package LCM::Landscape::ProductInstance;

use strict;
use LCM::Landscape::SoftwareComponent;

sub new {
	require XML::LibXML;

	my ($class, $systemComponentManager, $productInstanceNode, $context) = @_;
	my $self = bless({}, $class);

	$self->_detectProductInstanceTags($productInstanceNode, $context);
	$self->_detectedSoftwareComponents($systemComponentManager, $productInstanceNode, $context);

	return $self;
}

sub getSoftwareComponents {
	my ($self) = @_;
	return $self->{softwareComponents};
}

sub addSoftwareComponents {
	my ($self, $newSoftwareComponents) = @_;
	return unless $newSoftwareComponents;
	push @{$self->{softwareComponents}}, $_ for(@$newSoftwareComponents);
}

sub getName {
	my ($self) = @_;
	my $productInstanceTags = $self->_getProductInstanceTags();

	return $productInstanceTags->{caption};
}

sub isEmpty {
	my ($self) = @_;
	my $softwareComponents = $self->getSoftwareComponents() || [];

	return scalar(@{$softwareComponents}) == 0;
}

sub createElement {
	my ($self, $shallAddSoftwareComponentsData) = @_;
	my $productInstanceTags = $self->_getProductInstanceTags();
	my $installedSWFeature = XML::LibXML::Element->new('installedSWFeature');
	my @orderedTags = ('caption', 'instanceKey', 'identifyingNumber'); 

	for my $name (@orderedTags) {
		my $tag = XML::LibXML::Element->new($name);
		my $value = $productInstanceTags->{$name};
		$tag->appendTextNode($value);
		$installedSWFeature->appendChild($tag);
	}
	if($shallAddSoftwareComponentsData){
		$installedSWFeature->appendChild($self->_createSoftwareComponentsData());
	}
	return $installedSWFeature;
}

sub getID {
	my ($self) = @_;
	my %tags = %{$self->_getProductInstanceTags()};

	return sprintf("ID<%s>InstanceKey<%s>Caption<%s>", @tags{qw(identifyingNumber instanceKey caption)});
}

sub _createSoftwareComponentsData {
	my ($self) = @_;
	my $softwareComponents = $self->getSoftwareComponents();
	my $softwareComponentsTag = XML::LibXML::Element->new('software-components');

	for my $softwareComponent (@{$softwareComponents}){
		$softwareComponentsTag->appendChild($softwareComponent->createElement());
	}
	return $softwareComponentsTag;
}

sub _setProductInstanceTags {
	my ($self, $productInstanceTags) = @_;
	$self->{productInstanceTags} = $productInstanceTags;
}

sub _getProductInstanceTags {
	my ($self) = @_;
	return $self->{productInstanceTags};
}

sub _setSoftwareComponents {
	my ($self, $softwareComponents) = @_;
	$self->{softwareComponents} = $softwareComponents;
}

sub _detectProductInstanceTags {
	my ($self, $productInstanceNode, $context) = @_;
	my $tags = {};

	eval {
		my $productInstanceHeader = $productInstanceNode->getElementsByTagName('product-instance-header')->[0];
		my $productInstanceKey = $productInstanceHeader->getElementsByTagName('product-instance-key')->[0];
		my $captionElement = $productInstanceHeader->getElementsByTagName('caption')->[0];
		my $instanceKeyElement = $productInstanceKey->getElementsByTagName('instance-ID')->[0];
		my $identifyingNumberElement = $productInstanceKey->getElementsByTagName('PPMS-ID')->[0];

		$tags->{caption} = $captionElement ? $captionElement->textContent() : '';
		$tags->{instanceKey} = $instanceKeyElement ? $instanceKeyElement->textContent() : '';
		$tags->{identifyingNumber} = $identifyingNumberElement ? $identifyingNumberElement->textContent() : '';
	};

	$self->_setProductInstanceTags($tags);
}

sub _detectedSoftwareComponents {
	my ($self, $systemComponentManager, $productInstanceNode, $context) = @_;
	my $softwareComponents = $productInstanceNode->getElementsByTagName('software-component-version') || [];
	my @detectedSoftwareComponents = map { new LCM::Landscape::SoftwareComponent($systemComponentManager, $_) } @{$softwareComponents};
	my @nonEmptySoftwareComponents = grep { !$_->isEmpty() } @detectedSoftwareComponents;

	$self->_setSoftwareComponents([ @nonEmptySoftwareComponents ]);
}

1;