package LCM::Persistence::XML;

use strict;
use SDB::Install::Persistence::Schema;
use SDB::Common::Utils qw(createXMLParser);
use LCM::Persistence::XML::Extension;
use LCM::Component qw ( PREPARE_PHASE OFFLINE_PHASE CONFIGURE_PHASE ONLINE_PHASE SYSTEM_ONLINE_PHASE ALL);

use parent 'LCM::Persistence::XMLBase';

my %XML_TAG_TO_SETTER_MAP = (
	'source_version' => \&_setSourceVersionString,
	'target_version' => \&_setTargetVersionString,
	'date' => \&_setCreationDateString,
	'next_phase' => \&_setNextPhaseString,
	'action' => \&_setActionString,
	'step' => \&_setCurrentStepString,
	'component_name' => \&_setComponentNameString,
);
my $EXTENSIONS_TAG_NAME = 'extensions';

sub new {
	my ($class, $filePath) = @_;
	require XML::LibXML;
	my $self = $class->SUPER::new($filePath);
	$self->{extensions} = [];

	eval {
		$self->_loadXMLData();
	};

	if($@){
		$self->getErrMsgLst()->addError("Error parsing file '$filePath': $@");
		$self->_setExistsMandatoryValidationError(1);
	}

	return $self;
}

sub getSourceVersionString {
	my ($self) = @_;
	return $self->{_componentSourceVersion} || '<unknown>';
}

sub getTargetVersionString {
	my ($self) = @_;
	return $self->{_componentTargetVersion} || '<unknown>';
}

sub getNextPhaseString {
	my ($self) = @_;
	return $self->{_nextPhase} || '<unknown>';
}

sub getActionString {
	my ($self) = @_;
	return $self->{_action} || '<unknown>';
}

sub getCurrentStepString {
	my ($self) = @_;
	return $self->{_stepName} || '<unknown>';
}

sub getComponentNameString {
	my ($self) = @_;
	return $self->{_componentName} || '<unknown>';
}

sub getExtensions {
	my ($self) = @_;
	return $self->{extensions};
}

sub getExtension {
	my ($self, $name) = @_;
	for my $extension (@{$self->getExtensions()}) {
		return $extension->getValue() if ($extension->getName() eq $name);
	}
	return undef;
}

#
# Only private subs below this line
#

sub _loadXMLData {
	my ($self) = @_;
	my $parser = createXMLParser();
	my $filePath = $self->getFilePath();
	my $tree = $parser->parse_file($filePath);

	eval {
		SDB::Install::Persistence::Schema::validate($tree);
	};
	if ($@) {
		$self->getErrMsgLst()->addError("Error validating file '$filePath' against XML schema: $@");
	}
	my $xPathContext = XML::LibXML::XPathContext->new($tree);
	$xPathContext->registerNs('tns' => 'http://www.hdblcm.org/persistence');

	for my $tagName (keys(%XML_TAG_TO_SETTER_MAP)){
		my $element = $self->_getXmlNodeByTag($tagName, $xPathContext);
		if(defined($element)){
			my $setterMethodName = $XML_TAG_TO_SETTER_MAP{$tagName};
			&$setterMethodName($self, $element->textContent());
		}
	}
	my $extensions = $self->_getXmlNodeByTag($EXTENSIONS_TAG_NAME, $xPathContext);
	my @nodes = $extensions ? $extensions->nonBlankChildNodes() : ();
	$self->_addExtension($_->nodeName(), $_->textContent()) for @nodes;
}

sub _getXmlNodeByTag {
	my ($self, $tagName, $xPathContext) = @_;
	my $xPathString = sprintf("/tns:persistence/%s", $tagName);
	return $xPathContext->findnodes($xPathString)->get_node(1);
}

sub _setSourceVersionString {
	my ($self, $sourceVersionString) = @_;
	$self->{_componentSourceVersion} = $sourceVersionString;
}

sub _setTargetVersionString {
	my ($self, $targetVersionString) = @_;

	if ($targetVersionString =~ /^\d+(\.\d+)*$/) {
		$self->{_componentTargetVersion} = $targetVersionString;
		return;
	}

	$self->_setExistsMandatoryValidationError(1);
}

sub _setCreationDateString {
	my ($self, $dateString) = @_;
	$self->{_creationDate} = $dateString;
}

sub _setNextPhaseString {
	my ($self, $nextPhase) = @_;
	my $validPhases = [PREPARE_PHASE, OFFLINE_PHASE, CONFIGURE_PHASE, ONLINE_PHASE, SYSTEM_ONLINE_PHASE, ALL];

	if (grep {$nextPhase eq $_} @{$validPhases}) {
		$self->{_nextPhase} = $nextPhase;
		return;
	}

	$self->_setExistsMandatoryValidationError(1);
}

sub _setActionString {
	my ($self, $action) = @_;
	$self->{_action} = $action;
}

sub _setCurrentStepString {
	my ($self, $stepName) = @_;
	$self->{_stepName} = $stepName;
}

sub _setComponentNameString {
	my ($self, $componentName) = @_;
	$self->{_componentName} = $componentName;
}

sub _addExtension {
	my ($self, $name, $value) = @_;
	push @{$self->{extensions}}, LCM::Persistence::XML::Extension->new($name, $value);
}

1;