package LCM::Task::CommonTask::RemoveLocalRolesTask;
use parent qw (LCM::Task::SHASupportedTask);
use strict;

use LCM::HostsParser;
use LCM::ProcessExecutor;
use SDB::Install::Globals qw ($gHostRoleAcceleratorWorker $gHostRoleAcceleratorStandby $gLogDir $gHostRoleXS2Standby $gHostRoleXS2Worker);
use SDB::Install::System  qw (isSidadmin);
use SDB::Install::SysVars qw ($path_separator);

my $HOSTCTRL_OPERATION      = 'hdblcm_remove_roles_v5';
my $PROGRESS_MESSAGE        = 'Removing Local Host Roles';
my $PASSWORD_KEYS           = ['Password', 'AcceleratorPassword', 'SQLSysPasswd', 'SQLTenantUserPassword'];

our @EXPORT = qw ($PROGRESS_MESSAGE);

sub _executeInternal {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $sid = $configuration->getSID();
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName() . '...');
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $returnCode = isSidadmin($sid) ? $self->_executeSHAOperation() : $self->_executeAsRoot();

    if($returnCode) {
        $self->getStatus()->_setFinishedState();
        $message->endMessage (undef, sprintf("%s finished", $self->getName()));
    } else {
        $self->getStatus()->_setErrorState();
        $message->endMessage (undef, sprintf("%s failed", $self->getName()));
    }
    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $self->setMsgLstContext($saveContext);
}

sub _executeSHAOperation {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $localHost = $self->_createRemoteHostsHostctrlObject();
    $localHost->useSidadm();
    $localHost->setMsgLstContext( $self->getMsgLstContext() );

    my $rc = 1;
    my $optionsMap = $self->buildSHAOptionsMap();
    my $exitCode = $localHost->executeHostctrlParallel($HOSTCTRL_OPERATION, $configuration, undef, $PASSWORD_KEYS, undef, undef, undef,
                                                        undef, undef, $optionsMap, undef, undef, undef, 0);
    my $outputBuffer = $localHost->getOutputBuffer();
    my $executionOutput = [ split('\n', $outputBuffer->[0]) ];
    $self->_setLogLocation($self->_parseLogFileLocation($executionOutput));

    if (!defined $exitCode || ($exitCode != 0)) {
        $self->setErrorMessage($self->getFailedExecutionMessage(), $localHost->getErrMsgLst());
        $rc = undef;
    }
    return $rc;
}

sub _createRemoteHostsHostctrlObject {
    my ($self) = @_;
    my $progressHandler  = $self->getMsgLst()->getProgressHandler();
    my $configuration = $self->_getConfiguration();
    my @localHost = ($configuration->getLocalHanaHost());
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts(@localHost);
    $localExecutionHost->setOutputHandler($localHost[0], $progressHandler);
    return $localExecutionHost;
}

sub buildSHAOptionsMap{
    my $self = shift();
    my $result = $self->SUPER::buildSHAOptionsMap();
    my $rolesString = $self->_getLocalHostRolesString();
    my $configuration = $self->_getConfiguration();
    $result->{INSTLOG_DIR} = $gLogDir;
    $result->{ROLES} = $rolesString;
    if($rolesString =~ /$gHostRoleAcceleratorWorker|$gHostRoleAcceleratorStandby/){
        $result->{ASE_USER} = $configuration->getValue('AcceleratorUser');

        my $systemUser = $configuration->getValue('SystemUser');
        $systemUser = 'SYSTEM' if(!defined($systemUser) || length($systemUser) == 0);
        $result->{SYSTEM_USER} = $systemUser;
    }
    if (!$configuration->isSkipped('AutoInitializeServices')) {
        $result->{AIS} = $configuration->getValue('AutoInitializeServices');
    }
    if (!$configuration->isSkipped('TenantUser')) {
        $result->{TENANT_USER} = $configuration->getValue('TenantUser');
    }
    if($rolesString =~ /$gHostRoleXS2Standby|$gHostRoleXS2Worker/){
        my $keepXsUsers = $configuration->getValue('KeepXsUsers') ? 'on' : 'off';
        my $skipModifySudoers = $configuration->getValue('SkipModifySudoers') ? 'on' : 'off';

        $result->{KXU} = $keepXsUsers;
        $result->{SUD} = $skipModifySudoers;
    }
    my $isForced = !$configuration->isSkipped('ForceRemoveHosts') && $configuration->getValue('ForceRemoveHosts');
    $result->{'FORCE'} = $isForced ? 'on' : 'off';
    return $result;
}

sub _executeAsRoot {
    my ($self) = @_;
    my $executor = $self->_createProcessExecutor();
    my $saveCntxt = $self->getMsgLstContext();
    $executor->setMsgLstContext($self->getMsgLstContext());

    my $rc = $executor->executeProgram(1);
    $self->_setLogLocation($self->_parseLogFileLocation($executor->getOutputLines()));
    $self->setMsgLstContext($saveCntxt);

    return 1 if (defined $rc && $rc == 0);

    $self->getErrMsgLst()->addError($self->getFailedExecutionMessage());
    return undef;
}

sub _createProcessExecutor {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $command = $self->_getExecutable();
    my $args = $self->_buildArgs();
    my $executor;
    if ( $self->_isPasswordArgsExists() ){
        my $xmlPasswordStream = $configuration->getXmlPasswordStream();
        $executor = LCM::ProcessExecutor->new($command, $args, $xmlPasswordStream);
    } else {
        $executor = LCM::ProcessExecutor->new($command, $args);
    }
    $executor->setOutputHandler($self->getMsgLst()->getProgressHandler());
    $executor->setProcessEnvironment (LCM::TraceLoggingEnvironment::PrepareProcessExecutorHdbEnvironment($self->_getExecutable()));
    return $executor;
}

sub _buildArgs{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $localRolesString = $self->_getLocalHostRolesString();
    my $removeRolesOpt = $configuration->getOpt('RemoveRoles');
    my $localRolesArgument = "${removeRolesOpt}=${localRolesString}";
    my $args = ['-b', $localRolesArgument, '--read_password_from_stdin=xml',"--instlog_dir=$gLogDir"];

    if($localRolesString =~ /$gHostRoleAcceleratorWorker|$gHostRoleAcceleratorStandby/){
        my $acccelartorUserOpt = $configuration->getOpt('AcceleratorUser');
        my $acceleratorUser = $configuration->getValue('AcceleratorUser');
        my $acceleratorUserArgument = "${acccelartorUserOpt}=${acceleratorUser}";
        push(@{$args}, $acceleratorUserArgument);

        my $systemUserOpt = $configuration->getOpt('SystemUser');
        my $systemUser = $configuration->getValue('SystemUser');
        $systemUser = 'SYSTEM' if(!defined($systemUser) || length($systemUser) == 0);
        my $systemUserArgument = "${systemUserOpt}=$systemUser";
        push(@{$args}, $systemUserArgument);
    }
    if (!$configuration->isSkipped('AutoInitializeServices')) {
        my $autoInitOpt = $configuration->getOpt('AutoInitializeServices');
        my $autoInit = $configuration->getValue('AutoInitializeServices');
        my $autoInitArgument = "${autoInitOpt}=$autoInit";
        push(@{$args}, $autoInitArgument);
    }
    if (!$configuration->isSkipped('TenantUser')) {
        my $tenantUserOpt = $configuration->getOpt('TenantUser');
        my $tenantUser = $configuration->getValue('TenantUser');
        my $tenantUserArgument = "${tenantUserOpt}=$tenantUser";
        push(@{$args}, $tenantUserArgument);
    }
    if($localRolesString =~ /$gHostRoleXS2Standby|$gHostRoleXS2Worker/){
        my $keepXsUsers = $configuration->getValue('KeepXsUsers') ? 'on' : 'off';
        my $keepXsUsersOption = $configuration->getOpt('KeepXsUsers');
        my $skipModifySudoers = $configuration->getValue('SkipModifySudoers') ? 'on' : 'off';
        my $skipModifySudoersOption = $configuration->getOpt('SkipModifySudoers');

        push(@{$args}, sprintf('%s=%s', $keepXsUsersOption, $keepXsUsers));
        push(@{$args}, sprintf('%s=%s', $skipModifySudoersOption, $skipModifySudoers));
    }
    if(!$configuration->isSkipped('ForceRemoveHosts') && $configuration->getValue('ForceRemoveHosts')){
        push(@{$args}, sprintf('-force=%s', 'on'));
    }
    return $args;
}

sub _getLocalHostRolesString {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $localHost = $configuration->getLocalHanaHost();

    my $rolesMap = $configuration->getValue('RemoveRoles') || {};
    return '' if(length($rolesMap->{$localHost}) <= 0);

    my $rolesString = join(':role=', split(',', $rolesMap->{$localHost}));
    return sprintf('%s:role=%s', $localHost, $rolesString);
}

sub _isPasswordArgsExists{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    return 1 if($configuration->getValue("Password"));
    return 1 if($configuration->getValue("AcceleratorPassword"));
    return 1 if($configuration->getValue("SQLSysPasswd"));

    return 0;
}

sub _getExecutable {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $hdbExecutable =
           join($path_separator,
                $configuration->getSAPSystem()->get_globalTrexInstallDir(),
                'bin',
                'hdbmodify');

    return $hdbExecutable;
}

sub getId {
    return 'remove_local_roles';
}

sub getSlppLogFileName{
    return 'remove_local_roles.log';
}

sub getName {
    return "Remove Specified Local Host Roles";
}

sub getParameterToOptionMapping {
    return {
        'Target' => 'SAPMNT',
        'SID' => 'SID',
    };
}

sub getExecutionName {
    return $PROGRESS_MESSAGE;
}

sub _getNumberOfExpectedOutputLines{
    return 100;
}

1;