package LCM::Task::ConfigureInternalNetwork::ConfigureInternalNetworkSubtask;

use strict;
use File::Spec;
use SDB::Install::Globals qw ($gLogDir);
use parent 'LCM::Task::ConfigureInternalNetwork::BaseInternalNetworkSubtask';

# Override
sub getId { return 'configure_internal_network_internal'; }

# Override
# Windows is currently not supported
sub _configure {
	my ($self, $configuration) = @_;
	my $result = $self->SUPER::_configure($configuration);

	if($result && $self->_shallCreatePersistenceFile()){
		$self->_createPersistenceFile($configuration);
		$self->_addScopeWarnings($configuration);
	}

	return $result;
}

# Override
sub _getHdbToolName {
	my ($self) = @_;
	my $configuration = $self->_getConfiguration();
	my $isScopeInstance = $configuration->getValue('Scope') eq 'instance';
	my $isRestartNeeded = $configuration->isNetworkReconfigRestartRequired();
	return ($isScopeInstance && $isRestartNeeded) ? 'hdbreg' : 'hdbmodify';
}

# Override
sub _createHdbToolArgs {
	my ($self, $instconfig) = @_;
	my $listenInterface = $instconfig->getListenInterface();
	my @args = ('-b', "--listen_interface=$listenInterface");
	push(@args, '--instlog_dir=' . $gLogDir);

	my $internalNetwork = $instconfig->getInternalNetworkAddress();
	if(!$instconfig->isSkipped('InternalNetwork') && length($internalNetwork) > 0 && $internalNetwork ne 'none') {
		push(@args, "--internal_network=$internalNetwork");
	}

	return \@args if(!$instconfig->isDistributedSystem());

	my $noStartValue = $instconfig->getValue('NoStart');
	push(@args, "--nostart=$noStartValue");
	my $scope = $instconfig->getValue('Scope');
	push(@args, "--scope=$scope");

	return \@args if($scope ne 'system');

	my $remoteExecution = $instconfig->getValue('RemoteExecution');
	push(@args, "--remote_execution=$remoteExecution");

	if(!$instconfig->isUseSAPHostagent()){
		my $rootUser = $instconfig->getRootUser();
		push @args, ("--root_user=$rootUser");
	}
	return \@args;
}

# Override
sub _getPasswordKeys {
	return ['Password', 'RootPassword'];
}

# Override
sub _getNumberOfExpectedOutputLines {
	return 50;
}

sub _addScopeWarnings {
	my ($self, $instconfig) = @_;
	my $hdbInstance = $instconfig->getOwnInstance();
	my $hosts = $hdbInstance->get_hosts();
	my $message = "Inter-Service Communication configuration is not yet finished.\nTo complete this action, run 'hdblcm --action=configure_internal_network' on host(s): " . join(', ', @{$hosts});
	LCM::ExecutionWarningsObservable->getInstance()->notifyWarning($message);
}

sub _createPersistenceFile {
	my ($self, $instconfig) = @_;
	my $hdbInstance = $instconfig->getOwnInstance();
	my $instanceDir = $hdbInstance->get_instanceDir();
	my $hosts = $hdbInstance->get_hosts();

	for my $host(@{$hosts}){
		my $hostDir = File::Spec->catfile($instanceDir, $host);

		next if(!-d $hostDir);

		my $persistenceFilePath = File::Spec->catfile($instanceDir, $host, 'hdblcm_configure_internal_network');
		my $persistenceData = $self->_getPersistenceData($hdbInstance, $instconfig);
		$instconfig->pers_store($persistenceFilePath, $persistenceData);
		$self->getMsgLst()->addMessage("Created persistence file '" . $persistenceFilePath . "'");
	}
}

sub _getPersistenceData {
	my ($self, $hdbInstance, $instconfig) = @_;
	my $localHost = $hdbInstance->get_host();
	my $allHosts = $hdbInstance->get_allhosts();
	my $data = { listenInterface => $instconfig->getListenInterface() };
	return $data;
}

# Override
sub _shallCreatePersistenceFile {
	my ($self) = @_;
	my $configuration = $self->_getConfiguration();
	my $scope = $configuration->getValue('Scope');
	my $listenInterface = $configuration->getListenInterface();
	my $isDistributed = $configuration->isDistributedSystem();
	my $isRestartNeeded = $configuration->isNetworkReconfigRestartRequired();
	return ($scope eq 'instance' && $isDistributed && $isRestartNeeded && $listenInterface ne 'local');
}

1;