package LCM::Task::ConfigureSLDTask::ConfigureSLDTask;

#####################################################################################
# 					Authors : Philip Mihaylov, Martin Lakov	and Georgi Petkov		#
#####################################################################################
#	Before execution there are four mandatory parameters and one optional			#
#		- first (m)  : hostname of the SLD system									#
#		- second (m) : SLD port on the SLD system									#
#		- third (m)  : username for the SLD 										#
#		- fourth (m) : password for the SLD											#
#		- fifth      : flag for HTTPS												#
#	There are four steps to be done in order to configure SLD registration  		#
#		- first  : call the sldreg tool with the parameters provided				#
#		- second : provide lm structure path in the sld section in nameserver.ini	#
#		- third  : enable sld in the nameserver.ini file							#
#		- fourth : reconfigure sld													#
#####################################################################################

use strict;
use File::Spec;
use parent 'LCM::Task';

use LCM::SLD::SldregUtility;
use LCM::Python::HanaPropertiesHelper;
use LCM::Python::SLDReconfigurationHelper;

use SDB::Install::System qw(isSidadmin);
use SDB::Install::SysVars qw($isWin);
use SDB::Install::SAPSystem;
  
my $CONFIGURATION_NAME = 'SLD registration';
my $PROGRESS_MESSAGE = "Configuring $CONFIGURATION_NAME...";
my $SUCCESS_END_MESSAGE = "Finished configuration of $CONFIGURATION_NAME";
my $FAIL_END_MESSAGE = "Configuration of $CONFIGURATION_NAME failed";

my $UNABLE_TO_GET_HDB = 'Unable to get HDB Instance';


sub getId {
	return 'configure_sld_internal';
}

sub getName {
	return 'Configure System Landscape Directory Registration';
}

sub getExecutionName {
	return 'Configuring SLD Registration';
}

sub _executeInternal {
	my $self = shift();
	my $configuration = $self->_getConfiguration();
    
	if (!defined $self->_configure($configuration)) {
		$self->getStatus()->_setErrorState();
		$self->setErrorMessage( 'SLD configuration failed', $self->getErrMsgLst());
		return;
	}
	
	$self->getStatus()->_setFinishedState();
}

sub _configure {
	my ($self, $configuration) = @_;
	my $message = $self->getMsgLst()->addProgressMessage($PROGRESS_MESSAGE);
	my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
	my $returnCode = undef;
	
	# Windows is currently not supported
	if($isWin){
		$self->setErrorMessage ('Windows currently not supported.', undef);
		$self->getStatus()->_setErrorState();
	}
	else {
		# Execute step one
		$returnCode = $self->_runSldreg($configuration);
	
		# Execute steps two, three and four
		if(defined($returnCode)){
			$returnCode = $self->_enableSLD($configuration) 
		}
		if(defined $returnCode){
			$self->getStatus()->_setFinishedState();
		} else {
			$self->getStatus()->_setErrorState();
		}
	}

	my $endMessage = (defined $returnCode) ?  $SUCCESS_END_MESSAGE : $FAIL_END_MESSAGE;
	$message ->endMessage(undef, $endMessage);
	$self->setMsgLstContext($saveContext);
	return $returnCode;
}
	
# Here we try to execute step one
sub _runSldreg {
    my ($self, $configuration) = @_;
	if (!defined $configuration->getOwnInstance()) {
		$self->getErrMsgLst()->addMessage($UNABLE_TO_GET_HDB);
		return undef;
	}
	my $sldreg = LCM::SLD::SldregUtility->new($configuration);

	$sldreg->setOutputHandler($self->getMsgLst()->getProgressHandler());
	# sets delay in seconds when data are send to the called process.
	# it is required in SLDregistration scenario to workaround a bug with parsing of provided
	# arguments in sldreg ver 8.0 
	$sldreg->setDelayOnWrite(5);

	my $returnCode = $sldreg->configureSLD();
	$self->getMsgLst()->addMessage(undef, $sldreg->getMsgLst());

	if (!$returnCode){
	    $self->setErrorMessage ("Execution of sldreg failed: $@", undef);
	    return undef;
	}

	return 1;
}

sub _enableSLD {
	my ($self, $configuration) = @_;
    my $hdbInstance = $configuration->getOwnInstance();
	if(!defined $hdbInstance){
		$self->getErrMsgLst()->addMessage($UNABLE_TO_GET_HDB);
		return undef;
	}

	my $pythonExecutor = new LCM::Python::SLDReconfigurationHelper($hdbInstance, SDB::Install::System::isSidadmin(), $self->getMsgLst()->getProgressHandler());
	$pythonExecutor->setMsgLstContext($self->getMsgLstContext());

	# This will try to execute step two 
	my $sapSystem = $configuration->getSAPSystem();
	my $hanaSharedSidPath = $sapSystem->get_globalSidDir();
	my $lmStructurePath = File::Spec->catdir($hanaSharedSidPath , "lm_structure");
	my $exitCode = $pythonExecutor->setLmStructurePath($configuration->getSID(),$lmStructurePath);

	# This will try to execute step three
	if(defined($exitCode)){
        if($pythonExecutor->isSldEnabled($configuration->getSID())){
            $exitCode = $pythonExecutor->setEnableSld($configuration->getSID(),'no');
        }
		$exitCode = $pythonExecutor->setEnableSld($configuration->getSID(),'yes');
	}

	# This will try to execute step four
	my $host = $hdbInstance->get_host();
	my $instanceNumber = $hdbInstance->get_nr();
	require SDB::Install::SAPControl;
	my $sapControl = new SDB::Install::SAPControl($host,
	                                              $instanceNumber,
	                                              undef, # user
	                                              undef, # password,
	                                              $configuration->isUseHttps(),
	                                              $configuration->getValue ('SSOCertificate'));
	if(defined($exitCode) && $sapControl->isRunning()){
		$exitCode = $pythonExecutor->reconfigureSLD($configuration->getSID());
	}
	return $exitCode;
}

sub _getNumberOfExpectedOutputLines{
    return 50;
}

1;
