package LCM::Task::DownloadComponentsProcessTask;

use strict;
use URI::Escape qw(uri_unescape);
use parent qw(LCM::Task::SLProcessTask);
use LCM::FileUtils qw(isEmptyDirectory);
use LCM::Task::ExtractComponentsTask::ExtractDownloadedComponentsTask;
use LCM::Task::DownloadComponents::DownloadSingleComponentTask;
use LCM::Task::DownloadComponents::CleanupDownloadDirectoryTask;

my $ACTION_DONE_MESSAGE = "Download of components finished successully";
my $ACTION_DONE_WITH_WARNINGS_MESSAGE = "Download of components finished warnings";
my $ACTION_FAILED_MESSAGE = "Download of components failed";
my $ACTION_ABORTED_MESSAGE = "Download of components failed";
my $ENCODED_EMPTY_COLLECTION = '%7B%22components%22%3A%20%7B%7D%7D';

# Override
sub new{
    my ($class, $configuration) = @_;
    my $self = $class->SUPER::new($configuration);
    my $status = $self->getStatus();

    $status->_setActionDoneMessage($ACTION_DONE_MESSAGE);
    $status->_setActionDoneWithWarningsMessage($ACTION_DONE_WITH_WARNINGS_MESSAGE);
    $status->_setActionFailedMessage($ACTION_FAILED_MESSAGE);
    $status->_setActionAbortedMessage($ACTION_ABORTED_MESSAGE);
    $self->_createSubtasks();

    return $self;
}

# Override
sub getId {
    return "download_components";
}

# Override
sub getName {
    return "Download Components";
}

sub _createSubtasks {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $selectedComponentsCsv = $configuration->getValue('SelectedComponents');
    my $downloadableComponentsInfo = $self->_getDownloadableComponentsInformation($configuration);
    my $shallCleanupDownloadDir = $configuration->getValue('OverwriteExtractDir');
    my $downloadTarget = $configuration->getValue('DownloadDir');
    my @computedSubtasks = ();
    my @downloadFiles = ();

    for my $componentEntry (split(',', $selectedComponentsCsv)){
        if($componentEntry =~ /([^:]+):version=(.+)/){
            my ($ppmsId, $selectedVersion) = ($1, $2);
            my $downloadInformation = $self->_getDownloadInformation($downloadableComponentsInfo, $ppmsId, $selectedVersion);
            push(@downloadFiles, @{$downloadInformation->{'files'}});
            push(@computedSubtasks, new LCM::Task::DownloadComponents::DownloadSingleComponentTask($downloadInformation, $configuration));
        }
    }
    if($shallCleanupDownloadDir && -d $downloadTarget && !isEmptyDirectory($downloadTarget)){
        unshift(@computedSubtasks, new LCM::Task::DownloadComponents::CleanupDownloadDirectoryTask(\@downloadFiles, $configuration));
    }
    if(!$configuration->isSkipped('ExtractAfterDownload') && $configuration->getValue('ExtractAfterDownload')){
        push(@computedSubtasks, new LCM::Task::ExtractComponentsTask::ExtractDownloadedComponentsTask($configuration));
    }
    $self->addSubtask($_) for(@computedSubtasks);
}

sub _getDownloadableComponentsInformation {
    local %ENV = %ENV;
    $ENV{PERL_JSON_BACKEND} = 'JSON::backportPP';
    require JSON;

    my ($self, $configuration) = @_;
    my $updatableComponentsEncodedString = $configuration->getValue('UpdatableComponents') || $ENCODED_EMPTY_COLLECTION;
    my $installableComponentsEncodedString = $configuration->getValue('InstallableComponents') || $ENCODED_EMPTY_COLLECTION;
    my $updatableComponents = JSON::from_json(uri_unescape($updatableComponentsEncodedString));
    my $installableComponents = JSON::from_json(uri_unescape($installableComponentsEncodedString));

    return { %{$updatableComponents->{'components'}}, %{$installableComponents->{'components'}} };
}

sub _getDownloadInformation {
    my ($self, $downloadableComponentsInfo, $ppmsId, $selectedVersion) = @_;
    my $componentInfo = { %{$downloadableComponentsInfo->{$ppmsId}} };
    ($selectedVersion) = grep { $_->{'version'} eq $selectedVersion } @{$componentInfo->{versions}};

    $selectedVersion->{'name'} = $componentInfo->{'name'};
    $selectedVersion->{'ppmsId'} = $ppmsId;

    return $selectedVersion;
}

sub getDownloadedArchiveNames {
    my $self = shift;
    return $self->{downloadedArchiveNames};
}

sub addDownloadedArchiveName {
    my ($self, $name) = @_;
    $self->{downloadedArchiveNames} = [] if (!exists $self->{downloadedArchiveNames});
    push(@{$self->{downloadedArchiveNames}}, $name);
}

1;
