package LCM::Task::ExtractComponentsTask::ExtractComponentsTask;

use strict;
use parent qw(LCM::Task);

use LCM::ProcessExecutor;
use LCM::FileUtils;
use LCM::Manifests::XS2ApplicationManifest;
use LCM::ComponentManager::MediumComponentManager;
use LCM::Manifests::GenericManifest;

use SDB::Install::User;
use SDB::Install::System qw(copy_file);
use SDB::Install::DirectoryWalker;
use SDB::Install::SysVars qw($isWin);
use SDB::Install::Globals qw($gProductNamePlatform);

use SAPDB::Install::Jar;

use File::Spec;
use File::Copy;
use File::Path;

my $SCENARIO_NAME = 'Extract Components';
my $PROGRESS_MESSAGE = "Extracting components...";
my $SUCCESS_END_MESSAGE = "Finished $SCENARIO_NAME";
my $FAIL_END_MESSAGE = "$SCENARIO_NAME failed";
my $WARNING_END_MESSAGE = "$SCENARIO_NAME finished with warnings";

my $multispanRarPartRegex = '.*_P[0-9]+\.[Ee][Xx][Ee]';
my $shaSarRegex = '.*SAPHOSTAGENT.*';

sub getId {
    return 'extract_components_task';
}

sub getName {
    return 'Extract Component Archives';
}

sub getExecutionName {
    return 'Extracting Component Archives';
}

sub _executeInternal {
    my ($self) = @_;

    my $message = $self->getMsgLst()->addProgressMessage($PROGRESS_MESSAGE);
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);

    my $configuration = $self->_getConfiguration();
    my $componentArchivesPath = $configuration->getValueOrBatchValue("ComponentArchivesPath");
    my $extractDir = $configuration->getValueOrBatchValue("ExtractTempDir");

    my $returnCode = $self->_extractAll($componentArchivesPath, $extractDir);

    if (!$returnCode) {
        $self->getStatus()->_setErrorState();
    } else {
        $self->getStatus()->_setFinishedState();
    }

    my $endMessage = ($returnCode) ?  $SUCCESS_END_MESSAGE : $FAIL_END_MESSAGE;
    $message ->endMessage(undef, $endMessage);
    $self->setMsgLstContext($saveContext);
    return $returnCode;
}

sub _extractAll {
    my ($self, $archivesPath, $extractDir) = @_;
    my $progressHanlder = $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $returnCode = $self->_handleCreationOfExtractTempDir($extractDir);
    return 0 if (! $returnCode);

    $returnCode = $self->_extractSars($archivesPath, $extractDir);
    return 0 if (! $returnCode);

    $returnCode = $self->_extractZips($archivesPath, $extractDir);
    return 0 if (! $returnCode);
    
    $returnCode = $self->_extractTars($archivesPath, $extractDir);
    return 0 if (! $returnCode);
    
    $returnCode = $self->_extractMultispanRars($archivesPath, $extractDir);
    return 0 if (! $returnCode);

# This handles SARs extracted from TAR archives
    $returnCode = $self->_extractSars($extractDir, $extractDir);
    return 0 if (!$returnCode);

    $returnCode = $self->_cleanUp($extractDir);

    return 1;
}

sub _extractZips {
    my ($self, $archivesPath, $extractDir) = @_;
    my $zips = $self->_filterArchiveNamesForExtraction($archivesPath, ".ZIP");

    return 0 if(! defined($zips));
    return 1 if(! scalar(@$zips));

    for my $zip (@{$zips}) {
        my $zipFileName = (split (/\./, $zip))[0];
        my $zipPath = File::Spec->catfile($archivesPath, $zip);
        my $xsAppManifest = new LCM::Manifests::XS2ApplicationManifest($zipPath);

        if($xsAppManifest->isXS2Application()){
            next if(MyRealPath($archivesPath) eq MyRealPath($extractDir));

            $self->getMsgLst()->addProgressMessage("Copying ZIP archive '$zipPath'...");
            my $msgLst = SDB::Install::MsgLst->new();
            if(!copy_file($zipPath, File::Spec->catdir($extractDir, $zip), $msgLst)){
                my $errors = $msgLst->getMsgLstString();
                $self->setErrorMessage("Failed to copy file '$zipPath': $$errors");
                return 0;
            }
            next;
        }

        my $zipExtractDir = File::Spec->catdir($extractDir, $zipFileName);
        if (! -d $zipExtractDir && !$self->_createDir($zipExtractDir)) {
            $self->setErrorMessage("Failed to create '$zipExtractDir' directory: $!", undef);
            return 0;
        }

        if (!$self->_isValidZIP($zipPath, $zipExtractDir)) {
            return 0 if !$self->_deleteDir($zipExtractDir);
            next;
        }

        $self->getMsgLst()->addProgressMessage("Extracting ZIP archive $zipPath ...");
        if (!$self->_extractZip($zipPath, $zipExtractDir)) {
            return 0;
        }
    }
    return 1;
}

sub _isValidZIP {
    my ($self, $zipPath, $extractionPath) = @_;
    my $componentDirPattern = '^([^\/]+?\/)?('.(join '|', @componentDirNames).')/manifest$';
    if(!$self->_extractZip($zipPath, $extractionPath, $componentDirPattern)) {
        return 0;
    }

    my $componentManager = LCM::ComponentManager::MediumComponentManager->new();
    $componentManager->detectComponentsByFsRoot($extractionPath);
# Filter out the installer because it is being always added by _compileComponentList in MediumComponentManager.pm
    my $numberOfComponents = grep { !$_->isInstaller() } @{$componentManager->getAllComponents()};
    my $isArchiveValid = ($numberOfComponents == 1) ? 1 : 0;
    if (!$isArchiveValid) {
        my $reason = $numberOfComponents ? "more than one" : "no";
        my $message = "Archive '$zipPath' contains $reason $gProductNamePlatform components. Skipping extraction...";
        $self->getMsgLst()->addWarning($message);
        return 0;
    }

    return 1;
}

sub _extractZip {
    my ($self, $zipPath, $extractDir, $filePattern) = @_;
    my $archive = SAPDB::Install::Jar->new();
    my $buffer;

    $archive->OpenArchive($zipPath);
    for (my $i = 0; $i < $archive->{'num_of_files'}; $i++) {
        my $file = $archive->Next();

        next if (defined $filePattern && $file->{'filename'} !~ /$filePattern/);

        my ($volume, $directory, $filename) = File::Spec->splitpath($file->{'filename'});
        foreach my $fsnode (split (/[\\\/]+/, $directory)){
            if ($fsnode eq '..'){
                $archive->CloseArchive();
                $self->setErrorMessage("Cannot extract '$file->{filename}': '..' in path is forbidden");
                return 0;
            }
        }
        my $outDir = File::Spec->catdir($extractDir, $directory);
	    if (! -d $outDir && !$self->_createDir($outDir, 1)) {
            $self->setErrorMessage("Failed to create '$outDir' directory: $!", undef);
            $archive->CloseArchive();
	        return 0;
	    }
	    next if $filename eq "";

        my $outfile = File::Spec->catfile($outDir, $filename);
        if(!open (OUTFILE, ">", $outfile)){
            $self->setErrorMessage("Failed to create '$outfile' in '$outDir': $!", undef);
            $archive->CloseArchive();
            return 0;
        }

        $archive->Open();
	    while($archive->Read($buffer, 65536)){
	        print OUTFILE $buffer;
	    }
        $archive->Close();

	    if(!close(OUTFILE)) {
	        $self->setErrorMessage("Cannot close extracted file '$outfile': " . $!, undef);
	        $archive->CloseArchive();
	        return 0;
	    }

    }
    $archive->CloseArchive();

    if(!$self->_adjustPermissions($extractDir)){
        $self->setErrorMessage("Failed to adjust file permissions of files in directory '$extractDir'");
        return undef;
    }

    return 1;
}

sub _extractMultispanRars {
    my ($self, $archivesPath, $extractDir) = @_;

    my $multispanRars = $self->_filterArchiveNamesForExtraction($archivesPath, ".EXE");
    if (! defined $multispanRars) {
        return 0;
    }

    if (! scalar(@$multispanRars)) {
        return 1;
    }

    for my $multispanRar (@{$multispanRars}) {
        my ($rarFileName) = ($multispanRar =~ /^(.+)$multispanRarPartRegex$/);
        ($rarFileName) = ($multispanRar =~ /^(.+)\.[Ee][Xx][Ee]/) if (! $rarFileName);
        my $rarExtractDir = File::Spec->catdir($extractDir, $rarFileName);
         if (! -d $rarExtractDir && !$self->_createDir($rarExtractDir)) {
            $self->setErrorMessage("Failed to create '$rarExtractDir' directory: $!", undef);
            return 0;
        }

        my $refdataManifest = undef;
        my $rarPath = File::Spec->catfile($archivesPath, $multispanRar);
        my $refdataManfiestRelativePath = File::Spec->catfile('refdata', 'manifest');
        my $refdataManifestPath = File::Spec->catfile($rarExtractDir, $refdataManfiestRelativePath);
        if (!$self->_extractRar($rarPath, $rarExtractDir, $refdataManfiestRelativePath)) {
            return 0;
        }
        if(-f $refdataManifestPath){
            $refdataManifest = LCM::Manifests::GenericManifest->new($refdataManifestPath);
        }

        if(!defined $refdataManifest || !$refdataManifest->isReferenceData()){
        	return 0 if !$self->_deleteDir($rarExtractDir);
        	next;
        }else{
        	return 0 if !$self->_deleteDir(File::Spec->catfile($rarExtractDir,'refdata'));
        }
        
        $self->getMsgLst()->addProgressMessage("Extracting Reference Data multispan rar archive $rarPath ..."); 
        if(!$self->_extractRar($rarPath, $rarExtractDir)) {
            return 0;
        }
    }
    return 1;
}

sub _extractRar {
    my ($self, $rarPath, $extractDir, $file) = @_;

    my $arguments = [];
    if(! defined $file){
    	push (@$arguments, split(" ", "x $rarPath $extractDir"));
    }else{
    	push (@$arguments, split(" ", "x $rarPath $file $extractDir"));
    }

    my $configuration = $self->_getConfiguration();
    my $unrarExecutable = $configuration->getValueOrBatchValue("UnrarExecutableLocation");
    my $processExecutor = new LCM::ProcessExecutor($unrarExecutable, $arguments);

    my $returnCode = $processExecutor->executeProgram();
    if (! defined $returnCode || $returnCode) {
        $self->setErrorMessage("Failed to extract multispan rar archive $rarPath.");
        return 0;
    } elsif (!$self->_adjustPermissions($extractDir)){
        $self->setErrorMessage("Failed to adjust file permissions of files in directory '$extractDir'");
        return 0;
    }
    return 1;
}

sub _extractTars {
    my ($self, $archivesPath, $extractDir) = @_;
    my $allTars = $self->_filterArchiveNamesForExtraction($archivesPath, "\.(TAR|TGZ|TAR.GZ)");

    return 0 if (! defined $allTars);
    return 1 if (! scalar(@$allTars));

    my @tarCommonNames = grep { $_ !~ /^.+P[0-9]+\./ } @{$allTars};
    for my $commonName (@tarCommonNames) {
        return 0 if (!$self->_extractTarByCommonName($commonName, $archivesPath, $extractDir));
    }
    return 1;
}

sub _extractTarByCommonName {
    my ($self, $archive, $sourceDir, $extractDir) = @_;

# find all parts if multi-volume TAR
    my ($name, $extension) = ($archive =~ /^(.+)\.(tar|tgz|tar\.gz)$/);
    my $archiveParts = $self->_filterArchiveNamesForExtraction($sourceDir, "${name}P[0-9]+\.${extension}");
    my @sortedArchiveParts = sort(@$archiveParts);

    my $arguments = [];
    my $tarNameDisplayMsgSuffix = '';
    if (! scalar(@sortedArchiveParts)) {
        my $fullTarPath = File::Spec->catfile($sourceDir, $archive);
        $tarNameDisplayMsgSuffix = " '$fullTarPath'";
        push (@$arguments, split(" ", "--directory $extractDir -xv -f $fullTarPath"));
    } else {
        my $pathToTarParts = my $fullTarPath = File::Spec->catfile($sourceDir, $archive);
        $tarNameDisplayMsgSuffix = "s with prefix '$pathToTarParts...'";
        push (@$arguments, "--directory $extractDir");
        for my $tarPart (@sortedArchiveParts) {
            my $fullTarPath = File::Spec->catfile($sourceDir, $tarPart);
            push (@$arguments, split(" ", "-xvM -f $fullTarPath"));
        }
    }

    my $configuration = $self->_getConfiguration();
    my $tarExecutable = $configuration->getValueOrBatchValue("TarExecutableLocation");
    my $processExecutor = new LCM::ProcessExecutor($tarExecutable, $arguments);
    $processExecutor->setOutputHandler($self->getMsgLst()->getProgressHandler());

    my $multiVolumeTarDisplayMsgSuffix = (scalar(@sortedArchiveParts));
    $self->getMsgLst()->addProgressMessage("Extracting TAR archive$tarNameDisplayMsgSuffix ...");
    my $returnCode = $processExecutor->executeProgram();
    if (! defined $returnCode || $returnCode) {
        $self->setErrorMessage("Failed to extract TAR archive$tarNameDisplayMsgSuffix.");
        return 0;
    }
    return 1;
}

sub _extractSars {
    my ($self, $archivesPath, $extractDir) = @_;

    my $sars = $self->_filterArchiveNamesForExtraction($archivesPath, ".SAR");
    if (! defined $sars) {
        return 0;
    }

    if (! scalar(@$sars)) {
        return 1;
    }

    for my $sar (@{$sars}) {
        my $sarPath = File::Spec->catfile($archivesPath, $sar);
        if (!$self->_extractSingleSar($sarPath, $extractDir)) {
            return 0;
        }
    }
    return 1;
}

# The _extractSingleSar() sub first extracts the .SAR archive to
# a 'tmp' dir and only after it has tried to move an existing
# SIGNATURE.SMF file into the component's directory it tries
# to move it to the original target directory
sub _extractSingleSar {
    my ($self, $sarPath, $extractDir) = @_;

    my $isShaSar = ( (split("/", $sarPath))[-1] =~ /$shaSarRegex/ );
# SAPHOSTAGENT SAR should be extracted directly into its own directory
    if ($isShaSar) {
        my $shaTargetDir = File::Spec->catfile($extractDir, "SAP_HOST_AGENT");
        return $self->_executeSapcarCall($sarPath, $shaTargetDir);
    }
    my $tmpTarget = File::Spec->catfile($extractDir, "tmp");
    if (! -d $tmpTarget && !$self->_createDir($tmpTarget)) {
        return 0;
    }

    my $exitCode = $self->_executeSapcarCall($sarPath, $tmpTarget);
    if (!$exitCode) {
        return 0;
    }

    my $sarContents = $self->_listDirBySuffix($tmpTarget, '');
    if (! defined $sarContents) {
        $self->_cleanUp($extractDir);
        return 0;
    }

    if (scalar(@$sarContents) > 2 && $sarPath =~ /([^\/]+)\.SAR$/i) {
        my $sarName = $1;
        my $target = File::Spec->catfile($extractDir, "SAP_HANA_$sarName");
        $self->getMsgLst()->addMessage("Renaming '$tmpTarget' to '$target'...");
        if (! File::Copy::move($tmpTarget, $target)) {
            $self->setErrorMessage("Failed to rename '$tmpTarget' to '$target': $!", undef);
            $self->_cleanUp($extractDir);
            return 0;
        }
        return 1;
    }

    my $tmpComponentDirPath = $self->_getPathToFirstSubDirectory($tmpTarget);
    if ($tmpComponentDirPath && !$self->_moveSignatureSmfIfAvailable($tmpTarget, $tmpComponentDirPath)) {
        $self->_cleanUp($extractDir);
        return 0;
    }

    if ($tmpComponentDirPath && $tmpComponentDirPath =~ /([^\/]+)\/*$/) {
        my $componentDirName = $1;
        my $componentTargetPath = File::Spec->catfile($extractDir, $componentDirName);
        $self->getMsgLst()->addMessage("Moving '$tmpComponentDirPath' to '$componentTargetPath'...");
        if (! File::Copy::move($tmpComponentDirPath, $componentTargetPath)) {
            $self->setErrorMessage("Failed to move extracted component from '$tmpComponentDirPath' to '$extractDir': $!", undef);
            $self->_cleanUp($extractDir);
            return 0;
        }
    }

    $self->_cleanUp($extractDir);
    return 1;
}

sub _executeSapcarCall {
    my ($self, $sarPath, $targetDir) = @_;

    my $configuration = $self->_getConfiguration();
    my $sapcarLocation = $configuration->getValueOrBatchValue("SapcarLocation");

    my $arguments = ['-R', $targetDir, '-xf', $sarPath, "-manifest", "SIGNATURE.SMF"];
    my $processExecutor = new LCM::ProcessExecutor($sapcarLocation, $arguments);
    $processExecutor->setMsgLstContext($configuration->getMsgLstContext());
    $processExecutor->setOutputHandler($self->getMsgLst()->getProgressHandler());
    $self->getMsgLst()->addProgressMessage("Extracting SAR archive '$sarPath'...");
    my $exitCode = $processExecutor->executeProgram();

    if (! defined $exitCode || $exitCode) {
        $self->setErrorMessage("Failed to extract SAR archive '$sarPath'.");
        return 0;
    }

    return 1;
}

# This method is needed because there is no uniform
# agreement on what a component SAR archive should contain.
# i.e if it has more than a .SMF file and a single directory
sub _getPathToFirstSubDirectory {
    my ($self, $location) = @_;
    my $contents = $self->_listDirBySuffix($location, '');
    if (! defined $contents) {
        return 0;
    }
    for my $contentElement (@$contents) {
        my $contentElementPath = File::Spec->catfile($location, $contentElement);
        next if (! -d $contentElementPath);
        return $contentElementPath;
    }
    return '';
}

sub _moveSignatureSmfIfAvailable {
    my ($self, $source, $target) = @_;
    my $sourceSignatureSmfLocation = File::Spec->catfile($source, "SIGNATURE.SMF");
    if (-f $sourceSignatureSmfLocation) {
        $self->getMsgLst()->addMessage("Moving '$sourceSignatureSmfLocation' to '$target'...");
        if (! File::Copy::move($sourceSignatureSmfLocation, $target)) {
            $self->setErrorMessage("Failed to move '$sourceSignatureSmfLocation' to '$target': $!", undef);
            return 0;
        }
    }
    return 1;
}

sub _handleCreationOfExtractTempDir {
    my ($self, $extractDir) = @_;
    my $configuration = $self->_getConfiguration();
    if ($configuration->getValue('OverwriteExtractDir') && -d $extractDir) {
        $self->getMsgLst()->addMessage("Overwriting the directory '$extractDir'...");
        my $returnCode = File::Path::remove_tree($extractDir, {error => \[]}); #ignore error msgs
        if (! $returnCode) {
            $self->setErrorMessage("Failed to overwrite '$extractDir': $!", undef);
            return 0;
        }
    }
    return $self->_createDir($extractDir) if (! -d $extractDir);
    return 1;
}

sub _createDir {
    my ($self, $path, $recurse) = @_;
    my $currentUser = new SDB::Install::User();
    my ($fileMode, $uid, $gid, $isRecursive) = (0755, $currentUser->id(), $currentUser->gid(), $recurse);
    my $errlst = new SDB::Install::MsgLst();
    if (! LCM::FileUtils::createDirectory($path, $fileMode, $uid, $gid, $isRecursive, $errlst)) {
        my $errorMessage = $errlst->getMsgLstString();
        $self->setErrorMessage($$errorMessage, undef);
        return 0;
    }
    return 1;
}

sub _cleanUp {
    return $_[0]->_deleteDir(File::Spec->catfile($_[1], 'tmp'));
}

sub _deleteDir {
    my ($self, $extractDir) = @_;
    if (-d $extractDir && ! File::Path::remove_tree($extractDir, {error => \[]})) { #ignore error msgs
        $self->setErrorMessage("Could not remove '$extractDir' directory: $!");
        return 0;
    }
    return 1;
}

sub _filterArchiveNamesForExtraction {
    my $self = shift;
    return $self->_listDirBySuffix(@_);
}

sub _listDirBySuffix {
    my ($self, $dir, $suffix) = @_;

    my $errlst = new SDB::Install::MsgLst();
    my $ls = LCM::FileUtils::listDirectory($dir, $errlst);
    if (! defined $ls) {
# Error opening $dir
        my $errorMessage = $errlst->getMsgLstString();
        $self->setErrorMessage($$errorMessage, undef);
        return undef;
    }
    my @filteredLs = grep(/$suffix$/i, @$ls);
    @filteredLs = grep(!/^\.{1,2}$/, @filteredLs);
    return \@filteredLs;
}

sub _getNumberOfExpectedOutputLines{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $componentArchivesPath = $configuration->getValueOrBatchValue("ComponentArchivesPath");
    my $extractableArchives = $self->_filterArchiveNamesForExtraction($componentArchivesPath, ".(SAR|ZIP|TAR|TGZ)") || [];
    my $expectedOutputLines = scalar(@{$extractableArchives}) * 6 || 20;

    return $expectedOutputLines;
}

sub _adjustPermissions {
    my ($self, $directory) = @_;

    if(!$isWin){
        $self->getMsgLst()->addMessage("Adjusting file permissions of files in directory '$directory'...");

        my $dirWalker = new SDB::Install::DirectoryWalker(undef, undef, undef, undef, undef, undef, sub {
            my $filePath = File::Spec->catfile($_[3], $_[4]);
            my (undef, undef, $mode) = stat($filePath);

            chmod(($mode & 0777) | 0555, $filePath) or return undef;
        }, undef, undef, 1, 1, 0);

        return undef if(!$dirWalker->findAndProcess($directory));
    }
    return 1;
}

1;
