package LCM::Task::GenericStackTask::ComponentTasklistFactory::ComponentTasklistFactoryBase;

use strict;
use warnings;
use experimental qw (smartmatch);
use LCM::Task::GenericStackTask::ComponentTask;
use LCM::Task::GenericStackTask::ComponentTask::ComponentPhasedTask;
use LCM::Task::StartStopHostsTask::EnableSpecificServicesTask;
use LCM::Task::StartStopHostsTask::StartSpecificHostsTask;
use LCM::Task::StartStopHostsTask::StartSpecificServicesTask;
use LCM::Task::TaskStatus qw(FINISHED);
use LCM::Task::GenericStackTask::PostConfigurePhaseTaskGenerator;
use LCM::Utils::ComponentsDependencySort qw(sortComponentsTopologically constructComponentDependencyGraph);
use LCM::Component qw(PREPARE_PHASE ONLINE_PHASE OFFLINE_PHASE CONFIGURE_PHASE SYSTEM_ONLINE_PHASE ALL);
use SDB::Install::Globals qw($gHostRoleWorker $gHostRoleStandby);


sub new {
    my ($class, $configuration) = @_;
    my $self = bless({}, $class);
    $self->{configuration} = $configuration;
    $self->{_postPhaseTaskGenerator} = {};

    $self->addPostPhaseTaskGenerator(CONFIGURE_PHASE, LCM::Task::GenericStackTask::PostConfigurePhaseTaskGenerator->new());
    return $self;
}

sub createComponentTasklist {
    my ($self) = @_;
    ... # Implement in child classes
}

sub _getHostsWithRoles {
    my ($self, $roles) = @_;
    ... # Implement in child classes
}

sub _getInstancesToStartBeforeOnlinePhase {
    my ($self, $components) = @_;
    ... # Implement in child classes
}

sub getConfiguration {
    my ($self) = @_;
    return $self->{configuration};
}

sub addPostPhaseTaskGenerator {
    my ($self, $phase, $taskGenerator) = @_;
    $self->{_postPhaseTaskGenerator}->{$phase} //= [];
    push(@{$self->{_postPhaseTaskGenerator}->{$phase}}, $taskGenerator);
}

sub getPostPhaseTaskGenerators {
    my ($self, $phase) = @_;
    return $self->{_postPhaseTaskGenerator}->{$phase} // [];
}

sub _createDefaultTasklist {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $taskList = [];

    for my $component (@{$components}) {
        if ($component->isPostAddHostsComponent()) {
            next;
        }
        push (@{$taskList}, LCM::Task::GenericStackTask::ComponentTask->new($configuration,$component));
    }
    return $taskList;
}

sub _createPhasedTasklist {
    my($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $taskList = [];
    $components = sortComponentsTopologically($configuration->getComponentManager(), $components, 0);

    return $taskList if !$self->_recomputePhases($components);

    my $prepareTaskList         = $self->_createPrepareTasklist($components);
    my $offlineTaskList         = $self->_createOfflineTasklist($components);
    my $configureTaskList       = $self->_createConfigureTasklist($components);
    my $onlineTaskList          = $configuration->isServerNoStart() ? [] : $self->_createOnlineTasklist($components);
    my $systemOnlineTaskList    = $configuration->isServerNoStart() ? [] : $self->_createSystemOnlineTasklist($components);

    push(@{$taskList}, @{$prepareTaskList});
    push(@{$taskList}, @{$offlineTaskList});
    push(@{$taskList}, @{$configureTaskList});
    push(@{$taskList}, @{$onlineTaskList});
    push(@{$taskList}, @{$systemOnlineTaskList});

    if(defined $configuration->getResumeExecutionList()){
        $self->_updateResumedTasks($taskList);
    }

    return $taskList;
}

sub _updateResumedTasks {
    my ($self, $taskList) = @_;
    my $resumedExecutionList = $self->getConfiguration()->getResumeExecutionList();
    for my $task (@$taskList) {
        my $resumeStatus = $resumedExecutionList->{$task->getId()};
        $task->setSkipped(1) if (defined $resumeStatus && $resumeStatus eq FINISHED);
    }
}

sub _createPrepareTasklist {
    my ($self, $components) = @_;
    return $self->_createComponentPhasedTasks($components, PREPARE_PHASE);
}

sub _createOfflineTasklist {
    my ($self, $components) = @_;
    my $offlineTaskList = [];
    push (@{$offlineTaskList}, @{$self->_createOfflinePrerequisiteTasks($components)});
    push (@{$offlineTaskList}, @{$self->_createComponentPhasedTasks($components, OFFLINE_PHASE)});

    return $offlineTaskList;
}

sub _createConfigureTasklist {
    my ($self, $components) = @_;
    my $taskList = [];
    push (@{$taskList}, @{$self->_createComponentPhasedTasks($components, CONFIGURE_PHASE)});
    push (@{$taskList}, @{$self->_createPostPhaseTaskList(CONFIGURE_PHASE)});

    return $taskList;
}

sub _createOnlineTasklist {
    my ($self, $components) = @_;
    my $onlineTaskList = [];
    push (@{$onlineTaskList}, @{$self->_createOnlinePrerequisiteTasks($components)});
    push (@{$onlineTaskList}, @{$self->_createComponentPhasedTasks($components, ONLINE_PHASE)});
    push (@{$onlineTaskList}, @{$self->_createPostPhaseTaskList(ONLINE_PHASE)});

    return $onlineTaskList;
}

sub _createOnlinePrerequisiteTasks {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $result = [];

    my $instancesToStartBeforeOnlinePhase = $self->_getInstancesToStartBeforeOnlinePhase($components);
    if (@{$instancesToStartBeforeOnlinePhase}) {
        # Start everything but non-db hosts that require instance restart
        push(@{$result}, LCM::Task::StartStopHostsTask::StartSpecificHostsTask->new($configuration, undef, $instancesToStartBeforeOnlinePhase));
    }
    return $result;
}

sub _createSystemOnlineTasklist {
    my ($self, $components) = @_;
    my $systemOnlineTaskList = [];
    push (@{$systemOnlineTaskList}, @{$self->_createSystemOnlinePrerequisiteTasks($components)});
    push (@{$systemOnlineTaskList}, @{$self->_createComponentPhasedTasks($components, SYSTEM_ONLINE_PHASE)});
    push (@{$systemOnlineTaskList}, @{$self->_createPostPhaseTaskList(SYSTEM_ONLINE_PHASE)});

    return $systemOnlineTaskList;
}

sub _createSystemOnlinePrerequisiteTasks {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $result = [];

    my $servicesToRestart = $self->_getServicesToRestart($components);
    if (@{$servicesToRestart}) {
        push(@{$result}, LCM::Task::StartStopHostsTask::EnableSpecificServicesTask->new($configuration, $servicesToRestart));
    }

    my $instancesToRestart = $self->_getInstancesToStart($components);
    if (@{$instancesToRestart}) {
        push(@{$result}, LCM::Task::StartStopHostsTask::StartSpecificHostsTask->new($configuration, undef, $instancesToRestart));
    }

    my $instancesWithServicesToRestart = $self->_getInstancesWithServicesToRestart($components);
    if (@{$instancesWithServicesToRestart} && @{$servicesToRestart}) {
        push(@{$result}, LCM::Task::StartStopHostsTask::StartSpecificServicesTask->new($configuration, $servicesToRestart));
    }

    return $result;
}

sub _createOfflinePrerequisiteTasks {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $servicesToRestart = $self->_getServicesToRestart($components);
    my $result = [];

    if ($self->_isSystemRestartRequired($components)) {
        if (@{$servicesToRestart}) {
            push(@{$result}, LCM::Task::StartStopHostsTask::DisableAndStopSpecificServicesTask->new($configuration, $servicesToRestart, undef));
        }
        push(@{$result}, LCM::Task::StartStopHostsTask::StopSystemTask->new($configuration));
        return $result;
    }

    if (@{$servicesToRestart}) {
        my $instancesWithServicesToRestart = $self->_getInstancesWithServicesToRestart($components);
        my $shouldStop = @{$instancesWithServicesToRestart} ? 1 : 0;
        push(@{$result}, LCM::Task::StartStopHostsTask::DisableAndStopSpecificServicesTask->new($configuration, $servicesToRestart, $shouldStop));
    }

    my $instancesToRestart = $self->_getInstancesToStop($components);
    if (@{$instancesToRestart}) {
        push(@{$result}, LCM::Task::GenericStackTask::DisableAutoFailoverTask->new($configuration));
        push(@{$result}, LCM::Task::StartStopHostsTask::StopSpecificHostsTask->new($configuration, undef, $instancesToRestart));
        $configuration->{disableAutoFailoverTaskAdded} = 1;
    }
    return $result;
}

sub _createComponentPhasedTasks {
    my ($self, $components, $phase) = @_;
    my $configuration = $self->getConfiguration();
    my $tasks = [];

    for my $component (@{$components}) {
        next if $component->isPostAddHostsComponent();
        next if !$component->canBeCalledInPhase($phase, $configuration);

        if ($component->supportsPhase($phase)) {
            push @{$tasks}, LCM::Task::GenericStackTask::ComponentTask::ComponentPhasedTask->new($configuration, $component, $phase);
        } else {
            push @{$tasks}, LCM::Task::GenericStackTask::ComponentTask->new($configuration, $component);
        }
    }

    return $tasks;
}

sub _createPostPhaseTaskList {
    my ($self, $phase) = @_;
    my $configuration = $self->getConfiguration();
    my $generators = $self->getPostPhaseTaskGenerators($phase);
    my $result = [];

    for my $taskGenerator (@{$generators}) {
        push @{$result}, @{$taskGenerator->createTaskList($configuration)};
    }

    return $result;
}

sub _getServicesToRestart {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $servicesToRestart = {};

    for my $component (@{$components}) {
        my $componentRoles = $component->getHostRoles();
        my $hostsWithRoles = $self->_getHostsWithRoles($componentRoles);
        next if !@{$hostsWithRoles};
        next if $component->isComponentFinished($configuration);

        map { $servicesToRestart->{$_}++ } @{$component->getServicesToRestart($configuration)};;
    }
    return [ keys %{$servicesToRestart} ];
}

sub _getInstancesWithServicesToRestart {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $instancesWithServicesToRestart = {};

    for my $component (@$components) {
        my $componentServicesToRestart = $component->getServicesToRestart($configuration);
        if (@{$componentServicesToRestart}) {
            my $componentRoles = $component->getHostRoles();
            my $hostsWithRoles = $self->_getHostsWithRoles($componentRoles);
            map { $instancesWithServicesToRestart->{$_}++ } @{$hostsWithRoles};
        }
    }

    my $instancesToRestart = [];
    my $instancesToStart = $self->_getInstancesToStart($components);
    for my $host (keys %{$instancesWithServicesToRestart}) {
        if (! ($host ~~ @{$instancesToStart})) {
            push(@{$instancesToRestart}, $host);
        }
    }
    return $instancesToRestart;
}

sub _getInstancesToStart {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $dbHosts = $self->_getHostsWithRoles([$gHostRoleWorker, $gHostRoleStandby]);
    my $instancesToRestart = {};

    for my $component (@{$components}) {
        if ($component->isInstanceRestartRequired()) {
            my $componentRoles = $component->getHostRoles();
            my $hostsWithRoles = $self->_getHostsWithRoles($componentRoles);
            map { $instancesToRestart->{$_}++ if (!($_ ~~ @{$dbHosts})) } @{$hostsWithRoles};
        }
    }

    return [ keys %{$instancesToRestart} ];
}

sub _getInstancesToStop {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $instancesToRestart = {};

    for my $component (@{$components}) {
        if ($component->isInstanceRestartRequired()) {
            my $componentRoles = $component->getHostRoles();
            my $hostsWithRoles = $self->_getHostsWithRoles($componentRoles);
            map { $instancesToRestart->{$_}++ } @{$hostsWithRoles};
        }
    }
    return [ keys %{$instancesToRestart} ];
}

sub _isSystemRestartRequired {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    for my $component (@$components) {
        return 1 if ($component->isSystemStopRequired($configuration));
    }
    return 0;
}

sub _recomputePhases {
    my ($self, $components) = @_;
    return undef if (! $self->_forceDependentComponentsInOfflinePhase($components));
    return 1;
}

sub _forceDependentComponentsInOfflinePhase {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $graph = constructComponentDependencyGraph($configuration->getComponentManager(), $components, 0);
    my $offlineComponentsWithDependencies = $self->_getOfflineComponentsWithDependencies($components, $graph);
    my %componentKeynameToComponent = map {$_->getComponentKeyName() => $_} @{$components};

    for my $componentWithDependencies (@{$offlineComponentsWithDependencies}){
        for my $dependentComponentKeyname (@{$graph->{$componentWithDependencies->getComponentKeyName()}}){
            my $dependentComponent = $componentKeynameToComponent{$dependentComponentKeyname};
            if($dependentComponent->canBeCalledInPhase(OFFLINE_PHASE, $configuration)){
                next;
            }
            if(!$dependentComponent->canSetDependencyPhase()){
                $configuration->getMsgLst()->addMessage("Unresolved component dependency. ".
                $componentWithDependencies->getComponentName ." requires " . $dependentComponent->getComponentName() );
                return undef;
            }
            $dependentComponent->setPhase(OFFLINE_PHASE);
        }
    }
    return 1;
}

sub _getOfflineComponentsWithDependencies {
    my ($self, $components, $componentDependencyGraph) = @_;
    my $configuration = $self->getConfiguration();
    my $offlineComponentsWithDependencies = [];

    for my $component (@{$components}) {
        if ($component->canBeCalledInPhase(OFFLINE_PHASE, $configuration) &&
            (scalar(@{$componentDependencyGraph->{$component->getComponentKeyName()}}) > 0)) {
            push @{$offlineComponentsWithDependencies}, $component;
        }
    }

    return $offlineComponentsWithDependencies;
}

1;