package LCM::Task::Hosts::HostsBaseTask;

use strict;
use File::Spec;
use LCM::Component;
use LCM::ProcessExecutor;
use SDB::Install::SAPSystem;
use SAPDB::Install::Hostname;
use LCM::ComponentProgressHandler;
use SDB::Install::Globals qw($gLogDir);
use SDB::Install::System qw(isSidadmin);
use parent qw(LCM::Task::SHASupportedTask);
use SDB::Install::SysVars qw($isWin $path_separator);  
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::Utils::CommonUtils qw(getSpecialOptionString);

# Override
sub _executeInternal {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName());
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $isSuccess = isSidadmin() ? $self->_executeSHAOperation() : $self->_executeAsRoot();
    my $endMessage = $self->getExecutionName() . ($isSuccess ?  ' finished' : ' failed');

    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $message ->endMessage(undef, $endMessage);
    $self->setMsgLstContext($saveContext);

    if (!$isSuccess) {
        $self->getStatus()->_setErrorState();
        $self->setErrorMessage(sprintf('%s failed', $self->getName()), $self->getErrMsgLst());
        return;
    }
    $self->getStatus()->_setFinishedState();
}

sub _executeAsRoot {
    my ($self) = @_;
    my $executor = $self->_createProcessExecutor();
    my $saveCntxt = $self->getMsgLstContext();

    $executor->setMsgLstContext($self->getMsgLstContext());

    my $rc = $executor->executeProgram(1);
    $self->_setLogLocation($self->_parseLogFileLocation($executor->getOutputLines()));
    $self->setMsgLstContext($saveCntxt);

    return 1 if (defined $rc && $rc == 0);

    $self->getErrMsgLst()->addError($self->getFailedExecutionMessage());
    return undef;
}

sub _createProcessExecutor {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $command = $self->_getCommand();
    my $args = $self->_buildArgs();
    my $xmlPasswordStream = $configuration->getXmlPasswordStream();
    my $executor = new LCM::ProcessExecutor( $command, $args, $xmlPasswordStream );

    $executor->setOutputHandler($self->getMsgLst()->getProgressHandler());
    $executor->setProcessEnvironment (LCM::TraceLoggingEnvironment::PrepareProcessExecutorHdbEnvironment($self->_getExecutable()));
    return $executor;
}

sub _executeSHAOperation {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $shaOperation = $self->_getSHAOperation();
    my $passwordKeys = $self->_getPasswordKeys();
    my $remoteHostObject = $self->_createRemoteHostsHostctrlObject();

    $remoteHostObject->setMsgLstContext($self->getMsgLstContext());

    my $rc = 1;
    my $optionsMap = $self->buildSHAOptionsMap();
    my $exitCode = $remoteHostObject->executeHostctrlParallel($shaOperation, $configuration, undef, $passwordKeys, undef, undef, undef, undef, undef, $optionsMap, undef, undef, undef, 0);
    my $outputBuffer = $remoteHostObject->getOutputBuffer();
    my $executionOutput = [ split('\n', $outputBuffer->[0]) ];

    $self->_setLogLocation($self->_parseLogFileLocation($executionOutput));

    if (!defined $exitCode || ($exitCode != 0)) {
        $self->setErrorMessage($self->getFailedExecutionMessage(), $remoteHostObject->getErrMsgLst());
        $rc = undef;
    }
    return $rc;
}

sub _createRemoteHostsHostctrlObject {
    my ($self) = @_;
    my $localHost = hostname();
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts($localHost);

    $localExecutionHost->setOutputHandler($localHost, $progressHandler);
    return $localExecutionHost;
}

sub _getCommand {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $sapmnt = $configuration->getValueOrBatchValue("Target");
    my $sid = $configuration->getValueOrBatchValue("SID");
    my $executable = $self->_getExecutable();

    return File::Spec->catfile($sapmnt, $sid, 'global', 'hdb', 'install', 'bin', $executable);
}

sub _buildCommandLineOptions {
    my ($self, $configuration, $parameterIds) = @_;
    my $args = [];

    for my $parameterId (@{$parameterIds}){
        next if(!$configuration->hasValue($parameterId));
        next if($configuration->isSkipped($parameterId));

        my $option = $configuration->getOpt($parameterId);
        my $value = $configuration->getValue($parameterId);

        if($configuration->getType($parameterId) =~ /bool/){
            $value = ($value =~ /$bool_true_pattern/) ? 'on' : 'off';
        }
        push(@{$args}, sprintf('%s=%s', $option, $value));
    }
    return $args;
}

sub _getExecutable {
    ...
}

sub _buildArgs {
    my ($self) = @_;
    my $args = [ '-b', '--read_password_from_stdin=xml', "--instlog_dir=$gLogDir" ];
    my $configuration = $self->_getConfiguration();
    my $targetConfiguration = $self->_getTargetConfiguration();

    my $ignoreOptionString = getSpecialOptionString($configuration->getOptionIgnore(), $targetConfiguration->getOptionIgnore());
    my $timeoutOptionString = getSpecialOptionString($configuration->getOptionTimeout(), $targetConfiguration->getOptionTimeout());

    my @paramIds = ( keys(%{$self->getParameterToOptionMapping()}) );
    push (@paramIds, "RootUser") if (!$configuration->isUseSAPHostagent());
    my $globalArgs = $self->_buildCommandLineOptions($configuration, \@paramIds);

    for my $optionString ($ignoreOptionString, $timeoutOptionString){
        if(defined($optionString) && $optionString ne ''){
            push(@{$args}, $optionString);
        }
    }

    return [ @{$args}, @{$globalArgs} ];
}

sub _getTargetConfiguration {
    ...
}

sub _getSHAOperation {
    ...
}

sub _getPasswordKeys {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
#_TODO_ Should this be the default behaviour for all tasks?
    my @passwordKeys = grep { $configuration->getType($_) =~ /passwd/ } @{$configuration->getParamIds()};

    return \@passwordKeys;
}

1;