package LCM::Task::SLProcessTask;

use strict;
use parent 'LCM::Task';

use LCM::Task::TaskType;
use LCM::SlAnalytics::SLAMonitor;
use LCM::Task::CLIProcessTaskMetadata;
use SDB::Install::Tools qw(printTableToArrayOfLines formatSecondsIntoHhMmSs);

sub new{
	my ($class, $configuration) = @_;
	my $self = $class->SUPER::new($configuration);
	$self->_setParentTask(undef);
	$self->getMsgLst()->setProgressHandler(new LCM::Task::TaskProgressHandler($self));
	$self->_setCLIMetadata(new LCM::Task::CLIProcessTaskMetadata());
	$self->getStatus()->_setDialogState();
	return $self;
}

sub getType{
	return LCM::Task::TaskType->PROCESS;
}

sub addSubtaskFailedHandler{
	my ($self, $handler) = @_;
	my $handlers = $self->_getSubtaskFailedHandlers();
	push (@{$handlers}, $handler);
}

sub clearAllSubtasks {
	my $self = shift;
	$self->{_subtask} = undef;
}

sub getAllSubtasks{
	my $self = shift();
	if(!defined $self->{_subtask}){
		$self->{_subtask} = [];
	}

	return $self->{_subtask};
}

sub getSubtasks{
	my ($self) = @_;
	my $allSubtasks = $self->getAllSubtasks();
	my @result = grep { !$_->isHidden() } @{$allSubtasks};
	return \@result;
}

sub getSubtaskById{
    my($self,$taskId) = @_;
    my @result = grep {$_->getId() eq $taskId } @{$self->getAllSubtasks()};
    return $result[0];
}

sub addSubtask{
	my ($self, $subtask) = @_;
	$subtask->_setParentTask($self);
	my $subtasks = $self->getAllSubtasks();
	push(@{$subtasks}, $subtask);
}

sub getCLIMetadata{
	return $_[0]->{_cliMetadata};
}

sub _getNumberOfExpectedOutputLines{
	my $self = shift();
	my $totalLines = 0;
	for my $subtask (@{$self->getSubtasks()}){
		$totalLines += $subtask->_getNumberOfExpectedOutputLines();
	}
	
	return $totalLines;
}

sub _setCLIMetadata{
	my ($self, $cliMetadata) = @_;
	$self->{_cliMetadata} = $cliMetadata;
}

sub _executeInternal{
	my $self = shift();
	$self->_startSLAnalyticsMonitor();

	for my $subtask (@{$self->getAllSubtasks()}){
		$self->setCurrentTask($subtask);
		if($subtask->isSkipped()){
			next;
		}
		$subtask->setMsgLstContext($self->getMsgLstContext());
		if(!$subtask->isHidden()){
			$subtask->initProgressHandler();
		}
		$subtask->execute();
		$self->_notifyTaskChanged();
		if($subtask->getStatus()->isInErrorState() && $subtask->isCritical()){
			$self->getStatus()->_setErrorState();
			$subtask->getMsgLst()->addProgressMessage($subtask->getExecutionName()." failed");
			if(!$self->_handleFailedSubtask($subtask)){
				$self->_stopSLAnalyticsMonitor($self->getName().' failed.');
				my $executionSummaryMessages = $self->getExecutionSummaryMessages();
				$self->getMsgLst()->addMessages($executionSummaryMessages) if defined $executionSummaryMessages;
				return;
			}else{
				$subtask->getMsgLst()->addMessage($subtask->getExecutionName()." failure has been ignored");
			}
		}
	}

	my $isInErrorState = $self->getStatus()->isInErrorState();
	my $slaMessage = $isInErrorState ? $self->getName().' failed.' : undef;

	if(! $isInErrorState){
		$self->getStatus()->_setFinishedState();
	}
	$self->_stopSLAnalyticsMonitor($slaMessage);
	my $executionSummaryMessages = $self->getExecutionSummaryMessages();
	$self->getMsgLst()->addMessages($executionSummaryMessages) if defined $executionSummaryMessages;
	$self->getMsgLst()->setProgressHandler(new LCM::Task::TaskProgressHandler($self));
}

sub _getSubtaskFailedHandlers{
	my $self = shift();
	if(!defined $self->{_subtaskFailedHandlers}){
		$self->{_subtaskFailedHandlers} = [];
	}
	return $self->{_subtaskFailedHandlers};
}

sub _handleFailedSubtask{
	my ($self, $failedSubtask) = @_;
	my @applicableHandlers = grep { $_->isApplicable($failedSubtask) } @{$self->_getSubtaskFailedHandlers()};
	my $handlersCount = scalar (@applicableHandlers);
	if($handlersCount == 0){
		return  0;
	}
	my $exitCode = 1;
    for my $handler (@applicableHandlers){
		if( !$handler->handle($failedSubtask) ){
			$exitCode = 0;
		}
	}
	return $exitCode;
}

sub _startSLAnalyticsMonitor{
    my $configuration = $_[0]->_getConfiguration();
    LCM::SlAnalytics::SLAMonitor::start($configuration);
}

sub _stopSLAnalyticsMonitor{
    my ($self, $executionStatusMessage) = @_;
    if($self->getStatus()->isInErrorState()){
    	LCM::SlAnalytics::SLAMonitor::stop(1, $executionStatusMessage, $self->getAllSubtasks());
    } else {
        LCM::SlAnalytics::SLAMonitor::stop(undef, undef, $self->getAllSubtasks());
    }
}

sub getCurrentTask{
	my $self = shift();
	return $self->{_currentTask};
}

sub setCurrentTask {
	my ($self, $currentTask) = @_;
	$self->{_currentTask} = $currentTask;
}

sub indexOf{
	my($self, $taskId) = @_;
	my $subtasks = $self->getAllSubtasks();
	my $index = 0;
	for my $subtask(@{$subtasks}){
		if($subtask->getId() eq $taskId){
			return $index;
		}
		$index++;
	}
	return -1;
}

sub getAllWarningMessages {
    my $self = shift();
     
    my $allWarnings = $self->getWarningMessages(); 
    my $subtasks = $self->getAllSubtasks();
    for my $subtask(@{$subtasks}){
        my $subtaskWarnings = $subtask->getWarningMessages();
        push @$allWarnings, @$subtaskWarnings;              
    }   
    return $allWarnings;       
}

sub setAborted {
	my ($self) = @_;

	$self->getStatus()->_setAbortedState();

	for my $subtask (@{$self->getAllSubtasks()}) {
		$subtask->getStatus()->_setAbortedState();
	}
}

sub getExecutionSummaryMessages {
    my ($self) = @_;
    my $runtimeInfoTable = $self->_getExecutionInfoTable();
    return [] if (scalar(@$runtimeInfoTable) == 0);
    my $indexTable = printTableToArrayOfLines($runtimeInfoTable, ' | ', 1);
    my @messages = ("Execution Summary:", '', @$indexTable, '');
    return \@messages;
}

sub _getExecutionInfoTable {
    my ($self) = @_;
    my @table = ();

    my $subtasks = $self->getAllSubtasks();
    my $totalRuntime = 0;
    for my $subtask(@{$subtasks}){
        next if $subtask->isHidden();
        $totalRuntime += $subtask->getRuntime() // 0;
        my $runtime = $subtask->getRuntimeInHhMmSs();
        push @table, [$subtask->getExecutionName(), $runtime];
    }
    return [] if (scalar(@table) == 0);

    push @table, ['', '']; # Empty row before the 'Total Runtime' row
    push @table, ['Total execution time', formatSecondsIntoHhMmSs($totalRuntime)];

    unshift @table, ['Step Name', 'Runtime'];
    return \@table;
}

1;
