package LCM::Task::UninstallationProcessTask;

use strict;
use parent 'LCM::Task::GenericStackProcessTask';
use SDB::Install::Globals qw($gHostRoleXS2Worker);
use LCM::Task::CommonTask::RemoveLocalRolesTask;
use LCM::Task::CommonTask::RemoveRemoteRolesTask;
use LCM::Task::CommonTask::RemoveRemoteXsControllerRolesTask;
use LCM::Task::CommonTask::RemoveRemoteHostsTask;
use LCM::Task::CommonTask::UpdateLocalHostTask;
use LCM::Task::CommonTask::UpdateRemoteHostsTask;
use LCM::Task::CommonTask::UpdateComponentListTask;
use LCM::Utils::RemoveHosts qw(IsRemovingRemoteHost IsRemovingLocalHost);
use LCM::Utils::RemoveRoles qw(IsRemovingLocalRole IsRemovingRemoteRole IsRemovingRoleFromHost);
use LCM::Task::GenericStackTask::ComponentTask::UninstallTask;
use LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager;
use LCM::Task::Cockpit::XSALogoutHandler;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::SkippedComponentsManager qw ( $SCENARIO_UNINSTALL );
use LCM::Utils::ComponentsDependencySort qw(sortComponentsTopologically);
use LCM::Task qw($gFlavourProductName $gActionScope);
use SDB::Install::Globals qw ($gHostRoleWorker $gHostRoleStandby $gSecureStoreTypeSSFS);
use LCM::Task::GenericStackTask::MigrateSecureStoreTask;
use LCM::Task::StartStopHostsTask::StopSystemTask;
use LCM::Task::StartStopHostsTask::StartSpecificHostsTask;

my $UNINSTALL_SUCCESS_END_MESSAGE = "Uninstallation of the $gFlavourProductName $gActionScope finished";
my $UNINSTALL_FAIL_END_MESSAGE    = "Uninstallation of the $gFlavourProductName $gActionScope failed";
my $UNINSTALL_ABORT_END_MESSAGE   = "Uninstallation of the $gFlavourProductName $gActionScope aborted";

my $UNINSTALL_NAME = "Uninstall $gFlavourProductName $gActionScope";
my $PROGRESS_MESSAGE_TEMPLATE = "Uninstalling $gFlavourProductName $gActionScope";

# Override
sub getId {
	my $self = shift;
	return $self->_getConfiguration()->getAction();
}

# Override
sub getName {
    return $_[0]->_formatMessage($UNINSTALL_NAME);
}

# Override
sub isUpdate {
	return 0;
}

# Override
sub getComponentManager {
	my $self = shift();
	return $self->_getConfiguration()->getSystemComponentManager();
}

#
# Only protected/private subs bellow this line
#

# Override
sub _shallAddDeployXSAComponentsTask { return 0; }

# Override
sub _executeInternal {
	my $self = shift();
	my $configuration = $self->_getConfiguration ();

	$self->_initActionMessages ();
	$self->_addProgressMsg ();
	$self->SUPER::_executeInternal ();
}

# Override
sub _createSubtasks {
	my $self = shift();
	my $configuration = $self->_getConfiguration ();
	my $componentManager = $self->getComponentManager();
	my $installedComponents = $componentManager->getAllComponents();
	my $selectedComponents = $self->_getSortedComponentList();
	my $localComponentsManager = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager($selectedComponents, $installedComponents, $SCENARIO_UNINSTALL);
	my $remoteComponentsManager = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager($selectedComponents, $installedComponents,$SCENARIO_UNINSTALL);

	if($localComponentsManager->existRegistrableComponents()){
		$self->addSubtask(new LCM::Task::CommonTask::UpdateLocalHostTask($self->_createLocalHostConfiguration($localComponentsManager)));
	}
	if($configuration->isDistributedSystem() && $remoteComponentsManager->existRegistrableComponents()){
		$self->addSubtask(new LCM::Task::CommonTask::UpdateRemoteHostsTask($self->_createRemoteHostConfiguration($remoteComponentsManager)));
	}
	$self->_addRemoveHostsAndRolesSubtasks();

	my $isServerSelected = $self->_isServerSelected();
	my @serverPlugins = grep {$_->isServerPlugin()} @$selectedComponents;
	if (!$isServerSelected && scalar(@serverPlugins)) {
		$serverPlugins[-1]->setRequireSystemRestart(1);
	}
	for my $selectedComponent (@{$selectedComponents}) {
# If we are uninstalling the whole system we should handle
# non-server components' failure to uninstall as a warning
		my $isFailureCritical = $isServerSelected ? ($selectedComponent->isServer() ? 1 : 0) : 1;
		if(!$isServerSelected && $selectedComponent->isLSS()){
			$self->addSubtask(LCM::Task::StartStopHostsTask::StopSystemTask->new($configuration));
			$self->addSubtask(LCM::Task::GenericStackTask::MigrateSecureStoreTask->new($configuration, $gSecureStoreTypeSSFS));
		}
		$self->addSubtask(new LCM::Task::GenericStackTask::ComponentTask::UninstallTask($configuration, $selectedComponent, $isFailureCritical));
		if(!$isServerSelected && $selectedComponent->isLSS()){
			$self->addSubtask(LCM::Task::StartStopHostsTask::StartSpecificHostsTask->new($configuration, undef, $configuration->getSystemHosts()));
		}
		$self->_addPostXSAUninstallationUpdateHostTasks() if !$isServerSelected && $selectedComponent->isXS2();

		if ($selectedComponent->isCockpitStack()) {
            $self->addSubtaskFailedHandler(LCM::Task::Cockpit::XSALogoutHandler->new());
		}
	}

	$self->addSubtask(LCM::Task::CommonTask::UpdateComponentListTask->new($configuration)) if !$isServerSelected;
}

# after uninstalling XSA component we need to call update_host on
# all remote database hosts to delete any existing XSA Users
sub _addPostXSAUninstallationUpdateHostTasks {
	my ($self) = @_;
	my $configuration = $self->_getConfiguration();
	my $ownInstance   = $configuration->getOwnInstance();
	my $allDatabaseHosts = $ownInstance->getHostsByRole($gHostRoleWorker, $gHostRoleStandby);
	return if (!defined $allDatabaseHosts);
	my $localHost = $ownInstance->get_host();
	if (!IsRemovingLocalHost($configuration) && grep {$_ eq $localHost} @$allDatabaseHosts) {
		$self->addSubtask(new LCM::Task::CommonTask::UpdateLocalHostTask($self->_createPostXSUninstallLocalHostConfiguration()));
	}
	if (!$configuration->isDistributedSystem()) {
		return;
	}
	my $remoteHostsObjBeforeUninst = $configuration->getRemoteHosts();
	if (!defined $remoteHostsObjBeforeUninst) {
		return;
	}

	my @nonDatabaseRemoteHosts = ();
	for my $remoteHostname (@{$remoteHostsObjBeforeUninst->getHostNames()}) {
		if (! grep {$_ eq $remoteHostname} @$allDatabaseHosts) {
			push(@nonDatabaseRemoteHosts, $remoteHostname);
		}
	}
	if (scalar(@nonDatabaseRemoteHosts) == scalar(@{$remoteHostsObjBeforeUninst->getHostNames()})) {
		return;
	}
# Create a new RemoteHosts object containing only remote database hosts
# which should never be removed at this point because this method is only
# reached if you're not uninstalling the server
	$configuration->clearRemoteHosts();
	$configuration->initRemoteHosts();
	my $remoteDbHostsObjAfterUninst = $configuration->getRemoteHosts();
	$remoteDbHostsObjAfterUninst->copyConnections($remoteHostsObjBeforeUninst);
	$remoteDbHostsObjAfterUninst->removeRemoteHosts(\@nonDatabaseRemoteHosts);

	$self->addSubtask(new LCM::Task::CommonTask::UpdateRemoteHostsTask($self->_createPostXSUninstallRemoteHostConfiguration($remoteDbHostsObjAfterUninst)));
# Restore original value
	$configuration->setRemoteHosts($remoteHostsObjBeforeUninst);
}

sub _addRemoveHostsAndRolesSubtasks {
	my ($self) = @_;
	my $configuration = $self->_getConfiguration ();
	my $ownInstance = $configuration->getOwnInstance();
	my $xsControllerHost = $ownInstance->getXsControllerHostname();
	my $isLocalXsController = defined($xsControllerHost) && ($xsControllerHost eq $ownInstance->get_host());
	my $removeLocalXsControllerLast = $isLocalXsController && IsRemovingRoleFromHost($configuration, $xsControllerHost, $gHostRoleXS2Worker);
	my $removeRemoteXsControllerLast = $configuration->shouldRemoveXsControllerHostRolesSeparately();

	if(IsRemovingLocalRole($configuration) && !$removeLocalXsControllerLast){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveLocalRolesTask($configuration));
	}
	if(IsRemovingRemoteRole($configuration)){
		#if $removeRemoteXsControllerLast is true, a the removal of roles from the xs controller host will be skipped 
		$self->addSubtask(new LCM::Task::CommonTask::RemoveRemoteRolesTask($configuration));
	}
	if(IsRemovingRemoteHost($configuration)){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveRemoteHostsTask($configuration));
	}
	if(IsRemovingLocalRole($configuration) && $removeLocalXsControllerLast){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveLocalRolesTask($configuration));
	}
	if($removeRemoteXsControllerLast){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveRemoteXsControllerRolesTask($configuration));
	}
}

# Override
sub _shallAssignAdditionalLocalRoles { return 0; }

# Override
sub _shallAssignAdditionalRemoteRoles { return 0; }

# Override
sub _initActionMessages {
	my ( $self ) = @_;
	my $status = $self->getStatus();
	$status->_setActionDoneWithWarningsMessage( "" );
	$status->_setActionDoneMessage($self->_formatMessage($UNINSTALL_SUCCESS_END_MESSAGE));
	$status->_setActionFailedMessage($self->_formatMessage($UNINSTALL_FAIL_END_MESSAGE));
	$status->_setActionAbortedMessage($self->_formatMessage($UNINSTALL_ABORT_END_MESSAGE));
}

sub _formatMessage {
    my ( $self, $message ) = @_;
    my $isServerUninstallation = $self->_isServerSelected();
    my $actionscope = $self->_isServerSelected() ? 'System' : 'Components';
    $message =~ s/$gActionScope/$actionscope/g;
    return $self->SUPER::_formatMessage($message);
}

sub _isInstallingComponent {
    return 0;
}

# Override
sub _addProgressMsg {
	$_[0]->getMsgLst()->addProgressMessage($_[0]->_formatMessage($PROGRESS_MESSAGE_TEMPLATE));
}

sub _isServerSelected {
	my ($self) = @_;
	my $componentManager = $self->getComponentManager();
	return $componentManager->isHDBServerComponentSelected();
}

sub _getSortedComponentList {
	my ($self) = @_;
	my $componentManager = $self->getComponentManager();
	my $selectedComponents = $componentManager->getSelectedComponents();
	my $isServerSelected = $componentManager->isHDBServerComponentSelected();

	return $selectedComponents if($isServerSelected);
	return sortComponentsTopologically($componentManager, $selectedComponents, 1);
}

sub _createLocalHostConfiguration {
	my($self, $componentsManager) = @_;
	my $configuration = $self->_getConfiguration ();
	$componentsManager->detectComponents();
	my $skippedComponents = $componentsManager->getSkippedComponents();
	my $paramHash = {
		ComponentsRegistration => 'unregister',
		SkippedComponents => $skippedComponents,
		GenerateCertificates => 0,
		DeployShaConfigurations => 0,
		UpdateComponentListLocation => 0,
	};
	return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createLocalConfiguration($configuration, $paramHash);
}

sub _createRemoteHostConfiguration {
	my($self, $componentsManager) = @_;
	my $configuration = $self->_getConfiguration();
	$componentsManager->detectComponents();
	my $skippedComponents = $componentsManager->getSkippedComponents();
	my $remoteHostsObjects = defined($configuration->getRemoteHosts()) ? [ $configuration->getRemoteHosts() ] : [];

	my $paramHash = {
        ComponentsRegistration => 'unregister',
        SkippedComponents => $skippedComponents,
        GenerateCertificates => 0,
        DeployShaConfigurations => 0,
        UpdateComponentListLocation => 0,
    };
	return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createRemoteConfiguration($configuration, $remoteHostsObjects, $paramHash);
}

sub _createPostXSUninstallLocalHostConfiguration {
	my($self) = @_;
	my $configuration = $self->_getConfiguration ();
	my $paramHash = {
		ComponentsRegistration => undef,
		SkippedComponents => undef,
		GenerateCertificates => 0,
		DeployShaConfigurations => 0,
		UpdateComponentListLocation => 0,
	};
	$self->_addXSASpecificUpdateHostParams($paramHash);
	return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createLocalConfiguration($configuration, $paramHash);
}

sub _createPostXSUninstallRemoteHostConfiguration {
	my ($self, $remoteDbHostsObjAfterUninst) = @_;
	my $configuration = $self->_getConfiguration ();
	my $paramHash = {
		ComponentsRegistration => undef,
		SkippedComponents => undef,
		GenerateCertificates => 0,
		DeployShaConfigurations => 0,
		UpdateComponentListLocation => 0,
	};
	$self->_addXSASpecificUpdateHostParams($paramHash);
	return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createRemoteConfiguration($configuration, [$remoteDbHostsObjAfterUninst], $paramHash);
}

sub _addXSASpecificUpdateHostParams {
	my ($self, $paramHash) = @_;
	$self->SUPER::_addXSASpecificUpdateHostParams($paramHash);
# Adds XsEaDataPath here because it is deleted from cfg file after XSA uninstallation
	my $configuration = $self->_getConfiguration ();
	my $ownInstance = $configuration->getOwnInstance();
	my $eaDataPath = $ownInstance->getXSEADataPath(1);
	if (defined $eaDataPath && $eaDataPath) {
		$paramHash->{XsEaDataPath} = $eaDataPath;
	}
}

1;
