
#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc:  database upgrade application class

package SDB::Install::App::Console::DBUpgrade;

#
# configuration parameters 
#   - installation path or installation key
#   - mode (system/user/portable)
#   - kit directory
#   - os user 
#   - os group
#   - plain text logfile 


# ways of parameter initialization: implicit (upgrade), per command line option, per configfile, per default, interactive 

use SDB::Install::App::Console;
use SDB::Install::App::Installation;
use SDB::Install::Kit;
use Getopt::Long;
use SDB::Install::Configuration::Upgrade;
use SDB::Install::SAPSystem;
use SDB::Install::Tools qw (askConfirmation printTableToArrayOfLines);
use SDB::Install::System qw (isPASE isAdmin);
use SDB::Install::Globals qw($gPhaseOnlineOption $gProductNameEngine $gProductNameInstaller);
use SDB::Install::SysVars qw ($isWin);

use strict;


our @ISA = qw (SDB::Install::App::Console SDB::Install::App::Installation);

#---------------------------------------------------------------------------------------------------------------

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

#---------------------------------------------------------------------------------------------------------------

sub init{
    my ($self) = @_;

    if (!defined $self->SDB::Install::App::Installation::init()){
        return undef;
    }
        
    if (defined $self->{return}){
        return 1;
    }

    if (!$self->{'kit'}->HasDBKernel){
        $self->setErrorMessage ("installation kit provides no $gProductNameEngine kernel");
        return undef;
    }

    return 1;
}

#---------------------------------------------------------------------------------------------------------------

sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}

#---------------------------------------------------------------------------------------------------------------

sub InitCmdLineArgs{
    my ($self,$args ) = @_;
    
    $self->{options} = {};
    
    my $rc = $self->SUPER::InitCmdLineArgs ($args,1);

    if (!defined $rc){
        return undef;
    }
   $rc = $self->ParseInstallationCmdLineArgs ($rc);
    if (!defined $rc){
        return undef;
    }

    local @ARGV = @$rc;

    my $instconfig = new SDB::Install::Configuration::Upgrade ($self->{options},$self->{configfile});

    $instconfig->getMsgLst()->setProgressHandler ($self->getMsgLst ()->getProgressHandler ());

    my $instopt_ctrl = $instconfig->GetOptionCtrl();
    
    $self->{instconfig} = $instconfig;

    if ($self->isHelp()) {
        $self->PrintUsage();
        $self->{return} = 0;
        return undef;
    }


    $rc = $self->ParseConfigCmdLineArgs ($rc, 1);
    if (!defined $rc){
        return undef;
    }

    @ARGV = @$rc;


    my %upgradeopt_ctrl = (
        'force_downgrade' => \$self->{options}->{force_downgrade}
    );


    if (! defined $self->getOptions (\%upgradeopt_ctrl, 0)){
        $self->{return} = -1;
        return 0;
    }

    return $rc;
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing help/info options of the program.

sub GetHelpSwitches {
    my ($self, $indent) = @_;
    return [@{$self->SUPER::GetHelpSwitches($indent)}, '--list_packages'];
}

#---------------------------------------------------------------------------------------------------------------
# Returns a reference to an array containing hdbupd options
# without help/info options.

sub GetSwitches{

    return [@{$_[0]->SUPER::GetSwitches()},
            '--check_files',
            '--no_debug_packages'];
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of program options without common batch/help options.

sub GetUsage{

    return [@{$_[0]->SUPER::GetUsage()},
            ['--check_files',       undef, 'Checks files, if same package is already installed'],
            ['--list_packages',     '-l',  'Shows available features/packages'],
            ['--no_debug_packages', undef, 'Suppresses the installation of debug packages']];
}

#---------------------------------------------------------------------------------------------------------------

sub configure {
    my ($self) = @_;

    if ($isWin){
        if(!isAdmin ()){
            $self->setErrorMessage ('Start upgrade as local administrator!');
                                #( isPASE() ? 'Please restart upgrade as user QSECOFR!' :
                                #            'Please restart upgrade as user root!'));
            return undef;
        }
    }
    else{
        if ($>){
            my $username = getpwuid ($>);
            if ($username !~ /^[a-z][a-z0-9]{2}adm$/){
                $self->setErrorMessage ("Wrong user ($username). It's recommended to use the root user. But you can use the <sid>adm user as well.");
                return undef;
            }
        }
    }

    my $instconfig = $self->{instconfig};
    my $msglst = $self->getMsgLst();
    $self->warnIfShouldWarnIfCalledStandalone();
    my $caption = "$gProductNameInstaller - " . $instconfig->getShortProductName () . " Upgrade " . $instconfig->getProductVersion ();
    $msglst->addProgressMessage ("\n\n".$caption . "\n" . ('*' x length ($caption)) . "\n\n");
    $self->addStartDateMsg($msglst, $self->getProgramName());
    
    my $selected = $self->SelectSoftwareComponentsByCmdLineArguments ();
    
    if (!defined $selected){
        return undef;
    }
    
    if (!$selected){
        $self->{auto_selectall} = 1;
        $self->{kit}->SelectAllPackages ();
    }
    

    my $msg = $msglst->addMessage ("Checking configuration...");
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst()]);
    my $rc = $instconfig->CheckParams ($self->{batch_mode});

    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();
    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    }

    if($self->{batch_mode}){
        if (!$rc){
            if (defined $rc){
                $self->setErrorMessage ("running in batch mode", $instconfig->getErrMsgLst());
                #$self->PrintUsage ();
            }
            else{
                $self->getErrMsgLst ()->setMsgLst ($instconfig->getErrMsgLst());
            }
            return undef;
        }
    }
    else{
        if ($instconfig->ErrorState ()){
            $self->getErrMsgLst ()->setMsgLst ($instconfig->getErrMsgLst());
            return undef;
        }
        my $saveMsgLst = $self->getMsgLst();
        my $icfgMsg = $msg->getSubMsgLst()->addMessage ('Configuring interactively');
        my $saveContext = $self->setMsgLstContext ([$icfgMsg->getSubMsgLst()]);
        if (!defined $self->ConfigureInteractive ($instconfig)){
            $msg->endMessage ();
            $self->setMsgLstContext ($saveContext);
            return undef;
        }
        $self->setMsgLstContext ($saveContext);
    }

    if ($instconfig->isCheckOnly()) {
        $self->getMsgLst()->addProgressMessage ("Database Upgrade: check only");
    }

    if ($instconfig->isa('SDB::Install::Configuration::Upgrade')) {
        $instconfig->displayParameterSummary();
        $self->printWarnings ();
        if (!$self->{batch_mode} && !$instconfig->getValue('CheckOnly')
                       && !$self->{options}->{noprompt} && !askConfirmation()) {
            $self->{msglst}->addProgressMessage('Upgrade cancelled');
            $self->{return} = 0;
            return 1;
        }
    }

    return 1;

}

#---------------------------------------------------------------------------------------------------------------

sub ConfigureInteractive{
    my ($self, $config) = @_;


    my $param = $config->{params}->{SID};
    my $msglst = $self->getMsgLst ();
    my $errlst = $self->getErrMsgLst ();
    if (!$config->isInstallationSelected ()){

        my $systems = $config->getCollectSAPSystems();

        SDB::Install::SAPSystem::dumpSapSystems (undef, $msglst);

        my $id = 0;
        my @tab;
        my $system;
        
        my ($proper, $nonproper) = $config->getProperSIDs (1);
        
        
        if (@$nonproper){
        
            $msglst->addProgressMessage ("Systems which can't be upgraded:\n");
        
            foreach my $line (@{printTableToArrayOfLines ($nonproper,
                                                          '   ', # col separator
                                                          undef,
                                                          undef,
                                                          '   ', # table indent
                                                          1      # space between rows
                                                         )}){
                $msglst->addProgressMessage ($line);
            }
            print "\n\n";
        }
        
        push @tab, ['No', 'System', 'Properties'];
        
        foreach my $sid (@$proper){
            push @tab, [$id++, $sid->[0], $sid->[1]];
        }
        if (@tab == 1){
            $errlst->addError ("No proper $gProductNameEngine installation found");
            return undef;
        }
        
        push @tab, [$id, 'None', '(Abort upgrade)']; 
        
        $msglst->addProgressMessage ("Select a $gProductNameEngine installation:");
        
        print "\n";
        
        foreach my $line (@{printTableToArrayOfLines (\@tab,
                                                      ' | ', # col separator
                                                      1,     # first row is header
                                                      undef,
                                                      undef,
                                                      1      # space between rows
                                                     )}){
            $msglst->addProgressMessage ($line);
        }
        print "\n";
        
        my $result;
        for (;;){
            print 'Specify the sequence number of the system to be upgraded ['.$id.']: ';
            $result = <STDIN>;
            chomp $result;
            if ($result !~ /\S/){
                $result = $id;
            }
            if ($result !~ /\D/ and $result =~ /\d/ and int ($result) <= $id){
                $result = int $result;
                last;
            }
        }
        
        if ($result == $id){
            $msglst->addMessage ("User selected '$tab[$result + 1]->[2]'");
            die ('__SDB_ABORT__');
        }

        my $selectedSID = $tab[$result + 1]->[1];
        my $sapSys      = $systems->{$selectedSID};
        if (defined $sapSys) {
            $config->{params}->{Target}->{value} = $sapSys->get_target();
        }
        my $msg = $msglst->addMessage ("User selected '$selectedSID'");
       $config->setMsgLstContext ([$msg->getSubMsgLst()]);
        if (!$config->setValue('SID', $selectedSID)){
            $errlst->addError (undef, $config->getErrMsgLst ());
            return undef;
        }
    }
    return $self->SUPER::ConfigureInteractive ($config);
}

#---------------------------------------------------------------------------------------------------------------

sub preinstall{
    my ($self, $msglst) = @_;
        
    #
    #   database check
    #

    my $packageManagerCollection = $self->{'packageManagerCollection'} =
        new SDB::Install::PackageManagerCollection ();

    my $inst = $self->{instconfig}->getSAPSystem();

    $self->{sapSystem} = $inst;
    $self->{sapSystem}->setConfiguration($self->{instconfig});
    
    if ($inst->ErrorState){
        $self->setErrorMessage ('software installation defect', $inst->getErrMsgLst());
        return undef;
    }

    if ($inst->RegistryFileExists ()){
        $inst->setMsgLstContext([$msglst]);
        $inst->GenPackageList ();
        if ($inst->ErrorState){
            $self->setErrorMessage ('software installation defect', $inst->getErrMsgLst());
            return undef;
        }
    }
    $self->{kit}->setMsgLstContext([$msglst]);
    if (!defined $self->{kit}->GenerateTree ($inst)){
        $self->setErrorMessage ('error in installation kit', $self->{kit}->getErrMsgLst ());
        return undef;
    }

    if ($self->{auto_selectall}){
        $self->{kit}->SelectAllPackages ();
    }
    else{
        $self->SelectSoftwareComponentsByCmdLineArguments ();
    }
    
    $packageManagerCollection->addPackageManagerObject('kit', # key
        $self->{kit}, # value
        $inst);

    if ($self->{instconfig}->isa ('SDB::Install::Configuration::Upgrade')){
        $inst->setMsgLstContext($self->getMsgLstContext());
        if (!$inst->checkUpdate ($self->{instconfig})){
            $self->setErrorMessage ("Checking upgrade failed", $inst->getErrMsgLst());
            return undef;
        }
        $self->{checkOnly} = $self->{instconfig}->isCheckOnly();
    }

    #
    #   software check
    #
    $self->SUPER::preinstall($self->{checkOnly}, $msglst);
}

#---------------------------------------------------------------------------------------------------------------

sub update{
    my ($self) = @_;

    my $inst = $self->{sapSystem};

    my $instconfig = $self->{instconfig};

    my $step = int $instconfig->getStep ();
    my $allpackages = $self->{kit}->GetPackages();
    my $msglst = $self->getMsgLst();
    my $phase = $instconfig->getValue('Phase');
    if (!$step && (!defined $allpackages || !@$allpackages)){
        $msglst->addProgressMessage ("The $gProductNameEngine instance of system '" . $instconfig->getValue ('SID') . "' is up-to-date.");
        if (defined $phase && $phase eq $gPhaseOnlineOption && $instconfig->getIgnore ('check_pending_upgrade')){
            $instconfig->setMsgLstContext ([$msglst]);
            $instconfig->pers_remove ();
        }
        return 1;
    }
    $inst->setMsgLstContext ([$msglst]);

    if ($self->{checkOnly}) {

        foreach my $currPackage (@$allpackages){
            $msglst->addProgressMessage('Package checked for update: '
                                                     . $currPackage->GetName());
        }

        return 1;
    }

    if (!defined $phase || $phase ne $gPhaseOnlineOption){
        if (!defined $inst->OpenExclusive ()){
            $self->setErrorMessage ('Cannot open installation exclusively', $inst->getErrMsgLst());
            return undef;
        }

        $instconfig->addPhaseToPersistenceFile( $phase ) if defined $phase;
        if (!defined $inst->update($instconfig->getValue ('Password'),
            $self->{kit}, $instconfig)){
            $self->setErrorMessage (undef, $inst->getErrMsgLst ());
            return undef;
        }
    }

    my $rc = $inst->restart_after_upgrade ($instconfig);

    if (!defined $rc){
        $self->setErrorMessage (undef, $inst->getErrMsgLst());
        return undef;
    }

    if (!$rc){
        # import error => set exit code to 2
        $msglst->addProgressMessage ('');
        $msglst->addProgressMessage ("  ##############################################################");
        $msglst->addProgressMessage ("  # Problems occurred while importing delivery units (content).");
        $msglst->addProgressMessage ("  # To complete the content update, refer to SAP Note 1795885.");
        $msglst->addProgressMessage ("  #############################################################");
        $msglst->addProgressMessage ('');
        $self->{return} = 2;
    }

    my $nextPhase = $instconfig->getNextPhaseFromCurrent();
    if (defined $nextPhase && $nextPhase eq 'online' && $instconfig->getValue('NoStart')) {
        return 1;
    }
    $instconfig->addPhaseToPersistenceFile($nextPhase) if defined($nextPhase);
    return 1;
}

#---------------------------------------------------------------------------------------------------------------

sub CleanUp{
    my ($self) = @_;
    return $self->SDB::Install::App::Installation::CleanUp($self->{checkOnly});
}

#---------------------------------------------------------------------------------------------------------------

sub main{
    my $app = new __PACKAGE__;
    
    my $rc = $app->InitCmdLineArgs (\@_);

    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }

    if (!defined $rc){
        $app->ShowErrorMsg ('Upgrade failed',$app->getErrMsgLst());
        $app->CleanUp();
        undef $app;
        return 1;
    }


        
    $rc = $app->init ();

    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }
    my $msglst = $app->getMsgLst ();
    if (!defined $rc || !defined $app->checkSystemRequirements ()){
        $app->ShowErrorMsg ('Upgrade failed',$app->getErrMsgLst());
        $app->CleanUp();
        undef $app;
        return 1;
    }
    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval{

        $rc = $app->configure ();
        if (defined $app->{return}) {
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            return $app->{return};
        }

        if (defined $rc){
            my $msg = $msglst->addProgressMessage ("Checking upgrade...");
            my $saveContext = $app->setMsgLstContext ([$msg->getSubMsgLst ()]);
            $rc =   $app->preinstall ();
            $app->setMsgLstContext ([$saveContext->[0]]);
            if (defined $rc){
                $rc = $app->update ();
                if (!defined $rc){
                    $app->setErrorMessage ('error upgrading',
                                                      $app->getErrMsgLst());
                }
            }
            else{
                $app->setErrorMessage ('Error checking Upgrade',
                                                     $app->getErrMsgLst());
            }
        }

    };

    if ($@){
        my $signalInfoText;
        if ($@ =~ /^__SIGINT__/){
            $msglst->addMessage ('User canceled upgrade with Ctrl + c');
            $rc = 2;
        }
        elsif ($@ =~ /^__SDB_ABORT__/){
            $msglst->addMessage ('User canceled upgrade');
            $rc = 2;
        }
        elsif ($@ =~ /^__SIGPIPE__/){
            $app->setErrorMessage ('Broken pipe', $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        elsif ($@ =~ /^__SIG(\w+)__/){
            $app->setErrorMessage ("Caught signal $1", $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        else{
            $app->setErrorMessage ('unhandled exception: '. $@, $app->{stackBacktraceMsglst});
            $rc = undef;
        }
        if ($signalInfoText){
            $app->appendErrorMessage ($signalInfoText);
        }
    }

    if (defined $rc){
        if ($rc == 2){
            print "\n";
            $msglst->addProgressMessage('Upgrade aborted');
        }
        elsif ($app->{checkOnly}) {
            $msglst->addProgressMessage('Upgrade check done');
        }
        elsif ($app->{instconfig}->getValue ('PrepareUpgrade')) {
            $msglst->addProgressMessage('Upgrade prepared, resumable');
        }
        elsif (my $phase = $app->{instconfig}->getValue ('Phase')) {
            if($phase eq $gPhaseOnlineOption) {
                $msglst->addProgressMessage('Upgrade  done');
            }
            else {
                $msglst->addProgressMessage("Upgrade partially done, up to and including phase $phase, resumable");
            }
        }
        elsif (!defined $app->{return}) {
            $msglst->addProgressMessage('Upgrade done');
        }

    }
    else{
        $app->ShowErrorMsg ('Upgrade failed',$app->getErrMsgLst());
    }

    $app->CleanUp();

    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }

    undef $app;

    return defined $rc ? 0 : 1;
}

sub shouldWarnIfCalledStandalone{
    return 1;
}


#---------------------------------------------------------------------------------------------------------------

1;
