package SDB::Install::App::Installation;

use strict;

use SAPDB::Install::Hostname;
use SDB::Install::Configuration::NewDB;
use SDB::Install::Configuration::Upgrade;
use SDB::Install::Installation qw (SDB_O_CREAT);
use SDB::Install::Kit;
use Getopt::Long;
use SDB::Install::SysVars;
use SDB::Install::PackageManagerCollection;
use SDB::Install::Tools qw (printTableToArrayOfLines);
use SDB::Install::SAPSystem;
use SDB::Install::Globals;
use SDB::Install::Installation::Client;
use SDB::Install::Installation::DatabaseStudio;
use SDB::Install::IniFile;
use SDB::Install::Log;
use SDB::Install::DebugUtilities;
use SDB::Install::System;
use SDB::Install::App::AnyApp;

our @ISA = qw (SDB::Install::App::AnyApp);
#----------------------------------------------------------------------------



sub init{
    my ($self) = @_;

    if (defined $self->{kit}){
        return 1;
    }

    my $kit_dir = $self->{'options'}->{'archive_dir'};
    
    my $installer_dir = $self->GetInstaller ()->getInstallerDir ();

    if (defined $kit_dir){
        if ($isWin){
            if ($kit_dir !~ /^[\/\\]|^[a-z]:/i){
                $kit_dir = $installer_dir .
                    $path_separator . $kit_dir;
            }
        }
        else{
            if ($kit_dir !~ /^\//){
                $kit_dir = $installer_dir .
                    $path_separator . $kit_dir;
            }
        }
    }
    else{
        $kit_dir = $installer_dir;
        my $tdir;
        foreach my $dir ('server','client','studio','lm','excel','packages'){
            $tdir = $kit_dir . $path_separator . $dir;
            if (-d $tdir){
                $kit_dir = $tdir;
                last;
            }
        }
    }
    my $instKitMsg = $self->getMsgLst ()->addMessage ("Detecting installation kit");
    $self->{'kit'} = new SDB::Install::Kit ($kit_dir, $self->GetSysInfo());
    $instKitMsg->getSubMsgLst()->appendMsgLst ($self->{kit}->getMsgLst());

    $self->{'packageManagerCollection'} = new SDB::Install::PackageManagerCollection();
    $self->{'packageManagerCollection'}->getMsgLst ()->setProgressHandler ($self->getMsgLst ()->getProgressHandler ());
    $self->{'packageManagerCollection'}->setApplicationOptions($self->{'options'});

    my $saveContext = $self->setMsgLstContext ([$instKitMsg->getSubMsgLst()]);
    if (!defined $self->initInstConfig()){
        $self->setMsgLstContext ($saveContext);
        return undef;
    }
    $self->setMsgLstContext ($saveContext);
    if (defined $self->{options}->{no_debug_packages}){
        $self->{kit}->EnableDebugPackages (0);
    }

    #
    # Parse cmdln args here only to get the ignore values.
    # We have to know whether check_platform is set.
    # Parsing is done later again incl. error handling.
    #
    my @args = @ARGV;
    $self->ParseConfigCmdLineArgs (\@args,1);

    if ($self->{options}->{list_packages}){
        my $packages = $self->{'kit'}->GenPackageList($self->getInstconfig ());
            
        my @table;
        my $info = "\nPossible Package(s):\n\n";
        my $mode;
        foreach my $package (sort {$a->GetName() cmp $b->GetName()} values %$packages){
            $mode = $package->GetObjectMode;
            push @table , [$package->GetName, $package->GetVersion, $mode ? "$mode bit" : '']; 
        }
        
        $info .= join ("\n", @{printTableToArrayOfLines (\@table, '    ')}) . "\n";
        $self->PrintText($info, 'List Packages');
        $self->{return} = 0;
        return 1;
    }


    if ($self->isHelp()) {
        $self->PrintUsage();
        $self->{return} = 0;
        return 1;
    }

    if (defined $self->{return} && $self->{return} == 0) {
        return 1;
    }

    my $msg = $self->{msglst}->addMessage ("Looking for installation packages in '$kit_dir'");
    $msg->setEndTag ('Installation Packages');
    $self->{kit}->setMsgLstContext ([$msg->getSubMsgLst ()]);

    unless (defined $self->{'kit'}->FindArchives ($self->getInstconfig ())){
        $self->setErrorMessage ('Checking installation kit failed', $self->{'kit'}->getErrMsgLst ());
        $msg->endMessage ();
        return undef;
    }
    $msg->endMessage ();
    return 1;
}


sub checkSystemRequirements{
    my ($self) = @_;

    my $instconfig = $self->{instconfig};

    if ($instconfig->isAdminRequired()){

        #
        #   Is Admin?
        #

        my ($rc,$name) = isAdmin();

        if (defined $rc){
            if(!$rc){
                $self->setErrorMessage ($isWin ?
                    'Please start installation as local administrator (your user has to be in group "'.$name.'")!' :
                     ( isPASE() ? 'Please restart installation as user QSECOFR!' :
                     'Please restart installation as user root!'));
                return undef;
            }

        }
        else{
            $self->setErrorMessage ('Cannot check account: '.$name);
        }
    }
    my $oldContext = $self->{instconfig}->setMsgLstContext ([$self->getMsgLst()]);
    if (!defined $self->{instconfig}->checkSystemRequirements ($self->GetSysInfo(), $self->{kit}->getManifest ())){
        $self->setErrorMessage ("Checking system requirements failed ", $self->{instconfig}->getErrMsgLst());
        if ($self->{instconfig}->getIgnore ('check_hardware')){
            $self->{msglst}->addMessage ("Ignoring error due to command line switch '--ignore=check_hardware'");
        }
        else{
            $self->{instconfig}->setMsgLstContext ($oldContext);
            return undef;
        }
    }
    $self->{instconfig}->setMsgLstContext ($oldContext);
    return 1;
}


sub initInstConfig{
    my ($self) = @_;
    my $archiveDir = $self->{kit}->getArchiveDir ();
    my $xmlFile = $archiveDir . $path_separator . 'InstallParams.xml';
    if (defined $self->{instconfig}){
         if (!defined $self->{instconfig}->InitDefaults ($self->{kit})){
            $self->getErrMsgLst ()->setMsgLst ($self->{instconfig}->getErrMsgLst());
            return undef;
         }
    }
    else{
        if ($self->{kit}->isServer){
            $self->{instconfig} = new SDB::Install::Configuration::NewDB ($self, $self->{options}, $self->{configfile});
        }
        elsif ($self->{kit}->isStudio){
            require SDB::Install::Configuration::DatabaseStudio;
            $self->{instconfig} = new SDB::Install::Configuration::DatabaseStudio ($self->{options}, $self->{configfile},$self);
        }
        elsif ($self->{kit}->isClient){
            require SDB::Install::Configuration::Client;
            $self->{instconfig} = new SDB::Install::Configuration::Client ($self->{options}, $self->{configfile});
        }
        elsif ($self->{kit}->isOfficeClient){
            require SDB::Install::Configuration::OfficeClient;
            $self->{instconfig} = new SDB::Install::Configuration::OfficeClient ($self->{options}, $self->{configfile});
        }
        elsif (-f $xmlFile) {
            my $dtdFile = $archiveDir . $path_separator . 'installcfg.dtd';
            if (!-f $dtdFile){
                $self->setErrorMessage ("Installation configuration DTD file '$dtdFile' not found");
                return undef;
            }
            require SDB::Install::Configuration::Generic;
            $self->{instconfig} = new SDB::Install::Configuration::Generic ($self->{options}, $self->{configfile});
            $self->{instconfig}->setLogger($self->{log});
            if (!defined $self->{instconfig}->parseInstallParamsXml($xmlFile, $dtdFile)){
                $self->setErrorMessage (undef, $self->{instconfig}->getErrMsgLst());
                return undef;
            }

            my $kitClass = $self->{instconfig}->getKitClass ();
            if (defined $kitClass){
                eval ("require $kitClass;");
                if ($@){
                    $self->setErrorMessage ("Kit class '$kitClass' not implemented: " . $@);
                    return undef;
                }
                bless ($self->{kit},$kitClass);
            }
            $self->{kit}->setConfigParser ($self->{instconfig}->{configparser});
            if (!defined $self->{kit}->initEventHandler ()){
                $self->setErrorMessage (undef, $self->{kit}->getErrMsgLst());
                return undef;
            }

            my $eventHandler = $self->{kit}->getEventHandler ();

            $eventHandler->setInstConfig ($self->{instconfig});

            my $customConfigurationClass = $self->{instconfig}->getCustomConfigurationClass ();
            if (defined $customConfigurationClass){
                local @INC = @INC;
                if ($self->{kit}->isa ('SDB::Install::Kit')){
                    push @INC, $self->{kit}->getArchiveDir();
                    push @INC, $self->{kit}->getCustomModulesDir();
                }
                eval ("require $customConfigurationClass;");
                if ($@){
                    $self->setErrorMessage ("Configuration class '$customConfigurationClass' not implemented: " . $@);
                    return undef;
                }
                bless ($self->{instconfig},$customConfigurationClass);
           }

            # execute custom event preCheck
            if (!$eventHandler->triggerEvent ('preCheck', $self->getMsgLst(), $self->getErrMsgLst())){
                return undef;
            }
        }
        else{
            $self->setErrorMessage ("No installation kit found in '$archiveDir'");
            return undef;
        }
        if (!defined $self->{instconfig}->InitDefaults ($self->{kit})){
            $self->setErrorMessage (undef, $self->{instconfig}->getErrMsgLst());
            return undef;
        }

        my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();
        if (!defined $cfgDumpRc){
            $self->{return} = 1;
            return undef;
        }

        if ($cfgDumpRc){
            $self->{return} = 0;
            return 1;
        }
        my $textDetectedKit = $self->{instconfig}->getProductName() . ' installation kit detected.';
        if ($self->{dump_config_xml}){
            $self->{msglst}->addMessage ($textDetectedKit);
        }
        else{
            $self->{msglst}->addProgressMessage ($textDetectedKit);
        }
    }
    return 1;
}



sub ParseInstallationCmdLineArgs{
    my ($self, $args) = @_;
    local @ARGV = @$args;
    my %opt_ctrl = (
        'archive_dir=s'    => \$self->{options}->{archive_dir},
        'a=s'              => \$self->{options}->{archive_dir},
        'cluster'          => \$self->{options}->{cluster},       # hidden
        'force_extract'    => \$self->{options}->{force_extract}, # hidden
        'kill_running_processes' =>
                              \$self->{options}->{kill_running_processes}, # hidden
        'list_packages'    => \$self->{options}->{list_packages},
        'l'                => \$self->{options}->{list_packages},
        'list_systems'     => \$self->{options}->{list_systems},
        'L'                => \$self->{options}->{list_systems},
        'overview'         => \$self->{options}->{list_systems},  # depricated
        'no_debug_packages'=> \$self->{options}->{no_debug_packages},
        'package=s'        => \$self->{options}->{package},       # hidden

        'check_files'      => \$self->{repair_mode} );

    if (!defined $self->getOptions  (\%opt_ctrl,1)){
        $self->{return} = -1;
        return undef;
    }
    
    return \@ARGV;
}


sub ResetConfiguration{
    my ($self) = @_;
    my $old_config = $self->{instconfig};
    
    my $instconfig;
    
    if (defined $old_config &&
        $old_config->isa ('SDB::Install::Configuration::Upgrade')){
    
        $instconfig = new SDB::Install::Configuration::Upgrade ($self->{options});
    }
    else{
        $instconfig = new SDB::Install::Configuration::NewDB ($self, $self->{options});
    }
    $instconfig->ImportConfigFile ($old_config);
    $self->{instconfig} = $instconfig;
}

sub ResetPrepare{
    my ($self) = @_;
    if (defined $self->{sapSystem}){
        $self->{sapSystem}->FreePackages();
        delete $self->{sapSystem};
    }
    if (defined $self->{kit}){
        $self->{kit}->Reset ();
    }
    if (defined $self->{packageManagerCollection}){
        $self->{packageManagerCollection}->removePackageManagersByKey ('kit')
    }
    return undef;
}


#----------------------------------------------------------------------------
# Returns a reference to an array containing installation options
# without help/info options.

sub GetInstallationSwitches{

    return [@{$_[0]->SUPER::GetSwitches()},
            '--archive_dir=<dir>',
            '--check_files',
            '--no_debug_packages'];
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing help/info options of the program.

sub GetInstallationHelpSwitches {
    my ($self, $indent)= @_;
    return $self->GetModifiedHelpSwitches(['--archive_dir=<dir>'], $indent);
 }

#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of batch and help options.

sub GetInstallationBatchHelpUsage {

    my ($self, $usage) = @_;

    if (!defined $self->{'options'}->{'archive_dir'}) {

        foreach my $currUsage (@$usage) {
            if ($currUsage->[0] eq '--help') {
                $currUsage->[2] = $currUsage->[2]
                    . "\nFor specific help, use with the parameter '--archive_dir'";
                last;
            }
        }
    }
    return $usage;
}

#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays containg the description
# of hdbinst options.
sub GetInstallationUsage{

    return [ @{$_[0]->SUPER::GetUsage()},
        ['--archive_dir=<dir>', '-a <path>','Installation Kit Directory'],
        ['--check_files',       undef,      'Checks files, if same package is already installed'],
        ['--list_packages',     '-l',       'Shows available packages'],
        ['--list_systems',      '-L',       "Shows installed $gProductName systems"],
        ['--no_debug_packages', undef,      'Suppresses the installation of debug packages']];

       #['--package=<package list>', undef, 'Select packages to be installed'], # hidden
}


sub SelectSoftwareComponentsByCmdLineArguments{
    
    my ($self) = @_; 
    if (defined $self->{options}->{package}){
        if (!defined $self->{kit}->GenerateTree()){
            $self->setErrorMessage ('error in installation kit', $self->{kit}->getErrMsgLst());
            return undef;
        }
        
        if (defined $self->{options}->{package}){
            foreach my $name (split (',', $self->{options}->{package})){
                my $package = $self->{kit}->GetPackageByName ($name);
                if (!defined $package){
                    $self->setErrorMessage ("package \"$name\" not found");
                    return undef;
                }
                if (!defined $self->{kit}->SelectPackage ($package)){
                    $self->setErrorMessage ("cannot select package $package", $self->{kit}->getErrMsgLst());
                    return undef;
                }
            }
        }
        return 1;
    }
    else{
        return 0;
    }
}


sub preinstallClient{
    my ($self) = @_;

    my $instconfig = $self->getInstconfig ();
    $self->{installation}->setConfiguration ($instconfig);
    my $phase = $instconfig->getPhase ();
    if (defined $phase &&
        $phase ne $gPhasePrepareOption &&
        $phase ne $gPhasePrepareOptionAlias &&
        $phase ne $gPhaseOfflineOption &&
        $phase ne $gPhaseConfigureOption){
        $self->{kit}->{installation} = $self->{installation};
        return 1;
    }

    my $msglst = $self->getMsgLst ();
    my $msg = $msglst->addProgressMessage ("Checking installation...");
    $msg->setEndTag ('Installation Check');
    my $packageManagerCollection = $self->{'packageManagerCollection'};


    $packageManagerCollection->SetProgressHandler ($msglst->getProgressHandler());
    if (!defined $packageManagerCollection->getPackageManagersByKey ('kit')){
        $packageManagerCollection->addPackageManagerObject ('kit', $self->{kit}, $self->{'installation'});
    }
    
    my $force_downgrade = $self->{options}->{force_downgrade} ||
            $self->{'instconfig'}->getIgnore('check_version');
    $packageManagerCollection->setMsgLstContext([$msg->getSubMsgLst()]);
    if (!$packageManagerCollection->Preinstall ($self->{repair_mode},$force_downgrade)){
        $self->setErrorMessage ("Preinstall check failed", $packageManagerCollection->getErrMsgLst());
        $msg->endMessage ();
        return  undef; 
    }
    $msg->endMessage ();
    return 1;
}

#-------------------------------------------------------------------------------
# Installation or upgrade of a system or component.
#
# Parameter: $checkOnly boolean   # may be undefined

sub preinstall{
    my ($self, $checkOnly, $msglst) = @_;
    my $retcode = 1;
    my $appName = 'hdbinst';
    if ($self->isa ('SDB::Install::App::Gui')){
        $appName = 'hdbsetup';
    }
    $self->{_logNameInst} = $appName;
    if (!defined $msglst){
        $msglst = $self->getMsgLst ();
    }
    my $instconfig = $self->{instconfig};
    $instconfig->setMsgLstContext ([$msglst]);
	if (!$instconfig->checkMinMem()) {
        $self->setErrorMessage ("Checking minimum memory requirement failed", $instconfig->getErrMsgLst ());
        if ($instconfig->getIgnore ('check_min_mem')){
            $self->resetError();
            $msglst->addMessage ("Ignoring error due to command line switch '--ignore=check_min_mem'");
            $instconfig->{'params'}->{'RestrictMaxMem'}->{'skip'} = 1;
            $instconfig->{'params'}->{'MaxMem'}->{'skip'} = 1;
        }
        else {
            return undef;
        }
	}
    if (!$instconfig->checkSystemRequirementsAfterConfig ($self->GetSysInfo())){
        $self->setErrorMessage ("Checking system requirements failed", $instconfig->getErrMsgLst ());
        if ($instconfig->getIgnore ('check_hardware')){
             $msglst->addMessage ("Ignoring error due to command line switch '--ignore=check_hardware'");
        }
        else{
            if (!$checkOnly && defined $instconfig->getMounter()){
                $instconfig->umountStorageDevices();
            }
            return undef;
        }
    }
    my $packageManagerCollection = $self->{packageManagerCollection};
    
    if ($instconfig->getIgnore('check_diskspace')){
        $packageManagerCollection->{_ignore_fs_check} = 1;
    }
    
    if ($instconfig->getIgnore('check_busy_files') || $self->{options}->{force_extract}){
        $packageManagerCollection->{_ignore_busy_files} = 1;
    }
    my $dataFootprints = $instconfig->getFootprints();

    if (defined $dataFootprints && @$dataFootprints){
        $self->{kit}->setDataFootprints ($dataFootprints);
    }
	if ($instconfig->isa ('SDB::Install::Configuration::Generic')) {
        my $instClass = $instconfig->getInstallationClass ();
        eval ("require $instClass;");
        if ($@){
            die ($@);
        }
        $self->{installation} = $instClass->new($instconfig->get_InstallationPath (), SDB_O_CREAT);
        my $productKey = $self->{kit}->getProductKey ();

        my $logPathName;

        if ($productKey){
            $logPathName = lc ($productKey);
            $self->{_logNameInst} = $appName.'_'.$logPathName;
        }
        else {
            $logPathName = $appName;
        }

	my $phase = $instconfig->getPhase ();

	if (defined $phase){
		$self->{_logNameInst} .= '_' . $phase;
	}

        if (!$self->defineLog($logPathName, $self->{_logNameInst}, $checkOnly)) {
            return undef;
        }
        $instconfig->dumpConfigFile($self->{_logNameInst});

        if (!$checkOnly) {
            $retcode = $self->preinstallClient ();
        }
        if ($retcode && $self->{kit}->isa ('SDB::Install::Kit::Generic')){
            my $eventHandler = $self->{kit}->getEventHandler ();
            # execute custom event postCheck
            if (!$eventHandler->triggerEvent ('postCheck', $self->getMsgLst(), $self->getErrMsgLst())){
                return undef;
            }
        }
    }
    elsif($instconfig->isa ('SDB::Install::Configuration::Client')){

        $self->{_logNameInst} = $appName.'_client';

        if (!$self->defineLog('client', $self->{_logNameInst}, $checkOnly)) {
            return undef;
        }
        $instconfig->dumpConfigFile($self->{_logNameInst});

        $self->{'installation'} = new SDB::Install::Installation::Client(
            $instconfig->get_InstallationPath (),
            SDB_O_CREAT,
            $instconfig->{'ClientInstUID'},
            $instconfig->{'ClientInstGID'},
            $instconfig->{'params'}->{'SID'}->{'value'}
        );
        $self->{installation}->{kit_version} = $self->{kit}->GetVersion ();
        if (!$checkOnly) {
            $retcode = $self->preinstallClient ();
        }
    }
    elsif ($instconfig->isa ('SDB::Install::Configuration::DatabaseStudio')){
		if(defined($self->{'installation'}) && $self->{'installation'}->can('CloseRegistry')){
			$self->{'installation'}->CloseRegistry();
		}
    	
        $self->{'installation'} = new SDB::Install::Installation::DatabaseStudio ($instconfig->get_InstallationPath (), SDB_O_CREAT);
        require SDB::Install::Kit::DatabaseStudio;
        bless ($self->{kit},'SDB::Install::Kit::DatabaseStudio');

        $self->{installation}->{kit_version} = $self->{kit}->GetVersion ();
        
        my $copy_repository = $instconfig->getValue ('TargetRepositoryPath');

        if (defined $copy_repository){
            $self->{kit}->{repository_destination} = $copy_repository;
            $self->{'installation'}->{copy_repository} = $copy_repository;

            $self->{_logNameInst} = $appName.'_studio_repository';

            if (!$self->defineLog('studio_repository',
                                  $self->{_logNameInst},
                                  $checkOnly)) {
                return undef;
            }
            $instconfig->dumpConfigFile($self->{_logNameInst});
            $retcode = 1;
        }
        else{
            $self->{_logNameInst} = $appName.'_studio';
            if (!$self->defineLog('studio', $self->{_logNameInst}, $checkOnly)) {
                return undef;
            }
            $instconfig->dumpConfigFile($self->{_logNameInst});
            $self->{kit}->{java_vm} = $instconfig->getValue ('JavaVm');
            $self->{kit}->{repository} = $instconfig->getValue ('SourceRepositoryUrl');
			if(!$self->{kit}->{repository}) {
				$self->{kit}->{repository} = $instconfig->getBatchValue ('SourceRepositoryUrl');
			}
            if (!$checkOnly) {
                $retcode = $self->preinstallClient ();
            }
        }
    }
    elsif($instconfig->isa ('SDB::Install::Configuration::OfficeClient')){
        require SDB::Install::Kit::OfficeClient;
        require SDB::Install::Installation::OfficeClient;
        bless ($self->{kit},'SDB::Install::Kit::OfficeClient');

        $self->{_logNameInst} = $appName.'_excel';
        if (!$self->defineLog('excel', $self->{_logNameInst}, $checkOnly)) {
            return undef;
        }
        $instconfig->dumpConfigFile($self->{_logNameInst});

        $self->{'installation'} = new SDB::Install::Installation::OfficeClient(
            $instconfig->get_InstallationPath (),
            SDB_O_CREAT);

        $self->{installation}->{kit_version} = $self->{kit}->GetVersion ();
        if (!$checkOnly) {
            $retcode = $self->preinstallClient ();
        }
    }
    else{
        if (!defined $self->{sapSystem}){
            if ($instconfig->isa ('SDB::Install::Configuration::Upgrade')){
                $self->{sapSystem} = CollectSAPSystems ()->{$instconfig->getValue('SID')};
            }
            else{
                my $usrSapSid = join ($path_separator, $isWin ? $instconfig->getValue('Drive') : '' , 'usr','sap', $instconfig->getValue('SID'));
                my $sapSys    = new SDB::Install::SAPSystem();
                $sapSys->initSAPSystem($usrSapSid, SDB_O_CREAT, $instconfig->getValue ('Target'));
                $self->{sapSystem} = $sapSys;
            }
        }

        my $logPathName = $instconfig->isUpdate() ? 'upgrade' : 'install';
        my $logName = $instconfig->isUpdate() ? 'hdbupd' : 'hdbinst';
        my $phase = $instconfig->getPhase();
        if ($phase) {
            $logName .= "_$phase";
        }
        $self->{_logNameInst} = $logName;

        if (!$self->defineLog($logPathName,
                              $self->{_logNameInst},
                              $checkOnly,
                              $self->{sapSystem}->{_sid})) {
            return undef;
        }
        $instconfig->dumpConfigFile($self->{_logNameInst});
        $self->{sapSystem}->{kit_version} = $self->{kit}->GetVersion ();
        my $msg = $msglst->addMessage("Existing SAP Systems");
        $self->{sapSystem}->dumpSapSystems($msg->getSubMsgLst());
        my $step = $instconfig->getStep();
        my $isBeforeExtract = $step <= ($instconfig->isUpdate() ? $STEP_UPGRADE_EXTRACT : $STEP_EXTRACT);
        if ($isBeforeExtract) {
            my $msg = $msglst->addProgressMessage ("Checking installation...");
            $msg->setEndTag('Installation Check');

            if (!defined $packageManagerCollection->getPackageManagersByKey ('kit')){
                $packageManagerCollection->addPackageManagerObject ('kit', $self->{kit}, $self->{sapSystem});
            }

            my $force_downgrade = $self->{options}->{force_downgrade} ||
                $instconfig->getIgnore('check_version');
            $packageManagerCollection->setMsgLstContext([$msg->getSubMsgLst ()]);
            if (!$packageManagerCollection->Preinstall ($self->{repair_mode}, $force_downgrade)){
                $self->setErrorMessage ("Preinstall check failed", $packageManagerCollection->getErrMsgLst());
                if (!$checkOnly && defined $instconfig->getMounter()){
                    $instconfig->umountStorageDevices();
                }
                $msg->endMessage ();
                return  undef;
            }
            if (!$checkOnly && defined $instconfig->getMounter()){
                $instconfig->umountStorageDevices();
            }
            $msg->endMessage ();
        }
    }

    my $rc = undef;

    if($checkOnly) {
        $rc = 1;
    }
    elsif(defined $self->{'installation'}) {
        $self->{'installation'}->setConfiguration($instconfig);
        $self->{'installation'}->getMsgLst->setProgressHandler($self->getMsgLst->getProgressHandler());
        $rc = $self->{'installation'}->loadPersistentInstallationParams();
    }
    elsif(defined $self->{'sapSystem'}) {
        $self->{'sapSystem'}->setConfiguration($instconfig);
        $rc = $self->{'sapSystem'}->loadPersistentInstallationParams();
    }
    return defined $rc ? $retcode : undef;
    
}

sub installClient{
    my ($self) = @_;
    my $rc = 1;
    
    
    my $product = $self->{instconfig}->getProductName ();
    
    my $msg = $self->{msglst}->addProgressMessage ("Installing $product to " .
        $self->{instconfig}->get_InstallationPath () . "...");

    $msg->setEndTag ("Installing $product");
    $self->{'installation'}->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $self->{'installation'}->initNewInstallation ()){
        $self->setErrorMessage (undef, $self->{'installation'}->getErrMsgLst());
        $msg->endMessage ();
        return undef;
    }
    $self->{kit}->setMsgLstContext ([$msg->getSubMsgLst ()],1);
    if (!defined $self->{kit}->Install()){
        $self->setErrorMessage ("Cannot install",$self->{kit}->getErrMsgLst());
        $rc = undef;
    }
    $msg->endMessage ();
    return $rc;
}

sub _constructProgressMessage {
    my ($self, $instconfig) = @_;
    my $phase = $instconfig->getPhase();
    if (!$phase || $phase eq $gPhaseOfflineOption) {
        return 'Creating System';
    } elsif ($phase eq $gPhaseConfigureOption) {
        return 'Configuring System';
    }
    return 'Finalizing System Installation';
}

#----------------------------------------------------------------------------

sub install{
    my ($self) = @_;
    my $rc = 1;

    my $return = 0;
    my $retcode;
    my $instconfig = $self->getInstconfig();

    if ($instconfig->isa ('SDB::Install::Configuration::Generic') ||
        $instconfig->isa ('SDB::Install::Configuration::Client') ||
        $instconfig->isa ('SDB::Install::Configuration::OfficeClient') ||
        $instconfig->isa ('SDB::Install::Configuration::DatabaseStudio')){
        $retcode = $self->installClient ();
        $return = 1;
    }
    if($return) {
        return $retcode;
    }

    my $progressMessage = $self->_constructProgressMessage($instconfig);
    my $msg = $self->{msglst}->addProgressMessage ("$progressMessage...");
    $msg->setEndTag('Create System');
    $self->{sapSystem}->setMsgLstContext ([$msg->getSubMsgLst ()]);
    $self->{kit}->setMsgLstContext([$msg->getSubMsgLst ()], 1);

    if (!defined $self->{sapSystem}->create ($self->{kit}, $instconfig)){
        $msg->endMessage ();
        $self->setErrorMessage ("Cannot create system",$self->{sapSystem}->getErrMsgLst ());
        return undef;
    }
    
    if (!$self->isa ('SDB::Install::App::Gui')){
        if (!defined $self->{sapSystem}->createInstance ($self->{instconfig}, $self->GetSysInfo())){
            $msg->endMessage ();
            $self->setErrorMessage ("Cannot create Instance",$self->{sapSystem}->getErrMsgLst());
            return undef;
        }
    }
    my $nextPhase = $instconfig->getNextPhaseFromCurrent();
    if (defined $nextPhase && $nextPhase eq 'online' && $instconfig->getValue('NoStart')) {
        return 1;
    }
    $instconfig->addPhaseToPersistenceFile($nextPhase) if defined $nextPhase;
    return 1;
}

#-------------------------------------------------------------------------------
# Writes the log and clean-up the installation.
#
# Parameter: boolean $checkOnly   # may be undefined

sub CleanUp{
    my ($self, $checkOnly) = @_;
    if(defined $self->{'installation'}) {
        $self->{'installation'}->writePersistentInstallationParams();
    }
    elsif(defined $self->{'sapSystem'}) {
        $self->{'sapSystem'}->writePersistentInstallationParams();
        my $instances = $self->{'sapSystem'}->getNewDBInstances ();
        if (defined $instances && @$instances){
            $self->addLogToTraceDir($self->{'sapSystem'}, $self->{_logNameInst},
                                                                    $checkOnly);
        }
    }
    if (defined $self->{kit}){
        $self->{kit}->setMsgLstContext ([$self->{msglst}]);
        $self->{kit}->FreePackages();
        delete $self->{kit};
    }
    if (defined $self->{sapSystem}){
        $self->{sapSystem}->setMsgLstContext([$self->{msglst}]);
        $self->{sapSystem}->FreePackages();
        delete $self->{sapSystem};
    }
    $self->SUPER::CleanUp;
}


sub shouldWarnIfCalledStandalone{
    return 0;
}

1;
