package SDB::Install::ClientRegisterTools;

use strict;

use SDB::Install::Globals qw ($gProductNameClient);
use SDB::Install::System qw (makedir getAllUsersAppDataPath isSameFile normalizePath);
use SAPDB::Install::Hostname;
use SAPDB::Install::MD5Sum;
use SDB::Install::SysVars   qw($isWin $path_separator);
use SDB::Install::MsgLst;


require Exporter;

our @ISA = qw (Exporter);

our @EXPORT = qw (
    registerClient
    unregisterClient
);

#-------------------------------------------------------------------------------


#-------------------------------------------------------------------------------

sub registerClient {
    my (
        $instPath,
        $msglst,
        $errlst
    ) = @_;
    my $nPath = $instPath;
    if($isWin) {
        $nPath = normalizePath($instPath);
    }
    my $manifestFile = $nPath.$path_separator.'manifest';
    if (not -f $manifestFile){
        $errlst->addError("Manifest file '$manifestFile' not found: $!");
        return undef;
    }
    my $manifest = new SDB::Install::Manifest($manifestFile);
    if(not defined $manifest) {
        $errlst->addError("Could not read Manifest file '$manifestFile'.");
        return undef;
    }
    my $version = $manifest->getVersion ();
    if(not defined $version) {
        $errlst->addError('Could not get version', $manifest);
        return undef;
    }
    my $isStudio = $manifest->isStudio();
    my $regfile_dir = join ($path_separator, getAllUsersAppDataPath(),
    '.hdb', lc (hostname ()));
    my $regfile = $manifest->isStudio() ? 'installations.dbstudio' : 'installations.client';
    $regfile = $regfile_dir.$path_separator.$regfile;
    if(!-d $regfile_dir) {
        my $cfg = {'mode' => 0755};
        if(!defined makedir ($regfile_dir, $cfg)) {
            $errlst->addError("Could not create client registry", $cfg);
            return undef;
        }
    }
    my @buffer;
    my $create_new = 1;
    if(-f $regfile) {
        $create_new = 0;
        if(!open(RG, $regfile)) {
            $errlst->addError("Could not open client registry: $!");
            return undef;
        }
        $msglst->addProgressMessage("Using client registry at '$regfile'.");
        @buffer = <RG>;
        close(RG);
    }
    my $found = 0;
    my ($cmp, $cmpPath);
    foreach $cmp (0 .. $#buffer){
        if($buffer[$cmp] =~ /^\s*#/) {
            next;
        }
        ($cmpPath) = ($buffer[$cmp] =~ /(.*)=.*/);
        if(!defined $cmpPath) {
            next;
        }
        $cmpPath =~ s/^\s*//;
        $cmpPath =~ s/\s*$//;
        if(isSameFile($cmpPath, $nPath)) {
            $found = 1;
            $buffer[$cmp] = "$nPath=".$version."\n";
            $msglst->addMessage("Installation '$nPath' already registered. Setting version to '$version'.");
        }
    }
    if(not $found) {
        push @buffer, "$nPath=".$version."\n";
    }
    my $tmp_file = $regfile.MD5Str($nPath);
    if(!open (RG, ">$tmp_file")) {
        $errlst->addError ("Could not create client registry ($tmp_file): $!");
        return undef;
    }
    foreach my $i (0 .. $#buffer) {
        print RG $buffer[$i];
    }
    close(RG);
    if(!$create_new) {
        if(!unlink($regfile)) {
            $errlst->addError("Could not delete origin client registry: $!");
            return undef;
        }
    }
    else {
        $msglst->addProgressMessage("Creating client registry at '$regfile'.");
    }
    if(!rename($tmp_file, $regfile)) {
        $errlst->addError("Could not rename client registry $tmp_file => ".$regfile.": $!");
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------

sub unregisterClient {
    my (
        $instPath,
        $msglst,
        $errlst
    ) = @_;
    my $nPath = $instPath;
    if($isWin) {
        $nPath = normalizePath($instPath);
    }
    my $manifestFile = $nPath.$path_separator.'manifest';
    if (not -f $manifestFile){
        $errlst->addError ("Manifest file '$manifestFile' not found: $!");
        return undef;
    }
    my $manifest = new SDB::Install::Manifest($manifestFile);
    if(not defined $manifest) {
        $errlst->addError("Could not read Manifest file '$manifestFile'.");
        return undef;
    }
    my $isStudio = $manifest->isStudio();
    my $regfile_dir = join ($path_separator, getAllUsersAppDataPath(),
    '.hdb', lc (hostname ()));
    my $regfile = $manifest->isStudio() ? 'installations.dbstudio' : 'installations.client';
    $regfile = $regfile_dir.$path_separator.$regfile;
    if(not -f $regfile) {
        $msglst->addMessage("'$gProductNameClient' registry '$regfile' not existent anyway: Exiting.");
        return 1;
    }
    if(!open(RG, $regfile)) {
        $errlst->addError("Could not open client registry: $!");
        return undef;
    }
    $msglst->addProgressMessage("Using client registry at '$regfile'.");
    my @buffer = <RG>;
    close(RG);
    my $found = undef;
    my ($cmp, $cmpPath);
    foreach $cmp (0 .. $#buffer){
        if($buffer[$cmp] =~ /^\s*#/) {
            next;
        }
        ($cmpPath) = ($buffer[$cmp] =~ /(.*)=.*/);
        if(!defined $cmpPath) {
            next;
        }
        $cmpPath =~ s/^\s*//;
        $cmpPath =~ s/\s*$//;
        if(isSameFile($cmpPath, $nPath)) {
            $found = $cmp;
            last;
        }
    }
    if(not defined $found) {
        $msglst->addMessage("'$gProductNameClient' registry entry '$nPath' not existent anyway: Exiting.");
        return 1;
    }
    my $tmp_file = $regfile.MD5Str($nPath);
    if(!open (RG, ">$tmp_file")) {
        $errlst->addError ("Could not create client registry ($tmp_file): $!");
        return undef;
    }
    foreach my $i (0 .. $#buffer) {
        if($i == $found) {
            next;
        }
        print RG $buffer[$i];
    }
    close(RG);
    if(!unlink($regfile)) {
        $errlst->addError("Could not delete origin client registry: $!");
        return undef;
    }
    if(!rename($tmp_file, $regfile)) {
        $errlst->addError("Could not rename client registry $tmp_file => ".$regfile.": $!");
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------

1;
