#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#

package SDB::Install::ConfigXMLParser;

use SDB::Install::DebugUtilities qw (dumpThings);
use SDB::Install::System qw (getSysProgPath getSystemStartMenuPath getSystemDesktopPath);
use SDB::Install::SysVars qw ($isWin);
use SDB::Install::Tools qw (URLencode);
use SDB::Common::Utils qw(createXMLParser);

use base XML::Parser::Expat;


our %systemSubst = (
	'$(ProgramFiles)' => getSysProgPath (),
	'$(StartMenuPrograms)' => getSystemStartMenuPath (),
	'$(Desktop)' => getSystemDesktopPath ()
);


sub new {
	my $self = shift->SUPER::new (@_);
	$self->{root} = undef;
	$self->{parent} = undef;
	$self->{current} = undef;
	$self->setHandlers (
		'Start' => \&sh,
		'End'   => \&eh,
		'Char'  => \&ch);
	$self->{Product} = undef;
	$self->{Parameters} = undef;
	$self->{Environment} = undef;
	$self->{Directories} = [];
	$self->{ReplaceInFiles} = undef;
	$self->{ExecuteCommands} = [];
	$self->{Shortcuts} = [];
	$self->{UninstallationEntry} = undef;
	$self->{RunProgramAfterInstallation} = undef;
	$self->{ExecuteCommandsAtActivation} = [];
	$self->{ImportContent} = undef;


	return $self;
}


sub sh{
	my ($self, $el, %atts) = @_;
	my $cur = $self->{current} = {'type' => $el, 'attr' => \%atts};

	if (defined $self->{parent}){
		$cur->{parent} = $self->{parent};
	}
	else{
		$self->{root} = $cur;
	}
	$self->{parent} = $cur;
}

sub eh{
	my ($self, $el) = @_;
	my $cur = $self->{current};
	if (defined $cur->{parent}){
		if (! defined $cur->{parent}->{child}) {
			$cur->{parent}->{child} = $cur;
		}
		else{
			my $element = $cur->{parent}->{child};
			while (defined $element->{neighbor}) {
				$element = $element->{neighbor};
			}
			$element->{neighbor} = $cur;
		}
	}
	$self->{current} = $self->{parent} = $cur->{parent};
}

sub ch{
	my ($self, $cn) = @_;
	$self->{current}->{content} = $cn;
#	print "Element: " . $self->{current}->{type} . ", Content: $cn \n";
}

sub printTree{
	my ($self,$node, $indent) = @_;
	if (!defined $node){
		$node = $self->{root};
	}

	if (!defined $node){
		return;
	}

	print "$indent $node->{type} = $node->{content}" . "\n";

	if (defined $node->{child}){
		$self->printTree ($node->{child}, $indent . "  ");
	}

	if (defined $node->{neighbor}){
		$self->printTree ($node->{neighbor}, $indent);
	}
}

sub getElementByTagName{
	my ($self,$tag, $node) = @_;
	my $result;

	if (!defined $node){
		return undef;
	}

	if ($node->{type} eq $tag){
		return $node;
	}

	$node = $node->{child};
	while ($node) {
		$result = $self->getElementByTagName ($tag, $node);
		if (defined $result){
			return $result;
		}
		$node = $node->{neighbor};
	}

	return undef;
}


sub parsefile{
	my $self = shift;
	my ($filename, $dtdFilename) = @_;
	if(defined $dtdFilename) {
	   $self->validateFile($filename, $dtdFilename);
	}
	# call superclass parser, which uses our handler functions
	$self->SUPER::parsefile ($filename);
	$self->{varSubst} = {};
	# find'Variables' node
	my $node = $self->getElementByTagName ('Variables', $self->{root});
	# find first 'Variable' node
	$node = $self->getElementByTagName ('Variable', $node);
	my ($varNameNode,$varValueNode);
	while ($node){
		$varNameNode = $self->getElementByTagName ('VarName', $node);
		$varValueNode = $self->getElementByTagName ('VarValue', $node);
		$self->{varSubst}->{'$(' . $varNameNode->{content} . ')'} = $varValueNode->{content};
		# continue with next 'Variable' node
		$node = $node->{neighbor};
	}
	# add systemSubst variables, overwriting existing variables
	foreach my $key (keys (%systemSubst)){
		$self->{varSubst}->{$key} = $systemSubst{$key};
	}
	# apply variable substitution to parsed XML
	$self->replaceVariables();

	# transfer data from XML to specialized hashes available via getter functions
	$self->extractConfiguration();
}


sub validateFile{
    my (
        $self,
        $filename,
        $dtdFilename
    ) = @_;
    eval {
        require XML::LibXML;
        import XML::LibXML;
    };
    if(!$@) {
        eval {
            my $parser = createXMLParser();
            my $doc = $parser->parse_file($filename);
            my $dtdURL = URLencode($dtdFilename);
            my $dtd = XML::LibXML::Dtd->new("SOME // Public / ID / 1.0", $dtdURL);
            $doc->validate($dtd);
        };
        if($@) {
            my $errmsg = "\nCould not validate \"".$filename.
                         "\"\n                   against\n                   \"".
                         $dtdFilename."\".\n".
                         "Root  cause: ".$@."\n";
            die($errmsg);
        }
    }
}

sub replaceVariables{
    my ($self, $node) = @_;
    if (!defined $node){
        $node = $self->{root};
    }
    my $string = $node->{content};
    if ($string){
        # get all occurences like $(NAME)
        my @vars = ($string =~ /\$\([^\)]+\)/g);
        # found something that may have to be replaced
        if (@vars){
            my ($pattern,$replaced);
            $replaced = 0;
            foreach my $var (@vars){
                # is it a known variable
                if (exists $self->{varSubst}->{$var}){
                    $pattern = quotemeta ($var);
                    $string =~ s/$pattern/$self->{varSubst}->{$var}/g;
                    $replaced++;
                }
            }
            if ($replaced){
                # change parse tree node
                $node->{content} = $string;
            }
        }
    }
    # continue replacing
    if (defined $node->{child}){
        $self->replaceVariables ($node->{child});
    }
    if (defined $node->{neighbor}){
        $self->replaceVariables ($node->{neighbor});
    }
}



# downward compatibility: no attribute means 'installation' and 'update',
# used by 'Parameters'
our $defaultAttributesInstallUpdate = {
    'installation' => 1,
    'update'       => 1
};

# downward compatibility: no attribute means 'installation', 'update' and uninstallation,
# used by 'CustomModules'
our $defaultAttributesInstallUpdateUninstall = {
    'installation'   => 1,
    'update'         => 1,
    'uninstallation' => 1
};


sub _shouldFilterNode{
    my ($node, $filterAttributes, $defaultAttributes, $tagDefaultAttributes) = @_;
    if (!defined $filterAttributes){
        return 0;
    }
    my $currentAttributes = $node->{attr};
    if (!defined $currentAttributes or !(keys %$currentAttributes)){
        my $tag = $node->{type};
        if (defined $tagDefaultAttributes && defined $tagDefaultAttributes->{$tag}){
            $currentAttributes = $tagDefaultAttributes->{$tag};
        }
        else{
            $currentAttributes = $defaultAttributes;
        }
    }
    my $filtered = 1;
    foreach my $attrib (@$filterAttributes){
        if (exists $currentAttributes->{$attrib}){
            if ($currentAttributes->{$attrib} == 0){
                $filtered = 1;
                last;
            }
            if ($currentAttributes->{$attrib} == 1){
                $filtered = 0;
            }
        }
    }
    return $filtered;
}




sub extractConfiguration{
    my ($self) = @_;
    my $node = undef;
    my ($ID, $subnode);
    my $filterAttributes = $self->{filterAttributes};

    # extract Product data
    $node = $self->getElementByTagName('Product', $self->{root});
    $node = $node->{child};
    while ($node){
        $self->{Product}->{$node->{type}} = $node->{content};
        $node = $node->{neighbor};
    }

    # extract Parameters
    $node = $self->getElementByTagName('Parameters', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'ParamsWindows' : 'ParamsUNIX'), $node);
    $node = $self->getElementByTagName('Parameter', $node);
    ($ID, $subnode) = undef;
    my $count = 1;
    while ($node){
        $subnode = $self->getElementByTagName('ID', $node);
        $ID = $subnode->{content};
        if (_shouldFilterNode ($node, $filterAttributes, $defaultAttributesInstallUpdate)){
            $node = $node->{neighbor};
            next;
        }
        $self->{Parameters}->{$ID} = undef;
        $subnode = $subnode->{neighbor};
        while ($subnode){
            $self->{Parameters}->{$ID}->{$subnode->{type}} = $subnode->{content};
            $subnode = $subnode->{neighbor};
        }
        $self->{Parameters}->{$ID}->{order} = $count++;
        $node = $node->{neighbor};
    }

    # extract Environment
    $node = $self->getElementByTagName('Environment', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'EnvWindows' : 'EnvUNIX'), $node);
    $node = $self->getElementByTagName('EnvVariable', $node);
    ($ID, $subnode) = undef;
    while ($node){
        $subnode = $self->getElementByTagName('EnvVarName', $node);
        $ID = $subnode->{content};
        $self->{Environment}->{$ID} = undef;
        $subnode = $subnode->{neighbor};
        while ($subnode){
            $self->{Environment}->{$ID}->{$subnode->{type}} = $subnode->{content};
            $subnode = $subnode->{neighbor};
        }
        $node = $node->{neighbor};
    }

    # extract Directories
    $node = $self->getElementByTagName('CreateDirectories', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'DirWindows' : 'DirUNIX'), $node);
    $node = $self->getElementByTagName('CreateDirectory', $node);
    $subnode = undef;
    while ($node){
        $subnode = $self->getElementByTagName('DirPath', $node);
        push(@{$self->{Directories}}, $subnode->{content});
        $node = $node->{neighbor};
    }

    # extract ReplaceInFiles
    $node = $self->getElementByTagName('ReplaceInFiles', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'ReplaceWindows' : 'ReplaceUNIX'), $node);
    $node = $self->getElementByTagName('ReplaceInFile', $node);
    $subnode = undef;
    while ($node){
        $subnode = $node->{child};
        $self->{ReplaceInFiles}->{$subnode->{content}} = $subnode->{neighbor}->{content};
        $node = $node->{neighbor};
    }

    # extract ExecuteCommands
    $node = $self->getElementByTagName('ExecuteCommands', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'ExecWindows' : 'ExecUNIX'), $node);
    $node = $self->getElementByTagName('ExecuteCommand', $node);
    $subnode = undef;
    while ($node){
        my $execHash = undef;
        my $envHash = undef;
        $subnode = $node->{child};
        my @argList;
        while ($subnode) {
            if ($subnode->{type} eq 'ExecArg'){
                push @argList, $subnode->{content};
            } elsif ($subnode->{type} eq 'ExecEnv'){
                my $var = $self->getElementByTagName('ExecEnvVar', $subnode)->{content};
                my $value = $self->getElementByTagName('ExecEnvValue', $subnode)->{content};
                $envHash->{$var} = $value;
            } else {
             $execHash->{$subnode->{type}} = $subnode->{content};
            }
            $subnode = $subnode->{neighbor};
        }
        if (@argList){
            $execHash->{ExecArg} = \@argList;
        }
        if ($envHash) {
            $execHash->{ExecEnv} = $envHash;
        }
        push(@{$self->{ExecuteCommands}}, $execHash);
        $node = $node->{neighbor};
    }

    # extract P2 director call
    $node = $self->getElementByTagName('P2Director', $self->{root});
    $node = $node->{child};
    while ($node){
        if ($node->{type} eq 'InstUnit') {
            if (!defined $self->{P2Call}->{'InstUnits'}) {
                $self->{P2Call}->{'InstUnits'} = [];
            }
            push (@{$self->{P2Call}->{'InstUnits'}}, $node->{content});
        } else {
            $self->{P2Call}->{$node->{type}} = $node->{content};
        }
        $node = $node->{neighbor};
    }

    # extract Shortcuts
    $node = $self->getElementByTagName('Shortcuts', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'ScWindows' : 'ScUNIX'), $node);
    $node = $self->getElementByTagName('Shortcut', $node);
    $subnode = undef;
    while ($node){
        my $scHash = undef;
        $subnode = $node->{child};
        while ($subnode) {
            $scHash->{$subnode->{type}} = $subnode->{content};
            $subnode = $subnode->{neighbor};
        }
        push(@{$self->{Shortcuts}}, $scHash);
        $node = $node->{neighbor};
    }

    # extract content import criterion
    $node = $self->getElementByTagName('ImportContent', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'ImportWindows' : 'ImportUNIX'), $node);
    if($node) {
        $self->{ImportContent} = $node->{content};
    }

    # extract UninstallationEntry
    $node = $self->getElementByTagName('UninstallationEntry', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'UninstWindows' : 'UninstUNIX'), $node);
    $node = $node->{child};
    $self->{UninstallationEntry}->{'UninstDirectories'} = [];
    $self->{UninstallationEntry}->{'UninstFiles'} = [];
    while ($node){
        if ($node->{type} eq 'DirPath') {
            push (@{$self->{UninstallationEntry}->{'UninstDirectories'}}, $node->{content});
        } elsif ($node->{type} eq 'FilePath') {
            push (@{$self->{UninstallationEntry}->{'UninstFiles'}}, $node->{content});
        } else {
            $self->{UninstallationEntry}->{$node->{type}} = $node->{content};
        }
        $node = $node->{neighbor};
    }

    # extract CustomModules
    $node = $self->getElementByTagName('CustomModules', $self->{root});
    $node = $node->{child};
    while ($node){
        if (_shouldFilterNode (
                $node,
                $filterAttributes,
                $defaultAttributesInstallUpdateUninstall,
                {'Configuration' => $defaultAttributesInstallUpdate})){

            $node = $node->{neighbor};
            next;
        }
        $self->{CustomModules}->{$node->{type}} = $node->{content};
        $node = $node->{neighbor};
    }

    # extract RunProgramAfterInstallation
    $node = $self->getElementByTagName('RunProgramAfterInstallation', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'RunWindows' : 'RunUNIX'), $node);
    $node = $self->getElementByTagName('ExecuteCommand', $node);
    $subnode = undef;
    my $execHash = undef;
    my $envHash = undef;
    $subnode = $node->{child};
    my @argList;
    while ($subnode) {
        if ($subnode->{type} eq 'ExecArg'){
            push @argList, $subnode->{content};
        } elsif ($subnode->{type} eq 'ExecEnv'){
            my $var = $self->getElementByTagName('ExecEnvVar', $subnode)->{content};
            my $value = $self->getElementByTagName('ExecEnvValue', $subnode)->{content};
            $envHash->{$var} = $value;
        } else {
            $execHash->{$subnode->{type}} = $subnode->{content};
        }
        $subnode = $subnode->{neighbor};
    }
    if (@argList){
        $execHash->{ExecArg} = \@argList;
    }
    if ($envHash) {
        $execHash->{ExecEnv} = $envHash;
    }
    $self->{RunProgramAfterInstallation} = $execHash;

    # extract ExecuteCommandsAtActivation
    $node = $self->getElementByTagName('ExecuteCommandsAtActivation', $self->{root});
    $node = $self->getElementByTagName(($isWin ? 'ExecAtActivationWindows' : 'ExecAtActivationUNIX'), $node);
    $node = $self->getElementByTagName('ExecuteCommand', $node);
    $subnode = undef;
    while ($node){
        my $execHash = undef;
        my $envHash = undef;
        $subnode = $node->{child};
        my @argList;
        while ($subnode) {
            if ($subnode->{type} eq 'ExecArg'){
                push @argList, $subnode->{content};
            } elsif ($subnode->{type} eq 'ExecEnv'){
                my $var = $self->getElementByTagName('ExecEnvVar', $subnode)->{content};
                my $value = $self->getElementByTagName('ExecEnvValue', $subnode)->{content};
                $envHash->{$var} = $value;
            } else {
             $execHash->{$subnode->{type}} = $subnode->{content};
            }
            $subnode = $subnode->{neighbor};
        }
        if (@argList){
            $execHash->{ExecArg} = \@argList;
        }
        if ($envHash) {
            $execHash->{ExecEnv} = $envHash;
        }
        push(@{$self->{ExecuteCommandsAtActivation}}, $execHash);
        $node = $node->{neighbor};
    }
}

sub getProduct{
	my ($self) = @_;
	return $self->{Product};
}

sub getCustomModules{
	my ($self) = @_;
	return $self->{CustomModules};
}

sub getParameters{
	my ($self) = @_;
	return $self->{Parameters};
}

sub getEnvironment{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{Environment});
	return defined $replaced ? $replaced : $self->{Environment};
}

sub getDirectories{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{Directories});
	return defined $replaced ? $replaced : $self->{Directories};
}

sub getReplaceInFiles{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{ReplaceInFiles});
	return defined $replaced ? $replaced : $self->{ReplaceInFiles};
}

sub getExecuteCommands{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{ExecuteCommands});
	return defined $replaced ? $replaced : $self->{ExecuteCommands};
}

sub getP2Call{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{P2Call});
	return defined $replaced ? $replaced : $self->{P2Call};
}

sub getShortcuts{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{Shortcuts});
	return defined $replaced ? $replaced : $self->{Shortcuts};
}

sub getImportContent{
	my ($self) = @_;
	return $self->{ImportContent};
}

sub getUninstallationEntry{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{UninstallationEntry});
	return defined $replaced ? $replaced : $self->{UninstallationEntry};
}

sub getRunProgramAfterInstallation{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{RunProgramAfterInstallation});
	return defined $replaced ? $replaced : $self->{RunProgramAfterInstallation};
}

sub getExecuteCommandsAtActivation{
	my ($self) = @_;
	my $replaced = $self->replaceInStruct ($self->{ExecuteCommandsAtActivation});
	return defined $replaced ? $replaced : $self->{ExecuteCommandsAtActivation};
}

sub setInstParamReplaceHash{
	$_[0]->{instParamReplaceHash} = $_[1];
}

sub setFilterAttributes{
    $_[0]->{filterAttributes} = $_[1];
}

sub replaceInString{
	my ($self, $string) = @_;

	if ($string){
		# get all occurences like %(NAME)
		my @vars = ($string =~ /%\([^\)]+\)/g);
		# found something that may have to be replaced
		if (@vars){
			my ($pattern,$replaced);
			$replaced = 0;
			foreach my $var (@vars){
				# is it a known variable
				if (exists $self->{instParamReplaceHash}->{$var}){
					$pattern = quotemeta ($var);
					$string =~ s/$pattern/$self->{instParamReplaceHash}->{$var}/g;
					$replaced++;
				}
			}
			if ($replaced){
				return $string;
			}
		}
	}
	return undef;
}

sub replaceInStruct{
	my ($self, $struct) = @_;
	if (!defined $_[0]->{instParamReplaceHash}){
		return undef;
	}
	my $type = ref ($struct);
	my $string;
	my $newStuct;
	if ($type eq 'ARRAY'){
		my @copy;
		foreach my $i (0 .. (scalar (@$struct) - 1)){
			if (ref ($struct->[$i])){
				$newStruct = $self->replaceInStruct ($struct->[$i]);
				if (defined $newStruct){
                    if (!@copy){
                        @copy = @$struct;
                    }
					$copy[$i] = $newStruct;
				}
			}
			else{
				$string = $self->replaceInString ($struct->[$i]);
				if (defined $string){
                    if (!@copy){
                        @copy = @$struct;
                    }
					$copy[$i] = $string;
				}
			}
		}
		if (@copy){
			return \@copy;
		}
		else{
			return undef;
		}
	}
	elsif ($type eq 'HASH'){
		my %copy;
		foreach my $key (keys %$struct){
			if (ref ($struct->{$key})){
				$newStruct = $self->replaceInStruct ($struct->{$key});
				if (defined $newStruct){
                    if (!%copy){
                        %copy = %$struct;
                    }
					$copy{$key} = $newStruct;
				}
			}
			else{
				$string = $self->replaceInString ($struct->{$key});
				if (defined $string){
                    if (!%copy){
                        %copy = %$struct;
                    }
					$copy{$key} = $string;
				}
			}
		}
		if (%copy){
			return \%copy;
		}
		else{
			return undef;
		}
	}
	elsif ($type eq 'SCALAR'){
		$string = $self->replaceInString ($$ref);
		if (defined $string){
			return \$string;
		}
		return undef;
	}
	elsif ($type eq 'REF'){
		$newStruct = $self->replaceInStruct ($$ref);
		if (defined $newStruct){
			return \$newStruct;
		}
		return undef;
	}
	return undef;
}

1;
