package SDB::Install::Configuration::DatabaseStudio;

use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Installation::DatabaseStudio;
use SDB::Install::SysVars;
use SDB::Install::System qw (isEmptyDir);
use SDB::Install::Globals qw($gProductNameStudio $gShortProductNameStudio);
use SDB::Install::System::FuserHangChecker;

use SDB::Install::Utils::DatabaseStudio::DatabaseStudioKnownFeatures;
use SDB::Install::Utils::DatabaseStudio::DatabaseStudioFeatureManager;
use SAPDB::Install::ProcState qw (SDB_PROCSTATE_FLAG_MODULES);
require File::stat;

use strict;

our @ISA = qw (SDB::Install::Configuration::AnyConfig);
our $section = 'Studio';

our $requiredDiskSpaceForInstallation = 400 * 1024 * 1024; # 400mb
our $requiredDiskSpaceForUpdate = 100 * 1024 * 1024; # 100mb

sub new {
    my ($class, $options, $configFile, $caller) = @_;
    my $self = shift->SUPER::new ($options, $configFile);
    
    my $progpath = $self->getDefaultProgPath ();
    my $order = 0;
    my $isGUI = defined $caller
                       && $caller->isa('SDB::Install::App::Gui::Installation');
    $self->{params} = {
    	( $isGUI ? () :  ('TargetRepositoryPath' => $self->getParamTargetRepositoryPath ($order++, $section))),
      
        'JavaVm' => $self->getParamJavaVm ($order++, $section),
        'SourceRepositoryUrl' => $self->getParamSourceRepositoryUrl ($order++, $section),
        'PATH' => {
            'order' => $order++,
            'opt' => 'path',
            'short_opt' => 'p',
            'type' => 'path',
            'section' => $section,
            'value' => undef,
            'default' => join ($path_separator, $progpath, 'sap',$isWin || $installerIs64bit ? 'hdbstudio' : 'hdbstudio32'),
            'str' => 'Installation Path',
            'mandatory' => 1,
            'required_disk_space' => $requiredDiskSpaceForInstallation
        },
        ($isWin ? (
        'SkipVCRedist' => $self->getParamSkipVCRedist ($order++)
        ) : ()),
        'StartStudioAfterInstallation' => {
            'order'                     => $order++,
            'type'                      => 'boolean',
            'section'                   => $section,
            'value'                     => undef,
            'str'                       => "Start $gProductNameStudio after installation",
            'set_interactive'           => 0,
            'default'                   => 0,
            'init_with_default'         => 1,
        'excludeFromConfigDump' => 1,
        },
        'SelectedFeatures' => $self->getParamSelectedFeatures ($order++, $section)
    };
    
    return $self;
}


sub newRepositoryCanAppleAppLayout{
    my ($self) = @_;
    if (!$isApple){
        return 0;
    }
    if (!defined $self->{canAppLayout}){
        my $featureDetecor =
            new SDB::Install::Utils::DatabaseStudio::DatabaseStudioFeatureDetector(
                $self->{kit},
                '/var/tmp',
                $self->getBatchValue ('SourceRepositoryUrl'),
                $self->getBatchValue ('JavaVm')
            );
        my $msg = $self->getMsgLst ()->addProgressMessage ('Checking whether repository supports app layout');
        $featureDetecor->setMsgLstContext ([$msg->getSubMsgLst ()]);
        $self->{canAppLayout} = $featureDetecor->repositoryCanAppleAppLayout ();
        if ($self->{canAppLayout}){
            $msg->getSubMsgLst ()->addMessage ("Yes, it's supported.");
        }
        else{
            $msg->getSubMsgLst ()->addMessage ("No, it isn't supported.");
        }
    }
    return $self->{canAppLayout};
}



sub getParamSelectedFeatures {
    my ($self, $order, $section) = @_;
                        
    my %param = (
            'order' => $order,
            'opt'            => 'features',
            'opt_arg'        => 'all|<feat1>[,<feat2>]...',
            'type' => 'csv',
            'section' => $section,
            'value' => undef,
            'default' => 'all',
            'init_with_default' => 1,
            'set_interactive' => 1,
            'str' => 'Feature',
            'mandatory' => 0,
            'help_with_default' => 1,
            'interactive_index_selection' => 1,
            'leaveEmptyInConfigDump' => 1,
            'desc' => 'Specifies the features (admin, appdev, dbdev, answers, importmetadata, ...) to be installed',
            'interactive_str' => 'comma-separated list of the selected indices',
            'console_text' => "\n".'Select features to install:'."\n",
            'hidden' => 0,
            );
    return \%param; 
}

sub _findDefaultPath{
    my ($self) = @_;
    my $pathParam = $self->{params}->{PATH};
    my $default;
    my $i = 0;
    my $appValue;

    while (1){
        $i++;
        if (!defined $default){
            $default = $self->getDefault('PATH');
        }
        else{
            $default = $self->getDefault('PATH') . "$i";
        }

        if ($isApple){
            $appValue = $default . '.app';
        }

        if ((!-e $default || isEmptyDir ($default)) &&
            ((!defined $appValue || !-e $appValue || isEmptyDir ($appValue)))){
            last;
        }
    }
    $self->setDefault('PATH', $default);
    return $default;
}


sub InitDefaults{
    my ($self,$kit) = @_;
    
    if (defined $kit){
	$self->{kit} = $kit;
    }
    
    $self->setDefault('SourceRepositoryUrl', 'file:' . $kit->{archive_dir} . $path_separator . 'repository');

    my $containsJVM = $self->_isKitContainingJvm();
    $self->{params}->{JavaVm}->{init_with_default}=!$containsJVM;

    if ( $containsJVM ){
  	my $vm = $self->{params}->{JavaVm}->{ 'mandatory' } = 0 ;
    }elsif ( ! $self->_findJavaInstallations( $kit ) ) {
	$self->appendErrorMessage ("No suitable Java installation found.");
    	return undef;
    }   

    my $msg = $self->AddMessage ("Looking for studio features and products...");
    $self->{featureManager} =  SDB::Install::Utils::DatabaseStudio::DatabaseStudioFeatureManager->new($self);
    $self->_addParameterListenerToPath($msg);
    $self->{params}->{'SelectedFeatures'}->{hidden} = 0;
    $self->{params}->{'SelectedFeatures'}->{skip} = 0;

    $self->{action}            = 'Install';
    $self->{actionProgressive} = 'Installing';
    $self->{actionDone}        = 'installed';

    $self->_findDefaultPath ();
    return 1;
}
 
sub _isKitContainingJvm{
    my ($self, $kit ) = @_;
    if(! defined $kit){
    	$kit = $self->{kit};
    }
    return $kit->getManifest()->isStudioWithEmbededJvm();
}

sub _addParameterListenerToPath {
	my ( $self, $msg ) = @_;
	$self->addParameterListener ( 'PATH', sub{
            my $valid = $self->{featureManager}->updateDefaultAndValidValues(@_);
            $self->AddSubMsgLst( $msg, $self->{featureManager} );
            
            if(!$valid && !$self->{featureManager}->getErrMsgLst()->isEmpty()){
            	$self->getErrMsgLst()->appendMsgLst ($self->{featureManager}->getErrMsgLst());
            	$self->{featureManager}->resetMsgLstContext();
            }
            if (! $valid || defined $self->getValue ( 'SelectedFeatures' )){
                return $valid;
            }
            
            my $installablefeatures = $self->{featureManager}->getAllInstallableFeatures();
            my $hasNoFeaturesToInstall = defined $installablefeatures ? (scalar @$installablefeatures )  == 0 : 1 ;
            my $isInGuiMode = ( %{Wx::} && exists ${Wx::}{wxTheApp} );
            
            # Initialize SelectFeatures parameter values. Cases :
            # - When in gui mode, so default values are pre-checked
            # - When in interactive mode, in update scenario, there are no features to be installed ( only to be updated) so the parameter is skipped
            if ( $isInGuiMode ||  ( $hasNoFeaturesToInstall && $self->isUpdate() ) ) { 
                my $batchValue   = $self->getBatchValue ( 'SelectedFeatures' );
                my $defaultValue = $self->getDefault    ( 'SelectedFeatures' );
                my $newValue = defined $batchValue ? $batchValue : $defaultValue;
                if ( ! $self->setValue ( 'SelectedFeatures', $newValue ) ) {
                    $self->setValue ( 'SelectedFeatures', $defaultValue );
                }
            }
            return $valid;
        } 
    );
}

sub _findJavaInstallations {
	my ($self, $kit ) = @_;
	
    my $vm = $self->{params}->{JavaVm};
    my $java = $self->findJava ();
    if (defined $java){
        $self->setDefault('JavaVm', $java);
    }
    return 1;
}

sub setProgressHandler {
	my ($self, $handler) = @_;
	$self->getFeatureManager()->setProgressHandler($handler);
}

sub getFeatureManager{
	my ( $self ) = @_;
	return $self->{featureManager};
}

sub getProductName{
	return $gProductNameStudio;
}

sub getShortProductName{
	return $gShortProductNameStudio;
}


sub setTargetRepositoryPath{
	my ($self,$value) = @_;
	if (!-d $value){
		$self->PushError ("Invalid repository destination '$value': $!");
		return 0;
	}
	$self->{params}->{TargetRepositoryPath}->{value} = $value;
	$self->{params}->{JavaVm}->{skip} = 1;
	$self->{params}->{PATH}->{skip} = 1;
	$self->{params}->{SelectedFeatures}->{skip} = 1;
	$self->{params}->{StartStudioAfterInstallation}->{hidden} = 1;
	
	$self->{action}            = 'Copy repository of';
    $self->{actionProgressive} = 'Copying repository of';
    $self->{actionDone}        = 'repository copied'; 
    
	return 1;
}

sub _convertAppleInstPath{
    my ($self, $instpath) = @_;
    if ($self->newRepositoryCanAppleAppLayout ()){
        # append '.app'
        if($instpath !~ /\.app$/){
            $instpath .= '.app';
        }
    }
    else{
        # remove '.app'
        if($instpath =~ /\.app$/){
            $instpath =~ s/\.app$//;
        }
    }
    return $instpath;
}


sub _checkForRunningProcesses{
    my ($self, $exePath) = @_;
    my $stat = File::stat::stat ($exePath);
    if (defined $stat && -f $stat){
        if ($isLinux){
            my $hangChecker = new SDB::Install::System::FuserHangChecker();
            $hangChecker->setMsgLstContext($self->getMsgLstContext());
            my $rc = $hangChecker->check();
            if (defined $rc && $rc == 0){
                return undef;
            }
        }
        my $procStateObj   = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_MODULES);
        my $pids = $procStateObj->WhoUsesModule ($exePath);
        my $msglst = new SDB::Install::MsgLst ();
        if (defined $pids){
            foreach my $pid (@$pids){
                my $cmdLn = $procStateObj->GetArgs($pid);
                if (!defined $cmdLn){
                    $cmdLn = $procStateObj->GetCmd($pid);
                }
                if (defined $cmdLn){
                    $msglst->addMessage ("'$cmdLn' (pid=$pid)");
                }
            }
        }
        if (!$msglst->isEmpty()){
            $self->setErrorMessage("$gProductNameStudio is running. Please stop the following processe(s)", $msglst);
            return 0;
        }
    }
    return 1;
}

sub checkPATH{
	my ($self,$value) = @_;
	$self->setBatchValueOfRequiredParams ('SourceRepositoryUrl');
	my $param = $self->{params}->{PATH};
	if( ! $value){
		$self->setErrorMessage ("$param->{str} is empty");
		return 0; 
	}

    #
    # reject special characters in installation path because eclipse
    # cannot handle it
    #
    my $chars = $isWin ? '%!#' : '\\%!#:';
    my $regex = '[' . quotemeta ($chars) . ']';
    if ($value =~ /$regex/){
        $self->setErrorMessage ("$param->{str} is invalid: Special characters ($chars) are not allowed.");
        return 0;
    }

    $self->{update} = 0;
    $self->{installedVersion} = undef;

	if (-d $value){
		my $inst = new SDB::Install::Installation ($value);
		if (!$inst->ErrorState () && $inst->isStudio()){
			bless ($inst, 'SDB::Install::Installation::DatabaseStudio');
			my $errlst = new SDB::Install::MsgLst();
			if (!$self->{kit}->canUpdate($inst, $self, $errlst) || !defined $inst->is64bit()){
				$self->PushError (undef, $errlst);
				return 0;			
			}
            $self->{update} = 1;
            $self->{installedVersion} = $inst->getVersionByManifest();
			$self->AddMessage (undef, $errlst);
		}
        if ($self->isUpdate()){
            if ($isApple && $value !~ /\.app$/ && $self->newRepositoryCanAppleAppLayout ()){
                $self->PushError ("Cannot upgrade $gProductNameStudio in '$value' due to file system changes on Mac OS.");
                $self->PushError ("Please perform a new installation.");
                return 0;
            }
            $self->{params}->{PATH}->{required_disk_space} = $requiredDiskSpaceForUpdate;
            $self->{action}            = 'Update';
            $self->{actionProgressive} = 'Updating';
            $self->{actionDone}        = 'updated';
            my $studioExe = $self->{kit}->getHanaStudioExe ($value);
            if (!$self->_checkForRunningProcesses($studioExe)){
                return 0;
            }

        } else {
            if ($isApple){
                $value = $self->_convertAppleInstPath ($value);
            }
            if (-d $value && !isEmptyDir ($value)){
                $self->PushError ("Existing directory '$value' is not empty");
                return 0;
            }
            $self->{params}->{PATH}->{required_disk_space} = $requiredDiskSpaceForInstallation;
        }
	}
	return 1;
}

sub setPATH{
    my ($self, $value) = @_;
    if (!$self->checkPATH ($value)){
        return undef;
    }
    if ($isApple && !$self->isUpdate ()){
        $value = $self->_convertAppleInstPath ($value);
    }
    $self->{params}->{PATH}->{value} = $value;
    return 1;
}


sub isUpdate {
    my ($self) = @_;	
	return $self->{update};
}

sub enumExistingInstallations{
	# simulate absence of existing installations
	#return {};

	my ($self) = @_;
	my $errlst = new SDB::Install::MsgLst();
	my $installations = SDB::Install::Installation::DatabaseStudio::EnumDBStudioInstallations ($errlst);
	if (!defined $installations){
		$self->AddError (undef, $errlst);
		return undef;
	}	
	my ($installation, $version, $objMode, $platform);
	my $result = {};
	foreach my $path (keys (%$installations)){
		$installation = new SDB::Install::Installation::DatabaseStudio ($path); 
		$version = $installation->getVersionByManifest();
		if (!$version){
			next;
		}
		if ($isWin || $isLinux){
			if (defined $installation->is64bit()) {
				$objMode = $installation->is64bit() ? '64bit' : '32bit';
			} else {
				$objMode = undef;
			}
		}
		
		$result->{$path} = {
			'version' => (defined $version ? $version : $installations->{$path}),
			'mode' => (defined $objMode ? $objMode : undef),
			'canUpdate' => $self->{kit}->canUpdate($installation, $self)
		};		
	}
	return $result;
}

sub getIgnoreValues{
    return ['check_diskspace'];
}


sub checkSelectedFeatures{
	my ($self, $value) = @_;
	my @keys = split(',',$value);
	
	if( ! $self->isUpdate() ) {
		if( scalar @keys == 0 ){
			$self->PushError ("At least one studio feature must be selected");
			return 0;
		}
	}
	$self->_refreshAnswersLegalNote( \@keys ); 
	return 1;
}
sub _refreshAnswersLegalNote{
	my ($self, $keys) =@_;
	
	
	my $repoFeatures = $self->isUpdate() ? 
			$self->{featureManager}->getAllInstallableFeatures():
			$self->{featureManager}->{detector}->getRepositoryFeatures();
	
	my $answersKey = GetBatchKey( FEATURE_ID_ANSWERS );
	my $answersContained = 0;
	for my $feature ( @{$repoFeatures} ) {
		if ( $feature->{'id'} eq GetKnownFeatureIdByKey($answersKey)){
			$answersContained = 1;
		}
	}
	if( ! $answersContained ){
		return;
	}
	my $answersNote = ANSWERS_LEGAL_NOTE;
	if( grep(/^$answersKey/,@{$keys}) || grep(/^all$/,@{$keys})){
		if( ! grep(/^$answersNote/,@{$self->{warnings}})){
			push @{$self->{warnings}}, $answersNote;
		}
	} else {
		my @filteredWarnings = grep { $_ != $answersNote } @{$self->{warnings}};	
		$self->{warnings}=\@filteredWarnings;
	}	
}


sub isInstallationSelected{
	if ($_[0]->isValuePreset ('TargetRepositoryPath')){
		return 1;
	}
	return $_[0]->isValuePreset ('PATH');
}

sub isAdminRequired{
    return 0;
}

sub setStartStudioAfterInstallation{
	my ($self, $value) = @_;
	my $rc = $self->checkValue('StartStudioAfterInstallation', $value);

	if($rc){
		my $installPath = $self->get_InstallationPath();

		$self->{params}->{StartStudioAfterInstallation}->{value}=$value;
		$self->{kit}->{InstallationPath} = $installPath;
		$self->{kit}->{StartStudioAfterInstallation} = $value;
	}

	return $rc;
}

sub getProductVersion {
	my ($self) = @_;
	return $self->{kit}->GetVersion();
}

1;
