package SDB::Install::Configuration::NewDB;
use strict;
use warnings;

use SDB::Install::Package::Installable;
use SDB::Install::Configuration;
use SDB::Install::Configuration::NewServerConfig;
use SDB::Install::Configuration::ValueChangeListeners::InstallationPhaseListener;
use SDB::Install::HdbMounter;
use SDB::Install::SysVars;
use SAPDB::Install::Hostname;
use SDB::Install::System;
use SDB::Install::Installer;
use SAPDB::Install::System::Unix qw (sysinfo);

use SDB::Install::Globals qw ($gDriveWin
                              $gProductName
                              $gProductNameEngine
                              $gSapmntDirUx
                              $gSapsysGroupIdDefault
                              $gSapsysGroupName
                              $gShortProductNameEngine
                              $gMinMemGB
                              $gPhaseOptionPerskey
                              $gPhaseOfflineOption
                              $gPhaseConfigureOption
                              $gPhaseOnlineOption);

use SAPDB::Install::System::Win32::API;
use SDB::Install::SAPSystem qw (CollectSAPSystems ExistsInstanceNumber
			$STEP_CREATE_INSTANCE $STEP_EXTRACT @STEP_NAMES FORBIDDEN_SIDS
			isForbiddenSID);
use SAPDB::Install::MachInfo;
use File::Spec;

our @ISA = qw (SDB::Install::Configuration::NewServerConfig);

our $section = 'Server';

our $defaultBasePathRoot = $isWin ? '$Drive\usr\sap\$SID\SYS\global\hdb\\' :
                                    '$Target/$SID/global/hdb/';

our $volumeSubDirPerHost = 'mnt00001';


sub getDefaultVolumePath{
    my ($self, $dataOrLog, $substitute) = @_;
    my $default = $defaultBasePathRoot . $dataOrLog;
    if($substitute){
        return $self->substituteValues($default);
    }
    return $default;
}

#
# reserve well known installation names
#


#our @user_params = $isWin ? ('Password','Domain') : ('Password','Shell',
#	'HomeDir', 'UID', 'GID');

sub InitDefaults{
    my ($self, $kit) = @_;
    my $physicalMemory = $self->getPhysicalMem();

	if (!defined $physicalMemory || ($physicalMemory < $gMinMemGB * 1024 * 1024 * 1024)) {
		$self->{params}->{RestrictMaxMem}->{skip} = 1;
		$self->{params}->{MaxMem}->{skip} = 1;
	} else {
		$self->setDefault('MaxMem', int($physicalMemory/(1024*1024)));
		$self->{params}->{MaxMem}->{minValue} = $gMinMemGB * 1024;
		$self->{params}->{MaxMem}->{maxValue} = int($physicalMemory/(1024*1024));
	}

    if (defined $kit){
        $self->{kit} = $kit;
        $self->{_kitversion} = $kit->GetVersion;
        $self->setValue ('versionId', $kit->getVersionIdString ());
        my $mf = $kit->getManifest();
        if (!defined $mf){
            $self->AddError (undef, $kit);
            return undef;
        }
        $self->{kitDir} = $kit->getArchiveDir ();
    }

	my $msglst = new SDB::Install::MsgLst ();

	$self->setDefault('InstanceNumber', undef);
	my $defaultInstanceNumber = $self->getFreeInstanceNumber ();
	$self->setDefault ('InstanceNumber', $defaultInstanceNumber);
	$self->{params}->{InstanceNumber}->{valid_values} = $self->getFreeInstanceNumbers ();

	if (!$isWin){
		my $gid = getgrnam ($gSapsysGroupName);
		if (defined $gid){
			$self->{params}->{GID}->{value} = $gid;
			$self->{sapsysids}->{$gid} = [hostname()];
			$self->{localSapsysGID}    = [ $gid, hostname() ];
		}
	}
	else{
		my $drive = getSAPDrive ($self->getMsgLst ());
		if ($drive){
			$self->{params}->{Drive}->{value} = uc ($drive);
			$self->{params}->{Drive}->{isStaticValue} = 1;
			#$self->{params}->{Target}->{value} = $path;
		}
	}

	return 1;
}

#-------------------------------------------------------------------------------
# Constructor
#
# Parameters: $caller      (derived class of SDB::Install::App,  may be undef)
#             $options     (used by SDB::Install::Configuration, may be undef)
#             $config_file (used by SDB::Install::Configuration, may be undef)

sub new{
    my ($class, $caller, @args) = @_;

    my $self        = $class->SUPER::new(@args);
    my $order       = 0;
    my $is_hdbsetup = (defined $caller
                       && $caller->isa('SDB::Install::App::Gui::Installation'));
   $self->{params} = {
        ($isWin
            ? ()
            : ('UsePMem' => $self->getParamUsePMem($order++, $section))
        ),
        'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $section),
        ($is_hdbsetup
            ? ()
            : ('CheckOnly' => $self->getParamCheckOnly($order++, $section, 'install'), )
        ),
        'RemoteExecution'  => $self->getParamRemoteExecution ($order++, $section),
        'UseHttp'          => $self->getParamUseHttp         ($order++, $section),
        'InstallHostagent' => $self->getParamInstallHostagent($order++, $section),
        'HostName'         => $self->getParamHostName        ($order++, $section),
        'CreateInitialContainer' => $self->getParamCreateInitialContainer($order++, $section),
        ($isWin
            ? ()
               # today add host via hdbinst is implemented for Linux only
            : ('AddHosts'  => $self->getParamAddHosts      ($order++, $section, 1),
              )
        ),
        ($isWin
            ? ('HostagentUserName' => $self->getParamHostagentUserName($order++, $section),
              )
            : ('CheckMnt'      => $self->getParamCheckMnt      ($order++, $section),
               'Target'        => $self->getParamTarget        ($order++, $section),
               'InstallSSHKey' => $self->getParamInstallSSHKey ($order++, $section),
               'RootUser'      => $self->getParamRootUser      ($order++, $section), # calls collect host info
               'RootPassword'  => $self->getParamRootPassword  ($order++ ,$section), # calls collect host info
              )
         ),
        'HostagentPassword'=> $self->getParamHostagentPassword($order++,$section),   # calls collect host info
        'SID'              => $self->getParamSID             ($order++, $section),
        'Phase'            => $self->getParamPhase ($order++, $section, [$gPhaseOfflineOption, $gPhaseConfigureOption, $gPhaseOnlineOption]),
        'InstanceNumber'   => $self->getParamInstanceNumber  ($order++, $section),
        'WorkerGroup'      => $self->getParamWorkerGroup     ($order++, $section,0 , $STEP_CREATE_INSTANCE),
        'DbMode'           => $self->getParamDbMode          ($order++, $section),
        'DbIsolation'      => $self->getParamDbIsolation     ($order++, $section),
        'CreateInitialTenant' => $self->getParamCreateInitialTenant ($order++, $section),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $section),
		'SystemUsage'      => $self->getParamSystemUsage     ($order++, $section),
        ($isWin
            ? ('DistributedSystem' => {'order'             => $order++,
                                       'opt'               => 'cfs',
                                       'type'              => 'boolean',
                                       'section'           => $section,
                                       'value'             => undef,
                                       'default'           => 0,
                                       'str'               => 'Multiple Host System',
                                       'set_interactive'   => 0,
                                       'init_with_default' => 1,
                                       'persStep'          => $STEP_EXTRACT },
               'Drive' => $self->getParamDrive ($order++, $section),
               'Domain' => $self->getParamDomain ($order++, $section),
              )
            : ()
        ),
        'Password'         => $self->getParamPassword        ($order++, $section),
        ($isWin
            ? ()
            : ('HomeDir'          => $self->getParamHomeDir         ($order++, $section, '/usr/sap/$SID/home'),
               'Shell'            => $self->getParamShell           ($order,   $section),
               'UID'              => $self->getParamUID             ($order++, $section),
               'GID'              => $self->getParamGID             ($order++, $section, $gSapsysGroupIdDefault, 1),
               'StorageConfigDir' => $self->getParamStorageConfigDir($order++, $section),
               'ListenInterface'  => $self->getParamListenInterface ($order++, $section, 1),
               'InternalNetwork'  => $self->getParamInternalNetwork ($order++, $section, 0, undef, 0, 0),
              )
        ),
        'VolumeEncryption'     => $self->getParamVolumeEncryption  ($order++, $section),
        'BasePathDataVolumes'  => $self->getParamBasePathDataVolumes($order++, $section, $self->getDefaultVolumePath('data',0)),
        'BasePathLogVolumes'   => $self->getParamBasePathLogVolumes ($order++, $section, $self->getDefaultVolumePath('log',0)),
        ($isWin
            ? ()
            : ('BasePathPMem'   => $self->getParamBasePathPMem ($order++, $section))
        ),
        'CustomConfigDir' => $self->getParamCustomConfigDir($order++, $section),
        'SecureStore'          => $self->getParamSecureStore($order++, $section),
        'ChangeInitialSSFSKey' => $self->getParamChangeInitialSSFSKey  ($order++, $section),
        'installType' => {
            'order' => $order++,
            'opt' => 'installType',
            'short_opt' => 'T',
            'type' => 'string',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'Installation Type',
            'init_with_default' => 1,
            'set_interactive' => 0,
            'hidden' => 1,
            'persist_for_upgrade' => 1,
            'DEPRECATED' => 1
        },
        'StripSymbols'       => $self->getParamStripSymbols      ($order++, $section),
		'NoStart'            => $self->getParamNoStart           ($order++, $section,
		                        'Do not start the instance after installation', 'Does not start the instance after installation', 1),
		'MachineUtilization' => $self->getParamMachineUtilization($order++, $section),
        'RestrictMaxMem'     => $self->getParamRestrictMaxMem    ($order++, $section),
        'MaxMem'             => $self->getParamMaxMem            ($order++, $section),
		'SQLSysPasswd'       => $self->getParamSQLSysPasswd      ($order++, $section,
		                                                          'initial_passwd', 0, 'SYSTEM'),
        'AutoStart'          => $self->getParamAutoStart         ($order++, $section),
        'Repository'         => $self->getParamRepository        ($order++, $section),
        'ISCMode'            => $self->getParamISCMode           ($order++, $section),
		'EnableXSEngine' => {
			'order' => $order++,
			'opt' => 'xs_engine',
			'alias_opts' => ['xsengine','execution_service'],
			'type' => 'boolean',
			'section' => $section,
			'value' => undef,
			'default' => 1,
			'str' => 'Enable XS Engine',
            'desc'=> 'Enables the XS engine',
			'console_omit_word_Enter' => 1,
			'init_with_default' => 1,
			'set_interactive' => 0,
			'persStep' => $STEP_CREATE_INSTANCE
		},
        'XSEngineHttpPort' => {
            'order' => $order++,
            'opt' => 'xs_engine_http_port',
            'alias_opts' => ['execution_service_http_port'],
            'type' => 'number',
            'opt_arg' => '<port>',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'XS Engine HTTP Port',
            'desc'=> 'Specifies the HTTP port of the XS engine',
            'init_with_default' => 0,
			'log_value' => sub {defined $_[0] ? $_[0] : '80' . $self->getValue('InstanceNumber');},
            'set_interactive' => 0,
            'skip' => 1
        },
        'XSEngineHttpsPort' => {
            'order' => $order++,
            'opt' => 'xs_engine_https_port',
            'alias_opts' => ['execution_service_https_port'],
            'type' => 'number',
            'opt_arg' => '<port>',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'XS Engine HTTPS Port',
            'desc'=> 'Specifies the HTTPS port of the XS engine',
            'init_with_default' => 0,
			'log_value' => sub {defined $_[0] ? $_[0] : '43' . $self->getValue('InstanceNumber');},
            'set_interactive' => 0,
            'skip' => 1
        },
        'ImportContent' => $self->getParamImportContent ($order++, $section),
        # internal parameter to build versioned exe directory
        'versionId' => {
            'order' => $order++,
            'type' => 'string',
            'value' => undef,
            'default' => undef,
            'init_with_default' => 0,
            'set_interactive' => 0,
            'hidden' => 1,
            'persStep' => $STEP_EXTRACT
        },
        ($isWin
            ? ('RetryHostagentUserName' => $self->getParamRetryParam($order++, 'HostagentUserName'))
            : ('SkipModifySudoers' => $self->getParamSkipModifySudoers ($order++, $section))
        ),
        'RetryHostagentPassword'        => $self->getParamRetryParam($order++, 'HostagentPassword'),
	};

    $self->setSummaryOrder(['Target',
                            'SID',
                            'InstanceNumber',
                            'DbMode',
                            'DbIsolation',
                            'SystemUsage',
                            'HomeDir',
                            'Shell',
                            'UID',
                            'GID',
                            'BasePathDataVolumes',
                            'BasePathLogVolumes',
                            'HostName',
                            'AddHosts',
                            'ListenInterface',
                            'InternalNetwork',
                            'ImportContent',
                            'InstallHostagent',
                            'RemoteExecution',
                            'RootUser',
                            'HostagentUserName',
                            'SkipModifySudoers',
                            'CheckOnly',
                           ]);

    $self->addParameterListener('Phase', SDB::Install::Configuration::ValueChangeListeners::InstallationPhaseListener->new());
	return $self;
}


sub getParamCreateInitialContainer{
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'opt'               => 'create_initial_container',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 0,
        'str'               => 'Skip createSecureStore and initTopology ',
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'hidden'            => 1,
        'skip'              => 0
    };
}

sub getProductName{
	return $gProductNameEngine;
}

sub getProductVersion {
    my ($self) = @_;
    if (!defined $self->{kit}){
        return $self->SUPER::getProductVersion ();
    }
    return $self->{kit}->GetVersion();
}

sub getShortProductName{
	return $gShortProductNameEngine;
}


sub checkCreateInitialContainer{
    my ($self, $createInitialContainer) = @_;
    if ($createInitialContainer){
        return $self->setValue('NoStart', 1);
    }
    return 1;
}


sub checkNoStart{
	my ($self, $value) = @_;
	if ($value =~ /$bool_true_pattern/i){
        if (defined $self->getAddHostsParser()) {
            $self->PushError ("Parameter '" . $self->getOpt('NoStart')
                    . "' cannot be applied together with additional hosts");
            return 0;
        }
		$self->setValue('ImportContent', 'false');
	}
	return 1;
}


#-------------------------------------------------------------------------------
# Checks the root user name for additional hosts.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkRootUser{
    my ($self, $value) = @_;

    if (defined $self->getRemoteHosts()
        && !$self->tryRemoteKeyAuthorization($value)) {

        return undef;
    }
    return 1;
}

sub pers_keys {
    my ($self) = @_;
    my $keys = $self->SUPER::pers_keys();
    my $phase = $self->getPhase();
    if ($phase) {
        $self->{next_phase} = $phase;
        push @{$keys}, 'next_phase';
    }
    push @{$keys}, $gPhaseOptionPerskey;
    return $keys;
}

sub validatePersistency{
	my ($self, $pers) = @_;
	if (!defined $pers){
		$pers = $self->pers_load();
	}

	if (!defined $pers){
		return undef;
	}

    my $resumeType = $pers->{$gPhaseOptionPerskey} ? "pending" : "broken";
	if ($pers->{_kitversion} ne $self->{_kitversion}){
		$self->AddError ("Cannot resume $resumeType installation of '$self->{current_sid}': wrong version ($self->{_kitversion})");
		$self->PushError ("$gProductNameEngine installation kit '$pers->{_kitversion}' required");

		return undef;
	}
	if (!defined $self->SUPER::validatePersistency ($pers)){
		return undef;
	}

    my $stepName = $self->pers_getstepname($pers->{step});
    my $pendingMsg = "Resuming $resumeType installation at step '$stepName'.";
    if (defined $self->getBatchValue('Phase')) {
        # Just phased call
        $self->getMsgLst()->addMessage($pendingMsg);
    } else {
        # Actual resume
        $self->getMsgLst()->addProgressMessage($pendingMsg);
    }
    $self->setResume();
    return 1;
}


sub checkMinMem{
    my ($self) = @_;
    my $mem = $self->getPhysicalMem();
	if (defined $mem) {
		if ($mem < $gMinMemGB * 1024 * 1024 * 1024) {
			my $errorText = "Physical memory on this host is " . sprintf("%.3f", $mem / (1024 * 1024 * 1024)) . " GB. Minimum required: 24 GB.";
			my $errorMsg = $self->getMsgLst()->addError($errorText);
			if ($self->getIgnore('check_min_mem')){
				$self->getMsgLst()->addMessage("Ignoring error due to command line switch '--ignore=check_min_mem'");
			}
			else{
				$self->getErrMsgLst()->appendMsg($errorMsg);
				$self->appendErrorMessage('Override memory check with command line option --ignore=check_min_mem');
                return undef;
			}
		}
	} else {
		$self->AddWarning('Cannot check physical memory: GetMachInfo() failed');
	}
	return 1;
}


sub getPhase {
    my ($self) = @_;
    return $self->getValue('Phase');
}

sub checkSID{
    my ($self, $value) = @_;
    my $rc = $self->SUPER::checkSID ($value);

    if (!$rc){
        return $rc;
    }

    if (exists $self->{params}->{UID}
        && !defined $self->getDefault('UID')
        && !defined $self->getBatchValue('UID')) {

        $self->setDefault('UID', $self->getFreeUserID());
    }

    #
    # checking whether there is an unregistered or to another host assigned
    # HANA system
    #

    my $target = $self->getValue ('Target');
    if (defined $target && !$self->pers_exists ()){
        my $sapSys = new SDB::Install::SAPSystem ();
        $sapSys->initWithGlobDir($value, $target);
        if ($sapSys->ErrorState ()){
            return 1;
        }
        if($sapSys->hasNewDB){
            my $msglst = new SDB::Install::MsgLst ();
            my $hdbInstance = $sapSys->getNewDBInstances()->[0];
            $sapSys->asString ($msglst);
            if (defined $hdbInstance->get_host()){
                # assigned to own host
                $self->setErrorMessage ("$self->{params}->{SID}->{str} '$value' is already"
                                . " in use by an un-registered $gProductNameEngine System at $target on this host:", $msglst);
            }
            else{
                # not assigned to own host
                $self->setErrorMessage ("$self->{params}->{SID}->{str} '$value' is already"
                                . " in use by a $gProductNameEngine System at $target on another host:", $msglst);
            }
            return 0;
        }
    }
    return 1;
}

sub checkRepository{
    my ($self, $value) = @_;
    foreach my $repositoryParam ('EnableXSEngine', 'ImportContent'){
        $self->setDefault ($repositoryParam, ($value =~ /$bool_true_pattern/i) ? 1 : 0);
    }
    return 1;
}

sub checkSystemRequirementsAfterConfig{
    my ($self, $sysinfo, $manifest) = @_;

    return 1 if ($isWin);

    my $datapath = $self->getValue('BasePathDataVolumes');
    my $logpath  = $self->getValue('BasePathLogVolumes');
    $datapath //= $self->getDefault('BasePathDataVolumes');
    $logpath  //= $self->getDefault('BasePathLogVolumes');

    my $storageCfgDir = $self->getValue ('StorageConfigDir');

    if (defined $storageCfgDir && $self->getStep() == 0){
        if (!$self->mountStorageDevices()) {
            return undef;
        }
        my $uid = $self->getValue ('UID');
        my $gid = $self->getValue ('GID');

        return 1 if (!defined $uid || !defined $gid);

        my $dataVolumeSubDir = ($datapath) ? File::Spec->catfile($datapath, $volumeSubDirPerHost) : undef;
        my $logVolumeSubDir = ($logpath) ? File::Spec->catfile($logpath, $volumeSubDirPerHost) : undef;

        foreach ($dataVolumeSubDir, $logVolumeSubDir) {
            next if (! defined $_);
            my $rc = chown(int($uid), int($gid), $_);
            if ($rc) {
                $self->getMsgLst()->addMessage("Owner of $_ changed (uid = $uid, gid = $gid)");
            } else {
                $self->getMsgLst()->addWarning("Cannot change owner of $_: $!");
            }
        }
    }
    return 1;
}

sub mountStorageDevices {
    my ($self) = @_;
    return $self->_execMounterSub('mount');
}

sub umountStorageDevices {
    my ($self) = @_;
    return $self->_execMounterSub('umount');
}

sub _execMounterSub {
    my ($self, $subName) = @_;

    my $mounter = $self->getMounter() // $self->_initMounter();
    my $sub = $mounter->can($subName);
    if (!defined $sub) {
        $self->getErrMsgLst()->addError("Object $mounter does not have a sub named $subName");
        return undef;
    }

    my $packagedPython = $self->extractPackagedPython();
    if (!defined $packagedPython) {
        return undef;
    }

    my $msg = $self->getMsgLst()->addMessage(ucfirst($subName) . 'ing storage devices...');
    my $saveContext = $mounter->setMsgLstContext([$msg->getSubMsgLst(), $self->getErrMsgLst()]);

    $mounter->setPython($packagedPython);
    my $rc = &$sub($mounter);
    if (!$rc) {
        $self->getErrMsgLst()->addError("Cannot $subName storage devices");
    }
    $mounter->setMsgLstContext($saveContext);

    if (!$self->removeExtractedPython()) {
        $self->getErrMsgLst()->addError("Cannot remove the temporary python extracted from the installation kit");
        return undef;
    }
    return $rc;
}

sub _initMounter {
    my ($self) = @_;
    my $storageCfgDir = $self->getValue('StorageConfigDir');
    my $datapath = $self->getValue('BasePathDataVolumes');
    my $logpath  = $self->getValue('BasePathLogVolumes');
    $datapath //= $self->getDefault('BasePathDataVolumes');
    $logpath  //= $self->getDefault('BasePathLogVolumes');

    my $mounter = SDB::Install::HdbMounter->new($self->{kitDir});
    $mounter->setParams(
        $storageCfgDir,
        File::Spec->catfile($datapath, $volumeSubDirPerHost),
        File::Spec->catfile($logpath, $volumeSubDirPerHost),
        $self->getValue ('SID'),
        1
    );
    $self->setMounter($mounter);
    return $mounter;
}

sub extractPackagedPython {
    my ($self) = @_;

    my $archiveDir = $self->{kitDir};
    if (!defined $archiveDir || !isDirectory($archiveDir)) {
        $self->getErrMsgLst()->addError("Could not retrieve installation kit directory");
        return undef;
    }

    my $pythonArchivePath = File::Spec->catfile($archiveDir, 'PYTHON.TGZ');
    if (!isRegularFile($pythonArchivePath)) {
        $self->getErrMsgLst()->addError("Could not find python package '$pythonArchivePath'");
        return undef;
    }

    # .LST files should be taken relative to the tool's instruntime
    # to cover the cases when the installation kit is copied to a secure location
    my $installer = SDB::Install::Installer->new();
    my $runtimeDir = $installer->GetRuntimeDir();
    my $pythonArchiveLst = File::Spec->catfile($runtimeDir, '..', 'server', 'PYTHON.TGZ.lst');
    if (!isRegularFile($pythonArchiveLst)) {
        $self->getErrMsgLst()->addError("Could not find python package lst file '$pythonArchiveLst'");
        return undef;
    }

    my $targetPythonDir = $self->_getTempPythonDir();

    if (isDirectory($targetPythonDir) && !$self->removeExtractedPython()) {
        $self->getErrMsgLst()->addError("Could not delete pre-existing directory '$targetPythonDir'");
        return undef;
    }
    $self->getMsgLst()->addMessage("Creating PYTHON.TGZ temporary target extraction directory '$targetPythonDir'");
    my $cfg = $self->_getTempPythonDirCfg();
    if (!makedir($targetPythonDir, $cfg)) {
        $self->getErrMsgLst()->addError("Could not create PYTHON.TGZ tepmorary target extraction directory '$targetPythonDir'", $cfg->getErrMsgLst());
        return undef;
    }

    my $installationKit = $self->{kit};
    my $verifyChecksums = $installationKit->shouldVerifyChecksums($self);
    # A Dummy installation is used to set the extraction dir for package
    my $dummyInstallation = SDB::Install::Installation->new($targetPythonDir);

    my $pythonPackage = SDB::Install::Package::Installable->new($pythonArchivePath, $pythonArchiveLst, $verifyChecksums);
    $pythonPackage->{installation} = $dummyInstallation;
    $pythonPackage->setMsgLstContext([undef, $self->getErrMsgLst()]);

    $self->getMsgLst()->addMessage("Extracting '$pythonArchivePath' to '$targetPythonDir'");
    if (!$pythonPackage->init($archiveDir) || !$pythonPackage->Extract()) {
        $self->getErrMsgLst()->addError("Failed to extract '$pythonArchivePath' to '$targetPythonDir'");
        return undef;
    }
    return File::Spec->catfile($targetPythonDir, 'Python', 'bin', 'python');
}

sub _getTempPythonDir {
    my ($self) = @_;
    my $targetDir = $self->getValue('Target');
    return File::Spec->catfile($targetDir, 'python_' . $self->{_kitversion});
}

sub _getTempPythonDirCfg {
    return {mode => 0755};
}

sub removeExtractedPython {
    my ($self) = @_;
    my $targetPythonDir = $self->_getTempPythonDir();
    return 1 if (!isDirectory($targetPythonDir));

    $self->getMsgLst()->addMessage("Removing '$targetPythonDir");
    return deltree($targetPythonDir, undef, $self->getErrMsgLst());
}

sub checkSystemRequirements{
	my ($self, $sysinfo, $manifest) = @_;
    $self->{sysinfo} = $sysinfo;
	my $result = 1;
	my $required_instruction_set = 'sse4.2';
	my $required_instruction_set_pattern = quotemeta ($required_instruction_set);

	if (defined $sysinfo->{processor_features} &&
		($sysinfo->{processor_features} !~ /$required_instruction_set_pattern/)){

		$self->appendErrorMessage ("CPU is not supported. Instruction set '$required_instruction_set' is missing.");
        if ($self->getIgnore ('check_platform')){
            $self->resetError ();
            $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
        }
        else{
            $result = undef;
        }
	}

    if ($sysinfo->isSles()) {
		$sysinfo->setMsgLstContext ($self->getMsgLstContext ());
        if (!$sysinfo->isSles12SpSupported()){
            if ($self->getIgnore ('check_platform')){
                $self->resetError ();
                $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
            }
            else{
                $result = undef;
            }
        }
    }
    elsif($sysinfo->isRedHat()) {
        $sysinfo->setMsgLstContext ($self->getMsgLstContext ());
        if (!$sysinfo->isRhel7SpSupported()){
            if ($self->getIgnore ('check_platform')){
                $self->resetError ();
                $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
            }
            else{
                $result = undef;
            }
        }
    }
    if (!defined $sysinfo->_checkGccRuntime($self, $manifest)){
        $result = undef;
    }

	if (!$isWin){
		return $result;
	}

	if (!defined $sysinfo->{version}){
		$self->AddWarning ("SysInfo.version is unknown: skipping platform check");
		return $result;
	}

	my ($major,$minor) = ($sysinfo->{version} =~ /NT\s+(\d+)\.(\d+)/);

	if (!defined $major){
		$self->AddWarning ("Cannot parse version string '$sysinfo->{version}': skipping platform check");
		return $result;
	}

	if (int ($major) < 6 ||
		int ($major) == 6 && int ($minor) < 1){
		$self->PushError ("At least MS Windows 7 / Windows Server 2008 R2 is required");
        if ($self->getIgnore ('check_platform')){
            $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
        }
        else{
            $result = undef;
        }
	}
	return $result;
}

sub getIgnoreValues{
	return [qw (check_diskspace check_min_mem)];
}

sub getHiddenIgnoreValues{
	return [qw (check_hardware check_platform scan_password_database check_hostname check_pmem)];
}

sub getTimeoutValues{
    return [qw (start_instance start_service)];
}

#-------------------------------------------------------------------------------
# Collect the SAP HANA systems and fills the specified array/hash references
# and returns an array of installed SIDs.

sub getNewDBInstallations {
    my ($self,
        $installedSystems,    # hash  reference to insert SAPSystem classes for update
        $pendingInstallations # array reference to insert SAPSystem classes
       ) = @_;

    my $savedSID    = $self->{current_sid};
    my $savedTarget = $self->{params}->{Target}->{value};

    my @installedSIDs = ();
    my $allSystems    = CollectSAPSystems();

    foreach my $currSID (sort keys %{$allSystems}) {

        my $currSystem = $allSystems->{$currSID};
        $self->{current_sid}               = $currSID;
        $self->{params}->{Target}->{value} = $currSystem->get_target();

        if ($self->pers_exists()) {
            push(@$pendingInstallations, $currSystem);
        }
        elsif ($currSystem->hasNewDB()) {
            push(@installedSIDs, $currSID);
            $installedSystems->{$currSID} = $currSystem;
        }
    }

    $self->{current_sid}               = $savedSID;
    $self->{params}->{Target}->{value} = $savedTarget;

    return @installedSIDs
}

#-------------------------------------------------------------------------------
# Writes a summary of the parameters onto the console and into the log file.
#
# Hides the parameter RemoteExecution in case of local work only.

sub displayParameterSummary {

    my ($self) = @_;

    if (!defined $self->getRemoteHosts()) {
        $self->{params}->{RemoteExecution}->{hidden} = 1; # hide summary output
    }

    $self->SUPER::displayParameterSummary();
}

sub shouldWarnIfCalledStandalone{
    my ($self) = @_;
    return $self->_calledStandaloneCriterion();
}

sub getExeName{
    return "hdbinst";
}

sub getResidentHdblcmPath{
    my ($self) = @_;
    return $self->_constructResidentHdblcmPath();
}

sub checkBasePathAccess {
    my($self, $value) = @_;

    if ($self->usesStorageApi()){
        return 1;
    }

    if (!-e $value){
        $self->PushError ("Path '$value' does not exist");
        return 0;
    }

    if (!-d _){
        $self->PushError ("Path '$value' is no directory");
        return 0;
    }

    # checking r-x of patent directories
    return 0 if(!$self->check_path_access ($value,0));

    return 1;

}

1;
