#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: MaxDB installation package (installable archive)


package SDB::Install::DebugPackage;

use SDB::Install::Archive;
use SDB::Install::Tools;
use strict;


our @ISA = qw (SDB::Install::Archive);

our $filelist = 'files.lst';


sub new{
	my $self = shift->SUPER::new (@_);
	$self->init ();
	return $self;
}



#
# open archive + read meta data
#

sub init{
	my ($self) = @_;
	
	unless (defined $self->open()){
		$self->AddError ('cannot open archive', $self);
		return undef;
	}
	
	my $next = $self->next ();
	
	unless (defined $next){
		return undef;
	}
	
	unless ($next->{name} eq 'PACKAGEDATA'){
		$self->AddError ('no meta data found');
		return undef;
	}
	
	my $data = $self->read_meta ();	
	
	unless (defined $data){
		return undef;
	}	

	$self->{meta_data} = $data;
		

	$self->initProperties ();

	my $file = $self->next ();
		
	if ($file->{name} eq $filelist) {
		$self->{files} = $self->read_filelist();
	}
	else{
		$self->AddError ('archive is not a valid MaxDB debug package');
		return undef;
	}
	return 1;
}

sub initProperties{
	my ($self) = @_;
	return 1 if $self->{initialized};
	my $packagedata = $self->{meta_data};
	$self->{name} = $packagedata->{NAME};
	$self->{id} = $packagedata->{ID};
	$self->{buildstring} = $packagedata->{BUILD_STRING};
	$self->{checksum}  = $packagedata->{CHECKSUM};
	$self->{size} = $packagedata->{SIZE};
	$self->{initialized} = 1;
	delete $self->{meta_data};
}

sub Reset{
	my ($self) = @_;
	delete $self->{not_extracted};
	$self->rewind();
	$self->next(); # PACKAGEDATA
	$self->next(); # filelist
}

sub Extract{
	my ($self,$path) = @_;

	if (defined $self->{not_extracted} and !%{$self->{not_extracted}}){
		return 1;
	}

	my $set_setuidroot = 0;

	my $files = $self->{files};

	unless (defined $self->{archive}){
		$self->AddError ('no archive defined');
		return undef;
	}
	
	unless (defined  $files){
		$self->AddError ('no files defined');
		return undef;
	}
	
	my $archive_handle = $self->GetArchiveHandle ();

	$archive_handle->SetDestDir ($path);

	if (!defined $self->{not_extracted}){
		my %not_extracted = %$files;
		$self->{not_extracted} = \%not_extracted;
	}
	
	my $filename;
	my $file;
	my $msg;
	my $reset_owner_group;
	my $rc;
	
	
	while (1){
		
		if (defined $self->{cur_file}){
			$file = $self->{cur_file};
		}else{
			$file = $self->next ();
			$self->{cur_file} = $file;
		}
		# error
		return undef unless (defined $file);
		if ($file->{mode} & 040000){
			# extract a directory
			$archive_handle->ExtractFile ();
			undef $self->{cur_file};
			next;
		}
				
		# eof reached
		last if not exists $file->{name};
		$filename = $file->{name};
		if (exists $self->{not_extracted}->{$filename}){
			$msg = $self->AddProgressMessage ('Extracting '.$file->{name});
			$reset_owner_group = 0;
			if ($set_setuidroot && $files->{$filename}->{setRootAsOwner}){
				$reset_owner_group = 1;
				$archive_handle->SetOwner (0, $self->{installation}->{gid});
				$files->{$filename}->{uid} = 0;
			}
			$rc = $archive_handle->ExtractFile ();
			unless (defined $rc and $rc == 0){
				$self->AddError (join (': ',$archive_handle->GetErr));
				return undef;
			}
			if ($reset_owner_group){
				$archive_handle->SetOwner ($self->{installation}->{uid},
								$self->{installation}->{gid});
			}
			$self->AddSubMessage(
				$msg, sprintf ("file info: mode = 0%o%s, size = %d bytes",
					 $file->{mode} & 07777,
					$reset_owner_group ? 
					', file owner is root' : '',
					 $file->{size}));
			delete $self->{not_extracted}->{$filename};
			undef $self->{cur_file};
		}
	}

	if (%{$self->{not_extracted}}){
		my $msg = $self->AddError ('could not extract all files: EOF reached');
		my @submsgs;
		foreach my $file (keys (%{$self->{not_extracted}})){	
			push @submsgs, $self->GenMsg ('ERR', "file '$file' not extracted");
		}
		$msg->{submsg} = \@submsgs;
		return undef;
	}
	return 1;
}



1;

