#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: 


package SDB::Install::DirectoryWalker;

use base SDB::Install::Base;
use SDB::Install::SysVars qw ($path_separator $isWin);
use SDB::Install::System qw(isAbsolute listDir);
use Fcntl ':mode';
use File::stat qw(stat lstat);

use strict;

sub new {
    my $self = shift->SUPER::new (); 
    (
        $self->{'actionMatcher'},            # optional, "match all" if undef,
                                             # reference of function/method,
                                             # "$self->{'action'}" is applied to matching files/dirs 
        $self->{'actionMatcherObj'},         # optional, if "$self->{'actionMatcher'}" refs a method, this is
                                             # the Object to call it on.
        $self->{'actionMatcherData'},        # optional, user defined data, passed to the "actionMatcher" callback
        
        $self->{'pruneMatcher'},             # optional, "match nothing" if undef,
                                             # matching dirs are not traversed
                                             # and the "$self->{'actionMatcher'}" and $self->{'action'} functions
                                             # are not performed on these.
        $self->{'pruneMatcherObj'},          # optional, if "$self->{'pruneMatcher'}" refs a method, this is
                                             # the Object to call it on.
        $self->{'pruneMatcherData'},         # optional, user defined data, passed to the "pruneMatcher" callback
                                   
        $self->{'action'},                   # optional, reference of function/method,
                                             # The function is applied to all files/dirs encountered which match
                                             # "$self->{'action'}".
        $self->{'actionObj'},                # optional, if "$self->{'action'}" refs a method, this is
                                             # the Object to call it on.
        $self->{'actionData'},               # optional, user defined data, passed to the "action" callback
                                   
        $self->{'breadthFirst'},             # optional, apply "$self->{'action'}" to files after applying it to their parent directories.
                                             # if not set, depth-first mode is used.

        $self->{'dontCollectProcessed'},     # optional, return reference to empty array instead of list
                                             # of inodes visited.
                                             
        $self->{'followSymlinks'}            # optional, by default we do not follow symlinks.
                                             # potentially dangerous (infinite loop on circular link structures,
                                             # unwanted deletions elsewhere, etc)
    #
    # the *Matcher and action callback functions/methods have these parameters:
    # (for convenience/performance, there is some redundancy)
    #     $errlst,       # msglist for errors
    #     $msglst,       # msglist for messages
    #     $userData,     # user defined data, passed as was set in the ctor for this callback
    #     $root,         # full path of the root dir
    #     $relentry,     # path of the file/dir in process, relative to $root
    #     $basename,     # the basename of the file/dir in process
    #     $isDir,        # 1 iff true, '' iff no dir, undef iff inexistent or error
    #     $statbufObj    # File::stat::stat object (backwards compatible with the previous version,
    #                      which was ordinary ref to the result of perl's stat callas these objects are just blesser array refs)
    #     $isSymlink     # 1 iff symlink, 0 otherwise
    #     $lstatbufObj   # File::stat::stat object (backwards compatible with the previous version,
    #                      which was ordinary ref to the result of perl's lstat call, as these objects are just blesser array refs)
    #                      undef if not a symlink
    # the *Matcher functions must return 1 or 0  to indicate match/nomatch and
    # undef on error (which terminates traversing).
    # the action callback must return 1 on success and 0 or undef on error
    # (which terminates traversing).
    #
    ) = @_;
    return $self;
}

#----------------------------------------------------------------------------

=functionComment

traverses a directory tree in the file system and performs a user defined action
on all inodes which match a user defined criterion.
It collects and returns a list of the inodes processed.
On error, this function returns undef.

note that it behaves like unix find: it does not process the root itself

=cut
sub findAndProcess {
    my (
        $self,
        $root # mandatory, root of tree to traverse
        
    ) = @_;
    if(length($root) > 1 ){
        $root =~ s/(\/|\\)+$//g; # remove trailing slashes
    }
    if($isWin) {
        $root =~ s/\//\\/g;
    }
    $self->{'result'} = [];

    my $rc = $self->findAndProcess_aux($root);
    if(!defined $rc) {
        return undef;
    }
    return $self->{'result'};
}

#----------------------------------------------------------------------------

=functionComment

iterates over a list of nodes in the file system and performs
a user defined action on all nodes which match a user defined criterion.
It collects and returns a list of the nodes processed.
If a list entry does not exist, it is ignored.
If a pruneMatcher is set, it is ignored.
On error, this function returns undef.

=cut

=disabled

#Although we have a concrete use case for this function, we had not yet the time to test and actually
#use it. So dont use it too, or test it (in principle it should work, but has never been run).

sub ProcessByList {
    my (
        $self,
        $listRef,   # mandatory, ref to array of file/dir paths
        $prefix     # optional, absolute(!) path to prepend to all
                    # relative elements of the list;
                    # if undef, all list elements must be absolute paths.

    ) = @_;
    if(defined $prefix) {
        $prefix =~ s/(\/|\\)+$//g; # remove trailing slashes
        if($isWin) { # sanity
            $prefix =~ s/\//\\/g;
        }
        if(!isAbsolute($prefix)) {
            $self->getErrMsgLst()->AddError ("Prefix '$prefix' is not an absolute path.");
            return undef;        
        }
    }
    else {
        $prefix = '';
    }
    $self->{'result'} = [];
    foreach my $entry (@{$listRef}) {
        if($isWin) {
            $entry =~ s/\//\\/g;
        }
        my $fullentry = $entry;
        if(defined $prefix && !isAbsolute($entry)) {
            $fullentry = $prefix.$path_separator.$entry;
        }
        my $statbufObj = undef;
        my $lstatbufObj = undef;
        my $isDir = undef;
        my $isSymlink = 0;
        if(-e $fullentry) {
            my @statbuf = stat(_);
            if (!@statbuf){
                $self->getErrMsgLst()->AddError ("Could not stat '$fullentry' : $!");
                return undef;
            }
            $statbufObj = \@statbuf;
            $isDir = -d _;  # 1 iff true, '' iff no dir, undef iff inexistent or error
        }
        if((not $isWin) && (-l $fullentry)) {
            my @lstatbuf = lstat($fullentry);
            if (!@lstatbuf){
                $self->getErrMsgLst()->AddError ("Could not lstat '$fullentry' : $!");
                return undef;
            }
            $lstatbufObj = \@lstatbuf;
            $isSymlink = 1;
        }
        my ($basename) = ($entry =~ /([^\/]+$)/);
        my $rc = $self->process_aux($prefix, $entry, $basename, $isDir, $statbufObj, $isSymlink, $lstatbufObj);
        if(not $rc) {
            return $rc;
        }
    }
    return $self->{'result'};
}
=cut

#----------------------------------------------------------------------------
# only "private" methods below this line.
#----------------------------------------------------------------------------

sub findAndProcess_aux {
    my (
        $self,
        $globalroot,
        $relroot
    ) = @_;
    my $root = $globalroot;
    if (defined $relroot){
        $root .= $path_separator . $relroot;
    }
    if(($self->{'followSymlinks'}) || (!SDB::Install::System::isLink($root))) {
        my $entries = SDB::Install::System::listDir($root);
        foreach my $entry (@{$entries}) {
            my $fullentry = $root.$path_separator.$entry;
            my $relentry = (defined $relroot) ? $relroot.$path_separator.$entry : $entry;
            my $isSymlink = !$isWin && SDB::Install::System::isLink($fullentry) ? 1 : 0;
            my $isDir = undef;
            my $entryMode = undef;
            my $stat_obj = File::stat::stat($fullentry);
            my $lstat_obj = undef;

            if ($stat_obj) {
                $isDir = S_ISDIR($stat_obj->mode());  # 1 iff true, '' iff no dir, undef iff inexistent or error
            }
            if ($isSymlink) {
                $lstat_obj = File::stat::lstat($fullentry);
                if (!$lstat_obj) {
                    $self->getErrMsgLst()->AddError ("Could not lstat '$fullentry' : $!");
                    return undef;
                }
            }
            if (!$stat_obj && !$lstat_obj) {
                # Neither existing file nor symlink - fail
                $self->getErrMsgLst()->AddError ("Could not stat '$fullentry' : $!");
                return undef;
            }

            if(defined $self->{'pruneMatcher'}) {
                my $rc;
                if(not defined $self->{'pruneMatcherObj'}) {
                    $rc = &{$self->{'pruneMatcher'}}($self->getErrMsgLst(),
                                                     $self->getMsgLst(),
                                                     $self->{'pruneMatcherData'},
                                                     $globalroot,
                                                     $relentry,
                                                     $entry,
                                                     $isDir,
                                                     $stat_obj,
                                                     $isSymlink,
                                                     $lstat_obj);
                }
                else {
                    $rc = &{$self->{'pruneMatcher'}}($self->{'pruneMatcherObj'},
                                                     $self->getErrMsgLst(),
                                                     $self->getMsgLst(),
                                                     $self->{'pruneMatcherData'},
                                                     $globalroot,
                                                     $relentry,
                                                     $entry,
                                                     $isDir,
                                                     $stat_obj,
                                                     $isSymlink,
                                                     $lstat_obj);
                }
                if(not defined $rc) {
                    return undef;
                }
                if($rc) {
                    next;
                }
            }
            if($self->{'breadthFirst'}) {
                my $rc = $self->process_aux($globalroot,
                                            $relentry,
                                            $entry,
                                            $isDir,
                                            $stat_obj,
                                            $isSymlink,
                                            $lstat_obj);
                if(not $rc) {
                    return $rc;
                }
            }
            if($isDir) {
                my $rc = $self->findAndProcess_aux($globalroot, $relentry);
                if(not $rc) {
                    return $rc;
                }
            }
            if(not $self->{'breadthFirst'}) {
                my $rc = $self->process_aux($globalroot,
                                            $relentry,
                                            $entry,
                                            $isDir,
                                            $stat_obj,
                                            $isSymlink,
                                            $lstat_obj);
                if(not $rc) {
                    return $rc;
                }
            }
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub process_aux {
    my (
        $self,
        $globalroot,
        $relentry,
        $entry,
        $isDir,
        $statbufObj,
        $isSymlink,
        $lstatbufObj
    ) = @_;
    my $doIt = 1;
    if(defined $self->{'actionMatcher'}) {
        my $rc;
        if(not defined $self->{'actionMatcherObj'}){
            $rc = &{$self->{'actionMatcher'}}($self->getErrMsgLst(),
                                              $self->getMsgLst(),
                                              $self->{'actionMatcherData'},
                                              $globalroot,
                                              $relentry,
                                              $entry,
                                              $isDir,
                                              $statbufObj,
                                              $isSymlink,
                                              $lstatbufObj);
        }
        else {
            $rc = &{$self->{'actionMatcher'}}($self->{'actionMatcherObj'},
                                              $self->getErrMsgLst(),
                                              $self->getMsgLst(),
                                              $self->{'actionMatcherData'},
                                              $globalroot,
                                              $relentry,
                                              $entry,
                                              $isDir,
                                              $statbufObj,
                                              $isSymlink,
                                              $lstatbufObj);
        }
        if(not defined $rc) {
            return undef;
        }
        if(not $rc) {
            $doIt = 0;
        }
    } 
    if($doIt) {
        if(defined $self->{'action'}) {
            my $rc;
            if(not defined $self->{'actionObj'}) {
                $rc = &{$self->{'action'}}($self->getErrMsgLst(),
                                           $self->getMsgLst(),
                                           $self->{'actionData'},
                                           $globalroot,
                                           $relentry,
                                           $entry,
                                           $isDir,
                                           $statbufObj,
                                           $isSymlink,
                                           $lstatbufObj);
            }
            else {
                $rc = &{$self->{'action'}}($self->{'actionObj'},
                                           $self->getErrMsgLst(),
                                           $self->getMsgLst(),
                                           $self->{'actionData'},
                                           $globalroot,
                                           $relentry,
                                           $entry,
                                           $isDir,
                                           $statbufObj,
                                           $isSymlink,
                                           $lstatbufObj);
            }
            if(not $rc) {
                return undef;
            }
        }
        if(not $self->{'dontCollectProcessed'}) {
            push @{$self->{'result'}}, $relentry;
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

1;
