package SDB::Install::Group;

use SDB::Install::User; # => load BackSlasher class

use strict;

use SDB::Install::System qw (exec_program);
use SAPDB::Install::System::Win32::API;
use SDB::Install::SysVars;

our @ISA = qw (SDB::Install::BaseLegacy);


sub getSid{
    my ($self) = @_;
    if (defined $self->{sid}){
        return $self->{sid};
    }

    my ($rc,$sid,$name,$type);

    if ($self->{bsname}->{wellknown}){
        if (lc $self->{bsname}->{second} eq 'administrators'){
            ($name, $sid) = LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
                                    ,DOMAIN_ALIAS_RID_ADMINS);
        }
        elsif (lc $self->{bsname}->{second} eq 'users'){
            ($name,$sid) = 	LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
                                    ,DOMAIN_ALIAS_RID_USERS);
        }
        else{
            die "wellknown $self->{bsname}->{second} is UNKNOWN";
        }


        my $newbsname = new BackSlasher ($name);

        if (!$newbsname->equals ($self->{bsname})){
            $self->{bsname} = $newbsname;
            $self->{name} = $name;
        }
    }
    else{
         ($rc,$sid,$type) = SAPDB::Install::System::Win32::API::lookupAccount ($self->{bsname}->repr(), undef);
    }
    $self->{sid} = $sid;
    return $sid;
}


sub new{
    my $self = shift->SUPER::new ();
    ($self->{name}) = @_;
    if (!defined $self->{name}){
        if ($isWin){
            $self->AddError ("Empty group name not supported");
        }
        else{
            $self->{name} = getgrgid ( $) );
            $self->{gid} = $);
        }
    }
    elsif (!$isWin){
        $self->{gid} = getgrnam ($self->{name});
    }

    if ($isWin){
        $self->{bsname} = new BackSlasher ($self->{name});
        $self->getSid ();
    }
    return $self;
}


sub existsUx{
    defined $_[0]->{gid};
}

sub existsWin{
    return defined $_[0]->{sid};
    my ($self) = @_;

    #return SAPDB::Install::System::Win32::API::ExistsGroup ($self->{bsname}->{concat});

    my $rc;

    if ($self->{bsname}->{local}){
        ($rc) = SAPDB::Install::System::Win32::API::infoLocalGroup ($self->{bsname}->{second});
    }
    else{
        ($rc) = SAPDB::Install::System::Win32::API::infoGroup ($self->{bsname}->{second},undef,$self->{bsname}->{first});
    }
    return $rc == 0;
}

sub createUx{
    my ($self,$comment,$id) = @_;

    if ($self->exists() ){
        return 1;
    }

    my $cmd = 'groupadd';
    my @args = ((defined $id ? ('-o') : ()), $self->{name});

    if (defined $id){
        push @args, ('-g', $id);
    }

    my $msg = $self->AddMessage ("Creating group \"$self->{name}\"");

    my $msglst = new SDB::Install::MsgLst ();


    my $rc;

    {
        local %ENV = %ENV;
        $ENV{'PATH'} = '/sbin:/usr/sbin:'  . $ENV{PATH};
        $rc = exec_program ($cmd, \@args, $msglst);
    }

    if ($rc != 0){
        $self->AddError ('Cannot create group', $msglst);
        $self->AddSubMsgLst ($msg, $msglst);
        return undef;
    }
    $self->AddSubMsgLst ($msg, $msglst);

    my $gid = getgrnam ($self->{name});
    my $tries = 15;
    my $count = 0;
    while (!defined $gid && $tries--){
        $self->AddSubMessage ($msg, sprintf ('Getting gid of %s (%d)', $self->{name}, ++$count));
        $gid = getgrnam ($self->{name});
        last if defined $gid;
        sleep (1);
    }
    if (!defined $gid){
        $self->AddError ("Cannot get gid: tried getgrent() several times to get the gid after successful $cmd");
        return undef;
    }

    $self->{gid} = $gid;
    return 1;
}

sub createWin{
    my ($self, $comment) = @_;

    my $rc;

    if ($self->{bsname}->{local}){
        $rc = SAPDB::Install::System::Win32::API::AddLocalGroup (
                $self->{bsname}->{second},$comment);
    }
    else{
        $rc = SAPDB::Install::System::Win32::API::createGroup (
                $self->{bsname}->{second},
                $comment,
                undef,
                $self->{bsname}->{first});
    }

    if ($rc != 0){
        return undef;
    }

    $self->getSid ();

    return 1;
}

sub deleteUx{
    my ($self) = @_;

    if ( !$self->exists() ){
        return 1;
    }

    my $cmd = 'groupdel';
    my @args = ($self->{name});

    my $msg = $self->AddMessage ("Deleting group \"$self->{name}\"");

    my $msglst = new SDB::Install::MsgLst ();

    my $rc;

    {
        local %ENV = %ENV;
        $ENV{'PATH'} = '/sbin:/usr/sbin:'  . $ENV{PATH};
        $rc = exec_program ($cmd, \@args, $msglst);
    }

    if ($rc != 0){
        $self->AddError ('Cannot delete group', $msglst);
        $self->AddSubMsgLst ($msg, $msglst);
        return undef;
    }
    $self->AddSubMsgLst ($msg, $msglst);
    undef $self->{gid};
    return 1;
}

sub deleteWin{
    my ($self) = @_;

    my $rc;

    if ($self->{bsname}->{local}){
        $rc = SAPDB::Install::System::Win32::API::RemoveLocalGroup (
            $self->{bsname}->{second});
    }
    else{
        $rc = SAPDB::Install::System::Win32::API::deleteGroup (
                $self->{bsname}->{second}, undef, $self->{bsname}->{first});
    }
    if ($rc == 0){
        undef $self->{sid};
        return 1;
    }
    return undef;
}


sub getMembersUx{
    my ($self, $atLeastOne) = @_;
    my @users;
    setpwent();
    while (my ($name,$passwd,$uid,$gid) = getpwent ()){
        if ($gid == $self->{gid}){
            push @users, $name;
            if ($atLeastOne){
                last;
            }
        }
    }
    endpwent();
    return \@users;
}

sub getMembersWin{
    my ($self) = @_;
    if ($self->{bsname}->{local}){
        my ($rc,$members) = SAPDB::Install::System::Win32::API::getLocalGroupMembers ($self->{bsname}->{second});
        my @result;
        if ($rc == 0){
            #
            # TODO: filter already deleted users and groups
            #		in getLocalGroupMembers()
            #
            @result = grep {!/\\$/} @$members;
            return \@result;
        }
    }
    return undef;
}


if ($isWin){
    *exists = \&existsWin;
    *create = \&createWin;
    *delete = \&deleteWin;
    *getMembers = \&getMembersWin;
    sub has{
        my ($self, $prospect) = @_;
        my ($rc,$groups) = SAPDB::Install::System::Win32::API::getGroups ($prospect);

        foreach my $group (@$groups){
            if (lc ($group) eq $self->{bsname}->{concat}){
                return 1;
            }
        }

        return 0;
    }
}
else{
    *exists = \&existsUx;
    *create = \&createUx;
    *delete = \&deleteUx;
    *getMembers = \&getMembersUx;
    sub id {$_[0]->{gid}}
}

1;
