package SDB::Install::IniFile;

use base 'SDB::Install::BaseLegacy';
use SDB::Install::SysVars qw ($isWin);
use SDB::Install::Tools qw (trim ltrim rtrim getFirstLowercaseMatch);
use SDB::Install::Globals qw ($gBoolTruePattern);
use strict;
#use SDB::Install::DebugUtilities;

sub new{
    my $self = shift->SUPER::new ();
    (
        $self->{fullname},
        $self->{_uid},
        $self->{_gid},
        $self->{_mode}
    ) = @_;
    if (-f $self->{fullname}){
        $self->read();
    }
    return $self;
}

sub read{
	my ($self) = @_;
	if (defined $self->{_buffer}){
		return 1;
	}
	if (!open (FD, $self->{fullname})){
		$self->AddError ("Cannot open file '$self->{fullname}'");
		return undef;
	}
	$self->{_mtime} = (stat ($self->{fullname}))[9];
	$self->{_changed} = 0;
	if (!$isWin){
		$self->{_mode} = (stat (_))[2];
		$self->{_uid} = (stat (_))[4];
		$self->{_gid} = (stat (_))[5];
	}
	my @buffer = <FD>;
	close (FD);
	$self->{_buffer} = \@buffer;
	if (!$self->parse()){
		$self->AddError ("Cannot parse '$self->{fullname}'");
		return undef;
	}
	return 1;
}

sub parse{
	my ($self) = @_;
	if (defined $self->{data}){
		return 1;
	}
	my %data;
	my ($line, $key, $value);
	my $section = '';
	foreach my $lineNr (0.. (scalar (@{$self->{_buffer}}) - 1)){
		$line = $self->{_buffer}->[$lineNr];
		chomp ($line);
		if ($line =~/^\s*#/){
			next;
		}
		trim (\$line);
		if ($line =~/^\[.*\]$/){
			($section) = ($line =~/^\[(.*)\]$/);
			$section = $self->_getRealSection($section);
			if (!defined $data{$section}){
				$data{$section} = [$lineNr, {}];
			}
			next;
		}
		if ($line =~ /\S.*=/){
			($key,$value) = $self->_parseKeyValuePair($line, $section);
			if ($key){
				if (!defined $data{$section}->[1]->{$key}){
					$data{$section}->[1]->{$key} = [];
				}
				$data{$section}->[1]->{$key}->[0] = $lineNr;
				$data{$section}->[1]->{$key}->[1] = $value;
			}
		}
	}
	$self->{data} = \%data;
	return 1;
}

sub _parseKeyValuePair {
	my ($self, $line, $section) = @_;
	my ($key,$value) = ($line =~ /^([^=]+)=(.*)/);
	rtrim (\$key);
	ltrim (\$value);

	if ($key) {
		$key = $self->_getRealKey($section, $key);
		return ($key, $value);
	}
	return ();
}

sub getLastSectionValueIndex{
	my ($self, $section) = @_;
    $section = $self->_getRealSection($section // '');

	if (defined $self->{data}->{$section}){
		if (!%{$self->{data}->{$section}->[1]}){
			# section has no values
			# return section line number
			return $self->{data}->{$section}->[0];
		}
		my $section_hash = $self->{data}->{$section}->[1];
		my @sorted_keys = sort {$section_hash->{$a}->[0] <=> 
			$section_hash->{$b}->[0]} keys %{$section_hash};
		return $section_hash->{$sorted_keys[$#sorted_keys]}->[0];
	}
	return undef;
}

sub syncLineIndices{
	my ($self, $lineNr, $inc) = @_;
	if (!defined $inc){
		$inc = 1;
	}
	my ($section_hash, $key);
	foreach my $section (keys %{$self->{data}}){
		if ($self->{data}->{$section}->[0] >= $lineNr){
			$self->{data}->{$section}->[0] += $inc;
		}
		$section_hash = $self->{data}->{$section}->[1];
		foreach $key (keys %$section_hash){
			if ($section_hash->{$key}->[0] >= $lineNr){
				$section_hash->{$key}->[0] += $inc;
			}
		}
	}
	return 1;
}

sub setValue{
    my ($self, $section, $key, $value, $comment) = @_;
    my $index;
    my $isArrayValue = 0;
    my $isHashValue  = 0;
    my $outputValue  = '';
    $section = $self->_getRealSection($section // '');
    $key = $self->_getRealKey($section, $key);

    if (defined $value) {
        if (ref($value) =~ /ARRAY/) {
            $isArrayValue = 1;
        }
        elsif (ref($value) =~ /HASH/) {
            $isHashValue = 1;
        }
        else {
            $outputValue = $value;
        }
    }
    if (!defined $self->{_buffer}){
        $self->{_buffer} = [];
    }
    if (!defined $self->{data}->{$section}){
        $self->AddMessage ("Creating section '$section'");
        $index = scalar @{$self->{_buffer}};
        push @{$self->{_buffer}}, "\n";
        $index++;
        push @{$self->{_buffer}}, "[$section]\n";
        $self->{data}->{$section} = [$index, {}];
        if ($comment){
            $index++;
            push @{$self->{_buffer}}, "\n";
            $index++;
            push @{$self->{_buffer}}, "# $comment\n";
        }
        if ($isArrayValue) {
            foreach my $currVal (@$value) {
                $index++;
                push @{$self->{_buffer}}, "$key=$currVal\n";
            }
        }
        elsif ($isHashValue) {
            foreach my $valKey (sort keys %$value) {
                $index++;
                push @{$self->{_buffer}},
                     "$key=$valKey=" . $value->{$valKey} . "\n";
            }
        }
        else {
            $index++;
            push @{$self->{_buffer}}, "$key=$outputValue\n";
        }
        $self->AddMessage ("Adding value '$key' = '$outputValue' to section '$section'");
        $self->{data}->{$section}->[1]->{$key} = [$index, $value];
        $self->{_changed} = 1;
        return 1;
    }
    if (!defined $self->{data}->{$section}->[1]->{$key}){
        $index = $self->getLastSectionValueIndex ($section);

        if (!defined $index){
            die ('Internal error');
        }

        $self->AddMessage ("Adding value '$key' = '$outputValue' to section '$section'");
        my @linesToAdd = ();

        if ($comment){
            push(@linesToAdd, "\n", "# $comment\n");
        }
        if ($isArrayValue) {
            for my $currVal (@$value) {
                push(@linesToAdd, "$key=$currVal\n");
            }
        } elsif ($isHashValue) {
            for my $valKey (sort keys %$value) {
                push(@linesToAdd, "$key=$valKey=" . $value->{$valKey} . "\n");
            }
        } else {
            push(@linesToAdd, "$key=$outputValue\n");
        }
        $self->syncLineIndices ($index + 1, scalar(@linesToAdd));
        for my $additionalLine (@linesToAdd){
            splice (@{$self->{_buffer}}, ++$index, 0, $additionalLine);
        }
        $self->{data}->{$section}->[1]->{$key} = [$index, $value];
        $self->{_changed} = 1;
        return 1;
    }
    my $old_value = $self->{data}->{$section}->[1]->{$key}->[1];
    if (defined $old_value && defined $value && ($old_value eq $value)) {
        $self->AddMessage ("Value '$key' in section '$section' is up to date. ('$outputValue')");
        return 1;
    }
    $self->AddMessage ("Updating value '$key' in section '$section' ('$old_value' => '$outputValue')");
    $self->{data}->{$section}->[1]->{$key}->[1] = $value;
    $self->{_buffer}->[$self->{data}->{$section}->[1]->{$key}->[0]] = "$key=$outputValue\n";
    $self->{_changed} = 1;
    return 1;
}

sub removeKey{
    my ($self, $section, $key) = @_;
    my $index;
    $section = $self->_getRealSection($section // '');
    $key = $self->_getRealKey($section, $key);

    my $msg = $self->AddMessage ("Removing key '$key' in section '$section'");

    if (!defined $self->{data}->{$section}){
        $self->AddSubMessage ($msg, "Section '$section' is already gone.");
        return 1;
    }

    if (!defined $self->{data}->{$section}->[1]->{$key}){
        $self->AddSubMessage ($msg, "Key '$key' in section '$section' is already gone.");
        return 1;
    }
    my $value = $self->{data}->{$section}->[1]->{$key}->[1];

    $index = $self->{data}->{$section}->[1]->{$key}->[0];
    delete $self->{data}->{$section}->[1]->{$key};
    splice (@{$self->{_buffer}}, $index, 1);
    $self->syncLineIndices ($index + 1, -1);
    $self->AddSubMessage ($msg, "Done (value was '$value').");
    $self->{_changed} = 1;
    return 1;
}

sub getValue{
    my ($self, $section, $key) = @_;
    $section = $self->_getRealSection($section // '');
    $key = $self->_getRealKey($section, $key);

    if (defined $self->{data} && defined $self->{data}->{$section} &&
        defined $self->{data}->{$section}->[1]->{$key}){
        return $self->{data}->{$section}->[1]->{$key}->[1];
    }
    return undef;
}

sub getBoolValue {
    my ($self, $section, $key) = @_;
    my $value = $self->getValue($section, $key);
    return undef if (!defined $value);
    return $value =~ /$gBoolTruePattern/i;
}

sub getSections{
    my ($self) = @_;
    if (defined $self->{data}){
        return [keys %{$self->{data}}];
    }
    return [];
}

sub getKeys{
    my ($self, $section) = @_;
    $section = $self->_getRealSection($section);

    if (defined $self->{data} && defined $self->{data}->{$section}){
        return [keys %{$self->{data}->{$section}->[1]}];
    }
    return [];
}

sub getSection {
    my ($self, $section) = @_;
    $section = $self->_getRealSection($section // '');

    if (defined $self->{data} && defined $self->{data}->{$section}){
        return $self->{data}->{$section}->[1];
    }
    return undef;
}

sub write{
	my ($self) = @_;
	if (!$self->{_changed}){
		$self->AddMessage ("File '$self->{fullname}' is up to date");
		return 2;
	}

	if ($self->{_mtime} != (stat ($self->{fullname}))[9]){
		$self->AddError ("File '$self->{fullname}' changed in the meantime");
		return undef;
	}
	
	my $bak_file = $self->{fullname} . '.bak';
	
	if (-f $bak_file){
		unlink ($bak_file);
	}
	
	rename ($self->{fullname}, $bak_file);
	
	if (!open (FD, '>'.$self->{fullname})){
		$self->AddError ("Cannot create inifile '$self->{fullname}': $!");
		rename ($bak_file, $self->{fullname});
		return undef;
	}
	if (! print FD @{$self->{_buffer}}){
		$self->AddError ("Cannot write ini file '$self->{fullname}': $!");
		close (FD);
		unlink ($self->{fullname});
		rename ($bak_file, $self->{fullname});
		return undef;
	}
	close (FD);
	unlink ($bak_file);
	if (!$isWin){
        if (defined $self->{_mode}){
            chmod ($self->{_mode}, $self->{fullname});
        }
        if ($> == 0 && (defined $self->{_uid} || defined $self->{_gid})){
            my $uid = defined $self->{_uid} ? $self->{_uid} : -1;
			my $gid = defined $self->{_gid} ? $self->{_gid} : -1;
            chown ($uid, $gid, $self->{fullname});
		}
	}
	$self->{_changed} = 0;
	$self->{_mtime} = (stat ($self->{fullname}))[9];
	$self->AddMessage ("File '$self->{fullname}' updated");
	return 1;
}

sub _getRealSection {
    my ($self, $section) = @_;
    return getFirstLowercaseMatch($self->getSections(), $section) // $section;
}

sub _getRealKey {
    my ($self, $section, $key) = @_;
    return getFirstLowercaseMatch($self->getKeys($section), $key) // $key;
}

1;