#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: HDB Installer

package SDB::Install::Installer;
use SDB::Install::Base;
use SAPDB::Install::MD5Sum;
use SAPDB::Install::Version;
use SDB::Install::SysVars qw ($installerPlatformName $isWin $path_separator $isApple);
use File::Spec;
use strict;
our @ISA = qw (SDB::Install::Base);


our $pattern_path_separator = quotemeta ($path_separator);

our $pattern_multiple_path_separators = $pattern_path_separator . '+';

our $sid_pattern = '[A-Z][A-Z0-9][A-Z0-9]';

our @server_runtime_dir_nodes = qw (global hdb install bin instruntime$);
our @server_hdblcm_runtime_dir_nodes = qw (hdblcm instruntime$);

our $kit_runtime_dir_pattern = $pattern_path_separator . 'instruntime$';

our $server_runtime_dir_pattern = join ($pattern_multiple_path_separators , '' ,
    @server_runtime_dir_nodes);

our $server_hdblcm_dir_pattern = join ($pattern_multiple_path_separators,
                                       '', @server_hdblcm_runtime_dir_nodes);

our @instruntime_dir_nodes = $isApple ?
                            ('instruntime', 'sdbrun.app', 'Contents', 'MacOS')
                            :
                            ('instruntime');

our $instruntime_pattern = join ($pattern_multiple_path_separators,'', @instruntime_dir_nodes) . $pattern_path_separator . '*$';

our $no_server_instpath_pattern = $isApple ? 
    join ($pattern_multiple_path_separators, '(.*)', 'install(','instruntime', 'sdbrun.app', 'Contents', 'MacOS)?$') :
    '(.*)' . $pattern_multiple_path_separators . 'install(' . $pattern_multiple_path_separators . 'instruntime)?$';

our $usr_sap_pattern = join ($pattern_multiple_path_separators, ($isWin ? '^[a-z]:' : '^'), 'usr', 'sap');

our $unc_pattern = '^' . ($pattern_path_separator x 2) .
    "[^$pattern_path_separator]+" . $pattern_multiple_path_separators .
    "[^$pattern_path_separator]+";

#
# matches [c:]/usr/sap/(sid)/SYS/global/hdb/install/bin/instruntime
# returns sid
#
sub getServerUsrSapSidPattern
{
    my $server_usr_sap_sid_pattern =  join ($pattern_multiple_path_separators, $usr_sap_pattern,
            '(' . $sid_pattern. ')' , 'SYS', @server_runtime_dir_nodes);
    return $server_usr_sap_sid_pattern;
}

#
# matches [c:]/usr/sap/(sid)/hdblcm/instruntime
# returns sid
#
sub getServerHdblcmUsrSapPattern
{
    my $server_usr_sap_sid_pattern =  join ($pattern_multiple_path_separators, $usr_sap_pattern,
            '(' . $sid_pattern. ')' , @server_hdblcm_runtime_dir_nodes);
    return $server_usr_sap_sid_pattern;
}

#-------------------------------------------------------------------------------
# linux matches (sapmnt)/(sid)/global/hdb/install/install
# windows matches (\\<hostname>\<share>)\<sid>\SYS\global\hdb\install\bin\instruntime
# returns sapmnt/network share, sid
#
sub getServerSapmntSidPattern
{
    return  $isWin ?
        join ($pattern_multiple_path_separators, '(' . $unc_pattern . ')',
            '(' . $sid_pattern. ')' , 'SYS', @server_runtime_dir_nodes)
        :
        join ($pattern_multiple_path_separators, '^(.*)',
            '(' . $sid_pattern. ')' , @server_runtime_dir_nodes);
}

#-------------------------------------------------------------------------------
# linux matches (sapmnt)/(sid)/hdblcm/instruntime
# windows matches (\\<hostname>\<share>)\<sid>\SYS\hdblcm\instruntime
# returns sapmnt/network share, sid
#
sub getServerHdblcmSapmntSidPattern
{
    return $isWin ?
        join ($pattern_multiple_path_separators, '(' . $unc_pattern . ')',
            '(' . $sid_pattern. ')' , 'SYS', @server_hdblcm_runtime_dir_nodes)
        :
        join ($pattern_multiple_path_separators, '^(.*)',
            '(' . $sid_pattern. ')' , @server_hdblcm_runtime_dir_nodes);
}

sub new{
	my $self = shift->SUPER::new(@_);
	$self->init ();
	return $self;
}


#-------------------------------------------------------------------------------
# Initializes this class.
#
# When starting an installer program, the program name and path is detected by
# perl/sdbrun/libSDBRun.c and inserted into the hash
# '$SAPDB::Install::Config{ProgramName}' and '...Config{RuntimeDir}' .
# There is not class named '$SAPDB::Install::Config'

sub init {
	my ($self) = @_;

    $self->{version}     = $SAPDB::Install::Version::Version;
    $self->{buildstring} = $SAPDB::Install::Version::Buildstring;
    $self->{program}     = $SAPDB::Install::Config{ProgramName};
    $self->{runtimedir}  = $SAPDB::Install::Config{RuntimeDir};
    $self->{callerdir}   = $SAPDB::Install::Config{CallerDir};
    $self->{git_hash}    = $SAPDB::Install::Version::GitHash;

    if ($isWin){
        $self->{runtimedir} =~ s/\//\\/g  if (defined $self->{runtimedir});
        $self->{callerdir}  =~ s/\//\\/g  if (defined $self->{callerdir});
        # additional if-parts avoid Eclipse warnings under Windows
    }
}

sub _renameInternalInstallerPath {
    my ($self, $oldPath, $oldSid, $newSid, $target) = @_;
    local $path_separator = quotemeta($path_separator);
    my $sid_pattern = qr/${path_separator}${oldSid}(?=\z|${path_separator})/;
    my $new_sid     = File::Spec->catfile('', $newSid);
    my $renamedPath = $oldPath;
    $target =~ s[/+$][]; # Remove trailing slashes

    my $targetPattern = quotemeta($target);
    if (my ($partToRename) = ($oldPath =~ /^$targetPattern(.*)$/)) {
        if ($partToRename =~ s/$sid_pattern/${new_sid}$1/g) {
            $renamedPath = $target.$partToRename;
        }
    } else {
        $renamedPath =~ s/$sid_pattern/${new_sid}$1/g;
    }
    return $renamedPath;
}

sub _adaptINCPaths {
    my ($self, $oldSid, $newSid, $target) = @_;
    my $changed = 0;

    for my $i (0..$#INC){
        my $oldPath = $INC[$i];
        my $renamedPath = $self->_renameInternalInstallerPath($oldPath, $oldSid, $newSid, $target);
        if ($renamedPath ne $oldPath) {
            $self->getMsgLst()->addMessage ("Changing \@INC item '$oldPath' =>'$renamedPath'");
            $INC[$i] = $renamedPath;
            $changed = 1;
        }
    }

    return $changed;
}

sub _adaptGlobalConfigProperty {
    my ($self, $oldSid, $newSid, $target, $property) = @_;
    my $oldPath = $SAPDB::Install::Config{$property};
    return 0 if !defined($oldPath);

    my $renamedPath = $self->_renameInternalInstallerPath($oldPath, $oldSid, $newSid, $target);
    if ($renamedPath ne $oldPath){
        $self->getMsgLst()->addMessage ("Changing global $property '$oldPath' =>'$renamedPath'");
        $SAPDB::Install::Config{$property} = $renamedPath;
        return 1;
    }
    return 0;
}

sub renameHanaSID{
    my ($self, $oldSid, $newSid, $target) = @_;
    my $changed = $self->_adaptINCPaths($oldSid, $newSid, $target);
    $changed = $self->_adaptGlobalConfigProperty($oldSid, $newSid, $target, 'RuntimeDir')    || $changed;
    $changed = $self->_adaptGlobalConfigProperty($oldSid, $newSid, $target, 'RawRuntimeDir') || $changed;
    $changed = $self->_adaptGlobalConfigProperty($oldSid, $newSid, $target, 'CallerDir')     || $changed;

    if ($changed){
        $self->init();
    }
    return $changed;
}

sub appendInstallerInfo{
    my ($self, $msglst) = @_;

    if (!defined $msglst){
        $msglst = $self->getMsgLst ();
    }

    if ($self->isInstalled) {
        $msglst->addMessage ('using installed Installer');
    } else {
        $msglst->addMessage ('Installer is part of installation kit');
    }

    $msglst->addMessage('Version is '.$self->{buildstring});
    $msglst->addMessage('GitHash is '.$self->{git_hash});
    $msglst->addMessage('Platform is '.$installerPlatformName);
    $msglst->addMessage('Perl version is '. $^V);
}


#------------------------------------------------------------------

sub isInstalled{
    my ($self) = @_;
    my $instRuntimeDir = $self->getInstRuntimeDir ();
    return defined $self->{runtimedir}
           &&
           (($self->{runtimedir} =~ /$server_runtime_dir_pattern/) ||
            ($self->{runtimedir} =~ /$server_hdblcm_dir_pattern/) ||
            (($self->{runtimedir} =~ /$no_server_instpath_pattern/) &&
             ((-f $instRuntimeDir .$path_separator.'INSTREG') || (-f $instRuntimeDir .$path_separator.'..'.$path_separator.'INSTREG') )));
}

#------------------------------------------------------------------

sub isServerInstaller{
    my ($self) = @_;
    return ($self->{runtimedir} =~ /$server_runtime_dir_pattern/) ||
           $self->isServerHdblcmInstaller();
}

#------------------------------------------------------------------

sub isServerHdblcmInstaller {
    return $_[0]->{runtimedir} =~ /$server_hdblcm_dir_pattern/;
}


#------------------------------------------------------------------
# Find out if the installer program is part of an installation.
#
# In case of rename without master/slave status the parameter $isMasterRename
# has to be set true.
#
# Parameters: $msgLst         SDB::Install::MsgLst
#             $isMasterRename boolean

sub getInstallation{
    my ($self, $msgLst, $isMasterRename) = @_;

    if (!$self->isInstalled ()){
        $self->setErrorMessage ("Installer is not part of an installation");
        return undef;
    }

    if ($self->isServerInstaller ()){

        my $server_usr_sap_sid_pattern = ($self->isServerHdblcmInstaller)
                                         ? $self->getServerHdblcmUsrSapPattern()
                                         : $self->getServerUsrSapSidPattern();
        my $server_sapmnt_sid_pattern  = ($self->isServerHdblcmInstaller)
                                         ? $self->getServerHdblcmSapmntSidPattern()
                                         : $self->getServerSapmntSidPattern();
        my ($sid,$sapmnt);
        if ($isWin){
            ($sid) = ($self->{runtimedir} =~ /$server_usr_sap_sid_pattern/i);
            if (defined $sid){
                $sid = uc ($sid);
            }
        }
        else{
            ($sid) = ($self->{runtimedir} =~ /$server_usr_sap_sid_pattern/);
        }

        if (!defined $sid){
            if ($isWin){
                ($sapmnt, $sid) = ($self->{runtimedir} =~ /$server_sapmnt_sid_pattern/i);
                if (defined $sid){
                    $sid = uc ($sid);
                }
            }
            else{
                ($sapmnt,$sid) = ($self->{runtimedir} =~ /$server_sapmnt_sid_pattern/);
            }
        }
        if (defined $sid){
            my $sapSys = undef;
            require SDB::Install::SAPSystem;
            if (defined $sapmnt){
                if ($isMasterRename) {
                    $sapSys =
                        $self->getSapSysUsingTargetSID($msgLst, $sid, $sapmnt);
                }
                if (!defined $sapSys) { # no target SID for rename detected
                    $sapSys = new SDB::Install::SAPSystem();
                    $sapSys->initWithGlobDir($sid, $sapmnt);
                }
                return $sapSys;
            }
            else{
                my $systems = SDB::Install::SAPSystem::CollectSAPSystems();
                $sapSys  = $systems->{$sid};
                if (defined $sapSys){
                    return $sapSys;
                }
            }
        }
    }
    else{
        # no server
        my ($path, undef) = ($self->{runtimedir} =~ /$no_server_instpath_pattern/i);
        if (defined $path){
            require SDB::Install::Installation;
            my $installation = new SDB::Install::Installation ($path);
            if (defined $installation->getVersionByManifest()){
                return $installation;
            }
        }
    }
    return undef;
}

#-------------------------------------------------------------------------------
# Returns $sapSys (an instance of SDB::Install::SAPSystem)
# if an installation directory with a target SID already exists
# and the content within this directory is not yet renamed.

sub getSapSysUsingTargetSID {

    my ($self, $msgLst, $sid, $sapmnt) = @_;

    my $sapSys           = undef;
    my $pathSapmntSid    = join ($path_separator, $sapmnt, $sid);
    my $pathUsrSapSid    = join ($path_separator, '', 'usr', 'sap', $sid);
    my $pathUsrSapSidSys = join ($path_separator, $pathUsrSapSid, 'SYS');
    my $mayBeTargetSid   = 0;

    if (!-d $pathUsrSapSid) {
        $mayBeTargetSid = 1;
    }
    elsif (!-d $pathUsrSapSidSys) {

        if (!opendir (DH, $pathUsrSapSid)){
            $self->setErrorMessage ("Cannot open dir '$pathUsrSapSid': $!");
            return undef;
        }

        # grab all links HDB##
        my @linksHDB = grep {/^HDB\d\d$/ && -l "$pathUsrSapSid/$_"} readdir (DH);

        close (DH);

        if (!@linksHDB) {
            $mayBeTargetSid = 1;
        }
    }

    if ($mayBeTargetSid) {

        # if the system is unregistered, SID can also be the source SID
        # this is performed in SDB::Install::Configuration::Rename::checkSID

        $sapSys = new SDB::Install::SAPSystem();
        $sapSys->preInitWithTargetSID($sid, $sapmnt);

        require SDB::Install::SAPProfiles;
        require SDB::Install::IniFile;

        my $profiles       = new SDB::Install::SAPProfiles($sapSys->get_ProfileDir());
        my $defaultProfile = $profiles->get_DefaultProfile();
        my $iniFile        = new SDB::Install::IniFile ($defaultProfile);

        if (defined $iniFile->read()) { # ignore error
            my $systemName = $iniFile->getValue (undef, 'SAPSYSTEMNAME');
            if (defined $systemName && ($systemName eq $sid)) {
                # SID of <sapmnt>/<SID> matches content
                #             --> source SID used  OR  already renamed
                $sapSys = undef;
            }
        }
    }

    if (defined $sapSys) {
        $msgLst->addProgressMessage
            ("Target installation directory '$pathSapmntSid' already exists");
    }

    return $sapSys;
}

sub getCallerDir{
    $_[0]->{callerdir};
}

#
# resolve '.' and '..' in $path
#

sub _normalizePath{
    my ($path) = @_;
    my @result;
    foreach my $node (split (/[\/\\]+/, $path)){
        if ($node eq '.'){
            next;
        }
        if ($node eq '..'){
            if ($result[$#result] ne ''){
                pop @result;
            }
            next;
        }
        push @result, $node;
    }
    my $result = join ($path_separator, @result);

    #
    # Resolution could go wrong, if there are symbolic links.
    # Check whether $path and $result points to the same fs node.
    #

    my @statbuf_src = stat ($path);
    if (@statbuf_src){
        my @statbuf_result = stat ($result);
        if (!@statbuf_result ||
            $statbuf_src[1] != $statbuf_result[1] ||
            $statbuf_src[0] != $statbuf_result[0]){
            # wrong resolution
            # => return unmodified $path
            return $path;
        }
    }
    return $result;
}

sub getInstallerDir{
    my ($self) = @_;
    if (!defined $self->{installerdir}){
        my $runtimeDir;
        if (defined $SAPDB::Install::Config{RawRuntimeDir}){
            $runtimeDir = _normalizePath ($SAPDB::Install::Config{RawRuntimeDir});
        }
        else{
            $runtimeDir = $self->{runtimedir};
        }

        if (!File::Spec->file_name_is_absolute ($runtimeDir)){
            $runtimeDir = File::Spec->rel2abs ($runtimeDir, $self->{callerdir});
        }

        ($self->{installerdir}) = ($runtimeDir =~ /(.*)$instruntime_pattern/);
    }
    $self->{installerdir};
}

sub getInstRuntimeDir{
    my ($self) = @_;
    if (!$isApple){
        return $self->{runtimedir};
    }
    return $self->getInstallerDir () . $path_separator . 'instruntime';
}


sub GetVersion ($){
	$_[0]->{version};
}

sub GetProgramName ($){
	$_[0]->{program};
}


sub GetGitHash{
    $_[0]->{git_hash};
}

sub GetMakeId;
*GetMakeId = \&GetGitHash;

sub GetRuntimeDir ($){
    $_[0]->{runtimedir};
}


1;
