package SDB::Install::LSS::LssInstance;

use strict;
use warnings;

use parent qw(SDB::Install::Base);

use File::Spec;
use File::stat qw(stat);
use SDB::Install::System qw(isLink readLink makedir changeOwn deltree);
use SDB::Install::Globals qw($gShortProductNameLSS 
                             $gSAPLocation
                             $gLogDir);
use SDB::Common::Utils qw(getSidcryptName);
use SDB::Install::LSS::LssSidcryptUser;
use SDB::Install::Group;
use File::Basename qw(dirname);
use SDB::Install::LSS::Tools::LSSConfig;
use SDB::Install::LayeredConfig qw (CFG_LAYER_HOST);
use SDB::Install::LSS::LssUserConfig;

use LCM::ProcessExecutor;
use IO::File;
use SDB::Common::BuiltIn;

sub new {
    my($class, $instance , $sid , $lssSharedDir) = @_; # remove instance with lss shared
    my $self = bless({},$class);
    $self->{instance} = $instance;
    $self->{sid} = $sid // $self->{instance}->get_sid();
    $self->{lssSharedDir} = $lssSharedDir;
    return $self;
}

sub configureDatabaseHost {
    my($self, $sapsys, $host, $instconfig) = @_;
    if($self->isConfiguredOnHost()){
        $self->getMsgLst ()->addMessage ("$gShortProductNameLSS is already configured...");
        return 1;
    }
    if(!$self->isInstalled()){
        $self->getMsgLst ()->addMessage ("$gShortProductNameLSS is not installed...");
        return 0;
    }

    my $rc = $self->_runConfigure($sapsys, $instconfig);
    $rc &&= $self->enableService($host);
    return $rc;
}

sub _runConfigure {
    my($self, $sapsys, $instconfig) = @_;
    my $msg = $self->getMsgLst ()->addProgressMessage ("Configure $gShortProductNameLSS...");
    my $hdbinst =  File::Spec->catfile($self->getLssSidDir(), 'install', 'hdbinst');
    my $args = $self->getAddHostArgs($sapsys);
    return undef if(!defined $args);

    my $stdinLines = $instconfig->getXmlPasswordStream(['LSSPassword']);
    my $processExecutor = new LCM::ProcessExecutor($hdbinst, $args, $stdinLines);
    $processExecutor->setMsgLstContext([$msg->getSubMsgLst() , $self->getErrMsgLst()]);
    my $rc = $processExecutor->executeProgram();
    if(!defined($rc) || ($rc != 0)){
        $self->setErrorMessage("Configure $gShortProductNameLSS failed");
        return undef;
    }
    return 1;
}

sub getAddHostArgs {
    my ($self, $sapsys) = @_;
    my $userData =  $self->initUser($sapsys);
    return undef if(!defined $userData);

    my $hdbinst =  File::Spec->catfile($self->getLssSidDir(), 'install', 'hdbinst');
    return [
        '--main', 'SDB::Install::App::Console::LSS::AddHostLss::main',
        '--lss_userid='.$userData->[0],
        '--lss_groupid='.$userData->[1],
        '--lss_user_shell='.$userData->[3],
        '--lss_user_home='.$userData->[2],
        '--instlog_dir='.$gLogDir,
        '--read_password_from_stdin=xml',
        '--batch',
    ];
}

sub getSid {
    my $self = shift;
    return $self->{sid};
}

sub getInstance {
    my $self = shift;
    return $self->{instance};
}

sub getLocalLssRootDir {
    my ($self) = @_;
    my $sid = $self->getSid();
    return  File::Spec->catfile($gSAPLocation, $sid, 'lss');
}

sub getLocalLssLocalDir {
    my ($self) = @_;
    return  File::Spec->catfile($self->getLocalLssRootDir(), 'local');
}

sub getLocalLssSharedDir {
    my ($self) = @_;
    return  File::Spec->catfile($self->getLocalLssRootDir(), 'shared');
}

sub getLocalLssExeDir {
    my ($self) = @_;
    return  File::Spec->catfile($self->getLocalLssRootDir() ,'exe');
}

sub getLssSharedDir {
    my ($self) = @_;
    if(!defined $self->{lssSharedDir} && $self->isInstalled()){
        $self->{lssSharedDir} = dirname(SDB::Install::System::readLink($self->getLssLink()));
    }
    return $self->{lssSharedDir};
}

sub getLssSidDir {
    my ($self) = @_;
    return File::Spec->catfile($self->getLssSharedDir(), $self->getSid());
}

sub getLssLink {
    my ($self) = @_;
    return File::Spec->catfile($self->getInstance()->get_globalSidDir(), 'lss');
}

sub isInstalled {
    my ($self, $skipCheckLinkTarget) = @_;
    my $lssDir = $self->getLssLink();
    $self->getMsgLst()->addMessage("Checking for existing $gShortProductNameLSS installation via link '$lssDir'...");
    return 0 if(!File::stat::lstat($lssDir));
    if(SDB::Install::System::isLink($lssDir) && (File::stat::stat($lssDir) || $skipCheckLinkTarget)){
         $self->getMsgLst()->addMessage("$lssDir link exists");
        return 1;
    }
    return 0;
}

sub isConfiguredOnHost {
    my $self = shift;
    $self->getMsgLst()->addMessage("Checking for installed $gShortProductNameLSS...");
    my $lssDir = $self->getLocalLssExeDir();
    return 0 if(!File::stat::lstat($lssDir));
    if(SDB::Install::System::isLink($lssDir) && File::stat::stat($lssDir)){
        $self->getMsgLst()->addMessage("$lssDir link exists");
        return 1;
    }
    return 0;
}

sub initUser {
    my ($self,$sapsys) = @_;
    $self->{user}  = {};
    my $userConfiguration = $self->getUserConfig($sapsys);
    $userConfiguration->setMsgLstContext($self->getMsgLstContext());
    my $userData = [ $userConfiguration->getUID() , $userConfiguration->getGID() , $userConfiguration->getHome() , $userConfiguration->getShell() ]; 
    return undef if($userConfiguration->errorState());
    return $userData;
}

sub configureHost {
    my ($self, $uid, $password, $shell, $homeDir, $gid) = @_;

    my $builtIn = SDB::Common::BuiltIn->get_instance();
# Create directory /usr/sap/<SID>/lss
    my $path = $self->getLocalLssRootDir();
    my $config = {
        'mode' => 0751,
        'uid'  => $uid,
        'gid'  => $gid,
    };
    $self->getMsgLst()->addMessage("Creating directory $path");
    if (!defined SDB::Install::System::makedir($path, $config)) {
        $self->setErrorMessage("Cannot create directory $path: $!");
        return undef;
    }
# Create group
    my $groupName = $builtIn->getgrgid($gid);
    my $sidcryptGroup = $self->getLssCryptGroup($gid);
    if(defined $groupName){
        $self->getMsgLst()->addMessage("Group '$groupName' exists");
    }else{
        if (!$sidcryptGroup->create(undef, $gid)) {
            $self->setErrorMessage("Cannot create group " .$sidcryptGroup->{name} , $sidcryptGroup->getErrMsgLst());
            return undef;
        }
    }
    $gid //= $sidcryptGroup->id();
# Create user
    my $sidcryptUser = $self->getLssCryptUser();
    $self->getMsgLst()->addMessage("Creating user " . $sidcryptUser->getname());
    if (!$sidcryptUser->create($uid, $password, undef, $gid, $homeDir, $shell)) {
        $self->setErrorMessage("Cannot create user " . $sidcryptUser->getname(), $sidcryptUser->getErrMsgLst());
        return undef;
    }
# Create local directory /usr/sap/<SID>/lss/local
    my $localPath = $self->getLocalLssLocalDir();
    $config->{mode} = 0750;
    $self->getMsgLst()->addMessage("Creating local directory $localPath");
    if (!defined SDB::Install::System::makedir($localPath, $config)) {
        $self->setErrorMessage("Cannot create local directory $localPath: $!");
        return undef;
    }
# Create symbolic link /usr/sap/<SID>/lss/exe -> $installationPath/exe
    my $installationPath = $self->getLssSidDir();

    my $exeLink = File::Spec->catfile($path, 'exe');
    my $target = File::Spec->catfile($installationPath, 'exe');
    $self->getMsgLst()->addMessage("Creating symbolic link $exeLink -> $target");

    $builtIn->unlink($exeLink) if (SDB::Install::System::isLink($exeLink));

    if (!$builtIn->symlink($target, $exeLink)) {
        $self->setErrorMessage("Cannot create symbolic link $exeLink -> $target: $!");
        return undef;
    }
# Create symbolic link /usr/sap/<SID>/lss/shared -> $installationPath
    my $sharedLink = File::Spec->catfile( $path, 'shared');
    $self->getMsgLst()->addMessage("Creating symbolic link $sharedLink -> $installationPath");

    $builtIn->unlink($sharedLink) if (SDB::Install::System::isLink($sharedLink));

    if (!$builtIn->symlink($installationPath, $sharedLink)) {
        $self->setErrorMessage("Cannot create symbolic link $sharedLink -> $installationPath: $!");
        return undef;
    }

    $sidcryptUser->setMsgLstContext($self->getMsgLstContext());
    if(!$sidcryptUser->configureHome($self->getSid() , $self->getLssSidDir())){
        return undef;
    }
    return 1;
}

sub createLssRootDir {
    my ($self, $uid, $gid) = @_;
    my $path = $self->getLocalLssRootDir();
    if (File::stat::stat($path)) {
        $self->getMsgLst()->addMessage("Directory '$path' already exists");
        return 1;
    }
    my $config = {
        'mode' => 0751,
        'uid'  => $uid,
        'gid'  => $gid,
    };
    $self->getMsgLst()->addMessage("Creating directory $path");
    if (!defined SDB::Install::System::makedir($path, $config)) {
        $self->setErrorMessage("Cannot create directory $path: $!");
        return undef;
    }
    return 1;
}

sub createExeLink {
    my ($self) = @_;
    my $installationPath = $self->getLssSidDir();
    my $path = $self->getLocalLssRootDir();

    my $exeLink = File::Spec->catfile($path, 'exe');
    my $target = File::Spec->catfile($installationPath, 'exe');
    $self->getMsgLst()->addMessage("Creating symbolic link $exeLink -> $target");

    my $builtIn = SDB::Common::BuiltIn->get_instance();
    $builtIn->unlink($exeLink) if (SDB::Install::System::isLink($exeLink));

    if (!$builtIn->symlink($target, $exeLink)) {
        $self->setErrorMessage("Cannot create symbolic link $exeLink -> $target: $!");
        return undef;
    }
    return 1;
}

sub createSharedLink {
    my ($self) = @_;
    my $installationPath = $self->getLssSidDir();
    my $path = $self->getLocalLssRootDir();

    my $sharedLink = $self->getLocalLssSharedDir();
    $self->getMsgLst()->addMessage("Creating symbolic link $sharedLink -> $installationPath");

    my $builtIn = SDB::Common::BuiltIn->get_instance();
    $builtIn->unlink($sharedLink) if (SDB::Install::System::isLink($sharedLink));

    if (!$builtIn->symlink($installationPath, $sharedLink)) {
        $self->setErrorMessage("Cannot create symbolic link $sharedLink -> $installationPath: $!");
        return undef;
    }
    return 1;
}

sub getLssCryptUser {
    my ($self)  = @_;
    my $sidcryptUser = SDB::Install::LSS::LssSidcryptUser->new(getSidcryptName($self->getSid()));
    $sidcryptUser->setMsgLstContext($self->getMsgLstContext());
    return $sidcryptUser;
}


sub initialize {
    my($self, $systemUsage, $isAddhost) = @_;
    my $markerFileName =  File::Spec->catfile($self->getLocalLssLocalDir(), ($isAddhost) ? '.addhost' : '.initial' );
    return undef if (!$self->createMarkerFiles($markerFileName));
    if(!$isAddhost){
        return undef if (!$self->createLssIni($systemUsage));
        return undef if (!$self->initPersistence());
    }

    return 1;
}

sub createMarkerFiles {
    my ($self , $markerFileName) = @_;
    $self->getMsgLst()->addMessage("Creating initial LSS container marker $markerFileName");
    my $fd = IO::File->new(">$markerFileName");
    if (!$fd){
        $self->setErrorMessage("Cannot create initial container LSS marker '$markerFileName': $!");
        return undef;
    }
    undef $fd;
    my $user = $self->getLssCryptUser();
    $self->getMsgLst()->addMessage("Adjusting the ownership of initial LSS container marker");
    if (!SDB::Install::System::changeOwn (undef,$user->id(),undef, $user->gid(), $markerFileName)) {
        $self->setErrorMessage("Cannot change the ownership of initial LSS container marker: $!");
        return undef;
    }
    my $builtIn = SDB::Common::BuiltIn->get_instance();
    if (!$builtIn->chmod(0750, $markerFileName)) {
        $self->setErrorMessage("Cannot change the permissions of initial LSS container marker: $!");
        return undef;
    }
    return 1;
}

sub createLssIni {
    my ($self,$systemUsage) = @_;
    my $msg = $self->getMsgLst()->addMessage("Creating lss.ini");
    my $saveContext = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    my $iniFileName =  File::Spec->catfile($self->getLocalLssExeDir(), 'lss.ini' );
    my $user = $self->getLssCryptUser();
    my $ini = SDB::Install::IniFile->new($iniFileName, $user->id(), $user->gid(), 0640);
    if ($ini->ErrorState()){
        $self->getErrMsgLst()->addError ('Cannot access ini-file', $ini);
        return undef;
    }
    $ini->setMsgLstContext($self->getMsgLstContext());
    my $section = 'system';
    $ini->setValue($section, 'sid', $self->getSid());
    $section = 'setup';
    $ini->setValue($section, 'initial', 'true');
    $ini->setValue($section, 'system_usage', $systemUsage);
    $ini->setValue($section, 'client_type', 'hana');
    $section = 'nfs';
    $ini->setValue($section, 'shared_path', $self->getLocalLssSharedDir());

    if (!defined $ini->write()){
        $self->setMsgLstContext($saveContext);
        return undef;
    }
    $self->setMsgLstContext($saveContext);
    return 1;
}

sub initPersistence {
    my ($self) = @_;
    return $self->getLssCfg()->initPersistence();
}

sub storeUserInfo {
    my($self,$sapsys) = @_;
    my $user = $self->getLssCryptUser();
    my $cfg  = $self->getUserConfig($sapsys);
    $cfg->setMsgLstContext($self->getMsgLstContext());
    return $cfg->storeUserInfo($user);
}

sub disableService {
    my ($self, $host) = @_;
    return $self->_enableDisableService(0, $host);
}

sub enableService {
    my ($self, $host) = @_;
    return $self->_enableDisableService(1, $host);
}

sub _enableDisableService {
    my ($self, $value, $host) = @_;
    $self->getInstance()->setMsgLstContext($self->getMsgLstContext());
    return $self->getInstance()->writeIntoDaemonIni(CFG_LAYER_HOST, 'localsecurestore', 'instances', $value, undef, $host);
}

sub getUserConfig {
    my ($self,$sapsys) = @_;
    return  SDB::Install::LSS::LssUserConfig->new($sapsys);
}

sub setBackupPassphrase {
    my ($self, $backupPasswd) = @_;
    return $self->getLssCfg()->setBackupPassphrase($backupPasswd);
}

sub getLssCfg {
    my ($self) = @_;
    my $user = $self->getLssCryptUser();
    my $lssCfg = SDB::Install::LSS::Tools::LSSConfig->new($self->getSid(),$user->id(), $user->gid());
    $lssCfg->setMsgLstContext($self->getMsgLstContext());
    return $lssCfg;
}

sub adaptSpecialExecutable {
    my ($self, $filename, $uid, $gid, $mode, $useSharedDir) = @_;
    my $builtIn = SDB::Common::BuiltIn->get_instance();
    my $exeDir = $useSharedDir ? File::Spec->catfile($self->getLssSidDir(), 'exe') : $self->getLocalLssExeDir();
    my $fullPath = File::Spec->catfile($exeDir, $filename);
    my $shouldChown = (defined $uid && $uid != -1) || (defined $gid && $gid != -1);

    $self->getMsgLst()->addMessage("Adjusting ownership of '$fullPath'...");
    if ($shouldChown && !$builtIn->chown($uid, $gid, $fullPath)) {
        $self->getErrMsgLst()->addError("Failed to change ownership of file '$fullPath': $!");
        return 0;
    }

    $self->getMsgLst()->addMessage(sprintf("Adjusting mode of '$fullPath' to %o...", $mode));
    if (!$builtIn->chmod($mode, $fullPath)) {
        $self->getErrMsgLst()->addError(sprintf("Failed to set permissions '%o' to file '%s': %s.", $mode, $fullPath, $!));
        return 0;
    }
    return 1;
}

sub removeFromLocalHost {
    my($self, $keepUser , $keepUserGroup) = @_;
    my $instance = $self->getInstance();
    my $localhost = $instance->get_host();
    my $msg = $self->getMsgLst ()->addProgressMessage ("Removing $gShortProductNameLSS from host $localhost...");

    my $hdbunst =  File::Spec->catfile($self->getLssSidDir(), 'install', 'hdbuninst');
    my $args = [
        '--main', 'SDB::Install::App::Console::LSS::RemoveHost::main',
        '--instlog_dir', $gLogDir,
    ];
    push(@$args, '--keep_user=1') if ($keepUser);
    push(@$args, '--keep_user_group=1') if ($keepUserGroup);
    my $processExecutor = new LCM::ProcessExecutor($hdbunst, $args);
    $processExecutor->setMsgLstContext([$msg->getSubMsgLst() , $self->getErrMsgLst()]);
    my $rc = $processExecutor->executeProgram();
    if(!defined($rc) || ($rc != 0)){
        $self->setErrorMessage("Removing $gShortProductNameLSS failed");
        return undef;
    }
    return 1;
}

sub renameInstance {
    my ($self, $instconfig) = @_;

    my $stdinPasswds = $instconfig->getXmlPasswordStream(['LSSPassword']);

    my $newSid = $instconfig->getTargetLssSid();
    my $oldSid = $instconfig->getValue('SID');
    my $lssUid = $instconfig->getValue('LSSUserID');
    my $lssGid = $instconfig->getValue('LSSGroupID');
    my $shell = $instconfig->getValue('LSSUserShell');
    my $home = $instconfig->getValue('LSSUserHomeDir');
    my $isRegisterLSS = $instconfig->getValue('IsRegisterLSS');

    my $msg = $self->getMsgLst ()->addProgressMessage ("Renaming $gShortProductNameLSS instance...");
    my $hdbinst =  File::Spec->catfile($self->getLssSidDir(), 'install', 'hdbinst');
    my $args = [
        '--main', 'SDB::Install::App::Console::LSS::RenameInstanceLss::main',
        '--target_sid', $newSid,
        '--source_sid', $oldSid,
        '--read_password_from_stdin=xml',
        '--instlog_dir', $gLogDir,
        '--batch',
        '--lss_userid', $lssUid,
        '--lss_groupid', $lssGid,
        '--lss_user_shell', $shell,
        '--lss_user_home', $home,
        ($isRegisterLSS) ? '--is_register=1' : (),
    ];
    my $processExecutor = LCM::ProcessExecutor->new($hdbinst, $args, $stdinPasswds);
    $processExecutor->setMsgLstContext([$msg->getSubMsgLst() , $self->getErrMsgLst()]);
    my $rc = $processExecutor->executeProgram();
    if(!defined($rc) || ($rc != 0)){
        $self->setErrorMessage("Renaming $gShortProductNameLSS instance failed");
        $msg->endMessage();
        return undef;
    }
    my $instance = $instconfig->getOwnInstance(1);
    $self->enableService($instance->get_host());
    $msg->endMessage();
    return 1;
}

sub remove {
    my( $self, $keepUser , $keepGroup, $removeFromHostOnly) = @_;

    if(!$keepUser){
        my $user = $self->getLssCryptUser();
        my $groupName = $user->group();
        my $gid = $user->gid();
        return undef if(!$self->deleteUser());

        my $isDefaultGroup = $groupName eq getSidcryptName($self->getSid()) ? 1 : 0;
        if($isDefaultGroup && !$keepGroup){
            return undef if(!$self->deleteGroup($gid));
        }
    }

    my $removePaths = File::stat::stat($self->getLocalLssRootDir()) ? [$self->getLocalLssRootDir()] : []; #/usr/sap/SID/lss could be removed already from the db uninstall
    if(!$removeFromHostOnly){
        push (@$removePaths, $self->getLssSharedDir());
    }

    my ($message,$saveContext);
    my $rc = 1;
    foreach my $path(@$removePaths){
        $message = $self->getMsgLst()->addProgressMessage("Deleting $path...");
        $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
        if(!SDB::Install::System::deltree($path,$message->getSubMsgLst(),$self->getErrMsgLst())){
            $rc = undef;
            last;
        }
        $message->endMessage();
    }

    $self->setMsgLstContext($saveContext);

    return $rc;
}

sub deleteUser {
    my ($self) = @_;
    my $user = $self->getLssCryptUser();
    my $userName = $user->getname();

    my $rc  = 1;
    my $msg = $self->getMsgLst()->addProgressMessage ("Deleting user $userName ...");
    $user->setMsgLstContext ([$msg->getSubMsgLst ()]);

    if ($user->exists()) {
        if (!defined $user->delete()) {
            $self->setErrorMessage ("Cannot delete $userName", $user->getErrMsgLst());
            $rc = undef;
        }
    }
    else{
        $msg->getSubMsgLst()->addMessage ("$userName is already deleted");
    }

    return $rc;
}

sub deleteGroup {
    my ($self, $gid) = @_;
    my $group = $self->getLssCryptGroup($gid);
    my $groupName = $group->{name};
    my $rc  = 1;
    my $msg = $self->getMsgLst()->addProgressMessage ("Deleting group $groupName ...");
    $group->setMsgLstContext ([$msg->getSubMsgLst ()]);

    if ($group->exists()) {
        if (!defined $group->delete()) {
            $self->setErrorMessage ("Cannot delete $groupName", $group->getErrMsgLst());
            $rc = undef;
        }
    }
    else{
        $msg->getSubMsgLst()->addMessage ("$groupName is already deleted");
    }

    return $rc;
}

sub getLssCryptGroup {
    my ($self, $gid) = @_;
    my $builtIn = SDB::Common::BuiltIn->get_instance();
    my $groupName =  $builtIn->getgrgid($gid) // getSidcryptName($self->getSid());
    my $group = SDB::Install::Group->new($groupName);
    $group->setMsgLstContext($self->getMsgLstContext());
    return $group;
}

sub createLssSymLink {
    my ($self, $targetDir) = @_;
    $self->getMsgLst()->addMessage ("Create symbolic link for $gShortProductNameLSS...");
    my $lssSharedLink = $self->getLocalLssSharedDir();
    if(!SDB::Install::System::isLink($lssSharedLink)){
        $self->setErrorMessage("Cannot find symbolic link '$lssSharedLink'");
        return 0;
    }

    my $lssInstPath = SDB::Install::System::readLink ( $lssSharedLink );
    my $targetLink = File::Spec->catfile($targetDir, $self->getSid(), 'lss');
    my $builtIn = SDB::Common::BuiltIn->get_instance();
    $builtIn->unlink($targetLink) if (SDB::Install::System::isLink($targetLink));
    if (!SDB::Common::BuiltIn->get_instance()->symlink($lssInstPath, $targetLink)){
        $self->setErrorMessage("Failed to create '$targetLink'");
        return 0;
    }
    return 1;
}

sub configureMasterHost {
    my ($self, $sapsys) = @_;
    $self->getMsgLst()->addProgressMessage("Configuring $gShortProductNameLSS on local host...");

    my $rc = $self->createLssSymLink($sapsys->get_target());
    $rc &&= $self->storeUserInfo($sapsys);
    $rc &&= $self->enableService($self->getInstance()->get_host());
    return $rc;
}

sub adaptSettings {
    my ($self, $instconfig) = @_;

    my $lssCfg = $self->getLssCfg();
    my $iniFilePath =  File::Spec->catfile($self->getLocalLssExeDir(), 'lss.ini');
    my $iniFile = SDB::Install::IniFile->new($iniFilePath);

    if ($instconfig->isSidChanged()) {
        my $newSid = $instconfig->getTargetLssSid();
        $iniFile->setMsgLstContext($self->getMsgLstContext());
        $iniFile->setValue('nfs', 'shared_path', $self->getLocalLssSharedDir());
        $iniFile->setValue('system', 'sid', $newSid);
        return undef if (!$iniFile->write());
    }
    return undef if (!$self->adaptSystemUsage($instconfig));
    return undef if (!$self->reconfig());
    return undef if (!$self->restorePSE($instconfig));

    my $hostmap_hash = $instconfig->getValue ('HostMap');
    if (defined $hostmap_hash) {
        return undef if (!$lssCfg->renameHosts($hostmap_hash));
    }
    my $targetBackupPassword = $instconfig->getValue('TargetLSSBackupPassword');
    if (defined $targetBackupPassword) {
        return undef if (!$lssCfg->setBackupPassphrase($targetBackupPassword));
    }
    if ($instconfig->getValue('IsRegisterLSS')) {
        return undef if (!$lssCfg->rotatePSEs());
    }
    return 1;
}

sub adaptSystemUsage {
    my ($self, $instconfig, $shouldReconfig) = @_;
    my $iniFilePath =  File::Spec->catfile($self->getLocalLssExeDir(), 'lss.ini');
    my $iniFile = SDB::Install::IniFile->new($iniFilePath);
    my $newSystemUsage = $instconfig->getValue('SystemUsage');
    my $instance = $instconfig->getOwnInstance();
    if ($instance->isSystemUsageChanged($newSystemUsage)) {
        $self->getMsgLst()->addProgressMessage("Updating system usage in lss.ini to '$newSystemUsage'");
        $iniFile->setValue('setup', 'system_usage', $newSystemUsage);
        return undef if (!$iniFile->write());
    }
    return ($shouldReconfig) ? $self->reconfig() : 1;
}

sub reconfig {
    my ($self) = @_;
    my $lssCfg = $self->getLssCfg();
    $lssCfg->setMsgLstContext($self->getMsgLstContext());
    return $lssCfg->reconfig();
}

sub restorePSE {
    my ($self, $instconfig) = @_;
    if (!$instconfig->getValue('IsRegisterLSS')) {
        return 1;
    }
    my $lssCfg = $self->getLssCfg();
    my $passwd = (!$instconfig->isSkipped('TargetLSSBackupPassword') && $instconfig->{isSlave})
        ? $instconfig->getValue('TargetLSSBackupPassword')
        : $instconfig->getValue('LSSBackupPassword');
    return $lssCfg->restorePSE($passwd);
}


1;
