package SDB::Install::LSS::Tools::LSSConfig;
use strict;
use warnings;

use SDB::Install::Globals qw($gProductNameLSS $gSAPLocation $gSapsysGroupName $gTrustLevelFinalAssembly);
use LCM::ProcessExecutor;
use File::Basename qw(dirname);
use SDB::Install::System qw(isLink readLink);
use parent qw(SDB::Install::Base);

sub new{
    my($class,$sid,$sidcryptUID,$sidcryptGID) = @_;
    my $self =  bless({},$class);
    $self->{sid} = $sid;
    $self->{lssUID} = $sidcryptUID;
    $self->{lssGID} = $sidcryptGID;
    return $self;
}

sub getSid{
    my $self = shift;
    return $self->{sid};
}

sub _createEnvironment {
    my ($self) = @_;
    my %env = %ENV;
    my $exeDir = $self->_getLssLibrariesPath();
    my $localPath = File::Spec->catfile($gSAPLocation, $self->getSid(), 'lss' ,'local');
    $env{LSS_LOCAL_PATH} = $localPath;
    $env{LD_LIBRARY_PATH} = $env{LD_LIBRARY_PATH} ? $env{LD_LIBRARY_PATH}.":$exeDir" : $exeDir;
    return \%env;
}

sub _createProcessExecutor {
    my ($self,$args,$passwdInputStream, $gid) = @_;
    $gid //= $self->{lssGID};
    my $workDir = $self->_getLssLibrariesPath();
    if(! defined $workDir){
        return undef;
    }
    my $command = File::Spec->catfile($workDir,'lsscfg');
    my $executor = LCM::ProcessExecutor->new($command, $args, $passwdInputStream, $workDir, undef, $self->{lssUID}, $gid);
    $executor->setProcessEnvironment($self->_createEnvironment());
    $executor->setMsgLstContext($self->getMsgLstContext());
    return $executor;
}


sub _getLssLibrariesPath {
    my ($self) = @_;
    my $exeLink = File::Spec->catfile($gSAPLocation, $self->getSid(), 'lss' , 'exe');
    my $isLink = SDB::Install::System::isLink($exeLink);
    my $exeDir = $isLink ? SDB::Install::System::readLink ( $exeLink ) : undef;

    if(! defined $exeDir){
        $self->getErrMsgLst()->addError("$gProductNameLSS libraries path cannot be found.");
        return undef;
    }
    return $exeDir;
}

sub deleteComponent{
    my ($self, $componentManifest, $progressMessage) = @_;
    my $args=[ 'deleteProduct', '-f', $componentManifest->getKeyname()];
    my $lssCfg = $self->_createProcessExecutor($args);
    return $self->execute($lssCfg, $progressMessage);
}

sub importComponentHashes{
    my($self, $componentManifest, $signaturePath, $progressMessage) = @_;
    my @arguments = ();
    push( @arguments, 'importComponentHashes');
    push( @arguments, '-f');
    push( @arguments, $componentManifest->getKeyname());
    push( @arguments, $componentManifest->getVersion());
    push( @arguments, dirname($componentManifest->getFileName()));
    push( @arguments, $signaturePath) if(defined $signaturePath);
    my $lssCfg = $self->_createProcessExecutor(\@arguments);
    return $self->execute($lssCfg, $progressMessage);
}

sub setOutputHandler {
    my ($self, $handler) = @_;
    $self->{'outputHandler'} = $handler;
}

sub getOutputHandler {
    my ($self) = @_;
    return $self->{'outputHandler'};
}

sub initPersistence{
    my ($self) = @_;
    my $builtin = SDB::Common::BuiltIn->get_instance();
    my $gid = $builtin->getgrnam($gSapsysGroupName);
    my $lssCfg = $self->_createProcessExecutor(['initialize'], undef, $gid);
    return $self->execute($lssCfg, 'Initializing persistence');
}

sub reconfig {
    my ($self, $sid) = @_;
    my $lssCfg = $self->_createProcessExecutor(['reconfig']);
    return $self->execute($lssCfg, "Updating local lss.ini via 'lsscfg reconfig'");
}

sub renameHosts{
    my ($self, $hostMap) = @_;
    while (my ($sourceName, $targetName) = each(%$hostMap)) {
        next if ($sourceName eq $targetName);
        return undef if (!$self->renameHost($sourceName, $targetName));
    }
    return 1;
}

sub renameHost{
    my ($self, $sourceHost, $targetHost) = @_;
    my $lssCfg = $self->_createProcessExecutor(['renameHost', $sourceHost, $targetHost]);
    return $self->execute($lssCfg, "Renaming '$sourceHost' to '$targetHost' through lsscfg");
}

sub restorePSE{
    my ($self, $passwd) = @_;
    my $lssCfg = $self->_createProcessExecutor(['restorePSE'], [$passwd]);
    return $self->execute($lssCfg, 'Restoring PSE');
}

sub rotatePSEs {
    my ($self) = @_;
    my $lssCfg = $self->_createProcessExecutor(['rotatePSEs']);
    return $self->execute($lssCfg, 'Rotating PSEs');
}

sub setBackupPassphrase {
    my($self, $backupPasswd) = @_;
    my $lssCfg = $self->_createProcessExecutor(['setBackupPassphrase'], [$backupPasswd]);
    return $self->execute($lssCfg, 'Setting backup passphrase');
}

sub trustComponent {
    my ($self, $manifest, $trustLevel, $progressMessage) = @_;
    $trustLevel //= $gTrustLevelFinalAssembly;
    my @arguments = ();
    push( @arguments, 'trust');
    push( @arguments, $manifest->getKeyname());
    push( @arguments, $manifest->getVersion());
    push( @arguments, $trustLevel);
    my $lssCfg = $self->_createProcessExecutor(\@arguments);
    return $self->execute($lssCfg, $progressMessage);
}

sub execute {
    my ($self, $lssCfg, $progressMessage) = @_;
    return 0 if(!defined $lssCfg);

    $progressMessage //= '';
    my $message = $self->getMsgLst()->addProgressMessage($progressMessage.'...');
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    my $outputHandler = $self->getOutputHandler();
    $lssCfg->setOutputHandler($outputHandler) if($outputHandler);
    $lssCfg->setMsgLstContext($self->getMsgLstContext());

    my $rcLssCfg = $lssCfg->executeProgram(1);
    my $rc = 1;
    if (!defined($rcLssCfg) || ($rcLssCfg != 0)){
        $self->getErrMsgLst()->addError($progressMessage." failed!");
        $rc = 0;
    }
    $message->endMessage(undef, $progressMessage);
    $self->setMsgLstContext($saveContext);
    return $rc;
}

1;