#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: Find class for searching in the tree of content and cache.

package SDB::Install::LogViewer::Search::Find;

use Wx qw(
        wxWHITE
        );

my $YELLOW = Wx::Colour->new(255,255,0);


##
# Useless constructor
##

sub new () {
    my $invocant = shift;
    my $class = ref ($invocant) || $invocant;
    
    my $self = {
    };
    
    return bless ($self, $class);
}


##
# Resets the this find object.
##

sub Reset () {
    my ($self) = @_;
    $self->{HIGHLIGHTING} = undef;
    delete $self->{HIGHLIGHTING};
    $self->{SEARCHNODE} = undef; # temp scalar to save current (find dependent) cache node
    delete $self->{SEARCHNODE};
    1;
}


##
# Returns if search is enable or not.
#
# @return 1 if enable, otherwise 0
##

sub isSearchEnable {
    my ($self) = @_;
    return (defined $self->{SEARCHNODE}); 
}

#>----------- FindNext & FindPrevious ----------------<#


my $FOUND = 'found'; # return code


##
# Find next fitting tree item.
#
# @params
#   Wx::TreeItem xlvnode - Current node, could be undef
#   Wx::TreeCtrl tree    - Full tree to search in
#   Cache cache          - cache
#   scalar searchstring  - text to search 
#   scalar matchcase     - if defined then search case sensitive
#
# @return 
#   Wx::TreeItem         - result node if found, otherwise -1
##

sub FindNext () {
    my ($self, $xlvnode, $xlvtree, $cache, $searchstring, $matchcase) = @_;
    return $self->SearchNow($xlvnode, $xlvtree, $cache->GetCacheTree, $searchstring, $matchcase, 'NEXT');
}


##
# Find previous fitting tree item.
#
# @params
#   Wx::TreeItem node   - Current node, could be undef
#   Wx::TreeCtrl tree   - Full tree to search in
#   Cache cache          - cache
#   scalar searchstring - text to search 
#   scalar matchcase    - if defined then search case sensitive
#
# @return 
#   Wx::TreeItem        - result node if found, otherwise -1
##

sub FindPrevious () {
    my ($self, $xlvnode, $xlvtree, $cache, $searchstring, $matchcase) = @_;
    return $self->SearchNow($xlvnode, $xlvtree, $cache->GetCacheTree, $searchstring, $matchcase, 'PREVIOUS');
}


##
# Start searching.
#
# @params
#   Wx::TreeItem xlvnode - Current node, could be undef
#   Wx::TreeCtrl xlvtree - Full tree to search in
#   CacheTree cachetree  - Cache Tree
#   scalar searchstring  - text to search 
#   scalar matchcase     - if defined then search case sensitive
#   scalar nextprevious  - defines next or previous search mode
#
# @return 
#   Wx::TreeItem         - result node if found, otherwise -1
##

sub SearchNow() {
    my ($self, $xlvnode, $xlvtree, $cachetree, $searchstring, $matchcase, $nextprevious) = @_;
    
    #
    # set start node
    #
    
    unless ($xlvnode) {
        $xlvnode = $self->{SEARCHNODE};
        unless ($xlvnode) {
            $xlvnode = $xlvtree->GetRootItem();
        }
    } 
    $self->{SEARCHNODE} = undef;
    delete $self->{SEARCHNODE};
    
    #
    # get cachenode from xlvnode
    # 
    
    my $data = $xlvtree->GetPlData($xlvnode);
    my $cachenode;
    if (!defined $data) {
        $cachenode = $cachetree->GetRootItem();
    } else {
        if (exists $data->{id}) {
            $cachenode = $cachetree->GetItemById ( $data->{id} );
        } else {
            $cachenode = $cachetree->GetRootItem();
        }    
    }

    #
    # Start searching on cache tree
    #
        
    unless ($self->SearchNextString($cachenode, $cachetree, $searchstring, $matchcase, $nextprevious) eq $FOUND) {
        $self->Reset();
        return -1;
    }
    
    #
    # open the xlvtree to the chachenode
    #
    
    my @hierarchy=();
     
        # go to cache root and save all cache nodes
        my $node = $self->{SEARCHNODE};
        while ((defined $node) && ($node ne "ROOT")) {
            push @hierarchy, $node;
            $node = $cachetree->GetItemParent ($node);
        }
    
        # expand the xlvtree to the found node
        my $xlvItem = $xlvtree->GetRootItem();
        my $notRoot = 0;
        while ($cacheItem = pop @hierarchy) {
            
            if ($notRoot == 1) {
               # set fitting xlvItem
               my $itemId = $cachetree->GetItemId($cacheItem);
               my $found = undef;
               my ($xlvchild, $cookie) = $xlvtree->GetFirstChild( $xlvItem );
               while( $xlvchild->IsOk() )
               {
                   my $itemData = $xlvtree->GetPlData($xlvchild);
                   if ((exists $itemData->{id}) && ($itemData->{id} eq $itemId)) {
                        $found = $xlvchild;
                        last;
                   }
                   ($xlvchild,$cookie) = $xlvtree->GetNextChild( $xlvItem, $cookie );
               }
               $xlvItem = $found;
            }

            last if (!defined $xlvItem);                        
            $xlvtree->Expand($xlvItem, $cacheItem);
            $notRoot = 1;
        }        
        
        # set SEARCHNODE
        $self->{SEARCHNODE} = $xlvItem;
        
    return $self->{SEARCHNODE};
}


##
# Searching for the next fitting item.
#
# @params
#   TreeElement cachenode - Current node, could be undef
#   CacheTree cachetree   - Full tree to search in
#   scalar searchstring   - text to search 
#   scalar matchcase      - if defined then search case sensitive
#   scalar nextprevious   - defines next or previous search mode
#
# @return 
#   scalar              - $FOUND if something found, otherwise 1
##

sub SearchNextString() {
    my ($self, $cachenode, $cachetree, $searchstring, $matchcase, $nextprevious) = @_;

    #
    # Start sequential search on the tree
    #
    
    while (defined $cachenode) {
        
        #
        # Get item
        #
        
        if ($nextprevious eq 'NEXT') {
            $cachenode = $self->GetNext( $cachetree, $cachenode );
            #respect filtering
            while ((defined $cachenode) && ($cachetree->IsFiltered($cachenode))) {
                $cachenode = $self->GetNext( $cachetree, $cachenode );
            }
        } elsif ($nextprevious eq 'PREVIOUS') {
            $cachenode = $self->GetPrev( $cachetree, $cachenode );
            #respect filtering
            while ((defined $cachenode) && ($cachetree->IsFiltered($cachenode))) {
               $cachenode = $self->GetPrev( $cachetree, $cachenode );
            }
        }
        
        #
        # Check item
        #
        
        if (defined $cachenode) {
           my $text = $cachetree->GetItemText($cachenode);
           if ($matchcase) {
                if ($text =~ /$searchstring/) {
                    $self->{SEARCHNODE} = $cachenode;
                    return $FOUND;      
                }          
            } else {
                if ($text =~ /$searchstring/i) { #BUG: i für Groß- und Kleinschreibung
                    $self->{SEARCHNODE} = $cachenode;
                    return $FOUND;      
                }          
            } 
            
        }
        
    }
    
	1;
}


##
# Get next tree node item.
#
# @params
#   CacheTree cacheTree - CacheTree
#   TreeElement node    - Current node, could be undef
#
# @return 
#   TreeElement        - next found node
##

sub GetNext() {
    my ($self, $cachetree, $node) = @_;
    # has children?
    my ($item, $cookie) = $cachetree->GetFirstChild( $node );
    if( defined $item ) {
        return $item;
    } else {
        # has silblings?
        $item = $cachetree->GetNextSibling($node);
        if( defined $item ) {
            return $item;
        } else {
            return $cachetree->GetNextVisible( $node );
        }
    }    
}


##
# Get previous tree node item.
#
# @params
#   CacheTree cacheTree - CacheTree
#   TreeElement node    - Current node, could be undef
#
# @return 
#   TreeElement        - next found node
##
sub GetPrev() {
    my ($self, $cachetree, $node) = @_;
    
    # has silblings?
    my $lastitem;
    my $item = $cachetree->GetPrevSibling($node);
    if( defined $item ) {
        while ( defined $item ) {
            $lastitem = $item;
            $item = $cachetree->GetLastChild( $item );
        }
        return $lastitem;
    } else {
        my $returnParent = $cachetree->GetItemParent( $node );
        if ($returnParent eq 'ROOT') { return undef; }
        return $returnParent;
    }
}

#>--------------- Highlight ------------------<#


##
# Set background white of every leaf.
#
# @params
#   Wx::TreeCtrl tree     - Full tree
#   Wx::TreeItem parent - node
##

sub TrashHighlights() {
    my ($self, $tree, $parent) = @_;

    if (!$parent) { $parent = $tree->GetRootItem(); }

    my ($item, $cookie) = $tree->GetFirstChild( $parent );

    while( $item->IsOk() )
    {
        $tree->SetItemBackgroundColour($item, wxWHITE);
        $self->TrashHighlights($tree,$item);        
        ($item,$cookie) = $tree->GetNextChild( $parent, $cookie );
    }
   
   1;
}


##
# Returns if highlighting is enable or not.
#
# @return 1 if enable, otherwise 0
##

sub isHighlightingEnable {
    my ($self) = @_;
    return (defined $self->{HIGHLIGHTING}); 
}


##
# Highlight fitting leafs. 
#
# @params
#   Wx::TreeCtrl xlvtree   - Full tree to search in
#   scalar searchstring    - Text to search for
#   scalar matchcase       - if defined then search case sensitive
##

sub HighlightAll () {
    my ($self, $xlvtree, $searchstring, $matchcase) = @_;
    $self->{HIGHLIGHTING} = 1;
    $self->SearchHighlightString($xlvtree->GetRootItem, $xlvtree, $searchstring, $matchcase);
    $xlvtree->Refresh();
    1;
}

##
# Search in the treectrl to highlight fitting leafs. 
#
# @params
#   Wx::TreeItem node     - current leaf
#   Wx::TreeCtrl $xlvtree - Full tree to search in
#   scalar searchstring   - Text to search for
#   scalar matchcase      - if defined then search case sensitive
##

sub SearchHighlightString() {
    my ($self, $node, $xlvtree, $searchstring, $matchcase) = @_;
    
    #
    # look in data
    #
    
    my $text = $xlvtree->GetItemText($node);
    
    #
    # fitting?
    #
    
    if ($matchcase) {
        if ($text =~ /$searchstring/) {
            $xlvtree->SetItemBackgroundColour($node, $YELLOW);
        } else {
            $xlvtree->SetItemBackgroundColour($node, wxWHITE);
        }
    } else {
        if ($text =~ /$searchstring/i) {
            $xlvtree->SetItemBackgroundColour($node, $YELLOW);
        } else {
            $xlvtree->SetItemBackgroundColour($node, wxWHITE);
        }
    } 
    
    #
    # search in children
    #
    
    my ($item, $cookie) = $xlvtree->GetFirstChild( $node );
    while( $item->IsOk() )
    {
        $self->SearchHighlightString($item, $xlvtree, $searchstring, $matchcase);
        ($item,$cookie) = $xlvtree->GetNextChild( $node, $cookie );
    }
	1;
}

1;
