#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: Cache File Informations.

package SDB::Install::LogViewer::Type::Cache;

use XML::Parser::Expat;
use SDB::Install::LogViewer::Type::CacheTree;
use SDB::Install::Globals qw ($gProductNameInstaller);
use strict;

#temp variable for parsing
our $lastItem;

#root tags
my $ROOTSTARTTAG = "<root>\n";
my $ROOTENDTAG = "</root>";

# Tag contants
my $MGSL_TAG     = 'MSGL';
my $MSGD_TAG     = 'MSGD';
my $MSGC_TAG     = 'MSGC';
my $MSG_TAG      = 'MSG';
my $MSG_ARGS_TAG = 'MSG_ARGS';
my $NO_ATTR      = '_NO';
my $TIME_ATTR    = '_TIME';
my $TEXT_ATTR    = '_TEXT';
my $TYPE_ATTR    = '_TYPE';

my $LOGTITLE = "$gProductNameInstaller Log";


##
# Boring Constructor.
##

sub new () {
    my $invocant = shift;
    my $class = ref ($invocant) || $invocant;
    my $self = {
    };
    return bless ($self, $class);
}

#------------------ Public Methods ------------


##
# Parse log file to fill the cache.
#
# @param file
#           filename
##

sub LoadFromFile () {
    my ($self, $file) = @_;
    
    #    
    # Read all lines from file
    #

    $self->{content} = ''; 
    open (FH,"<$file") or return "Can't open file.";
   
	my $bufSize= (stat ($file))[11] || 32768;
	my $buf;
	my $offset=0;
	my $written;
	while(my $len = sysread(FH,$buf,$bufSize)){
		unless(defined $len){
			next if $! =~ /^Interrupted/;
			#AddError ($config, 'Read failure: ' . $!)
			#	if defined $config;
			return 'Read failure: ' . $!;
		}
		$self->{content} .= $buf;
	} 
   
   
    #while (<FH>) {
    #    $self->{content} .= $_;                
    #}
    close FH;

	#
    # Validate file
    #
    
    if ($self->ValidationFile( $self->{content} ) == 0) {
        $self->{cacheTree} = undef;
        return '2';
    } 

    my $result = $self->ParseTheContent( $file );
    if (exists $self->{contentwithlines}) { delete $self->{contentwithlines}; }
    if ($result eq '1' ) {
        return '1';
    }
    else { return $result; }
}


sub AddMsg{
	my ($tree,$msg,$item) = @_;

	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime ($msg->{time});
	
	my %data = (
		'Module' => $msg->{src_file},
		'Code Line' => $msg->{line},
		'Object' => $msg->{obj},
		'Message Type' => (defined $msg->{type} ? ($msg->{type} eq 'WRN' ? 'Warning' :
						$msg->{type} eq 'ERR' ? 'Error' : 'Info') : 'Info'),
		'Time' => sprintf ('%d-%02d-%02d %02d:%02d:%02d.%03d',
								$year + 1900,
								$mon  + 1,
								$mday,
								$hour,
								$min,
								$sec,
								$msg->{msec})
	);
	my $child = $tree->AppendItem ($item, $msg->{text}, \%data);
	if (defined $msg->{submsglst}){
		foreach my $submsg (@{$msg->{submsglst}->{msg_lst}}){
			AddMsg ($tree, $submsg, $child);
		}
	}
}


sub LoadFromMsgLst () {
	my ($self, $msglst) = @_;
	if (defined $self->{cacheTree}) { $self->{cacheTree}->DeleteAllItems; $self->{cacheTree} = undef; } 
	$self->{cacheTree} = SDB::Install::LogViewer::Type::CacheTree->new ();
	my $item = $self->{cacheTree}->AddRoot ('MsgLst',{});
	foreach my $msg (@{$msglst->{msg_lst}}){
		AddMsg ($self->{cacheTree}, $msg,$item);
	}
}





##
# Load log from scalar to fill the cache.
#
# @param file
#           filename
##

sub LoadFromScalar () {
    my ($self, $scal) = @_;
    $self->{content} = $scal;
    
    my $result = $self->ParseTheContent( $LOGTITLE );
    if ($result ne "1" ) {
        return $result;
    }
    else { return '1'; }
}


##
# Returns the plain text from the cache as one scalar.
# Only attributes _TEXT content is used.
#
# @return scalar plain text
#           
##

sub GetPlainText () {
    my ($self) = @_;
    if (defined $self->{cacheTree}) {
        my @List;
        $self->GetTextValue($self->{cacheTree}->GetRootItem(), \@List, "");
        return join "\n", @List;
    } else {
        return $self->{content};
    }
}


##
# Returns the plain text from the cache as an array ref.
# Only attributes _TEXT content is used.
#
# @param
#      TreeElement item        - current item
#      Array Reference listRef - ref to array for results
#      Scalar prefix           - only for spacing (pseudo tabs)
#
# @return array reference to plain text
#           
##

sub GetTextValue() {
    my ($self, $item, $listRef, $prefix) = @_;

    my $formated_time = $item->{data}->{$TIME_ATTR};
    if (!defined $formated_time){
        $formated_time = '';
    }
    $formated_time =~ s/.*\s//;
    my $type = $item->{data}->{$TYPE_ATTR};
    if (!defined $type){
        $type = ' - INFO:';
    }
    elsif ($type eq 'Warning'){
        $type = ' - WRN:';
    }
    elsif ($type eq 'Error'){
        $type = ' - ERR:';
    }
    else{
        $type = ' - INFO:';
    }

    push @{$listRef}, ($formated_time ? $formated_time : '' ). $type . $prefix . $item->{text}; #value is unimportant, only key is important
    #search in children
    my ($node, $cookie) = $self->{cacheTree}->GetFirstChild( $item );
    $prefix .= "  ";
    
    while( defined $node )
    {
        $self->GetTextValue($node, $listRef, $prefix);
        ($node,$cookie) = $self->{cacheTree}->GetNextChild( $item, $cookie );
    }
    
    return $listRef;
}


##
# Returns the whole text without temporary root tags.
#
# @return scalar content
##

sub GetContent {
    my ($self) = @_;
    return substr ($self->{content},(length $ROOTSTARTTAG), (length $self->{content}) - (length $ROOTSTARTTAG) - (length $ROOTENDTAG));
}


##
# Returns the whole text without temporary root tags
# but with line numbers in every line.
#
# @return scalar content with line numbers
##

sub GetContentWithLines {
    my ($self) = @_;
    if (defined $self->{cacheTree}) {
        if (!exists $self->{contentwithlines}) {
            $self->{contentwithlines} = '';
            my $lineCounter = 0;

            foreach my $tmp ( split (/\r?\n/, $self->GetContent())) {
               $self->{contentwithlines} .= sprintf ("%04d: %s\n",++$lineCounter,$tmp);
            }        
        }
        return $self->{contentwithlines};
    } else {
        return "";
    }
}


##
# Return the CacheTree.
#
# @return CacheTree
##

sub GetCacheTree {
    my ($self) = @_;
    return $self->{cacheTree};
}


##
# Reset filter attributes for every known cache tree element.
##

sub ResetFilterAttributes {
    my ($self) = @_;
    
    # set alls filter attributes to nothing
    $self->ResetFilterAttributeForItem( $self->{cacheTree}->GetRootItem() );
    
    1;
}


##
# Reset filter attributes for explicit node and all there childrens.
#
# @param
#   TreeElement item - parent item
##

sub ResetFilterAttributeForItem {
    my ($self, $item) = @_;
    
    my $data = $item->{data};
    
    # set filter attribute to nothing
    if (defined $data) {
        $data->{filter} = 'Require';
        #$item->{data} = $data;
    }
    
    my ($node, $cookie) = $self->{cacheTree}->GetFirstChild( $item );
    
    while( defined $node )
    {
        $self->ResetFilterAttributeForItem( $node );
        ($node,$cookie) = $self->{cacheTree}->GetNextChild( $item, $cookie );
    }
    
    1;
}


##
# Check the filter attributes for all
# nodes.
##

sub SetRequiredToParentsFilterAttributes {
    my ($self) = @_;
    
    my $nodeCache = $self->{cacheTree}->GetNodeCache();
    
    #attantion: unsorted!
	my ($id, $node);

    foreach $id (0 .. (scalar (@$nodeCache) - 1)){
		$node = $nodeCache->[$id];
		if (!defined $node){
			next;
		}
        my $data = $node->{data};
        if ($data->{filter} eq 'Require') {
            $self->SetAllParentsFilterAttributesToRequire( $node );            
        }
    }
    
    1;
}    


##
# Set the filter attributes for the givon node parents to require.
#
# @parent
#     TreeElement node - parent node
##

sub SetAllParentsFilterAttributesToRequire {
    my ($self, $node) = @_;
    
    my $parent = $self->{cacheTree}->GetItemParent ( $node );
    
    # Save for Roots parent
    if ($parent eq "ROOT") { return 1; }
    
    my $data = $parent->{data};
    if (defined $data) {
        if ($data->{filter} eq 'Exclude') {
            $data->{filter} = 'Require';
            $self->SetAllParentsFilterAttributesToRequire ( $parent );
        }
    }
    
    return 1;
}

#------------------ Parse ---------------------


##
# Reset and upgrades the tree with content.
#
# @param
#   scalar rootText    - root text
##

sub ParseTheContent {
    my ($self, $rootText) = @_;
    
    # Precondition
    return "no file content" unless $self->{content};
    return "no roottext" unless $rootText;
    
    eval {
        #parse
        my $expatParser = new XML::Parser::Expat;

        $expatParser->setHandlers (
                Start  => sub {
                        my ($expatParser, $type, %attributes) = @_;
                        
                        my $line = $expatParser->current_line - 1;
                        my %data = ();
                        $data{'Expand'} = 0;
                        if ($type eq $MGSL_TAG) {
                            # Tag
							$lastItem = $self->{cacheTree}->AppendItem ($lastItem, $type, \%data);
                            
                            # Attributes
                            my $addstring = "";
                            foreach my $key ( keys %attributes) {
                                if (($key eq $NO_ATTR) || ($key eq $TIME_ATTR)) {
                                    $addstring = '  '.$key.'='.$attributes{$key}.$addstring;
                                }
                            }
                            #$self->{cacheTree}->SetItemText($lastItem, $self->{cacheTree}->GetItemText( $lastItem ).$addstring);
                            $lastItem->{text} .= $addstring;
						} elsif ($type eq $MSG_TAG) {
                            # Tag
                            
                            # Attributes
                            foreach my $key ( keys %attributes) {
                                if ($key eq $TEXT_ATTR) {
                                    $lastItem = $self->{cacheTree}->AppendItem ($lastItem, $attributes{$key}, \%data);
                                } else {
                                    #all other attributes add to data
                                    #my $pldata = $self->{cacheTree}->GetPlData( $lastItem );
                                    my $pldata = $lastItem->{data};
									$pldata->{$key} = $attributes{$key};
                                    #Set data
                                    #$self->{cacheTree}->SetPlData($lastItem, $pldata);
									#$lastItem->{data} = $pldata
								}
                            }
                        } elsif (($type eq $MSGD_TAG) || ($type eq $MSGC_TAG)) {
                            # indent
                            $lastItem = $self->{cacheTree}->GetLastChild($lastItem);
                        } elsif ($type eq $MSG_ARGS_TAG) {
                            # Tag
                            
                            # Attributes - add tupels to parent MSG Node
                            my %dollars;
                            #my $pldata = $self->{cacheTree}->GetPlData($lastItem);
                            my $pldata = $lastItem->{data};
							foreach my $key ( keys %attributes) {
                                $pldata->{$key} = $attributes{$key};
                                unless ($key =~ /^_/) {
                                    $dollars{$key} = $attributes{$key};
                                }
                            }
                            #Set data
                            #$self->{cacheTree}->SetPlData($lastItem, $pldata);
                           # $lastItem->{data} = $pldata;
							#Set titel
                            if ((scalar keys %dollars) > 0) { #if dollars there
                                #my $tmptitel = $self->{cacheTree}->GetItemText($lastItem);
                                my $tmptitel = $lastItem->{text};
								$tmptitel =~ s/\$//g;
                                foreach my $key (keys %dollars) {        
                                      $tmptitel =~ s/$key/$dollars{$key}/;
                                }
                                #$self->{cacheTree}->SetItemText($lastItem, $tmptitel);
								$lastItem->{text} = $tmptitel;
							}	
                        } 

                    },
                End    => sub {
                        my ($expatParser, $type) = @_;
                        if (($type eq $MSG_TAG) || ($type eq $MGSL_TAG) || ($type eq $MSGD_TAG) || ($type eq $MSGC_TAG)) {
                            #$lastItem = $self->{cacheTree}->GetItemParent ($lastItem);        
							$lastItem = $lastItem->{parent};
						}
                    });
    
        $self->parseXMLtoTree ($expatParser, $rootText);
    
        $expatParser->release();

		delete $self->{content};

        #parse done
    };
    if($@) {
        chomp($@);
        $@ =~ s/^\n//g;
        return $@;
    }
    
    1;
}


##
# Start the expat parsing algorithm.
#
# @param
#   XML::Parser::Expat expatParser - ExpatParser
#   scalar filecontent             - xml content
#   scalar rootText                - root text
##

sub parseXMLtoTree {
    my ($self, $expatParser, $rootText) = @_;

    # Precondition
    return 0 unless $expatParser;
    return 0 unless $rootText;
    
    # Set parser-ignored pseudo root for multiple MSGL parsing
    $self->{content} = $ROOTSTARTTAG.$self->{content}.$ROOTENDTAG;
    
    if (defined $self->{cacheTree}) { $self->{cacheTree}->DeleteAllItems; $self->{cacheTree} = undef; } 
    $self->{cacheTree} = SDB::Install::LogViewer::Type::CacheTree->new ();
    my %data = ();
    $lastItem = $self->{cacheTree}->AddRoot( $rootText,\%data );
    $expatParser->parse( $self->{content} );
    1;
} 


#---------------------------------------------------------------------

##
# Validate if file supported.
#
# @param
#   scalar file - array reference
#
# @return
#   scalar      - if validate 1, otherwise 0 
##

sub ValidationFile () {
   my ($self, $file) = @_;
   return ($file =~ m/^<MSGL/)?(1):(0);
}

##
# Cut scalar if more then 30 characters.
#
# @param
#   scalar name      - string to cut
#   scalar maxlength - max length of string
#   scalar offsetl   - left offset
#   scalar offsetr   - right offset
#
# @return
#   scalar           - new cut string
##

sub cutName () {
    my ($self, $name, $maxlength, $offsetl, $offsetr ) = @_;
    
    if (length $name > $maxlength ) {
        return (substr ($name,0,$offsetl)).'...'.(substr ($name, length ($name) -$offsetr, length ($name)-1));
    }
    
    return $name;
}


#>------------------- Attribute Searching -------------


##
# Get all different attributes values by named attribute from data.
# For every different attribute is one entry.
#
# @param
#   scalar attrname - attribute name
#
# @return
#   array - list of different attribute values
##

sub GetDifferentAttributeValuesByName() {
    my ($self, $attrname) = @_;
    return unless $attrname;
    
    my %List;
    $self->GetAttrValue($self->{cacheTree}->GetRootItem(), $attrname, \%List);
    
    return keys %List;
}


##
# Helper method for GetDifferentAttributeValuesByName
# to traverse the whole tree.
#
# @param
#   Wx::TreeItem item - item node
#   scalar attrname   - attribute name
#   hash listref      - helper hash to store results
#
# @return
#   listref - result hash
##

sub GetAttrValue() {
    my ($self, $item, $attrname, $listRef) = @_;

    # look in data
    my $data = $item->{data};
    
    if (defined $data->{$attrname}) {
        if (defined $data->{filter}) {
          if ($data->{filter} eq 'Require') {
              $listRef->{$data->{$attrname}} = "Lala"; #value is unimportant, only key is important
          }  
        } else {
            $listRef->{$data->{$attrname}} = "Lala"; #value is unimportant, only key is important
        }
    }
    
    #search in children
    my ($node, $cookie) = $self->{cacheTree}->GetFirstChild( $item );
    while( defined $node )
    {
        $self->GetAttrValue($node, $attrname, $listRef);
        ($node,$cookie) = $self->{cacheTree}->GetNextChild( $item, $cookie );
    }
    
    return $listRef;
}

1;