package SDB::Install::Msg;
use Time::HiRes qw (gettimeofday);
use LCM::DevelopmentTrace;
use strict;

our $next_sequence_number = 0;

#----------------------------------------------------------------------------

# 3 params
#  1. $rresult: (out) scalar reference
#  2. $time: (in) seconds since 1.1.1970
#  3. $msec: (in) milliseconds
#
# Result (1st param): time string, e.g. "09:30:12.178"

sub getTimeStamp ($$$){
    my ($sec,$min,$hour) = localtime ($_[1]);
    ${$_[0]} = sprintf ("%02d:%02d:%02d.%03d",$hour, $min, $sec,  $_[2]);
}

#----------------------------------------------------------------------------

# 3 params
#  1. $rresult: (out) scalar reference
#  2. $time: (in) seconds since 1.1.1970
#  3. $msec: (in) milliseconds
#
# Result (1st param): date time string, e.g. "2013-02-06 12:23:13.348"

sub getDateTimeStamp ($$$){
    my ($sec,$min,$hour,$mday,$mon,$year) = localtime ($_[1]);
    ${$_[0]} = sprintf ('%d-%02d-%02d %02d:%02d:%02d.%03d',
                $year + 1900,
                $mon  + 1,
                $mday,
                $hour,
                $min,
                $sec,
                $_[2]
                );
}

#----------------------------------------------------------------------------

sub new{
	my ($class, $type,$text,$level, $obj) = @_;
    my $self = bless ({}, $class);

	unless (defined $level){
		$level = 0;
	}

	$self->{sequence_number} = $next_sequence_number++;
	$self->{text} = $text;
	$self->{type} = $type;
	($self->{module},
	$self->{src_file},
	$self->{line}) = caller ($level);
    if (defined $obj){
        $self->{obj} = "$obj";
    }
    else{
        $self->{obj} = "not defined";
    }
	($self->{time},$self->{msec}) = gettimeofday ();
	$self->{msec} /= 1000;
	$self->{method} = (caller ($level + 1))[3];
	
	if(LCM::DevelopmentTrace::isEnabledDevelopmentTrace()){
		LCM::DevelopmentTrace::Trace(${$self->getMsgString(undef, 1, 1)});
	}
	
	return $self;
}

sub setProgressHandler{
	$_[0]->{progress_handler} = $_[1];
}

sub copy{
    my ($self) = @_;
    my %copy = %$self;
    return bless \%copy, ref($self);
}

sub setMsgText {
    my($self, $text) = @_;
    $self->{text} = $text;
}

sub setEndTag{
    $_[0]->{endtag} = $_[1];
}

sub getRawText {
    return sprintf("%s", $_[0]->{text});
}

#----------------------------------------------------------------------------

sub getType {
    my ($self) = @_;
    return $self->{type};
}

#----------------------------------------------------------------------------

sub isError {
    my ($self) = @_;
    return ($self->getType() eq 'ERR') ? 1 : 0;
}

#----------------------------------------------------------------------------
#
# marks message as highlighted incl. end tag in log file
# required for proper time stamp of end tag
#

sub endMessage{
    my ($self, $withoutDuration, $endTag) = @_;
    ($self->{endtime},$self->{endmsec}) = gettimeofday ();
    $self->{frmtattr} = 'h1';
    if (!defined $withoutDuration || !$withoutDuration){
        $self->{frmtduration} = 1;
    }
    $self->{endmsec} /= 1000;
    if (defined $endTag){
        $self->{endtag} = $endTag;
    }
    return 1;
}

#----------------------------------------------------------------------------

sub getSubMsgLst{
    my ($self) = @_;
    if (!defined $self->{submsglst}){
        $self->{submsglst} = new SDB::Install::MsgLst ();
		if (defined $self->{progress_handler}){
			$self->{submsglst}->setProgressHandler ($self->{progress_handler});
		}
    }
    return $self->{submsglst};
}

#----------------------------------------------------------------------------

sub resetSubMsgLst{
    my ($self) = @_;
    if (defined $self->{submsglst}){
        $self->{submsglst} = undef;
    }
}

#----------------------------------------------------------------------------

sub getMsgString ($$$$){
	my ($self, $indent, $withTimestamp, $withType) = @_;
    
    if (!defined $indent){
		$indent = '';
	}

	my $formatted_time = '';
	my $line_start_wo_time;
    if ($withType){
        my $type = defined $self->{type} ? $self->{type} : 'INFO';
        $type .= ' ' x (4 -length ($type));
        $line_start_wo_time = "$type: $indent";
    }
    else{
        $line_start_wo_time = $indent;
    }
	my $line_start = $line_start_wo_time;
	if ($withTimestamp){
		getTimeStamp (\$formatted_time, $self->{time}, $self->{msec});
		$line_start = $formatted_time . ' - ' . $line_start_wo_time;
	}
	
	my $string = '';
	
	my $format_line;

	if (defined $self->{frmtattr}){
		$format_line = $line_start;
		$format_line .= ('-' x (80 - length ($format_line)));
		$string = $format_line . "\n";
	}
	
	$string .= $line_start . $self->{text} . "\n";

	if (defined $self->{frmtattr}){
		$string .= $format_line . "\n";
	}
	if (defined $self->{submsglst}){
        $string .= ${$self->{submsglst}->getMsgLstString ($indent, $withTimestamp, $withType)};
	}
	if (defined $self->{endtime}){

        my $duration;
        my $duration_msec;
        my $formatted_starttime;
        if (defined $self->{endtime}){
            if ($self->{frmtduration}){
                $formatted_starttime = $formatted_time;
                $duration = $self->{endtime} - $self->{time}; # duration in sec
                $duration_msec = $self->{endmsec} - $self->{msec};
                if ($duration_msec < 0){
                    $duration--;
                    $duration_msec += 1000;
                }
                $duration_msec = int ($duration_msec);
            }
            if ($withTimestamp){
                getTimeStamp (\$formatted_time, $self->{endtime}, $self->{endmsec});
                $format_line = $line_start = $formatted_time . ' - ' . $line_start_wo_time;
                $format_line .= ('-' x (80 - length ($format_line)));
            }
        }
		my $endstr = $self->{endtag};
		if (!defined $endstr) {
		    $endstr =  $self->{text};
		    $endstr =~ s/\.\.\.$//;
		}
		$string .= "$format_line\n";
        $string .= $line_start . "END: $endstr";
        if (defined $duration){
            $string .= (defined $formatted_starttime)
                       ? " (start: $formatted_starttime " : ' (';
            my $formattedDuration = SDB::Install::Tools::formatSecondsIntoHhMmSs($duration);
            $string .= "duration: $formattedDuration)";
        }
		$string .= "\n" . $format_line . "\n";
	}
	return \$string;
}

#----------------------------------------------------------------------------

sub getMsgXML{
    my ($self,$indent) = @_;

    if (!defined $indent){
        $indent = '';
    }

    my $buf = '';

    my $text = $self->{text};

    $text =~ s/&/&amp;/gm;
    $text =~ s/</&lt;/gm;	
    $text =~ s/>/&gt;/gm;
    $text =~ s/\'/&apos;/gm;
    $text =~ s/\"/&quot;/gm;

    my $formatted_time;
    getDateTimeStamp (\$formatted_time, $self->{time},$self->{msec});
    my $formatted_end_time;
    if ($self->{endtime}){
        getDateTimeStamp (\$formatted_end_time, $self->{endtime},$self->{endmsec});
    }

    $buf .=  $indent . '<MSG _NO="' . $self->{sequence_number} . '"' .
                        (defined $self->{type} ? (($self->{type} eq 'WRN') ? ' _TYPE="Warning"' :
                        ($self->{type} eq 'ERR') ? ' _TYPE="Error"' : ($self->{type} eq 'DEBUG') ? ' _TYPE="Debug"' : '') : '') .
                        (defined $self->{frmtattr} ? " _FRMTTYPE=\"$self->{frmtattr}\"" : '').
                        (defined $self->{endtag} ? " _FRMTEND=\"$self->{endtag}\"" : '').
                        ' _ID="0" _COMP="Install" _TEXT="' . $text . '">' . "\n";
    $buf .=  $indent . ' <MSG_ARGS'."\n";
    $buf .=  $indent . '   _FILE="' .$self->{src_file}. "\"\n";
    $buf .=  $indent . '   _LINE="' .$self->{line}. "\"\n";
    $buf .=  $indent . '   _OBJECT="' .$self->{obj}. "\"\n";
    $buf .=  $indent . '   _TIME="' .$formatted_time. "\"\n";
    if (defined $formatted_end_time){
        $buf .=  $indent . '   _ENDTIME="' .$formatted_end_time. "\"\n";
    }
    $buf .=  $indent . '   _MESSAGEVERSION="1"'."\n";
    $buf .=  $indent .  ' />' . "\n";
    $buf .=  $indent . '</MSG>'."\n";
    if (defined $self->{submsglst}){
        $buf .= ${$self->{submsglst}->getMsgLstXMLString ($indent)};
    }
    return \$buf;
}

#----------------------------------------------------------------------------

1;
