package SDB::Install::Recovery;

use strict;
use warnings;

use parent 'SDB::Install::Base';
use SDB::Install::SysVars qw($path_separator);
use SDB::Install::Sql::SqlConnectionProvider;
use SDB::Install::OutputHandler::RecoverSysOutHndlr;
use SDB::Install::LayeredConfig qw(CFG_LAYER_SYSTEM);
use SDB::Install::Tools qw(printTableToArrayOfLines);
use SDB::Common::BuiltIn;

our $recoverScript                = 'recoverSys.py';
our $recoverSnapshotStmntSystemDB = 'RECOVER DATA USING SNAPSHOT CLEAR LOG';
our $recoverSnapshotStmntTenantDB = 'RECOVER DATA FOR %s USING SNAPSHOT CLEAR LOG';
our $requiredPrivilege            = 'DATABASE ADMIN';
our $checkPrivilegeStmnt          = "SELECT PRIVILEGE FROM EFFECTIVE_PRIVILEGES WHERE USER_NAME = CURRENT_USER AND PRIVILEGE = '$requiredPrivilege' AND IS_VALID='TRUE'";
our $checkServicesStmnt           = "SELECT * FROM SYS.M_SERVICES WHERE SERVICE_NAME = 'nameserver' AND ACTIVE_STATUS = 'NO'";
our $checkRunningTenantsStmnt     = "SELECT * FROM SYS.M_DATABASES WHERE DATABASE_NAME <> 'SYSTEMDB' AND ACTIVE_STATUS = 'YES'";
our $checkConnectionsStmnt        = "SELECT c.CONNECTION_ID, c.HOST, c.PORT, c.CLIENT_HOST, c.CLIENT_IP, c.CLIENT_PID, c.START_TIME FROM SYS.M_SESSION_CONTEXT s, SYS.M_CONNECTIONS c WHERE s.KEY = 'APPLICATION' AND (s.VALUE = 'sdbrun' OR s.VALUE = 'HDBLCM') AND s.CONNECTION_ID = c.CONNECTION_ID";
our $renameTenantStmnt            = 'RENAME DATABASE %s TO %s';

sub checkPrivilege{
    my ($self, $sqlConn, $sqlUser) = @_;
    if (!defined $sqlConn->execute($checkPrivilegeStmnt)){
        $self->setErrorMessage ("Cannot check privileges of sql user '$sqlUser'", $sqlConn->getErrMsgLst ());
        return undef;
    }
    my $resultSet = $sqlConn->getResultSet ();
    if (!defined $resultSet){
        $self->setErrorMessage ('Cannot get result set', $sqlConn->getErrMsgLst ());
        return undef;
    }
    if (!@$resultSet){
        $self->setErrorMessage ("Database user '$sqlUser' requires privilege '$requiredPrivilege'");
        return undef;
    }
    return 1;
}

sub _recoverSnapshotTenant{
    my ($self,$sqlConn, $tenantDB,$msglst) = @_;
    my $msg = $msglst->addProgressMessage("Reverting tenant database '$tenantDB' to snapshot ...");
    my $errmsglst = new SDB::Install::MsgLst();
    $sqlConn->setMsgLstContext([$msg->getSubMsgLst, $errmsglst]);
    my $rc = $sqlConn->execute(sprintf($recoverSnapshotStmntTenantDB, $tenantDB));
    if (!$rc){
        $self->getErrMsgLst()->appendMsgLst($errmsglst);
    }
    return $rc;
}

sub _renameTenant{
    my ($self, $sqlConn, $name, $newName, $msglst) = @_;
    my $stmnt = sprintf($renameTenantStmnt,$name,$newName);
    my $submsg = $msglst->addMessage("Renaming tenant database '$name' => '$newName' ...");
    my $errmsglst = new SDB::Install::MsgLst();
    $sqlConn->setMsgLstContext([$submsg->getSubMsgLst(), $errmsglst]);
    if (!$sqlConn->execute($stmnt)){
        $self->getErrMsgLst()->appendMsgLst($errmsglst);
        return undef;
    }
    return 1;
}

sub _renameTenants{
    my ($self,$sqlConn, $tenants, $tenantMap, $msglst) = @_;
    my %existingTenants = map { lc($_->getName) => 1 } @$tenants;
    my $msg = $msglst->addProgressMessage("Renaming tenant databases ...");
    my $submsglst = $msg->getSubMsgLst();
    my $rc = 1;
    foreach my $name (sort {uc($a) cmp uc($b)} keys %$tenantMap){
        my $newName = $tenantMap->{$name};
        $name    = uc($name);
        $newName = uc($newName);
        if (defined $existingTenants{lc($name)}){
            if(!defined $self->_renameTenant($sqlConn,$name,$newName,$submsglst)){
                $rc = undef;
            }
            elsif(defined $rc){
                $rc = 2;
            }
            next;
        }
        if (defined $existingTenants{lc($newName)}){
            $submsglst->addMessage("Tenant database '$name' has been already renamed to '$newName'");
            next;
        }
        $submsglst->addMessage("Neighther tenant database '$name' nor '$newName' exists.");
    }
    return $rc;
}

sub _handleSqlResult{
    my ($self,$sqlConn,$msglst,$withCaption) = @_;
    my $resultSet = $sqlConn->getResultSet($withCaption);
    if (!defined $resultSet){
        $self->setErrorMessage ('Cannot get result set', $sqlConn->getErrMsgLst ());
        return undef;
    }
    my $rows = printTableToArrayOfLines($resultSet,' | ', $withCaption);
    foreach my $row (@$rows){
        $msglst->addMessage($row);
    }
    return $resultSet;
}

sub _checkRunningTenants{
    my ($self, $sqlConn, $msglst) = @_;
    my $msg = $msglst->addProgressMessage("Checking whether there are already running tenants ...");
    if (!defined $sqlConn->execute($checkRunningTenantsStmnt)){
        $self->setErrorMessage ("Cannot check concurrent installer connections", $sqlConn->getErrMsgLst ());
        return undef;
    }
    my $withCaption = 1;
    my $resultSet = $self->_handleSqlResult($sqlConn,$msg->getSubMsgLst(),$withCaption);
    if (@$resultSet > $withCaption){
        $msglst->addMessage("At least one tenant database is already running.");
        return 0;
    }
    return 1;
}


sub _checkInstallerConnections{
    my ($self, $sqlConn, $msglst) = @_;
    my $msg = $msglst->addProgressMessage("Checking concurrent installer connections ...");
    if (!defined $sqlConn->execute($checkConnectionsStmnt)){
        $self->setErrorMessage ("Cannot check concurrent installer connections", $sqlConn->getErrMsgLst ());
        return undef;
    }
    my $withCaption = 1;
    my $resultSet = $self->_handleSqlResult($sqlConn,$msg->getSubMsgLst(),$withCaption);
    if (@$resultSet > ($withCaption + 1)){
        $msglst->addMessage("There is more than one installer connected.");
        return 0;
    }
    return 1;
}

sub _checkSystemDBServices{
    my ($self, $sqlConn, $msglst) = @_;
    my $msg = $msglst->addProgressMessage("Checking system db services ...");
    if (!defined $sqlConn->execute($checkServicesStmnt)){
        $self->setErrorMessage ("Cannot check  system db services", $sqlConn->getErrMsgLst ());
        return undef;
    }
    my $withCaption = 1;
    my $resultSet = $self->_handleSqlResult($sqlConn,$msg->getSubMsgLst(),$withCaption);
    if (@$resultSet != $withCaption){
        $msglst->addMessage("System database doesn't run on all HANA worker nodes yet.");
        return 0;
    }
    return 1;
}

sub _waitForSystemDBServices{
    my ($self,$sqlConn,$msglst) = @_;
    my $retries = 20;
    my $rc;
    my $builtin = SDB::Common::BuiltIn->get_instance();
    my $waitMsg = $msglst->addMessage("Waiting for SystemDB services...");
    my $submsglst = $waitMsg->getSubMsgLst();
    while($retries-- > 0){
        $rc = $self->_checkSystemDBServices($sqlConn,$submsglst);
        if(!defined $rc){
            return undef;
        }
        if($rc){
            last;
        }
        $builtin->sleep(3);
    }
    if (!$rc){
        $self->setErrorMessage("Timeout during waiting for SystemDB services");
    }
    return $rc;
}

sub _isSlaveReady{
    my ($self,$sqlConn,$msglst) = @_;
    my $rc = $self->_checkSystemDBServices($sqlConn,$msglst);

    if (!defined $rc){
        return undef;
    }
    if(!$rc){
        $msglst->addMessage("System database isn't ready yet");
        return 0;
    }
    $rc = $self->_checkRunningTenants($sqlConn,$msglst);
    if (!defined $rc){
        return undef;
    }
    if(!$rc){
        $msglst->addMessage("Revert tenant to snpashot is already done, or in progress.");
        return 0;
    }
    $rc = $self->_checkInstallerConnections($sqlConn,$msglst);
    if (!defined $rc){
        return undef;
    }
    if(!$rc){
        $msglst->addMessage("Skipping revert tenant to snpashot from this host.");
        return 0;
    }
    return 1;
}

sub recoverSnapshotTenants{
    my ($self,$hdbInstance,$instconfig,$msglst) = @_;
    if (!defined $msglst){
        $msglst = $self->getMsgLst();
    }
    my $tenantMsg = $msglst->addProgressMessage("Reverting tenant databases to snapshot ...");
    my $tenantMsgLst = $tenantMsg->getSubMsgLst();

    $hdbInstance->setMsgLstContext([$tenantMsgLst]);
    my $tenants = $hdbInstance->getTenantDatabases(1);
    if (!defined $tenants){
        return undef;
    }

    if(!@$tenants){
        $tenantMsgLst->addMessage("There are no tenant databases");
        return 1;
    }

    my $sqlConn = $self->_getSqlConnection($hdbInstance, $instconfig, $tenantMsgLst);
    if (!defined $sqlConn){
        return undef;
    }

    if (!$self->checkPrivilege($sqlConn, $instconfig->getValue('SystemUser'))){
        return undef;
    }

    my $isScopeInstance = $instconfig->isScopeInstance();
    if ($instconfig->{isSlave}){
        my $isSlaveReady = $self->_isSlaveReady($sqlConn,$tenantMsgLst);
        if (!defined $isSlaveReady){
            return undef;
        }
        if (!$isSlaveReady){
            return 1;
        }
    }
    else{
        # on master host with scope=system
        # waitl until nameserver on all hana hosts are ready
        if (!$self->_waitForSystemDBServices($sqlConn,$tenantMsgLst)){
            return undef;
        }
    }

    my $rc = 1;
    my $tenantMap = $instconfig->getValue('TenantMap');
    if (defined $tenantMap && keys %$tenantMap){
        $rc = $self->_renameTenants($sqlConn, $tenants, $tenantMap, $tenantMsgLst);
        if(defined $rc && $rc > 1){
            # at least one tenant renamed => reload tenant databases
            $tenants = $hdbInstance->getTenantDatabases(1);
        }
    }

    if (!defined $rc){
        return undef;
    }

    foreach my $tenant (@$tenants){
        if(!defined $self->_recoverSnapshotTenant($sqlConn, $tenant->getName(), $tenantMsgLst)){
            $rc = undef;
        }
    }

    return $rc;
}


sub _getSqlConnection{
    my ($self,$hdbInstance,$instconfig, $msglst) = @_;
    my $connectionProvider = new SDB::Install::Sql::SqlConnectionProvider();
    $connectionProvider->setMsgLstContext([$msglst,$self->getErrMsgLst()]);

    my $user         = $instconfig->getValue('SystemUser');
    my $password     = $instconfig->getValue('SrcSQLSysPasswd');
    if(!defined $password){
        $password = $instconfig->getValue('SQLSysPasswd');
    }
    my $userstorekey = $instconfig->getValue('UserStoreKey');
    my $dbname       = $hdbInstance->getSQLDatabaseName();

    return $connectionProvider->waitForSqlConnection(
            $dbname,
            $hdbInstance,
            $user,
            $password,
            $userstorekey);
}

sub _resetLandscapeValue{
    my ($self,$ns_ini,$keyName,$port,$hostmap,$msglst) = @_;
    my $value = $ns_ini->getValue('landscape', $keyName);
    if(!$value){
        return 0;
    }
    my $changed = 0;
    my @new_value;
    my %rHostmap = reverse %$hostmap;
    foreach my $item (split(/\s+/,$value)){
        my ($oldhost, $oldport) = split(':', $item);
        my $newhost;
        my $item_changed = 0;
        if(defined $oldport && $port != $oldport){
            $changed = 1;
        }
        if($hostmap->{$oldhost} && $hostmap->{$oldhost} ne $oldhost){
            $newhost = $hostmap->{$oldhost};
            $changed = 1;
        }elsif(!defined $rHostmap{$oldhost}){
            $msglst->addWarning("Host '$oldhost' is unknown. Removing it from $keyName list!");
            $changed = 1;
            next;
        }
        else{
            $newhost = $oldhost;
        }
        my $newItem = $newhost;
        if (defined $oldport){
            $newItem .= ":$port";
        }
        push @new_value, $newItem;
    }
    if ($changed){
        $ns_ini->setValue (CFG_LAYER_SYSTEM,'landscape',$keyName, join(' ', @new_value));
    }
    return $changed;
}

sub _resetNameServerConfig{
    my ($self, $hdbInstance, $instconfig, $msglst) = @_;
    my $msg = $msglst->addMessage("Resetting nameserver.ini ...");
    my $submsglst = $msg->getSubMsgLst();
    $hdbInstance->setMsgLstContext([$submsglst,$self->getErrMsgLst]);
    my $layered_cfg = $hdbInstance->getLayeredConfig (1);
    if (!defined $layered_cfg){
        return undef;
    }
    my $ns_ini = $layered_cfg->getIniFile ('nameserver.ini');
    if (!defined $ns_ini){
        return undef;
    }
    my $lmsg = $submsglst->addMessage ("Reading nameserver.ini");
    $ns_ini->setMsgLstContext([$lmsg->getSubMsgLst ()]);
    if (!defined $ns_ini->readValues ()){
        $self->setErrorMessage("Cannot read nameserver.ini", $ns_ini->getErrMsgLst());
        return undef;
    }
    $ns_ini->setMsgLstContext([$submsglst]);

    my $port = sprintf("3%02d01",$hdbInstance->get_nr());
    my $changed = 0;
    my $hostmap = $instconfig->getValue ('HostMap');
    foreach my $keyName ('master', 'worker', 'active_master'){
        my $rc = $self->_resetLandscapeValue($ns_ini,$keyName,$port,$hostmap,$submsglst);
        if ($rc){
            $changed = 1;
        }
    }

    if (!$changed){
        return 1;
    }

    $lmsg = $submsglst->addMessage ("Writing nameserver.ini");
    $ns_ini->setMsgLstContext([$lmsg->getSubMsgLst ()]);
    if (!defined $ns_ini->write ()){
        $self->setErrorMessage ("Cannot flush ini file '$ns_ini->{name}'", $ns_ini->getErrMsgLst());
        return undef;
    }
    return 1;
}

sub recoverSnapshot{
    my ($self,$hdbInstance,$instconfig) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage("Reverting database to snapshot ...");
    my $msglst = $msg->getSubMsgLst();
    my $sysdbMsg = $msglst->addProgressMessage("Reverting system database to snapshot ...");
    my $sysdbMsgLst = $sysdbMsg->getSubMsgLst();
    my $script = join($path_separator, 'python_support', $recoverScript);
    # recoverSys.py doesn't support $DIR_INSTANCE starting with /hana/shared
    $hdbInstance->{_instanceDir} = sprintf('/usr/sap/%s/HDB%02d',$hdbInstance->{_sid}, $hdbInstance->{_nr});
    $self->_resetNameServerConfig($hdbInstance,$instconfig,$sysdbMsgLst);
    my $outputHandler = new SDB::Install::OutputHandler::RecoverSysOutHndlr(undef, "  $recoverScript: ");
    $hdbInstance->setMsgLstContext([$sysdbMsgLst, $self->getErrMsgLst()]);
    my @recoverArgs = ("--command=$recoverSnapshotStmntSystemDB", '--wait');
    my $isScopeInstance = $instconfig->isScopeInstance();
    if ($isScopeInstance){
        unshift @recoverArgs, '--masterOnly';
    }
    if (!defined $hdbInstance->runPythonScript(
                                                $script,
                                                \@recoverArgs,
                                                undef,
                                                undef,
                                                undef,
                                                undef,
                                                $outputHandler)){
        return undef;
    }
    if ($isScopeInstance){
        return 1;
    }
    return $self->recoverSnapshotTenants($hdbInstance,$instconfig,$msglst);
}

1;
