package SDB::Install::SAPInstance::TrexInstance;

use File::Spec;
use SDB::Install::SAPInstance;
use SDB::Install::SysVars;
use SDB::Install::Globals;
use SDB::Install::SAPProfiles;
use SDB::Install::NewDBUser;
use SDB::Install::DirectoryWalker;
use SAPDB::Install::Hostname;
use SDB::Common::Utils qw(createSysAdminUserName createXSSpaceSAPUserName createXSSpaceProdUserName);
use SDB::Install::System qw (changeOwn
                             copy_file
                             copySelectedFiles
                             deltree
                             exec_program
                             getDirectoryFilenames
                             is_local_address
                             isSameFile
                             makedir
                             nslookup
                             readLink
                             renameRecursive
                             isEmptyDir
                             listDir
                             $ipv4_regex
                             isDirectory);
use SAPDB::Install::SetUser;
use SDB::Install::LayeredConfig;
use SDB::Install::OSConfig qw ($SOFTLIMIT_NOFILE);
use SDB::Install::Tools qw (trim readini isMountPoint);
use SDB::Install::SAPControl;
use SDB::Install::SysInfo;
use SDB::Install::Log;
use SDB::Install::DeliveryUnit;
use SDB::Install::Manifest;
use SDB::Install::Version;
use SAPDB::Install::ProcState qw(SDB_PROCSTATE_FLAG_SOCKETS SDB_PROCSTATE_FLAG_MODULES);
use SDB::Install::SAPInstance::HanaTraceDirAnalyzer;
use SDB::Install::SAPInstance::TenantDatabase;
use SDB::Install::MonitoredTraceFile;
use SDB::Install::System::IPAddr;
use SDB::Install::Saphostagent;
use SDB::Install::Installer;
use SDB::Install::System::FuserHangChecker;
use Time::HiRes qw (usleep);
use SAPDB::Install::System::Unix qw (sigprocmask sigismember sigemptyset
                                    sigaddset sigfreeset SIG_UNBLOCK SIG_BLOCK
                                    SIG_SETMASK);
use IO::File;
use SDB::Install::OutputHandler::NoOutHndlr;
use SDB::Install::OutputHandler::HdbnameserverOutHndlr;
use File::Basename qw (basename dirname);
use experimental   qw (smartmatch);
use Exporter;

use SDB::Install::FsManager qw(
    $ssfsGlobalKeyFileTemplate
    $ssfsGlobalLockFileTemplate
    $ssfsDirectory
    $systemSsfsDataDirectory
    $systemSsfsKeyDirectory
);

use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::ProcessExecutor;

use Config;
use SDB::Install::SAPInstance::NameserverUtility;
use SDB::Install::SAPInstance::Services::UnregisterHostRoleService;
use SDB::Install::Sql::SqlConnectionProvider;
use SDB::Install::LSS::LssInstance;
use SDB::Common::BuiltIn;

use strict;



our @ISA = qw (SDB::Install::SAPInstance Exporter);

our @EXPORT = qw ($processRestartHandoverPattern);

our $persistence_section = 'persistence';
our $SAPSERVICES_FILE    = '/usr/sap/sapservices';

our $sapstartsrv_stream_template = '/tmp/.sapstream5%02d13';
our $sapstartsrv_log_template = '/tmp/.sapstartsrv%02d_sapstartsrv.log';

our $streamingRenameTool = 'strsidrenamenotify.sh';
our $extendedStorageRenameTool = 'esrename.sh';

our $sidadm_umask = 0027;

our $processRestartHandoverPattern = '^python\s-m\sattachShm.+fast-restart-handover|^hdbrsutil';

my $XSA_OS_USER_SUDOERS_PATTERN = '%s ALL=(%s) NOPASSWD: ALL #HDB_XSA';
# Bug: Bug 108779
my $DISABLE_REQUIRES_TERMINAL_SUDOERS_PATTERN = 'Defaults:%s !requiretty  #HDB_XSA';

use constant MAX_RECONFIG_LISTEN_IF_RETRIES => 10;
#-------------------------------------------------------------------------------
# Parameters of the constructor:
#      $instanceDir string  # '/usr/sap/<sid>' or '<sapmnt>/<sid>' in case of mount point
#     [$sourceSid   string] # source SID (mount point contains target SID)
#     [$user        SDB::Install::NewDBUser] # if user already exists / not used yet

sub new{
	my $self = shift->SUPER::new (@_);
	my (undef,undef,$user) = @_;
	$self->{_version} = undef;
	$self->{_hosts} = [];
	$self->{_host} = undef;
	if (defined $user){
		$self->{_user} = $user;
	}
	else{
		$self->{_user} = new SDB::Install::NewDBUser ($self->{_sid});
	}

	#if (!$self->{_raw}){
	$self->getTrexHostsFromProfileDir ($self->get_profileDir ());
	#}
	return $self;
}

# Returns the local host
sub get_host{
	$_[0]->{_host};
}

# Returns the instance number
sub get_nr{
	$_[0]->{_nr};
}

sub get_sid{
	$_[0]->{_sid};
}

sub get_instanceDir{
	$_[0]->{_instanceDir};
}

sub get_user{
	$_[0]->{_user};
}

sub get_hostNameDir{
    my ($self,
        $wantedHostName, # default is local host name
       ) = @_;

    my $hostName = (defined $wantedHostName) ? $wantedHostName
                                             : $self->{_host};
    if (!defined $hostName) {
        return undef;
    }

    if (defined $self->{_localUsrSapSid}){
        return $self->{_localUsrSapSid} . $path_separator . 'HDB' .
               $self->{_nr} . $path_separator . $hostName;
    }
    return $self->{_instanceDir} . $path_separator . $hostName;
}

sub get_usrSapHostNameDir{
    my ($self,
        $wantedHostName, # default is local host name
       ) = @_;

    my $hostName = (defined $wantedHostName) ? $wantedHostName
                                             : $self->{_host};
    if (!defined $hostName) {
        return undef;
    }

    my $usrSapSid = $self->{_localUsrSapSid};
    if (!defined $usrSapSid){
        if ($isWin){
            return undef;
        }
        $usrSapSid = join ($path_separator, '', 'usr', 'sap', $self->{_sid});
    }
    return join ($path_separator, $usrSapSid, 'HDB' . $self->{_nr}, $hostName);
}

sub get_hostNameTraceDir {
    my ($self, $wantedHost) = @_;
    my $hostDir = $self->get_hostNameDir($wantedHost);
    return (defined $hostDir) ? $hostDir . $path_separator . 'trace' : undef;
}

sub get_localInstanceDir{
	if (defined $_[0]->{_localUsrSapSid}){
		return $_[0]->{_localUsrSapSid} . $path_separator . 'HDB' .
			$_[0]->{_nr};
	}
	$_[0]->{_instanceDir};
}

sub get_lockDir{
	$_[0]->get_hostNameDir() . $path_separator . 'lock';
}

sub getLocalProfileDir {
    my ($self) = @_;
    return File::Spec->catfile('', 'usr','sap', $self->get_sid(), 'SYS', 'profile');
}

sub _getHanaTraceDirAnalyzer{
    my ($self) = @_;
    if (!defined $self->{_hanaTraceDirAnalyzer}){
        $self->{_hanaTraceDirAnalyzer} = new SDB::Install::SAPInstance::HanaTraceDirAnalyzer();
    }
    return $self->{_hanaTraceDirAnalyzer};
}

sub takeTraceDirSnapshot{
    my ($self, $host, $msglst) = @_;
    if (!defined $msglst){
        $msglst = $self->getMsgLst();
    }
    my $traceDirAnalyzer = $self->_getHanaTraceDirAnalyzer();
    $traceDirAnalyzer->setMsgLstContext([$msglst]);
    if (!$traceDirAnalyzer->existsMonitoredTraceDirectory($host)) {
        $traceDirAnalyzer->createMonitoredTraceDirectory($self, $host);
    }
    if ($traceDirAnalyzer->existsMonitoredTraceDirectory($host)){
        $traceDirAnalyzer->takeSnapshot($host);
    }
}

sub analyzeTraceDirDeltas{
    my ($self, $host) = @_;
    my $traceDirAnalyzer = $self->_getHanaTraceDirAnalyzer();
    $traceDirAnalyzer->setMsgLstContext([$self->getMsgLst()]);
    if ($traceDirAnalyzer->existsMonitoredTraceDirectory($host)) {
        $traceDirAnalyzer->analyzeDirectoryDeltas($host, $self->getMsgLst(), 1);
        $self->appendErrorMessage("Check the log file for $gProductNameEngine error messages");
    }
    return 1;
}

sub setCallback{
	$_[0]->{_f_callback} = $_[1];
}


# Returns an array containing the remote hosts
sub get_hosts{
	$_[0]->{_hosts};
}


# If hdbreg or hdbrename is running on a new host,
# none of the existing host names matches the name of the local host.
# So all host names are treated as remote hosts ($self->hasLocalHost returns false).
#
# This method removes the name $oldLocalHost from the array of remote hosts
# and assigns it as local host.

sub setLocalHost {
    my ($self, $oldLocalHost) = @_;

    if (!defined $self->{_host} && defined $self->{_hosts}
        && ($oldLocalHost ~~ @{$self->{_hosts}})) {

        $self->{_host} = $oldLocalHost;
        my @remainingRemoteHosts;
        foreach my $currHost (@{$self->{_hosts}}) {
            if ($currHost ne $oldLocalHost) {
                push @remainingRemoteHosts, $currHost;
            }
        }
        $self->{_hosts} = \@remainingRemoteHosts;
    }
}


# Adds a new hostname to the array of remote hosts.
sub setNewRemoteHost {
    my ($self, $newHost) = @_;

    if (defined $self->{_hosts} && ($newHost ~~ @{$self->{_hosts}})) {
        return 1; # already exists
    }
    push @{$self->{_hosts}}, $newHost;
    return 1;
}


# Returns 1 if any remote host exist
sub exists_remote_host {
    return (@{$_[0]->{_hosts}}) ? 1 : 0;
}

# Returns an array containing the local host (if exists) and the remote hosts
sub get_allhosts{
    [defined $_[0]->{_host} ? $_[0]->{_host} : (), @{$_[0]->{_hosts}}];
}

sub get_globalSidDir {
    return $_[0]->{_globalSidDir};
}

sub get_globalDir{
	join ($path_separator, $_[0]->{_globalSidDir}, 'global');
}

sub get_globalTrexDir{
	join ($path_separator, $_[0]->get_globalDir(), 'hdb');
}

sub get_globalContentDir{
	join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb', 'content');
}

sub get_globalAutoContentDir{
	join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb', 'auto_content');
}

sub get_globalMetadataDir{
	join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb',
			'metadata');
}

sub get_globalTrexInstallDir{
	join ($path_separator, $_[0]->get_globalTrexDir(), 'install');
}


sub get_globalSsfsDir{
    my ($self) = @_;
    my $global_ini = $self->getGlobalIni ();
    if (defined $global_ini){
        $global_ini->setMsgLstContext ([$self->getMsgLst ()]);
        my $customSSFSDir = $global_ini->getValue ('cryptography', 'ssfs_key_file_path');
        if ($customSSFSDir =~ /\S/){
            if ($customSSFSDir =~ /\$\(DIR_GLOBAL\)/){
                return $self->resolveGlobalDir ($customSSFSDir);
            }
            elsif ($customSSFSDir =~ /\$\(DIR_INSTANCE\)/){
                return $self->resolveInstanceDir ($customSSFSDir);
            }
            return $customSSFSDir;
        }
    }
    return join ($path_separator, $_[0]->{_globalSidDir}, $ssfsDirectory);
}

sub get_globalSystemSsfsDataDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, $systemSsfsDataDirectory);
}

sub get_globalSystemSsfsKeyDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, $systemSsfsKeyDirectory);
}

sub getSsfsGlobalLockFileName {
    my ($self) = @_;
    my $sid = $self->get_sid();
    my $globalSSFSDir = $self->get_globalSsfsDir();
    return File::Spec->catfile($globalSSFSDir, sprintf($ssfsGlobalLockFileTemplate, $sid));
}

sub getSsfsGlobalKeyFileName{
    join ($path_separator, $_[0]->get_globalSsfsDir(), sprintf ($ssfsGlobalKeyFileTemplate, $_[0]->{_sid}));
}

sub getSystemSsfsGlobalLockFileName {
    my ($self) = @_;
    my $sid = $self->get_sid();
    my $globalSystemSSFSDir = $self->get_globalSystemSsfsKeyDir();
    return File::Spec->catfile($globalSystemSSFSDir, sprintf($ssfsGlobalLockFileTemplate, $sid));
}

sub getSystemSsfsGlobalKeyFileName{
    join ($path_separator, $_[0]->get_globalSystemSsfsKeyDir(), sprintf ($ssfsGlobalKeyFileTemplate, $_[0]->{_sid}));
}

sub get_globalTrexInstallProgamDir{
    join ($path_separator, $_[0]->get_globalTrexInstallDir(), 'bin');
}

sub get_globalTrexInstallSupportDir{
	join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb',
			'install', 'support');
}

sub get_globalTrexInstallTemplateDir{
    join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb',
            'install', 'config');
}

sub get_globalTodoFileHostnameDir{
    my (
        $self,
        $host
    ) = @_;
    if(not defined $host) {
        $host = $self->{_host};
    }
    join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb',
            'install', 'config', $host);
}

sub get_globalHdbCustomConfigDir{
	join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb',
			'custom', 'config');
}

sub get_globalTrexExeDir{
	join ($path_separator, $_[0]->{_globalSidDir}, 'exe',
			$gPlatform, $isWin ? () : ('hdb'));
}

sub get_globalTrexExeDirAbs{
	readLink ($_[0]->get_globalTrexExeDir ());
}

sub get_instanceExeDir{
	join ($path_separator, $_[0]->{_instanceDir}, 'exe');
}

sub get_pluginsDir{
    return join ($path_separator, $_[0]->get_instanceExeDir (), 'plugins');
}

sub get_globalMdcDir{
    join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb',
            'mdc');
}

sub getMdcDatabasesFileName{
    $_[0]->get_globalMdcDir () . $path_separator . 'databases.lst';
}

sub get_version{
	my ($self,$nocache) = @_;

	$self->{_version} = undef if ($nocache);

	if (!defined $self->{_version}) {
        my $manifest = $self->getManifest ($nocache);
        if (defined $manifest) {
            $self->{_version} = $manifest->getVersion();
        }
	}

	if (!defined $self->{_version}) {
		my $mf = $self->get_instanceExeDir () . $path_separator .
			'saptrexmanifest.mf';
		if (!-f $mf){
			return undef;
		}
		if (!open (FD, $mf)){
			return undef;
		}
		while (my $line = <FD>){
			chomp ($line);
			if ($line =~ /^saptrex\s+version:/){
				$self->{_version} = (split (':',$line))[1];
				$self->{_version} =~ s/^\s*//;
				$self->{_version} =~ s/\s*$//;
				last;
			}
		}
		close (FD);
	}

	return $self->{_version};
}


sub isAutoStartEnabledLocal{
    my ($self) = @_;
    # local host
    if ($self->SUPER::isAutoStartEnabled ()){
        return 1;
    }
    return 0;
}


sub isAutoStartEnabledRemote{
    my ($self) = @_;
    # remote hosts
    foreach my $host (@{$self->{_hosts}}){
        if ($self->SUPER::isAutoStartEnabled ($host)){
            return 1;
        }
    }
    return 0;
}


sub enableAutoStartLocal {
    my ($self, $enable, $msglst) = @_;
    # local host
    return $self->SUPER::enableAutoStart ($enable, undef, $msglst);
}


sub enableAutoStartRemote{
    my ($self, $enable, $msglst) = @_;
    # remote hosts
    my $rc = 1;
    foreach my $host (@{$self->{_hosts}}){
        if (!defined $self->SUPER::enableAutoStart ($enable, $host, $msglst)){
            $rc = undef;
        }
    }
    return $rc;
}

sub set_localUsrSapSidDir{
	$_[0]->{_localUsrSapSid} = $_[1];
}



sub hasLocalHost{
	return defined $_[0]->{_host};
}


#
# _getSAPControlHost($hostname) returns undef if unix domain sockets should be used,
# otherwise the ogininal host name is passed through
#

sub _getSAPControlHostWin{
    return $_[1];
}

sub _getSAPControlHostUx{
    return (defined $_[0]->{_host} && $_[1] eq $_[0]->{_host}) ? undef : $_[1];
}

sub _getSAPControlHost;
*_getSAPControlHost = $isWin ? \&_getSAPControlHostWin : \&_getSAPControlHostUx;


sub asString{
    my ($self,$msglst, $rolesWithProductNames) = @_;
    my $string = join('  ', @{$self->asStringArray($rolesWithProductNames)});
    if (defined $msglst){
        $msglst->AddMessage ($string);
    }
    return $string;
}

sub asStringArray {
    my ($self, $rolesWithProductNames) = @_;
    my @lines;

    push @lines, $self->SUPER::asString();
    push @lines, 'version: '. $self->get_version;

    my $hostOutputList = $self->getAllHostsWithRoles($rolesWithProductNames);
    my $hostInfo       = (scalar @$hostOutputList == 1) ? 'host: ' : 'hosts: ';
    $hostInfo         .= join (', ', @$hostOutputList);
    push @lines, $hostInfo;

    my $manifest = $self->getManifest();
    my $flavour = (defined $manifest && $manifest->getHANAFlavour()) || $gFlavourPlatform;
    my $flavourProductName = getFlavourProductName($flavour);
    push @lines, "edition: $flavourProductName";


    my $pluginInstallations = $self->getPluginInstallations();
    if (defined $pluginInstallations && @$pluginInstallations){
        my @pluginKeys;
        foreach my $inst (@$pluginInstallations){
            push @pluginKeys, $inst->getProductKey();
        }
         push @lines, 'plugins: ' . join (',', @pluginKeys);
    }

	return \@lines;
}

sub isTrex{
	return 0;
}

sub isNewDB{
	return 1;
}

our @sap_services_cache;


#-------------------------------------------------------------------------------
# Returns 1 if the specified hostname parameter contains a valid hostname
# or IP address to access the local host.

sub isHostLocal{
	my ($self, $hostname, $skipPhysicalHostCheck) = @_;
	if (!defined $skipPhysicalHostCheck){
		$skipPhysicalHostCheck = 0;
	}
	if (!$skipPhysicalHostCheck && lc ($hostname) eq lc (hostname ())){
		return 1;
	}

	my $errlst = new SDB::Install::MsgLst ();

	my $address;

	if ($hostname =~ /$ipv4_regex/){
		$address = $hostname;
	}
	else{
		my $nslookup = nslookup ($hostname, $errlst);

		if(!$nslookup){
			$self->AddError ("Cannot resolve host name '$hostname': "
			                .'calling nslookup failed', $errlst);
			return undef;
		}
		elsif ($nslookup->{address}){
			$address = $nslookup->{address};
		}
		else{
		    my $lookupError = $nslookup->{msg};
		    my $errMessage  = "Cannot resolve host name '$hostname'";
		    $errMessage    .= ": $lookupError" if (defined $lookupError);
			$self->AddError ($errMessage);
			return undef;
		}
	}
	my $rc = is_local_address ($address, $errlst);
	if (!defined $rc){
		$self->AddError (undef, $errlst);
		return undef;
	}
	return $rc
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing the names of all profiles.
# Returns undef if the specified directory cannot be opened.

sub getProfileNames {
	my ($self, $profileDir, $msglst) = @_;
	$msglst = $self if (!defined $msglst);

	if (!opendir (DH, $profileDir)){
		$msglst->PushError ("Cannot open hdb profile directory \"$profileDir\": $!");
		return undef;
	}
	my $profile_pattern = '^' . $self->{_sid} . '_HDB\d\d_[^~\s]+$';

	my @profiles = grep {-f "$profileDir/$_" && /$profile_pattern/} readdir (DH);
	closedir (DH);
	return \@profiles
}


sub getTrexHostsFromProfileDir{
	my ($self, $profileDir, $ignore_missing_sapprofileini) = @_;

	my $profiles = $self->getProfileNames($profileDir);
	if (!defined $profiles) {
		return undef;
	}

	my $host;
    my $ignore_remote_sapprofile;
	my $foundHosts = {};
	foreach my $foundHost (@{$self->{_hosts}}) {
	    $foundHosts->{$foundHost} = $foundHost;
	}
	my $hostPattern = sprintf ('HDB%02d_([^_]+)$', $self->{_nr});
	foreach my $profile (@$profiles){
	    ($host) = ($profile =~ /$hostPattern/);
        next if (!$host);
        $self->AddMessage ("Found $gProductNameEngine host: $host");

		if ($isWin && !$self->isHostLocal ($host)){
			$ignore_remote_sapprofile = 1;
		}
		else{
			$ignore_remote_sapprofile = 0;
		}

		if ($ignore_missing_sapprofileini || $ignore_remote_sapprofile ||
			-e "$self->{_instanceDir}/$host/sapprofile.ini"){
			if ($self->isHostLocal($host)){
				$self->{_host} = $host;
                $self->AddMessage ("Host $host is local");
			}
			else{
			    if(not exists $foundHosts->{$host}) {
				    push @{$self->{_hosts}}, $host;
				    $foundHosts->{$host} = $host;
			    }
			}
		}
		else{
			$self->AddWarning ("No hostname dir found in '$self->{_instanceDir}' for host '$host'");
		}
	}

	return 1;
}


#@statuschecked("create instance profile")
sub createInstanceProfile{
	my ($self, $globalhost, $autostart, $mode_installation) = @_;

	if (!defined $globalhost){
		$globalhost = $self->{_host};
	}

	my $prof = new SDB::Install::SAPProfiles ($self->get_profileDir(), $self->{_sid}, $globalhost);

	my $restore = umask ($sidadm_umask) if (!$isWin);

	if (!defined $prof->createInstanceProfile($self->{_nr}, $self->{_host}, $autostart, $mode_installation, $self->getManifest ())){
		$self->AddError ("Cannot create instance profile", $prof);
		umask ($restore) if (!$isWin);
		return undef;
	}
	$self->AddMessage (undef,$prof);
	umask ($restore) if (!$isWin);
	return 1;
}

sub printProfileContent {
    my ($profile) = @_;
    my $file = new IO::File($profile);
    return "Cannot open '$profile': $!" if (!defined $file);

    my @buffer = $file->getlines();
    my $debugMessage = join('', map{"\t".$_} @buffer);
    return "Content of the profile '$profile' is:\n$debugMessage\n";
}

#@statuschecked("create service")
sub createService{
	my ($self, $password) = @_;

	my $pf = join ($path_separator,($isWin ? $self->get_profileDir () :
		sprintf ('/usr/sap/%s/SYS/profile', $self->{_sid})),
				$self->get_profileName ());

    $self->addLazyDebug(\&printProfileContent, $pf);
	my $dir = $self->get_instanceExeDir ();

	my $msg = $self->AddMessage ("Registering sapstartsrv");

	my $prog;
	my $params;
	my $cfg =  {};

	if ($isWin){
		$prog = $self->get_localInstanceDir() . '\exe\sapstartsrv.exe';
		$cfg->{fLogSubst} =
			sub {
				my ($txt) = @_;
				$txt =~ s/(-P\s+)\S+/$1***/;
				return $txt;
			};

        $params = [
                '-r',		# registrate
                '-q',		# quiet, no popups
                '-p', $pf,	# instance profile
                '-s', $self->{_sid},
                '-n', $self->{_nr},
                '-U', $self->{_user}->{serviceuser}->{name},
                '-P', $password,
                '-e', $self->{_user}->getSidAdmName()
		];
	}
	else{
		$prog = $dir . $path_separator . 'sapstartsrv';
		$params = ["pf=$pf", '-reg'];
	}

    my $autostart;
    my $msglst = $msg->getSubMsgLst();
    if ($isWin){
        $autostart = $self->SUPER::isAutoStartEnabled();
        if ($autostart){
            $msglst->addMessage ("Autostart is on");
            my $saveContext = $self->setMsgLstContext ([$msg->getSubMsgLst ()]);
            $self->SUPER::enableAutoStart (0);
            $self->setMsgLstContext ($saveContext);
        }
        else{
            $msglst->addMessage ("Autostart is off");
        }
    }

	#extend/create sapservices
	local %ENV = %ENV if (!$isWin);
	if (!$isWin){
		$ENV{LD_LIBRARY_PATH} = $dir;
	}
	$msglst->injectIntoConfigHash ($cfg);
	my $rc = exec_program ($prog,$params,$cfg);

    if (!defined $rc || $rc != 0){
		$self->AddError ('Registration of sapstartsrv FAILED', $cfg);
	}

    if ($isLinux) {
        $self->checkSAPServicesRlimitNofile(); # ignore errors
    }

    if ($isWin && $autostart){
        my $saveContext = $self->setMsgLstContext ([$msg->getSubMsgLst ()]);
        $self->SUPER::enableAutoStart (1);
        $self->setMsgLstContext ($saveContext);
    }

	if (!defined $rc || $rc != 0){
		return undef;
	}
	return 1;
}


#-------------------------------------------------------------------------------
# Adds or replaces the maximum number of open files in 'the file sapservices.
#
# Returns int retCode

sub checkSAPServicesRlimitNofile {

    my ($self) = @_;

    my $msg        = $self->AddMessage ("Checking os_rlimit_nofile");
    my $limitName  = 'limit.descriptors';

    my $fileLines = $self->readSapservicesFile();

    if (!defined $fileLines) {
        return undef;
    }

    my $oldLimit    = undef;
    my $replaceFile = 0;

    foreach my $i (0 .. (scalar @$fileLines-1)) {

        ($oldLimit) = ($fileLines->[$i] =~ /^$limitName=([0-9]*)/);

        if (defined $oldLimit) {

            if ($oldLimit < $SOFTLIMIT_NOFILE) {

                $replaceFile = 1;

                $self->AddSubMessage($msg, "Changing '$limitName' from "
                       ."$oldLimit to $SOFTLIMIT_NOFILE in '$SAPSERVICES_FILE'");

                $fileLines->[$i] =~ s/$oldLimit/$SOFTLIMIT_NOFILE/;
            }

            last;
        }
    }

    if ($replaceFile || !defined $oldLimit) {

        my $auxFilename = $self->getSapservicesAuxfile();

        if (!open (FD, '>', $auxFilename)) {
            $self->AddError("Cannot create '$auxFilename': $!");
            return undef;
        }

        foreach my $currLine (@$fileLines) {
            if (!print FD "$currLine") {
                $self->AddError("Cannot write into file '$auxFilename': $!");
                return undef;
            }
        }

        if (!defined $oldLimit) {
            my $newLimit = "$limitName=$SOFTLIMIT_NOFILE";

            $self->AddSubMessage($msg, "Adding '$newLimit' to '$SAPSERVICES_FILE'");

            if (!print FD "$newLimit\n") {
                $self->AddError("Cannot write into file '$auxFilename': $!");
                return undef;
            }
        }

        if (!close (FD)) {
            $self->AddError("Cannot close file '$auxFilename': $!");
            return undef;
        }

        if (!$self->renameSapservicesAuxFile($auxFilename)) {
           return undef;
        }
    }
    elsif (!$self->setSapservicesFilePrivileges()) {
        return undef;
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Clears an existing sapservcies auxiliary file
# and returns the auxiliary filename with the process ID as postfix.

sub getSapservicesAuxfile {

    my $auxFilename = $SAPSERVICES_FILE . '~' . $$;

    if (-e $auxFilename) {
        unlink $auxFilename;
    }

    return $auxFilename;
}

#-------------------------------------------------------------------------------
# Returns the lines contained in the file sapservices or undef in case of an error

sub readSapservicesFile {

    my ($self) = @_;

    if (!open(FD, $SAPSERVICES_FILE)) {
        $self->AddError("Cannot read file '$SAPSERVICES_FILE': $!");
        return undef;
    }

    my @lines = <FD>;
    close (FD);

    return \@lines;
}


#-------------------------------------------------------------------------------
# Renames the sapservices file into a backup file and renames the
# specified auxiliary file into sapservices.
#
# Returns int retCode

sub renameSapservicesAuxFile {

    my ($self, $auxFilename) = @_;

    my $bakFilename = getBackupName($SAPSERVICES_FILE);

    my ($dev, $ino, $mode, $nlink, $uid, $gid) = stat($SAPSERVICES_FILE);

    if (!CORE::rename ($SAPSERVICES_FILE, $bakFilename)) {

        $self->AddError("Cannot rename file '$SAPSERVICES_FILE' into '$bakFilename': $!");
        return undef;
    }

    if (!CORE::rename($auxFilename, $SAPSERVICES_FILE)) {

        $self->AddError("Cannot rename file '$auxFilename' into '$SAPSERVICES_FILE': $!");
        CORE::rename($bakFilename, $SAPSERVICES_FILE); # recover
        return undef;
    }

    if (!$self->setSapservicesFilePrivileges($gid)) {
        return undef;
    }

    #remove old backups

    my $keep_backups = 3;
    if (opendir (DH, '/usr/sap')){
        my @backups = sort grep {/^sapservices_\d/} readdir (DH);
        close (DH);
        if (@backups > $keep_backups){
            foreach my $i (0 .. ($#backups - $keep_backups)){
                unlink ('/usr/sap/' . $backups[$i]);
            }
        }
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Sets the privileges of the sapservices file:
#    mode  = 'rwxr-xr-x'
#    user  = root
#    group = group of old sapservices file
#
# Optional Parameter: $oldGroupID # if specified this group ID is set
#
# Returns int retCode

sub setSapservicesFilePrivileges {

    my ($self, $oldGroupID) = @_;

    my ($dev, $ino, $mode, $nlink, $uid, $gid) = stat($SAPSERVICES_FILE);

    my $bakFilename   = getBackupName($SAPSERVICES_FILE);
    my $wantedMode    = 0755; # rwxr-xr-x
    my $wantedUserID  = 0;    # user 'root'
    my $wantedGroupID = (defined $oldGroupID) ? $oldGroupID : $gid;

    if ($mode != $wantedMode) {

        if (!chmod ($wantedMode, $SAPSERVICES_FILE)) {
            $self->AddError("Cannot execute 'chmod ($wantedMode, $SAPSERVICES_FILE)': $!");
            return undef;
        }
    }

    if (($uid != $wantedUserID) || ($gid != $wantedGroupID)) {

        if (!chown ($wantedUserID, $wantedGroupID, $SAPSERVICES_FILE)) {
            $self->AddError("Cannot execute 'chmown ($wantedUserID, $wantedGroupID, $SAPSERVICES_FILE)': $!");
            return undef;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
#@statuschecked("create hostname dirs")
sub createHostNameDirs{
	my ($self) = @_;
	my $msg = $self->AddMessage ("Creating host filesystem");
	$self->SetFormatInfo ($msg, 'h1', 'Create Host Filesystem');
	$self->{_fsMan}->setMsgLstContext([$msg->getSubMsgLst ()]);
	if (!defined $self->{_fsMan}->createHostFS($self->{_host})){
		$self->AddError (undef, $self->{_fsMan});
		return undef;
	}
	return 1;
}

sub setPermissions{
	my ($self, $path, $recursive, $permissions, $msglst) = @_;

	if (!defined $permissions){
		$permissions = 0750;
	}

	if (!defined $msglst){
		$msglst = $self;
	}

	#if (! new SDB::Install::User()->isadmin()){
	#	return 1;
	#}

	if ($isWin and ($path =~ /^\\\\/ or 0)){
		return 1;
	}

	my $msg = $msglst->AddMessage ("Setting permissions of '$path' " .
		($recursive ? 'recursively' : ''));
	$self->{_user}->setMsgLstContext ([$msg->getSubMsgLst ()]);
	if (! defined $self->{_user}->possess($path, $recursive, $permissions)){
		$self->AddWarning ("Cannot possess $path", $self->{_user});
	}
	return 1;
}


#    @statuschecked("adapt sapprofile template")
sub prepareSAPProfileTemplate{
	my ($self) = @_;

	#sapprofile.ini prepared first time as template
	my $spTemplFile =  join($path_separator, $self->get_globalTrexInstallTemplateDir (), 'sapprofile_templ.ini');
	my $spFile =  join($path_separator, $self->get_globalTrexInstallTemplateDir (), 'sapprofile.ini');
	$self->AddMessage ("Adapting $spFile");
	my $sp = new SDB::Install::IniParser::SAPProfile ($spTemplFile);
	if (!defined $sp->read ()){
		$self->AddError ("Cannot read '$spFile'", $sp);
		return undef;
	}

	$sp->setValue ('SAPSYSTEM', $self->{_nr}, undef, 'Basis');
	$sp->setValue ('SAPSYSTEMNAME', $self->{_sid}, undef, 'Basis');
	$sp->setValue ('SAPLOCALHOST', $self->{_host}, undef, 'Basis');
    $sp->setValue ('DIR_PROFILE', $self->getLocalProfileDir());
    $sp->setValue ('DIR_EXECUTABLE', '$(DIR_INSTANCE)/exe');
	if ($isWin){
		$sp->setValue ('SAPGLOBALHOST', $self->{_host}, undef,'Basis');
	}

	if (!defined $sp->write ($gLogDir,$spFile)){
		$self->AddError (undef, $sp);
		return undef;
	}

	return 1;
}

sub checkUpdateTrexInstance {
    my ($self, $instconfig) = @_;
    my $successVal = 1;

    if(defined $instconfig && $instconfig->isa('SDB::Install::Configuration::Upgrade')) {
        my $rc = $self->checkForLogModeToKeep($instconfig);
        if(!$rc) {
            return undef;
        }

        $rc = $self->checkForDefaultTabletypeToKeep($instconfig);
        if(!$rc) {
            return undef;
        }
        $rc = $self->checkForSslClientPki($instconfig);
        if(!$rc) {
            return undef;
        }
        $successVal = 2;
    }

   return $successVal;
}


sub checkForLogModeToKeep {
    my ($self, $instconfig) = @_;

    #
    # find out the log mode
    #
    if(!defined $instconfig->getKeepDefaultLogMode()) {

        my $msg = $self->AddMessage ("Getting log mode...");

        my $layeredCfg = $self->getLayeredConfig ();
        if (!defined $layeredCfg){
            return undef;
        }
        my $global_ini = $layeredCfg->getIniFile ('global.ini');

        if (!defined $global_ini){
            $self->AddError (undef, $layeredCfg);
            return undef;
        }

        if (!defined $global_ini->readValues()){
            $self->AddError (undef, $global_ini);
            return undef;
        }
        my $layer;
        my $log_mode = $global_ini->getValue ('persistence', 'log_mode', \$layer);

        $self->AddSubMsgLst ($msg, $global_ini);

        if ($layer == CFG_LAYER_DEFAULT && ($log_mode eq 'overwrite')){
            $self->AddMessage ("Default log mode parameter will be moved to system layer");
            $instconfig->setKeepDefaultLogMode ($log_mode);
        }
        elsif ($log_mode eq 'legacy'){
            $self->setErrorMessage ("Log mode 'LEGACY' is no longer supported.");
            $self->appendErrorMessage ("Refer to SAP note 1823343.");
            return undef;
        }

    }
    return 1;
}


# find out the default table type. When it is not present in the system layer and set to 'row'
# in the default layer, we have to set it to 'row' in the system layer when it was set
# to 'column' in the default layer after extraction:
sub checkForDefaultTabletypeToKeep {
    my (
        $self,
        $instconfig
    ) = @_;
    if(!defined $instconfig->getKeepDefaultTableType()) {
        my $msg = $self->AddMessage("Getting default table type...");
        my $layeredCfg = $self->getLayeredConfig ();
        if (!defined $layeredCfg){
            return undef;
        }
        my $indexserver_ini = $layeredCfg->getIniFile('indexserver.ini');
        if(!defined $indexserver_ini) {
            $self->AddError(undef, $layeredCfg);
            return undef;
        }
        if(!defined $indexserver_ini->readValues()) {
            $self->AddError(undef, $indexserver_ini);
            return undef;
        }
        my $layer;
        my $default_table_type = $indexserver_ini->getValue('sql', 'default_table_type', \$layer);
        $self->AddSubMsgLst($msg, $indexserver_ini);
        my $kit = $instconfig->{'kit'};
        if(defined $kit) {
            my $mf = $kit->getManifest();
            if(!defined $mf || !$mf->isHanaExpressOrExpressForCockpit()) {
                if ($layer == CFG_LAYER_DEFAULT && ($default_table_type eq 'row')){
                    $self->AddMessage("The current 'default_table_type' parameter setting in the default layer will be moved");
                    $self->AddMessage("to the system layer on completion of this update (thereby keeping it active),");
                    $self->AddMessage("if software extraction sets it to 'column' on the default layer.");
                    $instconfig->setKeepDefaultTableType($default_table_type);
                }
            }
        }
    }
    return 1;
}

sub checkForSslClientPki {
    my (
        $self,
        $instconfig
    ) = @_;
    if(!defined $instconfig->getTurnOffSslClientPki()) {
        my $msg = $self->AddMessage("Getting sslclientpki parameter...");
        my $layeredCfg = $self->getLayeredConfig ();
        if (!defined $layeredCfg){
            return undef;
        }
        my $global_ini = $layeredCfg->getIniFile('global.ini');
        if(!defined $global_ini) {
            $self->AddError(undef, $global_ini);
            return undef;
        }
        if(!defined $global_ini->readValues()) {
            $self->AddError(undef, $global_ini);
            return undef;
        }
        if($global_ini->existsValue('communication', 'sslclientpki')) {
            # we want a "found value in layer" msg in this case:
            $global_ini->getValue('communication', 'sslclientpki');
	        $self->AddSubMsgLst($msg, $global_ini);
        }
        else {
            $self->AddMessage("Parameter 'communication/sslclientpki' not found in 'global.ini' on any layer, prior to update.");
            $self->AddMessage("Setting it to 'off' on system layer after update.");
            $instconfig->setTurnOffSslClientPki();
        }
    }
    return 1;
}

#
# check whether the system is locked down
#

sub checkLicense{
    my ($self, $sql, $msglst) = @_;
    if (!defined $msglst){
        $msglst = $self;
    }
    # use SQL statement that any user is allowed to run
    if (!defined $sql->execute ('select top 1 * from sys.tables')){
        if ($sql->getStmntErrorCode() == 437) {
            $msglst->AddError ('The system has no valid license.', $sql);
            return 0;
        } else {
            $msglst->AddError ('Cannot check license.', $sql);
            return undef;
        }
    }
    return 1;
}

sub migrateParameters{
    my ($self, $srcVersion) = @_;
    my $migrateParamsScript = join ($path_separator, $self->get_instanceExeDir(),
    'python_support',
    'migrate_parameters.py');
    my $msg = $self->getMsgLst ()->addMessage ("Migrating database parameters");

    if (!-f $migrateParamsScript){
        $msg->getSubMsgLst()->addWarning ("Script '$migrateParamsScript' not found => skipping migration");
        return 1;
    }

    if (!$self->runPythonScript (
            $migrateParamsScript,
            ['--sourceVersion',$srcVersion],
            $msg->getSubMsgLst ())){
        $self->setErrorMessage ("Migrating database parameters failed", $msg->getSubMsgLst ());
        return undef;
    }
    return 1;
}


sub saveHdbparamOutput{
    my ($self, $instconfig) = @_;
    my $saveHdbparamOut = 0;
    my $srcVersion;
    if (defined $instconfig && $instconfig->can ('getStartVersion')){
        $srcVersion = $instconfig->getStartVersion ();
    }
    if (defined $srcVersion){
        my $src = new SDB::Install::Version (split ('\.', $srcVersion));
        if ((new SDB::Install::Version(2,0))->isNewerThan ($src)){
            $saveHdbparamOut = 1;
        }
    }

    if (!$saveHdbparamOut){
        return 1;
    }

    my $outputFile = $self->get_globalHdbCustomConfigDir () .
                        $path_separator . 'hdbparam.out';

    my $msglst = $self->getMsgLst ();
    my $msg = $msglst->addMessage ("Saving hdbparam output in '$outputFile'");
    my $output;
    my $rc = $self->runUtilityInHDBEnv ( 'hdbparam', ['-p'], $msg->getSubMsgLst (), \$output);
    if (!defined $output || !$output || ($output !~ /Print\sconfiguration/m)){
        $msglst->addWarning ("File '$outputFile' is not created due to a hdbparam error.");
        return 1;
    }
    my $fh = new IO::File ('>' . $outputFile);
    if (!defined $fh){
        $msglst->addWarning ("Cannot create File '$outputFile': $!");
        return 1;
    }
    $fh->print ($output);
    $fh->close ();
    $self->{_user}->setMsgLstContext ([$msglst]);
    $self->{_user}->possess ($outputFile, 0, 0644);
    return 1;
}



sub migrate{
    my ($self, $instconfig, $fsMan) = @_;

    # save hdbparam parameters
    $self->saveHdbparamOutput ($instconfig);

    if(!$self->migrateLogMode($instconfig)) {
        return undef;
    }
    if(!$self->migrateDefaultTableType($instconfig)) {
        return undef;
    }
    if(!$self->migrateSslClientPki($instconfig)) {
        return undef;
    }
    my $msg;
    $fsMan->set_instanceNr ($self->{_nr});

    my @hosts = $isWin ? ($self->{_host}) : @{$self->get_allhosts()};

    foreach my $host (@hosts){
        $msg = $self->AddMessage ("Migrating host file system (host '$host')...");
        $fsMan->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $fsMan->migrateHost ($host)){
            $self->AddError ("Cannot migrate host filesystem for host '$host'", $fsMan);
            return undef;
        }
    }

    #
    # wdisp profile
    #
    # wdisp profile is obsolete => now configuration is stored in ini file
    return 1;
}

sub migrateLogMode{
    my ($self, $instconfig) = @_;
    my $keep_default_log_mode = $instconfig->getKeepDefaultLogMode ();
    if (defined $keep_default_log_mode){
        my $msg = $self->AddMessage ("Setting log mode in system layer...");

        my $layeredCfg = $self->getLayeredConfig ();
        if (!defined $layeredCfg){
            return undef;
        }
        my $global_ini = $layeredCfg->getIniFile ('global.ini');

        if (!defined $global_ini){
            $self->AddError (undef, $layeredCfg);
            return undef;
        }

        $global_ini->setMsgLstContext ([$msg->getSubMsgLst ()]);

        if (!defined $global_ini->readValues()){
            $self->AddError (undef, $global_ini);
            return undef;
        }
        my $installer = new SDB::Install::Installer();

        if (!defined $global_ini->setValue (
            CFG_LAYER_SYSTEM,
            'persistence',
            'log_mode',
            $keep_default_log_mode,
            sprintf ('set by upgrade tool (%s %s) : keep original default log mode (start version = %s)',
                $installer->GetProgramName(),
                $installer->GetVersion (),
                $instconfig->getStartVersion()))){
            $self->AddError('Cannot save original default log mode', $global_ini);
            return undef;
        }
        else {
             $msg->getSubMsgLst()->addMessage("Log mode was set to '$keep_default_log_mode' on system layer, since");
             $msg->getSubMsgLst()->addMessage("this was the previous setting on default layer.");
        }
        if (!defined $global_ini->write ()){
            $self->AddError('Cannot save global.ini', $global_ini);
            return undef;
        }
    }
    return 1;
}

sub migrateDefaultTableType{
    my ($self, $instconfig) = @_;
    my $keep_default_table_type = $instconfig->getKeepDefaultTableType ();
    if (defined $keep_default_table_type){
        my $msg = $self->AddMessage ("Setting default table type in system layer, if necessary...");
        my $layeredCfg = $self->getLayeredConfig ();
        if (!defined $layeredCfg){
            return undef;
        }
        my $indexserver_ini = $layeredCfg->getIniFile ('indexserver.ini');
        if (!defined $indexserver_ini){
            $self->AddError (undef, $layeredCfg);
            return undef;
        }
        $indexserver_ini->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $indexserver_ini->readValues()){
            $self->AddError (undef, $indexserver_ini);
            return undef;
        }
        my $layer;
        my $default_table_type = $indexserver_ini->getValue('sql', 'default_table_type', \$layer);
        if ($layer == CFG_LAYER_DEFAULT && ($default_table_type eq 'column')){
            my $installer = new SDB::Install::Installer();
            if (!defined $indexserver_ini->setValue (
                CFG_LAYER_SYSTEM,
                'sql',
                'default_table_type',
                $keep_default_table_type,
                sprintf ('set by upgrade tool (%s %s) : keep original default table type (start version = %s)',
                    $installer->GetProgramName(),
                    $installer->GetVersion (),
                    $instconfig->getStartVersion()))){
                $self->AddError('Cannot save original default table type', $indexserver_ini);
                return undef;
            }
            else {
                 $msg->getSubMsgLst()->addMessage("Default table type was set to '$keep_default_table_type' on system layer, since");
                 $msg->getSubMsgLst()->addMessage("this was the previous setting on default layer.");
            }
            if (!defined $indexserver_ini->write ()){
                $self->AddError('Cannot save indexserver.ini', $indexserver_ini);
                return undef;
            }
        }
        else {
             $msg->getSubMsgLst()->addMessage("Default table type setting on system layer, if any, was not changed.");
        }
    }
    return 1;
}

sub migrateSslClientPki{
    my ($self, $instconfig) = @_;
    my $turn_off_ssl_client_pki = $instconfig->getTurnOffSslClientPki();
    if ($turn_off_ssl_client_pki) {
        my $msg = $self->AddMessage ("Setting 'communication/sslclientpki' in 'global.ini' on system layer to 'off'...");
        my $layeredCfg = $self->getLayeredConfig ();
        if (!defined $layeredCfg){
            return undef;
        }
        my $global_ini = $layeredCfg->getIniFile ('global.ini');
        if (!defined $global_ini){
            $self->AddError (undef, $layeredCfg);
            return undef;
        }
        $global_ini->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $global_ini->readValues()){
            $self->AddError (undef, $global_ini);
            return undef;
        }
        my $installer = new SDB::Install::Installer();
        my $comment = sprintf('set by upgrade tool (%s %s) : turned off communication/sslclientpki (start version = %s)',
                $installer->GetProgramName(),
                $installer->GetVersion(),
                $instconfig->getStartVersion());
        my $setValueRc = $global_ini->setValue(CFG_LAYER_SYSTEM, 'communication', 'sslclientpki', 'off', $comment);
        if (!defined $setValueRc) {
            $self->AddError("Cannot turn off 'communication/sslclientpki' on system layer.", $global_ini);
            return undef;
        }
        else {
             $msg->getSubMsgLst()->addMessage("Turned off 'communication/sslclientpki' on system layer.");
        }
        if (!defined $global_ini->write ()){
            $self->AddError("Cannot save 'global.ini'", $global_ini);
            return undef;
        }
    }
    return 1;
}

sub setVolumePaths{
	my ($self, $datapath, $logpath, $global_ini) = @_;
	my $msg = $self->AddMessage ("Setting volume paths");
	my $cfg;
	if ($isWin){
		$cfg = {};
	}
	else{
		$cfg = {
		'mode' => 0750,
		'uid' => $self->{_user}->uidUx(),
		'gid' => $self->{_user}->gidUx()};
	}
	my $submsglst = $msg->getSubMsgLst ();
	$global_ini->setMsgLstContext ([$submsglst]);
	my $saveCntxt = $self->setMsgLstContext ([$submsglst]);
	if ($datapath){
		if (!-e $datapath){
			if (!defined makedir ($datapath, $cfg)){
				$self->setMsgLstContext ($saveCntxt);
				$self->AddError ("Cannot create data volume path", $cfg);
				return undef;
			}
			$submsglst->AddMessage ("Creating data volume path '$datapath'" , $cfg);
			if ($isWin){
				$self->setPermissions ($datapath, 1);
			}
		}
		else{
			$submsglst->AddMessage ("Data volume path '$datapath' already exists");
			$self->setPermissions ($datapath, 1, 0750);
		}
		$datapath = $self->substituteGlobalDir ($datapath);
		$global_ini->setValue (CFG_LAYER_SYSTEM, $persistence_section,'basepath_datavolumes',$datapath);
	}
	if ($logpath){
		if (!-e $logpath){
			if (!defined makedir ($logpath, $cfg)){
				$self->setMsgLstContext ($saveCntxt);
				$self->AddError ("Cannot create log volume path", $cfg);
				return undef;
			}
			$submsglst->AddMessage ("Creating log volume path '$logpath'" , $cfg);
			if ($isWin){
				$self->setPermissions ($logpath, 1);
			}
		}
		else{
			$submsglst->AddMessage ("Log volume path '$logpath' already exists");
			$self->setPermissions ($logpath, 1, 0750);
		}
		$logpath = $self->substituteGlobalDir ($logpath);
		$global_ini->setValue (CFG_LAYER_SYSTEM, $persistence_section,'basepath_logvolumes',$logpath);
	}
	$self->setMsgLstContext ($saveCntxt);
	return 1;
}


sub installSapprofileIni{
	my ($self, $mode) = @_;
	my $srFile = $self->get_hostNameDir () . $path_separator . 'sapprofile.ini';
    my $cfg = {};

	if (!defined copy_file ($self->get_globalTrexInstallTemplateDir () . $path_separator . 'sapprofile.ini', $srFile, $cfg)){
		$self->AddError ("Cannot copy SAPProfile template to '$srFile'", $cfg);
		return undef;
	}
	#sapprofile.ini
	my $sp = new SDB::Install::IniParser::SAPProfile ($srFile);
	if (!defined $sp->read ()){
		$self->AddError ("Cannot read '$srFile'", $sp);
			return undef;
	}
	#if self.fsLayout==FS_LOCAL:
	#    sp.setValue('SAPSYSTEM', self.instance, blockComment='Basis')
	#sp.setValue('SAPSYSTEMNAME', self.sid, blockComment='Basis')
	$sp->setValue ('SAPLOCALHOST', $self->{_host}, 'Basis');
    $sp->setValue ('DIR_PROFILE', $self->getLocalProfileDir());
    $sp->setValue ('DIR_EXECUTABLE', '$(DIR_INSTANCE)/exe');

	if (!defined $sp->write ($gLogDir,undef, $mode)){
		$self->AddError (undef,$sp);
		return undef;
	}
	return 1;
}


sub installHDBScripts{
	my ($self, $msglst) = @_;
	if (!defined $msglst){
		$msglst = $self->getMsgLst ();
	}

	my @copies = $isWin ? qw (HDBAdmin.bat HDBConsole.bat HDBSettings.bat) :
			qw (HDB HDBSettings.sh HDBSettings.csh HDBAdmin.sh xterms hdbenv.sh hdbenv.csh);
	my $cfg = $isWin ? {} : {'chown' => 1};

	my $src_dir = $self->get_globalTrexInstallSupportDir();
	my $dest_dir = $self->get_localInstanceDir;
	my ($src,$dst);
	my ($line, $cd_substituted, $td_substituted);

	my $host = $self->{_host};
	my $msg;

	my $sidadm_upgrade = 0;

	if (!$isWin && $>){
		# nonroot upgrade
		$sidadm_upgrade = 1;
	}

	foreach my $file (@copies){
		$src = $src_dir . $path_separator . $file;
		$dst = $dest_dir . $path_separator . $file;
		if (!open (SRC, $src)){
			$self->AddError ("Cannot open file '$src': $!");
			return undef;
		}

		if ($sidadm_upgrade){
			if (!-w $dst){
				chmod (((stat (_))[2]& 07777) | 0220, $dst);
			}
		}


		if (!open (DST, '>' . $dst)){
			$self->AddError ("Cannot create file '$dst': $!");
			close (SRC);
			return undef;
		}
		$msg = $msglst->addMessage ("Installing script '$dst'");
		$td_substituted = 0;
		$cd_substituted = 0;
		while ($line = <SRC>){
			chomp ($line);
			if ($line =~ s/TARGETDIR/$dest_dir/g){
				$td_substituted = 1;
			}
			if ($line =~ s/CONFIGDIR/$host/g){
				$cd_substituted = 1;
			}
			print DST "$line\n";
		}
		if ($td_substituted){
			$self->AddSubMessage ($msg, "Replaced 'TARGETDIR' with '$dest_dir'");
		}
		if ($cd_substituted){
			$self->AddSubMessage ($msg, "Replaced 'CONFIGDIR' with '$host'");
		}
		close (SRC);
		close (DST);
		$self->setPermissions ($dst,0,((stat ($src))[2] & 07777));
	}
	return 1;
}

#@statuschecked("install host specific inifiles")
sub installHostSpecificInis{
	my ($self) = @_;
	return 1;
	my $dest = $self->get_hostNameDir () . $path_separator . 'crontab.ini';

	if (-e $dest){
		return 1;
	}
	my $src =  $self->get_globalTrexInstallTemplateDir () .  $path_separator . 'crontab.ini';
	my $cfg = {};
	if (!defined copy_file ($src,$dest,$cfg)){
		$self->AddError ("Cannot copy host specific inifiles", $cfg);
		return undef;
	}
	$self->setPermissions ($dest, 0);
	return 1;
}


sub createSYSShortCut{
	my ($self) = @_;

	if (!$self->{_localUsrSapSid}){
		$self->AddError ('local userSapSid is not defined');
		return undef;
	}

	my $dst = $self->{_localUsrSapSid} . $path_separator . 'SYS.lnk';
	require SAPDB::Install::System::Win32::API;
	if (SAPDB::Install::System::Win32::API::CreateShortcut ($self->{_globalSidDir},$dst)){
		$self->AddError ('Creating shortcut failed');
		return undef;
	}
	$self->setPermissions ($dst);
	return 1;


}


sub addHost{
	my($self,$fsManager, $hostName, $instconfig, $sapSys, $sysinfo) = @_;
	$self->{_fsMan} = $fsManager;
	if ($isWin){
		if (!defined $self->createFS ()){
			return undef;
		}
		if (!defined $self->createSYSShortCut ()){
			return undef;
		}
	}
	my $autostart;
	my $master = $self->getMasters();
	if (defined $master){
		$master = $master->[0];
	}
	if (defined $master){
		my $p = new SDB::Install::SAPProfiles ($self->get_profileDir (),
			$self->{_sid});
		my $pf = $p->getInstanceProfile($self->{_nr}, $master);
		my $master_profile = new SDB::Install::IniParser::SAPProfile ($pf);
		if (!$master_profile->read()){
			$self->AddError ("Cannot read instance profile of master nameserver host", $master_profile);
			return undef;
		}
		$autostart = $master_profile->getValue ('Autostart', 0);
		$self->AddMessage ('Autostart is ' . (($autostart) ? 'on' : 'off'));
	}
	return $self->_addHost($hostName, undef, $autostart, $instconfig, $sapSys, $sysinfo);
}

sub enableXSEngineOnHost{
	my ($self, $global_ini) = @_;
	foreach my $service (qw (xsengine webdispatcher)){
            if ($global_ini->getValue($service, 'instances') == 0){
                $global_ini->setValue(CFG_LAYER_HOST, $service, 'instances', 1);
	    }
	}
	return 1;
}


sub substituteGlobalDir{
	my ($self,$path) = @_;
	my $pattern = '^' . quotemeta ($self->get_globalDir ());
	if ($isWin){
		$path =~ s/$pattern/\$\(DIR_GLOBAL\)/i;
	}
	else{
		$path =~ s/$pattern/\$\(DIR_GLOBAL\)/;
	}
	return $path;
}


sub resolveGlobalDir{
	my ($self,$path) = @_;
	my $resolved = $self->get_globalDir ();
	$path =~ s/\$\(DIR_GLOBAL\)/$resolved/;
	return $path;
}


sub resolveInstanceDir{
    my ($self,$path) = @_;
    my $resolved = $self->get_instanceDir ();
    $path =~ s/\$\(DIR_INSTANCE\)/$resolved/;
    return $path;
}


sub getSAPGLOBALHOST{
	my ($self) = @_;

	my $sapglobalhost;

	if ($isWin){
		($sapglobalhost) = ($self->{_instanceDir} =~ /^\\\\([^\\]+)\\sapmnt\\|^\\\\([^\\]+)\\saploc\\/i);
		if (defined $sapglobalhost){
			return $sapglobalhost;
		}
	}

	my $default_profile = (new SDB::Install::SAPProfiles ($self->get_profileDir()))->get_DefaultProfile ();

	my $ini = new SDB::Install::IniFile ($default_profile);

	if (!defined $ini->read ()){
		$self->AddError ("Cannot read default profile", $ini);
		return undef;
	}
	$sapglobalhost = $ini->getValue (undef, 'SAPGLOBALHOST');

	if (defined $sapglobalhost){
		return $sapglobalhost;
	}

	my $hostLocalInstanceDir = $self->{_instanceDir};
	my $hostLocalInstanceDirTmpl;

	if ($isWin and $self->{_instanceDir} =~ /^\\\\/){
		$hostLocalInstanceDirTmpl = '\\\%s\sapmnt\\' . $self->{_sid} . '\HDB' . $self->{_nr};
	}

	if ($hostLocalInstanceDirTmpl){
		$hostLocalInstanceDir = sprintf ($hostLocalInstanceDirTmpl, $self->{_hosts}->[0]);
	}
	my $sapprofile = new SDB::Install::IniParser::SAPProfile (
		join ($path_separator, $hostLocalInstanceDir,
		$self->{_hosts}->[0], 'sapprofile.ini'));
			if (!defined $sapprofile->read ()){
		$self->AddError ("Cannot read SapProfile", $sapprofile);
	}
	$sapglobalhost = $sapprofile->getValue ('SAPGLOBALHOST',0);
	return $sapglobalhost;
}

sub _configureFirstHost {
    my ($self, $instconfig) = @_;
    #
    # system wide parameter
    # volume paths, system usage, multidb mode, and internal network
    #
    my $storageCfgDir = $instconfig->getValue ('StorageConfigDir');
    my $global_ini = $self->getGlobalIni ();
    if (!defined $global_ini){
        return undef;
    }

    if ($storageCfgDir){
        my $msg = $self->AddMessage ("Importing storage configuration");
        $global_ini->setMsgLstContext ([$msg->getSubMsgLst()]);
        my $iniPath = $storageCfgDir . $path_separator . 'global.ini';
        my $storage_ini = new SDB::Install::IniFile ($iniPath);

        $global_ini->importIniFile (CFG_LAYER_SYSTEM, $storage_ini);
    }

    my $systemUsage = $instconfig->getValue ('SystemUsage');
    if (defined $systemUsage){
        if (($self->getSystemUsage() ne $systemUsage) &&
            !defined $self->setSystemUsage($global_ini, $systemUsage)) {
            return undef;
        }
    }

    my $dbMode = $instconfig->getValue ('DbMode');
    if (defined $dbMode){
        my $dbIsolation = $instconfig->getValue ('DbIsolation');
        if (!defined $self->setDbMode($global_ini, $dbMode, $dbIsolation)) {
            return undef;
        }
    }

    if (!defined $self->setRepositoryParam ($instconfig)){
        return undef;
    }

    my $allocationLimit = $instconfig->getValue ('MaxMem');
    # only change global allocation limit if would be set to something else than 0
    if ($allocationLimit){
        if (($self->getGlobalAllocationLimit() ne $allocationLimit) &&
            !defined $self->setGlobalAllocationLimit($global_ini, $allocationLimit)) {
            return undef;
        }
    }

    my $internalNetworkIP = $instconfig->getInternalNetworkIP();
    if ($internalNetworkIP){
        $self->setInternalNetworkPrefix ($internalNetworkIP, $global_ini);
    }

    my $defaultDatapath = $instconfig->getDefaultVolumePath('data',1);
    my $defaultLogpath  = $instconfig->getDefaultVolumePath('log',1);
    my $datapath = $instconfig->getValue ('BasePathDataVolumes');

    if (defined $datapath){
      my @statbuf = $isWin ? undef : stat ($defaultDatapath);
      if (isSameFile ($datapath, $defaultDatapath,
              $isWin ? undef : \@statbuf)){
          $datapath = undef;
      }
      elsif ($defaultDatapath eq $datapath){
          $datapath = undef;
      }
    }
    my $logpath = $instconfig->getValue ('BasePathLogVolumes');
    if (defined $logpath){
      my @statbuf = $isWin ? undef : stat ($defaultLogpath);
      if (isSameFile ($logpath, $defaultLogpath,
              $isWin ? undef : \@statbuf)){
          $logpath = undef;
      }
      elsif ($defaultLogpath eq $logpath){
          $logpath = undef;
      }
    }

    if (!defined $datapath && defined $defaultDatapath){
      if (-e $defaultDatapath){
          $self->setPermissions ($defaultDatapath, 1, 0750);
      }
    }

    if (!defined $logpath && defined $defaultLogpath){
      if (-e $defaultLogpath){
          $self->setPermissions ($defaultLogpath, 1, 0750);
      }
    }

    if ($datapath || $logpath){
      if (!defined $self->setVolumePaths ($datapath, $logpath, $global_ini)){
          return undef;
      }
    }
    my $pMemPath = $instconfig->getValue('BasePathPMem');
    if (defined $pMemPath){
        my $pMemLst = $instconfig->getPMemLst();
        if (defined $pMemLst){
            my $pMemValue = "$pMemPath$path_separator" .
                    join (";$pMemPath$path_separator", @$pMemLst);

            $global_ini->setValue(
                CFG_LAYER_SYSTEM,
                $persistence_section,
                'basepath_persistent_memory_volumes',
                $pMemValue
            );
        }
    }

    $global_ini->setOwner ($self->{_user}->uid(), $self->{_user}->gid());
    my $msg = $self->getMsgLst()->addMessage ("Writing global.ini");
    $global_ini->setMsgLstContext ([$msg->getSubMsgLst()]);
    delete $self->{global_ini};
    if (!defined $global_ini->write()){
        $self->AddError ("Cannot set global ini values", $global_ini);
        return undef;
    }

    return $self->createService ($instconfig->getValue('Password'));
}

sub initTopology {
    my ($self, $instconfig) = @_;
    if (!$instconfig->getValue('CreateInitialContainer')) {
        if (!defined $self->handleSecureStore($instconfig)) {
            return undef;
        }
        if($instconfig->getValue('VolumeEncryption')){
            $self->enableVolumeEncryption();
        }

        my $msg = $self->AddMessage ("Installing initial topology");
        $self->SetFormatInfo ($msg, 'h1', 'Install initial topology');
        my $submsglst = $msg->getSubMsgLst ();
        my $initTopologyArgs = [];
        my $listenInterface = $instconfig->getValue('ListenInterface');
        my $internalLocalIp = $instconfig->getInternalDeviceIP ();
        my $shouldSetListenInterface = $listenInterface && $listenInterface ne 'local';
        if ($shouldSetListenInterface) {
            push @{$initTopologyArgs}, "--hostnameResolution=$listenInterface";
            if ($internalLocalIp) {
                push @{$initTopologyArgs}, "--internalAddress=$internalLocalIp";
            }
        }
        my $workerGroupString = $instconfig->getValue ('WorkerGroup');
        if (defined($workerGroupString) && $workerGroupString){
            my @workerGroups = split(',', $workerGroupString);
            for my $group (@workerGroups){
                push(@{$initTopologyArgs}, "--workergroup=$group");
            }
        }

        if ($self->isMultiDb())
        {
          if (!$instconfig->getValue ('CreateInitialTenant')){
              push @$initTopologyArgs, '--no_initial_tenant';
          }
          elsif(!$instconfig->getValue ('EnableXSEngine')){
              push @$initTopologyArgs, '--no_initial_tenant_xs';
          }
        }

        my $sql_system_password = $instconfig->getValue('SQLSysPasswd');
        my $do_not_change_password = ($instconfig->isSkipped('SQLSysPasswd')) || ($sql_system_password eq 'manager');
        my @systemUserPWInputBuffer = ("$sql_system_password",
                                       "$sql_system_password");
        if(!$do_not_change_password) {
            if($self->isMultiDb() && $instconfig->getValue('CreateInitialTenant')){
                push @systemUserPWInputBuffer, @systemUserPWInputBuffer;
            }
            push @$initTopologyArgs, '--set_user_system_pw';
        }
        if (!defined $self->installTopology ($initTopologyArgs , $submsglst, \@systemUserPWInputBuffer)){
            $self->AddError ("Cannot create initial topology", $submsglst);
            return undef;
        }
    }
    return 1;
}

sub _configureAdditionalHost {
    my ($self, $instconfig, $sapSys, $hostname, $configureLss, $specialRoleMap) = @_;
    my $password = $instconfig->getValue('Password');

    if ($instconfig->usesNonSharedStorage()){
        # hdbaddhost
        my $cfg;
        my @volumePaths = ($self->getDataPath (1),
                          $self->getLogPath (1));

        my $mode = 0750;
        my $uid = $self->{_user}->uidUx();
        my $gid = $self->{_user}->gidUx();
        foreach my $path (@volumePaths){
            if (!$path){
                next;
            }
            if (!-e $path){
                $cfg = {
                    'mode' => $mode,
                    'uid'  => $uid,
                    'gid'  => $gid};
                if (!defined makedir ($path, $cfg)){
                    $self->AddError ("Cannot create volume path", $cfg);
                    return undef;
                }
            }
            else{
                $self->setPermissions ($path, 0, $mode);
            }
        }
    }
    if ($configureLss && $sapSys->isOutstandingAddHostStepConfigureLss($instconfig)) {
        my $lssInstance = $self->getLssInstance(1);
        $lssInstance->setMsgLstContext($self->getMsgLstContext());
        return undef if(!$lssInstance->configureDatabaseHost($sapSys, $instconfig->getValue('Hostname'), $instconfig));
    }
    if ($sapSys->isOutstandingAddHostStepExecuteHdbnsutil($instconfig) && !defined $self->registerInLandscape($instconfig)) {
        $self->AddProgressMessage("Registration of host '$hostname' in the landscape failed. Trying to revert changes made by the hdbnsutil and retry.");
        $self->runNameServerUtility(['-removeHost', $hostname]);
        return undef if (!defined $self->registerInLandscape($instconfig));
    }
    if ($sapSys->isOutstandingAddHostStepCreateService($instconfig) && !defined $self->createService($password)) {
        return undef;
    }
    if ($sapSys->isOutstandingAddHostStepStartSystem($instconfig) && !defined $sapSys->addHostStartInstance($password, $instconfig, $self)) {
        return undef;
    }
    if ($specialRoleMap && !$self->addSpecialHosts($hostname, $specialRoleMap, $instconfig, $sapSys)) {
        return undef;
    }
    return 1;
}

sub _addHost{
	my ($self, $hostname, $mode_installation, $autostart,
	$instconfig, $sapSys, $sysinfo, $progress_handler) = @_;

	if ($self->isHostLocal($hostname)){
		$self->{_host} = $hostname;
	} else {
		$self->AddError ("Hostname '$hostname' does not specify the local host");
		return undef;
	}
	delete $self->{_hostRolesInfo};

	if (defined $self->{layered_cfg}){
		# reset layered config due to new host layer
		$self->{layered_cfg} = undef;
	}

	my $firstHost = ($self->exists_remote_host()) ? 0 : 1;

	my $sapglobalhost;

	if ($sapSys->isOutstandingAddHostStepInitHostDir($instconfig)) {
		if (!defined $self->createHostNameDirs ()){
			return undef;
		}
	}

	#self.createApacheConfig()

	#--------------------------------------------------------------
	#sapprofile.ini
	#--------------------------------------------------------------
	if ($firstHost){
		if (!defined $self->prepareSAPProfileTemplate ()){
			return undef;
		}
	}

	if ($sapSys->isOutstandingAddHostStepInstallSAPProfile($instconfig)) {
		if (!defined $self->installSapprofileIni()){
			return undef;
		}
		if (!defined $self->installHostSpecificInis()) {
			return undef
		}
	}

	$self->setPermissions ($self->get_globalTrexInstallTemplateDir (), 1);
	$self->setPermissions ($self->get_localInstanceDir () . $path_separator .  $hostname, 1);
	$self->{_raw} = 0;

	if ($isWin && !$mode_installation){
		$self->installHDBScripts ();
	}

	if ($sapSys->isOutstandingAddHostStepCreateInstProfile($instconfig)) {
		if (!defined $self->createInstanceProfile($sapglobalhost,
		                                          $autostart,
		                                          $mode_installation)){
			return undef;
		}
	}

	#previously the Web Dispatcher profile had been generated here
	#but since the Web Dispatcher is configured using ini-files this is not necessary anymore

	my $specialRoleMap   = {};
	my $doAddSpecialHost = 0;
    my $configureLss = 0;

	if ($firstHost){

		my $daemon_ini = $self->readDaemonIni();
		if (!defined $daemon_ini) {
			return undef;
		}

		my $msg = $self->AddMessage ("Adjusting daemon.ini");

		$daemon_ini->setMsgLstContext ([$msg->getSubMsgLst(), $self->getErrMsgLst()]);
		if ($instconfig->getValue ('EnableXSEngine')){
			my $dbMode = $instconfig->getValue ('DbMode');
			if (!(defined $dbMode && $dbMode eq 'multidb')){
				$self->enableXSEngineOnHost ($daemon_ini);
			}
		}
		if (!defined $daemon_ini->write ()){
			return undef;
		}
    }else {
		# hdbaddhost
		# if es/streaming (not hana worker or standby), configure only nameserver in deamon.ini
        my $lssInstance = $self->getLssInstance(1);
		my $addRoles = $instconfig->getHostRoles();
		if (defined $addRoles) {

			my $switchOffHanaService = 1;
			foreach my $currRole (@$addRoles) {
				if (($currRole eq $gHostRoleWorker) ||
				    ($currRole eq $gHostRoleStandby)) {
					$switchOffHanaService = 0;
                    $configureLss = $lssInstance->isInstalled() && !$lssInstance->isConfiguredOnHost();
				}
				elsif (($currRole eq $gHostRoleAcceleratorWorker) ||
				       ($currRole eq $gHostRoleAcceleratorStandby)) {

					$self->{addAcceleratorHostPending} = $currRole;
				}
				else { # without accelerator <##  (background ?) ## how can $specialRoleMap->{$gHostRoleAccelerator*} ever be set?
					$specialRoleMap->{$currRole} = 1;
					$doAddSpecialHost            = 1;
				}
			}

			if ($switchOffHanaService || $doAddSpecialHost) {

				if (!$self->writeSpecialHostServersIntoDaemonIni
				                     ($switchOffHanaService, undef)) {
					return undef;
				}
			}
		}
	}

	$self->createHWInfoFile ($sysinfo, $hostname);
    if ($mode_installation) {
        return $self->_configureFirstHost($instconfig);
    } else {
        return $self->_configureAdditionalHost(
            $instconfig,
            $sapSys,
            $hostname,
            $configureLss,
            $doAddSpecialHost ? $specialRoleMap : undef,
        );
    }
}

sub handleSecureStore {
    my ($self, $instconfig) = @_;
    my $msglst = $self->getMsgLst();

    if (!$instconfig->isUpdate()) {
        if ($instconfig->shallUseLSS()){
            my $msg = $msglst->addMessage("Configuring $gProductNameLSS...");
            return $self->createSecureStore($instconfig, $msg->getSubMsgLst());
        }
        return $self->createSecureStoreAndSSFSKeys($instconfig);
    }

    if ($self->isLSSConfigured($msglst) || !$instconfig->shallUseLSS()) {
        return 1;
    }

    my $msg = $msglst->addMessage("Migrating secure store to $gProductNameLSS...");
    return $self->migrateSecureStore($gSecureStoreTypeLSS, $instconfig, $msg->getSubMsgLst());
}

sub isLSSConfigured {
    my ($self, $msglst) = @_;
    my $currentSecureStore = $self->getCurrentSecureStore();
    if ($currentSecureStore eq 'LSS') {
        $msglst->addMessage("Secure store already configured as '$gSecureStoreTypeLSS'");
        return 1;
    }
    return 0;
}

sub createSecureStoreAndSSFSKeys{
    my ($self, $instconfig) = @_;
    my $msglst = $self->getMsgLst ();
    my $msg = $msglst->addMessage ("Creating SecureStore");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst ()]);
    my $keyLockName = $self->getSsfsGlobalLockFileName();
    $self->setMsgLstContext ($saveCntxt);
    if (-f $keyLockName){
        $msg->getSubMsgLst ()->addWarning ("SSFS key lock '$keyLockName' found");
        if (!unlink ($keyLockName)){
            $self->setErrorMessage ("Cannot remove SSFS key lock '$keyLockName': $!");
            return undef;
        }
    }
    my $rc = $self->createSecureStore($instconfig, $msg->getSubMsgLst());
    $msg->endMessage (0, 'Create SecureStore');
    if (!defined $rc){
        $self->setErrorMessage ("Cannot create SecureStore", $msg->getSubMsgLst ());
        return undef;
    }
    if ($instconfig->getValue('ChangeInitialSSFSKey')) {
        my $keyName = $self->getSsfsGlobalKeyFileName();
        if (!-f $keyName){
            $msg = $msglst->addMessage ("Creating SSFS global key file...");
            $rc = $self->createSsfsGlobalKey ($msg->getSubMsgLst ());
            $msg->endMessage (0, 'Create SSFS global key file');
            if (!defined $rc){
                $self->setErrorMessage ("Cannot create global key file", $msg->getSubMsgLst ());
                return undef;
            }
        } else {
            $msglst->addMessage ("SSFS global key file '$keyName' already exists)")
        }
        $keyName = $self->getSystemSsfsGlobalKeyFileName ();
        if (!-f $keyName){
            $msg = $msglst->addMessage ("Creating SSFS global key file (system PKI)...");
            $keyLockName = $self->getSystemSsfsGlobalLockFileName();
            if (-f $keyLockName){
                $msg->getSubMsgLst ()->addWarning ("SSFS key lock '$keyLockName' found");
                if (!unlink ($keyLockName)){
                    $self->setErrorMessage ("Cannot remove SSFS key lock '$keyLockName': $!");
                    return undef;
                }
            }
            $rc = $self->createSsfsGlobalKey ($msg->getSubMsgLst (), 1);
            $msg->endMessage (0, 'Create SSFS global key file (system PKI)');
            if (!defined $rc){
                $self->setErrorMessage ("Cannot create global key file", $msg->getSubMsgLst ());
                return undef;
            }
        } else {
            $msglst->addMessage ("SSFS global key file (system PKI) '$keyName' already exists)")
        }
    }
    return 1;
}

sub enableVolumeEncryption {
    my ($self) = @_;
    my $globalIni = $self->getGlobalIni();
    my $msgLst = $self->getMsgLst();
    if (!defined $globalIni){
        $msgLst->addError('Cannot access global.ini');
        return undef;
    }

    $msgLst->addMessage('Updating global.ini');
    $globalIni->setMsgLstContext($self->getMsgLstContext());

    foreach my $param ('persistence_encryption', 'log_encryption'){
        $globalIni->setValue(CFG_LAYER_SYSTEM, 'database_initial_encryption', $param, 'on');
    }

    if (!defined $globalIni->write()){
        return undef;
    }
    return 1;
}

sub migrateSecureStore {
    my ($self, $secureStoreType, $instconfig, $msglst) = @_;
    my $nsutil = $self->getNameServerUtility();
    $msglst //= SDB::Install::MsgLst->new();
    $nsutil->setMsgLstContext([$msglst]);

    if (!$self->ensureTopologyFile($nsutil, $instconfig, $msglst)) {
        $self->setErrorMessage("Failed to ensure existing topology file.", $nsutil->getErrMsgLst());
        return undef;
    }

    my $target = $secureStoreType eq $gSecureStoreTypeLSS ? 'LSS' : 'SSFS';
    if (!$nsutil->migrateSecureStore($target)) {
        $self->setErrorMessage("Failed to migrate $gProductName secure store to $secureStoreType.", $nsutil->getErrMsgLst());
        return undef;
    }

    return 1;
}

sub ensureTopologyFile {
    my ($self, $nsutil, $instconfig, $msglst) = @_;
    my $topologyFile = $self->getTopologyFilename();
    return 1 if (File::stat::stat($topologyFile));

    my $errlst = $self->getErrMsgLst();
    my $isSecondary = $self->isSecondarySystem($msglst, $errlst);
    if (!defined $isSecondary) {
        return 0;
    }
    if (!$isSecondary) {
        return $nsutil->createOfflineTopology($topologyFile);
    }
    # Bug: 254121
    # Generate topology file via system restart for secondary systems
    my $sapSys       = $instconfig->getSAPSystem();
    my $userName     = $sapSys->getUser()->getSidAdmName();
    my $password     = $instconfig->getValue('Password');
    my $startTimeout = $instconfig->getTimeout('stop_service'),
    my $stopTimeout  = $instconfig->getTimeout('stop_instance'),
    my $useHttps     = $instconfig->isUseHttps();

    my $util = SDB::Install::SAPSystemUtilities->new();
    $util->setMsgLstContext([$msglst]);
    return $util->startStopSystem($self, $userName, $password, $startTimeout, $stopTimeout, $useHttps);
}

sub getTopologyFilename {
    my ($self, $host) = @_;
    $host //= $self->get_host();

    my $hostDir = $self->get_hostNameTraceDir($host);
    my $topologyFileName = sprintf("nameserver_topology_%s.json", $host);
    return File::Spec->catfile($hostDir, $topologyFileName);
}

#-------------------------------------------------------------------------------
# Calls specific scripts to add special hosts (i.e. extended storage host,
# streaming host, remote data sync host)

sub addSpecialHosts {

    my ($self,
        $hostname,
        $roleMap, # e.g. { $gHostRoleStreaming => 1, $gHostRoleEsWorker => 1 }
        $instconfig,
        $sapSys
    ) = @_;

	delete $self->{_hostRolesInfo};

	if ($roleMap->{$gHostRoleStreaming}
	    && $sapSys->isOutstandingAddHostStepCreateStreaming($instconfig)) {
        return undef if(!$self->addStreamingHost($hostname, $instconfig));
	}

	if (($roleMap->{$gHostRoleEsWorker} || $roleMap->{$gHostRoleEsStandby})
	    && $sapSys->isOutstandingAddHostStepCreateES($instconfig)) {
        return undef if(!$self->addEsHost($hostname, $instconfig, $roleMap));
	}

	return 1;
}

sub addStreamingHost {
    my ($self, $hostname, $instconfig) = @_;
    my $tenantUser = $instconfig->getValue('TenantUser');
    my $tenantPassword = $instconfig->getValue('SQLTenantUserPassword');

    $self->createStreamingBasepath();
    my $addhostScript = 'straddhost';
    my $straddhost    = File::Spec->catfile($self->getStreamingLcmDir(), $addhostScript . '.sh');
    my $msg = $self->getMsgLst ()->addProgressMessage ("Performing $addhostScript...");

    my @args = ('-host', $hostname, '-log_dir', $gLogDir);
    my $inputBuffer = [];
    if ($self->shouldAutoInitializeService($instconfig, $gDirNameStreaming)) {
        if (!defined $tenantUser || !defined $tenantPassword) {
            $self->getErrMsgLst->addMessage("Missing tenant user credentials to add '$gDirNameStreaming' host");
            return undef;
        }
        push(@args, '-auto_initialize_services');
        push(@args, ('-tenant_user', $tenantUser));
        $inputBuffer = ["$tenantPassword"];
    }
    my $rc = $self->runUtilityInEnv ($straddhost, \@args, $msg->getSubMsgLst, undef, $inputBuffer);
    if (!$rc){
        $self->setErrorMessage ("$addhostScript failed", $msg->getSubMsgLst ());
        return undef;
    }
    return 1;
}

sub createStreamingBasepath {
    my ($self, $basepathStreaming) = @_;
    $basepathStreaming //= $self->getStreamingBasepath();
    if (!defined($basepathStreaming)) {
        my $message = sprintf("Could not get the value of streaming basepath " .
                              "from the %s's environment.", $self->{_user}->getSidAdmName());
        $self->getMsgLst()->addWarning($message);
        return 0;
    }
    return 1 if (isDirectory($basepathStreaming));
    my $cfg = {'mode' => 0755};
    if (!makedir($basepathStreaming, $cfg)) {
        $self->getMsgLst()->addWarning("Could not create streaming basepath '$basepathStreaming'", $cfg);
        return 0;
    }
    my $uid = $self->{_user}->uid();
    my $gid = $self->{_user}->gid();
    my $builtin = SDB::Common::BuiltIn->get_instance();
    if (!$builtin->chown($uid, $gid, $basepathStreaming)) {
        $self->getMsgLst()->addWarning("Could not change ownership of streaming basepath '$basepathStreaming': $!");
        return 0;
    }
    if (!$builtin->chmod(0750, $basepathStreaming)) {
        $self->getMsgLst()->addWarning("Could not change mode of streaming basepath '$basepathStreaming': $!");
        return 0;
    }
    return 1;
}

sub getStreamingBasepath {
    my ($self) = @_;
    my $sidadm = $self->{_user}->getSidAdm();
    my $env = $sidadm->getInitialEnvironment();
    if (!defined($env) || !exists($env->{STREAMING_SHARED})) {
        return undef;
    }
    return $env->{STREAMING_SHARED};
}

sub addEsHost {
    my ($self, $hostname, $instconfig, $roleMap) = @_;
    my $addhostScript = 'esaddhost';
    my $esaddhost     =  File::Spec->catfile($self->getExtendedStorageLcmDir(), $addhostScript . '.sh');
    my $msg = $self->getMsgLst ()->addProgressMessage ("Performing $addhostScript...");

    my @args = ('-host', $hostname, '-log_dir', $gLogDir);
    if ($roleMap->{$gHostRoleEsWorker} && $self->shouldAutoInitializeService($instconfig, $gDirNameEs)) {
        push(@args, '-auto_initialize_services');
    }
    my $rc = $self->runUtilityInEnv ($esaddhost, \@args, $msg->getSubMsgLst());
    if (!$rc){
        $self->setErrorMessage ("$addhostScript failed", $msg->getSubMsgLst ());
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Calls a specific script to add an accelerator hosts

sub addAcceleratorHosts {

	my ($self,
	    $hostname,
	    $acceleratorHostRole,
	    $instconfig, # getting additional accelerator parameters
	   ) = @_;

	my $addhostScript = 'aseaddhost';
	my $addhostCall   = $self->getAcceleratorLcmDir() . $path_separator
						     . $addhostScript . '.sh';
	my $msg = $self->getMsgLst()->addProgressMessage ("Performing $addhostScript...");
	my $args = ['-host',    $hostname,
	            '-log_dir', $gLogDir,
	            '-role',    $acceleratorHostRole,
	           ];
	my $ref_input_buffer = undef;

	if (defined $instconfig) {
        #_TODO_ Decide if both SystemUser & TenantUser are necessary
        my $systemUser      = $instconfig->getValue('SystemUser');
        my $systemUserPw    = $instconfig->getValue('SQLSysPasswd');
        my $acceleratorUser = $instconfig->getValue('AcceleratorUser');
        my $acceleratorPw   = $instconfig->getValue('AcceleratorPassword');

        if (defined $systemUser   && defined $acceleratorUser &&
            defined $systemUserPw && defined $acceleratorPw) {

            push @$args, '-hdbuser';
            push @$args, $systemUser;
            push @$args, '-aseuser';
            push @$args, $acceleratorUser;
            $ref_input_buffer = ["$systemUserPw $acceleratorPw"];
        }
        if ($self->shouldAutoInitializeService($instconfig, $gDirNameAccelerator)) {
            my $tenantUser      = $instconfig->getValue('TenantUser');
            my $tenantPassword  = $instconfig->getValue('SQLTenantUserPassword');
            if (!defined $tenantUser || !defined $tenantPassword) {
                $self->getErrMsgLst->addMessage("Missing tenant user credentials to add '$gDirNameAccelerator' host");
                return undef;
            }
            push(@$args, '-auto_initialize_services');
            push(@$args, ('-tenant_user', $tenantUser));
#_TODO_ Input buffer password order
            push(@$ref_input_buffer, $tenantPassword);
        }
	}

	my $rc  = $self->runUtilityInEnv($addhostCall,
	                                 $args,
	                                 $msg->getSubMsgLst(),
	                                 undef,
	                                 $ref_input_buffer);
	if (!$rc){
		$self->setErrorMessage ("$addhostScript failed", $msg->getSubMsgLst());
		return undef;
	}
	return 1;
}

sub shouldAutoInitializeService {
    my ($self, $instconfig, $componentDirName) = @_;
    return 0 if (!$instconfig->isSystemInCompatibilityMode());
    my $autoInitializeServiceValue = $instconfig->getValue('AutoInitializeServices');
    return 0 if ($instconfig->isSkipped('AutoInitializeServices') || ! $autoInitializeServiceValue);
    my $componentManifestPath = $self->getInstalledComponentManifestPath($componentDirName);
    return undef if (!defined $componentManifestPath);
    my $manifest = SDB::Install::Manifest->new($componentManifestPath);
    return $manifest->isAutoInitializeServiceSupported();
}

sub getInstalledComponentManifestPath {
    my ($self, $componentDirName) = @_;
    my $componentDir = File::Spec->catfile($self->get_globalSidDir, $componentDirName);
    if (! -d $componentDir) {
        $self->getErrMsgLst->addMessage("Component directory $componentDirName not found");
        return undef;
    }
    my $manifestPath = File::Spec->catfile($componentDir, 'manifest');
    if (! -f $manifestPath) {
        $self->getErrMsgLst->addMessage("Manifest file '$manifestPath' could not be found");
        return undef;
    }
    return $manifestPath;
}

#-------------------------------------------------------------------------------
# Reads the file 'daemon.ini'.
# Returns a reference to SDB::Install::LayeredConfig::IniFile
# or undef in case of an error.

sub readDaemonIni {

    my ($self, $ignoreFileNotFound, $host) = @_;

    my $layered_cfg = undef;

    if (!defined $host || ($host eq $self->{_host})) {
        # local host
        $layered_cfg = $self->getLayeredConfig ();
        if (!defined $layered_cfg){
            return undef;
        }
    }
    else {
        # remote host
        my $uid = ($isWin) ? undef : $self->{_user}->uidUx();
        my $gid = ($isWin) ? undef : $self->{_user}->gidUx();
        $layered_cfg = new SDB::Install::LayeredConfig ($self->{_instanceDir},
                                                        $self->{_globalSidDir},
                                                        undef,
                                                        $host,
                                                        undef,
                                                        $uid,
                                                        $gid);
    }

    my $daemon_ini = $layered_cfg->getIniFile ('daemon.ini');

    if (!defined $daemon_ini){
        if (!$ignoreFileNotFound) {
            $self->AddError (undef, $layered_cfg);
        }
        return undef;
    }

    my $msg = $self->AddMessage ("Reading daemon.ini");
    $daemon_ini->setMsgLstContext([$msg->getSubMsgLst (), $self->getErrMsgLst ()]);
    if (!defined $daemon_ini->readValues()){
        return undef;
    }
    return $daemon_ini;
}


#-------------------------------------------------------------------------------
# Inserts entries for extended storage, streaming and/or remote data sync into
# the daemon.ini file.

sub writeSpecialHostServersIntoDaemonIni {
    my ($self, $switchOffHanaService, $roleMap ) = @_;

    my $daemon_ini = $self->readDaemonIni();
    if (!defined $daemon_ini) {
        return undef;
    }

    my $msg = $self->getMsgLst ()->addMessage ("Updating daemon.ini");
    $daemon_ini->setMsgLstContext ([$msg->getSubMsgLst(), $self->getErrMsgLst ()]);

    if ($switchOffHanaService){
        # host without hana worker or hana standby
        my @daemonPrograms;
        foreach my $section (@{$daemon_ini->getSections ()}){
            if ($daemon_ini->existsValue ($section, 'instances')){
                push @daemonPrograms, $section;
            }
        }
        my $instancesValue;
        foreach my $daemonProgram (@daemonPrograms){
            if (($daemonProgram eq 'nameserver') ||
                ($daemonProgram eq 'diserver')) {
                next;
            }
            $instancesValue = $daemon_ini->getValue ($daemonProgram, 'instances');
            if ($instancesValue > 0){
                $daemon_ini->setValue (CFG_LAYER_HOST, $daemonProgram, 'instances', 0);
            }
        }
    }

    my $isMultiDb = $self->isMultiDb (1); # no cache
    if ($roleMap->{$gHostRoleStreaming} && !$isMultiDb) {
        # switch streaming service on for single DB only
        $daemon_ini->setValue (CFG_LAYER_HOST, 'streamingserver', 'instances', 1);
    }
    if (($roleMap->{$gHostRoleEsWorker} || $roleMap->{$gHostRoleEsStandby}) && !$isMultiDb) {
        # switch es service 'on' in single DB only. In multi DB it is handled by hdbnsutil
        $daemon_ini->setValue (CFG_LAYER_HOST, 'esserver', 'instances', 1);
    }

    # accelerator is switched on by aseaddhost script!
    if (!defined $daemon_ini->write ()){
        return undef;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Inserts entries for extended storage, streaming and/or remote data sync into
# the daemon.ini file.

sub writeIntoDaemonIni {

    my ($self, $layer, $section, $key, $value, $comment, $host) = @_;

    my $daemon_ini = $self->readDaemonIni(0, $host);
    if (!defined $daemon_ini) {
        return undef;
    }

    my $msg = $self->getMsgLst ()->addMessage
                             ("Writing '$section/$key=$value' into daemon.ini");
    $daemon_ini->setMsgLstContext ([$msg->getSubMsgLst(), $self->getErrMsgLst()]);

    if (!$daemon_ini->setValue ($layer, $section, $key, $value, $comment)) {
        return undef;
    }

    if (!defined $daemon_ini->write ()){
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Returns 1 if this host provides a xs2 controller.

sub hasXS2Controller {

    my ($self, $host) = @_;

    my $daemon_ini = $self->readDaemonIni(1, $host);
    if (!defined $daemon_ini) {
        return 0;
    }

    my $section = 'xscontroller';
    my $key     = 'instances';

    if (!$daemon_ini->existsValue($section, $key)) {
        return 0;
    }

    my $value = $daemon_ini->getValue($section, $key);
    return (defined $value && ($value > 0)) ? 1 : 0;
}

sub getXSEADataPath {
    my ($self, $noCache) = @_;
    my $globalIni = $self->getGlobalIni(undef, $noCache);

    return undef if(!defined($globalIni));

    my $returnValue = $globalIni->getValue('persistence', 'basepath_xsa_appworkspace');
    return defined($returnValue) && length($returnValue) > 0 ? $returnValue : undef;
}

#-------------------------------------------------------------------------------


our $systemInfoSection = 'system_information';
our $systemUsageKey = 'usage';

sub getSystemUsage{
    my ($self) = @_;
    my $global_ini = $self->getGlobalIni ();
    if (!defined $global_ini){
        return undef;
    }
    return $global_ini->getValue ($systemInfoSection, $systemUsageKey);
}

sub isSystemUsageChanged {
    my ($self, $newValue) = @_;
    my $global_ini = $self->getGlobalIni ();
    if (!defined $newValue || !defined $global_ini){
        return undef;
    }
    return $global_ini->getValue ($systemInfoSection, $systemUsageKey) ne $newValue;
}

sub setSystemUsage{
    my ($self, $global_ini, $systemUsage) = @_;

    if (defined $systemUsage){
        my $layer;
        my $msg = $self->getMsgLst()->addMessage ('Setting system usage parameter');
        my $msglst = $msg->getSubMsgLst();
        $global_ini->setMsgLstContext ([$msglst]);
        my $value = $global_ini->getValue ($systemInfoSection, $systemUsageKey,\$layer);
        if (defined $value){
            if($value eq $systemUsage){
                $msglst->addMessage ("usage is already set to '$value'")
            }
            elsif (!defined $global_ini->setValue(CFG_LAYER_SYSTEM, $systemInfoSection, $systemUsageKey, $systemUsage)){
                $self->setErrorMessage ('Cannot set system usage parameter',
                $global_ini->getErrMsgLst());
                return undef;
            }
        }
    }
    return 1;
}

our $repositorySection = 'repository';
our $repositoryKey     = 'enable_repository';


sub setRepositoryParam{
    my ($self, $instconfig) = @_;
    my $isRepositoryOn = $instconfig->getValue ('Repository');

    if (!defined $isRepositoryOn){
        return 1;
    }

    my $msg = $self->getMsgLst ()->addMessage ($isRepositoryOn ? 'Enabling repository' : 'Disabling repository');

    my @ini_files;
    my $dbMode = $instconfig->getValue ('DbMode');

    my $layered_cfg = $self->getLayeredConfig ();
    if (!defined $layered_cfg){
        return undef;
    }

    if (defined $dbMode && $dbMode eq 'multidb'){
        my $nameserver_ini = $layered_cfg->getIniFile ('nameserver.ini');
        if (!defined $nameserver_ini){
            $self->setErrorMessage ('Cannot get nameserver.ini', $layered_cfg->getErrMsgLst ());
            return undef;
        }
        push @ini_files, $nameserver_ini;
    }


    my $indexserver_ini = $layered_cfg->getIniFile ('indexserver.ini');
    if (!defined $indexserver_ini){
        $self->setErrorMessage ('Cannot get indexserver.ini', $layered_cfg->getErrMsgLst ());
        return undef;
    }
    push @ini_files, $indexserver_ini;

    my $repositoryValue   =  $isRepositoryOn ? 'true' : 'false';

    my $submsglst = $msg->getSubMsgLst ();
    my $submsg;

    foreach my $ini (@ini_files){
        $submsg = $submsglst->addMessage ("Adjusting " . $ini->{name});
        $ini->setMsgLstContext([$submsg->getSubMsgLst (), $self->getErrMsgLst ()]);
        if (!defined $ini->readValues ()){
            return undef;
        }
        if (!defined $ini->setValue (CFG_LAYER_SYSTEM, $repositorySection, $repositoryKey, $repositoryValue)){
            return undef;
        }
        if (!defined $ini->write()){
            return undef;
        }
    }
    return 1;
}

sub isRepositoryEnabled{
    my ($self) = @_;
    my $iniFileName;
    my $layered_cfg = $self->getLayeredConfig ();
    if (!defined $layered_cfg){
        return undef;
    }
    if ($self->isMultiDb ()){
        $iniFileName = 'nameserver.ini';
    }
    else{
        $iniFileName = 'indexserver.ini';
    }
    $layered_cfg->setMsgLstContext([undef, $self->getErrMsgLst ()]);
    my $iniFile = $layered_cfg->getIniFile ($iniFileName);
    if (!defined $iniFile){
        return undef;
    }
    $iniFile->setMsgLstContext([undef, $self->getErrMsgLst ()]);
    if (!defined $iniFile->readValues ()){
        return undef;
    }
    if ($iniFile->existsValue ($repositorySection, $repositoryKey)){
        my $value = $iniFile->getValue ($repositorySection, $repositoryKey);
        if ($value eq 'false'){
            return 0;
        }
    }
    return 1;
}


our $multidbSection = 'multidb';
our $dbmodeKey = 'mode';
our $dbisolationKey = 'database_isolation';

sub getDbMode{
    my ($self, $nocache) = @_;

    if (!defined $self->{_dbmode} || $nocache){

        my $global_ini = $self->getGlobalIni(undef, $nocache);

        if (!defined $global_ini) {
            return undef;
        }
        $global_ini->setMsgLstContext ([$self->getMsgLst ()]);
        $self->{_dbmode} = [$global_ini->getValue ($multidbSection, $dbmodeKey),$global_ini->getValue ($multidbSection, $dbisolationKey)];
    }
    if (wantarray){
        return @{$self->{_dbmode}};
    }
    return $self->{_dbmode}->[0];
}

sub setDbMode{
    my ($self, $global_ini, $dbMode, $dbIsolation) = @_;

    if (defined $dbMode){
        my $layer;
        my $msg = $self->getMsgLst()->addMessage ('Setting multidb mode parameter');
        my $msglst = $msg->getSubMsgLst();
        $global_ini->setMsgLstContext ([$msglst]);
        my $value = $global_ini->getValue ($multidbSection, $dbmodeKey,\$layer);
        if (defined $value){
            if($value eq $dbMode){
                $msglst->addMessage ("mode is already set to '$value'")
            }
            elsif (!defined $global_ini->setValue(CFG_LAYER_SYSTEM, $multidbSection, $dbmodeKey, $dbMode)){
                $self->setErrorMessage ('Cannot set multidb mode parameter',
                $global_ini->getErrMsgLst());
                return undef;
            }
            if (defined $dbIsolation && !defined $global_ini->setValue(CFG_LAYER_SYSTEM, $multidbSection, $dbisolationKey, $dbIsolation)){
                $self->setErrorMessage ('Cannot set multidb database_isolation parameter',
                $global_ini->getErrMsgLst());
                return undef;
            }
            # force writing INI files and reread, otherwise getDbMode() calls will fail
            delete $self->{global_ini};
            $global_ini->write();
        }
        else{
            $msglst->addWarning ("Skipping multidb mode parameter, because the used database version doesn't know it.");
        }
    }
    return 1;
}

sub getDefaultPort{
    my ($self) = @_;
    if ($self->isMultiDb ()){
        return '13';
    }
    return '15';
}

sub getSQLDatabaseName{
    my ($self, $useInitialTenant) = @_;
    if (!defined $useInitialTenant) {
        $useInitialTenant = 0;
    }
    if ($self->isMultiDb ()){
        if ($useInitialTenant){
            return $self->{_sid};
        }
        else{
            return 'SYSTEMDB';
        }
    }
    # => single db indexserver
    return undef;
}

our $memoryManagerSection = 'memorymanager';
our $globalAllocationKey = 'global_allocation_limit';

sub getGlobalAllocationLimit{
    my ($self) = @_;
    my $global_ini = $self->getGlobalIni ();
    if (!defined $global_ini){
        return undef;
    }
    return $global_ini->getValue ($memoryManagerSection, $globalAllocationKey);
}

sub setGlobalAllocationLimit{
    my ($self, $global_ini, $allocationLimit) = @_;

    if (defined $allocationLimit){
        my $layer;
        my $msg = $self->getMsgLst()->addMessage ('Setting global allocation limit parameter');
        my $msglst = $msg->getSubMsgLst();
        $global_ini->setMsgLstContext ([$msglst]);
        my $value = $global_ini->getValue ($memoryManagerSection, $globalAllocationKey,\$layer);
        if (defined $value){
            if($value eq $allocationLimit){
                $msglst->addMessage ("global allocation limit is already set to '$value'")
            }
            elsif (!defined $global_ini->setValue(CFG_LAYER_SYSTEM, $memoryManagerSection, $globalAllocationKey, $allocationLimit)){
                $self->setErrorMessage ('Cannot set global allocation limit parameter',
                $global_ini->getErrMsgLst());
                return undef;
            }
        }
    }
    return 1;
}

sub getHWInfoDir{
    return $_[0]->get_globalTrexInstallDir () . $path_separator . 'hwinfo';
}

sub getHWInfoFileName{
    my ($self, $host) = @_;
    if (!defined $host){
        $host = $self->{_host};
    }
    if (!defined $host){
        return undef;
    }
    return $self->getHWInfoDir() . $path_separator . sprintf ('bios_%s.txt', $host);
}

sub createHWInfoFile{
    my ($self, $sysinfo, $host) = @_;
    my $filename = $self->getHWInfoFileName ($host);

    if (!defined $filename){
        return undef;
    }

    my $hwinfo_dir = $self->getHWInfoDir ();
    if (!-d $hwinfo_dir){
        my $cfg = {'mode' => 0750};
        if (!makedir ($hwinfo_dir,$cfg)){
	        $self->AddError ("Cannot create directory '$hwinfo_dir': $!");
	        return undef;
        }
        $self->{_user}->possess($hwinfo_dir, 0, 0750);
    }
    my $hwinfo_file = $self->getHWInfoFileName ($host);
    if (!defined $hwinfo_file){
        return undef;
    }
    if (!open (FH,">$hwinfo_file")){
        $self->AddError ("Cannot create file '$hwinfo_file': $!");
        return undef;
    }
    print FH $sysinfo->{hwinfo_out};
    close (FH);
    $self->{_user}->possess($hwinfo_file, 0, 0640);
    return 1;
}

sub removeHWInfoFile{
    my ($self, $host) = @_;
    my $filename = $self->getHWInfoFileName ($host);

    if (!defined $filename){
        return undef;
    }
    if (-f $filename){
        if (!unlink ($filename)){
            $self->AddError ("Cannot delete file '$filename': $!");
            return undef;
        }
    }
    return 1;
}

sub changeInstanceNr{
    my ($self, $oldNr, $newNr) = @_;
    my $instanceDir = $self->{_instanceDir};
    $self->addDebug("Current instance directory is '$instanceDir'");
    my $patternOld  = sprintf ('HDB%02d',$oldNr);
    my $newDir      = sprintf ('HDB%02d',$newNr);
    if ($instanceDir =~ s/$patternOld/$newDir/){
        $self->{_nr}          = sprintf ('%02d', $newNr);
        $self->{_instanceDir} = $instanceDir;
    }
    $self->addDebug("New instance directory is ".$self->{_instanceDir});
    $self->addDebug("New instance number is ".$self->{_nr});
    return 1;
}


#-------------------------------------------------------------------------------
# Renames the instance.
#
# Parameter: string                      $oldSid
#            string                      $newSid
#            int                         $newNr
#            hash                        $hostHashMap 1)
#            SDB::Install::Configuration $instconfig  2)
#            string array                $chownList
#            SDB::Install::SysInfo       $sysinfo
#
#    1)  key: origin host name, value: new host name
#    2)  e.g. SDB::Install::Configuration::Rename
#
# Returns int retCode

sub renameInstance {
	my ($self,
	    $oldSid, $newSid,
	    $oldNr,
	    $newNr,
	    $hostHashMap,
	    $instconfig,
	    $chownList,  # e.g.:  [$oldUid, $newUid, $oldGid, $newGid]
	    $sysinfo,
	    $userChanged,
	    $oldHostRoles) = @_;

	my $hosts       = $self->get_allhosts();
	my $sid_changed = $oldSid ne $newSid;
	my $changed     = $sid_changed;
	my $repl;

	if ($sid_changed){
		$self->{_user} = new SDB::Install::NewDBUser($newSid);
		$repl = {
			"SAPSYSTEMNAME=$oldSid" => "SAPSYSTEMNAME=$newSid",
			"setenv SAPSYSTEMNAME $oldSid" => "setenv SAPSYSTEMNAME $newSid",
			"setenv SAPSYSTEMNAME  $oldSid" => "setenv SAPSYSTEMNAME $newSid",
		};
	}

	#
	# instance nr
	#

	if ($oldNr != $newNr){
		$changed = 1;
	}


	if ($changed){
		#
		# reset layered config / inifile cache
		#

		$self->getLayeredConfig (1);
	}

	my $profiles =
	       new SDB::Install::SAPProfiles ($self->get_profileDir(), $newSid);

	#
	# host names
	#

	my @inis;
	my @wdisp_profiles;
	my $hostNameChanged = 0;
	my $oldLocalHost    = $self->{_host};
    $self->{_host}  = undef;
	$self->{_hosts} = [];
	my $acceleratorOldHost     = undef;
	my $extendedStorageOldHost = undef;
	my $xs2Hosts = [];
	my $databaseHosts = [];

	if (defined $oldHostRoles) {

		if (defined $oldHostRoles->{$gHostRoleXS2Worker}) {
			my $oldXsWorkerHosts = $oldHostRoles->{$gHostRoleXS2Worker};
			push(@{$xs2Hosts}, @{$oldXsWorkerHosts});
		}

		if (defined $oldHostRoles->{$gHostRoleXS2Standby}) {
			my $oldXsStandbyHosts = $oldHostRoles->{$gHostRoleXS2Standby};
			push(@{$xs2Hosts}, @{$oldXsStandbyHosts});
		}

		if (defined $oldHostRoles->{$gHostRoleWorker}) {
			my $oldDatabaseWorkerHosts = $oldHostRoles->{$gHostRoleWorker};
			push(@{$databaseHosts}, @{$oldDatabaseWorkerHosts});
		}

		if (defined $oldHostRoles->{$gHostRoleStandby}) {
			my $oldDatabaseStandbyHosts = $oldHostRoles->{$gHostRoleStandby};
			push(@{$databaseHosts}, @{$oldDatabaseStandbyHosts});
		}

		if (defined $oldHostRoles->{$gHostRoleAcceleratorWorker}) {
			$acceleratorOldHost = $oldHostRoles->{$gHostRoleAcceleratorWorker}->[0];
		}

		if (defined $oldHostRoles->{$gHostRoleEsWorker}) {
			$extendedStorageOldHost = $oldHostRoles->{$gHostRoleEsWorker}->[0];
		}
	}

	foreach my $oldHost (@$hosts) {

	    if (!defined $oldHost) {
			next;
		}

		@inis           = ();
		@wdisp_profiles = ();
		my $newHost     = $hostHashMap->{$oldHost};

        if (!defined $newHost) {
            $newHost = $oldHost;
        }

        my $oldHostDir = $self->get_hostNameDir($oldHost);
        my $newHostDir = $self->get_hostNameDir($newHost);

		if ($self->isHostLocal($newHost)) {
            $self->removeHWInfoFile ($oldHost);
            $oldLocalHost  = $oldHost;
            $self->{_host} = $newHost;
            $self->createHWInfoFile ($sysinfo, $newHost);
		}
		else {
		    push @{$self->{_hosts}}, $newHost;
		     $self->removeHWInfoFile ($oldHost);
            # save hdbrename/hdbreg file in the slave host directory
            # it's required by hdbrename/hdbreg on slave hosts

            my $pers_filename = join ($path_separator,
                                      $oldHostDir,
                                      $instconfig->getPersistentFile());

            my $isNoHostStart = $instconfig->noHostStart($self);
            my $initUserHomeDir = ($instconfig->getValue('InitUserHomeDir'))
                                  ? 1 : 0;
            my $isAutoStart = ($instconfig->getValue('AutoStart')) ? 1 : 0;

            my $isAccelerator = (defined $acceleratorOldHost &&
                                 ($acceleratorOldHost eq $oldHost)) ? 1 : 0;

            my $isExtendedStorage = (defined $extendedStorageOldHost &&
                                    ($extendedStorageOldHost eq $oldHost))
                                    ? 1 : 0;

            my $isXs2Host       = grep { $_ eq $oldHost } @{$xs2Hosts};
            my $isDatabaseHost  = grep { $_ eq $oldHost } @{$databaseHosts};
            my $oldXsEaDataPath = $self->getXSEADataPath();
            my $oldDatapath     = $self->getDataPath(1);
            my $oldLogpath      = $self->getLogPath(1);

            my $lssUid = $instconfig->getValue('LSSUserID');
            my $lssGid = $instconfig->getValue('LSSGroupID');
            my $lssShell = $instconfig->getValue('LSSUserShell');
            my $lssHome = $instconfig->getValue('LSSUserHomeDir');
            my $changeBackupPasswd = $instconfig->getValue('ChangeLSSBackupPassword');
            my $internalHostnameResolution = $instconfig->getValue('InternalHostnameResolution');
            my $isRegisterLSS = $instconfig->getValue('IsRegisterLSS');

            my $persFileData = {
                'oldHost'           => $oldHost,
                'oldSID'            => $oldSid,
                'oldInstanceNumber' => $oldNr,
                'isNoHostStart'     => $isNoHostStart,
                'initUserHomeDir'   => $initUserHomeDir,
                'AutoStart'         => $isAutoStart,
                'isRenameAccelerator'     => $isAccelerator,
                'isRenameExtendedStorage' => $isExtendedStorage,
                'isRenameXS2' => $isXs2Host || $isDatabaseHost,
                (
                    defined($oldXsEaDataPath) ? ( 'oldXsEaDataPath' => $oldXsEaDataPath ) : (),
                    defined($oldDatapath)     ? ( 'oldDataPath'     => $oldDatapath     ) : (),
                    defined($oldLogpath)      ? ( 'oldLogPath'      => $oldLogpath      ) : (),
                ),
                'oldUID' => $chownList->[0], # Old <sid>adm UID
                'oldGID' => $chownList->[2], # Old sapsys GID
                (
                    defined($lssUid)   ? ( 'LSSUserID' => $lssUid) : (),
                    defined($lssGid)   ? ( 'LSSGroupID'     => $lssGid) : (),
                    defined($lssShell) ? ( 'LSSUserShell'      => $lssShell) : (),
                    defined($lssHome)  ? ( 'LSSUserHomeDir'      => $lssHome) : (),
                    defined($changeBackupPasswd) ? ('ChangeLSSBackupPassword' => $changeBackupPasswd) : (),
                    defined($internalHostnameResolution) ? ('InternalHostnameResolution' => $internalHostnameResolution) : (),
                    defined($isRegisterLSS) ? ('IsRegisterLSS' => $isRegisterLSS) : (),
                ),
            };

            if ($instconfig->isScopeInstance()){
                my $revertToSnapshot = $persFileData->{'RevertToSnapshot'} = $instconfig->getValue('RevertToSnapshot');
                if ($revertToSnapshot){
                    my $tenantMap = $instconfig->getValue('TenantMap');
                    if (defined $tenantMap){
                        $persFileData->{'TenantMap'} = $tenantMap;
                    }
                }
            }

            if ($isAccelerator) {

                if (defined $chownList) {
                    $persFileData->{oldUidAccelerator} = $chownList->[0];
                    $persFileData->{oldGidAccelerator} = $chownList->[2];
                }

                if (exists $instconfig->{params}->{AcceleratorDataPath}) {

                    $persFileData->{AcceleratorDataPath} =
                        $instconfig->getValue('AcceleratorDataPath');

                    $persFileData->{oldAcceleratorDataPath} =
                        $instconfig->{params}->{AcceleratorDataPath}->{_originPath};
                }

                if (exists $instconfig->{params}->{AcceleratorLogPath}) {

                    $persFileData->{AcceleratorLogPath} =
                        $instconfig->getValue('AcceleratorLogPath');

                    $persFileData->{oldAcceleratorLogPath} =
                        $instconfig->{params}->{AcceleratorLogPath}->{_originPath};
                }
            }

            if ($isExtendedStorage) {

                if (defined $chownList) {
                    $persFileData->{oldUidEs} = $chownList->[0];
                    $persFileData->{oldGidEs} = $chownList->[2];
                }

                if (exists $instconfig->{params}->{EsDataPath}) {
                    $persFileData->{EsDataPath} = $instconfig->getValue('EsDataPath') || $instconfig->getDefault('EsDataPath');
                    $persFileData->{oldEsDataPath} = $instconfig->{params}->{EsDataPath}->{_originPath};
                }

                if (exists $instconfig->{params}->{EsLogPath}) {
                    $persFileData->{EsLogPath} = $instconfig->getValue('EsLogPath') || $instconfig->getDefault('EsLogPath');
                    $persFileData->{oldEsLogPath} = $instconfig->{params}->{EsLogPath}->{_originPath};
                }
            }

            if (!$instconfig->pers_store ($pers_filename, $persFileData)) {
                $self->AddError ("Cannot save $pers_filename", $instconfig);
                return undef;
            }
            if (!$isWin){
                $self->setPermissions ($pers_filename,0,0640);
            }
		}

        my $renameHostDirStatus = $self->renameHostDir($oldHost, $newHost, $oldHostDir, $newHostDir);
        if($renameHostDirStatus){
            $changed = 1;
            $hostNameChanged = 1;
        } else {
            return undef if(!defined($renameHostDirStatus));
        }

        my $oldProfileName = $self->get_profileName($oldSid, $oldNr, $oldHost);
        my $newProfileName = $self->get_profileName($newSid, $newNr, $newHost);

        my $newProfile     =
                $self->get_profileDir() . $path_separator . $newProfileName;

        return undef if(!$self->renameProfileDir($oldProfileName , $newProfileName));

		my $msg = $self->AddMessage
		      ("Updating instance configuration files for host ($newHost)...");
		my $msglst = $msg->getSubMsgLst ();

		@inis = (new SDB::Install::IniFile ($newProfile),
			     new SDB::Install::IniFile
			                ($newHostDir . $path_separator . 'sapprofile.ini'));

		foreach my $ini (@inis){

			if ($ini->ErrorState ()){
				$self->AddError (undef,$ini);
				return undef;
			}

			$ini->setValue(undef, 'SAPSYSTEMNAME', $newSid);
			$ini->setValue(undef, 'SAPSYSTEM',     sprintf('%02d', $newNr));

			foreach my $hostkey ('SAPLOCALHOST', 'SAPGLOBALHOST'){

				my $val = $ini->getValue (undef, $hostkey);

				if (!$val){
					next;
				}

				my $newVal = $hostHashMap->{$val};
                if (defined $newVal) {
                    $ini->setValue (undef, $hostkey, $newVal);
                }
			}
		}

		$inis[0]->setValue(undef,'INSTANCE_NAME', sprintf ('HDB%02d',$newNr));
		$inis[0]->setValue(undef,'_PF',      '$(DIR_PROFILE)/'.$newProfileName);
        if($sid_changed){
           $inis[1]->setValue(undef,'DIR_PROFILE', $self->getLocalProfileDir());
        }
		my $error = 0;
		my $writeMsg;
		foreach my $ini (@inis, @wdisp_profiles){
			$writeMsg = $msglst->addMessage ("Updating '$ini->{fullname}'");
			$ini->setMsgLstContext ([$writeMsg->getSubMsgLst]);
			if (!defined $ini->write ()){
				$self->PushError (undef, $ini);
				$error = 1;
			}
		}
		if ($error){
			return undef;
		}
	}

	if ($hostNameChanged){
		# reset master cache
		delete $self->{_masters};
	}

	if (!defined $self->adaptVolumePaths ($oldSid, $newSid,
	                                      $oldNr,  $newNr,
	                                      $instconfig,
	                                      $chownList)) {
		return undef;
	}

	my $newLandscapeIdRequired = 0;
	if (!$changed){
		# nothing changed => check whether there is a landscapeId
		my $landscapeId = $self->getLandscapeID ();
		if (!$landscapeId){
			$self->getMsgLst ()->addMessage ('Force convertTopology due to missing landscapeId');
			$newLandscapeIdRequired = 1;
		}
	}

	if ($changed || $userChanged){
		#
		# adjust env scripts
		#
		$self->set_localUsrSapSidDir (join ($path_separator, '','usr','sap', $newSid));
		my $msg = $self->AddProgressMessage ("Configuring environment scripts...");
		$self->installHDBScripts ($msg->getSubMsgLst());
		if (!$isWin){
			$self->{_user}->setMsgLstContext([$msg->getSubMsgLst ()]);
			$self->{_user}->configureHome ($self->get_globalTrexInstallSupportDir (),
			                       $self->{_instanceDir}, $self->{_host},$repl);
		}
	}

    if (defined $self->getLssInstance()) {
        return undef if(!$self->renameLssInstance($instconfig));
    }
	my $isRevertToSnapshot = $instconfig->getValue ('RevertToSnapshot');
    if (!$isRevertToSnapshot && ($changed || $newLandscapeIdRequired)){
        my $doTargetSystemReplicationCleanUp = $instconfig->getValue ('TrgSysReplicationCleanUp');
        my $allHostsMap = { %$hostHashMap };
        my @allHosts = @{$self->get_allhosts()};
        my $tenantMap = $instconfig->getValue('TenantMap');

        foreach my $host(@allHosts){
            if (!(grep { $host eq $_} values %$allHostsMap)){
                $allHostsMap->{$host} = $host;
            }
        }
		if (! defined $self->rebuildTopology($allHostsMap,
                                             $tenantMap,
		                                     $sid_changed,
		                                     $doTargetSystemReplicationCleanUp)) {
			$self->AddError ("Cannot rebuild topology",$self);
			return undef;
		}
	}

    my $activeRoles = $self->getActiveHostRoles();
    if (!defined $activeRoles) {
        return undef;
    }

    return undef if(!$self->renameStreamingInstance($oldSid, $newSid, $oldNr, $newNr, $hostHashMap,$activeRoles, $chownList));

    if (defined $acceleratorOldHost && ($acceleratorOldHost eq $oldLocalHost)) {
        if (!$self->renameAcceleratorInstance($instconfig,
                                              $oldSid,       $newSid,
                                              $oldNr,        $newNr,
                                              $oldLocalHost, $self->{_host},
                                              $chownList)) {
            return undef;
        }
    }


    my $esrename = $self->getExtendedStorageRenameScript();
    if (-f $esrename) {
        if (!$self->renameExtendedStorageInstance($instconfig,
                                                  $oldSid,       $newSid,
                                                  $oldNr,        $newNr,
                                                  $oldLocalHost, $self->{_host},
                                                  $chownList)) {
            return undef;
        }
    }
    elsif (defined $extendedStorageOldHost) {
        $self->setErrorMessage("Rename tool '$esrename' not found");
        return undef;
    }

    if (!$self->changeInternalNetAndWriteGlobalIni($instconfig)) {
        return undef;
    }

    my $isXS2Host            = grep { $_ eq $oldLocalHost } @{$xs2Hosts};
    my $isDatabaseHost       = grep { $_ eq $oldLocalHost } @{$databaseHosts};
    if ($isXS2Host || $isDatabaseHost){
        return undef if(!$self->recreateXSSpaceIsolationUsers($instconfig, $oldSid, $newSid));
        return undef if(!$self->recreateXsEaDataPath($instconfig, $oldLocalHost));
    }
    return 1;
}

sub renameLssInstance {
    my ($self, $instconfig) = @_;
    my $lssInstance = $self->getLssInstance();
    if (!defined $lssInstance) {
        $self->getMsgLst()->addMessage("Failed to detect existing LSS instance to rename");
        return 1;
    }
    $lssInstance->setMsgLstContext($self->getMsgLstContext());
    if (!$instconfig->isSidChanged() && !$instconfig->getValue('IsRegisterLSS')) {
        $self->getMsgLst()->addMessage("Skipping rename of LSS instance because SID wasn't changed and is not a register scenario");
        if (!$instconfig->{isSlave}) {
            return $lssInstance->adaptSystemUsage($instconfig, 1);
        } else {
            return 1;
        }
    }
    return undef if(!$lssInstance->renameInstance($instconfig));

# If sid chanhed -> adapt lss symlink under isntall dir
    $lssInstance = $self->getLssInstance(1, 1);
    $lssInstance->setMsgLstContext($self->getMsgLstContext());
    if ($instconfig->isSidChanged()) {
        my $target = $instconfig->getValue('Target');
        if (!$lssInstance->createLssSymLink($target)) {
            return undef;
        }
    }

    if (!$instconfig->{isSlave}) {
        return undef if (!$lssInstance->adaptSettings($instconfig));
    }
    return 1;
}

#-------------------------------------------------------------------------------
sub renameStreamingInstance{
    my($self, $oldSid, $newSid, $oldNr, $newNr, $hostHashMap, $activeRoles, $chownList) = @_;
    my $sid_changed = $oldSid ne $newSid;

    my $strrename = $self->getStreamingRenameScript ();
    if (-f $strrename){
        my $strSidRenameNotifyArgs = $self->getArgsStrSidRenameNotify(
                $oldSid,$newSid,
                $oldNr, $newNr,
                $hostHashMap);
        if (@$strSidRenameNotifyArgs){
            if ($sid_changed){
                my $streamingEnvScript = $self->getStreamingDir() .
                    $path_separator . 'STREAMING.sh';
                if (open (FD, $streamingEnvScript)){
                    my @STREAMING_SHARED = grep {/STREAMING_SHARED=/} <FD>;
                    close (FD);
                    my $volumePath;
                    foreach my $match (@STREAMING_SHARED){
                        ($volumePath) = ($match =~ /STREAMING_SHARED=(.*)$/);
                        if (-d $volumePath){
                            last;
                        }
                    }
                    if ($volumePath && -d _){
                        my $msg = $self->getMsgLst ()->addMessage ("Changing owner of '$volumePath'");
                        my $errlst = new SDB::Install::MsgLst ();
                        if (!defined changeOwn (@$chownList, $volumePath, $errlst, $msg->getSubMsgLst())){
                            $self->setErrorMessage ("Cannot change owner of streaming volume path", $errlst);
                            return undef;
                        }
                    }
                }
            }
            push @$strSidRenameNotifyArgs, ('-phase', 'offline');
            my $msg = $self->getMsgLst ()->addProgressMessage("Performing '$streamingRenameTool'...");
            my $rc = $self->runUtilityInEnv (
                $strrename,
                $strSidRenameNotifyArgs,
                $msg->getSubMsgLst ());
            if (!$rc){
                $self->setErrorMessage ("$streamingRenameTool failed", $msg->getSubMsgLst ());
                $msg->endMessage(undef, "Performing '$streamingRenameTool' failed");
                return undef;
            }
            $msg->endMessage();
        }else{
            $self->getMsgLst ()->addMessage ("Skipping '$streamingRenameTool'");
        }
    }
    elsif (defined $activeRoles && defined $activeRoles->{$gHostRoleStreaming}){
        $self->setErrorMessage ("Rename tool '$strrename' not found");
        return undef;
    }

    return 1;
}

sub renameHostDir{
    my($self, $oldHost, $newHost, $oldHostDir, $newHostDir) = @_;

    if (($oldHost ne $newHost) && !-d $newHostDir) {

        $self->AddMessage
                 ("Renaming host directory '$oldHostDir' => '$newHostDir'");

        if (!CORE::rename($oldHostDir, $newHostDir)) {
            $self->AddError
                       ("Cannot rename '$oldHostDir' => '$newHostDir': $!");
            return undef;
        }

        return 1;
    }

    return 0;
}

sub renameProfileDir{
    my($self, $oldProfileName , $newProfileName ) = @_;

    my $newProfile     =
                $self->get_profileDir() . $path_separator . $newProfileName;

    $self->addDebug("Old profile name is '$oldProfileName'");
    $self->addDebug("New profile name is '$newProfileName'");

    if (($oldProfileName ne $newProfileName) && !-e $newProfile) {

        my $oldProfile = $self->get_profileDir() . $path_separator .
                                                            $oldProfileName;
        $self->AddMessage ("Renaming instance profile '$oldProfile' => '$newProfile'");
        if (!CORE::rename ($oldProfile, $newProfile)){
            $self->AddError ("Cannot rename '$oldProfile' => '$newProfile': $!");
            return undef;
        }

        $self->addDebug("After profile rename:");
        $self->addDebug("Check if $oldProfile exists - ".(-e $oldProfile));
        $self->addDebug("Check if $newProfile exists - ".(-e $newProfile));
    }

    return 1;
}

sub renameXS2 {
    my ($self, $instconfig, $userConfig, $newSid) = @_;
    my $renameScriptXS2 = 'xsa-rename';
    my $xsInstallationPath = $self->getXS2Dir();
    my $osUserSAP = createXSSpaceSAPUserName($newSid);
    my $osUserProd = createXSSpaceProdUserName($newSid);
    my $scriptPath = join($path_separator, $xsInstallationPath, 'installation-scripts', 'installation', $renameScriptXS2);
    my $inputBuffer = undef;
    my $msg = $self->getMsgLst()->addProgressMessage("Performing rename of XS...");
    my $arguments = [
        '-new_sid', $newSid,
        '-new_sid_adm', createSysAdminUserName($newSid),
    ];

    if($instconfig->hasValue('XsEaDataPath') && !$instconfig->isSkipped('XsEaDataPath')){
        push(@{$arguments}, '-ea_data_path');
        push(@{$arguments}, $instconfig->getValue('XsEaDataPath'));
    }
    if(exists($userConfig->{xs_space_user_sap})){
        push(@{$arguments}, '-execution_os_user_sap');
        push(@{$arguments}, $osUserSAP);
    }
    if(exists($userConfig->{xs_space_user_prod})){
        push(@{$arguments}, '-execution_os_user_prod');
        push(@{$arguments}, $osUserProd);
    }
    if (! -f $scriptPath){
        $self->setErrorMessage ("Rename tool '$renameScriptXS2' not found");
        $msg->endMessage(undef, "Performing '$renameScriptXS2' failed");
        return undef;
    }
    if(($instconfig->getValue('ChangeSystemPasswd') =~ /$bool_true_pattern/)){
        # If we're changing DB password we need to pass it to the XSA rename script too
        my $dbUsername = $instconfig->getValue('SystemUser') || 'SYSTEM';
        $inputBuffer = [ $instconfig->getValue('TrgSQLSysPasswd') ];
        push(@{$arguments}, '-system_db_user', $dbUsername);
    }
    if (!$self->runUtilityInEnv($scriptPath, $arguments, $msg->getSubMsgLst(), undef, $inputBuffer, undef, undef, 1)){
        $self->setErrorMessage("$renameScriptXS2 failed", $msg->getSubMsgLst());
        $msg->endMessage(undef, "Performing '$renameScriptXS2' failed");
        return undef;
    }

    my $usersGID = getgrnam('users');
    my $configKeyMap = {
        xs_space_user_sap => [ 'XSSpaceUserIdSAP', $osUserSAP ],
        xs_space_user_prod => [ 'XSSpaceUserIdProd', $osUserProd ],
    };

    for my $key (keys(%{$configKeyMap})) {
        next if(!exists($userConfig->{$key}));

        my ($parameterId, $targetName) = @{$configKeyMap->{$key}};
        my $errorList = new SDB::Install::MsgLst();
        my $targetUser = new SDB::Install::User($targetName);
        my $targetGID = $targetUser->exists() ? $targetUser->gid() : $usersGID;
        my ($sourceUID, $targetUID) = ($userConfig->{"${key}_id"}, $instconfig->getValue($parameterId));
        my $subMessage = $self->AddSubMessage($msg, sprintf('Changing owner of files under \'%s\' (uid: %s -> %s, gid: %s)', $xsInstallationPath, $sourceUID, $targetUID, $targetGID));

        if (!defined changeOwn($sourceUID, $targetUID, $targetGID, $targetGID, $xsInstallationPath, $errorList, $subMessage->getSubMsgLst(), undef, undef, undef, 1)) {
            $self->AddError("Cannot chown '$xsInstallationPath'", $errorList);
            $msg->endMessage(undef, "Performing '$renameScriptXS2' failed");
            return undef;
        }
    }

    if($instconfig->hasValue('XsDomainName') && !$instconfig->isSkipped('XsDomainName')){
        my $xsDomainName = $instconfig->getValue('XsDomainName');
        $self->AddSubMessage($msg, sprintf('Setting XS Domain Name to \'%s\'', $xsDomainName));
        if (!defined $self->setXsDomainName($xsDomainName)) {
            $self->AddError("Cannot set XS Domain Name");
            $msg->endMessage(undef, "Performing rename of XS failed");
            return undef;
        }
    }

    $msg->endMessage();
    return 1;
}

#-------------------------------------------------------------------------------
sub renameAcceleratorInstance {

    my ($self,
        $instconfig,
        $oldSid,             $newSid,
        $oldNr,              $newNr,
        $oldAcceleratorHost, $newAcceleratorHost,
        $chownList, # e.g.:  [$oldUid, $newUid, $oldGid, $newGid]
       ) = @_;

    my $renameScriptAccelerator = 'aserename';
    my $renameCallAccelerator   = $self->getAcceleratorLcmDir() . $path_separator
                       . $renameScriptAccelerator . '.sh';

    if (-f $renameCallAccelerator){
        my $msg = $self->getMsgLst()->addProgressMessage("Performing '$renameScriptAccelerator'...");

        my $oldAcceleratorDataPath = undef;
        my $oldAcceleratorLogPath  = undef;
        my $newAcceleratorDataPath = undef;
        my $newAcceleratorLogPath  = undef;

        if (exists $instconfig->{params}->{AcceleratorDataPath}) {
            $oldAcceleratorDataPath = $instconfig->{params}->{AcceleratorDataPath}->{_originPath};
            $newAcceleratorDataPath = $instconfig->getValue('AcceleratorDataPath');
        }

        if (exists $instconfig->{params}->{AcceleratorLogPath}) {
            $oldAcceleratorLogPath = $instconfig->{params}->{AcceleratorLogPath}->{_originPath};
            $newAcceleratorLogPath = $instconfig->getValue('AcceleratorLogPath');
        }

        $oldAcceleratorDataPath = $self->getAcceleratorDataPath(1) if (!defined $oldAcceleratorDataPath);
        $oldAcceleratorLogPath  = $self->getAcceleratorLogPath(1)  if (!defined $oldAcceleratorLogPath);
        $newAcceleratorDataPath = $oldAcceleratorDataPath if (!defined $newAcceleratorDataPath);
        $newAcceleratorLogPath  = $oldAcceleratorLogPath  if (!defined $newAcceleratorLogPath);

        my $rc = $self->runUtilityInEnv ($renameCallAccelerator,
                    ['-old_sid',      $oldSid,
                     '-new_sid',      $newSid,
                     '-old_inst',     $oldNr,
                     '-new_inst',     $newNr,
                     '-old_ase_data', $oldAcceleratorDataPath,
                     '-new_ase_data', $newAcceleratorDataPath,
                     '-old_ase_log',  $oldAcceleratorLogPath,
                     '-new_ase_log',  $newAcceleratorLogPath,
                     '-old_host',     $oldAcceleratorHost,
                     '-new_host',     $newAcceleratorHost,
                     '-log_dir',      $gLogDir,
                    ],
                    $msg->getSubMsgLst());
        if (!$rc){
            $self->setErrorMessage("$renameScriptAccelerator failed", $msg->getSubMsgLst());
            $msg->endMessage(undef, "Performing '$renameScriptAccelerator' failed");
            return undef;
        }
        $msg->endMessage();
    }
    else {
        $self->setErrorMessage ("Rename tool '$renameCallAccelerator' not found");
        return undef;
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub renameExtendedStorageInstance {

    my ($self,
        $instconfig,
        $oldSid,    $newSid,
        $oldNr,     $newNr,
        $oldEsHost, $newEsHost,
        $chownList, # e.g.:  [$oldUid, $newUid, $oldGid, $newGid]
       ) = @_;

    my $esrename = $self->getExtendedStorageRenameScript();

    if (-f $esrename){
        my $msg = $self->getMsgLst ()->addProgressMessage("Performing '$extendedStorageRenameTool'...");

        my $oldEsDataPath = undef;
        my $oldEsLogPath  = undef;
        my $newEsDataPath = undef;
        my $newEsLogPath  = undef;

        if (exists $instconfig->{params}->{EsDataPath}) {
            $oldEsDataPath = $instconfig->{params}->{EsDataPath}->{_originPath};
            $newEsDataPath = $instconfig->getValue('EsDataPath') || $instconfig->getDefault('EsDataPath');
        }

        if (exists $instconfig->{params}->{EsLogPath}) {
            $oldEsLogPath = $instconfig->{params}->{EsLogPath}->{_originPath};
            $newEsLogPath = $instconfig->getValue('EsLogPath') || $instconfig->getDefault('EsLogPath');
        }

        $oldEsDataPath = $self->getEsDataPath(1) if (!defined $oldEsDataPath);
        $oldEsLogPath  = $self->getEsLogPath(1)  if (!defined $oldEsLogPath);
        $newEsDataPath = $oldEsDataPath if (!defined $newEsDataPath);
        $newEsLogPath  = $oldEsLogPath  if (!defined $newEsLogPath);

        my $rc = $self->runUtilityInEnv ($esrename, [
            '-old_sid',     $oldSid,
            '-new_sid',     $newSid,
            '-old_inst',    $oldNr,
            '-new_inst',    $newNr,
            '-old_es_data', $oldEsDataPath,
            '-new_es_data', $newEsDataPath,
            '-old_es_log',  $oldEsLogPath,
            '-new_es_log',  $newEsLogPath,
            '-old_host',    $oldEsHost,
            '-new_host',    $newEsHost,
            '-log_dir',     $gLogDir],
            $msg->getSubMsgLst ());
        if (!$rc){
            $self->setErrorMessage ("$extendedStorageRenameTool failed", $msg->getSubMsgLst ());
            $msg->endMessage(undef, "Performing '$extendedStorageRenameTool' failed");
            return undef;
        }
        $msg->endMessage();
    }
    else {
        $self->setErrorMessage ("Rename script '$esrename' not found");
        return undef;
    }
    return 1;
}


sub getArgsStrSidRenameNotify{
    my ($self, $oldSid, $newSid, $oldNr, $newNr, $hostHashMap) = @_;
    my @args;
    if ($oldSid ne $newSid){
        push @args, '-sid', "$oldSid=$newSid";
    }
    if ($oldNr != $newNr){
        push @args, '-number', sprintf ('%02d=%02d', $oldNr, $newNr);
    }
    my $tgtHost;
    my @hostArgs;
    foreach my $srcHost (keys %$hostHashMap){
        $tgtHost = $hostHashMap->{$srcHost};
        if ($tgtHost eq $srcHost){
            next;
        }
        push @hostArgs, "$srcHost=$tgtHost";
    }
    if (@hostArgs){
        push @args, '-host', join (',',@hostArgs);
    }

    if (@args){
         push @args, '-hanapath', $self->get_globalSidDir (), '-log_dir' , $gLogDir;
    }
    return \@args;
}



#-------------------------------------------------------------------------------
# Changes 'SystemUsage' and internal network if necessary.
# Writes the file global.ini
#
# Returns int retCode

sub changeInternalNetAndWriteGlobalIni {
    my ($self,
        $instconfig # SDB::Install::Configuration
       ) = @_;

    my $listenInterface = $instconfig->getValue('ListenInterface');
    my $internalNetwork = $instconfig->getInternalNetworkIP();
    my $currentInternalNetwork = $self->getInternalNetworkPrefix();
    if (!defined $internalNetwork && !defined $listenInterface) {
        $internalNetwork = $currentInternalNetwork;
    }

    my $isChangingInternalNetwork = (defined($currentInternalNetwork)
                                    && defined($internalNetwork)
                                    && ($internalNetwork eq $currentInternalNetwork))
                                    ? 0 : 1;
# This handles the batch execution of rename scenario when
# the internal_network and listen_interface parameters
# can be not set at all...
    $isChangingInternalNetwork = 0 if (!defined($currentInternalNetwork) && !defined($internalNetwork));

    my $global_ini = $self->getGlobalIni ();
    if (!defined $global_ini){
        return undef;
    }
    $global_ini->setMsgLstContext ([$self->getMsgLst ()]);

    my $isSetToLocal = (defined $listenInterface && $listenInterface eq 'local') ? 1 : 0;
    my $isAnyHostChanged = $instconfig->isAnyHostChanged();
    my $isInternalHostnameResolutionReset = 0;
    if ($isChangingInternalNetwork || $isSetToLocal || (defined($currentInternalNetwork) && $isAnyHostChanged)){
        $self->clearInternalNetworkPrefix($global_ini);
        $self->resetInternalHostNameResolution();
        $self->setInternalNetworkPrefix($internalNetwork, $global_ini) if (defined($internalNetwork));
        $isInternalHostnameResolutionReset = 1;
    }
    my $systemUsage = $instconfig->getValue ('SystemUsage');
    if (defined $systemUsage){
        if (!defined $self->setSystemUsage($global_ini, $systemUsage)) {
            return undef;
        }
    }
    delete $self->{global_ini};
    if (!$global_ini->write()){
        $self->setErrorMessage (undef, $global_ini->getErrMsgLst());
        return undef;
    }

    if($instconfig->hasValue('InternalHostnameResolution')) {
        $self->resetInternalHostNameResolution() if(! $isInternalHostnameResolutionReset);
        return $self->writeInternalNetworkConfig($instconfig);
    }

    my $doReconfigureNetwork = 0;
    my $isNetworkConfigUpdated = $self->updateInternalNetworkConfig($instconfig);

    if ($isChangingInternalNetwork || (defined($currentInternalNetwork) && $isAnyHostChanged) || $isNetworkConfigUpdated) {
        $doReconfigureNetwork = 1;
    }
    else {
        my $wantedInterface = $instconfig->getValue('ListenInterface');
        if (defined $wantedInterface
               && ($wantedInterface ne $instconfig->getListenInterfaceType())) {
            $doReconfigureNetwork = 1;
        }
    }

    if ($doReconfigureNetwork) {
        if (!defined $self->reconfigureTrexNet ($instconfig)){
            return undef;
        }
    }

	return 1;
}

sub updateInternalNetworkConfig {
    my ($self, $instconfig) = @_;

    my $global_ini = $self->getGlobalIni();
    $global_ini->setMsgLstContext($self->getMsgLstContext());

    my $isFileUpdated = 0;
    my $localHostname = hostname();
    my $internalIp = $instconfig->getInternalDeviceIP ();
    foreach my $address (@{$global_ini->getSectionKeys('internal_hostname_resolution')}){
        next if($global_ini->getValue('internal_hostname_resolution', $address) ne $localHostname);

        if($internalIp ne $address){
            $isFileUpdated = 1;
            $global_ini->removeKey (CFG_LAYER_SYSTEM, 'internal_hostname_resolution', $address);
        }
    }
    if ($isFileUpdated && !$global_ini->write()){
        $self->setErrorMessage (undef, $global_ini->getErrMsgLst());
        return undef;
    }
    return $isFileUpdated;

}

sub writeInternalNetworkConfig {
    my ($self, $instconfig) = @_;

    my $global_ini = $self->getGlobalIni();
    $global_ini->setMsgLstContext($self->getMsgLstContext());

    my $interface = $instconfig->getValue('ListenInterface');
    if($interface){
        $global_ini->getMsgLst()->addMessage ("Setting listen interface");
        $global_ini->setValue (CFG_LAYER_SYSTEM,'communication','listeninterface',".$interface");
    }

    $global_ini->getMsgLst()->addMessage ("Setting internal_hostname_resolution");

    my $hostnameResolution = $instconfig->getValue('InternalHostnameResolution');
    foreach my $hostEntry (split (',', $hostnameResolution)){
        my @hostInfo = split ('=', $hostEntry);
        $global_ini->setValue(CFG_LAYER_SYSTEM,'internal_hostname_resolution',$hostInfo[0], $hostInfo[1]);
    }

    if (!$global_ini->write()){
        $self->setErrorMessage (undef, $global_ini->getErrMsgLst());
        return undef;
    }
    return 1;
}

sub getLayeredConfig{
	my ($self,$no_cache) = @_;
	if (defined $self->{layered_cfg} && !$no_cache){
		return $self->{layered_cfg};
	}
	$self->{global_ini} = undef;

	my ($uid,$gid);
	if (!$isWin){
		$uid = $self->{_user}->uidUx();
		$gid = $self->{_user}->gidUx();
	}

	$self->{layered_cfg} = new SDB::Install::LayeredConfig (
		$self->{_instanceDir},
		$self->{_globalSidDir},
		undef,
		$self->{_host},
		$self->get_globalTrexExeDirAbs () . $path_separator . 'config',
		$uid,
		$gid
		);

	if ($self->{layered_cfg}->ErrorState ()){
		$self->AddError ('Error initializing layered configuration', $self->{layered_cfg});
		return undef;
	}
	return $self->{layered_cfg};
}

sub getScriptServerIni {
    my ($self, $hdbinstance) = @_;

    my $layered_cfg = $self->getLayeredConfig ();

    if (!defined $layered_cfg){
        return undef;
    }

    my $msg = $self->AddMessage ("Getting scriptserver.ini");
    $layered_cfg->setMsgLstContext ([$msg->getSubMsgLst ()]);
    my $scriptServerIni = $layered_cfg->getIniFile ('scriptserver.ini');
    if (!defined $scriptServerIni){
        $self->AddWarning ("scriptserver.ini does not exist");
        return undef;
    }

    my $readMsg = $msg->getSubMsgLst ()->addMessage ('Reading scriptserver.ini');
    if (!defined $scriptServerIni->readValues ()){
        $self->AddError ('Cannot read scriptserver.ini', $scriptServerIni);
        return undef;
    }

    return $scriptServerIni;
}

sub getGlobalIni{
	my ($self, $msglst, $nocache) = @_;

	if (!defined $msglst){
		$msglst = $self;
	}

	my $layered_cfg = $self->getLayeredConfig ();

	if (defined $self->{global_ini} && !$nocache){
		return $self->{global_ini};
	}

	if (!defined $layered_cfg){
		return undef;
	}
	my $msg = $msglst->AddMessage ("Getting global.ini");
	$layered_cfg->setMsgLstContext ([$msg->getSubMsgLst ()]);
	my $global_ini = $layered_cfg->getIniFile ('global.ini');
	if (!defined $global_ini){
		$msglst->AddError ('Cannot get global.ini', $layered_cfg);
		return undef;
	}
	my $readMsg = $msg->getSubMsgLst ()->addMessage ('Reading global.ini');
	$global_ini->setMsgLstContext([$readMsg->getSubMsgLst ()]);
	if (!defined $global_ini->readValues ()){
		$self->AddError ("Cannot read global.ini", $global_ini);
		return undef;
	}

    if (!$isWin && defined $self->{_user}){
        $global_ini->setOwner ($self->{_user}->uid(), $self->{_user}->gid());
    }

	$self->{global_ini} = $global_ini;
}


sub getBasePath{
    my ($self,$basepath, $expand) = @_;
    my $globalIni = $self->getGlobalIni ();
    if (!defined $globalIni){
        return undef;
    }
    my $result = $globalIni->getValue ($persistence_section,$basepath);
    if ($expand && defined $result && ($result =~ /\$\(DIR_GLOBAL\)/)){
        return $self->resolveGlobalDir ($result);
    }
    elsif ($expand && defined $result && ($result =~ /\$\(DIR_INSTANCE\)/)){
        return $self->resolveInstanceDir ($result);
    }
    else{
        return $result;
    }
}

sub isBasePathShared {
    my ($self) = @_;

    my $globalIni = $self->getGlobalIni ();
    if (!defined $globalIni){
        return undef;
    }
    my $result = $globalIni->getValue ($persistence_section,'basepath_shared');
    return (defined $result && $result =~ /$gBoolTruePattern/) ? 1 : 0;

}

sub setBasePath {
    my ($self, $iniKeyBasepath, $newBasepath, $skipWriting) = @_;
    my $global_ini = $self->getGlobalIni();
    if (!defined $global_ini){
        $self->AddError ('Cannot access global.ini');
        return undef;
    }
    my $msg = $self->getMsgLst()->addMessage ('Updating global.ini');
    $global_ini->setMsgLstContext([$msg->getSubMsgLst ()]);

    $newBasepath    = $self->substituteGlobalDir($newBasepath);
    my $oldBasepath = $global_ini->getValue($persistence_section,
                                            $iniKeyBasepath);

    if (!defined $oldBasepath || ($oldBasepath ne $newBasepath)) {
        $global_ini->setValue(CFG_LAYER_SYSTEM,
                              $persistence_section,
                              $iniKeyBasepath,
                              $newBasepath);
        if (!$skipWriting) {
            if (!defined $global_ini->write()){
                $self->AddError ('Cannot write global.ini', $global_ini);
                return undef;
            }
            if (!defined $global_ini->readValues(1)) {
                $self->AddError ('Cannot read global.ini', $global_ini);
                return undef;
            }
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Returns the data path.
# If the path contained the variables '$(DIR_GLOBAL)' or '$(DIR_INSTANCE)'
# the parameter $expand=1 resolves these variables.
#
# Parameter bool $expand

sub getDataPath{
    return $_[0]->getBasePath ('basepath_datavolumes', $_[1]);
}

sub getLogPath{
    return $_[0]->getBasePath ('basepath_logvolumes', $_[1]);
}

sub getEsDataPath{
    return $_[0]->getBasePath ('basepath_datavolumes_es', $_[1]);
}

sub getEsLogPath{
    return $_[0]->getBasePath ('basepath_logvolumes_es', $_[1]);
}

sub getAcceleratorDataPath{
    return $_[0]->getBasePath ('basepath_datavolumes_ets', $_[1]);
}

sub getAcceleratorLogPath{
    return $_[0]->getBasePath ('basepath_logvolumes_ets', $_[1]);
}

sub getPersistentMemoryVolumesPaths {
    my ($self) = @_;
    my @paths = split(/;/, $self->getBasePath ('basepath_persistent_memory_volumes'));
    return \@paths;
}

sub setAcceleratorBasePaths {
    my ($self, $dataPath, $logPath) = @_;
    my $rc = $self->setBasePath('basepath_datavolumes_ets', $dataPath, 1);
    if ($rc) {
        $rc = $self->setBasePath('basepath_logvolumes_ets', $logPath);
    }
    return $rc;
}

sub getLogBackupPath{
    return $_[0]->getBasePath ('basepath_logbackup', $_[1]);
}

sub getDataBackupPath{
    return $_[0]->getBasePath ('basepath_databackup', $_[1]);
}

sub getCatalogBackupPath{
    return $_[0]->getBasePath ('basepath_catalogbackup', $_[1]);
}

sub getReferenceDataPath{
    my ($self) = @_;
    my $scriptServerIni = $self->getScriptServerIni ();
    if (!defined $scriptServerIni){
        return undef;
    }
    return $scriptServerIni->getValue ('adapter_framework', 'dq_reference_data_path');
}

sub setReferenceDataPath {
    my ($self, $newBasepath) = @_;
    my $scriptServerIni = $self->getScriptServerIni();
    if (!defined $scriptServerIni){
        $self->AddError ('Cannot access scriptserver.ini');
        return undef;
    }

    my $msg = $self->getMsgLst()->addMessage("Updating scriptserver.ini");

    $scriptServerIni->setMsgLstContext([$msg->getSubMsgLst()]);

   $scriptServerIni->setValue(CFG_LAYER_SYSTEM,
                          'adapter_framework',
                          'dq_reference_data_path',
                          $newBasepath);

    if (!defined $scriptServerIni->write()){
        $self->AddError ('Cannot write scriptserver.ini', $scriptServerIni);
        return undef;
    }
    if (!defined $scriptServerIni->readValues(1)) {
        $self->AddError ('Cannot read scriptserver.ini', $scriptServerIni);
        return undef;
    }

    return 1;
}

sub usesStorageApi{
    my ($self) = @_;
    my $globalIni = $self->getGlobalIni ();
    if (!defined $globalIni){
        return undef;
    }
    if (!$globalIni->existsValue ('storage','ha_provider')){
        return 0;
    }
    my $ha_provider = $globalIni->getValue ('storage','ha_provider');
    if ($ha_provider){
        return 1;
    }
    return 0;
}

sub adaptVolumePaths{

	my ($self,
	    $oldSid, $newSid,
	    $oldNr,  $newNr,
	    $instconfig,
	    $chownList
	   ) = @_;

    my @ini_keys = qw (basepath_datavolumes
                       basepath_logvolumes
                       basepath_datavolumes_es
                       basepath_logvolumes_es
                       basepath_datavolumes_ets
                       basepath_logvolumes_ets
                       basepath_databackup
                       basepath_logbackup
                       basepath_catalogbackup);

    my @params = qw (BasePathDataVolumes
                     BasePathLogVolumes
                     EsDataPath
                     EsLogPath
                     AcceleratorDataPath
                     AcceleratorLogPath
                     BasePathDataBackup
                     BasePathLogBackup
                     BasePathCatalogBackup);

    my @doChangeDir = (1,1,  1,1,  1,1,  0,0,  1,1);

    my @paramValues;
    my $anyParamFound = 0;

    foreach my $i (0 .. $#params) {
        if ($instconfig->isSkipped ($params[$i])){
            next;
        }
        $paramValues[$i] = $instconfig->getValue ($params[$i]);
        if (defined $paramValues[$i]) {
            $anyParamFound = 1;
        }
    }

    if (($oldSid eq $newSid) &&
        ($oldNr  == $newNr)  &&
        !defined $chownList  &&
        !$anyParamFound) {

        return 1; # nothing to do
    }

    my $msg        = $self->AddMessage ("Adapting volume paths");
    my $submsglst = $msg->getSubMsgLst ();
    my $global_ini = $self->getGlobalIni ($submsglst);

    if (!defined $global_ini){
        $self->AddError ('Cannot get global.ini', $submsglst);
        return undef;
    }

    if (defined $chownList){
        $global_ini->setOwner ($chownList->[1], $chownList->[3]);
    }

    my $errlst = new SDB::Install::MsgLst();

    $global_ini->setMsgLstContext([$submsglst]);

	foreach my $i (0 .. $#params){

		if (!defined $paramValues[$i]){
			next;
		}

		$global_ini->setValue (CFG_LAYER_SYSTEM,
		                       $persistence_section,
		                       $ini_keys[$i],
		                       $paramValues[$i]);

		if (defined $chownList && $doChangeDir[$i]){
			my $submsg = $submsglst->addMessage ('Changing owner'
			    . " (uid: $chownList->[0] -> $chownList->[1]"
			    . ", gid: $chownList->[2] -> $chownList->[3]) of '$paramValues[$i]'...");
			if (!defined changeOwn(@$chownList,
			                       $paramValues[$i],
			                       $errlst,
			                       $submsg->getSubMsgLst (),
			                       SDB::Install::SAPSystem::SKIP_CHOWN_DIR())) {

				$self->AddError ("Cannot chown '$paramValues[$i]'", $errlst);
				return undef;
			}
		}
		$ini_keys[$i] = undef;
	}

    foreach my $i (0 .. $#ini_keys){

		my $key = $ini_keys[$i];
		if (!defined $key){
			next;
		}
		if (!$global_ini->existsValue($persistence_section, $key)){
			next;
		}
		my $oldPath =$global_ini->getValue ($persistence_section, $key, CFG_LAYER_SYSTEM);
		if (!defined $oldPath){
			next;
		}
		if ($oldPath =~ /\$\(/){
			next;
		}
		my $newPath = $self->getNewVolumePath($oldPath, $newSid, $newNr,
		                                                $oldSid, $oldNr, $instconfig->getValue('Target'));

		if ($doChangeDir[$i]) {
			if (!$self->changeOwnAndRenameVolumePath
			                       ($instconfig->{params}->{$params[$i]}->{str},
			                        $chownList,
			                        $oldPath,
			                        $newPath,
			                        $msg,
			                        $errlst)) {
				return undef;
			}
		}
		$global_ini->setValue (CFG_LAYER_SYSTEM, $persistence_section,$key,$newPath);
	}
	if (!defined $global_ini->write ()){
		$self->AddError ("Cannot flush ini file '$global_ini->{name}'", $global_ini);
		return undef;
	}
	return 1;
}


#-------------------------------------------------------------------------------

sub changeOwnAndRenameVolumePath {
    my ($self,
        $paramStr,
        $chownList,
        $oldPath,
        $newPath,
        $msg,
        $errlst,
       ) = @_;

    my $chownPath = $oldPath;
    if ($oldPath ne $newPath) {

        $chownPath = $newPath;
        $msg->getSubMsgLst()->addProgressMessage
                ("Renaming '$paramStr' from '$oldPath' to '$newPath'");

        if (-d $oldPath && -d $newPath) {

            if (rmdir $newPath) {
                $self->AddSubMessage($msg, "Empty directory '$newPath' deleted");
            }
            else {
                $self->AddSubMessage($msg, "Directory '$newPath' already exists");
            }
        }

        if (!-d $newPath) {
            my $submsg = $self->AddSubMessage
                    ($msg, "Renaming directory '$oldPath' => '$newPath'");
            if (!renameRecursive ($oldPath, $newPath, $submsg->getSubMsgLst())){
                $self->AddError ("Cannot rename '$oldPath' => '$newPath': $!");
                return undef;
            }
        }
    }

    if (defined $chownList) {
        my $submsg = $self->AddSubMessage ($msg,  'Changing owner'
                . " (uid: $chownList->[0] -> $chownList->[1]"
                . ", gid: $chownList->[2] -> $chownList->[3]) of '$chownPath'...");
        if (!defined changeOwn(@$chownList,
                               $chownPath,
                               $errlst,
                               $submsg->getSubMsgLst(),
                               SDB::Install::SAPSystem::SKIP_CHOWN_DIR())) {

            $self->AddError ("Cannot chown '$oldPath'", $errlst);
            return undef;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Calls: hdbnsutil --addHostRoles --role=<role1> [--role=<role2>]...

sub registerAdditionalHostRoles {
    my ($self, $roleArray, $configuration) = @_;
    my $hasXsRole = grep {$_ eq $gHostRoleXS2Worker || $_ eq $gHostRoleXS2Standby} @{$roleArray};
    my $rc = 1;

    my $args = ['-addHostRoles'];
    foreach my $role (@$roleArray){
        push @$args, '--role='.$role;
    }

    if($hasXsRole){
        $rc &= $self->createXSSpaceIsolationUsers($configuration, $configuration->getSID());
        $rc &= $self->createXsEaDataPath($configuration);
    }
    return undef if !$rc;

    my $msg = $self->getMsgLst()->addMessage("Registering additional host roles");
    my $msglst = $msg->getSubMsgLst ();

    $rc = $self->runNameServerUtility ($args, $msglst);
    if (!defined $rc){
        $self->setErrorMessage ("Cannot add host roles", $msglst);
    }

    return $rc;
}

#-------------------------------------------------------------------------------
# Calls: hdbnsutil -removeHostRole

sub unregisterHostRole{
    my ($self, $host, $role, $force, $configuration) = @_;

    delete $self->{_hostRolesInfo};
    if (!defined $force){
        $force = 0;
    }
    my $removeHostRoleService = SDB::Install::SAPInstance::Services::UnregisterHostRoleService->new($self);
    $removeHostRoleService->setMsgLstContext($self->getMsgLstContext());
    return $removeHostRoleService->unregisterHostRole($host, $role, $force, $configuration);
}

#-------------------------------------------------------------------------------
# Calls: hdbnsutil -removeHostRole --role=xs_worker --sapcontrol=1

sub removeHostRoleXS2Worker {
    my ($self, $configuration) = @_;
    return $self->unregisterHostRole (undef, $gHostRoleXS2Worker, undef, $configuration);
}

#-------------------------------------------------------------------------------
# Calls: hdbnsutil -removeHostRole --role=xs_standby --sapcontrol=1

sub removeHostRoleXS2Standby {
    my ($self, $configuration) = @_;
    return $self->unregisterHostRole (undef, $gHostRoleXS2Standby, undef, $configuration);
}

sub recreateXSSpaceIsolationUsers {
    my ($self, $configuration, $oldSID, $newSID) = @_;
    my $isChangingSID = ($oldSID ne $newSID);

    if($isChangingSID && !$self->deleteXSSpaceIsolationUsers($configuration, $oldSID)){
        return undef;
    }
    if(!$self->createXSSpaceIsolationUsers($configuration, $newSID)){
        return undef;
    }
    return 1;
}

sub createXSSpaceIsolationUsers {
    my ($self, $configuration, $sid) = @_;
    my $userConfig = $configuration->getSAPSystem(1)->getUserConfig() || {};
    my $sidadm = createSysAdminUserName($sid);
    my $keys = ['xs_space_user_sap', 'xs_space_user_prod'];
    my $shouldModifySudoers = ($configuration->getValue('SkipModifySudoers') || $isWin) ? 0 : 1;

    for my $key (@{$keys}){
        my $uid = $userConfig->{"${key}_id"};
        my $name = ($key eq 'xs_space_user_sap') ? createXSSpaceSAPUserName($sid) : createXSSpaceProdUserName($sid);
        my $user = new SDB::Install::User($name);
        my $gid = getgrnam('users');

        next if(!defined($uid) || $uid == 0 || $gid == 0 || $user->exists());

        my $msg = $self->AddMessage ("Creating XS Space OS User \"$name\"...");
        $user->setMsgLstContext ([$msg->getSubMsgLst()]);
        my $result = $user->create($uid, undef, 'XS Advanced Space Administrator', $gid, undef, '/bin/false');
        if(!defined($result)){
            $self->AddError("Creating XS Space OS User \"$name\" failed", $user);
            return undef;
        }

        if(!$shouldModifySudoers){
            $self->AddMessage ("Modification of file '/etc/sudoers' skipped...");
            next;
        }
        my $line = sprintf($XSA_OS_USER_SUDOERS_PATTERN, $sidadm, $name);
        my $logMessage = "Adding entry in '/etc/sudoers' file for user '$name'...";
        return undef if (! $self->_addLineToSudoers($line, $logMessage));
    }
    if ($shouldModifySudoers) {
        my $line = sprintf($DISABLE_REQUIRES_TERMINAL_SUDOERS_PATTERN, $sidadm);
        my $logMessage = "Adding entry in '/etc/sudoers' that disables 'requiretty' for '$sidadm'...";
        return undef if (! $self->_addLineToSudoers($line, $logMessage));
    }
    return 1;
}

sub recreateXsEaDataPath {
    my ($self, $configuration, $sourceHost) = @_;

    return undef if(!$self->deleteXsEaDataPath($configuration, $sourceHost));
    return undef if(!$self->createXsEaDataPath($configuration));
    return 1;
}

sub createXsEaDataPath {
    my ($self, $configuration) = @_;
    my $user = $self->get_user();
    my $cfg = { 'mode' => 0755 };
    my $targetXsEaDataPath = $configuration->hasValue('XsEaDataPath') ? $configuration->getValue('XsEaDataPath') : $self->getXSEADataPath(1);
    return 1 if(!defined($targetXsEaDataPath)); # Nothing to create.

    if(! -d $targetXsEaDataPath){
        $self->AddMessage("Creating XS Advanced Execution Agent data path '$targetXsEaDataPath'...");
        if (! defined(makedir($targetXsEaDataPath, $cfg))) {
            $self->getErrMsgLst()->addError("Failed to create directory '$targetXsEaDataPath'", $cfg);
            return undef;
        }
    }
    if(!$isWin){
        my ($uid, $gid) = ($user->uid(), $user->gid());

        $self->AddMessage("Adjusting ownership of directory '$targetXsEaDataPath'...");

        if(!chown($uid, $gid, $targetXsEaDataPath)){
            $self->getErrMsgLst()->addError("Failed to adjust the ownership of directory '$targetXsEaDataPath'");
            return undef;
        }
    }
    return 1;
}

sub deleteXsEaDataPath {
    my ($self, $configuration, $sourceHost) = @_;
    my $xsEaDataPathParameter = exists($configuration->{params}->{XsEaDataPath}) ? $configuration->{params}->{XsEaDataPath} : {};
    my $xsEaDataPathValue = $configuration->getValue('XsEaDataPath');
    # 'XsEaDataPath' value is used for update host scenarios called after
    # XSA is uninstalled and there is no 'basepath_xsa_appworkspace' value in global.ini
    my $sourceXsEaDataPath = (exists($xsEaDataPathParameter->{_originPath}) || $xsEaDataPathValue) ? $xsEaDataPathParameter->{_originPath} // $xsEaDataPathValue : $self->getXSEADataPath(1);
    return 1 if (!$sourceXsEaDataPath);
    my $hostName = defined($sourceHost) ? $sourceHost : $self->get_host();
    my $sourceXsEaDataHostDir = File::Spec->catfile($sourceXsEaDataPath, $hostName);
    if(-d $sourceXsEaDataHostDir){
        my $message = $self->getMsgLst()->addMessage("Deleting directory '$sourceXsEaDataHostDir'...");
        my $errorList = new SDB::Install::MsgLst();

        if (!deltree ($sourceXsEaDataHostDir, $message->getSubMsgLst(), $errorList)){
            $self->AddError("Failed to delete directory '$sourceXsEaDataHostDir'", $errorList);
            return undef;
        }
    }
    my $dataPathContents = listDir($sourceXsEaDataPath);
    if(scalar @$dataPathContents == 0 || (scalar @$dataPathContents == 1 && $dataPathContents->[0] eq 'stager')){
        my $message = $self->getMsgLst()->addMessage("Deleting empty directory '$sourceXsEaDataPath'...");
        my $errorList = new SDB::Install::MsgLst();

        if (!deltree ($sourceXsEaDataPath, $message->getSubMsgLst(), $errorList)){
            $self->AddError("Failed to delete directory '$sourceXsEaDataPath'", $errorList);
            return undef;
        }
    }
    return 1;
}

sub _addLineToSudoers {
    my ($self, $line, $logMessage) = @_;
    my $msg = $self->AddMessage($logMessage);
    my $osConfig = new SDB::Install::OSConfig();
    if(!$osConfig->addLineToSudoers($line, quotemeta($line))){
        $self->AddError("Modification of file '/etc/sudoers' failed", $osConfig);
        $self->AddSubMsgLst($msg, $osConfig);
        return undef;
    }
    $self->AddSubMsgLst($msg, $osConfig);
    return 1;
}

sub deleteXSSpaceIsolationUsers {
    my ($self, $configuration, $sid) = @_;
    my $userConfig = $configuration->getSAPSystem(1)->getUserConfig() || {};
    my $sidadm = createSysAdminUserName($sid);
    my $keys = ['xs_space_user_sap', 'xs_space_user_prod'];

    if($configuration->getValue('KeepXsUsers')){
        $self->AddMessage ("Deletion of XS Space OS Users skipped.");
        return 1;
    }

    for my $key (@{$keys}){
        my $name = ($key eq 'xs_space_user_sap') ? createXSSpaceSAPUserName($sid) : createXSSpaceProdUserName($sid);
        my $user = new SDB::Install::User($name);

        next if(!exists($userConfig->{$key}) && !$user->exists());

        my $msg = $self->AddMessage ("Deleting XS Space OS User \"$name\"...");
        if(!$user->delete()){
            $self->AddError("Cannot delete XS Space OS User \"$name\"", $user);
        }
        $self->AddSubMsgLst($msg, $user);

        next if (!$self->_shouldModifySudoers($configuration));

        my $sudoersLine = sprintf($XSA_OS_USER_SUDOERS_PATTERN, $sidadm, $name);
        return undef if (! $self->_removeLineFromSudoers($sudoersLine, $name));
    }
    if ($self->_shouldModifySudoers($configuration)) {
        $self->AddMessage("Removing entry in '/etc/sudoers' that disables 'requiretty' for '$sidadm'...");
        my $sudoersLine = sprintf($DISABLE_REQUIRES_TERMINAL_SUDOERS_PATTERN, $sidadm);
        return undef if (! $self->_removeLineFromSudoers($sudoersLine, $sidadm));
    }
    return 1;
}

sub _shouldModifySudoers {
    my ($self, $configuration) = @_;
    if ($isWin) {
        $self->AddMessage ("Modification of file '/etc/sudoers' is skipped for Windows OS...");
        return 0;
    }
    if ($configuration->getValue('SkipModifySudoers')) {
        $self->AddMessage ("Modification of file '/etc/sudoers' is skipped because of parameter 'SkipModifySudoers'...");
        return 0;
    }
    if (! -e '/etc/sudoers') {
        $self->AddMessage ("Modification of file '/etc/sudoers' is skipped because it doesn't exist...");
        return 0;
    }
    return 1;
}

sub _removeLineFromSudoers {
    my ($self, $sudoersLine, $userName) = @_;
    my $osConfig = new SDB::Install::OSConfig();
    my $msg = $self->AddMessage("Removing entry in '/etc/sudoers' file for user '$userName'...");
    if(!$osConfig->removeLineFromSudoers("\Q$sudoersLine\E")){
        $self->AddError("Modification of file '/etc/sudoers' failed", $osConfig);
        $self->AddSubMsgLst($msg, $osConfig);
        return undef;
    }
    $self->AddSubMsgLst($msg, $osConfig);
    return 1;
}

sub registerInLandscape{
    my ($self, $instconfig) = @_;
    my $roles = $instconfig->getHostRoles ();
    my $group = $instconfig->getValue ('HostFailoverGroup');
    my $storage_partition = $instconfig->getValue ('StoragePartitionNumber');
    my $workerGroupString = $instconfig->getValue ('WorkerGroup');
    my @workerGroups = defined($workerGroupString) && length($workerGroupString) ? split(',', $workerGroupString) : ();
    my @workerGroupArgs = map { sprintf('--workergroup=%s', $_) } @workerGroups;
    my $internalIp = $instconfig->getInternalDeviceIP ();
    my $hasXsRole       = grep {$_ eq $gHostRoleXS2Worker || $_ eq $gHostRoleXS2Standby} @{$roles};
    my $hasDatabaseRole = grep {$_ eq $gHostRoleWorker || $_ eq $gHostRoleStandby} @{$roles};
    my $rc = 1;

    my @role_options;
    foreach my $role (@$roles){
        push @role_options, '--role='.$role;
    }

    if($hasXsRole || $hasDatabaseRole){
        $rc &= $self->createXSSpaceIsolationUsers($instconfig, $instconfig->getSID());
        $rc &= $self->createXsEaDataPath($instconfig);
    }
    return undef if !$rc;

    my $msg = $self->getMsgLst()->addMessage ("Registering new host in $gProductNameEngine landscape");
    my $msglst = $msg->getSubMsgLst ();

    $rc = $self->runNameServerUtility (['-addHost', @role_options,
        $group ? ("--group=$group") : (),
        defined $storage_partition ? "--subpath=$storage_partition" : (),
        defined $internalIp ? "--internalAddress=$internalIp" : (),
        defined $workerGroupString ? @workerGroupArgs : ()],
        $msglst,
        undef,
        1);

    if (!defined $rc){
        $self->setErrorMessage ("Cannot register new host in landscape", $msglst);
    }

    if (!$isWin){
        my $errlst = new SDB::Install::MsgLst ();
        my ($uid,$gid) = ($self->{_user}->uid(), $self->{_user}->gid());
        my $chownError = 0;
        foreach my $dir (
                        $self->get_hostNameDir (),
                        $self->get_globalHdbCustomConfigDir (),
                        $self->get_globalMetadataDir ()
                        ){
            my $infoMsg = $msglst->AddMessage("Changing owner of '$dir'");
            if (!defined changeOwn (
                        0, $uid,
                        undef, $gid,
                        $dir,
                        $errlst,
                        $infoMsg->getSubMsgLst(),
                        SDB::Install::SAPSystem::SKIP_CHOWN_DIR ())) {
                $chownError = 1;
            }
        }
        if ($chownError){
            $self->getMsgLst()->addWarning ("Cannot chown by hdbnsutil generated files", $errlst);
        }
    }

    return $rc;
}

sub reconfigureTrexNet{
    my ($self, $instconfig) = @_;
    return 1 if($instconfig->hasValue('InternalHostnameResolution'));

    $self->updateInternalNetworkConfig($instconfig);

    my $msg = $self->getMsgLst()->addMessage ("Reconfigure $gProductNameEngine network settings of $self->{_host}");
    my $msglst = $msg->getSubMsgLst ();
    my @args = ('-reconfig', '--force');
    my $internalIp = $instconfig->getInternalDeviceIP ();
    my $listenInterface = $instconfig->getValue('ListenInterface');
    my $maxRetries = 1;

    if (defined $listenInterface) {
        $maxRetries = MAX_RECONFIG_LISTEN_IF_RETRIES;
        push @args, ("--hostnameResolution=$listenInterface");
        if (defined($internalIp)) {
            push @args, ("--internalAddress=$internalIp") if ($listenInterface ne 'local');
        }
    }
    else {
        if ($self->getInternalNetworkPrefix() && (!defined $internalIp || $internalIp =~ /^\s*$/)) {
            $self->setErrorMessage ("no internal ip address");
            return undef;
        }
        push @args, ("--internalAddress=$internalIp");
    }
    my $globalIniFileName = $self->get_globalHdbCustomConfigDir () .
        $path_separator . 'global.ini';

    my @statbufBefore = stat ($globalIniFileName);

    my $rc;
    my $builtin = SDB::Common::BuiltIn->get_instance();
    while(!$rc && $maxRetries > 0 ){
        $rc = $self->runNameServerUtility (\@args,$msglst);
        $maxRetries--;
        $builtin->sleep(1);
    }
    if (defined $rc){
        # wait until global.ini is written
        my $statbufAfter;
        my $endTime = time () + 10; # wait maximal 10s
        while (1){
            $statbufAfter = File::stat::stat($globalIniFileName);
            if (defined $statbufAfter){
                if (
                    $statbufBefore[1] != $statbufAfter->[1] || # indode
                    $statbufBefore[9] < $statbufAfter->[9]     #mtime
                ){
                    last;
                }
            }
            usleep (10);
            if (time () >= $endTime){
                last;
            }
        }
    }
    else{
        $self->setErrorMessage ("Cannot register new host in landscape", $msglst);
    }
    return $rc;
}



#-------------------------------------------------------------------------------
# Updates .ini-files in case of $resetLandscapeId
# and calls 'hdbnsutil -convertTopology'.
#
# Parameter: hash    $hostHashMap 1)
#            boolean $resetLandscapeId
#
#    1)  key: origin host name, value: new host name
#
# Returns int retCode

sub rebuildTopology{
	my ($self, $hostHashMap, $tenantMap, $resetLandscapeId, $doTargetSystemReplicationCleanUp) = @_;
	my $msg = $self->AddProgressMessage ("Converting topology...");
	my $msglst = $msg->getSubMsgLst ();
	my @hostargs;
	delete $self->{_hostRolesInfo};

	if (defined $hostHashMap) {

	    foreach my $oldHost (keys %$hostHashMap) {
			push @hostargs, ("--oldHost=$oldHost",
			                 "--newHost=$hostHashMap->{$oldHost}");
		}
	}

    if (defined $tenantMap) {
        my @tenantMapArgs = $self->_buildTenantMapArgs($tenantMap);
        push @hostargs, @tenantMapArgs;
    }

    if ($resetLandscapeId){
        #
        # force new landscapeId
        #
        my $layered_cfg = $self->getLayeredConfig (1);
        if (!defined $layered_cfg){
            return undef;
        }

        my $lmsg = $msg->getSubMsgLst()->addMessage ("Resetting landscapeId...");
        $layered_cfg->setMsgLstContext([$lmsg->getSubMsgLst ()]);

        my $ns_ini = $layered_cfg->getIniFile ('nameserver.ini');
        if (!defined $ns_ini){
            $self->AddError ('Cannot get nameserver.ini', $layered_cfg);
            return undef;
        }
        $lmsg = $lmsg->getSubMsgLst()->addMessage ("Reading nameserver.ini");
        $ns_ini->setMsgLstContext([$lmsg->getSubMsgLst ()]);
        if (!defined $ns_ini->readValues ()){
            $self->AddError ("Cannot read nameserver.ini", $ns_ini);
            return undef;
        }
        $ns_ini->setValue (CFG_LAYER_SYSTEM,'landscape','id',undef);
        $ns_ini->removeKey (CFG_LAYER_SYSTEM,'landscape','idsr');
        if (!defined $ns_ini->write ()){
            $self->AddError ("Cannot flush ini file '$ns_ini->{name}'", $ns_ini);
            return undef;
        }
    }

    if ($doTargetSystemReplicationCleanUp && $self->useSystemReplication () &&
        !defined $self->runNameServerUtility (['-sr_cleanup', '--force'], $msglst)){
            $self->AddWarning (undef, $msglst);
    }

	my $rc = $self->runNameServerUtility (['-convertTopology', @hostargs],
		$msglst);
    if (!defined $rc){
        $self->AddError (undef, $msglst)
    }
    if (defined $tenantMap){
        # force reload of databases.lst
        $self->{_mdcDatabases} = undef;
    }
    return $rc;
}

sub _buildTenantMapArgs {
    my ($self, $tenantMap) = @_;
    my @result = ();
    my @orderedKeys = ();

    foreach my $key (keys %$tenantMap) {
        if ($key eq lc($tenantMap->{$key})) {
            push @orderedKeys, $key;
            next;
        }
        $self->_sortByNamingDependency($key, \@orderedKeys, $tenantMap);
    }
    my @entries = ();
    foreach my $key (@orderedKeys) {
        @entries = ("--oldTenant=$key", "--newTenant=$tenantMap->{$key}");
        push @result, @entries;
    }
    return @result;
}

sub _sortByNamingDependency {
    my ($self, $key, $orderedKeys, $tenantMap) = @_;
    if (grep {$key eq lc($_)} @$orderedKeys) {
        return;
    }
    if ($self->tenantExists($tenantMap->{$key})) {
        $self->_sortByNamingDependency(lc($tenantMap->{$key}), $orderedKeys, $tenantMap);
    }
    push @$orderedKeys, $key;
}

sub getNameServerUtility{
    my($self)  = shift;
    return SDB::Install::SAPInstance::NameserverUtility->new($self);
}
sub runNameServerUtility{
    my ($self,$args,$msglst, $ref_output_buffer, $no_user_switch, $ref_input_buffer) = @_;
    my $nameSrvUitlity = $self->getNameServerUtility();
    return $nameSrvUitlity->run($args,$msglst, $ref_output_buffer, $no_user_switch, $ref_input_buffer);
}

sub runHDBDaemonUtility{
	my ($self,$args,$msglst, $ref_output_buffer, $no_user_switch) = @_;

    return $self->runUtilityInHDBEnv('hdbdaemon',$args, $msglst, $ref_output_buffer, undef, undef, undef, $no_user_switch);
}

sub runUtilityInHDBEnv{
    my ($self,$tool, $args, $msglst, $ref_output_buffer, $ref_input_buffer, $additionalEnviroment,$outputHandler, $no_user_switch, $fSubstLogEntry, $havePtyIo) = @_;
    my $prog = File::Spec->rel2abs($tool . ($isWin ? '.exe' : ''), $self->get_instanceExeDir());
    my $original_umask = umask ($sidadm_umask) if (!$isWin);
    my $rc = $self->runUtilityInEnv($prog, $args, $msglst, $ref_output_buffer, $ref_input_buffer, $additionalEnviroment,$outputHandler, $no_user_switch, $fSubstLogEntry, $havePtyIo);
    if (!$isWin && $original_umask != $sidadm_umask){
        umask ($original_umask);
    }
    return $rc;
}

sub _getHDBEnv{
    my ($self, $env) = @_;
    if (!defined $env){
        $env = {};
    }
    my $sysinfo     = new SDB::Install::SysInfo ();
    my $instanceDir = $self->get_instanceDir ();
    my $exeDir      = $self->get_instanceExeDir ();
    my $hostName    = $self->get_host ();
    my $hostDir     = $self->get_usrSapHostNameDir ();

    if (!defined $hostDir || !-d $hostDir){
        $hostDir     = $self->get_hostNameDir ();
    }

    if (!defined $hostDir){
        $self->setErrorMessage ("Cannot determine HANA environment: unknown host directory");
        return undef;
    }

    $env->{SAP_RETRIEVAL_PATH}      = $hostDir;
    $env->{SAPSYSTEMNAME}           = $self->{_sid};
    $env->{SECUDIR}                 = $hostDir . $path_separator . 'sec';
    $env->{DIR_EXECUTABLE}          = $exeDir;
    $env->{DIR_INSTANCE}            = $instanceDir;
    $env->{HDB_HOST}                = $hostName;
    $env->{DIR_GLOBAL}              = $self->get_globalDir ();
    $env->{PYTHONDONTWRITEBYTECODE} = 'x';

    if ($self->isAddressSanitizerBuild ()){
        $env->{ASAN_OPTIONS} = "alloc_dealloc_mismatch=0:log_path=$hostDir/trace/AddressSanitizer_Crashdump:handle_segv=0:allow_user_segv_handler=1:abort_on_error=1:check_initialization_order=1:detect_leaks=0:detect_odr_violation=0";
        $env->{ASAN_SYMBOLIZER_PATH} = "$exeDir/llvm-symbolizer";
    }

    my @PATH;
    if (defined $env->{PATH}){
        @PATH = split($env_path_separator, $env->{PATH});
    }
    if (!grep {$_ eq $exeDir} @PATH){
        unshift @PATH, $exeDir;
        $env->{PATH} = join($env_path_separator,@PATH);
    }

    $env->{TSAN_OPTIONS} = "exitcode=0:log_path=/tmp/ThreadSanitizer_Report";

    if (!$isWin){
        $env->{USER}= $self->{_user}->getSidAdmName();
        $env->{HOME}= $self->{_user}->getSidAdmHome();
        $env->{LD_LIBRARY_PATH} = $exeDir .
            ($env->{LD_LIBRARY_PATH} ?
            $env_path_separator . $env->{LD_LIBRARY_PATH} :
            '');
    }
    else{
        $env->{PATH} = $exeDir . $path_separator .
                'Python' . $env_path_separator . $env->{PATH};
    }
    if($env->{PYTHONPATH}) {
        $env->{PYTHONPATH} = $env->{PYTHONPATH}.$env_path_separator.$exeDir;
    }
    else {
        $env->{PYTHONPATH} = $exeDir.$path_separator;
    }
    $env->{PYTHONPATH} = $env->{PYTHONPATH}.$env_path_separator.$exeDir.$path_separator."python_support";
    if($env->{PYTHONHOME}) {
        $env->{PYTHONHOME} = $env->{PYTHONHOME}.$env_path_separator.$exeDir.$path_separator."Python";
    }
    else {
        $env->{PYTHONHOME} = $exeDir.$path_separator."Python";
    }

    my $lssInstance = $self->getLssInstance();
    if(defined $lssInstance){
        $env->{LSS_HOME} = $lssInstance->getLssSidDir();
        $env->{LSS_LOCAL_PATH} = $lssInstance->getLocalLssLocalDir();
    }

    return $env;
}


sub runUtilityInEnv{
	my ($self, $prog, $args, $msglst, $ref_output_buffer, $ref_input_buffer, $additionalEnvironment, $outputHandler, $no_user_switch, $fSubstLogEntry, $havePtyIo, $wantExitCode) = @_;
	my $errlst = $msglst;
	if (!defined $no_user_switch){
		$no_user_switch = 0;
	}
	if (!defined $msglst){
		$msglst = $self->getMsgLst ();
		$errlst = $self->getErrMsgLst ();
	}

    if (!defined $wantExitCode){
        $wantExitCode = 0;
    }

    local %ENV = %ENV;

    my ($uid, $gid, $home);
    if (!$isWin && !$no_user_switch){
        $uid = $self->{_user}->uid();
        if ($> == $uid){
            $uid = undef;
        }
        else{
            $gid = $self->{_user}->gid();
        }
        $home = $self->{_user}->getSidAdmHome ();
        my $sidadmUser = $self->{_user}->getSidAdm();
        $self->_updateEnvTimezone(\%ENV, $sidadmUser);
    }

    if (defined $home){
        $ENV{HOME} = $home;
    }
    if (!defined $self->_getHDBEnv (\%ENV)){
        return undef;
    }

    if ($additionalEnvironment) {
        my %combinedHash = (%ENV, %{$additionalEnvironment});
        %ENV = %combinedHash;
    }

    my $exer = new LCM::ProcessExecutor (
                    $prog,
                    $args,
                    $ref_input_buffer,
                    undef,
                    undef,
                    $uid,
                    $gid,
                    $havePtyIo);
    $exer->setMsgLstContext ([$msglst]);
    if (!defined $outputHandler){
        require SDB::Install::OutputHandler;
        $outputHandler = new SDB::Install::OutputHandler (
            undef,
            '  ' . basename ($prog) . ': '
            );
    }
    $outputHandler->setProgressHandler ($msglst->getProgressHandler());
    if (defined $fSubstLogEntry){
        $exer->setSubstLoggedOutputFunction ($fSubstLogEntry);
    }
    $exer->setOutputHandler ($outputHandler);
    my $rc = $exer->executeProgram ();

    if (defined $ref_output_buffer){
        my $arrayOutput = $exer->getOutputLines();
        $$ref_output_buffer =
            defined $arrayOutput ? join ("\n", @$arrayOutput) : '';
    }

    if (!defined $rc){
        $errlst->addError("$prog call failed", $exer->getErrMsgLst());
        return undef;
    }

    if ($wantExitCode){
        return $rc;
    }
    if ($rc){
        my $exerMsglstCopy = new SDB::Install::MsgLst();
        $exerMsglstCopy->appendMsgLst($exer->getMsgLst());
        $errlst->addError("$prog call failed [rc = $rc]", $exerMsglstCopy);
        return undef
    }

    return 1;
}

sub _updateEnvTimezone {
    my ($self, $env, $sidadmUser) = @_;
    my $sidadmEnv = $sidadmUser->getInitialEnvironment();
    return if (!defined $sidadmEnv);

    if (exists $sidadmEnv->{TZ}) {
        $env->{TZ} = $sidadmEnv->{TZ};
    }
    elsif (exists $env->{TZ}) {
        delete $env->{TZ};
    }
}

sub installTopology{
    my (
        $self,
        $initTopologyArgs,
        $submsglst,
        $systemUserPW
    ) = @_;
    return $self->runNameServerUtility(
        ['-initTopology', defined $initTopologyArgs ? @{$initTopologyArgs} : ()],
        $submsglst,
        undef,
        undef,
        $systemUserPW
    );
}

sub createSecureStore{
    my ($self, $instconfig, $msglst) = @_;
    my $createSecureStoreArgs = [];

    if (!$instconfig->isUpdate()) {
        my $shouldSkipInitialTenant = $self->isMultiDb() && !$instconfig->getValue('CreateInitialTenant');
        push @{$createSecureStoreArgs}, '--no_initial_tenant' if $shouldSkipInitialTenant;
        push @{$createSecureStoreArgs}, '--target=LSS'        if $instconfig->shallUseLSS();
    }

    return $self->runNameServerUtility(['-createSecureStore' , @{$createSecureStoreArgs}], $msglst);
}

sub createSsfsGlobalKey{
    my ($self, $msglst, $isSystemPKI) = @_;
    local %ENV = %ENV;
    my $saveCntxt;
    if (defined $msglst){
        $saveCntxt = $self->setMsgLstContext ([$msglst]);
    }

    if (defined $isSystemPKI && $isSystemPKI){
        $ENV{'RSEC_SSFS_DATAPATH'} = $self->get_globalSystemSsfsDataDir ();
        $ENV{'RSEC_SSFS_KEYPATH'}  = $self->get_globalSystemSsfsKeyDir ();
    }
    else{
        $ENV{'RSEC_SSFS_DATAPATH'} = $ENV{'RSEC_SSFS_KEYPATH'} = $self->get_globalSsfsDir ();
    }
    if (defined $saveCntxt){
        $self->setMsgLstContext ($saveCntxt);
    }

    my $key;

    my $outputHandler = new SDB::Install::OutputHandler::NoOutHndlr ();

    my $logHandler =
        sub {
            return '*' x 58;
        };

    my $rc = $self->runUtilityInHDBEnv (
        'rsecssfx',
        ['generatekey', '-getPlainValueToConsole'],
        $msglst,
        \$key,
        undef,
        undef,
        $outputHandler,
        undef,
        $logHandler);

    if (!$rc){
        return $rc;
    }

    if (!$key){
        $self->setErrorMessage ('ssfs global key not defined');
        return undef;
    }

    if ($key =~ /\n/){
        # strip libc error messages
        my @lines = split (/\n/, $key);
        $key = pop @lines;
    }


    return $self->runUtilityInHDBEnv (
            'rsecssfx',
            ['changekey', '-valuefile7bit' ,'-valuefile', '-'],
            $msglst,
            undef,
            [$key],
            undef,
            $outputHandler);
}


sub pingMasterNameServer{
	return $_[0]->runNameServerUtility (['-pingNameserver', '--master'], $_[1]);
}

sub getTimeOffset{
    my ($self, $msglst) = @_;
    my $output;
    my $rc = $self->runNameServerUtility (['-getTimeOffset', '-sapcontrol=1'], $msglst, \$output);
    if (!$rc){
        return undef;
    }
    my ($timeOffset) = ($output =~ /timeOffset=(-?\d+)/m);
    if (!defined $timeOffset){
        $self->setErrorMessage ('Cannot parse hdbnsutil output');
        return undef;
    }
    return int ($timeOffset);
}


sub getStartStopHostInfo {
    my ($self, $hostMapRoles, $host) = @_;
    my $thisHost  = (defined $host) ? $host : $self->get_host();
    $thisHost     = lc(hostname()) if (!defined $thisHost);
    my $hostRoles = $hostMapRoles->{$thisHost};
    if (!defined $hostRoles || ($hostRoles eq '')) {
        return "'$thisHost'";
    }
    $hostRoles =~ s/\s+/, /g;
    return "'$thisHost' ($hostRoles)";
}


sub startHost{
	my ($self, $user, $passwordForWindows, $host, $timeout, $useHttps, $sso_cert) = @_;

	if (!defined $host){
		if ($isWin){
			$host = $self->{_host};
		}
	}
	else{
		$host = lc ($host);
		my $found = 0;
		foreach my $lhost (@{$self->get_allhosts()}){
			if ($host eq $lhost){
				$found = 1;
				last;
			}
		}

		if (!$found){
			$self->AddError ("Host '$host' is not known");
			return undef;
		}
	}
	if ($isWin && !defined $host){
		$self->AddError ("No host defined");
		return undef;

	}

	my $hostInfo   = $self->getStartStopHostInfo($self->getHostRolesInfo(), $host);
	my $strHostMsg = "host $hostInfo";
	my $msg = $self->AddMessage ("Starting instance on $strHostMsg...");
	$msg = $self->AddMessage ("Parameters: instance number = $self->{_nr}, user = $user");

	$self->takeTraceDirSnapshot($host, $msg->getSubMsgLst());

	my $sapcontrol = new SDB::Install::SAPControl ($self->_getSAPControlHost ($host),
	                                               $self->{_nr},
	                                               $user, # sidadm
	                                               $passwordForWindows,
	                                               $useHttps, $sso_cert);
	$sapcontrol->setMsgLstContext ([$msg->getSubMsgLst ()]);

	if (defined $self->{_f_callback}){
		$sapcontrol->set_callback ($self->{_f_callback});
	}
    my ($start_wait_retries,$start_wait_delay);
    if (defined $timeout){
        $start_wait_delay = 2;
        $start_wait_retries = int ($timeout / $start_wait_delay);
    }

	if (!defined $sapcontrol->StartWait ($start_wait_retries, $start_wait_delay, $hostInfo) ){
		$self->AddError ("Start instance $self->{_nr} on $strHostMsg failed.", $sapcontrol);
		$self->analyzeTraceDirDeltas($host);
		return undef;
	}
	else{
		$self->AddMessage ("Instance on $strHostMsg started");
	}
	if ($self->hostIsActiveMaster ($host)){
		$self->onMasterStarted ();
	}
	return 1;
}


sub stopHost{
    my ($self, $user, $pw, $host, $timeout, $useHttps, $sso_cert) = @_;

    if (!defined $host){
        if ($isWin){
            $host = $self->{_host};
        }
    }
    else{
        $host = lc ($host);
        my $found = 0;
        foreach my $lhost (@{$self->get_allhosts()}){
            if ($host eq $lhost){
                $found = 1;
                last;
            }
        }
        if (!$found){
            $self->setErrorMessage ("Host '$host' is not known");
            return undef;
        }
    }
    if ($isWin && !defined $host){
    $self->setErrorMessage ("No host defined");
    return undef;

    }
    my $msglst = $self->getMsgLst ();
    my $hostInfo   = $self->getStartStopHostInfo($self->getHostRolesInfo(), $host);
    my $strHostMsg = "host $hostInfo";
    my $msg = $msglst->addMessage ("Stopping instance on $strHostMsg...");
    $msg->getSubMsgLst()->addMessage ("Parameters: instance number = $self->{_nr}, user = $user");

    $self->takeTraceDirSnapshot($host, $msg->getSubMsgLst());

    my $sapcontrol = new SDB::Install::SAPControl (
            $self->_getSAPControlHost ($host), $self->{_nr}, $user, $pw, $useHttps, $sso_cert);

    $sapcontrol->setMsgLstContext ([$msg->getSubMsgLst()]);
    if (defined $self->{_f_callback}){
        $sapcontrol->set_callback ($self->{_f_callback});
    }

    if (!defined $timeout){
        # default stop timeout
        $timeout = 800;
    }
    my $stop_wait_delay = 2;
    my $stop_wait_retries = int ($timeout / $stop_wait_delay);

  	my $pHandler = $self->getMsgLst()->getProgressHandler();
   	$sapcontrol->getMsgLst()->setProgressHandler($pHandler);
    if (!defined $sapcontrol->StopWait ($stop_wait_retries, $stop_wait_delay, $hostInfo)){
        $self->setErrorMessage ("Stop instance $self->{_nr} on $strHostMsg failed.", $sapcontrol->getErrMsgLst ());
        $self->analyzeTraceDirDeltas($host);
        return undef;

    }
    else{
        $msglst->addMessage ("Instance on $strHostMsg stopped");
    }
    return 1;
}

#@statuschecked("create instance filesystem")
sub createFS{
	my ($self, $instanceNumber) = @_;
	#This function creates the directories and symlinks for a TREX instance, which is shared by all hosts"
	my $msg = $self->AddMessage ("Creating instance file system");
    $self->{_fsMan}->setMsgLstContext([$msg->getSubMsgLst (), $self->getErrMsgLst ()]);
    $msg->setEndTag ('Create Instance Filesystem');
	if (!defined $self->{_fsMan}->createInstanceFS ($instanceNumber)){
        return undef;
	}
	if ($isWin){
		$self->setPermissions ($self->get_localInstanceDir, 1);
	}

	return 1;

}

sub createInstance{
	my ($self, $fsManager, $instanceNumber) = @_;
	$self->{_fsMan} = $fsManager;
	$self->{_fsMan}->set_instanceNr ($instanceNumber);
	if (!defined $self->createFS ()){
		return undef;
	}
	if (!defined $self->installHDBScripts ()){
		return undef;
	}
}

sub create{
	my ($self, $fsManager, $sidadmUser, $instconfig, $sapSys, $sysinfo, $progress_handler) = @_;
    my $hostname = $instconfig->getValue('HostName');
    my $autostart = $instconfig->getValue('AutoStart');
    my $instanceNr = $instconfig->getValue('InstanceNumber');
	$self->{_user} = $sidadmUser;
	$self->{_host} = $hostname;
	if (!defined $self->createInstance ($fsManager, $instanceNr)){
		return undef;
	}

	my $existingLogFiles = getDirectoryFilenames($gLogDir);

	my $rc = $self->_addHost ($hostname, 1, $autostart,
		$instconfig, $sapSys, $sysinfo, $progress_handler);

	if (defined $instconfig->{options}->{action_id}) {

		copySelectedFiles($gLogDir,
		             $self->get_hostNameTraceDir(),
		             $self->getMsgLst(),
		             $instconfig->{options}->{action_id},
		             $existingLogFiles,
		             $self->{_user}->uid(),
		             $self->{_user}->gid());
	}
	return $rc;
}



sub checkSapStartSrv{
    my ($self, $user, $passwd, $host, $msglst, $useHttps, $sso_cert) = @_;
    require SDB::Install::SAPControl;
    if (!defined $host){
        $host = $self->{_host};
    }
    my $sapcontrolHost = $self->_getSAPControlHost ($host);
    my $sapcontrol = new SDB::Install::SAPControl ($sapcontrolHost,
                                                   $self->{_nr},
                                                   $user,
                                                   $passwd,
                                                   $useHttps, $sso_cert);

    my $rc;

    if (!defined $sapcontrolHost){
        $rc = $sapcontrol->isServiceRunning () ? 0 : 1;
    }
    else{
        $rc =  $sapcontrol->accessCheck ();
    }
    if (defined $msglst){
        if ($sapcontrol->ErrorState()){
            $msglst->AddError (undef, $sapcontrol);
        }
        else{
            $msglst->AddMessage (undef, $sapcontrol);
        }
    }
    return $rc;
}


sub stopSapStartSrv{
	my ($self,
	    $user,          # sidadm name
	    $password,      # sidadm password
	    $timeout,
	    $localHostOnly,
	    $setError,      # does not ignore error when stopping sapstartsrv
	    $useHttps,
	    $sso_cert
	   ) = @_;

	my $msg = $self->AddProgressMessage ("Stopping sapstartsrv service...");
	my $msglst = $msg->getSubMsgLst ();
	if (!$localHostOnly) {
        foreach my $host (@{$self->{_hosts}}){
            $msglst->addMessage("Stopping sapstartsrv service at host '$host'...");
            my $sapcontrol = new SDB::Install::SAPControl ($self->_getSAPControlHost ($host),
                                                           $self->{_nr},
                                                           $user,
                                                           $password,
                                                           $useHttps, $sso_cert);
            if (defined $self->{_f_callback}){
                $sapcontrol->set_callback ($self->{_f_callback});
            }
            $sapcontrol->setMsgLstContext ([$msglst]);
            if (!defined $sapcontrol->stopService()){
                $self->PushError ("Cannot stop sapstartsrv", $sapcontrol);
                if ($setError) {
                    return undef;
                }
            }
        }
	}

	$msglst->addMessage("Stopping local sapstartsrv service...");
	my $sapcontrol = new SDB::Install::SAPControl ($isWin ? $self->{_host} : undef,
	                                               $self->{_nr},
	                                               $user,
	                                               $password,
	                                               $useHttps, $sso_cert);
	$sapcontrol->set_callback($self->{_f_callback});
	$sapcontrol->setMsgLstContext ([$msglst]);
	if (!defined $sapcontrol->stopServiceWait ($timeout)){
		$self->PushError ("Cannot stop local sapstartsrv", $sapcontrol);
        if ($setError) {
            return undef;
        }
	}
	return 1;
}


sub cleanupLocalLeftovers{
    my ($self, $msglst, $errlst, $useHttps) = @_;

    if (!defined $msglst){
        $msglst = $self;
    }
    my $nr = $self->{_nr};
    my $sc = new SDB::Install::SAPControl ('localhost',
                                           $nr,
                                           undef, # user
                                           undef, # password
                                           0);

    if ($sc->connect ()){
        $msglst->AddMessage ('old sapstartsrv is running');
        $sc->disconnect ();
        #
        # sapstartsrv is running, try to stop it via unix domain socket
        #
        my $unix_sc = new SDB::Install::SAPControl (undef,
                                                    $nr,
                                                    undef, # user
                                                    undef, # password
                                                    $useHttps);
        $unix_sc->setMsgLstContext ([$msglst->isa('SDB::Install::BaseLegacy') ? $msglst->getMsgLst () : $msglst]);
        if($unix_sc->stopServiceWait (5)){
            $msglst->AddMessage ('stop request via unix domain socket succeeded');
        }
        else{
            $msglst->AddMessage ('stop request via unix domain socket failed');
        }
        if ($sc->connect ()){
            #
            # sapstartsrv is still running, try to kill it
            #

            # get pid by tcp port 5<instance nr>13
            my $ps = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_SOCKETS);
            my $port = int sprintf ('5%02d13', $nr);
            my $pids = $ps->WhoUsesPort ($port);
            my $msg = $self->AddMessage ("Detecting sapstartsrv process via tcp socket port");
            my $submsglst = $msg->getSubMsgLst();
            if (defined $pids && @$pids){
                my $cmd;
                my $error = 0;
                foreach my $pid (@$pids){
                    $cmd = $ps->GetArgs ($pid);
                    $submsglst->addMessage("Process '$cmd'(pid = $pid) is using tcp port '$port'");
                    my $pgid=-$pid;
                    if ($ps->Terminate ($pgid, $ps->GetUid ($pid), $cmd) == 0){
                        $submsglst->addMessage(' => process killed.');
                    }
                    else{
                        my $errmsg = $submsglst->addError("Killing process '$cmd'(pid = $pid) failed: $!");
                        $errlst->appendMsg($errmsg);
                        $error = 1;
                    }
                }
                if ($error){
                    return undef;
                }
            }
            else{
                $self->AddSubMessage ($msg, 'No process found.');
            }
        }
        undef $sc;
    }

    $self->cleanipc();

    #
    # removing old sapservice entries
    #

    #
    # sapservicesPattern matches /profile/???_*<$nr>_*
    #
    my $sapservicesPattern = '.*' . quotemeta ('/profile/') .
        '[A-Z][A-Z0-9][A-Z0-9]_' .
        '[A-Z]{3,}' . sprintf ('%02d', $nr) . '_\S+.*';

    #
    # sapservicesPattern matches current profile name
    #
    my $profilePattern = '.*' . quotemeta ($self->get_profileName()) . '.*';

    #
    # pattern matches if $sapservicesPattern matches && $profilePattern does not
    #
    my $pattern = sprintf ('(?=%s)(?!%s)',$sapservicesPattern, $profilePattern);


    $self->removeSapservicesEntry (undef, $pattern, 1);

    #
    # delete files /tmp/.sapstream5##13 and /tmp/.sapstartsrv##_sapstartsrv.log
    # if they are owned by another user
    #


    my $file;
    my $rc = 1;
    foreach my $template ($sapstartsrv_stream_template, $sapstartsrv_log_template){
        $file = sprintf ($template,int $nr);
        if (-e $file){
            if ((stat(_))[4] != $self->{_user}->uid()){
                if (unlink ($file)){
                    $msglst->AddMessage ("File '$file' deleted");
                }
                else{
                    my $errmsg = $msglst->AddError("Cannot delete file '$file': $!");
                    $errlst->appendMsg($errmsg);
                    $rc = undef;
                }
            }
        }
    }
    return $rc;
}


sub _unblockStopSignals{
    my ($self) =  @_;

    #
    # check whether there are blocked signals SIGQUIT, SIGTERM, SIGINT, and SIGKILL
    #

    my %sig_num;
    @sig_num{split (' ', $Config{sig_name})} = split (' ', $Config{sig_num});
    my $oldset = 0;
    if (sigemptyset ($oldset) != 0){
        $self->setErrorMessage ("sigemptyset() failed: $!");
        return undef;
    }

    if (sigprocmask (SIG_BLOCK, 0, $oldset) != 0){
        $self->setErrorMessage ("sigprocmask() failed: $!");
        sigfreeset ($oldset);
        return undef;
    }

    my @stopsignals = qw (INT KILL QUIT TERM);
    my ($sig_num, $rc, $set, @blocked_signals);
    my $msglst = $self->getMsgLst();
    my $msg = $msglst->AddMessage ('Checking whether there are blocked stop signals');
    foreach my $signal (@stopsignals){
        $sig_num = $sig_num{$signal};
        if (!defined $sig_num){
            $self->setErrorMessage ("signal '$signal' not found");
            sigfreeset ($oldset);
            sigfreeset ($set) if ($set);
            return undef;
        }
        $sig_num = int ($sig_num);
        $rc = sigismember ($oldset, $sig_num);
        if ($rc < 0){
            $self->setErrorMessage ("sigismember() failed: $!");
            sigfreeset ($oldset);
            sigfreeset ($set) if ($set);
            return undef;
        }
        if ($rc){
            $msg->getSubMsgLst ()->addMessage ("signal '$signal' is blocked");
            if (!$set){
                $set = 0;
                if (sigemptyset ($set) != 0){
                    sigfreeset ($oldset);
                    return undef;
                }
            }
            if (sigaddset ($set, $sig_num) != 0){
                    $self->setErrorMessage ("sigaddset failed: $!");
                    sigfreeset ($set);
                    sigfreeset ($oldset);
                    return undef;
            }
            push @blocked_signals, $signal;
        }
        else{
            $msg->getSubMsgLst ()->addMessage ("signal '$signal' is not blocked");
        }
    }

    if ($set){
        if (sigprocmask (SIG_UNBLOCK, $set, $oldset) != 0){
            $self->setErrorMessage ("sigprocmask() failed: $!");
            sigfreeset ($set);
            sigfreeset ($oldset);
            return undef;
        }
        sigfreeset ($set);
        $msglst->addMessage ("Signal(s) " . join (', ', @blocked_signals) .
            ' successfully unblocked.')
    }

    if (@blocked_signals){
        return $oldset;
    }

    return 0;
}

sub _restoreSignalMask{
    my ($self, $oldset) = @_;
    if (sigprocmask (SIG_SETMASK, $oldset, 0) != 0){
        $self->setErrorMessage ("Cannot restore original signal mask: sigprocmask() failed: $!");
        sigfreeset ($oldset);
        return undef;
    }
    $self->getMsgLst ()->addMessage ("Original signal mask restored.");
    sigfreeset ($oldset);
    return 1;
}

sub startLocalSapStartSrvUx{
    my ($self, $msglst, $timeout, $useHttps) = @_;

    if (!defined $msglst){
        $msglst = $self;
    }

    if ($> == 0){
        my $errlst = new SDB::Install::MsgLst();
        if (!defined $self->cleanupLocalLeftovers ($msglst, $errlst, $useHttps)){
            $self->setErrorMessage('Cannot cleanup old running sapstartsrv service', $errlst);
            return undef;
        }
    }

    my $autoStartEnabled = $self->isAutoStartEnabledLocal ();

    if ($autoStartEnabled) {
        if (!$self->enableAutoStartLocal (0)) {
            return undef;
        }
    }

    my $oldsigset = $self->_unblockStopSignals ();

	#
	# send start request
	#

	my $prog = $self->get_instanceExeDir () . $path_separator . 'sapstartsrv';
    my $monitoredTraceFile = new SDB::Install::MonitoredTraceFile (
        $self->get_hostNameTraceDir() . $path_separator . 'sapstartsrv.log');
    $monitoredTraceFile->takeSnapshot ();

    $self->addDebug("Profile name is ".$self->get_profileName ());
    $self->addDebug("<sid>adm user is ".$self->{_user}->getSidAdmName());
    $self->addDebug("Instance number is ".$self->get_nr());
	my $args = ['pf=' . $self->get_profileDir () . $path_separator .
				$self->get_profileName () ,'-D', '-u',
				$self->{_user}->getSidAdmName()];
	my $cfg = {};
	my $msg = $self->getMsgLst ()->addMessage ('Starting service');
	$msg->getSubMsgLst ()->injectIntoConfigHash ($cfg);
	my $rc;
	{
		local %ENV = %ENV;
		$ENV{LD_LIBRARY_PATH} = join (':', $self->get_instanceExeDir (),
			defined $ENV{LD_LIBRARY_PATH} ? $ENV{LD_LIBRARY_PATH} : ());
		$rc = exec_program ($prog, $args, $cfg);
	}

    if ($oldsigset){
        $self->_restoreSignalMask ($oldsigset);
    }

	if (!defined $rc || $rc){
		$msglst->AddError ('Cannot start service', $cfg);
        if ($autoStartEnabled) {
            $self->enableAutoStartLocal (1);
        }
		return undef;
	}

	#
	# wait for sapstartsrv is listening
	#

	$msg = $msglst->AddMessage ("Waiting for sapstartsv...");

    my $sapcontrol;
    $rc = 1;

    my $sapstartsrvIsGone = 0;

    eval{
         $sapcontrol = new SDB::Install::SAPControl ($self->{_host},
	                                                   $self->{_nr},
	                                                   undef, # user
	                                                   undef, # password
	                                                   $useHttps);
	    $sapcontrol->setMsgLstContext ([$msg->getSubMsgLst()]);
	    $sapcontrol->set_callback($self->{_f_callback});
        my $wait_rc = $sapcontrol->waitForService ($timeout,undef,undef, $self->get_profilePattern());
        if (!defined $wait_rc){
            $sapstartsrvIsGone = 1;
        }
        if (!$wait_rc){
            $msglst->AddError (undef , $sapcontrol);
            $rc = undef;
        }
    };

    if ($autoStartEnabled) {
        $self->enableAutoStartLocal (1);
    }

    if ($@){
        if ($@ =~ /Can't\sload\s.*SSLeay/m){
            $msglst->AddWarning ("Skipping tcp socket check due to:\n$@");
        }
        else{
            if ($autoStartEnabled) {
                $self->enableAutoStartLocal (1);
            }
            $self->setErrorMessage ("Cannot check tcp socket conntection: $@");
            return undef;
        }
    }
    if (!$sapstartsrvIsGone){
        # try connect with unix domain socket
        $msg = $msglst->AddMessage ("Checking unix domain socket");
        $sapcontrol = new SDB::Install::SAPControl (undef,
                                                    $self->{_nr},
                                                    undef, # user
                                                    undef, # password
                                                    $useHttps);
        $sapcontrol->setMsgLstContext ([$msg->getSubMsgLst()]);
        $sapcontrol->set_callback($self->{_f_callback});
        if (!$sapcontrol->waitForService ($timeout)) {
            $msglst->PushError (undef , $sapcontrol);
            $rc = undef;
        }
    }

    if ($autoStartEnabled) {
        $self->enableAutoStartLocal (1);
    }

    if (!defined $rc){

         my $host = $self->{_host};
         if ($host eq lc (hostname ())){
                my $isHostLocal = $self->isHostLocal ($host, 1);
                if (defined $isHostLocal && $isHostLocal == 0){
                    $msglst->PushError ("MACHINE DNS CONFIGURATION ERROR!!!!");
                    $msglst->PushError ("Physical hostname '$host' doesn't resolve to a local ip address.");
                }
         }

        my $trcmsg = $msglst->AddMessage (
            "Looking for error messages in sapstartsrv trace file");
        $monitoredTraceFile->analyzeDirectoryDeltas ($trcmsg->getSubMsgLst (),0);
        if (!$trcmsg->getSubMsgLst ()->isEmpty()){
            if ($msglst->isa ('SDB::Install::Base')){
                $msglst->getErrMsgLst ()->appendMsg ($trcmsg);
            }
        }

        if ($sapstartsrvIsGone){
            # try to start sapstartsrv in foreground
            my $foregroundMsg = $msglst->AddMessage("Starting sapstartsrv in foreground to fetch error messages");
            $cfg = {'PTY_IO' => 1, 'TIMEOUT' => 5};
            $foregroundMsg->getSubMsgLst ()->injectIntoConfigHash($cfg);
            $args = [grep {$_ ne '-D'} @$args];
            {
                local %ENV = %ENV;
                $ENV{LD_LIBRARY_PATH} = join (':', $self->get_instanceExeDir (),
                    defined $ENV{LD_LIBRARY_PATH} ? $ENV{LD_LIBRARY_PATH} : ());
                $rc = exec_program ($prog, $args, $cfg);
                if (!$cfg->{WAS_A_TIMEOUT}){
                    $msglst->getErrMsgLst()->appendMsg($foregroundMsg);
                }
            }
        }

        my $http_port = int (sprintf ('5%02d13', $self->{_nr}));
        my $https_port = int (sprintf ('5%02d14', $self->{_nr}));
        my @ports = ($http_port, $https_port);
        my $sockMessage = $msglst->AddMessage ("Checking whether tcp ports '$http_port' and '$https_port' are already in use");
        my $ps = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_SOCKETS);
        my ($pids, $pid, $portMsg);
        my $uid;
        my $sidadmUid = $self->{_user}->uid ();
        my $wrongProcessFound;
        foreach my $port (@ports){
            $pids = $ps->WhoUsesPort ($port);
            if (!defined $pids || !@$pids){
                $sockMessage->getSubMsgLst ()->addMessage ("Tcp port '$port' isn't used");
                next;
            }
            $portMsg = $sockMessage->getSubMsgLst ()
                ->addMessage ("Tcp port '$port' is already in use by process:");
            $wrongProcessFound = 0;
            foreach $pid (@$pids){
                $uid = $ps->GetUid ($pid);
                if ($ps->GetCmd($pid) ne 'sapstartsrv' || $uid != $sidadmUid){
                    $wrongProcessFound = 1;
                }
                $portMsg->getSubMsgLst ()->addMessage ($ps->GetArgs ($pid) .
                    sprintf (" (pid = %d, user = %s)", $pid, getpwuid ($uid)));
            }
            if ($wrongProcessFound){
                if ($msglst->isa ('SDB::Install::Base')){
                    $msglst->getErrMsgLst ()->appendMsg ($portMsg);
                }
            }
        }
    }

	return $rc;
}

sub startLocalSapStartSrvWin{
	my ($self, $msglst, $timeout) = @_;

	if (!defined $msglst){
		$msglst = $self;
	}
    if (!defined $timeout){
        $timeout = 60;
    }

	#
	# send start request
	#

	my $prog = 'sc.exe';
	my $args = ['start', 'SAP' . $self->{_sid} . '_' . $self->{_nr}];

	my $cfg = {};
	my $rc = exec_program ($prog, $args, $cfg);

	my $alreadyRunning = 1056;

	if (defined $rc && ($rc == $alreadyRunning)) {
		return 1;
	}

	if (!defined $rc || $rc){
		$msglst->AddError ('Cannot start service', $cfg);
		return undef;
	}

	#
	# wait for state 'RUNNING'
	#

	$args->[0] = 'query';

	my ($output, $state_nr, $state_name);
	my $tries = 0;
	while (1){
		$output = '';
		$cfg = {'out' => \$output};
		$rc = exec_program ($prog, $args, $cfg);
		if (!defined $rc || $rc){
			$msglst->AddError ('Cannot query service status', $cfg);
			return undef;
		}
		($state_nr,$state_name) = ($output =~ /STATE\s*:\s*(\d+)\s*(\S*)/m);
		if (!defined $state_nr){
			$msglst->AddError ('Cannot parse query service state reply', $cfg);
			return undef;
		}
		if ($state_nr != 2){
			# !START_PENDING
			last;
		}
		sleep (1);
		if ($timeout != 0 && $timeout < ++$tries){
            $msglst->AddError ("Timout reached ($timeout)", $cfg);
            return undef;
        }
	}
	if ($state_nr != 4){
		# !RUNNING
		$msglst->AddError ('Query service state didn\'t reply \'RUNNING\'', $cfg);
		return undef;
	}
	return 1;
}

sub startLocalSapStartSrv;

*startLocalSapStartSrv = $isWin ? \&startLocalSapStartSrvWin : \&startLocalSapStartSrvUx;

sub removeUnreachableHost {
    my ($self, $host, $force) = @_;
    $self->AddMessage ("Removing unreachable host '$host' from landscape...");
    my $hostRoles = $self->getHostRolesByIniFile($host);
    if (!defined $self->removeHostFromLandscape ($force,
                                                 $host,
                                                 undef, # instconfig
                                                 $hostRoles,
                                                 undef, # sapSystem
                                                 0,     # shouldKillTrex
                                                 0,  #skipHdbnsutil
                                                 1      # host is not reachable
                                                 )){
        if ($force){
            $self->AddMessage ('Skipping errors due to force option');
        }
        else{
            return undef;
        }
    }

    if ($isWin){
        sleep (1);
    }
    if (!defined $self->removeHostNameDir ($host)){
        if ($force){
            $self->AddMessage ('Skipping errors due to force option');
        }
        else{
            return undef;
        }
    }

    $self->AddMessage ("Removing profile");
    if (!defined $self->removeProfile ($host)){
        if ($force){
            $self->AddMessage ('Skipping errors due to force option');
        }
        else{
            return undef;
        }
    }

    delete $self->{_hostRolesInfo};
    $self->getTrexHostsFromProfileDir ($self->get_profileDir ());

    return 1;
}

#-------------------------------------------------------------------------------
# Execute uninstall on each remote host

sub sendUninstallCommands {

    my ($self, $force, $instconfig, $deleteUser, $rHosts) = @_;

    my $msg = $self->getMsgLst()->addProgressMessage("Uninstalling hosts...");
    my $msglst = $msg->getSubMsgLst();
    $rHosts->setMsgLstContext([$msglst]);

    my $progress_message = "Uninstalling host '%s'...";
    my $done_message     = "Uninstallation of host '%s' done.";
    my $error_message    = "Uninstallation of host '%s' failed!";
    my $actionID         = $instconfig->{options}->{action_id};
    my $rc               = 0;

    if ($rHosts->isHostctrl) {

        my $sapmnt = substr($self->{_globalSidDir}, 0,
                            length($self->{_globalSidDir})
                            - length($path_separator)
                            - length($self->{_sid}));

        my $optionMap =
            {'SAPMNT'             => $sapmnt,
             'SID'                => $self->{_sid},
             'FORCE'              => (($force)       ? 'on' : 'off'),
             'KEEP_USER'          => ((!$deleteUser) ? 'on' : 'off'),
             'KXU'                => ($instconfig->getValue('KeepXsUsers') ? 'on' : 'off'),
             'SUD'                => ($instconfig->getValue('SkipModifySudoers') ? 'on' : 'off'),
             'KEEP_USER_HOME_DIR' =>
                    (($deleteUser && $instconfig->getValue('KeepUserHomeDir'))
                    ? 'on' : 'off'),
        };
        $optionMap->{ACTION_ID} = $actionID if (defined $actionID);

        $rc = $rHosts->executeHostctrlSerial($gOperationUninstallInstance,
                                             $instconfig,
                                             undef, # parameter IDs
                                             undef, # password keys
                                             undef, # opt ignore
                                             undef, # opt timeout
                                             $progress_message,
                                             $done_message,
                                             $error_message,
                                             $optionMap,
                                             undef, # host map option map
                                             undef, # execute only hosts
                                             $force);
    }
    else {
        # SSH connection

        my $hdbuninst = join ($path_separator,
                              $self->get_globalTrexInstallDir(),
                              'bin',
                              'hdbuninst');

        my $cmd = "$hdbuninst -b --noexclusive -s $self->{_sid} --scope=instance";
        $cmd .= ' --force' if ($force);
        $cmd .= " --action_id=$actionID" if (defined $actionID);

        if (!$deleteUser) {
            $cmd .= ' --keep_user';
        }
        elsif ($instconfig->getValue('KeepUserHomeDir')) {
            $cmd .= ' --keep_user_home_dir';
        }
        if($instconfig->getValue('KeepXsUsers')){
            $cmd .= sprintf(' %s=on', $instconfig->getOpt('KeepXsUsers'));
        }
        if($instconfig->getValue('SkipModifySudoers')){
            $cmd .= sprintf(' %s=on', $instconfig->getOpt('SkipModifySudoers'));
        }

        $rc = $rHosts->executeSerial($cmd,
                                     $force,
                                     $progress_message,
                                     $done_message,
                                     $error_message);
    }

    if ($rc != 0) {
        $self->AddError ("Uninstallation of other hosts failed", $rHosts);
        if (!$force){
            return undef;
        }
    }
    return 1;
}


sub removeTrexInstance{
	my ($self, $inst, $force, $instconfig, $deleteUser) = @_;

	if (!defined $self->{_host}){
		$self->getTrexHostsFromProfileDir ($self->get_profileDir (), 1);
	}
	if ($self->{_host}){
		my $forceRemoveFromLandscape = $force || (defined $instconfig && $instconfig->isa ('SDB::Install::Configuration::Uninstallation'));
		my $localHostRoles = $self->getHostRolesByIniFile($self->{_host});

        my $lastHost = 0;
		if ($self->exists_remote_host() && defined $instconfig){
			my $rHosts = $instconfig->getRemoteHosts();
			if(!$isWin && defined $rHosts){
				# uninstall other hosts
				if (!$self->sendUninstallCommands(	$force,
													$instconfig,
													$deleteUser,
													$rHosts)) {
					return undef;
				}
				$self->{_hosts} = [];
				$self->getTrexHostsFromProfileDir ($self->get_profileDir ());
				$lastHost = 1;
			}
		} else {
		   $lastHost = 1;
		}

        my $sidToUnregister  = $self->{'_sid'};
        my $nrToUnregister   = $self->{'_nr'};
        my $hostToUnregister = $self->{'_host'};

        #
        # In uninstall context data integrity checks must not
        # be performed => $forceRemoveFromLandscape = true
        # In removehost context it should => $forceRemoveFromLandscape = false
        #
		if (!defined $self->removeHostFromLandscape ($forceRemoveFromLandscape, $self->{_host}, $instconfig, $localHostRoles, $inst, 1, $lastHost)) {
			return undef if (!$forceRemoveFromLandscape);

            my $message = $force ? 'Skipping errors due to force option' : 'Ignore errors in uninstall context';
            $self->AddMessage($message);
		}

		if ($isWin){
			sleep (1);
		}

		if (!defined $self->removeHostNameDir ()){
			return undef;
		}

		if (!defined $self->removeSapservicesEntry ($self->get_profileName())){
			return undef;
		}

		if (!defined $self->removeProfile ()){
			return undef;
		}

		$self->{_host} = undef;

		$self->getTrexHostsFromProfileDir ($self->get_profileDir ());

		if (!$self->exists_remote_host()){
			$self->removeVolumes ();
			if (! defined $self->removeInstanceDir ()){
				return undef;
			}
		}
		elsif ($isWin && $self->{_globalSidDir} =~ /^\\\\/){
			if (! defined $self->removeInstanceDir ()){
				return undef;
			}
		}
        # if this fails, we don't really care. It is only done to refresh some cached
        # hostagent information on this system:
        if (!$instconfig->shouldSkipHostagentCalls()) {
            unregisterInstanceService($self, 0, $gLogDir, $sidToUnregister, $nrToUnregister, $hostToUnregister, 5);
        }
	}
	else{
		if (!$self->exists_remote_host()){
			$self->removeVolumes ();
			if (! defined $self->removeInstanceDir ()){
				return undef;
			}
		}
		elsif ($isWin && $self->{_globalSidDir} =~ /^\\\\/){
			if (! defined $self->removeInstanceDir ()){
				return undef;
			}
		}
		else{
			$self->AddWarning ("No local HDB host found to remove in '$SAPSERVICES_FILE'. Nothing to do.");
		}
	}
	return 1;
}

sub removeHostNameDir{
	my ($self, $host) = @_;
	my $msglst = new SDB::Install::MsgLst ();
	my $msg = $self->AddMessage ("Deleting host file system");
	#$self->SetFormatInfo ($msg, 'h1', 'Delete host file system');
    my $errlst = new SDB::Install::MsgLst ();
	if (!deltree ($self->get_hostNameDir ($host), $msg->getSubMsgLst (), $errlst)){
		$self->AddError ("Cannot remove host file system", $errlst);
		return undef;
	}
	return 1;
}


sub removeProfile{
	unlink ($_[0]->get_profileDir() . $path_separator .
			$_[0]->get_profileName(undef, undef, $_[1]));
}


#-------------------------------------------------------------------------------
# Tries to remove sapservice entry.
# Either profile or pattern is used.
# In case of cleanup 'not found' messages are suppressed.
#
# Parameter: string $profileName
#            string $pattern
#            bool   $isCleanup
#
#    1)  key: origin host name, value: new host name
#
# Returns int retCode
sub removeSapservicesEntry{
	my ($self, $profileName, $pattern, $isCleanup) = @_;

	my $strAction = ($isCleanup) ? 'Cleanup' : 'Unregistering';
	my $msg       = $self->AddMessage ("$strAction sapstartsrv");

	if ($isWin){
		my $prog = $self->get_instanceExeDir () . $path_separator . 'sapstartsrv.exe';

        if (!-f $prog){
            $prog = $self->get_globalTrexExeDir ()  . $path_separator . 'sapstartsrv.exe';
        }

		my $params = [
                '-u',		# unregister
                '-q',		# quiet, no popups
                '-s', $self->{_sid},
                '-n', $self->{_nr}
		];

		my $cfg = {};

		my $rc = exec_program ($prog,$params,$cfg);

		if (!defined $rc || $rc != 0){
			$self->AddError ('Unregistration of sapstartsrv FAILED', $cfg);
			$self->AddSubMsgLst ($msg, $cfg);
			return undef;
		}
		$self->AddSubMsgLst ($msg, $cfg);
		return 1;
	}

	if (!-e $SAPSERVICES_FILE){
		return 1;
	}

	my $lines = $self->readSapservicesFile();
	if (!defined $lines) {
	    return undef;
	}

    if (!defined $pattern){
        if (!defined $profileName){
            $profileName = $self->get_profileName ();
        }
        $pattern = quotemeta ($profileName);
    }

    my $found = 0;

    foreach my $line (@$lines){
        chomp ($line);
        if ($line =~ /$pattern/){
            $found = 1;
            last;
        }
    }

    if (!$found && !$isCleanup){
        $self->AddSubMessage ($msg, "No entry found in '$SAPSERVICES_FILE' PATTERN was $pattern");
    }

	if ($found) {
	    my $auxFilename = $self->getSapservicesAuxfile();

        if (!open (FD, '>', $auxFilename)){
            $self->AddError ("Cannot create file '$auxFilename': $!");
            return undef;
        }

        foreach my $line (@$lines){
            chomp ($line);
            if ($line =~ /$pattern/){
                $self->AddSubMessage ($msg, "Deleted line '$line' in '$SAPSERVICES_FILE'");
            }
            elsif (!print FD "$line\n") {
                $self->AddError ("Cannot write file '$auxFilename': $!");
                return undef;
            }
        }

        if (!close (FD)) {
            $self->AddError ("Cannot close file '$auxFilename': $!");
            return undef;
        }

        if (!$self->renameSapservicesAuxFile($auxFilename)) {
            return undef;
        }
	}

	return 1;
}

#@statuschecked("remove host from landscape")
sub removeHostFromLandscape{
	my ($self,
		$force,
		$host,
		$instconfig, # used to get additional accelerator parameters
		$roles,
		$sapSystem, # Must be defined if $shouldKillTrex = 1
		$shouldKillTrex,
		$skipHdbnsutil,
        $isHostUnreachable
	   ) = @_;

	if (defined $host){

		my ($isAcceleratorHost,
		    $isESHost,
		    $isStreamingHost) = (0,0,0);

		if (defined $roles) {
			foreach my $currRole (@$roles) {
				if (($currRole eq $gHostRoleAcceleratorWorker) ||
				    ($currRole eq $gHostRoleAcceleratorStandby)) {
					$isAcceleratorHost = 1;
				}
				elsif (($currRole eq $gHostRoleEsWorker) ||
				       ($currRole eq $gHostRoleEsStandby)) {
					$isESHost = 1;
				}
				elsif ($currRole eq $gHostRoleStreaming) {
					$isStreamingHost = 1;
				}

            }
        }
        delete $self->{_hostRolesInfo};

		my $hasXsRole = defined($roles) && grep {$_ eq $gHostRoleXS2Worker || $_ eq $gHostRoleXS2Standby} @{$roles};
        my $hasDbRole = defined($roles) && grep {$_ eq $gHostRoleWorker || $_ eq $gHostRoleStandby} @{$roles};
		if(($hasXsRole || $hasDbRole) && !$isHostUnreachable && !$self->deleteXSSpaceIsolationUsers($instconfig, $instconfig->getSID())){
			return undef if(!$force);
			$self->AddMessage('Skipping errors due to force option');
		}

        if (!$isHostUnreachable && !$self->removeSpecialHostRoles($force,
                                           $host,
                                           $instconfig,
                                           $isESHost,
                                           $isAcceleratorHost,
                                           $isStreamingHost)) {
            return undef if (!$force);
            $self->AddMessage('Skipping errors due to force option');
        }

        if ($shouldKillTrex) {
            if (!defined $self->killtrex ($sapSystem)){
                if ($force){
                    $self->AddMessage ('Skipping errors due to force option');
                }
                else{
                    return undef;
                }
            }
        }

        if(!$isHostUnreachable && $hasXsRole && !$self->deleteXsEaDataPath($instconfig, $host)){
            return undef if(!$force);
            $self->AddMessage('Skipping errors due to force option');
        }

        my $lss = $self->getLssInstance();
        if(!$isHostUnreachable && defined $lss && $hasDbRole){
            if(!$lss->removeFromLocalHost($instconfig->getValue('KeepLSSUser'),$instconfig->getValue('KeepLSSUserGroup'))){
                return undef if (!$force);
                $self->AddMessage('Skipping errors due to force option');
            }
        }

        if (!$skipHdbnsutil) {
            my $msg = $self->AddMessage ("Removing host '$host' from landscape");
            my $msglst = new SDB::Install::MsgLst ();
            my $rc = $self->runNameServerUtility(['-removeHost', $host, $force ? ('--force') : ()], $msglst);
            if (!defined $rc){
                $self->AddError ("Cannot remove host '$host' from landscape", $msglst);
                $self->AddSubMsgLst ($msg,$msglst);
                return undef;
            }
            $self->AddSubMsgLst ($msg,$msglst);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Calls specific scripts to remove special host roles
# (i.e. extended storage host, streaming host, remote data sync host)

sub removeSpecialHostRoles {

    my ($self,
        $force,
        $host,
        $instconfig,
        $isESHost,
        $isAcceleratorHost,
        $isStreamingHost,
       ) = @_;

    delete $self->{_hostRolesInfo};

    my $tenantUser = undef;
    my $tenantPassword = undef;
    if (defined $instconfig) {
        $tenantUser      = $instconfig->getValue('TenantUser');
        $tenantPassword  = $instconfig->getValue('SQLTenantUserPassword');
    }

    if ($isESHost){
        if (!$self->callEsRemoveHost($host, $force, $instconfig)) {
            return undef;
        }
    }

    if ($isAcceleratorHost){
        my $removeScript = 'aseremovehost';
        my $removeCall   = $self->getAcceleratorLcmDir() . $path_separator
                           . $removeScript . '.sh';
        my $msg = $self->getMsgLst ()->addMessage ("Performing '$removeScript'");
        my $args= ['-host', $host, '-log_dir', $gLogDir];
        if ($force) {
            push @$args, '-force';
        }
        my $inputBuffer = undef;

        if (defined $instconfig) {
            my $systemUser      = $instconfig->getValue('SystemUser');
            my $systemUserPw    = $instconfig->getValue('SQLSysPasswd');
            my $acceleratorUser = $instconfig->getValue('AcceleratorUser');
            my $acceleratorPw   = $instconfig->getValue('AcceleratorPassword');

            if (defined $systemUser   && defined $acceleratorUser &&
                defined $systemUserPw && defined $acceleratorPw) {

                push @$args, '-hdbuser';
                push @$args, $systemUser;
                push @$args, '-aseuser';
                push @$args, $acceleratorUser;
                $inputBuffer = ["$systemUserPw $acceleratorPw"];
            }
        }
        if ($self->shouldAutoInitializeService($instconfig, $gDirNameAccelerator)) {
            if (!defined $tenantUser || !defined $tenantPassword) {
                $self->getErrMsgLst->addMessage("Missing tenant user credentials to remove '$gDirNameAccelerator' host");
                return undef;
            }
            push(@$args, '-auto_initialize_services');
            push(@$args, ('-tenant_user', $tenantUser));
#_TODO_ Input buffer password order
            push(@$inputBuffer, $tenantPassword);
        }
        my $rc = $self->runUtilityInEnv ($removeCall,
                                         $args,
                                         $msg->getSubMsgLst(),
                                         undef,
                                         $inputBuffer);
        if (!$rc){
            $self->setErrorMessage ("$removeScript failed",
                                    $msg->getSubMsgLst ());
            return undef;
        }
    }

    if ($isStreamingHost){
        my @args = ('-host', $host, $force ? '-force' : (), '-log_dir', $gLogDir);
        my $inputBuffer = [];
        if ($self->shouldAutoInitializeService($instconfig, $gDirNameStreaming)) {
            if (!defined $tenantUser || !defined $tenantPassword) {
                $self->getErrMsgLst->addMessage("Missing tenant user credentials to remove '$gDirNameStreaming' host");
                return undef;
            }
            push(@args, '-auto_initialize_services');
            push(@args, ('-tenant_user', $tenantUser));
            $inputBuffer = ["$tenantPassword"];
        }
        my $strremovehost = File::Spec->catfile($self->getStreamingLcmDir(), 'strremovehost.sh');
        my $msg = $self->getMsgLst ()->addMessage ("Performing 'strremovehost.sh'");
        my $rc = $self->runUtilityInEnv ($strremovehost, \@args, $msg->getSubMsgLst, undef, $inputBuffer);
        if (!$rc){
            $self->setErrorMessage ("strremovehost failed", $msg->getSubMsgLst());
            return undef;
        }
    }

	return 1;
}


sub callEsRemoveHost {
    my ($self, $host, $force, $instconfig) = @_;

    my $esremovehost = $self->getExtendedStorageLcmDir()
                       . $path_separator . 'esremovehost.sh';
    my $msg    = $self->getMsgLst ()->addMessage ("Performing 'esremovehost'");
    my $args   = ['-host', $host, '-log_dir', $gLogDir];
    push @$args, '-force'      if ($force);
    if ($self->shouldAutoInitializeService($instconfig, $gDirNameEs)) {
        push(@$args, '-auto_initialize_services');
    }

    my $rc = $self->runUtilityInEnv ($esremovehost, $args, $msg->getSubMsgLst());
    if (!$rc){
        $self->setErrorMessage ("esremovehost failed", $msg->getSubMsgLst ());
        return undef;
    }
    return 1;
}


sub removeInstanceDir{
	my $errlst = new SDB::Install::MsgLst ();
	my $dir = $_[0]->{_localUsrSapSid};

	if (!defined $dir){
		$dir = $_[0]->{_instanceDir};
	}

	my $msg = $_[0]->AddMessage ("Removing instance directory '$_[0]->{_instanceDir}'");
	if ($isWin){
		if (!deltree ($_[0]->{_instanceDir}, $msg->getSubMsgLst(), $errlst)){
			$_[0]->AddError ("Cannot remove instance directory", $errlst);
			return undef;
		}
		if (-d $_[0]->{_instanceDir}){
			sleep (1);
		}
	}
	else{
		if (-l $_[0]->{_instanceDir}){
			my $globdir = readlink ($_[0]->{_instanceDir});
			if (-d $globdir && !deltree ($globdir,$msg->getSubMsgLst(), $errlst)){
				$_[0]->AddError ("Cannot remove global instance directory '$globdir'", $errlst);
				return undef;
			}
			if (!unlink ($_[0]->{_instanceDir})){
				$_[0]->AddError ("Cannot remove instance link '$_[0]->{_instanceDir}': $!");
				return undef;
			}
		}
		elsif (!deltree ($_[0]->{_instanceDir},$msg->getSubMsgLst(), $errlst)){
			$_[0]->AddError ("Cannot remove instance directory",$errlst);
			return undef;
		}
	}
	return 1;
}

sub removeVolumes{
	my ($self) = @_;
	my (@entries, $entry, $msg);
	my $rc = 1;
	my $errlst;
	my @volumes = (
		$self->getDataPath(),
		$self->getLogPath(),
		$self->getEsDataPath(1),
		$self->getEsLogPath(1),
		$self->getAcceleratorDataPath(1),
		$self->getAcceleratorLogPath(1),
		@{$self->getPersistentMemoryVolumesPaths()},
	);

	foreach my $path (@volumes){
		if (!defined $path){
			next;
		}
		if ($path =~ /\$\(/){
			next;
		}
		$msg = $self->AddMessage ("Deleting volumes in $path...");
		@entries = grep {/^hdb\d+$|^mnt\d+$|^NVM-GMD-SID-\w{3}$/} @{listDir($path)};
		foreach $entry (@entries){
			$entry = File::Spec->catfile($path, $entry);
			if (isDirectory($entry)){
				$errlst = new SDB::Install::MsgLst();
				if (!deltree($entry, $msg->getSubMsgLst(), $errlst)){
					$self->PushError("Cannot delete directory '$entry'", $errlst);
					$rc = undef;
				}
			}
			else{
				if (!SDB::Common::BuiltIn::unlink($entry)){
					$self->PushError("Cannot delete '$entry': $!");
					$rc = undef;
				}
			}
		}
		if (isEmptyDir($path) && !isMountPoint($path) && !deltree($path)){
			$self->AddWarning("Cannot delete directory '$path': $!");
		}
	}
	return $rc;
}


sub getSqlSSLProperties{
    my ($self, $msglst) = @_;
    if (!defined $msglst){
        $msglst = $self->getMsgLst ();
    }
    my $host = $self->getSqlHost ();
    require SDB::Install::Sql::SSLConnectProperties;
    my $sslProperties = new SDB::Install::Sql::SSLConnectProperties ();
    $sslProperties->setMsgLstContext ([$msglst]);
    my $secureConnectProperties = $sslProperties->findSSLProperties ($self, $host);
    if (!defined $secureConnectProperties){
        $self->setErrorMessage ('Cannot get ssl connect properties',
            $sslProperties->getErrMsgLst());
        return undef;
    }
    return $secureConnectProperties;
}

#
# Return codes:
#   1       success
#   0       import error (regi failed)
#   undef   error

sub importDeliveryUnits{
	my ($self, $instconfig, $path, $copy_autocontent_to_global, $msg, $support_resume) = @_;
	# if $path is undef -> all DUs in 'auto_content' dir are imported, also DUs in 'content' if they
	#    already exist in the database
	# if $path is a directory -> it replaces the 'auto_content' dir, and there is no 'content' dir
	# if $path is a file -> it is treated as the only DU to be imported

    my $isRepositoryEnabled = $self->isRepositoryEnabled ();
    if (defined $isRepositoryEnabled && !$isRepositoryEnabled){
        my $msg = $self->getMsgLst ()->addProgressMessage ("Skipping delivery units import...");
        $msg->getSubMsgLst->addMessage ("Repository is disabled");
        return 1;
    }

	my @files;
	my @subdirFiles;
	my $filesToImport;
	my @units;
	my @auto_units;
	my $installed_units;
	my $unit_counter = 0;

	my $user_store_key = $instconfig->getValue('UserStoreKey');
	my $multidb = $self->isMultiDb();
	my $wrnLst = new SDB::Install::MsgLst ();
	my $errLst = new SDB::Install::MsgLst ();
	my $STEP_IMPORT_CONTENT = $instconfig->isUpdate() ?
			$SDB::Install::SAPSystem::STEP_UPGRADE_IMPORT_CONTENT :
			$SDB::Install::SAPSystem::STEP_IMPORT_CONTENT;
	my $isResume = $instconfig->isResume($STEP_IMPORT_CONTENT);
	if (!defined $msg){
		$msg = $self->AddProgressMessage ("Importing delivery units...");
		$msg->setEndTag ('Importing delivery units');
	}
    if ((!defined $path) && $copy_autocontent_to_global) {
        # copy option has no effect since source and target are identical
        $copy_autocontent_to_global = 0;
    }
	$self->takeTraceDirSnapshot(undef, $msg->getSubMsgLst());
	my $msglst = $msg->getSubMsgLst ();
	if ($instconfig->isUpdate()) {
		# skip import if this is a secondary system (replication scenario)
		$msglst->setProgressHandler ($self->getMsgLst()->getProgressHandler());
		my $secondary = $self->isSecondarySystem($msglst, $errLst);
		if (!defined $secondary) {
			if ($instconfig->getIgnore('check_secondary_system')) {
				$msg->endMessage ();
				$wrnLst->addWarning ("Ignoring failed secondary system check due to switch ignore=check_secondary_system");
				$self->AddSubMsgLst ($msg, $wrnLst);
				$wrnLst->initMsgLst ();
			}
			else{
				$msg->endMessage ();
				$self->getErrMsgLst()->addError ("Error importing delivery units", $errLst);
				return undef;
			}
		}
		if ($secondary) {
			$msg->endMessage ();
			$wrnLst->setProgressHandler($self->getMsgLst()->getProgressHandler());
			$wrnLst->addProgressMessage ("Skipping import of delivery units because this is a secondary system");
			$self->AddSubMsgLst ($msg, $wrnLst);
			if ($instconfig->getIgnore('check_secondary_system')) {
				$msg->endMessage ();
				$wrnLst->addWarning ("Ignoring result of secondary system check due to switch ignore=check_secondary_system");
				$self->AddSubMsgLst ($msg, $wrnLst);
				$wrnLst->initMsgLst ();
			}
			else{
				return 1;
			}
		}
	}

	# create list of automatic delivery units in file system
	my $explicitPath = 0;
	if (defined $path && -f $path) {
		if (! -r $path) {
			$msg->endMessage ();
			$wrnLst->AddError ("Cannot read archive \"$path\"");
			$self->AddSubMsgLst ($msg, $wrnLst);
			$self->AddError (undef, $wrnLst);
			$wrnLst->initMsgLst ();
			return undef;
		}
		my $unit = new SDB::Install::DeliveryUnit($path);
		if (!defined $unit->init()){
			$msg->endMessage ();
			$wrnLst->AddError ("Cannot initialize archive \"$path\"", $unit);
			$self->AddSubMsgLst ($msg, $wrnLst);
			$self->AddError (undef, $wrnLst);
			$wrnLst->initMsgLst ();
			return undef;
		} else {
			$msglst->addMessage("Selected for import: $path");
			push @auto_units, $unit;
		}
		$explicitPath = 1;
	} else {
		my $archive_dir;
		if (defined $path && -d $path) {
			$archive_dir = $path;
			$explicitPath = 1;
		} else {
			$archive_dir = $self->get_globalAutoContentDir();
		}
		unless (opendir (DH, $archive_dir)){
			$msg->endMessage ();
			$self->AddError ("Cannot open archive directory \"".$archive_dir."\": $!");
			return undef;
		}
		@files = readdir (DH);
		closedir (DH);
		if (!$explicitPath) {
			foreach my $file (@files){
				my $filePath = join($path_separator, $archive_dir, $file);
				if (-d $filePath && (!($file =~ /^\./))) {
					if ($multidb || (!$multidb && ($file ne 'systemdb'))) {
						unless (opendir (DH, $filePath)){
							$msg->endMessage ();
							$self->AddError ("Cannot open archive directory \"".$filePath."\": $!");
							return undef;
						}
						@subdirFiles = readdir (DH);
						closedir (DH);
						@subdirFiles = grep {/\.tgz$/i} @subdirFiles;
						foreach my $subdirFile (sort @subdirFiles){
							if (! -r join($path_separator, $filePath, $subdirFile)) {
								$msg->endMessage ();
								$wrnLst->AddError ("Cannot read archive \"$subdirFile\"");
								$self->AddSubMsgLst ($msg, $wrnLst);
								$self->AddError (undef, $wrnLst);
								$wrnLst->initMsgLst ();
								return undef;
							}
							if (defined $filesToImport->{$subdirFile}) {
								$msg->endMessage ();
								$wrnLst->AddError ("Duplicate archive \"$subdirFile\"");
								$self->AddSubMsgLst ($msg, $wrnLst);
								$self->AddError (undef, $wrnLst);
								$wrnLst->initMsgLst ();
								return undef;
							} else {
								$filesToImport->{$subdirFile} = join($path_separator, $filePath, $subdirFile);
							}
						}
					}
				} elsif ($file =~ /\.tgz$/i) {
					if (! -r join($path_separator, $archive_dir, $file)) {
						$msg->endMessage ();
						$wrnLst->AddError ("Cannot read archive \"$file\"");
						$self->AddSubMsgLst ($msg, $wrnLst);
						$self->AddError (undef, $wrnLst);
						$wrnLst->initMsgLst ();
						return undef;
					}
					if (defined $filesToImport->{$file}) {
						$msg->endMessage ();
						$wrnLst->AddError ("Duplicate archive \"$file\"");
						$self->AddSubMsgLst ($msg, $wrnLst);
						$self->AddError (undef, $wrnLst);
						$wrnLst->initMsgLst ();
						return undef;
					} else {
						$filesToImport->{$file} = join($path_separator, $archive_dir, $file);
					}
				}
			}
			foreach my $fileToImport (sort keys (%$filesToImport)) {
				my $unit = new SDB::Install::DeliveryUnit($filesToImport->{$fileToImport});
				if (!defined $unit->init()){
					$msg->endMessage ();
					$wrnLst->AddError ("Cannot initialize archive \"$fileToImport\"", $unit);
					$self->AddSubMsgLst ($msg, $wrnLst);
					$self->AddError (undef, $wrnLst);
					$wrnLst->initMsgLst ();
					return undef;
				} else {
					$msglst->addMessage("Selected for import: $filesToImport->{$fileToImport}");
					push @auto_units, $unit;
				}
			}
		} else {
			@files = grep {/\.tgz$/i} @files;
			foreach my $file (sort @files){
				if (! -r join($path_separator, $archive_dir, $file)) {
					$msg->endMessage ();
					$wrnLst->AddError ("Cannot read archive \"$file\"");
					$self->AddSubMsgLst ($msg, $wrnLst);
					$self->AddError (undef, $wrnLst);
					$wrnLst->initMsgLst ();
					return undef;
				}
				my $unit = new SDB::Install::DeliveryUnit(join($path_separator, $archive_dir, $file));
				if (!defined $unit->init()){
					$msg->endMessage ();
					$wrnLst->AddError ("Cannot initialize archive \"$file\"", $unit);
					$self->AddSubMsgLst ($msg, $wrnLst);
					$self->AddError (undef, $wrnLst);
					$wrnLst->initMsgLst ();
					return undef;
				} else {
					$msglst->addMessage("Selected for import: ".join($path_separator, $archive_dir, $file));
					push @auto_units, $unit;
				}
			}
		} # end if (!$explicitPath) / else
	} # end if (defined $path && -f $path) / else

	my $secureConnectProperties;
    my $global_ini = $self->getGlobalIni();
    my $sslEnforce = $global_ini->getValue ('communication', 'sslEnforce');
    if ($sslEnforce eq 'true') {
        $secureConnectProperties = $self->getSqlSSLProperties ($msglst);
        if (!defined $secureConnectProperties){
            $self->setErrorMessage('Cannot get secure connect properties');
            return undef;
        }
    }
	# create list of delivery units in database
	if ($instconfig->isUpdate() || $isResume){
		my $sqlUser = $instconfig->getValue('SystemUser');
		$sqlUser //= 'SYSTEM';
		my $sqlPassword = $instconfig->getValue('SQLSysPasswd');
		my $connectionProvider = new SDB::Install::Sql::SqlConnectionProvider();
		$connectionProvider->setMsgLstContext([$msg->getSubMsgLst()]);
		my $sql = $connectionProvider->getSqlConnection(
					$self->getSQLDatabaseName(),
					$self,
					$sqlUser,
					$sqlPassword,
					$user_store_key);

		if (!defined $sql){
			$self->setErrorMessage('Cannot check already imported delivery units', $connectionProvider->getErrMsgLst());
			return undef;
		}

		if (!defined $sql->execute("SELECT DELIVERY_UNIT, VERSION, VERSION_SP, VERSION_PATCH FROM _SYS_REPO.DELIVERY_UNITS")){
			$msg->endMessage ();
			$self->AddError("Execute failed: " . $sql->GetErrorString());
			return undef;
		}
		my $result = $sql->getResultSet();
		if (!defined $result){
			$msg->endMessage ();
			$self->AddError("getResultSet failed: " . $sql->GetErrorString());
			return undef;
		}
		foreach my $row (@$result){
			$installed_units->{shift @$row} = $row;
		}
	} # end of: if ($instconfig->isUpdate())

	my $import_error = 0;
	my @deployed_units = ();
	my $current_delivery_unit = $instconfig->{'current_delivery_unit'} || '';
	# import automatic content
	foreach my $unit (@auto_units){
		my $name = $unit->getName();
		my $version = $unit->getVersion();
		my $sp = $unit->getSP();
		my $patch = $unit->getPatch();
		my $import_unit_failed = 0;
		my $fullVersion = new SDB::Install::Version($version, $sp, $patch);
		my $fullVersionInstalled;
 		# Import of DU should be skipped if it exists in the system view
		my $isUnitAlreadyInstalled;
		if ( $support_resume && $isResume ) {
			# $installed_units is not populated initially during install. Check if persisted in this case only
			my $isInSystemView = exists $installed_units->{$name};
			$isUnitAlreadyInstalled = $isInSystemView && ($current_delivery_unit ne $name);
		} else {
			$isUnitAlreadyInstalled = exists $installed_units->{$name};
		}
		if ($isUnitAlreadyInstalled){
			$fullVersionInstalled = new SDB::Install::Version(@{$installed_units->{$name}});
		}
		if (!$instconfig->isUpdate() && $isUnitAlreadyInstalled){
			$msglst->addMessage("Import of delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ") skipped, it is already imported.");
		} elsif ($instconfig->isUpdate() && $isUnitAlreadyInstalled &&
		    !$fullVersion->isNewerThan($fullVersionInstalled) &&
		    !$instconfig->getIgnore('check_version')
		) {
			$msglst->addMessage("Import of delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ") skipped, same or newer version " . $fullVersionInstalled->asString() . " already imported.");
		}
		else
		{
            if ($copy_autocontent_to_global) {
                # DU is located in plugin dir, needs to be copied to global auto_content dir
                my $source = $unit->getArchiveName();
                my $target = $self->get_globalAutoContentDir() . $path_separator . basename($source);
                my $fileCfg = {};
                if (!$isWin){
                    my $uid = $self->get_user()->uid();
                    my $gid = $self->get_user()->gid();
                    if (defined $uid && defined $gid){
                        $fileCfg->{uid} = $uid;
                        $fileCfg->{gid} = $gid;
                    }
                }
                my $copy_rc = copy_file($source, $target, $fileCfg);
                if (!defined $copy_rc) {
                    $msglst->addError ("Cannot copy DU archive $source to $target", $fileCfg);
                    return 0;
                } else {
                    $msglst->addMessage ("Copying DU archive $source to $target");
                }
            }
			if ($instconfig->getIgnore('check_version') && $isUnitAlreadyInstalled && !$fullVersion->isNewerThan($fullVersionInstalled)) {
				$msglst->addWarning("Importing delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ") even though same or newer version " . $fullVersionInstalled->asString() . " is already imported, due to switch ignore=check_version.");
			}
			my $importingMsg = $msglst->addMessage("Importing delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ")");
			$unit_counter += 1;
			$self->getMsgLst()->getProgressHandler()->SetProgress("Importing delivery unit " . $unit->getName());
			my $sqlUser = $instconfig->getValue('SystemUser');
			if (!$sqlUser) {
				$sqlUser = 'SYSTEM';
			}
			$unit->setMsgLstContext ([$importingMsg->getSubMsgLst()]);
			if ($support_resume){
				$instconfig->addCurrentDUToPersistenceFile($name);
			}
			if (!defined $unit->importContent($self, $sqlUser, $instconfig->getValue('SQLSysPasswd'), $user_store_key, $self->getDefaultPort(),$secureConnectProperties)){
				$self->AddError ("Cannot import delivery unit " . $unit->getArchiveName() . "");
				$import_error = 1;
				$import_unit_failed = 1;
				if (!$instconfig->isUpdate()){
					$msg->endMessage ();
					$self->analyzeTraceDirDeltas();
					return 0;
				}
			}
		}
		if($support_resume && !$import_unit_failed){
			push @deployed_units, $unit;
			$instconfig->addCurrentDUToPersistenceFile();
		}
		if (exists $installed_units->{$name}) {
			delete $installed_units->{$name};
		}
	}

	# import non-automatic content
	if ($instconfig->isUpdate() && !defined $path) {
		# non-automatic content installed?
		if (scalar keys %$installed_units) {

			# create list of non-automatic delivery units in file system
			my $archive_dir = $self->get_globalContentDir();
			unless (opendir (DH, $archive_dir)){
				$msg->endMessage ();
				$self->AddError ("Cannot open archive directory \"".$archive_dir."\": $!");
				return undef;
			}
			@files = readdir (DH);
			closedir (DH);
			@files = grep {/\.tgz$/i} @files;
			foreach my $file (sort @files){
				if (! -r join($path_separator, $archive_dir, $file)) {
					$msg->endMessage ();
					$wrnLst->AddError ("Cannot read archive \"$file\"");
					$self->AddSubMsgLst ($msg, $wrnLst);
					$self->AddError (undef, $wrnLst);
					$wrnLst->initMsgLst ();
					return undef;
				}
				my $unit = new SDB::Install::DeliveryUnit(join($path_separator, $archive_dir, $file));
				if (!defined $unit->init()){
					$wrnLst->addWarning ("Cannot initialize archive \"$file\"", $unit);
					$self->AddSubMsgLst ($msg, $wrnLst);
					$wrnLst->initMsgLst ();
				} else {
					$msglst->addMessage("Selected for import: ".join($path_separator, $archive_dir, $file));
					push @units, $unit;
				}
			}

			# import units
			foreach my $unit (@units){
				my $name = $unit->getName();
				my $version = $unit->getVersion();
				my $sp = $unit->getSP();
				my $patch = $unit->getPatch();
				my $import_unit_failed = 0;
				my $fullVersion = new SDB::Install::Version($version, $sp, $patch);
				my $isUnitAlreadyInstalled = !$support_resume || ($current_delivery_unit ne $name);
				if (exists $installed_units->{$name}) {
					my $fullVersionInstalled = new SDB::Install::Version(@{$installed_units->{$name}});
					if ($isUnitAlreadyInstalled) {
						$msglst->addMessage("Import of delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ") skipped, it is already imported.");
					} elsif (!$fullVersion->isNewerThan($fullVersionInstalled) && !$instconfig->getIgnore('check_version')) {
						$msglst->addMessage("Import of delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ") skipped, same or newer version " . $fullVersionInstalled->asString() . " already imported.");
					}
					else
					{
						if ($instconfig->getIgnore('check_version') && !$fullVersion->isNewerThan($fullVersionInstalled)) {
							$msglst->addWarning("Importing delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ") even though same or newer version " . $fullVersionInstalled->asString() . " is already imported, due to switch ignore=check_version.");
						}
						my $importingMsg = $msglst->addMessage("Importing delivery unit " . $unit->getArchiveName() . " ($name " . $fullVersion->asString() . ")");
						$unit_counter += 1;
						$self->getMsgLst()->getProgressHandler()->SetProgress("Importing delivery unit " . $unit->getName());
						my $sqlUser = $instconfig->getValue('SystemUser');
						if (!$sqlUser) {
							$sqlUser = 'SYSTEM';
						}
						$unit->setMsgLstContext ([$importingMsg->getSubMsgLst()]);
						if ($support_resume){
							$instconfig->addCurrentDUToPersistenceFile($name);
						}
						if (!defined $unit->importContent($self, $sqlUser, $instconfig->getValue('SQLSysPasswd'), $user_store_key, $self->getDefaultPort())){
							$self->AddError ("Cannot import delivery unit \"" . $unit->getArchiveName() . "\"");
							$import_error = 1;
							$import_unit_failed = 1;
							if (!$instconfig->isUpdate()){
								$msg->endMessage ();
								$self->analyzeTraceDirDeltas();
								return 0;
							}
						}
					}
				}
				if($support_resume && !$import_unit_failed){
					push @deployed_units, $unit;
					$instconfig->addCurrentDUToPersistenceFile();
				}
			}
		}
	} # end of: if ($instconfig->isUpdate() && !defined $path)

	if (!$unit_counter) {
		$self->getMsgLst()->getProgressHandler()->SetProgress("No delivery units to import");
		$wrnLst->AddMessage("No delivery units to import");
		$self->AddSubMsgLst ($msg, $wrnLst);
		$wrnLst->initMsgLst ();
	}
	$msg->endMessage ();

	if ($import_error){
		$self->analyzeTraceDirDeltas();
		return 0;
	}
	return 1;
}


sub getTenantDatabases{
    my ($self, $nocache) = @_;
    $nocache //= 0;
    if (!defined $self->{_mdcDatabases} || $nocache){
        my $mdc_file = $self->getMdcDatabasesFileName ();

        my $stat = File::stat::stat($mdc_file);
        if (!defined $stat){
            $self->getMsgLst ()->addMessage ("Cannot access mdc databases configuration file '$mdc_file': $!");
            return [];
        }
        my $file = new IO::File($mdc_file);
        if (!defined $file){
            $self->setErrorMessage ("Cannot open mdc databases configuration '$mdc_file': $!");
            return undef;
        }
        my @buffer = $file->getlines();
        $file->close();
        $self->{_mdcDatabases} = [];

        foreach my $line (@buffer){
            trim (\$line);
            next if $line =~ /^#/;
            my @tenantInfo = split(':', $line);
            my $tenant = SDB::Install::SAPInstance::TenantDatabase->new(\@tenantInfo);
            if (!$tenant->getName() || !defined($tenant->getUsername())){
                next;
            }
            push @{$self->{_mdcDatabases}}, $tenant;
        }
    }
    return $self->{_mdcDatabases};
}

sub tenantExists {
    my ($self, $tenantDB) = @_;
    my $tenantDatabases = $self->getTenantDatabases();
    my $tenants = grep { lc($_->getName()) eq lc($tenantDB) } @{$tenantDatabases};
    return ($tenants > 0) ? 1 : 0;
}


sub importPluginDeliveryUnits{
    my ($self, $instconfig) = @_;
    my $import_rc;
    my $rc = 1;
    my $pluginInstallations = $self->getPluginInstallations (1);
    if (defined $pluginInstallations && @$pluginInstallations){
        my $autocontentDir;
        my $pluginKey;
        my $msg;
        my $saveContext;
        foreach my $pluginInst (
            sort {$a->getProductKey () <=> $b->getProductKey ()} @$pluginInstallations){
            if ($isWin) {
                $autocontentDir = $self->get_pluginsDir() . $path_separator . $pluginInst->getProductKey() . $path_separator . 'auto_content';
            } else {
                $autocontentDir = $pluginInst->getProgramPath () . $path_separator . 'auto_content';
            }
            if (-d $autocontentDir){
                $pluginKey = $pluginInst->getProductKey ();
                $msg = $self->getMsgLst ()->addProgressMessage ("Importing delivery units of plugin '$pluginKey'...");
                $msg->setEndTag ("Import delivery units of plugin '$pluginKey'");
                $import_rc = $self->importDeliveryUnits($instconfig, $autocontentDir, 1, $msg);
                if (!defined $import_rc){
                    $pluginKey = $pluginInst->getProductKey ();
                    $self->appendErrorMessage ("Import of '$pluginKey' plugin delivery units failed", $self->getErrMsgLst ());
                    $rc = undef;
                }
                if (!$import_rc){
                    return 0;
                }
            }
        }
    }
    return $rc;
}




sub isSecondarySystem {
	my ($self, $msglst, $errlst) = @_;

	if (defined $self->{secondarySystem}) {
		return $self->{secondarySystem};
	}
	$msglst //= SDB::Install::MsgLst->new();
	$errlst //= SDB::Install::MsgLst->new();

	my $msg = $msglst->addMessage ("Performing secondary system check");

	my $mode = $self->getReplicationMode($msglst, $errlst) // $self->getReplicationModeByIniFile();
	if (!defined $mode){
		my $errmsg = $errlst->addError ("Could not detect whether this is a secondary system by using the hdbnameserver utility or from global.ini");
		$msglst->appendMsg ($errmsg->copy());
		return undef;
	}

	if (!($mode eq 'none' || $mode eq 'primary')) {
		$self->{secondarySystem} = 1;
		return 1;
	}
	$self->{secondarySystem} = 0;
	return 0;
}


sub getReplicationModeByIniFile{
    my ($self,$no_cache) = @_;
    if (defined $self->{replicationMode}) {
        return $self->{replicationMode};
    }

    my $global_ini = $self->getGlobalIni($self->getMsgLst(),$no_cache);
    if (!defined $global_ini){
        return undef;
    }
    my $replicationMode;
    my $sectionSysRepl = 'system_replication';
    my $keyActualMode  = 'actual_mode';

    if ($global_ini->existsValue ($sectionSysRepl, $keyActualMode)){
        $replicationMode = $global_ini->getValue ($sectionSysRepl, $keyActualMode);
    }

    if (!defined $replicationMode || $replicationMode eq ''){
        $replicationMode = 'none';
    }
    $self->{replicationMode} = $replicationMode;
    return $replicationMode;
}

# Works only when DB is offline!!!
sub getCurrentSecureStore {
    my ($self) = @_;

    if (defined $self->{_currentSecureStore}) {
        return $self->{_currentSecureStore};
    }
    my $msglst = $self->getMsgLst();
    my $msg = $msglst->addMessage('Checking current secure store setup');

    my $outputBuffer;
    my $rc = $self->runNameServerUtility(['-migrateSecureStore', '--target=LSS', '--dry-run'], $msg->getSubMsgLst(), \$outputBuffer);

    my $errMsg = "Failed to determine current secure store setup";
    if (!$rc) {
        $self->getErrMsgLst()->addError($errMsg);
        return undef;
    }

    my ($secureStoreSetup) = ($outputBuffer =~ /^current root key store:\s*(\S+)/m);
    if (defined $secureStoreSetup && ($secureStoreSetup =~ /LSS|SSFS/)) {
        $self->{_currentSecureStore} = $secureStoreSetup;
    } else {
        $self->getErrMsgLst()->addError($errMsg);
        return undef;
    }

    return $self->{_currentSecureStore};
}

sub getReplicationMode{
	my ($self, $msglst, $errlst) = @_;
	if (defined $self->{replicationMode}) {
		return $self->{replicationMode};
	}

	if (!defined $msglst) {
		$msglst = new SDB::Install::MsgLst();
	}
	if (!defined $errlst) {
		$errlst = new SDB::Install::MsgLst();
	}

	my $msg = $msglst->addMessage ("Checking system replication mode");
	my $execmsglst = $msg->getSubMsgLst ();
	my $output_buffer;
	my $rc = $self->runNameServerUtility (['-sr_state'],$execmsglst,\$output_buffer);
	if (!defined $rc){
		my $errmsg = $errlst->addError ("Cannot detect system replication mode");
		$msglst->appendMsg ($errmsg->copy());
		$errmsg->getSubMsgLst()->appendMsgLst ($execmsglst);
		return undef;
	}
	# check hdbnsutil output for 'mode' info
	my ($mode) = ($output_buffer =~ /^\s*mode\s*[:=]\s*(\S+)/m);
	if (defined $mode) {
		$self->{replicationMode} = $mode;
		$msglst->addMessage ("System replication mode is: $mode");
		return $mode;
	} else {
		my $errmsg = $errlst->addError ("Cannot detect system replication mode");
		$msglst->appendMsg ($errmsg->copy());
		return undef;
	}

}


sub runSapCpe{
	my ($self) = @_;

	my $path = $self->get_globalTrexExeDir . $path_separator;

	my $sapcpe = $path . 'sapcpe.exe';

	my $profile = join ($path_separator, $self->get_profileDir (),
				$self->get_profileName ());

	my $list = $path . 'hdbbin.lst';

	my $msg = $self->AddProgressMessage ("Copying binaries via sapcpe...");

	my $cfg = {};

	my $rc = exec_program ($sapcpe,["pf=$profile", "list:$list", 'stdoutput'],$cfg);

	if (!defined $rc){
		$self->AddError ("Error copying executables", $cfg);
		$self->AddSubMsgLst ($msg,$cfg);
		return undef;
	}

	if ($rc != 0){
		$self->AddError ("Error copying executables: sapcpe returned exit code '$rc'");
		$self->AddSubMsgLst ($msg,$cfg);
		return undef;
	}

	$self->AddSubMsgLst ($msg,$cfg);
	return 1;
}

sub runPythonScript{
    my ($self, $script, $args, $msglst, $ref_output_buffer, $ref_input_buffer, $additionalEnviroment, $outputHandler, $no_user_switch, $fSubstLogEntry, $wantExitCode) = @_;
    $script = File::Spec->rel2abs($script, $self->get_instanceExeDir());
    if (! -f $script){
        if ($!){
            $self->setErrorMessage ("Cannot start script '$script': $!");
        }
        else{
            $self->setErrorMessage ("Cannot start script. '$script' is no regular file");
        }
        return undef;
    }

    my $python;
    if ($self->isAddressSanitizerBuild ()){
        $python = join( $path_separator, 'python');
    }
    elsif ($isWin) {
        $python = join( $path_separator, 'Python', 'python.exe');
    }
    else{
        $python = join( $path_separator, 'Python', 'bin', 'python');
    }

    if (!defined $args){
        $args = [];
    }

    unshift @$args, $script;

    if (!defined $outputHandler){
        require SDB::Install::OutputHandler;
        $outputHandler = new SDB::Install::OutputHandler (
            undef,
            '  ' . basename ($script) . ': '
            );
    }

    return $self->runUtilityInHDBEnv ($python, $args, $msglst, $ref_output_buffer, $ref_input_buffer, $additionalEnviroment, $outputHandler, $no_user_switch, $fSubstLogEntry, undef, $wantExitCode);
}

sub stopAllHosts{
	my ($self,
	    $userName,
	    $password,
	    $ignore_not_running_nodes,
	    $timeout,
	    $useHttps, $sso_cert) = @_;

	if (!$self->exists_remote_host()){
		return $self->stopHost ($userName,
		                        $password,
		                        undef, # default is local host
		                        $timeout,
		                        $useHttps, $sso_cert);
	}
	return $self->stopHosts ($userName,
	                         $password,
	                         $self->get_allhosts(),
	                         $ignore_not_running_nodes,
	                         $timeout,
	                         $useHttps, $sso_cert);
}


sub stopHosts{
	my ($self,
	    $userName,
	    $password,
	    $hosts,
	    $ignore_not_running_nodes,
	    $timeout,
	    $useHttps,
	    $sso_cert) = @_;

	my $masters = $self->getMasters();
	my $master;
	if (!defined $masters){
		return undef;
	}
	if (@$masters){
		$master = $masters->[0];
	}

	my $nr = $self->{_nr};
	my @nonMasterHosts;
	my $host;
	my $masterIsInList = 0;
	my $hostMapRoles   = $self->getHostRolesInfo();
	foreach $host (@$hosts){
		if ($host ne $master){
			push @nonMasterHosts, $host;
		}
		else{
			$masterIsInList = 1;
		}
	}

	if (!$masterIsInList){
		$master = undef;
	}

	my ($msg,$rc, $i);

	my $result = 1;

	my (@sapcontrols,@msgs,$sapcontrol, @infoHosts);

    if (!defined $timeout){
        # default stop timeout
        $timeout = 800;
    }
    my $stop_wait_delay = 2;
    my $stop_wait_retries = int ($timeout/ $stop_wait_delay);
    my $stop_wait_retries_plus_60 = int (($timeout + 60)/ $stop_wait_delay);

	foreach $host (@nonMasterHosts){
		$sapcontrol = new SDB::Install::SAPControl (
				$self->_getSAPControlHost ($host),
				$nr, $userName, $password, $useHttps, $sso_cert);
		$sapcontrol->set_callback($self->{_f_callback});
		if ($ignore_not_running_nodes && !$sapcontrol->isServiceRunning()){
			push @msgs, $self->AddMessage ("Ignore not running host '$host'", $sapcontrol);
			next;
		}
		push @msgs, $self->AddMessage ("Stopping $host, parameters: instance number = $nr, user = $userName");
		push @sapcontrols, $sapcontrol;

        $self->takeTraceDirSnapshot($host, $msgs[$#msgs]->getSubMsgLst());

	}


	foreach $i (0 .. $#sapcontrols){
		$infoHosts[$i] = $self->getStartStopHostInfo($hostMapRoles,
		                                             $sapcontrols[$i]->{_host});
		if (defined $self->{_f_callback}){
			$sapcontrols[$i]->set_callback ($self->{_f_callback});
		}
		$sapcontrols[$i]->setMsgLstContext ([$msgs[$i]->getSubMsgLst()]);
		if (!defined $sapcontrols[$i]->Stop ($timeout, 3)){
			$self->PushError ("Stop instance $sapcontrols[$i]->{_nr} on host $infoHosts[$i] failed.", $sapcontrols[$i]);
			$sapcontrols[$i] = undef;
			$result = undef;
			$self->analyzeTraceDirDeltas($nonMasterHosts[$i]);
		}
	}

	foreach  $i (0 .. $#sapcontrols){
		if (!defined $sapcontrols[$i]){
			next;
		}
		my $pHandler = $self->getMsgLst()->getProgressHandler();
		$sapcontrols[$i]->getMsgLst()->setProgressHandler($pHandler);
		if (!defined $sapcontrols[$i]->WaitforStopped ($stop_wait_retries_plus_60, $stop_wait_delay, undef, $infoHosts[$i])){
			$self->PushError ("Stop instance $sapcontrols[$i]->{_nr} on host $sapcontrols[$i]->{_host} failed.", $sapcontrols[$i]);
			$result = undef;
			$self->analyzeTraceDirDeltas($nonMasterHosts[$i]);
		}
	}

	if ($master){
		$host = $master;
		my $hostInfo = $self->getStartStopHostInfo($hostMapRoles, $host);
		$msg = $self->AddMessage ("Stopping master host $hostInfo, parameters: instance number = $nr, user = $userName");
		$self->takeTraceDirSnapshot($host,$msg->getSubMsgLst());
		$sapcontrol = new SDB::Install::SAPControl (
				$self->_getSAPControlHost ($host),
				$nr, $userName, $password, $useHttps, $sso_cert);

		if (defined $self->{_f_callback}){
			$sapcontrol->set_callback ($self->{_f_callback});
		}

    	my $pHandler = $self->getMsgLst()->getProgressHandler();
    	$sapcontrol->getMsgLst()->setProgressHandler($pHandler);
		if (!defined $sapcontrol->StopWait ($stop_wait_retries, $stop_wait_delay, $hostInfo)){
			$self->PushError ("Stop instance $nr on master host $hostInfo failed.",$sapcontrol);
			$self->analyzeTraceDirDeltas($host);
			$result = undef;
		}
		$self->AddSubMsgLst ($msg, $sapcontrol);
		$self->AddMessage ("Master stopped");
	}

	return $result;
}

sub startAllHosts{
	my ($self, $userName, $password, $timeout, $useHttps, $sso_cert) = @_;

	if (!$self->exists_remote_host()){
		return $self->startHost ($userName,
		                         $password, # used for Windows only
		                         undef,     # localhost
		                         $timeout,
		                         $useHttps, $sso_cert);
	}
	return $self->startHosts ($userName,
	                          $password,
	                          $self->get_allhosts(),
	                          $timeout,
	                          $useHttps, $sso_cert);
}


sub startHosts{
	my ($self, $userName, $password, $hosts, $timeout, $useHttps, $sso_cert) = @_;

	my $masters = $self->getMasters();
	my $master;
	if (!defined $masters){
		return undef;
	}
	if (@$masters){
		$master = $masters->[0];
	}

	my $nr = $self->{_nr};
	my @nonMasterHosts;
	my $host;
	my $masterIsInList = 0;
	my $hostMapRoles   = $self->getHostRolesInfo();
	foreach $host (@$hosts){
		if ($host ne $master){
			push @nonMasterHosts, $host;
		}
		else{
			$masterIsInList = 1;
		}
	}

	if (!$masterIsInList){
		$master = undef;
	}

	my ($msg, $sapcontrol);
    my ($start_wait_retries, $start_wait_delay);
    if (defined $timeout){
        $start_wait_delay = 2;
        $start_wait_retries = int ($timeout / $start_wait_delay);
    }
	if (defined $master){
		$host = $master;
		my $hostInfo = $self->getStartStopHostInfo($hostMapRoles, $host);
		$msg = $self->AddMessage ("Starting master host $hostInfo, parameters: instance number = $nr, user = $userName");
		$self->takeTraceDirSnapshot($host, $msg->getSubMsgLst());
		$sapcontrol = new SDB::Install::SAPControl (
				$self->_getSAPControlHost ($host),
				$nr, $userName, $password, $useHttps, $sso_cert);

		$sapcontrol->setMsgLstContext ([$msg->getSubMsgLst ()]);

		if (defined $self->{_f_callback}){
			$sapcontrol->set_callback ($self->{_f_callback});
		}
		if (!defined $sapcontrol->StartWait ($start_wait_retries, $start_wait_delay, $hostInfo)){
			$self->AddError ("Start instance $nr on master host $hostInfo failed.", $sapcontrol);
			$self->analyzeTraceDirDeltas($host);
			return undef;
		}
		else{
			$self->AddMessage ("Master host $hostInfo started");
		}
	}

    $self->onMasterStarted ();

	my $rc = 1;
	my ($i, @sapcontrols, @infoHosts);
	foreach $host (@nonMasterHosts){
		$msg = $self->AddMessage ("Starting $host, parameters: instance number = $nr, user = $userName");
		$sapcontrol = new SDB::Install::SAPControl (
			$self->_getSAPControlHost ($host),
			$nr, $userName, $password, $useHttps, $sso_cert);
		$sapcontrol->setMsgLstContext ([$msg->getSubMsgLst()]);
		push @sapcontrols, $sapcontrol;
		$self->takeTraceDirSnapshot($host,$msg->getSubMsgLst());

        $sapcontrols[$#sapcontrols]->set_callback($self->{_f_callback});
	}

	foreach $i (0 .. $#sapcontrols){
		$infoHosts[$i] = $self->getStartStopHostInfo($hostMapRoles,
		                                             $sapcontrols[$i]->{_host});
		if (defined $self->{_f_callback}){
			$sapcontrols[$i]->set_callback ($self->{_f_callback});
		}
		if (!defined $sapcontrols[$i]->Start ()){
			$self->PushError ("Start instance $sapcontrols[$i]->{_nr} on host $infoHosts[$i] failed.",$sapcontrols[$i]);
			$sapcontrols[$i] = undef;
			$rc = undef;
			$self->analyzeTraceDirDeltas($nonMasterHosts[$i]);
		}
	}

	foreach $i (0 .. $#sapcontrols){
		if (!defined $sapcontrols[$i]){
			next;
		}
		if (!defined $sapcontrols[$i]->WaitforStarted ($start_wait_retries, $start_wait_delay, $infoHosts[$i])){
			$self->PushError ("Start instance $sapcontrols[$i]->{_nr} on host $infoHosts[$i] failed. Has to be restarted manually.",$sapcontrols[$i]);
			$rc = undef;
			$self->analyzeTraceDirDeltas($nonMasterHosts[$i]);
		}
	}

	return $rc;
}

sub killtrex{
	my ($self, $inst) = @_;

    if ($isLinux){
        my $hangChecker = new SDB::Install::System::FuserHangChecker();
        $hangChecker->setMsgLstContext($self->getMsgLstContext());
        my $rc = $hangChecker->check();
        if (defined $rc && $rc == 0){
            return undef;
        }
    }

	my $ps = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_MODULES);

	if (!$isWin){

        my %killed;
        my $msg = $self->AddMessage ("Killing processes with loaded modules of this installation");
        my $msgt = 'Killing process \'%s\', pid = %d';
        my $cmdLine;
        foreach my $try (0..1){ # search twice to get rid of automatically restarted services
            if ($try){
                # get a new snapshot
                $ps = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_MODULES);
            }

            my ($procs,$rc,$errmsg) = $inst->getFilesInUseByPID ($ps);
            if ($rc){

                #
                # find the sapstartsrv and daemon pids to kill them first
                #
                my @pids = @{$self->getSortedTrexKillPids($procs)};
                #
                # kill processes
                #
                %killed = ();
                #
                # find children, that are not found via fuser
                #

                $ps->getTree ();
                my ($children,$child, @add_children);
                foreach my $pid (@pids){
                    if ($pid == $$){
                        next; # skip own children
                    }
                    $children = $ps->getChildren ($pid, 1);
                    if (!defined $children){
                        next;
                    }
                    foreach $child (@$children){
                        if (!exists $procs->{$child}){
                            push @add_children, $child;
                        }
                    }
                }

                if (@add_children){
                    push @pids, @add_children;
                }

                foreach my $pid (@pids){
                    if ($pid == $$){
                        next;
                    }
                    if (exists $killed{$pid}){
                        next;
                    }
                    $cmdLine = defined $procs->{$pid} ?
                        $procs->{$pid}->{commandLine} :
                        $ps->GetArgs ($pid);
                    $self->AddSubMessage ($msg, sprintf ($msgt, $cmdLine, $pid));
                    my $pgid=-$pid;
                    if ($ps->Terminate ($pgid, $ps->GetUid ($pid), $cmdLine) != 0){
                        $self->PushError ("Killing process '$cmdLine', pid = $pid failed");
                        $rc = undef;
                    }
                    $killed{$pid} = undef;
                }

                if (!$try){
                    next; #skipping detection of other processes in first iteration
                }

                #
                # log other processes owned by sidadm for better analysis in qa infrastructure
                #

                @pids = ();
                my $sysAdmUID = $self->{_user}->uidUx();

                foreach my $proc (@{$ps->{procs}}){
                    if ($proc->{uid} != $sysAdmUID){
                        next;
                    }

                    my $pid = $proc->{pid};

                    if (exists $killed{$pid}) {
                        next;
                    }

                    if ($self->tryKillProcessRestartHandover($ps, $pid, $msg)) {
                        $killed{$pid} = undef;
                    }
                    else {
                        push @pids, $pid;
                    }
                }
                if (@pids){
                    $msg = $self->AddMessage ("Found other processes owned by sidadm");
                    foreach my $pid (@pids){
                        $self->AddSubMessage ($msg, "'" . $ps->GetArgs($pid) ."', pid = $pid");
                    }
                }
            }
            else{
                $self->AddError ("Cannot get pids : $errmsg");
                return undef;
            }
        }

        $self->cleanipc();
		if (!$inst->get_integratedSystem()){
			require SDB::Install::System::IpcCleaner;
			my $ipcCleaner = new SDB::Install::System::IpcCleaner();
			$ipcCleaner->setMsgLstContext($self->getMsgLstContext);
			$ipcCleaner->cleanIpcResources($self->{_user}->getSidAdmName());
		}

		my $lockdir = $self->get_lockDir();

		if (opendir (DH,$lockdir)){
			my $pattern = sprintf '^\w+@3' . $self->{_nr}. '\d\d\.pid$';
			my @files = grep {/$pattern/} readdir (DH);
			closedir (DH);
			my $file;
			foreach my $i (0..$#files){
				$file = $lockdir . $path_separator . $files[$i];
				if (!unlink ($file)){
					$self->AddWarning ("Cannot delete pid file '$file': $!");
				}
			}
		}
		else{
			$self->AddWarning ("Cannot open directory '$lockdir': $!");
		}
		unlink (sprintf ($sapstartsrv_log_template, int $self->{_nr}));
		unlink (sprintf ($sapstartsrv_stream_template, int $self->{_nr}));
		unlink ("/tmp/.hdb_$self->{_sid}_$self->{_nr}_lock");
	}
	else{
		my $sid = $self->{_user}->{serviceuser}->{sid};
		my $msg = $self->AddMessage ("Killing processes owned by '$self->{_user}->{serviceuser}->{name}'");
		my $msgt = 'Killing process \'%s\', pid = %d';
		my $pid;
		my $rc = 1;
		my $procFound = 0;
		foreach my $proc (@{$ps->{procs}}){
			$pid = $proc->{pid};
			if ($sid eq $ps->GetSID ($pid)){
				$procFound = 1;
				my $cmd = $ps->GetCmd ($pid);
				$self->AddSubMessage ($msg, sprintf ($msgt, $cmd, $pid));
				if ($ps->Terminate ($pid) != 0){
					$self->PushError ("Killing process '$cmd', pid = $pid failed");
					$rc = undef;
				}
			}
		}
		if (!$procFound){
			$self->AddSubMessage ($msg, "No process found");
		}
		return $rc;
	}
}



#-------------------------------------------------------------------------------
# Terminates the specified process if the process is used to speed restart.
#
# Returns 1 if terminated otherise 0
#
# Sample process:
#    python -m attachShm attach -n 30703 -s /tmp/hdbindexserver-30703-rowstore-fast-restart-handover -t 3600

sub tryKillProcessRestartHandover {

    my ($self, $procState, $pid, $msg) = @_;

    my $isTerminated = 0;
    my $processInfo  = $procState->GetArgs($pid);

    if ($processInfo =~ /$processRestartHandoverPattern/){
        my $info = "Killing process '".$processInfo."', pid = $pid";
        $self->AddSubMessage ($msg, $info);
        if ($procState->Terminate ($pid, $procState->GetUid ($pid), $processInfo) != 0){
            $self->PushError ("$info failed");
        }
        else {
            $isTerminated = 1;
        }
    }
    return $isTerminated;
}


#-------------------------------------------------------------------------------
# Terminates the specified process if the process
# runs an executable under '<sapmnt>/<SID>/xs'.
#
# Returns 1 if terminated otherise 0

sub tryKillProcessXS2 {

    my ($self, $procState, $pid, $msg) = @_;

    my $isTerminated = 0;
    my $xs2Dir       = $self->getXS2Dir();

    if (!defined $xs2Dir) {
        return $isTerminated;
    }

    my $processInfo    = $procState->GetArgs($pid);
    my $xs2PathPattern = $xs2Dir . $path_separator;

    if ($processInfo =~ m#$xs2PathPattern#) {
        my $info = "Killing process '".$processInfo."', pid = $pid";
        $self->AddSubMessage ($msg, $info);
        if ($procState->Terminate ($pid, $procState->GetUid ($pid), $processInfo) != 0){
            $self->PushError ("$info failed");
        }
        else {
            $isTerminated = 1;
        }
    }
    return $isTerminated;
}


#-------------------------------------------------------------------------------
# Scans the processes and kills the process that is provided to speed up restart.

sub scanProcessesKillRestartHandover {

    my ($self, $msg) = @_;

    if ($isWin) {
        return 1;
    }

    my $ps        = new SAPDB::Install::ProcState ();
    my $sysAdmUID = $self->{_user}->uidUx();
    my $xs2Exists = $self->existsXS2Dir();

    foreach my $proc (@{$ps->{procs}}) {

        if ($proc->{uid} == $sysAdmUID) {
            my $terminated =
                $self->tryKillProcessRestartHandover($ps, $proc->{pid}, $msg);
            if (!$terminated && $xs2Exists) {
                    $self->tryKillProcessXS2($ps, $proc->{pid}, $msg);
            }
        }
    }
    return 1;
}


sub cleanipc{
    my ($self, $msglst) = @_;
    my $cleanipc = $self->get_instanceExeDir () . $path_separator . 'cleanipc';

    if (!defined $msglst){
        $msglst = $self;
    }

    my $msg = $msglst->AddMessage ("Cleaning IPC resources...");

    if (-x $cleanipc){
        my $msgl = {};
        local %ENV = %ENV;
        $ENV{LD_LIBRARY_PATH} = join (':',$self->get_instanceExeDir (),
            defined $ENV{LD_LIBRARY_PATH} ? $ENV{LD_LIBRARY_PATH} : ());
        my $rc = exec_program ($cleanipc,[$self->{_nr}, 'remove'], $msgl);
        if (!defined $rc || $rc){
            $msglst->AddError ("cleanipc failed", $msgl);
            $msglst->AddSubMsgLst ($msg, $msgl);
            return undef;
        }
        $msglst->AddSubMsgLst ($msg, $msgl);
    }
    else{
        $msglst->AddError ("cleanipc ($cleanipc) not found");
        return undef;
    }
    return 1;
}


sub getIndexServerPidFile{
	return $_[0]->get_lockDir () . $path_separator .
		sprintf ('hdbindexserver@3%02d03.pid', int $_[0]->{_nr});
}

sub getSqlHost {
    my ($self) = @_;
    my $sqlHost = $self->getActiveMaster();
    my $isHostPartOfSystem = $sqlHost ? grep { $_ eq $sqlHost } @{$self->get_allhosts()} : 0;

    if (!$isHostPartOfSystem) {
# It is possible that the active master is not part of the system.
# For example after rename with --nostart. The value in the nameserver.ini file
# will be updated when the system is started. Hence, a check is needed whether the host
# is part of the system. If not -> just return the local host
        $sqlHost = $self->get_host ();
    }
    return $sqlHost;
}


sub getSystemServerPidFile{
    return $_[0]->get_lockDir () . $path_separator .
        sprintf ('hdbnameserver@3%02d01.pid', int $_[0]->{_nr});
}

sub getIndexServerPid{
	my ($self, $pidfile, $msglst) = @_;

	if (!defined $msglst){
		$msglst = $self;
	}


	if (!defined $pidfile){
		$pidfile = $self->getIndexServerPidFile;
	}
	if (!open (FD, $pidfile)){
		return undef;
	}
	my ($pid) = int <FD>;
	close (FD);

	if ($pid){
		my $ps = new SAPDB::Install::ProcState ();
		my $args;
		foreach my $proc (@{$ps->{procs}}){
			if ($proc->{pid} == $pid){
				$args = $ps->GetArgs($pid);
				if (($args =~ /hdbindexserver/)){
					$msglst->AddMessage ("IndexServer is alive: pid = $pid, args = $args");
					return $pid;
				}
			}
		}
	}
	return undef;
}

sub getSystemServerPid{
    my ($self, $pidfile, $msglst) = @_;

    if (!defined $msglst){
        $msglst = $self;
    }


    if (!defined $pidfile){
        $pidfile = $self->getSystemServerPidFile;
    }
    if (!open (FD, $pidfile)){
        return undef;
    }
    my ($pid) = int <FD>;
    close (FD);

    if ($pid){
        my $ps = new SAPDB::Install::ProcState ();
        my $args;
        foreach my $proc (@{$ps->{procs}}){
            if ($proc->{pid} == $pid){
                $args = $ps->GetArgs($pid);
                if (($args =~ /hdbnameserver/)){
                    $msglst->AddMessage ("NameServer is alive: pid = $pid, args = $args");
                    return $pid;
                }
            }
        }
    }
    return undef;
}

sub setSystemDBPassword{
    my ($self, $password) = @_;
    return $self->runUtilityInHDBEnv('hdbnameserver', ['-resetUserSystem'],
                    undef, undef, [$password], {'HDB_IPMM' => 'd'},
                    new SDB::Install::OutputHandler::HdbnameserverOutHndlr (undef, '  hdbnameserver: '));
}


our $createDatabaseStmnt             = 'create database %s at \'%s:%s\' SYSTEM user PASSWORD "%s"';
our $createDatabaseStmnt_defaultPort = 'create database %s SYSTEM user PASSWORD "%s"';


sub createInitialTenant{
    my ($self, $systemUser, $systemPassword, $password, $timeout, $port) = @_;

    if (!$self->isMultiDb()){
        $self->setErrorMessage ("Database is no MDC system");
        return undef;
    }

    my $escapedPassword = $password;
    $escapedPassword =~ s/\"/\"\"/g;
    my $msg = $self->getMsgLst()->addMessage("Connecting to system database");
    my $stmnt;
    if(defined $port) {
        $stmnt = sprintf($createDatabaseStmnt, $self->get_sid(), $self->get_host(), $port, $escapedPassword);
    }
    else {
        $stmnt = sprintf($createDatabaseStmnt_defaultPort, $self->get_sid(), $escapedPassword);
    }
    my $submsglst = $msg->getSubMsgLst();
    my $connectionProvider = new SDB::Install::Sql::SqlConnectionProvider();
    $connectionProvider->setMsgLstContext([$submsglst]);
    $connectionProvider->{_f_callback} = $self->{_f_callback};
    my $sql = $connectionProvider->waitForSqlConnection(
                'SYSTEMDB',
                $self,
                $systemUser,
                $systemPassword,
                undef, # user store key
                undef, # connect properties
                $timeout);

    if (!defined $sql){
        $self->setErrorMessage('Cannot create initial tenant', $connectionProvider->getErrMsgLst());
        return undef;
    }
    my $logstmnt;
    if(defined $port) {
        $logstmnt = sprintf($createDatabaseStmnt, $self->get_sid(), $self->get_host(), $port, '***');
    }
    else {
        $logstmnt = sprintf ($createDatabaseStmnt_defaultPort, $self->get_sid(), '***');
    }
    $msg = $submsglst->AddMessage ("Performing SQL statement: '$logstmnt'");
    $sql->setMsgLstContext ([$msg->getSubMsgLst()]);
    if (!defined $sql->execute ($stmnt)){
        $self->setErrorMessage ("Cannot perform sql statement:", $sql->getErrMsgLst());
        return undef;
    }
    return 1;
}

sub setEnginePassword{
	my ($self, $password, $old_password, $databaseUser, $initialTenant) = @_;
	my $username = (defined $databaseUser) ? $databaseUser : 'SYSTEM';
	if (!defined $initialTenant){
		$initialTenant = 0;
	}

	my $isMultiDB = $self->isMultiDb ();
	my $useSystemDB = $isMultiDB && !$initialTenant;

	my $label = $initialTenant ? ' (initital tenant database)' :
				$useSystemDB ? ' (system database)' : '';

	my $msg = $self->AddMessage ("Setting password of database user $username" . $label);
	$self->SetFormatInfo ($msg,'h1', "Set password of database user $username" . $label);
	my $errText = "Cannot set password of database user $username" . $label;
	my $envTimeout = $ENV{HDB_INSTALLER_TIMEOUT};
	my $timeout = (defined $envTimeout && ($envTimeout =~ /^\d+$/)) ? int ($envTimeout) : 900;
	my $submsglst = $msg->getSubMsgLst();
	my $connectionProvider = new SDB::Install::Sql::SqlConnectionProvider();
	$connectionProvider->setMsgLstContext([$submsglst]);
	$connectionProvider->{_f_callback} = $self->{_f_callback};
	$old_password //= 'manager';
	my $sql = $connectionProvider->waitForSqlConnection(
				$self->getSQLDatabaseName ($initialTenant),
				$self,
				$databaseUser,
				$old_password,
				undef, # user store key
				undef, # connect properties
				$timeout);

	if (!defined $sql && !$connectionProvider->wasATimeout()){
		$sql = $connectionProvider->getSqlConnection(
			$self->getSQLDatabaseName($initialTenant),
			$self,
			$databaseUser,
			$password);
		if (defined $sql){
			# connect via new password works
			# => password is already changed
			return 1;
		}
	}

	if (!defined $sql){
		$self->setErrorMessage($errText, $connectionProvider->getErrMsgLst());
		return undef;
	}
	$submsglst->AddMessage ("Performing SQL statement");
	my $escapedPassword = $password;
	$escapedPassword =~ s/\"/\"\"/g;
	if (!defined $sql->execute (
		"ALTER USER $username PASSWORD \"$escapedPassword\"")){
		$self->AddError ($errText, $sql);
		return undef;
	}
	return 1;
}

sub setInternalNetworkPrefix{
    my ($self,$prefix,$global_ini) = @_;
    delete $self->{_internNetworkPrefix};
    my $msg = $self->getMsgLst()->addMessage ("Setting internal network prefix");
    $global_ini->setMsgLstContext ([$msg->getSubMsgLst()]);
    $global_ini->setValue (CFG_LAYER_SYSTEM,'communication','internal_network',$prefix);
    return 1;
}

sub clearInternalNetworkPrefix{
    my ($self,$global_ini) = @_;
    delete $self->{_internNetworkPrefix};
    my $msg = $self->getMsgLst()->addMessage ("Removing internal network prefix");
    $global_ini->setMsgLstContext ([$msg->getSubMsgLst()]);
    $global_ini->removeKey(CFG_LAYER_SYSTEM, 'communication','internal_network');
    return 1;
}

sub getInternalNetworkPrefix {
    my ($self) = @_;

    if (defined $self->{_internNetworkPrefix}) {
        return ($self->{_internNetworkPrefix} eq '0')
               ? undef
               : $self->{_internNetworkPrefix};
    }

    my $global_ini = $self->getGlobalIni();

    return undef if (!defined $global_ini);

    if ($global_ini->existsValue('communication', 'internal_network')) {
        $self->{_internNetworkPrefix} =
                $global_ini->getValue('communication','internal_network');
        return $self->{_internNetworkPrefix};
    }

    my $networkInterfaces = SDB::Install::SysInfo::interfaceInfo($self->getErrMsgLst()) || {};
    my $ipAddressList = [ map { @{$_} } values(%{$networkInterfaces}) ];

    for my $ipAddressHash (@{$ipAddressList}){
        next if(!$global_ini->existsValue('internal_hostname_resolution', $ipAddressHash->{addr}));

        my $ipAddressString = sprintf('%s/%s', $ipAddressHash->{addr}, $ipAddressHash->{netbits});
        my $ipAddressObject = new SDB::Install::System::IPAddr($ipAddressString);
        if (defined $ipAddressObject) {
            $self->{_internNetworkPrefix} = $ipAddressObject->getPrefix()->print();
            return $self->{_internNetworkPrefix};
        }
    }
    $self->{_internNetworkPrefix} = '0'; # does not exist
    return undef;
}

sub getTrexNetListenInterface{
    my ($self,$prefix) = @_;
    my $global_ini = $self->getGlobalIni();
    if (!defined $global_ini){
        return undef;
    }
    return $global_ini->getValue ('communication','listeninterface');
}


sub isUsingInternalInterface{
    return ($_[0]->getTrexNetListenInterface () eq '.internal');
}

sub isUsingLocalInterface{
    return ($_[0]->getTrexNetListenInterface () eq '.local');
}

sub isUsingGlobalInterface{
    return ($_[0]->getTrexNetListenInterface () eq '.global');
}

sub resetInternalHostNameResolution{
    my ($self) = @_;
    my $global_ini = $self->getGlobalIni();
    if (!defined $global_ini){
        return undef;
    }
    my $msg = $self->getMsgLst()->addMessage ("Resetting internal hostname resolution");
    $global_ini->setMsgLstContext ([$msg->getSubMsgLst()]);
    my $internalIPs = $global_ini->getSectionKeys ('internal_hostname_resolution');
    if (!defined $internalIPs){
        return 1;
    }
    foreach my $ip (@$internalIPs){
        $global_ini->removeKey (CFG_LAYER_SYSTEM, 'internal_hostname_resolution', $ip);
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Returns the landscape ID containing read from file 'nameserver.ini'.

sub getLandscapeID {
    my ($self) = @_;
    return $self->getLandscapeValue('id');
}


#-------------------------------------------------------------------------------
# Returns the specified value of 'landscape' read from file 'nameserver.ini'.
#
# Parameter: $valueName string (e.g. 'id', 'master', etc.)
#            $errlst    SDB::Install::MsgLst (may be undef)

sub getLandscapeValue {

    my ($self, $valueName, $errlst) = @_;

    $errlst = new SDB::Install::MsgLst if (!defined $errlst);

    my $layered_cfg = $self->getLayeredConfig(1);
    if (!defined $layered_cfg) {
        $errlst->addError('Cannot get layered configuration');
        return undef;
    }
    $layered_cfg->setMsgLstContext([$self->getMsgLst ()]);

    my $ns_ini = $layered_cfg->getIniFile('nameserver.ini');
    if (!defined $ns_ini){
        $errlst->addError('Cannot get nameserver.ini',
                          $layered_cfg->getErrMsgLst());
        return undef;
    }
    $ns_ini->setMsgLstContext([$self->getMsgLst ()]);
    if (!defined $ns_ini->readValues ()){
        $errlst->addError("Cannot read nameserver.ini", $ns_ini->getErrMsgLst());
        return undef;
    }
    return $ns_ini->getValue('landscape', $valueName);
}


sub _readCustomNameServerIni{
    my ($self) = @_;
    my $nameserver_ini = $self->get_globalHdbCustomConfigDir () .
        $path_separator . 'nameserver.ini';

    if (-f $nameserver_ini){
        my $errlst = new SDB::Install::MsgLst ();
        my $data = readini ($nameserver_ini, $errlst);
        if (!defined $data){
            $self->AddError ('Cannot read nameserver.ini', $errlst);
            return undef;
        }
        return $data;
    }
    return {};
}

#-------------------------------------------------------------------------------
sub getManifest{
    my ($self, $no_cache) = @_;
    if ($no_cache || !defined $self->{_mf}){
        if (defined $self->{_addressSanitizerBuild}){
            $self->{_addressSanitizerBuild} = undef;
        }
        my $mf = $self->getManifestPath();
        if (-f $mf) {
            $self->{_mf} = new SDB::Install::Manifest($mf);
        }
    }
    return $self->{_mf};
}

sub getManifestPath {
    my ($self) = @_;
    return File::Spec->catfile($self->get_instanceExeDir(), 'manifest');
}

sub getPDXMLPath {
    my ($self) = @_;
    return File::Spec->catfile($self->get_instanceExeDir(), 'PD.XML');
}

#-------------------------------------------------------------------------------
sub getMasters{
	my ($self, $no_cache) = @_;

	if (defined $self->{_masters} && !$no_cache){
		return $self->{_masters};
	}

	my @masters;
	my $i = 0;
	my $data = $self->_readCustomNameServerIni ();
	if (!defined $data){
		return undef;
	}
	my $strAllMasters;
	if (defined $data->{landscape}){
		$strAllMasters = $data->{landscape}->{master};
	}

	if (!$strAllMasters){
		my $profile =  new SDB::Install::IniParser::SAPProfile (
			$self->get_hostNameDir() .$path_separator . 'sapprofile.ini');

		if (!defined $profile->read ()){
			$self->AddError ("Cannot read profile", $profile);
			return undef;
		}
		$strAllMasters = $profile->getValue ('HDB/NameServer/Master');
	}

	$strAllMasters =~ s/^\s*//;
	$strAllMasters =~ s/\s*$//;

	foreach my $master (split(' ', $strAllMasters)){
		push @masters, (split(':', $master))[0];
	}
	$self->{_masters} = \@masters;
	return $self->{_masters};
}

sub getActiveMaster{
    my ($self) = @_;
    my $data = $self->_readCustomNameServerIni ();
    if (defined $data && defined $data->{landscape}){
        my $active_master = $data->{landscape}->{active_master};
        $active_master = (split (':', $active_master))[0];
        if ($active_master){
            return $active_master;
        }
    }
    return undef;
}

sub hostIsActiveMaster{
    my ($self, $host) = @_;
    if (!defined $host){
        $host = $self->{_host};
    }
    my $activeMaster = $self->getActiveMaster ();
    if (!defined $activeMaster){
        return undef;
    }
    return $activeMaster eq $host;
}

sub getHostRemoveStatus{
    my ($self, $host, $config) = @_;
    my $namesrvUtility = $self->getNameServerUtility();
    my $success =  $namesrvUtility->checkCanRemoveHost($host, $config);
    if(!$success){
        $self->getErrMsgLst()->appendMsgLst($namesrvUtility->getErrMsgLst());
    }
    return $success;
}

sub getHostRoleRemoveStatus {
    my ($self, $host, $role) = @_;
    my $namesrvUtility = $self->getNameServerUtility();
    my $canRemove = $namesrvUtility->checkCanRemoveRole($host, $role);
    if(!$canRemove){
        $self->getErrMsgLst()->appendMsgLst($namesrvUtility->getErrMsgLst());
    }
    return $canRemove;
}

sub disableAutoFailover {
    my ($self, $msglst) = @_;

    my $msg = $msglst->addMessage ("Disabling auto-failover...");
    my $submsglst = $msg->getSubMsgLst();

    my $layeredCfg = $self->getLayeredConfig ();
    if (!defined $layeredCfg){
        $submsglst->addWarning ("Cannot get layered configuration");
        return undef;
    }
    $layeredCfg->setMsgLstContext([$submsglst]);
    my $nameserverIni = $layeredCfg->getIniFile ('nameserver.ini');
    if (!defined $nameserverIni){
        $submsglst->addWarning ('Cannot get nameserver.ini', $layeredCfg->getErrMsgLst());
        return undef;
    }

    $nameserverIni->setMsgLstContext ([$submsglst]);
    if (!defined $nameserverIni->readValues ()){
        $submsglst->addWarning ('Cannot load nameserver.ini', $nameserverIni->getErrMsgLst());
        return undef;
    }

    my $isIniModified = 0;
    my $hostFailoverEnabled = $nameserverIni->getValue('failover', 'enable');
    if ($hostFailoverEnabled eq 'false') {
        $submsglst->addMessage('Host auto-failover is already disabled');
    } else {
        $isIniModified = 1;
        $nameserverIni->setValue(CFG_LAYER_SYSTEM, 'failover', 'enable', 'false');
        $nameserverIni->setValue(CFG_LAYER_SYSTEM, 'failover', 'hplm_host_failover_disabled', 1);
    }

    my $masterFailoverEnabled = $nameserverIni->getValue('failover', 'enable_master_failover');
    if ($masterFailoverEnabled eq 'false') {
        $submsglst->addMessage('Master worker auto-failover is already disabled');
    } else {
        $isIniModified = 1;
        $nameserverIni->setValue(CFG_LAYER_SYSTEM, 'failover', 'enable_master_failover', 'false');
        $nameserverIni->setValue(CFG_LAYER_SYSTEM, 'failover', 'hplm_master_failover_disabled', 1);
    }

    if (!$isIniModified) {
        return 1;
    }

    if (!defined $nameserverIni->write ()){
        $submsglst->addWarning ('Cannot save nameserver.ini', $nameserverIni->getErrMsgLst());
        return undef;
    }

    my $hdbnsutilMsgLst = new SDB::Install::MsgLst();
    my $rc = $self->runNameServerUtility (['-reconfig'], $hdbnsutilMsgLst);
    $submsglst->appendMsgLst($hdbnsutilMsgLst);
    if (!defined $rc){
        $submsglst->addWarning ('Reconfigure failed', $hdbnsutilMsgLst);
    }
    $submsglst->addMessage ("Auto-failover disabled.");

    return 1;
}

sub enableAutoFailover {
    my ($self, $msglst) = @_;
    my $msg = $msglst->addMessage ("Enabling auto-failover...");
    my $submsglst = $msg->getSubMsgLst();

    my $layeredCfg = $self->getLayeredConfig ();
    if (!defined $layeredCfg){
        $submsglst->addWarning ("Cannot get layered configuration");
        return undef;
    }
    $layeredCfg->setMsgLstContext([$submsglst]);
    my $nameserverIni = $layeredCfg->getIniFile ('nameserver.ini');
    if (!defined $nameserverIni){
        $submsglst->addWarning ('Cannot get nameserver.ini', $layeredCfg->getErrMsgLst ());
        return undef;
    }

    $nameserverIni->setMsgLstContext ([$submsglst]);
    if (!defined $nameserverIni->readValues ()){
        $submsglst->addWarning ('Cannot load nameserver.ini', $nameserverIni->getErrMsgLst());
        return undef;
    }

    my $isIniModified = 0;
    my $hostFailoverEnabled = $nameserverIni->getValue('failover', 'enable');
    if ($hostFailoverEnabled eq 'true') {
        $submsglst->addMessage('Auto-failover is already enabled');
    } elsif ($nameserverIni->existsValue('failover', 'hplm_host_failover_disabled')) {
        $isIniModified = 1;
        $nameserverIni->setValue(CFG_LAYER_SYSTEM, 'failover', 'enable', 'true');
        $nameserverIni->removeKey(CFG_LAYER_SYSTEM, 'failover', 'hplm_host_failover_disabled');
    } else {
        $submsglst->addMessage('Skipping enabling of host auto-failover because of missing \'hplm_host_failover_disabled\' flag');
    }

    my $masterFailoverEnabled = $nameserverIni->getValue('failover', 'enable_master_failover');
    if ($masterFailoverEnabled eq 'true') {
        $submsglst->addMessage('Auto-failover is already enabled');
    } elsif ($nameserverIni->existsValue('failover', 'hplm_master_failover_disabled')) {
        $isIniModified = 1;
        $nameserverIni->setValue(CFG_LAYER_SYSTEM, 'failover', 'enable_master_failover', 'true');
        $nameserverIni->removeKey(CFG_LAYER_SYSTEM, 'failover', 'hplm_master_failover_disabled');
    } else {
        $submsglst->addMessage('Skipping enabling of master worker auto-failover because of missing \'hplm_master_failover_disabled\' flag');
    }

    if (!$isIniModified) {
        return 1;
    }

    if (!defined $nameserverIni->write ()){
        $submsglst->addWarning ('Cannot save nameserver.ini', $nameserverIni->getErrMsgLst());
        return undef;
    }
    my $hdbnsutilMsgLst = new SDB::Install::MsgLst();
    my $rc = $self->runNameServerUtility (['-reconfig'], $hdbnsutilMsgLst);
    $submsglst->appendMsgLst($hdbnsutilMsgLst);
    if (!defined $rc){
        $submsglst->addWarning ('Reconfigure failed', $hdbnsutilMsgLst);
    }
    $submsglst->addMessage ("Auto-failover enabled.");

    return 1;
}

sub isAutoFailoverDisabled {
    my ($self) = @_;

    my $layeredCfg = $self->getLayeredConfig ();
    return undef if ! defined $layeredCfg;

    my $nameserverIni = $layeredCfg->getIniFile ('nameserver.ini');
    return undef if ! defined $nameserverIni;
    return undef if ! defined $nameserverIni->readValues ();

    my $isHostFailoverDisabled = $nameserverIni->getValue('failover', 'enable') eq 'false';
    my $isMasterFailoverDisabled = $nameserverIni->getValue('failover', 'enable_master_failover') eq 'false';
    return $isHostFailoverDisabled || $isMasterFailoverDisabled;
}

#our @knownHostRoles = qw (worker standby streaming extended_storage_standby
#                     extended_storage_worker accelerator_standby accelerator_worker);

sub getStreamingDir{
    return join ($path_separator, $_[0]->get_globalSidDir (),
                                  $gDirNameStreaming);
}

sub getStreamingLcmDir{
    return join ($path_separator, $_[0]->getStreamingDir (),
                                  'hdblcm');
}

sub getAcceleratorLcmDir{
    return join ($path_separator, $_[0]->get_globalSidDir (),
                                  $gDirNameAccelerator, 'hdblcm');
}

sub getExtendedStorageLcmDir{
    return join ($path_separator, $_[0]->get_globalSidDir (),
                                  $gDirNameEs, 'hdblcm');
}

sub getXS2Dir{
    return join ($path_separator, $_[0]->get_globalSidDir (),
                                  $gDirNameXS2);
}

sub existsXS2Dir {
    my $xs2Dir = $_[0]->getXS2Dir();
    return (defined $xs2Dir && -d $xs2Dir) ? 1 : 0;
}

#-------------------------------------------------------------------------------
# Returns a reference to a string array containing the sorted hosts
# with their roles.
#
# The roles of all hosts are read from the nameserver file if the roles are
# not read before.

sub getAllHostsWithRoles {
    my ($self, $productNamesWanted) = @_;

    my $hosts         = $self->get_allhosts();
    my $hostRolesInfo = $self->getHostRolesInfo($productNamesWanted);
    my @hostOutputList;

    if (defined $hostRolesInfo) {
        foreach my $currHost (sort(@$hosts)) {
            my $roles    = $hostRolesInfo->{$currHost};
            my $currInfo = (defined $roles) ? "$currHost ($roles)"
                                            : $currHost;
            push @hostOutputList, $currInfo;
        }
    }
    else {
        @hostOutputList = sort(@$hosts)
    }

    return \@hostOutputList;
}


#-------------------------------------------------------------------------------
# Returns a hash map containing the roles for each host.

sub getHostRolesInfo {
    my ($self, $productNamesWanted) = @_;

    if (!defined $self->{_hostRolesInfo}) {
        $self->updateHostsRolesFromNameserverFile();
    }
    if (!defined $self->{_hostRolesInfo}) {
        return undef;
    }
    if (!$productNamesWanted) {
        return $self->{_hostRolesInfo};
    }

    my $productRolesInfo = {};
    my $roleProperties   = GetHostRoleProperties();

    foreach my $host (keys %{$self->{_hostRolesInfo}}) {
        my @names;
        foreach my $currRole (split(/\s+/, $self->{_hostRolesInfo}->{$host})) {
                push @names, $roleProperties->{$currRole}->{str};
        }
        $productRolesInfo->{$host} = join (', ', @names);
    }

    return $productRolesInfo;
}

sub getHostsByRole {
    my ($self) = shift;
    my @rolesToCheck = @_;
    my $hostRolesInfo = $self->getHostRolesInfo();
    return undef if (!defined $hostRolesInfo);
    my @hosts = ();
    while(my ($host, $roles) = each(%$hostRolesInfo)) {
        if (grep {$roles =~ /(^|\s)$_/} @rolesToCheck) {
            push(@hosts, $host);
            next;
        }
    }
    return \@hosts;
}

#-------------------------------------------------------------------------------
# Returns the name of the xscontroller host or undef if there isn't one.

sub getXsControllerHostname {
    my ($self) = @_;
    my $hosts = $self->getHostsByRole($gHostRoleXS2Worker);
    return undef if (!defined($hosts) || !@{$hosts}); # No xs_worker role => no xscontroller host

    for my $host (@$hosts) {
        my $daemonIni = $self->readDaemonIni(undef, $host);
        return $host if ($daemonIni->getValue('xscontroller', 'instances'));
    }
# Try to detect the xs controller host from the topology
    return $self->getNameServerUtility()->getXsControllerHost(); # may be undef
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing the host roles of the local host
# or specified host.
#
# Returns $roles->[$gHostRoleWorker, $gHostRoleEsWorker]
#
# Returns undef in case of an error
# (An error an error message is written into the message list of TrexInstance).

sub getHostRolesByIniFile{
    my ($self, $host) = @_;

    if (!defined $host){
        $host = $self->{_host};
    }
    if (!defined $host){
        $self->setErrorMessage ('Cannot get host roles of undefined host');
        return undef;
    }
    my $data = $self->_readCustomNameServerIni ();

    if (!defined $data){
        return undef;
    }

    if (!defined $data->{landscape}){
        $self->setErrorMessage ('landscape section not found in nameserver.ini');
        return undef;
    }


    my $landscape = $data->{landscape};
    my $roles = $landscape->{'roles_' . $host};

    if (!defined $roles){
        return [];
    }
    return [split (/\s+/,$roles)]
}

#-------------------------------------------------------------------------------
# Returns a reference to a hash list containing the host roles as hash keys
# with host arrays as hash values.
#
# Returns $roles->{$gHostRoleWorker}   ->['lu0001', 'lu0005']
#               ->{$gHostRoleStandby}  ->['lu0002']
#               ->{$gHostRoleEsWorker} ->['lu0003']
#               ->{$gHostRoleStreaming}->['lu0004']
#
# Returns undef if '<sapmnt>/<SID>/global/hdb/custom/config/nameserver.ini'
# cannot be read (an error an error message is written into the message list
# of TrexInstance).

sub getActiveHostRoles{
    my ($self) = @_;
    my $data = $self->_readCustomNameServerIni ();

    if (!defined $data){
        return undef;
    }

    if (!defined $data->{landscape}){
        $self->setErrorMessage ('landscape section not found in nameserver.ini');
        return undef;
    }
    my $landscape = $data->{landscape};
    my $roleHost;
    my %roles;
    my $hostRoles;
    my $role;
    foreach my $host (@{$self->get_allhosts()}){
        $hostRoles = $landscape->{'roles_'.$host};
        if (!defined $hostRoles){
            next;
        }
        foreach $role (split (/\s+/, $hostRoles)){
            if (!defined $roles{$role}){
                $roles{$role} = [];
            }
            push @{$roles{$role}}, $host;
        }
    }
    return \%roles;
}

sub getPluginInstallation{
    my ($self,$productKey) = @_;
    my $dir = $self->get_pluginsDir() . $path_separator . $productKey;
    my $stat = File::stat::stat($dir);
    if (!defined $stat){
        $self->setErrorMessage("Cannot access plugin directory '$dir': $!");
        return undef;
    }
    if (!-d $stat){
        $self->setErrorMessage("'$dir' isn't a directory");
        return undef;
    }

    require SDB::Install::Installation;
    my $inst = new SDB::Install::Installation ($dir);
    #
    # check if the plugin subdirectory matches the product key
    #
    my $productKeyFromManifest = $inst->getProductKey();
    if ($productKeyFromManifest ne $productKey){
       $self->setErrorMessage("Invalid server plugin in '$dir': product key = '$productKeyFromManifest'", $inst->getErrMsgLst);
       return undef;
    }
    return $inst;
}

#-----------------------------------------------------------------

sub getPluginInstallations{
    my ($self, $nocache) = @_;
    if (!defined $nocache){
        $nocache = 0;
    }

    if (defined $self->{_pluginInstallations} && !$nocache){
        return $self->{_pluginInstallations};
    }

    my $pluginsFolder = $self->get_pluginsDir();
    if (!-d $pluginsFolder){
        $self->{_pluginInstallations} = [];
        return $self->{_pluginInstallations};
    }
    #
    # find subdirectories in plugins dir
    #
    if (!opendir (DH, $pluginsFolder)){
        $self->AddError ("Cannot open directory '$pluginsFolder': $!");
        return undef;
    }
    my @subdirs = grep {!/^\.{1,2}$/ && -d "$pluginsFolder$path_separator$_"} readdir (DH);
    closedir (DH);
    if (!@subdirs){
        $self->{_pluginInstallations} = [];
        return $self->{_pluginInstallations};
    }
    my @result;
    my $inst;
    #
    # check subdirectories for a valid installation
    #
    foreach my $key (@subdirs){
        $inst = $self->getPluginInstallation ($key);
        if (defined $inst){
            push @result, $inst;
        }
    }
    $self->{_pluginInstallations} = \@result;
    return $self->{_pluginInstallations};
}
#-------------------------------------------------------------------------------
# Returns a pattern that matches a directory in the middle of the
# specified parth (e.g. /hana/shared/SID/data)

sub getDirectoryPatternMiddle {
    my ($dir) = @_;
    return quotemeta ($path_separator. $dir . $path_separator);
}


#-------------------------------------------------------------------------------
# Returns a pattern that matches a directory at the end of the
# specified parth (e.g. /hana/data/SID)

sub getDirectoryPatternEnd {
    my ($dir) = @_;
    return quotemeta ($path_separator) . $dir . '$';
}


#-------------------------------------------------------------------------------
# Returns prefix of a volume path in front of the first occurence of SID
# (e.g. '/hana/shared/SID/data' --> '/hana/shared')

sub getVolumePathPrefix {
    my ($path, $patternMiddleSID, $patternEndSID) = @_;

    my ($pathPrefix) = ($path =~ /^(.*)$patternMiddleSID/);
    if (!defined $pathPrefix){
            ($pathPrefix) = ($path =~ /^(.*)$patternEndSID/);
    }
    return $pathPrefix;
}


#-------------------------------------------------------------------------------
# Returns 1 if the specified path is already renamed or
# can be renamed by changing the SID and instance number.

sub isVolumePathStandardRename {
    my ($self, $oldPath, $specifiedPath, $newSID, $newNr) = @_;

    if (!defined $oldPath || (length($oldPath) != length($specifiedPath))) {
        return 0;
    }

    my $oldSID = $self->{_sid};
    my $prefix = getVolumePathPrefix($specifiedPath,
                                      getDirectoryPatternMiddle($newSID),
                                      getDirectoryPatternEnd   ($newSID));

    if (!defined $prefix) {
        # specified path does not contain a SID --> no standard rename
        return 0;
    }

    my $prefixWithOldSID = $prefix . $path_separator . $oldSID;
    my $prefixWithNewSID = $prefix . $path_separator . $newSID;

    if (!-d $prefixWithOldSID && !-d $prefixWithNewSID) {
        # prefix (e.g. '/hana/shared/<SID>') does not exist
        return 0;
    }

    my $prefixLen = length($prefixWithOldSID);

    if ($prefixLen < length($oldPath)) {

        my $specifiedTail = substr($specifiedPath, $prefixLen);
        my $oldTail       = substr($oldPath,       $prefixLen);
        my $newTail       = $self->getVolumePathWithReplacedNr($oldTail,
                                                               $self->get_nr(),
                                                               $newNr);

        if ($oldTail eq $newTail) {
            # tail does not contain an instance number

            if ($specifiedTail ne $oldTail) {
                # --> no standard rename
                return 0;
            }
        }
        elsif (($specifiedTail ne $oldTail) && ($specifiedTail ne $newTail)) {
            # --> no standard rename
            return 0;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Returns the new path by renamaing the first occurrence of '<oldSID>'
# and 'HDB<oldNr>' contained in $oldPath.
# $oldNr and $newNr may be undef.
#
# Examples: (old/new SID: DB1/DB2 - old/new Nr: 01/02)
# '/hana/data/DB1'                     --> '/hana/data/DB2'
# '/hana/shared/DB1/data'              --> '/hana/shared/DB2/data'
# '/hana/shared/DB1/HDB01/backup/data' --> '/hana/shared/DB2/HDB02/backup/data'
# '/hana/shared/DB1/xyz/DB1'           --> '/hana/shared/DB2/xyz/DB1'
# '/hana/shared/DB1a/xyz/DB1'          --> '/hana/shared/DB1a/xyz/DB2'

sub getNewVolumePath {

    my ($self,
        $oldVolumePath,
        $newSID,
        $newNr,
        $oldSID, # may be undef
        $oldNr,  # may be undef
        $target
       ) = @_;

    my $newVolumePath = $oldVolumePath;

	$oldSID = $self->{_sid}   if (!defined $oldSID);
	$oldNr  = $self->get_nr() if (!defined $oldNr);

    if ($oldSID ne $newSID) {
        my $prefix = getVolumePathPrefix($oldVolumePath,
                                         getDirectoryPatternMiddle($oldSID),
                                         getDirectoryPatternEnd   ($oldSID));

        if (defined $prefix) {
            my $renamePattern = quotemeta(File::Spec->catfile($prefix, $oldSID));
# Do not rename the volume path if it is part of the installation path
            return $oldVolumePath if $target =~ /^$renamePattern.*$/;

            my $patternSID = '^'.$renamePattern;
            my $newPrefix  = File::Spec->catfile($prefix, $newSID);
            $newVolumePath =~ s/${patternSID}/${newPrefix}/;
        }
    }

    $newVolumePath = $self->getVolumePathWithReplacedNr($newVolumePath,
                                                        $oldNr, $newNr);

    return $newVolumePath;
}


#-------------------------------------------------------------------------------
# Replaces the first occurrence of 'HDB<oldNr>' by 'HDB<newNR>' of the
# specified path.

sub getVolumePathWithReplacedNr {

	my ($self, $path, $oldNr, $newNr) = @_;

    if ($oldNr != $newNr) {
        my $patternDir = getDirectoryPatternMiddle(sprintf('HDB%02d', $oldNr));
        my $newDir     = $path_separator . sprintf('HDB%02d', $newNr)
                                                              . $path_separator;
        $path =~ s/${patternDir}/${newDir}/;
    }
    return $path;
}

sub useSystemReplication{
    my($self,$no_cache) = @_;
    my $replicationMode = $self->getReplicationModeByIniFile ($no_cache);
    return defined $replicationMode && $replicationMode ne 'none';
}

sub isMultiDb{
    my $dbMode = $_[0]->getDbMode($_[1]);
    if (defined $dbMode && (lc($dbMode) eq 'multidb')) {
        return 1;
    }
    return 0;
}

sub isIsolatedMultiDb{
    my ($dbMode, $isolation) = $_[0]->getDbMode();
    if (defined $dbMode && (lc($dbMode) eq 'multidb') &&
        defined $isolation && (lc($isolation) eq 'high')) {
        return 1;
    }
    return 0;
}

sub isAddressSanitizerBuild {
    my ($self, $no_cache) = @_;
    if (defined $self->{_addressSanitizerBuild} && !$no_cache){
        return $self->{_addressSanitizerBuild};
    }
    my $ret = 0;
    my $mf = $self->get_instanceExeDir () . $path_separator . 'manifest';
    if (!-f $mf) {
        $mf = $self->get_instanceExeDir () . $path_separator . 'saptrexmanifest.mf';
        if (!-f $mf){
            return undef;
        }
    }
    if (!open (FD, $mf)){
        return undef;
    }
    while (my $line = <FD>){
        chomp ($line);
        if ($line =~ /(^saptrex |^)addresssanitizer: yes$/){
            $ret = 1;
            last;
        }
    }
    close (FD);
    $self->{_addressSanitizerBuild} = $ret;
    return $ret;
}

sub getReservedInstanceNumbers{
    my ($self) = @_;
    if ($self->isMultiDb ()){
        my $global_ini = $self->getGlobalIni ();
        if (!defined $global_ini){
            return undef;
        }
        my $reserved_instance_numbers =
            $global_ini->getValue ('multidb', 'reserved_instance_numbers');
        if (defined $reserved_instance_numbers){
            my $upper_limit = 97 - $self->{_nr};
            if ($reserved_instance_numbers > $upper_limit){
                $reserved_instance_numbers = $upper_limit;
            }
            return $reserved_instance_numbers;
        }
    }
    if ($self->useSystemReplication ()){
        # replication system
        # reserve also nr + 1
        return 1;
    }
    return 0;
}


sub getAdditionalReservedInstanceNumbers{
    my ($self) = @_;
    my @numbers;
    foreach my $i (1..$self->getReservedInstanceNumbers()){
        push @numbers, sprintf ('%02d',$self->{_nr} + $i);
    }
    return \@numbers;
}


#-------------------------------------------------------------------------------
# reads the nameserver file and sets the hash '_hostRolesInfo' consisting
# of the remote host names with their roles.

sub updateHostsRolesFromNameserverFile {
	my ($self, ) = @_;
	my $layerdConfig = $self->getLayeredConfig();
	if ( ! defined $layerdConfig ) {
		return;
	}

	my $nameserverFile = $layerdConfig->getNameserverFile();
	if ( ! defined $nameserverFile ) {
        return;
    }
    $nameserverFile->readValues(1);
    $self->{_hostRolesInfo} = $nameserverFile->getHostsRolesMap();
}


#-------------------------------------------------------------------------------
# If 'service/init_system_pki' is not contained in any profile,
# this value is inserted into the local profile.
#
# Disable ccms agent on each host, if 'ccms/enable_agent_custom' isn't set
#
# Removes LD_PRELAOD of compat-sap c++ runtime on RedHat
#
#

sub updateProfile {

    my ($self, $msglst, $mode, $uid, $gid) = @_;

    my $insertBehind = 'service/protectedwebmethods';
    my $valueName    = 'service/init_system_pki';
    my $wantedValue  = 'ON';
    my $profileDir   = $self->get_profileDir();
    my $profileNames = $self->getProfileNames($profileDir, $msglst);
    my $ownProfile   = undef;
    my $ownProfPath  = undef;
    my $ccmsAgentValueName = 'ccms/enable_agent';
    my $ccmsAgentWantedValue = '0';
    my $ld_preload_compat_sap_line_pattern = '^\s*SETENV_\d+\s*=\s*LD_PRELOAD\s*=.*compat-sap-c\+\+';
    if (!defined $profileNames) {
        return undef;
    }

    my $sysPKIAlreadyEnabled = 0;
    my @changedProfiles;

    foreach my $currProfName (@$profileNames) {
        my $profPath = join ($path_separator, $profileDir, $currProfName);
        my $profile  = new SDB::Install::IniParser::SAPProfile($profPath);
        my $profile_changed = 0;
        if (!defined $profile->read ()){
            $msglst->AddError ("Cannot read '$profPath'", $profile);
            return undef;
        }

        if (defined $profile->getValue($valueName)) {
            $sysPKIAlreadyEnabled = 1; # value found --> nothing to do
        }

        if ($profile->getValue($ccmsAgentValueName)
            && !$profile->getValue('ccms/enable_agent_custom')) {

            $msglst->AddMessage
                ("Updating '$ccmsAgentValueName = 0' in profile '$profile->{fullname}'");
                    $profile->setValue ($ccmsAgentValueName,
                      0,
                      1,            # active (i.e. not a comment)
                      undef,        # block comment
                      1             # with spaces (i.e. ' = ')
            );
            $profile_changed = 1;
        }

        while (my $deletedLine = $profile->delLineRegex($ld_preload_compat_sap_line_pattern)){
            $msglst->AddMessage
                ("Deleting line '$deletedLine' in profile '$profile->{fullname}'");
            $profile_changed = 1;
        }

        if ($profile_changed){
            push @changedProfiles, $profile;
        }
        if (!defined $ownProfile) {
            my $ownHost  = $self->get_host();
            if ($currProfName =~ /$ownHost/) {
                $ownProfile  = $profile;
                $ownProfPath = $profPath;
            }
        }
    }

    if ($sysPKIAlreadyEnabled && !@changedProfiles){
        $msglst->AddMessage ('Profile is up-to-date');
        return 1;
    }

    if (defined $ownProfile && !$sysPKIAlreadyEnabled){
        $msglst->AddMessage
            ("Adding '$valueName = $wantedValue' into profile '$ownProfPath'");

        $ownProfile->setValue($valueName,
                          $wantedValue,
                          1,            # active (i.e. not a comment)
                          undef,        # block comment
                          1,            # with spaces (i.e. ' = ')
                          $insertBehind);
        if (!(grep {$ownProfile eq $_} @changedProfiles)){
            push @changedProfiles, $ownProfile;
        }
    }

    my $rc = 1;
    foreach my $profile (@changedProfiles){
        my $msg = $msglst->addMessage("Writing profile '$profile->{fullname}'");
        $profile->setMsgLstContext([$msg->getSubMsgLst(), $self->getErrMsgLst()]);
        if (!defined $profile->write ($gLogDir, undef, $mode, $uid, $gid)){
            $rc = undef;
        }
    }

    return $rc;
}

sub updateSapprofileIni {
    my ($self, $ini) = @_;

    if (!defined $ini || !$ini->read()){
        return undef;
    }
    $ini->setValue(undef, 'DIR_PROFILE', $self->getLocalProfileDir());
    $ini->setValue(undef, 'DIR_EXECUTABLE', '$(DIR_INSTANCE)/exe');

    if (!defined $ini->write()){
        return undef;
    }

    return 1;
}

sub updateInstanceSapprofileInis {
    my ($self) = @_;
    my ($file, $ini);

    foreach my $host (@{$self->get_allhosts()}){
        $file = File::Spec->catfile($self->get_hostNameDir($host), 'sapprofile.ini');
        $ini = SDB::Install::IniFile->new($file);
        $ini->setMsgLstContext($self->getMsgLstContext());
        $self->updateSapprofileIni($ini);
    }
}

sub getStreamingRenameScript {
    my ($self) = @_;
    return File::Spec->catfile($self->getStreamingLcmDir(), $streamingRenameTool);
}

sub getExtendedStorageRenameScript {
    my($self) = @_;
    return File::Spec->catfile($self->getExtendedStorageLcmDir(), $extendedStorageRenameTool);
}

sub addMasterStartedHandler {
    my ($self, $masterStartedHandler) = @_;
    if (!defined $self->{masterStartedHandlers}){
        $self->{masterStartedHandlers} = [];
    }
    push @{$self->{masterStartedHandlers}}, $masterStartedHandler;
}


sub onMasterStarted{
    my ($self) = @_;
    my $rc = 1;

    if (!defined $self->{masterStartedHandlers}){
        return $rc;
    }
    my $msg = $self->getMsgLst ()->addMessage ("Calling onMasterStarted handler"),
    my $errlst;

    foreach my $onMasterStartedHandler (@{$self->{masterStartedHandlers}}){
        $errlst = new SDB::Install::MsgLst ();
        if (!$onMasterStartedHandler->($msg->getSubMsgLst, $errlst)){
            $self->appendErrorMessage ("onMasterStarted failed", $errlst);
            $rc = 0;
        }
    }
    return $rc;
}


sub sapstartsrvIsRunning(){
    my ($self) = @_;
    if (defined $self->{_host} && $self->isHostLocal($self->{_host})) {
        return $self->SUPER::sapstartsrvIsRunning();
    }
    return 0;
}

sub getSidAdmName{
    if (!defined $_[0]->{_user}){
        return undef;
    }
    return $_[0]->{_user}->getSidAdmName ();
}

sub stripSymbols{
    my ($self) = @_;
    if ($isWin) {
        return 1;
    }
    my $exeDir = $self->get_instanceExeDir();
    unless (-d $exeDir && -r $exeDir) {
        $self->AddError ("Cannot read exe directory \"".$exeDir."\": $!");
        return undef;
    }

    my $dirWalker = SDB::Install::DirectoryWalker->new(
        sub{  # actionMatcher callback, returns 1 for files that can be stripped
            my $isDir = $_[6];
            my $isSymLink = $_[8];
            if($isDir || $isSymLink) {
                return 0;
            }
            if ($_[4] =~ /lexicon\/lang/ || $_[4] =~ /.py$/ || $_[4] =~ /.h$/
            || $_[4] =~ /.c$/ || $_[4] =~ /.txt$/ || $_[4] =~ /.gif$/ || $_[4] =~ /.png$/) {
                # language files, source files, text files, image files
                return 0;
            }
            my $file = join($path_separator, $_[3], $_[4]);
            my $outbuffer;
            my $cfg = {'out' => \$outbuffer};
            my $rc = exec_program('file',
                [$file],
                 $cfg);
            if (!defined $outbuffer || !($outbuffer =~ /not stripped/)) {
                $self->AddMessage ("Strip symbols: not processing: " . $outbuffer);
                return 0;
            }
            return 1;
        },
        undef,
        undef,
        undef,
        undef,
        undef,
        sub{  # action callback, strip file
            my $file = join($path_separator, $_[3], $_[4]);
            my $cfg = {};
            my $rc = exec_program('strip',
                ['-s',
                $file],
                $cfg);
            $self->AddMessage (undef,$cfg);
            return 1;
        },
        undef,
        undef,
        0,  # no breadth-first traversal, so we do it depth-first
        1,  # do not collect list
        1   # follow symlinks
    );  # end of $dirWalker constructor

    my $rc = $dirWalker->findAndProcess($exeDir);

    return $rc;

}


sub getXsIniFile {
    my ($self) = @_;

    my $layered_cfg = $self->getLayeredConfig(1);
    if (!defined $layered_cfg) {
        $self->addError('Cannot get layered configuration');
        return undef;
    }
    $layered_cfg->setMsgLstContext([$self->getMsgLst()]);

    my $xs_ini = $layered_cfg->getIniFile('xscontroller.ini');
    if (!defined $xs_ini){
        $self->addError('Cannot get xscontroller.ini', $layered_cfg->getErrMsgLst());
        return undef;
    }

    return $xs_ini;
}

sub getXsRoutingMode {
    my ($self) = @_;

    my $xs_ini = $self->getXsIniFile();
    if (!defined $xs_ini){
        $self->addError('Cannot get xscontroller.ini');
        return undef;
    }
    $xs_ini->setMsgLstContext([$self->getMsgLst()]);
    if (!defined $xs_ini->readValues()){
        $self->addError("Cannot read xscontroller.ini", $xs_ini->getErrMsgLst());
        return undef;
    }
    return $xs_ini->getValue('communication', 'routing_mode');
}

sub getXsDomainName {
    my ($self) = @_;

    my $xs_ini = $self->getXsIniFile();
    if (!defined $xs_ini){
        $self->addError('Cannot get xscontroller.ini');
        return undef;
    }
    $xs_ini->setMsgLstContext([$self->getMsgLst()]);
    if (!defined $xs_ini->readValues()){
        $self->addError("Cannot read xscontroller.ini", $xs_ini->getErrMsgLst());
        return undef;
    }
    return $xs_ini->getValue('communication', 'default_domain');
}

sub setXsDomainName {
    my ($self, $value) = @_;

    my $xs_ini = $self->getXsIniFile();
    if (!defined $xs_ini){
        $self->addError('Cannot get xscontroller.ini');
        return undef;
    }

    $xs_ini->setValue (CFG_LAYER_SYSTEM,'communication','default_domain',$value);

    if (!defined $xs_ini->write()){
        $self->AddError ("Cannot flush ini file '$xs_ini->{name}'", $xs_ini);
        return undef;
    }
}

sub getLssInstance {
    my ($self,$noCheckForInstalled, $no_cache) = @_;
    if ($no_cache) {
        $self->{lssInstance} = SDB::Install::LSS::LssInstance->new($self);
    } else {
        $self->{lssInstance} //= SDB::Install::LSS::LssInstance->new($self);
    }
    return undef if(!$noCheckForInstalled && !$self->{lssInstance}->isInstalled());
    $self->{lssInstance}->setMsgLstContext($self->getMsgLstContext());
    return $self->{lssInstance};
}

sub getSortedTrexKillPids {
    my($self,$procs) = @_;
    my @pids;
    my $prio = 1;
    my @prioProcFiles = (
        'sapstartsrv',
        'sapstart',
        'hdbdaemon',
    );

    foreach my $prioProcFile (@prioProcFiles){
        foreach my $pid (keys (%$procs)){
            my @lockedFiles = @{$procs->{$pid}->{lockedFiles}};
            if(grep { $_ =~ /\/$prioProcFile$/ } @lockedFiles ) {
                push @pids, $pid;
                last;
            }
        }
        last if(scalar(@pids) == scalar(@prioProcFiles));
    }

    my %prioPids = map {$_ => 1} @pids;
    my @nonPrioPids = grep {not $prioPids{$_}} keys (%$procs);
    push @pids, @nonPrioPids ;

    return \@pids // [];
}

1;
__END__

=head1 NAME
A HANA Instance
=head1 DESCRIPTION
=over
=item getReplicationMode
Reads the replication mode from hdbnsutil.It is more accurate than getReplicationModeByIniFile,
as it retrieves the  most recent result.

=item getReplicationModeByIniFile
Reads replication mode from global.ini.
This method must be issued in performance critical situations or when the DB is offline.
=back
=cut
