package SystemCocktail;

#"""A type class that covers the different instance types"""

our ($UNDEFINED, $TREX_ALONE, $OTHERS, $TREX_AND_OTHERS) = (0..3);
our @SYSTEM_COCKTAILES = qw (UNDEFINED HDB_ALONE OTHERS HDB_AND_OTHERS);

sub new{
    my ($class,$type) = @_;
    my $self = bless {{}, $class};
    if (!defined $type){
        $self->{_type} = 0;
    }
    else{
        $self->set_type ($type);
    }
    return $self;
}

sub get_type{
    return $_[0]->{_type};
}

sub set_type{
    my ($self,$type) = @_;
    if ($type =~ /\D/){
        my $i = 0;
        my $found = 0;
        foreach my $str (@SYSTEM_COCKTAILES){
            if ($str eq $type){
                $self->{_type} = $i;
                $found = 1;
                last;
            }
            $i++;
        }
        if (!$found){
            die (__PACKAGE__ . "::set_type($type): EINVAL");
        }
    }
    else{
        if ($type >= 0 and $type < scalar @SYSTEM_COCKTAILES){
            $self->{_type} = int $type;
        }
        else{
            die (__PACKAGE__ . "::set_type($type): EINVAL");
        }
    }
}

sub checkType{
    my ($listOfTypes) = @_;
    my $trex = 0;
    my $others = 0;
    my $result = 0;
    foreach my $type (@$listOfTypes){
        if ($type eq 'HDB'){
            $result |= $TREX_ALONE;
        }
        else{
            $result |= $OTHERS;
        }
        if (($result & $TREX_AND_OTHERS) == $TREX_AND_OTHERS){
            last;
        }
    }
    return $result;
}

sub getStringType{
    return $SYSTEM_COCKTAILES[$_[0]->{_type}];
}

1;

package SDB::Install::Config;

our @ISA = qw (SDB::Install::BaseLegacy);

sub new{
    my $self = shift->SUPER::new ();
    ($self->{filename}) = @_;
    return $self;
}

sub load{
    my ($self, $nocache) = @_;

    if (defined $self->{data} && !$nocache){
        return $self->{data};
    }

    if (!$self->{filename}){
        $self->AddError ('No filename');
        return undef;
    }
    if (!-f $self->{filename}){
        $self->AddError ("File '$self->{filename}' not found");
        return undef;
    }
    if (!open (FD, $self->{filename})){
        $self->AddError ("Cannot open file '$self->{filename}': $!");
        return undef;
    }
    
    my $buffer = '';
    my $line;
    while (!eof (*FD)){
        if (!defined ($line = <FD>)){
            $self->AddError ("readline() failed: $!");
            close (FD);
            return undef;
        }
        chomp ($line);
        $line =~ s/\sNone([,}])/\ undef$1/g;
        $line =~ s/\s*:\s/\ =>\ /g;
        $buffer .= $line . "\n";
    }
    close (FD);

    $buffer .=  ';';
    
    my $hash = eval ($buffer);
    if ($@){
        $self->AddError ("Parse error: $@");
        return undef;
    }
    if (!defined $hash){
        $self->AddError ("Error parsing config: buffer = '$buffer'");
    }
    $self->{data} = $hash;
}

sub setData{
    $_[0]->{data} = $_[1];
}

sub getData{
    $_[0]->{data};
}

sub save{
    my ($self) = @_;
    if (!$self->{filename}){
        $self->AddError ('No filename');
        return undef;
    }
    if (!open (FD, '>'.$self->{filename})){
        $self->AddError ("Cannot create file '$self->{filename}': $!");
        return undef;
    }
    my $val;
    my $first = 1;
    print FD '{';
    foreach my $key (sort keys %{$self->{data}}){
        $val = $self->{data}->{$key};
        if (!defined $val){
            $val = 'None';
        }
        elsif ($val !~ /^\d+$/ or $val =~ /^0\d/){
            $val = "'$val'";
        }
        
        if ($first){
            print FD "'$key': $val";
            $first = 0;
        }
        else{
            print FD ",\n '$key': $val";
        }
    }
    print FD '}';
    
    close (FD);
    return 1;
}

1;

#===============================================================================

package SDB::Install::SAPSystem;

use strict;
use SDB::Install::SysVars;
use Exporter;
use SAPDB::Install::Hostname;
use SDB::Install::SAPInstance;
use SDB::Install::SAPInstance::TrexInstance qw ($processRestartHandoverPattern);
use SDB::Install::FsManager;
use SDB::Install::NewDBUser;
use SDB::Install::Installation;
use SDB::Install::SAPProfiles;
use SDB::Install::ServiceManager;
use SDB::Install::Sql::Hana2PersistenceMigrationHelper;
use File::Basename qw (dirname);
use File::Spec;

use SDB::Install::Globals qw($gHostRoleAcceleratorStandby
                             $gHostRoleAcceleratorWorker
                             $gHostRoleEsStandby
                             $gHostRoleEsWorker
                             $gHostRoleStreaming
                             $gLogDir
                             $gPlatform
                             $gProductName
                             $gProductNameEngine
                             $gSapsysGroupName
                             $gShortProductNameAccelerator
                             $gShortProductNameEs
                             $gShortProductNameStreaming
                             $gShortProductNameXS2
                             $gShortProductNameLSS
                             $gPhaseOfflineOption
                             $gPhaseConfigureOption
                             $gPhaseOnlineOption
                             $gPhaseOptionPerskey
                             $gPhasePrepareOption);
use SDB::Install::System qw (copy_file
                             copy_tree
                             copySelectedFiles
                             deltree
                             exec_program
                             getDirectoryFilenames
                             getFileId
                             isAdmin
                             isSameFile
                             isSameFile2
                             makedir
                             nfsidmap
                             readLink
                             which
                             setFilePermissions);
use SDB::Install::Tools qw (iso8601DateTime trim);
use SDB::Install::OSConfig;
use SDB::Install::Manifest;
use SDB::Install::Kit qw (ScanFilesForVCRedists InstallVCRedists);
use SDB::Install::DebugUtilities;
use SDB::Install::Saphostagent qw(installSaphostagent getSHAVersion);
use SDB::Install::HdbMountAPIInfo;
use SDB::Install::SysInfo;
use SAPDB::Install::System::Unix qw(lchown);
use File::Basename qw (basename);
use SDB::Common::ProcessUtils qw(terminateProcessAfterTimeout);

our @ISA = qw (SDB::Install::Installation Exporter);

our @FORBIDDEN_SIDS = qw (ADD ALL AMD AND ANY ARE ASC AUX AVG BIT
                          CDC COM CON DBA END EPS FOR GET GID IBM
                          INT KEY LOG LPT MAP MAX MIN MON NIX NOT
                          NUL OFF OLD OMS OUT PAD PRN RAW REF ROW
                          SAP SET SGA SHG SID SQL SUM SYS TMP TOP
                          UID USE USR VAR);


sub FORBIDDEN_SIDS {\@FORBIDDEN_SIDS};

our $SKIP_CHOWN_DIR = {'.snapshot'=>1};

sub SKIP_CHOWN_DIR {return $SKIP_CHOWN_DIR};

our @_known_instance_types = qw (HDB TRX DVEBM[A-Z]+ SCS ASCS D G JC J W SMDA ERS);

our $_instance_type_pattern = '^' . join ('$|^', @_known_instance_types). '$';

our $_sid_pattern = '^[A-Z][A-Z0-9][A-Z0-9]$';

our $legacy_manifest = 'saptrexmanifest.mf';

our $SAPINIT_SCRIPT = '/etc/init.d/sapinit';

our $STEP_EXTRACT            = 7;
our $STEP_ADAPT_CONFIG       = 14;
our $STEP_CREATE_INSTANCE    = 15;
our $STEP_CONFIGURE_TOPOLOGY = 20;
our $STEP_START_INSTANCE     = 23;
our $STEP_IMPORT_CONTENT     = 30;
our $STEP_INSTALL_ADD_HOSTS  = 31;


our ($STEP_UPGRADE_EXTRACT,
    $STEP_UPGRADE_PREPARE_INSTANCE,
    $STEP_UPGRADE_STOP_INSTANCE,
    $STEP_UPGRADE_SWITCH_VERSION,
    $STEP_UPGRADE_HANDLE_HOSTS,
    $STEP_UPGRADE_CONFIGURE_INSTANCE,
    $STEP_UPGRADE_START_INSTANCE,
    $STEP_UPGRADE_IMPORT_CONTENT) = (7, 10, 15, 23, 30, 31, 32, 33);

our @STEP_NAMES;

$STEP_NAMES[$STEP_EXTRACT] = 'Extracting Software';
$STEP_NAMES[$STEP_ADAPT_CONFIG] = 'Adapting System Configurations';
$STEP_NAMES[$STEP_CREATE_INSTANCE] = 'Creating System';
$STEP_NAMES[$STEP_CONFIGURE_TOPOLOGY] = 'Configuring Topology';
$STEP_NAMES[$STEP_START_INSTANCE] = 'Starting System';
$STEP_NAMES[$STEP_IMPORT_CONTENT]    = 'Importing Content';
$STEP_NAMES[$STEP_INSTALL_ADD_HOSTS] = 'Adding Hosts';

our @UPGRADE_STEP_NAMES;

$UPGRADE_STEP_NAMES[$STEP_UPGRADE_EXTRACT] = 'Extracting Software';
$UPGRADE_STEP_NAMES[$STEP_UPGRADE_PREPARE_INSTANCE] = 'Preparing instance';
$UPGRADE_STEP_NAMES[$STEP_UPGRADE_STOP_INSTANCE] = 'Stopping System';
$UPGRADE_STEP_NAMES[$STEP_UPGRADE_SWITCH_VERSION]  = $isWin ? 'Copying Binaries' : 'Switching used software to new version';
$UPGRADE_STEP_NAMES[$STEP_UPGRADE_HANDLE_HOSTS]   = 'Updating hosts';
$UPGRADE_STEP_NAMES[$STEP_UPGRADE_CONFIGURE_INSTANCE] = 'Configuring Instance';
$UPGRADE_STEP_NAMES[$STEP_UPGRADE_START_INSTANCE] = 'Starting System / Importing Content';
$UPGRADE_STEP_NAMES[$STEP_UPGRADE_IMPORT_CONTENT] = 'Importing Content';

our $STEP_ADDHOST_CREATE_DIRECTORIES  =  7;
our $STEP_ADDHOST_UPDATE_HOSTAGENT    =  8;
our $STEP_ADDHOST_ADD_INSTANCES       =  9;
our $STEP_ADDHOST_INIT_HOST_DIR       = 15;
our $STEP_ADDHOST_INSTALL_SAPPROFILE  = 16;
our $STEP_ADDHOST_CREATE_INST_PROFILE = 17;
our $STEP_ADDHOST_CONFIGURE_LSS       = 18;
our $STEP_ADDHOST_EXECUTE_HDBNSUTIL   = 19;
our $STEP_ADDHOST_CREATE_SERVICE      = 20;
our $STEP_ADDHOST_START_SYSTEM        = 21;
our $STEP_ADDHOST_CREATE_STREAMING    = 22;
our $STEP_ADDHOST_CREATE_ES           = 23;
#our $STEP_ADDHOST_CREATE_XS2         = 24;
our $STEP_ADDHOST_CREATE_ACCELERATOR  = 30;
our $STEP_ADDHOST_RESTART_SERVICES    = 31;

our @ADDHOST_STEP_NAMES;

$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CREATE_DIRECTORIES]  = 'Creating Directories';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_ADD_INSTANCES]       = 'Adding Instances';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_INIT_HOST_DIR]       = 'Initializing Host Directory';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_INSTALL_SAPPROFILE]  = 'Installing SAPProfile';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CREATE_INST_PROFILE] = 'Creating Instance Profile';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_EXECUTE_HDBNSUTIL]   = 'Executing hdbnsutil';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CREATE_SERVICE]      = 'Creating Service';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_START_SYSTEM]        = 'Starting System';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CREATE_STREAMING]    = "Adding $gShortProductNameStreaming";
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CREATE_ES]           = "Adding $gShortProductNameEs";
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CONFIGURE_LSS]          = "Adding $gShortProductNameLSS";
#$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CREATE_XS2]         = "Adding $gShortProductNameXS2";
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_UPDATE_HOSTAGENT]    = 'Updating SAP Host Agent';
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_CREATE_ACCELERATOR]  = "Adding $gShortProductNameAccelerator";
$ADDHOST_STEP_NAMES[$STEP_ADDHOST_RESTART_SERVICES]    = "Restarting HANA options services";

our @EXPORT = qw (CleanupSAPSystemCache
                  CollectSAPSystems
                  ExistsInstanceNumber
                  FORBIDDEN_SIDS
                  getPendingInfoFromPersFile
                  isForbiddenSID
                  matchesPatternSID
                  OccupiedInstanceNumbers
                  SKIP_CHOWN_DIR
                  @ADDHOST_STEP_NAMES
                  $STEP_ADDHOST_CREATE_DIRECTORIES
                  $STEP_CREATE_INSTANCE
                  $STEP_EXTRACT
                  @STEP_NAMES
                  $STEP_UPGRADE_EXTRACT
                  @UPGRADE_STEP_NAMES);

our @storageApiSudoerMinimumProgs = (
    '/sbin/multipath',
    '/sbin/multipathd',
    '/usr/bin/sg_persist',
    '/bin/mount',
    '/bin/umount',
    '/bin/kill',
    '/usr/bin/lsof'
);


sub getTarget{
    my ($globalSidDir) = @_;
    
    if ($isWin){
        if ($globalSidDir =~ /^[a-z]:\\usr\\sap\\/i){
            return undef;
        }
    }
    else{
        if ($globalSidDir =~ /^\/usr\/sap\/[^\/]+\/?$/){
            return undef;
        }
    }
        
    my $ret = undef;
    my ($dirname,$basename) =
        ($globalSidDir =~ /^(.*)$re_path_separator([^$re_path_separator]+)$/);
    
    if ($basename eq 'SYS'){
        $dirname =~ s/$re_path_separator[^$re_path_separator]+$//;
        $ret = $dirname;
    }
    else{
        $ret = $dirname;
    }
    return $ret;
}

sub getUsrSapSid {
    $_[0]->{_usrSapSid}; # returns a string containing the directory usr/sap/<sid>
}


#
# returns e.g. /hana/shared on Linux and <drive:>\usr\sap on Windows
#

sub getHanaInstPath {
    my $instPath = $_[0]->{_target};
    if (!defined $instPath && defined $_[0]->{_usrSapSid}){
        $instPath = dirname ($_[0]->{_usrSapSid});
    }
    return $instPath;
}

sub getGlobalSidDir{
    my ($usrSapSid) = @_;
    if ($isWin){
        my $dir= readLink ($usrSapSid . $path_separator . 'SYS');
        return $dir;
    }
    else{
        if ($usrSapSid !~ /^\/usr\/sap\/[^\/]+\/?$/){
            return $usrSapSid;
        }
        if (-l "$usrSapSid/SYS/profile"){
            my $dir = readlink ("$usrSapSid/SYS/profile");
            if ($dir eq '../profile'){
                return $usrSapSid;
            }
            $dir =~ s/\/profile\/*$//;
            return $dir;
        }
        elsif (-l "$usrSapSid"){
            my $dir = readlink ("$usrSapSid");
            return "$dir/SYS";
        }
        else{
            return "$usrSapSid/SYS";
        }
    }
}

sub getSysPath{
    my ($pathToSYSDir) = @_;
    
    if ($isWin){
        return readLink ($pathToSYSDir);
    }
    else{
        if (-e $pathToSYSDir){
            my ($dir) = ($pathToSYSDir =~ /(.*)$re_path_separator/);
            return readLink ($dir) . '/SYS';
            
            
            if (-d _ ){
                return $pathToSYSDir;
            }
            else{
                my $path = readLink ($pathToSYSDir . $path_separator .
                    'profile');
                $path =~  s/$re_path_separator[^$re_path_separator]+$//;
                return $path;
            }
        }
        else{
            return $pathToSYSDir;
            #die ('SYS dir is missing: ' . $pathToSYSDir);
        }
    }
 }

#-------------------------------------------------------------------------------
# Creates a new instance that has to be initialized later via 'initSAPSystem...'
#
# Without any parameter

sub new{
    my ($class) = @_;
    my $self = $class->SUPER::new();
    return $self;
}

#-------------------------------------------------------------------------------
# Initializes SAPSystem and the SUPER class Installer.
#
# Parameters: $usrSapSid     string   # '/usr/sap/<sid>' or '<sapmnt>/<sid>' or undef
#             $mode          bit map  # flags: SDB_O_RDONLY / SDB_O_RDWR / SDB_O_CREAT
#             $target        string   # '/usr/sap/<sid>' or '<sapmnt>/<sid>' or undef
#             $drive         string   # undef or e.g. 'C:' in case of windows

sub initSAPSystem {

    my ($self, $usrSapSid, $mode, $target, $drive) = @_;

    $self->initInstallationWithInstPath($usrSapSid, $mode);

    if (!$isWin && defined $target && $target eq '/usr/sap'){
        $target = undef;
    }

    my ($sid) = ($usrSapSid =~ /([^\\\/]+)[\\\/]*$/);

    $self->{_usrSapSid}      = $usrSapSid;
    $self->{_globalSidDir}   = undef;
    $self->{_instances}      = {};
    $self->{_raw}            = 0;
    $self->{_sid}            = uc($sid);
    $self->{_state}          = undef;
    $self->{_systemCocktail} = $SystemCocktail::UNDEFINED;
    
    $self->checkSid();

    my $syspath = getGlobalSidDir ($usrSapSid);
    
    if (!-d "$syspath/profile"){
        if($!{'ENOENT'}){
            $self->{_raw} = 1;
            if (defined $mode && $mode & SDB_O_CREAT){
                if (defined $target){
                    if (!-e $target){
                        $self->AddError ("Target $target does not exist");
                        return undef;
                    }
                    $self->{_target} = $target;
                    $self->{_globalSidDir} = $target . $path_separator .
                    $self->{_sid}
                }
                else{
                    $self->{_globalSidDir} = $self->{_usrSapSid};
                }
                if ($isWin){
                    $self->{_globalSidDir} .= $path_separator . 'SYS';
                }
            }
            else{
                $self->PushError ("Installation in '$usrSapSid' not found");
                return undef;
            }
        }
        else{
            $self->PushError ("Cannot look into $usrSapSid: $!");
            return undef;
        }
    }
    else{
        my $errmsgl = new SDB::Install::MsgLst();

        $self->{_globalSidDir} = $syspath;
                
        $self->{_target} = getTarget($self->{_globalSidDir});
        
        if (!defined $self->{_globalSidDir}){
            $self->PushError (undef,$errmsgl);
        }
        $self->collectInstances ();
        #self._systemCocktail = SystemCocktail.checkType(self.getTypeList())
    }
    
    if (defined $self->{_globalSidDir}){
        $self->setRegistryPath ($self->get_globalTrexInstallTemplateDir());
    }
    
    if (!$isWin){
        if ($usrSapSid =~ /^\/usr\/sap\/[^\/]+\/?$/){
            $self->{_usrSapSid} = $usrSapSid;
        }
        else{
            $self->{_usrSapSid} = '/usr/sap/' . $self->{_sid};
        }
    }
    else{
        if ($usrSapSid =~ /^[A-Za-z]:\\usr\\sap\\/){
            $self->{_usrSapSid} = $usrSapSid;
        }
        else{
            $self->{_usrSapSid} = $drive . '\usr\sap\\' . $self->{_sid};
        }
    }
    $self->{_user} = new SDB::Install::NewDBUser ($self->{_sid},undef, $self->get_integratedSystem());


    return 1;
}


#-------------------------------------------------------------------------------
# Initializes SAPSystem and the SUPER class Installer.
#
# Parameters: $sid      string
#             $globDir  string    # e.g. '/hana/shared'

sub initWithGlobDir {

    my ($self, $sid, $globDir) = @_;

    my $globSidDir = join($path_separator, $globDir, $sid);
    return $self->initSAPSystem($globSidDir);
}


#-------------------------------------------------------------------------------
# Completes the initialization of this class with the source SID if the
# class is initialized with a target SID only.

sub completeInitWithSourceSID {

    my ($self, $sourceSid, $usrSapSid) = @_;

    $self->{_sid}       = uc($sourceSid);
    $self->{_usrSapSid} = $usrSapSid;

    $self->initInstallationWithInstPath($self->{_usrSapSid});
    $self->checkSid();
    $self->collectInstances();
    $self->setRegistryPath($self->get_globalTrexInstallTemplateDir());

    $self->{_user} = new SDB::Install::NewDBUser
                           ($self->{_sid},undef, $self->get_integratedSystem());
    return 1;
}


#-------------------------------------------------------------------------------
# Pre-initializes this class with the specified target SID
# if '/usr/sap/<targetSID>' does not yet exist.

sub preInitWithTargetSID {

    my ($self,
        $targetSID,  # string
        $globDir     # string, e.g. '/hana/shared'
       ) = @_;

    $self->{_usrSapSid}      = undef;
    $self->{_globalSidDir}   = join($path_separator, $globDir, $targetSID);
    $self->{_instances}      = {};
    $self->{_targetSID}      = uc($targetSID);
    $self->{_raw}            = 0;
    $self->{_sid}            = undef;
    $self->{_state}          = undef;
    $self->{_systemCocktail} = $SystemCocktail::UNDEFINED;
    $self->{_target}         = $globDir;

    if (!-d "$self->{_globalSidDir}/profile") {
        $self->PushError ("Installation in '$self->{_globalSidDir}' not found");
        return undef;
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub get_sid {
    $_[0]->{_sid}; # returns a string containing the SAP system ID
}


sub get_target {
    $_[0]->{_target};
}


sub get_globalSidDir {
    $_[0]->{_globalSidDir};  # returns string containing the directory
}


sub get_globalTrexDir{
    return  join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb');
}


sub get_globalTrexCustomConfigDir{
    return  join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb', 'custom', 'config');
}

sub get_globalTrexInstallDir{
    return  join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb', 'install');
}

sub get_globalTrexInstallSupportDir{
    return  join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb', 'install', 'support');
}

sub get_globalTrexInstallTemplateDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb', 'install', 'config');
}

sub get_globalSAPHostAgentSetupDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, 'global', 'hdb', 'saphostagent_setup');
}

sub GetDataPath;
*GetDataPath = \&get_globalTrexInstallTemplateDir;
    
sub get_ProfileDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, 'profile');
}

sub get_trexBinDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, 'exe',
        $gPlatform, $isWin ? () : ('hdb'));
}

sub getPythonPath {
    my ($self) = @_;
    return File::Spec->catfile($self->get_trexBinDir(), 'Python', 'bin', 'python');
}

sub get_platformDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, 'exe',
        $gPlatform);
}


sub get_newVersionExeDir{
    return join ($path_separator, $_[0]->{_globalSidDir}, 'exe',
        $gPlatform, 'HDB_' . $_[0]->{'kit_versionid'});
}


sub get_pluginsDir{
    return join ($path_separator, $_[0]->get_platformDir (), 'plugins');
}

sub get_integratedSystem{
    return ($_[0]->{_systemCocktail} > $SystemCocktail::TREX_ALONE);
}

sub get_instances{
    $_[0]->{_instances};  # returns a hash of SDB::Install::SAPInstance
}

sub getFileSystemManager {
    my ($self, $nocache) = @_;
    if (!defined $self->{_fsMan} || defined $nocache && $nocache){

        $self->{_fsMan} = new SDB::Install::FsManager (
            $self->{_usrSapSid},
            $self->{_globalSidDir},
            $self->getUID,
            $self->getGID,
            $self->getShmGID);
    }
    return $self->{_fsMan};  # returns SDB::Install::FsManager
}

sub setFileSystemManager {
    # Parameter: SDB::Install::FsManager $fsManager
    my ($self, $fsManager) = @_;
    $self->{_fsMan} = $fsManager;
}


#-------------------------------------------------------------------------------
# Returns the target SID if the target SID does not match '/usr/sap/<SID>'.
# Otherwise undef is returned.

sub getTargetSID {
    my ($self) = @_;
    return $self->{_targetSID};
}


#-------------------------------------------------------------------------------
sub getUser {
    $_[0]->{_user};  # returns SDB::Install::NewDBUser
}


#-------------------------------------------------------------------------------

sub getIgnoreBusyProgramsPattern{
    return $processRestartHandoverPattern;
}

#-------------------------------------------------------------------------------
# Returns 1 and pushes an error if the given SID matches a reserved word
#                                                       otherwise 0 is returned.
# Parameters: string $sid
#             MsgLst $msglst  # may be undef in order to suppress an error message
sub isForbiddenSID {
    my ($self, $sid, $msglst) = @_;
    
    foreach my $isid (@FORBIDDEN_SIDS){
        if ($isid eq $sid){
            if (defined $msglst) {
                $msglst->PushError
                    ("'$sid' is not allowed because the following members are forbidden: " .
                     join (', ', @FORBIDDEN_SIDS));
            }
            return 1;
        }
    }
    return 0;
}


#-------------------------------------------------------------------------------
# Returns 1 if a target SID is used and
# the source SID of '/usr/sap/<SID>' is not yet assigned.

sub isPreInitWithTargetSID {
    my ($self) = @_;
    return (!defined $self->{_sid} && defined $self->{_targetSID});
}


#-------------------------------------------------------------------------------
# Returns 1 if the given SID matches the SID pattern.
# Parameter: string $sid
sub matchesPatternSID {
    my ($self, $sid) = @_;
    return ($sid =~ /$_sid_pattern/);
}


sub checkSid{
    my ($self,$sid, $msglst) = @_;

    if (!defined $msglst){
        $msglst = $self;
    }
    
    if (!defined $sid){
        $sid = $self->{_sid};
    }
    
    if (!$self->matchesPatternSID($sid)){
        $msglst->PushError ("'$sid' is not allowed for a $gProductNameEngine System Id.");
        return undef;
    }
    
    if ($self->isForbiddenSID($sid, $msglst)) {
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Scans the directory '/usr/sap/<sid>' or '<sapmnt>/<sid>'
# for directories names 'HDB<nr>' (i.e. Trex instance) or '<nr>' (any instance)
#
# the instances are assigned to $self->{_instances}

sub collectInstances{
    my ($self) = @_;

    my $pathSidDir = (defined $self->getTargetSID())
                     ? $self->{_globalSidDir}
                     : $self->{_usrSapSid};

    if (-x $pathSidDir){
        $self->{_instances} = {};

        if (!opendir (DH, $pathSidDir)){
            $self->PushError ("Cannot open directory '$pathSidDir': $!");
            return undef;
        }
        my @instanceDirs = grep {/^[A-Z]+\d\d$/ && -d "$pathSidDir/$_"} readdir (DH);
        closedir (DH);
        my ($type,$num);

        foreach my $instanceDir (@instanceDirs){
            ($type,$num) = ($instanceDir =~ /(.*)(\d\d)/);

            # Prevents overwriting already detected newDb instances in
            # case of multiple entries with the same number. Bug: 191507
            if (!$self->hasNewDBInstanceWithNumber($num) && $type =~ /$_instance_type_pattern/){
                my $path = $pathSidDir . $path_separator . $instanceDir;
                my $msg = $self->AddMessage ("Checking instance in $path"); 

                my $instance;
                if ($type eq 'HDB'){
                    $instance = new SDB::Install::SAPInstance::TrexInstance
                                                         ($path, $self->{_sid});
                }
                else{
                    $instance = new SDB::Install::SAPInstance ($path);
                }
                $self->{_instances}->{$num} = $instance;
                $self->AddSubMsgLst ($msg, $instance);
            }
        }
        $self->{_systemCocktail} = 
            SystemCocktail::checkType ($self->getTypeList ());
    }
    elsif (! -e _ ){
        $self->{_instances} = {};
        $self->{_state} = 'INST_STATE_NEW';
    }
    return 1;
}

sub hasNewDBInstanceWithNumber {
    my ($self, $number) = @_;
    my $instance = $self->{_instances}->{$number};
    return (defined $instance && $instance->isNewDB()) ? 1 : 0;
}


sub dumpProperties{
    my ($self) = @_;
    my $msg = $self->AddMessage ("Properties of $gProductNameEngine System $self->{_sid}");
    $self->SetFormatInfo ($msg, 'h1',"$gProductNameEngine System Properties");
    my $msglst = new SDB::Install::MsgLst ();
    $msglst->AddMessage ('IntegratedSystem: ' .($self->get_integratedSystem ? 'YES' : 'NO'));
    $msglst->AddMessage ("GlobalSidDir: $self->{_globalSidDir}");
    $msglst->AddMessage ('SystemCocktail: ' . (new SystemCocktail ($self->{_systemCocktail}))->getStringType());
    $msglst->AddMessage ("Types: " . join (', ', @{$self->getTypeList()}));
    if (!$isWin && defined $self->{_user} && $self->{_user}->exists()){
        $msglst->AddMessage ('Sidadm user id: '.$self->getUser()->uidUx());
        $msglst->AddMessage ('Sidadm group id: ' . $self->{_user}->{group}->id());
    }
    if (%{$self->{_instances}}){
        my $msg = $msglst->AddMessage('Instances:');
        my $instancemsglst = new SDB::Install::MsgLst ();
        foreach my $instance_name (keys %{$self->{_instances}}){
            my $instance = $self->{_instances}->{$instance_name};
            my $msg = $instancemsglst->AddMessage ('Instance dir: ' . $instance->{_instanceDir});
            if ($instance->isNewDB()){
                $instancemsglst->AddMessage ('Hosts: ' . join (', ', $instance->{_host}, @{$instance->{_hosts}}));
            }
        }
        $self->AddSubMsgLst ($msg, $instancemsglst);
    }
    $self->AddSubMsgLst ($msg, $msglst);
}


sub dumpSapSystems{
    my ($self, $msglst) = @_;
    
    my ($msg,$submsglst);
    if (!defined $msglst){
        $msglst = $self->getMsgLst();
        $msg = $msglst->addMessage ("Found SAP Systems");
        $submsglst = $msg->getSubMsgLst();
    }
    else{
        $submsglst = $msglst;
    }
    my $systems = CollectSAPSystems ();
    foreach my $sid (sort keys (%$systems)){
        $systems->{$sid}->asString ($submsglst);
    }
    if (defined $msg){
        $msg->endMessage(1, 'SAP Systems');
    }
}

sub asString{
    my ($self,
        $msglst,
        $newdb_only,
        $multipleLinesWithRoles,
        $rolesWithProductNames,
       ) = @_;

    my $dir = $isWin ? substr ($self->{_globalSidDir},0,-4) :
        $self->{_globalSidDir};
    my $string =  "$self->{_sid} $dir " .
        (new SystemCocktail ($self->{_systemCocktail}))->getStringType();
    my $submsglst;
    my $msg;
    
    if (defined $msglst){
        $msg = $msglst->AddMessage ($string);
        $submsglst = new SDB::Install::MsgLst ();
    }
    
    $string .= "\n";
    
    if (!$self->{_raw}){
        foreach my $instanceNo (sort keys (%{$self->{_instances}})){
            my $instance = $self->{_instances}->{$instanceNo};
            my $isNewDB  = $instance->isNewDB();
            if ($newdb_only && !$isNewDB){
                next;
            }
            if ($multipleLinesWithRoles && $isNewDB) {
                my $lines = $instance->asStringArray($rolesWithProductNames);
                foreach my $currLine (@$lines) {
                    if (defined $msglst) {
                        $msglst->AddMessage($currLine);
                    }
                    $string .= "\t" . $currLine . "\n";
                }
            }
            else {
                my $currWithProdNames = ($isNewDB) ? $rolesWithProductNames
                                                   : undef;
                $string .= "\t"
                        .  $instance->asString($submsglst, $currWithProdNames)
                        .  "\n";
            }
        }
    }
    if (defined $msglst){
        $msglst->AddSubMsgLst ($msg,$submsglst);
    }
    return $string;
}

sub getOccupiedInstanceNumbers{
    my $hdbInstance = $_[0]->getNewDBInstances->[0];
    my $additionalReservedInstanceNumbers;
    if (defined $hdbInstance){
        $additionalReservedInstanceNumbers =
            $hdbInstance->getAdditionalReservedInstanceNumbers ();
    }
    my @result;
    if (defined $additionalReservedInstanceNumbers){
       @result = @$additionalReservedInstanceNumbers;
    }
    push @result, keys %{$_[0]->{_instances}};
    return [sort  @result];
}

sub hasNewDB{
    foreach my $inst (values (%{$_[0]->{_instances}})){
        if ($inst->isNewDB ()){
            return 1;
        }
    }
    return 0;
}

sub hasTrex{
    foreach my $inst (values (%{$_[0]->{_instances}})){
        if ($inst->isTrex ()){
            return 1;
        }
    }
    return 0;
}



sub getTypeList{
    my %list;
    
    foreach my $inst (values (%{$_[0]->{_instances}})){
        $list{$inst->get_type()} = undef;
    }
    return [sort keys %list];
}

our $systems;

our $directory_usr_sap;
our $sapshare;

sub CleanupSAPSystemCache{
    undef $systems;
}


if ($isWin){
    require SAPDB::Install::System::Win32::API;
    my ($error,$dir,$comment) = SAPDB::Install::System::Win32::API::ShareInfo ('saploc');
    if ($error){
        $directory_usr_sap = '\usr\sap';
    }
    else{
        $directory_usr_sap = $dir; 
        $sapshare = $dir;
    }
    
    #my $hostname = SAPDB::Install::System::Win32::API::GetComputerName ();
    
    #if (! -d "$sapshare\\"){
    #   undef $sapshare;    
    #} 
    #$directory_usr_sap = '\usr\sap';
}
else{
    $directory_usr_sap = '/usr/sap';
}

our $instance_numbers;

sub CollectSAPSystems{
    my ($msglst, $no_cache) = @_;
    
    if (!$no_cache && defined $systems){
        return $systems;    
    }

    my %result = ();

    my @sap_locations;

    if ($isWin){
        if (defined $sapshare){
            @sap_locations = ($sapshare); 
        }
        else{
            my $path;
            my @drives = SAPDB::Install::System::Win32::API::GetHDDrives ();
            foreach my $drive (@drives){
                $path = "$drive$directory_usr_sap";
                if (-d $path){
                    push @sap_locations, $path;
                }
            }   
        }
    }
    else{
        @sap_locations = ($directory_usr_sap);
    }   

    $instance_numbers = {};
    
    foreach my  $sap_loc (@sap_locations){
        if (!-d $sap_loc){
            return undef;
        }

        if (!opendir (DH, $sap_loc)){
            if (defined $msglst){
                $msglst->PushError ("Cannot open directory '$sap_loc': $!");
            }
            next;
        }
    
        my @sids = grep {/$_sid_pattern$/ && -d "$sap_loc/$_"} readdir (DH);
    
        closedir (DH);
    
        my @instances;
        my ($type,$num);
        foreach my $sid (@sids){
            my $system = new SDB::Install::SAPSystem();
            $system->initSAPSystem($sap_loc . $path_separator . $sid);
            
            if ($system->ErrorState ()){
                #print ">>> ERROR: ".$system->GetErrorString."\n";
                if (defined $msglst){
                    $msglst->PushError ("Error opening system $sid", $system);
                }
                next;
            }
            foreach my $num (@{$system->getOccupiedInstanceNumbers()}){
                $instance_numbers->{$num} = $sid;
            }
            $result{$sid} = $system;
        }
    }
    $systems = \%result;
    return \%result;
}


sub getLinkedExePath{
    my ($self) = @_;
    
    if ($isWin){
        return $self->get_trexBinDir ();
    }
    
    my $version = $self->{kit_versionid};
    if (!defined $version){
        my $link = $self->get_trexBinDir ();
        if (-l $link && -d $link){
            return join ($path_separator, $self->{_globalSidDir},
                    'exe', $gPlatform,
                    readlink ($self->get_trexBinDir()));
        }
        else{
            $version = $self->GetVersion (1);
        }
    }
    
    if (!defined $version){
        return undef;
    }
    
    return join ($path_separator, $self->{_globalSidDir},
        'exe', $gPlatform, 'HDB_' . $version);
}

sub getProductName{
    return $gProductNameEngine;
}


sub getProgramPath;
*getProgramPath = \&getLinkedExePath;


sub genSapCpeFileList{
    my ($self) = @_;
    my %allfiles;
    my $files;

    my @dirs = qw (
        config
        filter
        llvm
        Python
        python_support
        servicehttp
        testscripts
        clienttest
        extensions
        Dictionary
        wdisp
        plugins
        di
        DataQuality
    );

    
    my $pattern;
    if ($self->isDebugBuild ()){
        $pattern = '^' . join ('\/|^', @dirs) . '\/|^trexbin.lst$';
    }
    else{
        $pattern = '^' . join ('\/|^', @dirs) . '\/|\.pdb$|^trexbin.lst$';
    }

    
    foreach my $package (values %{$self->{packages}}){
        if ($package->isGlobal){
            next;
        }
        $files = $package->GetFiles();
        if (defined $files){
            %allfiles = (%allfiles,%$files);
        }
    }

    my $bindir = $self->getLinkedExePath ();

    my $lstname = $bindir . $path_separator . 'hdbbin.lst';

    $self->AddMessage ("Generating sapcpe file list '$lstname'...");

    if (-f $lstname){
        rename ($lstname, $lstname . '.origin');
    }

    if (!open (FD, ">$lstname")){
        $self->AddError ("Cannot create file '$lstname': $!\n");
        return undef;
    }

    foreach my $file (sort (keys %allfiles)){

        # ignore files in subdirecories deeper than two layers
        if ($file =~ /\/[^\/]+\//){
            next;
        }

        if ($file =~ /$pattern/){
            next;
        }
        print FD "$file\n";
    }

    foreach my $dir (@dirs){
        if (-e "$bindir/$dir"){
            print FD "$dir\n";
        }
    }


    close (FD);

    return 1;
}


sub create{
    my ($self, $kit, $instconfig) = @_;
    my $instanceNr = $instconfig->getValue('InstanceNumber');
    my $password = $instconfig->getValue('Password');
    my $target = $instconfig->getValue('Target');
    my $hostname = $instconfig->getValue('HostName');
    my $id = $instconfig->getValue('UID');
    my $sapsysid = $instconfig->getValue('GID');
    my $shell = $instconfig->getValue('Shell');
    my $home = $instconfig->getValue('HomeDir');
    my $domain = $instconfig->getValue('Domain');

    if ($instconfig->isPhase($gPhaseOnlineOption)) {
        return 1;
    }

    $self->{kit_versionid} = $instconfig->getValue ('versionId');

    my $resume_step = $instconfig->getStep ();
    if (defined $resume_step){
        $resume_step = int $resume_step;
    }
    else{
        $resume_step = 0;
    }
    
    my $integratedSystem = 0;

    if (@{$self->getTypeList ()}){
        $integratedSystem = 1;
    }
    if ($integratedSystem){
        $self->{_globalSidDir} = getGlobalSidDir ($self->{_usrSapSid});
        $self->{_target} = getTarget ($self->{_globalSidDir});
    }
    
    
    if (defined $target && !$integratedSystem){
        if (!-e $target){
            $self->AddError ("Target $target does not exist");
            return undef;
        }
        $self->{_globalSidDir} =  join ($path_separator,
                                            $target,
                                            $self->{_sid},$isWin ? ('SYS') : ());
        $self->{_target} = $target;
    }
    
    if (!defined $self->{_target}){
        ($self->{_target}) = ($self->{_usrSapSid} =~ /(.*)$re_path_separator/);
        $self->{_globalSidDir} = join($path_separator,
             $self->{_usrSapSid}, $isWin ? ('SYS') : ());
    }

    $self->{_kitversion} = $kit->GetVersion ();

    if (!defined $hostname){
        $hostname = lc (hostname());
    }
    

    if ($resume_step <= $STEP_EXTRACT){
    
        if ($self->hasNewDB ()){
            $self->AddError ("System $self->{_sid} already has a $gProductNameEngine instance");
            return undef;
        }
    
        $self->{_user} = new SDB::Install::NewDBUser($self->{_sid}, $domain, $integratedSystem);
    
    
        $self->dumpProperties ();
    
        if (!defined $self->createRoot ()){
            return undef;
        }
    
        my $msg = $self->AddMessage ("Creating user...");
    
        $self->SetFormatInfo ($msg, 'h1', 'Create User');

        my $templateDir;
        $self->{_user}->resetMsgLstContext();
        if (!defined $self->{_user}->create ($templateDir, "$gProductNameEngine System Administrator", $id, $password, $sapsysid, $home, $shell)){
            $self->AddError ("Cannot create user", $self->{_user}); 
            $self->AddSubMsgLst ($msg,$self->{_user});
        return undef;
        }

        $self->AddSubMsgLst ($msg,$self->{_user});
        $self->{_user}->resetMsgLst();
        if (!defined $self->chownRoot ()){
            return undef;
        }

        my $fsMan = $self->getFileSystemManager();

        if (!defined $self->createFs ($instanceNr, $self->{kit_versionid})){
            return undef;
        }
    	
    	require SDB::Install::SHAOperationsManager;
    	my $shaOperationsManager = new SDB::Install::SHAOperationsManager($self, $kit->getArchiveDir(), $self->{_globalSidDir});
    	if (!$shaOperationsManager->copySHAOperationsAndSignature()){
            return undef;
        }
    
        if (!defined $self->OpenExclusive ()){
            return undef;
        }
    
        $instconfig->setStep ($STEP_EXTRACT, 1);
    
        $msg = $self->AddProgressMessage ("Extracting software...");
        $msg->setEndTag ('Extract Software');
        my $submsglst = $msg->getSubMsgLst ();
        $submsglst->setProgressHandler ($kit->getMsgLst ()->getProgressHandler ());
        $kit->setMsgLstContext([$submsglst]);
        my $saveMsgLst = $self->getMsgLst();
        if (!defined $kit->Install ($self)){
            $msg->endMessage ();
            $self->setMsgLstContext ([$saveMsgLst]);
            $self->AddError ("Extracting software failed", $kit);
            $msg->endMessage ();
            return undef;
        }
        $msg->endMessage ();
        $self->setMsgLstContext ([$saveMsgLst]);

        if (!defined $self->_linkRedHatCompatCPlusPlusLib ()){
            return undef;
        }

        #
        # checking global filesystem permissions after extraction phase, because
        # some packages (e.g.DAT Languages) use the config directory as
        # installation root
        #

        $msg = $self->getMsgLst->addMessage ("Checking global file system...");
        $fsMan->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $fsMan->createGlobalDirs($self->{kit_versionid})){
            $self->AddError ("Cannot check global file system", $fsMan);
            return undef;
        }

        if ($isWin){
            if (!defined $self->genSapCpeFileList ()){
                return undef;
            }
        }

        if (!$self->copyCustomConfigurations($instconfig)){
            return undef;
        }

        $instconfig->setStep ($STEP_ADAPT_CONFIG);
    }
    


    if ($instconfig->getStep() <= $STEP_ADAPT_CONFIG) {
        if (!defined $self->createDefaultProfile ($hostname)){
            return undef;
        }

        if (!$self->saveConfig ()){
            return undef;
        }
        if (!$isWin){
            if (!$self->adaptSystemConfiguration($integratedSystem || $instconfig->getValue('CreateInitialContainer'), $instconfig, $kit)) {
                return undef;
            }
        }
        $instconfig->setStep ($STEP_CREATE_INSTANCE);
    }

    return 1;
}

sub getNoLock{
	return $_[0]->{_nolock};
}

sub setNoLock{
	$_[0]->{_nolock} = $_[1];
}

sub createInstance{
    my ($self, $instconfig, $sysinfo) = @_;
    my $instanceNr = $instconfig->getValue('InstanceNumber');
    my $password = $instconfig->getValue('Password');
    my $hostname = $instconfig->getValue('HostName');
    my $nostart = $instconfig->getValue('NoStart');
    my $phase = $instconfig->getPhase();
    my $isOnlinePhase = $instconfig->isPhase($gPhaseOnlineOption);
    my $shouldStartSystem = !$nostart && (!$phase || $isOnlinePhase);
    my $progress_handler = $self->GetProgressHandler ();
    my $func_finished = sub {};

    if (defined $progress_handler){
        my @progressSteps = ('Creating System');
        push @progressSteps, 'Copying Binaries' if $isWin;
        push @progressSteps, 'Starting System'  if $shouldStartSystem;
        push @progressSteps, 'Importing delivery units' if $instconfig->getValue('ImportContent');
        $progress_handler->InitProgress (scalar @progressSteps, 0, \@progressSteps);
        $func_finished = $progress_handler->can ('StepFinished');
    }
    
    my $instanceNrStr = sprintf ('%02d',$instanceNr);
    my $instance      = new SDB::Install::SAPInstance::TrexInstance
            ($self->{_usrSapSid} . $path_separator . 'HDB' . $instanceNrStr);
    $self->{_instances}->{$instanceNrStr} = $instance;

    if (defined $self->{_f_callback}){
        $instance->setCallback ($self->{_f_callback});
    }

    $instance->{_target} = $self->{_target};

    my $resume_step = int($instconfig->getStep());
    my $resumingInstallationWithStart = 1;
    if ($resume_step <= $STEP_CREATE_INSTANCE) {
        $resumingInstallationWithStart = 0;
        my $msg = $self->AddProgressMessage ("Creating instance...");
        $self->{_user}->setMsgLstContext([$msg->getSubMsgLst ()]);
        $self->{_user}->configureHome ($self->get_globalTrexInstallSupportDir (), $instance->get_instanceDir, $hostname);
        $instance->SetProgressHandler ($self->GetProgressHandler);
        $self->SetFormatInfo ($msg, 'h1', 'Create instance');
        
        my $fsMan = $self->getFileSystemManager();

        $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $instance->create($fsMan, $self->{_user}, $instconfig, $self, $sysinfo, $progress_handler)){
            $self->AddError ("Cannot create instance", $instance);
            $func_finished->($progress_handler, 1);
            return undef;
        }

        $instconfig->setStep($STEP_CONFIGURE_TOPOLOGY);
        if ($instconfig->isPhase($gPhaseOfflineOption)) {
            $instconfig->{$gPhaseOptionPerskey} = $gPhaseOfflineOption;
            return 1;
        }
    }

    if ($resume_step <= $STEP_CONFIGURE_TOPOLOGY) {
        delete $instconfig->{$gPhaseOptionPerskey};
        my $msg = $self->AddProgressMessage ("Configuring instance...");
        $msg->setEndTag("Configure instance");

        $instance->setMsgLstContext([$msg->getSubMsgLst()]);
        if (!$instance->initTopology($instconfig)) {
            return undef;
        }
        my $iscMode = $instconfig->getValue ('ISCMode');
        if (defined $iscMode) {
            $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
            if (!defined $instconfig->handleGlobalIniISCMode($iscMode,
                                                             0, # $skipChanging
                                                             $instance)) {
                # if 0 is returned, the mode is already set
                $self->setErrorMessage ('Cannot set ISC mode properly', $instconfig->getErrMsgLst ());
                return undef; # error occurred
            }
        }

        if ($instconfig->getValue('StripSymbols')) {
            $instance->stripSymbols();
        }

        if (!$self->instUpdSaphostagent ($instconfig)){
            return undef;
        }
        $instconfig->setStep ($STEP_START_INSTANCE);
    }

    $func_finished->($progress_handler);

    if (!$isWin && !$instconfig->getValue('CreateInitialContainer')){
        my $serviceIsAlreadyRunning = 0;
        if ($resumingInstallationWithStart){
            my $sapControl = new SDB::Install::SAPControl (undef, $instanceNr);
            $serviceIsAlreadyRunning = $sapControl->isServiceRunning ();
        }
        if (!$serviceIsAlreadyRunning){
            my $msg = $self->getMsgLst ()->addMessage ("Starting sapstartsrv...");
            my $saveCntxt = $instance->setMsgLstContext ([$msg->getSubMsgLst()]);
            if (!defined $instance->startLocalSapStartSrv (undef,
                                      $instconfig->getTimeout ('start_service'),
                                      $instconfig->isUseHttps())){
                $self->setErrorMessage ("Cannot start sapstartsrv", $instance->getErrMsgLst ());
                return undef;
            }
        }
    }

    if ($instconfig->isPhase($gPhaseConfigureOption)) {
        $instconfig->{$gPhaseOptionPerskey} = $gPhaseConfigureOption;
        if ($nostart) {
            $self->removePersFiles($instconfig);
        }
        return 1;
    }

    if (!$nostart){
        delete $instconfig->{$gPhaseOptionPerskey};
        my $msg = $self->AddProgressMessage ("Starting $gProductNameEngine system...");
        $msg->setEndTag ("Start $gProductNameEngine system");
        $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $instance->startHost (
                    $self->getUser()->getSidAdmName(),
                    $password, # used for Windows only
                    undef,     # localhost
                    $instconfig->getTimeout ('start_instance'),
                    $instconfig->isUseHttps())){

            $self->AddError ("Cannot start system", $instance);
            $func_finished->($progress_handler, 1);
            $msg->endMessage ();
            return undef;
        }
        $func_finished->($progress_handler);
        $msg->endMessage ();
    }

	if ((!$phase || $isOnlinePhase) && $instconfig->getValue('ImportContent')
	    && ($resume_step <= $STEP_IMPORT_CONTENT)) {
        $instconfig->setStep ($STEP_IMPORT_CONTENT);

        $instance->SetProgressHandler ($self->GetProgressHandler());
        $instance->setMsgLstContext([$self->getMsgLst ()]);

	    my $rc = $instance->importDeliveryUnits($instconfig, undef, undef, undef, 1);
        if (!$rc){
            $self->AddError ('Import of delivery units failed', $instance);
            return undef;
        }
        if ($isOnlinePhase) {
# No point in trying to import plugin DUs during non-phased execution of hdbinst
# because there still aren't any installed plugins. On the other hand, during a
# phased execution, it is possible that plugins have been already installed (
# for example in the offline phase)
            $rc = $instance->importPluginDeliveryUnits ($instconfig);
            if (!$rc){
                $self->setErrorMessage ('Import of plugin delivery units failed', $instance->getErrMsgLst ());
                return undef;
            }
        }
	}

    if (!defined($phase) && !$nostart && ($resume_step <= $STEP_INSTALL_ADD_HOSTS)) {

        if ($resume_step < $STEP_INSTALL_ADD_HOSTS){
            $instconfig->setStep ($STEP_INSTALL_ADD_HOSTS);
        }

        if (!$instconfig->sendAddHostCommands($self, $self, $instance)) {
            $self->removePersFiles($instconfig);
            return undef;
        }
    }
    $self->removePersFiles($instconfig);
    return 1;
}

#-------------------------------------------------------------------------------
# Adapts the operating system configuration of HDB
#
# Parameter: optional boolean $skipInstallSAPInit
#
# Returns int retCode

sub adaptSystemConfiguration {
    my ($self, $skipInstallSAPInit, $instconfig, $kit) = @_;

    my $sidadm = $self->{_user}->getSidAdm()->getname();
    $self->_configureOS($instconfig, $sidadm);

    my $usesStorageApi = 0;
    my $hdbmountAPISkriptDir = undef;
    my $globalIniDir = undef;

    if ($instconfig->getValue ('StorageConfigDir')){
        #
        # during new installation StorageConfigDir cfg parameter has to be checked
        #
        $usesStorageApi = 1;
        $globalIniDir = $instconfig->getValue ('StorageConfigDir');
    }
    else{
        #
        # during hdbrename/hdbreg/hdbaddhost instance parameters have to be checked
        # # this alternative includes hdbupd as well.
        my $hdbInstances = $self->getNewDBInstances();
        if (defined $hdbInstances && defined $hdbInstances->[0]){
            my $hdbInstance = $hdbInstances->[0];
            $usesStorageApi = $hdbInstance->usesStorageApi ();
            $globalIniDir = $hdbInstance->get_globalHdbCustomConfigDir();
            # this will be overwritten in the hdbupd case, see below:
            $hdbmountAPISkriptDir = join ($path_separator,
                                          $hdbInstance->get_globalTrexInstallProgamDir(),
                                          'server');
        }
    }
    if(defined $kit) {
        # i.e. when we are hdbinst or hdbupd:
        # (setting $hdbmountAPISkriptDir for other scenarios has already been handled above)
        $hdbmountAPISkriptDir = join ($path_separator, $self->get_globalTrexInstallDir(), 'bin', 'server');
    }


    if ($usesStorageApi && !$instconfig->getValue('SkipModifySudoers')){
        my $uid = $self->{_user}->uidUx();
        my $gid = $self->{_user}->gidUx();
        my $mntinfo = SDB::Install::HdbMountAPIInfo->new($hdbmountAPISkriptDir, $globalIniDir, $uid, $gid);
        $mntinfo->setPython($self->getPythonPath());
        my $msg = $self->getMsgLst()->addMessage("Modifying sudoers...");

        my ($retcode, $retval) = $mntinfo->getSudoersLineCommandArray();
        if(not defined $retcode) {
            my $wrn = $self->AddWarning ("'hdbmount.py --sudoers...' did not return a valid sudoers line.");
            $self->AddSubMsgLst($wrn, $mntinfo);
        }
        else {
            if($retcode == 0) {
                my $err = $self->AddError ("Could not execute 'hdbmount.py --sudoers...'");
                $self->AddSubMsgLst($err, $mntinfo);
                return undef;
            }
            my $addCmds = $retval;
            ($retcode, $retval) = $mntinfo->getSudoersLinePrefix();
            if(not defined $retcode) {
                my $wrn = $self->AddWarning ("'hdbmount.py --sudoers...' did not return a valid sudoers line.");
                $self->AddSubMsgLst($wrn, $mntinfo);
            }
            else {
                if($retcode == 0) {
                    my $err = $self->AddError ("Could not execute 'hdbmount.py --sudoers...'");
                    $self->AddSubMsgLst($err, $mntinfo);
                    return undef;
                }
                my $sudoersPrefix = $retval;
                $self->AddSubMsgLst($msg, $mntinfo);

                $self->adaptSudoers(
                    $sidadm,
                    $addCmds,
                    undef,
                    $sudoersPrefix,
                    [$msg->getSubMsgLst(), $self->getErrMsgLst()]
                );
            }
        }
    } elsif ($usesStorageApi) {
        my $skipOption = $instconfig->getOpt('SkipModifySudoers');
        $self->AddMessage ("Modification of sudoers skipped due to command line option '$skipOption'");
    }

    if (!defined $skipInstallSAPInit || !$skipInstallSAPInit) {
        return $self->installSapinit();
    }
    return 1;
}

sub _configureOS {
    my (
        $self,
        $instconfig,
        $sidadm
    ) = @_;
    # we only try to handle the ip port range, if there is
    # no SHA >= 7.20.162 installed or going to be installed,
    # since such SHAs handle the port range anyway,
    # cf bug 164734.
    my $processPortRange = 1;
    my $msgText = "No SAP Host Agent of version >= 7.20.162 found or going to be installed: processing IP port range OS kernel parameter.";
    my $installParam = $instconfig->getValue('InstallHostagent');
    if(!defined $installParam || $installParam) {
        # skip if option '--install_hostagent=off' is not set, i.e.
        # if hostagent is going to be installed or updated.
        $msgText = "SAP Host Agent is going to be installed/updated: not processing IP port range OS kernel parameter.";
        $processPortRange = 0;
    }
    else {
        # we are always root here.
        my $fullVersion = getSHAVersion();
        if(defined $fullVersion && $fullVersion =~ /^[0-9]+\.[0-9]+\.[0-9]+$/) {
            my ($release, $revision, $patch) = split('\.', $fullVersion);
            if(100000000*int($release)+10000*int($revision)+int($patch) >= 700200162) {
                $msgText = "Found  SAP Host Agent of version $fullVersion (>= 7.20.162): not processing IP port range OS kernel Parameter.";
                $processPortRange = 0;
            }
        }
    }
    $self->getMsgLst()->addMessage($msgText);

    my $osConfig = SDB::Install::OSConfig->new();
    $osConfig->setMsgLstContext($self->getMsgLstContext());
    return $osConfig->adaptSystemConfigNewDB($sidadm, 0, $processPortRange);
}

sub copyCustomConfigurations {
    my ($self, $instconfig) = @_;

    my $customConfigDir = $instconfig->getValue ('CustomConfigDir');
    if (!$customConfigDir) {
        return 1;
    }

    my $msg = $self->getMsgLst ()->addMessage ("Copying custom configuration files...");
    my $msglst = $msg->getSubMsgLst ();
    my $destDir = $self->get_globalTrexCustomConfigDir();
    my $filenames = getDirectoryFilenames($customConfigDir);
    my $rc = 1;
    foreach my $filename (@$filenames) {
       if ($filename eq "hdbconfiguration.ini") {
           $msglst->addWarning ("'hdbconfiguration.ini' is skipped! hdbparam parameters aren't supported anymore.");
           next;
       }
       my $source = $customConfigDir . $path_separator . $filename;
       my $dest = $destDir . $path_separator . $filename;
       my $copyCfg = {};
       $msglst->addMessage ("Copy $source to $dest");
       if (!copy_file($source, $dest, $copyCfg)) {
           $msglst->addMessage("Cannot copy new file '$source' to '$dest'", $copyCfg);
           $rc = undef;
           last;
       }
       $self->setPermissions($dest, 0, 0644, $msglst);
    }
    return $rc;
}

sub createFs{
    my ($self, $instanceNr,$version) = @_;
    my $msg = $self->AddMessage ("Creating global filesystem");
    $self->SetFormatInfo ($msg, 'h1', 'Create Global Filesystem');
    my $rc = 1;
    my $fsMan = $self->getFileSystemManager();
    $fsMan->resetMsgLstContext ();
    if (!defined $fsMan->createGlobalFS($version)){
        $self->AddError ("Cannot create global file system", $fsMan);
        $rc = undef;
    }
    $self->AddSubMsgLst ($msg, $fsMan);

    return $rc;
}


sub update{
    my ($self, $password, $kit, $instconfig) = @_;
    #"""called from outside to update the system"""
    
    my $resume_step = $instconfig->getStep ();
    
    $resume_step = int $resume_step;
    
    $self->dumpProperties ();
    
    $self->{kit_has_filter_package} = defined $kit->GetPackageById ('filter');
    my $fsMan = new SDB::Install::FsManager (
        $self->{_usrSapSid},
        $self->{_globalSidDir},
        $self->getUID,
        $self->getGID,
        $self->getShmGID);
    my $msg = $self->AddMessage ("Migrating file system layout (system)...");
    $fsMan->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $fsMan->migrate ($self->{kit_versionid})){
        $self->AddError ('Migration of file system layout failed', $fsMan);
        return undef;
    }
    $msg = $self->AddMessage ('Adapting existing user');
    $self->{_user}->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $self->{_user}->adaptExistingUser()){
        $self->AddError ("Cannot adapt existing user", $self->{_user});
        return undef;
    }

    if ($resume_step <= $STEP_UPGRADE_EXTRACT){
    
        $instconfig->setStep ($STEP_UPGRADE_EXTRACT, 1);
    
        my $msg = $self->AddProgressMessage ("Extracting software...");
        $self->SetFormatInfo ($msg, 'h1', 'Extract Software');
        my $submsglst = $msg->getSubMsgLst ();
        $submsglst->setProgressHandler ($kit->getMsgLst ()->getProgressHandler ());
        $kit->setMsgLstContext([$submsglst]);
        if (!defined $kit->Install ($self)){
            $self->AddError ("Extracting software failed", $kit);
            return undef;
        }

        if ($isWin){
            if (!defined $self->genSapCpeFileList ()){
                return undef;
            }
        }
    }

    if ($resume_step <= $STEP_UPGRADE_PREPARE_INSTANCE) {
        $instconfig->setStep($STEP_UPGRADE_PREPARE_INSTANCE);

        my $instance = $self->getNewDBInstances ()->[0];
        $instance->setMsgLstContext ([$self->getMsgLst ()]);
        if (!defined $instance->migrate ($instconfig, $fsMan)){
            $self->AddError ('Cannot migrate HDB instance', $instance);
            return undef;
        }

        $msg = $self->AddMessage ("Updating environment scripts");
        $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $instance->installHDBScripts()){
            $self->AddError ("Error updating environment scipts", $instance);
            return undef;
        }
        
        require SDB::Install::SHAOperationsManager;
        my $shaOperationsManager = new SDB::Install::SHAOperationsManager($self, $kit->getArchiveDir(), $self->{_globalSidDir});
       	if (!$shaOperationsManager->copySHAOperationsAndSignature()){
    		return undef;
    	}
        
        my $sapprofileini_templ = $self->get_globalTrexInstallTemplateDir(). $path_separator . 'sapprofile.ini';
        if (! -f $sapprofileini_templ){
            if (defined $instance){
                $instance->setMsgLstContext ([$self->getMsgLst ()]);
                if (!$instance->prepareSAPProfileTemplate ()){
                    $self->AddError ("Cannot pepare SAPProfile template", $instance);
                }
                $self->setPermissions ($sapprofileini_templ,0, 0644);
            }
        } else {
            my $sapprofileIni = SDB::Install::IniFile->new($sapprofileini_templ);
            $sapprofileIni->setMsgLstContext($self->getMsgLstContext());
            $instance->updateSapprofileIni($sapprofileIni);
        }

        if (!defined $self->prepareVersionSwitch ()){
            return undef;
        }
        
        if((not $isWin) && (not $>)) {
            # only when we are root on a linux box:
            if (!defined $self->adaptSystemConfiguration (undef, $instconfig, $kit)){
                return undef;
            }
        }
    }
    
    return 1;
}



sub _isRunning {
    my ($self, $instconfig, $username, $password, $msglst) = @_;
    my $hdbInstance = $instconfig->getOwnInstance ();
    my $hostname = $isWin ? $hdbInstance->get_host() : undef;
    # $hostname == undef => use unix domain socket, no credentials required
    my $sapControl = new SDB::Install::SAPControl ($hostname,
                                                   $hdbInstance->get_nr(),
                                                   $username,
                                                   $password,
                                                   $instconfig->isUseHttps(),
                                                   $instconfig->getValue ('SSOCertificate'));
    my $isRunning = $sapControl->isRunning(undef, $msglst);

    if (!defined $isRunning) {
        $self->getMsgLst()->addWarning ("Could not check running service.", $sapControl->getErrMsgLst());
    }

    return $isRunning;
}

sub tryHana2PersistenceMigrationHelper{
    my ($self, $instconfig, $hdbInstance, $extractionRootOfHana2SW) = @_;
    if (!defined $extractionRootOfHana2SW){
        $extractionRootOfHana2SW = $hdbInstance->get_instanceExeDir ();
    }

    my $migrationScriptLocation = $extractionRootOfHana2SW.$path_separator.'python_support'.$path_separator.'Hana2PersistenceMigrationHelper.sql';
    my $systemUser = $instconfig->getValue('SystemUser');
    my $sqlSysPw = $instconfig->getValue('SQLSysPasswd');

    if (!defined $systemUser || !defined $sqlSysPw){
        return 1;
    }

    my $helper = new SDB::Install::Sql::Hana2PersistenceMigrationHelper();
    my $proMsg = $self->getMsgLst()->addProgressMessage ("Migrating table persistencies for HANA2...");
    $proMsg->setEndTag("Migrating table persistencies for HANA2");
    $helper->setMsgLstContext([$proMsg->getSubMsgLst()]);

    return $helper->executeHana2PersistenceMigrationHelper(
        $migrationScriptLocation,
        undef, # user store key, not supported by hdbupd
        $instconfig->getValue('SystemUser'),
        $instconfig->getValue('SQLSysPasswd'),
        $hdbInstance->getSqlHost (),
        $hdbInstance->get_nr (),
        $hdbInstance->isMultiDb (),
        $hdbInstance
    );
}




sub restart_after_upgrade{
    my ($self,$instconfig) = @_;
    require SDB::Install::SAPControl;
    my $password = $instconfig->getValue ('Password');
    my $nostart = $instconfig->getValue ('NoStart');
    my $trexes = $self->getNewDBInstances ();
    my $runHana2PersistenceMigrationHelperAfterRestart = 0;

    if($instconfig->isa ('SDB::Install::Configuration::Upgrade')) {
        my $srcVersionString = $instconfig->getStartVersion();
        my $srcVersion;
        my $maxVersion;
        if (defined $srcVersionString){
             $srcVersion = new SDB::Install::Version (split ('\.', $srcVersionString));
             $maxVersion = new SDB::Install::Version (2, '00', '00', '00');
        }

        if(defined $srcVersion && $maxVersion->isNewerThan ($srcVersion)) {
            my $Phase = $instconfig->getValue ('Phase');
            if ($instconfig->isPhase($gPhaseOfflineOption) || $instconfig->isPhase($gPhaseConfigureOption)) {
                $runHana2PersistenceMigrationHelperAfterRestart = 1;
            } else {
                my $rc = $self->tryHana2PersistenceMigrationHelper ($instconfig, $trexes->[0], $self->get_newVersionExeDir ());
                if (defined $rc && $rc == 2){
                    # $rc == 2 => database not running
                    $runHana2PersistenceMigrationHelperAfterRestart = 1;
                }
            }
        }
    }

    if($instconfig->isPrepareUpdate()) {
        $instconfig->{$gPhaseOptionPerskey} = $gPhasePrepareOption;
        $instconfig->setStep ($STEP_UPGRADE_STOP_INSTANCE);
        return 1;
    }

    my $resume_step = $instconfig->getStep ();
    $resume_step = int $resume_step;
    
    my $progress_handler = $self->GetProgressHandler ();
    
    if (defined $progress_handler){
        my @progress_steps = ('Stopping System',($isWin ? ('Copying Binaries') : ('Switching used software to new version')), ($nostart ? () : 'Starting System'));
        if ($instconfig->getValue('ImportContent')) {
            push @progress_steps, 'Importing delivery units';
        }
        $progress_handler->InitProgress (scalar @progress_steps,0,\@progress_steps);
    }
    
    my $username =  $self->{_user}->getSidAdm()->getname ();
    
    
    my ($msg);
    
    if ($resume_step <= $STEP_UPGRADE_STOP_INSTANCE){
    
        $instconfig->setStep ($STEP_UPGRADE_STOP_INSTANCE);
        delete $instconfig->{$gPhaseOptionPerskey};
        my $processListMsgLst = new SDB::Install::MsgLst();
        my $isRunning = $self->_isRunning($instconfig, $username, $password, $processListMsgLst);
        if (!defined $isRunning || $isRunning) {
            $msg = $self->AddProgressMessage ('Stopping system...');
            $self->SetFormatInfo ($msg, 'h1', 'Stop system');
            my $ignore_hosts = (defined $instconfig->getIgnore('check_hosts'));
            my $useHttps = $instconfig->isUseHttps();
            my $sso_cert = $instconfig->getValue ('SSOCertificate');

            foreach my $trex (@$trexes){
                $trex->setMsgLstContext ([$msg->getSubMsgLst ()]);
                if (!defined $trex->stopAllHosts (
                                    $username,
                                    $password,
                                    $ignore_hosts,
                                    $instconfig->getTimeout ('stop_instance'),
                                    $useHttps,
                                    $sso_cert)){

                    $self->PushError ("Cannot stop system", $trex);
                    if (defined $progress_handler){
                        $progress_handler->StepFinished (1);
                    }
                    return undef;
                 }
            }
        }
        else{
            $self->getMsgLst()->addMessage("$gProductNameEngine isn't runnning.", $processListMsgLst);
        }
    }

    if (defined $progress_handler){
        $progress_handler->StepFinished ();
    }
    
    if ($resume_step <= $STEP_UPGRADE_SWITCH_VERSION){
        $instconfig->setStep ($STEP_UPGRADE_SWITCH_VERSION);

        my $oldVersionLink = File::Spec->catfile($self->get_platformDir(), 'hdb');
        my $hdbnsutil = File::Spec->catfile($self->get_platformDir(), readlink($oldVersionLink), 'hdbnsutil');
        return undef if(!setFilePermissions(0440, $hdbnsutil, $self->getMsgLst()));

        $self->getMsgLst()->addMessage("Check for runnning hdbnsutil process...");
        return undef if(!terminateProcessAfterTimeout($self->getUID(), 'hdbnsutil', 60, $self->getMsgLst()));

        if (!defined $self->activateNewVersion ()){
            if (defined $progress_handler){
                $progress_handler->StepFinished (1);
            }
            return undef;
        }
        return undef if(!setFilePermissions(0551, $hdbnsutil, $self->getMsgLst()));
    }
    
    if (defined $progress_handler){
        $progress_handler->StepFinished ();
    }

    if (!defined $self->_linkRedHatCompatCPlusPlusLib ()){
        return undef;
    }

    if (!$self->instUpdSaphostagent ($instconfig)){
        return undef;
    }

    my $trex = $trexes->[0];

    if ($resume_step <= $STEP_UPGRADE_HANDLE_HOSTS) {
        $instconfig->setStep ($STEP_UPGRADE_HANDLE_HOSTS);
        my $srcVersion = $instconfig->isa ('SDB::Install::Configuration::Upgrade') ? $instconfig->getStartVersion() : undef;
        if (defined $srcVersion){
            $trex->setMsgLstContext($self->getMsgLstContext());
            if (!defined $trex->migrateParameters ($srcVersion)){
                return undef;
            }
        } else{
            $self->getMsgLst ()->addMessage (
            'Skipping database parameter migration due to known source version');
        }

        $self->deleteHdbParamFiles ();

        $msg = $self->AddMessage ("Updating profile");
        $self->SetFormatInfo ($msg, 'h1', 'Update profile');
        if (!defined $trex->updateProfile ($msg->getSubMsgLst (),
                                           0644, # mode 'rw-r--r--'
                                           $self->getUID(),
                                           $self->getGID())) {
            $self->AddWarning ("Cannot update profile", $msg->getSubMsgLst ());
        }
        $trex->updateInstanceSapprofileInis();
    }

    if ($resume_step <= $STEP_UPGRADE_CONFIGURE_INSTANCE) {
        $instconfig->setStep($STEP_UPGRADE_CONFIGURE_INSTANCE);
        if ($instconfig->isPhase($gPhaseOfflineOption)) {
            $instconfig->{$gPhaseOptionPerskey} = $gPhaseOfflineOption;
            return 1;
        }

        delete $instconfig->{$gPhaseOptionPerskey};

        my $rc   = 1;

        #
        # restart local sapstartsrv
        #
        require SDB::Install::SAPSystemUtilities;
        my $util = new SDB::Install::SAPSystemUtilities();
        if (!$util->startSapstartsrv($self, $trex, $instconfig)) {
            $rc = undef;
        }

        $trex->setMsgLstContext ($self->getMsgLstContext ());
        if (!defined $trex->handleSecureStore($instconfig)){
            return undef;
        }
        $trex->resetError ();
        $instconfig->setMsgLstContext ([$self->getMsgLst ()]);
        my $iscMode = $instconfig->getValue ('ISCMode');
        if (defined $iscMode) {
            if (!defined $instconfig->handleGlobalIniISCMode($iscMode)) {
                # if 0 is returned, the mode is already set
                return undef; # error occurred
            }
        }

        if ($instconfig->getValue('StripSymbols')) {
            $trex->stripSymbols();
        }

        #
        # upgrade remote hosts (restarts remote sapstartsrv) in case of scope=system
        #
        if ($rc && !$instconfig->tryUpgradeRemoteHosts()) {
            $self->AddError (undef, $instconfig);
            $rc = undef;
        }

        if (defined $progress_handler){
            $progress_handler->StepFinished (!defined $rc);
        }

        if (!defined $rc) {
            return undef;
        }

        $self->removePersFiles($instconfig);
    }
    my $called_by_hdblcm = defined $instconfig->{options}->{action_id};
    my $importContentWarning;
    if (!$nostart && $instconfig->getValue('ImportContent')){
        my $sidadm = $self->{_user}->getSidAdm()->getname();
        $importContentWarning =
        ['###############################################################################',
        'WARNING: The import of delivery units is skipped due to a failed database restart!',
        'Please perform the import manually when the database is online by running this',
        "command as the $sidadm user:",
        '\'' .join ($path_separator, $self->getUsrSapSid (), qw (SYS global hdb install bin hdbupdrep)). '\'',
        '###############################################################################'
       ];
    }

    my $printMissingImportContentWarning = sub {
         if (defined $importContentWarning){
            foreach my $line (@$importContentWarning){
                $self->getMsgLst ()->addProgressMessage ($line, undef, undef, $called_by_hdblcm);
            }
        }
    };

    if (!$nostart && ($resume_step <= $STEP_UPGRADE_START_INSTANCE)
        && $trex->exists_remote_host()
        && !defined $instconfig->getRemoteHosts()) {
        
        #
        # restart sapstartsrv on remote hosts in case of scope=instance
        #
        
        my @sapcontrols;
        my $sapcontrol;
        my $topmsg = $self->AddMessage ("Restarting sapstartsrv on remote hosts...");

        my $autoStartEnabled = $trex->isAutoStartEnabledRemote();

        if ($autoStartEnabled){
            my $msg = $self->AddSubMessage ($topmsg, "Switching Autostart parameter off");
            $trex->setMsgLstContext ([$msg->getSubMsgLst ()]);
            if (!defined $trex->enableAutoStartRemote(0)){
                $self->AddError ("Cannot switch off Autostart of instance", $trex);
                if (defined $progress_handler){
                    $progress_handler->StepFinished(1);
                }
                $printMissingImportContentWarning->();
                return undef;
            }
        }

        my $rc = 1;
        my $msg;
        my $useHttps = $instconfig->isUseHttps();
        my $sso_cert = $instconfig->getValue ('SSOCertificate');
        foreach my $host (@{$trex->get_hosts ()}){
            if (!defined $host){
                next;
            }
            $msg = $self->AddSubMessage ($topmsg, "Sending restart request to host '$host'...");
            $sapcontrol = new SDB::Install::SAPControl ($host,
                                                        $trex->{_nr},
                                                        $username,
                                                        $password,
                                                        $useHttps, $sso_cert);
            $sapcontrol->set_callback($self->{_f_callback});
            $sapcontrol->setMsgLstContext ([$msg->getSubMsgLst ()]);
            if (!$sapcontrol->restartService ()){
                if ($!{ECONNREFUSED}){
                    #
                    # ECONNREFUSED can be caused by autostart mechanism of sapstartsrv
                    # => sapstartsrv restarts automatically when it notices, that it's own
                    # image file has been changed
                    #
                    $self->AddWarning ("Cannot send restart request to sapstartsrv on host '$host'", $sapcontrol);
                    $self->AddMessage (" => Assuming automatic sapstartsrv restart");
                }
                else{
                    $self->PushError ("Cannot restart sapstartsrv on host '$host'", $sapcontrol);
                    $rc = undef;
                }
            }
            else{
                push @sapcontrols, $sapcontrol;
            }
        }

        my $waitSecondsAfterStopDetection = 1;
        foreach my $sapcontrol (@sapcontrols){
            $msg = $self->AddSubMessage ($topmsg, "Waiting for '$sapcontrol->{_host}'...");
            $sapcontrol->setMsgLstContext ([$msg->getSubMsgLst ()]);
            if (!$sapcontrol->waitForService ($instconfig->getTimeout ('start_service'),undef, $waitSecondsAfterStopDetection)){
                $self->PushError ("Wait for sapstartsrv on host '$sapcontrol->{_host}' failed", $sapcontrol);
                $rc = undef;
            }
            if ($waitSecondsAfterStopDetection){
                # wait just for the 1st host
                $waitSecondsAfterStopDetection = 0;
            }
        }

        if ($autoStartEnabled){
            $msg = $self->AddSubMessage ($topmsg, "Switching Autostart parameter on");
            $trex->setMsgLstContext ([$msg->getSubMsgLst ()]);

            if (!defined $trex->enableAutoStartRemote(1)){
                $self->AddError ("Cannot switch on Autostart of instance", $trex);
                $rc = undef;
            }
        }

        if (!defined $rc) {
            if (defined $progress_handler){
                $progress_handler->StepFinished(1);
            }
            $printMissingImportContentWarning->();
            return undef;
        }
    }

    my $rc = 1;

    if (!$nostart && $resume_step <= $STEP_UPGRADE_START_INSTANCE){
    #if (!$nostart){

        #
        # start system
        #
        if ($instconfig->isPhase($gPhaseConfigureOption)) {
            $instconfig->setStep ($STEP_UPGRADE_START_INSTANCE);
            $instconfig->{$gPhaseOptionPerskey} = $gPhaseConfigureOption;
            return 1;
        }
        delete $instconfig->{$gPhaseOptionPerskey};
        $self->removePersFiles($instconfig);

        my $isRunning = $self->_isRunning($instconfig, $username, $password);

        if (!defined $isRunning || !$isRunning ) {
            $msg = $self->AddProgressMessage ("Starting system...");
            $self->SetFormatInfo ($msg, 'h1', 'Starting system');
            my $useHttps = $instconfig->isUseHttps ();
            my $sso_cert = $instconfig->getValue ('SSOCertificate');
            foreach my $trex (  @$trexes){
                $trex->setMsgLstContext ([$msg->getSubMsgLst ()]);
                if (!defined $trex->startAllHosts (
                            $username,
                            $password,
                            $instconfig->getTimeout ('start_instance'),
                            $useHttps, $sso_cert)){

                    $self->PushError ("Cannot start system", $trex);
                    if (defined $progress_handler){
                        $progress_handler->StepFinished(1);
                    }
                    $printMissingImportContentWarning->();
                    return undef;
                }
            }
        }
        if ($runHana2PersistenceMigrationHelperAfterRestart){
            $self->tryHana2PersistenceMigrationHelper ($instconfig, $trexes->[0]);
        }

        if (defined $progress_handler){
            $progress_handler->StepFinished (!$rc);
        }
    }

    if ($resume_step <= $STEP_UPGRADE_IMPORT_CONTENT) {
        if ($instconfig->getValue('ImportContent')) {
            $instconfig->setStep ($STEP_UPGRADE_IMPORT_CONTENT);

            $trex->setMsgLstContext ([$self->getMsgLst ()]);
            ## during upgrade:
            my $import_rc = $trex->importDeliveryUnits($instconfig, undef, undef, undef, 1);
            if (!defined $import_rc){
                $self->AddError ('Import of delivery units failed', $trex);
                if (defined $progress_handler){
                    $progress_handler->StepFinished(1);
                }
                return undef;
            }
            my $plugin_import_rc = $trex->importPluginDeliveryUnits ($instconfig);
            if (!defined $plugin_import_rc){
                $self->setErrorMessage ('Import of plugin delivery units failed', $trex->getErrMsgLst ());
                if (defined $progress_handler){
                    $progress_handler->StepFinished(1);
                }
                return undef;
            }
            if (!$import_rc || !$plugin_import_rc){
                $self->AddWarning ('Import of delivery units failed', $trex);
                $rc = 0;
            }
     
            if (defined $progress_handler){
                $progress_handler->StepFinished (!$rc);
            }
        }

        if($rc) {
            $self->removePersFiles($instconfig);
        }
    }

    return $rc;
}

sub prepareVersionSwitch{
    if ($isWin){
        return 1;
    }
    my ($self) = @_;
    require SDB::Install::ServerVersionLinker;

    $self->{_versionLinker} = new SDB::Install::ServerVersionLinker (
        $self->get_platformDir (),
        undef,
        'hdb',
        'HDB_'.$self->{kit_versionid},
        $self->getUID(),
        $self->getGID());

    if (!defined $self->{_versionLinker}->prepareNewVersion()){
        $self->AddError (undef, $self->{_versionLinker});
        $self->AddMessage (undef, $self->{_versionLinker});
        return undef;
    }
    $self->AddMessage (undef, $self->{_versionLinker});
    return 1;
}


#
# search for '*_new' plugins, which has to be activated
#

sub getNewPlugins{
    my ($self, $pluginsFolder) = @_;
    if ($isWin){
		$pluginsFolder .= '_new';
    }

    if (!-d $pluginsFolder){
        return [];
    }

    if (!opendir (DH, $pluginsFolder)){
        $self->AddError ("Cannot open directory '$pluginsFolder': $!");
        return undef;
    }

    my @subdirs;

    if ($isWin){
		#
		# Win: find subdirectories in plugins_new dir
		#
		@subdirs = grep {!/^\.{1,2}$/ && -d "$pluginsFolder$path_separator$_"} readdir (DH);
    }
    else{
		#
		# Linux: find subdirectories '*_new' in plugins dir
		#
		@subdirs = grep {/_new$/ && !/^\.{1,2}$/ && -d "$pluginsFolder$path_separator$_"} readdir (DH);
    }
    closedir (DH);


    if (!@subdirs){
        return [];
    }
    my @result;
    #
    # check subdirectories for a valid installation
    #
    my ($key,$inst);
    foreach my $dir (@subdirs){
        $inst = new SDB::Install::Installation ($pluginsFolder . $path_separator . $dir);
		if ($isWin) {
			$key = $dir;
		} else {
			($key) = ($dir =~ /(.*)_new$/);
		}
        #
        # check if the plugin subdirectory matches the product key
        #
        if ($inst->getProductKey() ne $key){
            next;
        }
        push @result, $inst;
    }
    return \@result;
}


sub activateNewVersion{
    my ($self, $customPluginsPath) = @_;

    my $has_filter = $self->{kit_has_filter_package};
    my $instance = $self->getNewDBInstances ()->[0];
    if ($isWin){
		if (!defined $self->activateServerPlugins($customPluginsPath)){
			return undef;
		}
        if (!defined $instance->runSapCpe()){
            $self->AddError (undef, $instance);
            $self->AddMessage (undef, $instance);
            return undef;
        }
        my $instance_exeDir = $instance->get_instanceExeDir ();
        if (!$has_filter && -d "$instance_exeDir\\filter"){
            deltree ("$instance_exeDir\\filter");
        }
        
        $self->AddMessage (undef, $instance);
        return 1;
    }

    #
    # activate server binaries
    #

    my $versionLinker = $self->{_versionLinker};
    
    if (!defined $versionLinker){
        require SDB::Install::ServerVersionLinker;
        $versionLinker = new SDB::Install::ServerVersionLinker (
        $self->get_platformDir (),
        undef,
        'hdb',
        'HDB_'.$self->{kit_versionid},
        $self->getUID(),
        $self->getGID());
    }

    $versionLinker->setSyncLinksInDir ('plugins');

    if (!defined $versionLinker->activateNewVersion()){
        $self->AddError (undef, $versionLinker);
        $self->AddMessage (undef, $versionLinker);
        return undef;
    }
    $self->AddMessage (undef, $versionLinker);

    #
    # reload manifest
    #
    $self->getManifest (1);
    $instance->getManifest (1);

    #
    # activate server plugin binaries
    #
    if (!defined $self->activateServerPlugins($customPluginsPath)){
		return undef;
    }

	#
	# deactivate plugins with wrong version in case of ignore=check_plugin_dependencies
	#
	my $instconfig = $self->getConfiguration();
	if (defined $instconfig) {
		if ($instconfig->getIgnore('check_plugin_dependencies')) {
			if (!$self->deactivateOutdatedServerPlugins()){
				$self->AddMessage ("Deactivating of outdated plugins failed");
			}
		}
	}

    return 1;
}

sub deactivateOutdatedServerPlugins{
	my ($self, $packageManager) = @_;
	my $instance = $self->getNewDBInstances ()->[0];
	my $activePlugins = $instance->getPluginInstallations(1);

	if (!defined $activePlugins || !@$activePlugins){
		return 1;
	}

	my $msg = $self->AddProgressMessage ("Checking for outdated server plugins...");

	if (!$packageManager){
		$packageManager = $self
	}

	my $outdated_plugins = {};
	my $rc = 1;

	if (defined $activePlugins){
		foreach my $plugin (@$activePlugins){
			$self->AddSubMessage ($msg, "Checking '".$plugin->getProductName()."'");
			if (!$plugin->checkComponentDependency ($packageManager)){
				$outdated_plugins->{$plugin->getProductKey()} = $plugin;
				$self->AddSubMessage ($msg, "Plugin '".$plugin->getProductName()."' is outdated");
			}
		}
	}

	if ($isWin) {
		# Windows
		# to be implemented
	} else {
		# Linux
		my ($plugin, $link, $link_target);
		foreach my $key (keys %$outdated_plugins){
			$plugin = $outdated_plugins->{$key};
			$self->AddSubMessage ($msg, "Deactivating '".$plugin->getProductName()."'");
			$link = $instance->get_pluginsDir() . $path_separator . $key;
			if (-l $link) {
				$link_target = readlink($link);
				$self->AddSubMessage ($msg, "Removing symbolic link $link -> $link_target");
				if (!unlink($link)) {
					$rc = 0;
					$self->AddSubMessage ($msg, "Removal of symbolic link $link -> $link_target failed");
				} else {
				}
			} else {
				$rc = 0;
				$self->AddSubMessage ($msg, "Cannot remove symbolic link $link because it does not exist or is not a link");
			}
		}
	}

	return $rc;
}

sub activateServerPlugins{
	my ($self, $customPluginsPath) = @_;
	my $rc = 1;
    my $versionLinker = $self->{_versionLinker};
	my $instance = $self->getNewDBInstances()->[0];
    my $pluginsDir = defined $customPluginsPath ? $customPluginsPath : $self->get_pluginsDir();

    my $plugins = $self->getNewPlugins($pluginsDir);

    if (defined $plugins && @$plugins){
        my $msg = $self->AddMessage ("Activating plugins");
        my $pluginsmsglst = new SDB::Install::MsgLst();
        foreach my $plugin (@$plugins){
            my $mf = $plugin->getManifest();
            if (!$plugin->checkComponentDependency ($self)){
                $pluginsmsglst->AddMessage ("Skipping activation of plugin '".$plugin->getProductName."' due to dependency error", $plugin);
                if ($mf) {
					my $dep = $mf->getValue('required-components');
					if ($dep && @$dep){
						$dep = join('|', @$dep);
						$pluginsmsglst->AddMessage ("Plugin requires: '$dep'", $plugin);
						if ($self->GetVersion()) {
							$pluginsmsglst->AddMessage ("Server version: " . $self->GetVersion(), $plugin);
						}
					}
				}
                next;
            }
            my $msg = $pluginsmsglst->AddMessage ("Processing plugin '".$plugin->getProductName."'");
			my $newPluginDir;
            if ($isWin){
				my $key = $plugin->getProductKey ();
				my $oldPluginsDir = $self->get_pluginsDir() . '_old';
				my $activePluginsDir = $self->get_pluginsDir();

				my $old =  $oldPluginsDir . $path_separator . $key;
				my $new = $newPluginDir = $self->get_pluginsDir()  . '_new' . $path_separator . $key;
				my $active = $activePluginsDir . $path_separator . $key;
				if (-e $old){
					my $errlst = new SDB::Install::MsgLst ();
					if (!defined deltree ($old,$errlst)){
						$self->AddError (undef, $errlst);
						return undef;
					}
				}

				if (-e $active){
					if (!-e $oldPluginsDir){
						my $cfg = {};
						if (!defined makedir ($oldPluginsDir, $cfg)){
							$self->AddError (undef, $cfg);
							return undef;
						}
					}

					my $errlst = new SDB::Install::MsgLst ();
					$self->AddMessage ("Moving '$active' to '$old'");
					if (!copy_tree ($active, $old, $errlst)){
						$self->AddError ("Cannot copy '$active' => '$old'", $errlst);
						return undef;
					}
					if (!deltree ($active, $errlst)){
						$self->AddError ("Cannot delete '$active'", $errlst);
						return undef;
					}
				}

				if (!-e $activePluginsDir){
					my $cfg = {};
					if (!defined makedir ($activePluginsDir, $cfg)){
						$self->AddError (undef, $cfg);
						return undef;
					}
				}
				my $errlst = new SDB::Install::MsgLst ();
				$self->AddMessage ("Moving '$new' to '$active'");
				if (!copy_tree ($new, $active, $errlst)){
					$self->AddError ("Cannot rename '$new' => '$active'", $errlst);
					return undef;
				}
				if (!deltree ($new, $errlst)){
					$self->AddError ("Cannot delete '$new'", $errlst);
					return undef;
				}
			}
			else{
				# Linux
	            $versionLinker = new SDB::Install::ServerVersionLinker (
	                $pluginsDir,
	                $instance->get_pluginsDir(),
	                $plugin->getProductKey (),
	                undef,
	                $self->getUID(),
	                $self->getGID());
                if (!defined $customPluginsPath){
	                $versionLinker->setRelativeBaseDir (join ($path_separator, qw (.. .. plugins)));
                }
                $versionLinker->setMsgLstContext ([$msg->getSubMsgLst()]);
                if (!defined $versionLinker->activateNewVersion ()){
                    $self->PushError (undef, $versionLinker);
                    $rc = undef;
                    next;
                }
                $newPluginDir = $instance->get_pluginsDir() . $path_separator . $plugin->getProductKey ();
            }
			$plugin->setInstallationPath ($newPluginDir);

			# copy 3rd party libs
			if (!defined $self->copyLibraries($plugin)) {
				$self->PushError ("Cannot copy libraries for plugin");
				$rc = undef;
				next;
			}

            # execute commands from XML file
            my $cmds = $plugin->getExecuteCommandsAtActivation();

			if (defined $cmds && @$cmds){
				my $exec_rc;
				foreach my $cmd (@$cmds){
					my $cfg = {};
					my $cmdArray;

					if (defined $cmd->{ExecArgs}) {
						$cmdArray = [split (',', $cmd->{ExecArgs})];
					} else {
						$cmdArray = $cmd->{ExecArg}
					}
					$exec_rc = exec_program ($cmd->{ExecCommand},$cmdArray,$cfg);
					if ((!defined $exec_rc || $exec_rc != $cmd->{ReturnCodeOK}) && !$cmd->{IgnoreError}){
						$self->PushError ('External program failed', $cfg);
						$self->AddSubMsgLst ($msg, $cfg);
						$rc = undef;
						next;
					}
					$self->AddSubMsgLst ($msg, $cfg);
				}
			}

        }
        $self->AddSubMsgLst($msg, $pluginsmsglst);
    }
	return $rc;
}

sub copyLibraries{
    my ($self, $plugin) = @_;

    return undef if (!defined $plugin);

    my $mf = $plugin->getManifest();
    my $thirdPartyLibs = $mf->getThirdPartyLibs();
    if ($thirdPartyLibs) {
        my $destDir = File::Spec->catfile($self->get_globalTrexDir(), 'plugins', '3rd_party_libs');
        return undef if (!$self->_copyLibrariesSet($plugin, $thirdPartyLibs, $destDir));
    }

    my $firstPartyLibs = $mf->getFirstPartyLibs();
    if ($firstPartyLibs) {
        my $destDir = File::Spec->catfile($self->get_globalTrexDir(), 'plugins', '1st_party_libs');
        return undef if (!$self->_copyLibrariesSet($plugin, $firstPartyLibs, $destDir));
    }
    return 1;
}

sub _copyLibrariesSet {
    my ($self, $plugin, $libsHash, $destDir) = @_;

    my $cfg = {
        uid => $self->getUID(),
        gid => $self->getGID(),
        createdir => 1,
        dir_mode => 0750,
    };
    my $srcDir = $plugin->GetProgramPath();

    foreach my $libName (keys %$libsHash) {
        my $filenames = $libsHash->{$libName}->{'filenames'};
        my @libFiles = split(/\,/, $filenames);

        foreach my $libFile (@libFiles) {
            trim(\$libFile);
            my $src = $srcDir . $path_separator . $libFile;
            my $dest = $destDir . $path_separator . $libFile;

            $self->AddMessage("Copying library file '$src' to '$dest'");
            if (!defined copy_file ($src, $dest, $cfg, 1)){
                $self->AddError ("Cannot copy library file '$src' to '$dest'", $cfg);
                return undef;
            }
        }
    }
    return 1;
}

sub createRoot{
    my ($self, $msglst, $wantedUID, $wantedGID) = @_;
    if (!defined $msglst){
        $msglst = $self;
    }
    
    if ($isWin){
        my $cfg = {};
        if (!-x $self->{_usrSapSid}){
            require SAPDB::Install::System::Win32::API;
            $msglst->AddMessage ("Creating '$self->{_usrSapSid}");
            if (!defined makedir ($self->{_usrSapSid},$cfg)){
                $msglst->AddError ("Cannot create '$self->{_usrSapSid}'", $cfg);
                return undef;
            }
        }
        
        my ($systemRoot) = 
                ($self->{'_usrSapSid'} =~ /^(.*)$re_path_separator/);
                    
        my ($error,$dir,$comment);
        my %comments = {
            'saploc' => "Used by SAP Server to access local information for system '$self->{_sid}'",
            'sapmnt' => "Used by SAP Server to access system specific information for system '$self->{_sid}'"
        };
            
        foreach my $share ('saploc', 'sapmnt'){
            ($error,$dir,$comment) = SAPDB::Install::System::Win32::API::ShareInfo ($share);
            if ($error){
                my $rc=SAPDB::Install::System::Win32::API::ShareAdd (
                    $share,
                    $systemRoot,
                    undef,
                    $comments{$share},
                );
                if ($rc != 0){
                    $msglst->AddError ("Failed to share '$systemRoot' as '$share', net api error: $rc");
                    return undef;
                }
                $self->{"${share}created"} = 1;
                $msglst->AddMessage ("Share '$share' => '$systemRoot' created");
            }
            else{
                $msglst->AddMessage ("Share '$share' => '$dir' already exists");
            }
        }       
        return 1;
        
    }
    else{
        my $cfg = {'mode' => 0755};
        if (-d '/usr/sap') {
            $msglst->AddMessage("Checking permissions for the existing '/usr/sap' directory");
            my @stat = stat('/usr/sap');
            my $mode = $stat[2];
            my $gid  = $stat[5];
            my $minPerm = 0755;
            if ($gid == getgrnam($gSapsysGroupName)) {
                $minPerm = 0750;
            }
            if (($mode & $minPerm) != $minPerm) {
                my $targetPerm = $mode | $minPerm;
                my $msgStr = "Existing directory '/usr/sap' has invalid permissions. Adding missing permissions";
                $msglst->AddMessage($msgStr);
                if (! chmod($targetPerm, '/usr/sap')) {
                    $msglst->AddError("Could not change premissions for the '/usr/sap' directory");
                    return undef;
                }
            }
        } else {
            $msglst->AddMessage ("Creating '/usr/sap'");
            if (!defined makedir ('/usr/sap',$cfg)){
                $msglst->AddError ("Cannot create directory '/usr/sap'",
                    $cfg);
                return undef;
            }
        }
        
        my $path;
        my $link;
        $cfg->{uid} = (defined $wantedUID) ? $wantedUID : $self->getUID();
        $cfg->{gid} = (defined $wantedGID) ? $wantedGID : $self->getGID();
        if ($self->{_target}){
        
            $path = $self->{_target} . $path_separator .
                $self->{_sid};
                #. $path_separator . 'home';
            #$link = $self->{_usrSapSid} . $path_separator .
            #   $path_separator . 'home';
            if (!-e $path){
                if (!defined makedir ($path, $cfg)){
                    $msglst->AddError ("Cannot create directory '$path'",
                        $cfg);
                    return undef;
                }
            }
        }
        if (!-e $self->{_usrSapSid}){
            $msglst->AddMessage ("Creating directory '$self->{_usrSapSid}'");
            if (!makedir ($self->{_usrSapSid}, $cfg)){
                $self->AddError ("Cannot create directory '$self->{_usrSapSid}'", $cfg);
                return undef;
            }
        }
        #if ($link){
        #   if (!-l $link && !symlink ($path, $link)){
        #       $msglst->AddError ("Cannot create symlink \'$link\': $!");
        #           return undef;
        #   }
        #   lchown ($link);
        #}

        #if (!-e $self->{_usrSapSid}){
        #   my ($dest) = ($self->get_globalSidDir =~ /(.*)$re_path_separator/);
        #   $msglst->AddMessage ("Creating link '$self->{_usrSapSid}' => '$dest'");
        #   
        #   if (!symlink ($dest, $self->{_usrSapSid})){
        #       $msglst->AddError ('Cannot create symlink \'' .
        #           $self->{_usrSapSid}. "': $!");
        #       return undef;
        #   }
        #   lchown ($self->getUID,$self->getGID,$self->{_usrSapSid});
        #}
    }
    return 1;
}

sub chownRoot{
    my ($self) = @_;
    if ($isWin){
        if (! -e $self->{_globalSidDir}){
            makedir ($self->{_globalSidDir});
        }
        #
        # set permissions to SYS directory
        #
        
        $self->setPermissions($self->{_globalSidDir},1);
    
        my $msglst = new SDB::Install::MsgLst();
    
        my $host = hostname();
    
        foreach my $share ('saploc','sapmnt'){
            #if ($self->{"${share}created"}){
                $self->{_user}->possessShare ("\\\\$host\\$share",$msglst);
            #}
        }
        $self->AddMessage (undef,$msglst);
        return 1;
    }
    #my $home = $self->{_usrSapSid} . '/home';
    chown ($self->getUID,$self->getGID,$self->{_globalSidDir});
    chown ($self->getUID,$self->getGID,$self->{_usrSapSid});
    #chown ($self->getUID,$self->getGID, $home);
    
    #if (-l $home){
    #   lchown ($self->getUID,$self->getGID, $home);
    #}
    
    return 1;
}


sub ExistsInstanceNumber{
    if (!defined $systems){
        CollectSAPSystems ();
    }
    if (!defined $instance_numbers || !defined $instance_numbers->{$_[0]}){
        return undef;
    } 
    return $instance_numbers->{$_[0]};
}

sub OccupiedInstanceNumbers{
    if (!defined $systems){
        CollectSAPSystems ();
    }
    if (!defined $instance_numbers){
        return undef;
    } 
    return [sort keys %$instance_numbers];
}



sub installSapinit{
    my ($self) = @_;

    my $sapinit = basename ($SAPINIT_SCRIPT);

    my $dir = $self->get_globalSAPHostAgentSetupDir();
    my $installsapinit = join($path_separator, $dir, 'installsapinit.sh');

    local %ENV = %ENV;
    $ENV{PATH} = join (':', '/sbin', '/usr/sbin', $ENV{PATH});
    undef $ENV{LD_LIBRARY_PATH};

    my $hasChkconfig = defined which('chkconfig');

    if (!$hasChkconfig && ! -f $SAPINIT_SCRIPT){
        # without chkconfig installsapinit.sh doesn't work (e.g. on Ubuntu)
        # => copy sapinit directly and use update-rc.d instead
        my $sapinit = $dir . $path_separator . 'sapinit';
        my $cfg = {};
        my $msg = $self->getMsgLst()->addMessage("Copying file '$sapinit' => '$SAPINIT_SCRIPT'");
        $msg->getSubMsgLst()->injectIntoConfigHash($cfg);
        if (!defined copy_file ($sapinit, $SAPINIT_SCRIPT, $cfg)){
            $self->setErrorMessage("Cannot copy file $sapinit' => '$SAPINIT_SCRIPT'", $msg->getSubMsgLst());
            return undef;
        }
    }

    if (-f $SAPINIT_SCRIPT){
        my $needsUpdate = $self->sapinitNeedsUpdate();
        if(not defined $needsUpdate) {
            return undef;
        }

        if(not $needsUpdate) {
            my $msg = $self->AddMessage ("Checking $sapinit");
            if ($hasChkconfig){
                my @check_options = ('--check', $sapinit);
                # check whether '--check' option is known
                {
                    $ENV{LC_MESSAGES} = 'en_US.UTF-8';
                    $ENV{LANG} = 'en_US.UTF-8';
                    my $output;
                    my $cfg = {'out' => \$output};
                    my $rc = exec_program ('chkconfig',\@check_options, $cfg);
                    if (!defined $rc){
                        $self->AddError ("Error checking $sapinit", $cfg);
                        return undef;
                    }
                    if ($rc != 0){
                        if ($output =~ m/unknown\s+option|wrong\s+usage/i){
                            @check_options = ('sapinit');
                        }
                    }
                }

                my $msglst = $msg->getSubMsgLst ();
                my $rc = exec_program ('chkconfig', \@check_options, $msglst);
                if (!defined $rc){
                    $self->AddError ("Error checking $sapinit", $msglst);
                    return undef;
                }
                if ($rc == 0){
                    $msglst->addMessage ('sapinit is already installed');
                    return 1;
                }
                $msglst->addMessage ('chkconfig reported an error');
                {
                    my $msg = $msglst->addMessage ('Trying to repair');
                    $msglst = $msg->getSubMsgLst();
                    $rc = exec_program ('chkconfig',[$sapinit, 'off'],$msglst);
                    $rc = exec_program ('chkconfig',[$sapinit, 'on'],$msglst);
                    $rc = exec_program ('chkconfig',\@check_options,$msglst);
                    if (!defined $rc){
                        $self->AddError ("Error repairing $sapinit", $msglst);
                        return undef;
                    }
                }
                if ($rc == 0){
                    $msglst->addMessage ("$sapinit repaired");
                    return 1;
                }
            }
            else{
                # no chkconfig
                my $outbuffer;
                my $cfg = {'outLines' => 1};
                my @update_rc_d_options = ($sapinit, 'defaults');
                $msg->getSubMsgLst ()->injectIntoConfigHash ($cfg);
                my $rc = exec_program ('update-rc.d', ['-n', @update_rc_d_options], $cfg);
                if (!defined $rc || $rc != 0){
                    $self->setErrorMessage ("Error checking $sapinit", $msg->getSubMsgLst ());
                    return undef;
                }
                my @missing_links = grep {/\s->\s/} @{$cfg->{'outLines'}};
                if (!@missing_links){
                    $msg->getSubMsgLst()->addMessage ("sapinit is already installed");
                    return 1;
                }
                $msg->getSubMsgLst()->addMessage ("update-rc.d reported an error");
                $msg = $msg->getSubMsgLst()->addMessage ("Trying to repair");
                $msg->getSubMsgLst ()->injectIntoConfigHash ($cfg);
                $rc = exec_program ('update-rc.d', \@update_rc_d_options, $cfg);
                if (!defined $rc || $rc != 0){
                    $self->setErrorMessage ("Error repairing $sapinit", $msg->getSubMsgLst ());
                    return undef;
                }
                return 1;
            }
        }
    }
    chdir ($dir);
    my $msg = $self->AddMessage ("Installing $sapinit");
    my $msglst = $msg->getSubMsgLst();
    my $rc = exec_program ($installsapinit,[], $msglst);
    if (!defined $rc || $rc){
        $self->AddError ("installsapinit.sh FAILED", $msglst);
        chdir ($gLogDir);
        return undef;
    }
    chdir ($gLogDir);
    return 1;
}

# we update /etc/init.d/sapinit (using installsapinit.sh) only if
# an older version is present. Currently (as of 8/2012), we detect 
# an old version by presence of the line /^PGMNAME=sapstart$/
sub sapinitNeedsUpdate{
    my ($self) = @_;
    if (!open (FD, $SAPINIT_SCRIPT)){
        $self->AddError ("Cannot open file '$SAPINIT_SCRIPT': $!");
        return undef;
    }
    my $retval = 0;
    while (my $line = <FD>){
        if($line =~ /^\s*PGMNAME=sapstart\s*$/) {
            $retval = 1;
            last;
        }
    }
    close (FD);
    return $retval;
}

sub getUID{
    if ($isWin || !defined $_[0]->{_user}){
        return undef;
    }
    $_[0]->{_user}->uidUx();
}

sub getGID{
    if ($isWin || !defined $_[0]->{_user}){
        return undef;
    }
    $_[0]->{_user}->gidUx();
}

sub getShmGID{
    if ($isWin){
        return undef;
    }
    $_[0]->{_user}->{shmgroup}->id();
}

sub createDefaultProfile{
    my ($self, $sapglobalhost) = @_;
    
    my $msg = $self->AddMessage ("Creating default profile...");
    my $msglst = $msg->getSubMsgLst ();

    my $prof = new SDB::Install::SAPProfiles ($self->get_ProfileDir (), $self->{_sid}, $sapglobalhost);
    $prof->setMsgLstContext ([$msglst]);
    my $rc = $prof->createDefaultProfile ();
    
    if (!defined $rc){
        $self->AddError ("Cannot create default profile", $prof);
        return undef;
    }
    if (!$isWin){
        $self->setPermissions ($prof->get_DefaultProfile (), 0, 0644,$msglst);
    }
        
    return 1;
}


sub saveConfig {
    my ($self, $customData, $deleteKeys) = @_;
    my %data = ();
    my $filePath = $self->getUserConfigFile();
    my $config = new SDB::Install::Config ($filePath);

    $customData = $customData || {};
    $deleteKeys = $deleteKeys || [];

    # Load current file content to allow persistent storage of additionally defined data too
    if(-f $filePath && defined($config->load(1))){
        %data = %{$config->getData()};
    }
    # Add custom data entries
    for my $key (keys(%{$customData})){
        $data{$key} = $customData->{$key};
    }
    # Remove selected key-value pairs if given
    for my $key (@{$deleteKeys}){
        next if(!exists($data{$key}));
        delete($data{$key});
    }
    $data{installation_path} = $self->get_target();
    if(!$isWin){
        $data{sapsys_groupid} = new SDB::Install::Group($gSapsysGroupName)->id();
        $data{sidadm_comment} = $self->{_user}->getSidAdm()->{comment};
        $data{sidadm_home} = $self->{_user}->getSidAdm()->{home};
        $data{sidadm_shell} = $self->{_user}->getSidAdm()->{shell};
        $data{sidadm_id} = $self->{_user}->uidUx();
    } elsif (! $self->{_user}->getSidAdm()->{bsname}->{local}) {
        $data{domain} = $self->{_user}->getSidAdm()->{bsname}->{first};
    }

    $config->setData (\%data);

    if (!defined $config->save()){
        $self->AddError ("Cannot save user configuration", $config);
        return undef;
    }
    if (!defined $self->setPermissions ($filePath,0,0640)){
        $self->AddError ("Cannot set '$filePath' permissions");
        return undef;
    }
    return 1;
}

sub getUserConfigFile{
    return join ($path_separator,
        $_[0]->get_globalTrexInstallSupportDir, 'cfg');
}


sub getUserConfig{
    my ($self, $msglst) = @_;
    if (!defined $msglst){
        $msglst = $self;
    }
    
    my $cfg_path = $self->getUserConfigFile ();
    my $config = new SDB::Install::Config ($cfg_path);

    if (!defined $config->load()){
        $msglst->AddError ("Cannot load user config '$cfg_path'", $config);
        return undef;
    }
    return $config->getData ();
}

sub setPermissions{
    my ($self, $path, $recursive, $permissions, $msglst) = @_;
    
    if (!defined $permissions){
        $permissions = 0750;
    }
    
    if (!defined $msglst){
        $msglst = $self->getMsgLst ();
    }
    elsif ($msglst->isa ('SDB::Install::BaseLegacy')){
        $msglst = $msglst->getMsgLst ();
    }
        
    if (!isAdmin()){
        return 1;
    }
    
    if ($isWin and ($path =~ /^\\\\/ or 0)){
        return 1;
    }
    my $msg = $msglst->addMessage ("Setting permissions of '$path' " .
        ($recursive ? 'recursively' : ''));
    $self->{_user}->setMsgLstContext ([$msglst]);
    if (! defined $self->{_user}->possess($path, $recursive, $permissions)){
        $self->AddWarning ("Cannot possess $path", $self->{_user});
    }
    return 1;
}


sub removeSAPSystem {
    my ($self, $deleteUser, $force, $instconfig) = @_;

    require SDB::Install::Configuration::NewDB;
    require SDB::Install::Configuration::Upgrade;

    if (!defined $deleteUser){
        $deleteUser = 1;
    }

    my $deleteUserHomeDir = $deleteUser;
    my $keepUserHomeDir   = 0;
    if ($deleteUserHomeDir && $instconfig->getValue('KeepUserHomeDir')) {
        $deleteUserHomeDir = 0;
        $keepUserHomeDir   = 1;
    }

    my $msg;
    $self->dumpProperties();
    if ($self->hasNewDB()){
        my $trexes = $self->getNewDBInstances ();
        foreach my $trex (@$trexes){
            $msg = $self->AddProgressMessage ("Removing $gProductNameEngine instance...");
            $self->SetFormatInfo ($msg, 'h1', 'Remove Instance');
            $trex->setMsgLstContext ([$msg->getSubMsgLst()]);
            if (!$instconfig->getValue('SkipModifySudoers') && !$self->removeSudoersEntries($trex)) {
                return undef;
            }
            if (!defined $trex->removeTrexInstance($self,
                                                   $force,
                                                   $instconfig,
                                                   $deleteUser)) {
                $self->PushError ("Cannot remove instance",$trex);
                return undef;
            }
        }
    }
    $self->collectInstances ();
    $msg = $self->AddMessage ("Remaining instances for system '$self->{_sid}'");
    my $submsglst = $msg->getSubMsgLst ();
    $self->SetFormatInfo ($msg, 'h1', 'Remaining Instances');

    my $userDeleted = 0;

    if (!%{$self->{_instances}}){
        $submsglst->AddMessage ('None');
        $self->removeGlobalDirs($deleteUserHomeDir);
        $self->removeUsrSapSid ($deleteUserHomeDir);
        if($deleteUser){
            $self->deleteUser(1, # force
                              $keepUserHomeDir);
            $userDeleted = 1;
        }
        
    }
    elsif (!$self->hasNewDB() && $self->get_integratedSystem()){
        $self->asString ($submsglst);
        $self->removeGlobalDirs (0); #deleteUser=False
    }
    else{
        $self->asString ($submsglst);
    }

    if (!$self->get_integratedSystem() && !@{$self->getLocalInstances()}){
        if ($isWin || !isSameFile ($self->{_usrSapSid},$self->{_globalSidDir})){
            $self->removeUsrSapSid ($deleteUserHomeDir);
            if($deleteUser && !$userDeleted){
                $self->deleteUser(1, # force
                                  $keepUserHomeDir);
            }
        }
    }

    # delete installer status files
    
    foreach my $currInstconfig (
                new SDB::Install::Configuration::NewDB (),
                new SDB::Install::Configuration::Upgrade ()){

        $currInstconfig->{current_sid} = $self->{_sid};
        if ($currInstconfig->pers_exists){
            $self->removePersFiles($currInstconfig);
        }
    }

    my $logdirs = $self->findLogFileLocations ();

    if (defined $logdirs){
        my $ownLogFileId = getFileId (\$gLogDir);
        foreach my $logdir (@$logdirs){
            if (isSameFile2 ($logdir, undef, $ownLogFileId)){
                # skipping own log
                next;
            }
            $self->AddMessage ("Deleting old logdir '$logdir'");
            my $errlst = new SDB::Install::MsgLst ();
            if (!defined deltree ($logdir, $errlst)){
                $self->AddError ("Cannot remove logdir '$logdir'", $errlst);
            }
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Removes sudoers entries in case of storage API.

sub removeSudoersEntries {

    my ($self, $trex) = @_;

    if (!$trex->usesStorageApi()) {
        return 1;
    }

    my $hdbmountAPISkriptDir = join ($path_separator,
                                     $trex->get_globalTrexInstallProgamDir(),
                                     'server');

    my $globalIniDir = $trex->get_globalHdbCustomConfigDir();
    my $msg          = $self->AddMessage ("Modifying sudoers...");

    my $mntinfo = new SDB::Install::HdbMountAPIInfo ($hdbmountAPISkriptDir, $globalIniDir, $self->{_user}->uidUx(), $self->{_user}->gidUx());
    $mntinfo->setPython($self->getPythonPath());
    my ($retcode, $retval) = $mntinfo->getSudoersLineCommandArray();
    if(not defined $retcode) {
        my $wrn = $self->AddWarning ("'hdbmount.py --sudoers...' did not return a valid sudoers line.");
        $self->AddSubMsgLst($wrn, $mntinfo);
        $retval = \@storageApiSudoerMinimumProgs;
    }
    elsif($retcode == 0) {
        my $err = $self->AddError ("Could not execute 'hdbmount.py --sudoers...'");
        $self->AddSubMsgLst($err, $mntinfo);
        return undef;
    }
    my $removeCmds = $retval;
    ($retcode, $retval) = $mntinfo->getSudoersLinePrefix();
    if(not defined $retcode) {
        my $wrn = $self->AddWarning ("'hdbmount.py --sudoers...' did not return a valid sudoers line");
        $self->AddSubMsgLst($wrn, $mntinfo);
        $retval = 'ALL=NOPASSWD:';
    }
    elsif($retcode == 0) {
        my $err = $self->AddError ("Could not execute 'hdbmount.py --sudoers...'");
        $self->AddSubMsgLst($err, $mntinfo);
        return undef;
    }
    my $sudoersPrefix = $retval;
    $self->AddSubMsgLst($msg, $mntinfo);

    my $sidadm = $self->{_user}->getSidAdm()->getname();
    $self->adaptSudoers(
        $sidadm,
        undef,
        $removeCmds,
        $sudoersPrefix,
        [$msg->getSubMsgLst(), $self->getErrMsgLst()]
    );

    return 1;
}

sub adaptSudoers {
    my ($self, $sidadm, $addCmds, $removeCmds, $sudoersPrefix, $msgLstContext) = @_;
    my $osConfig = SDB::Install::OSConfig->new();
    $msgLstContext //= $self->getMsgLstContext();
    $osConfig->setMsgLstContext($msgLstContext);

    $osConfig->adaptSudoers ($sidadm, $addCmds, $removeCmds, $sudoersPrefix);

    my $sysinfo = SDB::Install::SysInfo->new();
    return 1 if (!$sysinfo->isRedHat());

    # BZ 55083:
    if (defined $addCmds) {
        $osConfig->addLineToSudoers("Defaults:$sidadm !requiretty", '^\s*Defaults:'.$sidadm.'\s+!requiretty');
        $osConfig->addLineToSudoers('Defaults:#0 !requiretty', '^\s*Defaults:#0\s+!requiretty');
    } elsif (defined $removeCmds) {
        $osConfig->removeLineFromSudoers('^\s*Defaults:'.$sidadm.'\s+!requiretty');
        $osConfig->removeLineFromSudoers('^Defaults:#0\s+!requiretty$');
    }
    return 1;
}


sub tryRemoveSudoersEntries {
    my ($self) = @_;

    if ($self->hasNewDB()){
        my $trexes = $self->getNewDBInstances();

        foreach my $trex (@$trexes) {
            if (!$self->removeSudoersEntries($trex)) {
                return undef;
            }
        }
    }
}

sub Uninstall;
*Uninstall = \&removeSAPSystem;


sub removeGlobalDirs{
    my ($self, $deleteUser) = @_;
    
    if ($isWin && ($self->{_globalSidDir} =~ /^\\\\/)){
        $self->AddMessage ("Skipping uninstallation of global filesystem");
        return 1;
    }   
    
    if (!defined $deleteUser){
        $deleteUser = 1;
    }
    
    if (!defined $self->{_globalSidDir}){
        $self->AddWarning ("globalSidDir is undefined");
        return 1;
    }
    
    my $_msg = $self->AddMessage ("Removing global filesystem");
    $self->SetFormatInfo ($_msg, 'h1', 'Remove Global Filesystem');
    
    my $swmsglst = new SDB::Install::MsgLst ();
    
    my $msg = $_msg->getSubMsgLst ()->addMessage ("Removing software packages");
    
    my $saveContext = $self->setMsgLstContext ([$msg->getSubMsgLst ()]);
    #$self->SetFormatInfo ($msg, 'h1', 'Remove Software');
    
    if (defined $self->GetPackages (undef, 1)){
        $self->SelectAllPackages  ();
        $self->SUPER::Uninstall ();
    }
    $self->setMsgLstContext ($saveContext);

    my $msglst = new SDB::Install::MsgLst ();


    my @dirlist = ($self->get_globalTrexDir ());

    if ($self->get_integratedSystem()){
    
        #
        # remove just HDB* directories and 'hdb*' links
        #
    
        my $platformDir = $self->get_platformDir ();
        if (opendir (DH, $platformDir)){
            my @HDBs = grep {/^hdb/i} readdir (DH);
            closedir (DH);
            my $path;
            foreach my $hdb (@HDBs){
                $path = $platformDir . $path_separator . $hdb;
                if (!$isWin && -l $path){
                    unlink ($path);
                }
                elsif(-d $path){
                    push @dirlist, $path;
                }
            }
        }
    }
    else{
        push @dirlist, $self->get_platformDir();
        my $homeDir = $self->{_user}->getSidAdm()->home();
        my @homeDirList = split ($re_path_separator, $homeDir);
        my ($globalSidRoot) = ($self->{_globalSidDir} =~ /(.*)$re_path_separator/);
        my $usrsapsid_pattern = '^'.quotemeta ('/usr/sap/'. $self->{_sid}.'/');
        my $globalsidroot_pattern = '^' . quotemeta ($globalSidRoot);
        if ($deleteUser || !(($homeDir =~ /$usrsapsid_pattern/) ||
            ($homeDir =~ /$globalsidroot_pattern/))){
            
            push @dirlist, $self->{_globalSidDir};
            
            my ($basename) = ($globalSidRoot =~
                /$re_path_separator([^$re_path_separator]+)$/);
            
            if ($basename eq $self->{_sid}){
                push @dirlist, $globalSidRoot;
            }
        }
        else{
            if (scalar (@homeDirList) > 3){
                my @checkList = qw (profile global exe);
                foreach my $checkDir (@checkList){
                    if ($homeDirList[4] ne $checkDir ||
                        scalar (@homeDirList) < 4){
                            push @dirlist, $self->{_globalSidDir} .
                                $path_separator . $checkDir;
                    }
                }
            }
        }
    }
    foreach my $dir (@dirlist){
        $self->AddMessage ("Deleting tree '$dir'");
        if (!deltree ($dir, $msglst)){
            $self->PushError ("Cannot remove directory '$dir'", $msglst);
        }
    }
    return 1;
}


sub checkUpdate{
    my ($self, $instconfig) = @_;

    $self->{kit_versionid} = $instconfig->getValue ('versionId');

    require SDB::Install::SAPControl;
    #
    # check if sapstartsrv is running
    #

    my $instance = $self->getNewDBInstances ()->[0];
    
    my @user_passwd = ($instconfig->{user}->getname (), $instconfig->getValue ('Password'));
    
    my $ok = 1;

    my $hosts = $instance->get_hosts();
    my $msglst;
    my $useHttps = $instconfig->isUseHttps ();
    my $sso_cert = $instconfig->getValue ('SSOCertificate');
    if ($hosts && @{$hosts}){
        foreach my $host (@$hosts){
            $self->AddProgressMessage ("Checking sapstartsrv on host $host...");
            $msglst = new SDB::Install::MsgLst ();
            my $rc = $instance->checkSapStartSrv (@user_passwd,
                                                  $host,
                                                  $msglst,
                                                  $useHttps,
                                                  $sso_cert);

            if ($rc){
                if ($instconfig->getIgnore ('check_hosts')
                    &&
                    !$instance->isHostLocal(lc($host))) {

                    $self->getMsgLst()->addError
                            ("Checking sapstartsrv ($host)", $msglst);
                    $self->getMsgLst()->addMessage
                        ("Ignoring error due to command line switch '--ignore=check_hosts'");
                }
                else {
                    $self->PushError ("Error checking sapstartsrv ($host)", $msglst);
                    if ($rc == 1){
                        $self->PushError ("Please start sapstartsrv on host '$host'!");
                    }
                    $ok = 0;
                }
            }
        }
    }

    if (!$ok){
        return undef;
    }

    $self->AddProgressMessage ("Checking local sapstartsrv...");

    $msglst = new SDB::Install::MsgLst ();
    my $rc = $instance->checkSapStartSrv (@user_passwd,
                                          undef, # default is local host
                                          $msglst,
                                          $useHttps,
                                          $sso_cert);
    if ($rc){
        if ($rc == 1){
            $self->AddWarning ("Local sapstartsrv is not running", $msglst);
            my $msg = $self->AddProgressMessage ("Starting local sapstartsrv...");
            $instance->setMsgLstContext([$msg->getSubMsgLst ()]);
            if (!defined $instance->startLocalSapStartSrv (undef,
                                      $instconfig->getTimeout ('start_service'),
                                      $instconfig->isUseHttps())){
                $self->PushError ('Cannot start local sapstartsrv', $instance);
                return undef;
            }
            $msglst = new SDB::Install::MsgLst ();
            $rc = $instance->checkSapStartSrv (@user_passwd,
                                               undef, # default is local host
                                               $msglst,
                                               $useHttps,
                                               $sso_cert);
        }
        if ($rc){
            $self->PushError ("Error checking local sapstartsrv", $msglst);
            return undef;
        }   
    }

    #
    # checking instance
    #
     my $oldContext = $instance->setMsgLstContext($self->getMsgLstContext());
     my $retcode = $instance->checkUpdateTrexInstance ($instconfig);
     $instance->setMsgLstContext($oldContext);
     if (!defined $retcode){
        return undef;
     }
    return 1;
}



#
# checking plugins
#

sub checkPlugins{
    my ($self, $packageManager, $customPluginsPath, $ignoreDependencies) = @_;

    my $rc = 1;
    my $instance = $self->getNewDBInstances ()->[0];
    my $activePlugins = $instance->getPluginInstallations ();

    if (!defined $activePlugins || !@$activePlugins){
        return 1;
    }
    my $pluginsDir = $customPluginsPath // $self->get_pluginsDir();
    my $msg = $self->AddProgressMessage ("Checking server plugins...");
    my $newPlugins = $self->getNewPlugins ($pluginsDir);

    if (!$packageManager){
        $packageManager = $self
    }


    my $failing_plugins = {};
    my $new_plugins = {};


    if (defined $activePlugins){
        foreach my $plugin (@$activePlugins){
            $self->AddSubMessage ($msg, "Checking '".$plugin->getProductName()."'");
            if (!$plugin->checkComponentDependency ($packageManager)){
                $failing_plugins->{$plugin->getProductKey()} = $plugin;
            }
        }
    }

    if (defined $newPlugins){
        foreach my $plugin (@$newPlugins){
            my $key = $plugin->getProductKey();
            $new_plugins->{$key} = $plugin;
            if (defined $failing_plugins->{$key} && $plugin->checkComponentDependency ($packageManager)){
                delete $failing_plugins->{$key};
            }
        }
    }
    my $msglst = SDB::Install::MsgLst->new();

    my ($plugin, $string);
    my $instructionString;
    foreach my $key (keys %$failing_plugins){
        $plugin = $failing_plugins->{$key};
        $instructionString = " => Update the '".$plugin->getProductName()."' plugin!";
        $msg = $msglst->AddMessage ($plugin->getProductName());
        $msglst->AddSubMessage($msg, 'Currently active version: ' . $plugin->GetVersion());
        $string = $plugin->asString();
        if (defined $new_plugins->{$key}){
            $msglst->AddSubMessage($msg, 'Inactive version (should be newer): ' . $new_plugins->{$key}->GetVersion() . $instructionString, $plugin->getErrMsgLst());
        }
        else{
            $msglst->AddSubMessage($msg, "Newer inactive version not found.");
            $msglst->AddSubMessage($msg, $instructionString);
            if (!$ignoreDependencies) {
                $msglst->AddSubMessage($msg, "Skip this plugin dependency check with the command line option ");
                $msglst->AddSubMessage($msg, "--ignore=check_plugin_dependencies if you want to deactivate the ");
                $msglst->AddSubMessage($msg, "'".$plugin->getProductName()."' plugin and update it later, or ");
                $msglst->AddSubMessage($msg, "if you no longer use the functions provided by this plugin. ");
                $msglst->AddSubMessage($msg, "Follow the instructions in SAP Note 1920457.");
            }
        }

        $rc = undef;
    }
    if (!$rc){
        my $msgString = "$gProductNameEngine " . $packageManager->GetVersion() . " is not compatible with installed plugin(s):";
        if ($ignoreDependencies) {
            $self->AddMessage($msgString, $msglst);
        } else {
            $self->AddError($msgString, $msglst);
        }
    }
    return $rc;
}


sub removeUsrSapSid{
    my ($self, $deleteUser) = @_;
    
    if (!defined $deleteUser){
        $deleteUser = 1;
    }
    my $usrSapSid = $self->{_usrSapSid};
    if (-e $usrSapSid || -l $usrSapSid){
        my $homeDir = $self->{_user}->getSidAdm()->home();
        my $usrsapsid_pattern = '^' . quotemeta ('/usr/sap/' .
            $self->{_sid});
        if (($homeDir !~ /$usrsapsid_pattern/) || $deleteUser){
            if (-l $usrSapSid){
                $self->getMsgLst()->addMessage("Deleting $usrSapSid");
                unlink ($usrSapSid);
            }
            elsif (-d $usrSapSid){
                  deltree ($usrSapSid, $self->getMsgLst());
            }
        }
        else{
            if (opendir (DH, $usrSapSid)){
                my @nodes = grep {!/^\.{1,2}$/} readdir (DH);
                closedir (DH);
                my @homeDirList = split($path_separator, $homeDir);
                my $lNode;
                foreach my $node (@nodes){
                    if ($node eq $homeDirList[4]){
                        next;
                    }
                    $lNode = $usrSapSid . $path_separator . $node;
                    if (-d $lNode && !-l $lNode){
                        deltree ($lNode, $self->getMsgLst());
                    }
                    else{
                        $self->getMsgLst()->addMessage("Deleting $lNode");
                        unlink ($lNode);
                    }
                }
            }
        }
    }
    if (!$isWin){
        my $ipcDir = '/var/lib/hdb/' . $self->{_sid};
        if (-d $ipcDir){
            deltree ($ipcDir, $self->getMsgLst());
        }
    }
    return 1;
}

sub deleteUser{
    my ($self, $force, $keepUserHomeDir) = @_;
    if (!defined $force){
        $force = 0;
    }
    my $msg = $self->AddMessage ("Deleting user");
    $self->SetFormatInfo ($msg, 'h1', 'Delete User');
    $self->{_user}->resetMsgLstContext();
    $self->{_user}->delete ($force, $keepUserHomeDir);
    $self->AddSubMsgLst ($msg, $self->{_user});
    return 1;
}

sub deleteHdbParamFiles{
    my ($self) = @_;
    my $customConfigDir = $self->get_globalTrexCustomConfigDir();
    my @hdbparamFiles;
    if (opendir (DH, $customConfigDir)){
        @hdbparamFiles = grep {/^hdbconfiguration_\d+$/} readdir (DH);
        closedir (DH);
    }
    if (@hdbparamFiles){
        my $submsglst = $self->getMsgLst ()->addMessage ('Removing old binary database configurations')->getSubMsgLst ();
        my $path;
        foreach my $file (sort @hdbparamFiles){
            $path = $customConfigDir . $path_separator . $file;
            if (unlink ($path)){
                $submsglst->addMessage ("File '$path' removed");
            }
            else{
                $submsglst->addWarning ("Cannot remove file '$path': $!");
            }
        }
    }
    return 1;
}


sub getNewDBInstances{
    my @result;
    foreach my $inst (values %{$_[0]->{_instances}}){
        if ($inst->isNewDB()){
            push @result, $inst;
        }
    }
    return \@result;
}

sub getLocalInstances{
    my ($self) = @_;
    my @result;
    foreach my $inst (values %{$_[0]->{_instances}}){
        if (!$inst->isNewDB() || $inst->hasLocalHost ()){
            push @result, $inst;
        }
    }
    return \@result;
}


sub GetVersion{
    my ($self, $no_manifest) = @_;
    if (defined $self->{kit_version}){
        return $self->{kit_version};
    }
    if ($no_manifest){
        # prevent recursion across getLinkedExePath <=> GetVersion
        return undef;
    }
    return $self->getVersionByManifest();
}


sub getVersionByManifest{
    my ($self,$nocache) = @_;
    my $manifest = $self->getManifest ($nocache);
    my $dir = $self->get_trexBinDir ();

    if (defined  $manifest){
        my $version = $manifest->getVersion ();
        if (!defined $version){
            $self->AddError ('Cannot get version', $manifest);
            return undef;
        }
        return $version;
    }
    
    if (!-f "$dir/$legacy_manifest"){
        return undef;
    }
    if (!open (FD, "$dir/$legacy_manifest")){
        return undef;
    }
    while (my $line = <FD>){
        chomp;
        if ($line =~ /^saptrex\s+version:/){
            my $version = (split (':',$line))[1];
            $version =~ s/^\s*//;
            $version =~ s/\s*$//;
            close (FD);
            return $version;
        }
    }
    close (FD);
    return undef;
}

sub getManifestDir{
    $_[0]->get_trexBinDir ();
}


sub getNewDBInstanceNr{
    my ($self) = @_;
    foreach my $num (keys %{$self->{_instances}}){
        if ($self->{_instances}->{$num}->isNewDB()){
            return $num;
        }
    }
    return undef;
}


sub addHost{
    my ($self, $hostName, $password, $instconfig,$sysinfo) = @_;

    my $rc         = 1;
    my $instanceNr = $self->getNewDBInstanceNr();
    my $instance   = $self->getNewDBInstances()->[0];
    my $addRoles   = $instconfig->getHostRoles();
    my $isExtendedStorage = 0;
    my $acceleratorRole   = undef;
    my $rolesWithServicesToBeRestarted = {};

    if (defined $addRoles) {
        foreach my $currRole (@$addRoles) {
            if (($currRole eq $gHostRoleAcceleratorWorker) ||
                ($currRole eq $gHostRoleAcceleratorStandby)) {
                $rolesWithServicesToBeRestarted->{$currRole} = 1;
                $acceleratorRole = $currRole;
            }
            elsif (($currRole eq $gHostRoleEsWorker) ||
                   ($currRole eq $gHostRoleEsStandby)) {
                $isExtendedStorage = 1;
                $rolesWithServicesToBeRestarted->{$currRole} = 1;
            }
            elsif ($currRole eq $gHostRoleStreaming){
                $rolesWithServicesToBeRestarted->{$currRole} = 1;
            }
        }
    }

    if ($instconfig->getStep() > 0) {
        if (defined $acceleratorRole) {
            $instance->{addAcceleratorHostPending} = $acceleratorRole;
        }
    }
    else {
        $rc = $self->addHostCreateHostDir($hostName, $instconfig, $instance);
    }

    if ($rc && $instconfig->isOutstandingAddHostStep
                                           ($STEP_ADDHOST_CREATE_DIRECTORIES)) {

        $rc = $self->addHostCreateDirectories($hostName,
                                              $password,
                                              $instconfig,
                                              $sysinfo,
                                              $instanceNr,
                                              $instance);
    }

    if ($rc && $instconfig->isOutstandingAddHostStep
                                             ($STEP_ADDHOST_UPDATE_HOSTAGENT)) {

        $rc = $self->instUpdSaphostagent($instconfig);

    }

    if ($rc && $instconfig->isOutstandingAddHostStep
                                                ($STEP_ADDHOST_ADD_INSTANCES)) {

        if ($acceleratorRole
            && !$instconfig->tryCreateAcceleratorPaths($self,    # sapsys
                                                       $instance,
                                                       $self)) { # msglst
            return undef;
        }

        if ($isExtendedStorage
            && !$instconfig->tryCreateExtendedStoragePaths($self,    # sapsys
                                                           $instance,
                                                           $self)) { # msglst
            return undef;
        }

    }
    # steps are checked within TrexInstance
    $rc = $self->addHostCreateInstance($hostName,
                                       $password,
                                       $instconfig,
                                       $sysinfo,
                                       $instanceNr,
                                       $instance);
    my $isAseAddHostPending = $instance->{addAcceleratorHostPending};
    if ($rc && defined($isAseAddHostPending) && $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_CREATE_ACCELERATOR)) {
        $rc = $instance->addAcceleratorHosts($hostName, $isAseAddHostPending, $instconfig);
        if (!$rc) {
            $self->getErrMsgLst()->setMsgLst($instance->getErrMsgLst());
        }
    }
    if ($rc && $self->isOutstandingAddHostStepRestartServices($instconfig)) {
        if(!$instance->isMultiDb (1) && scalar(keys%{$rolesWithServicesToBeRestarted}) > 0 ){
            $rc = $instance->writeSpecialHostServersIntoDaemonIni(undef,$rolesWithServicesToBeRestarted);
        }
        $rc = $self->addHostRestartServices($hostName,$instconfig,$instance,$rolesWithServicesToBeRestarted);
    }
    if($rc){
        $rc = $rc && $self->addHostFinishPersFile($hostName, $instconfig, $instance);
    }
    return $rc;
}

sub _linkRedHatCompatCPlusPlusLib{
    my ($self) = @_;

    if ($isWin){
        return 1;
    }
    my $libstdc_plusplus = $self->get_trexBinDir() . $path_separator . 'libstdc++.so.6';
    if (-f $libstdc_plusplus || -l $libstdc_plusplus){
        return 1
    }
    my $sysinfo = new SDB::Install::SysInfo ();
    my $manifest = $self->getManifest ();

    if (!$sysinfo->requiresRedHatCppCompatLib ($manifest)){
        return 1;
    }
    my $compatLib = $sysinfo->getRedHatCppCompatLib ($manifest);

    $self->getMsgLst ()->addMessage ("Creating symlink '$libstdc_plusplus' => '$compatLib'");
    if (!symlink ($compatLib, $libstdc_plusplus)){
        $self->setErrorMessage ("Cannot create symbolic link '$libstdc_plusplus' => '$compatLib': $!");
        return undef;
    }
    lchown ($self->getUID, $self->getGID, $libstdc_plusplus);
    return 1;
}

sub addHostCreateDirectories {
    my ($self,
        $hostName,
        $password,
        $instconfig,
        $sysinfo,
        $instanceNr,
        $instance) = @_;
    
    if (!defined $instanceNr){
        $self->AddError ("Cannot determine $gProductNameEngine instance number");
        return undef;
    }
    
    if (ExistsInstanceNumber ($instanceNr)){
        $self->AddError ("Instance '$instanceNr' number is already occupied on this host. AddHost not possible. Use another host or remove this instance.");
        return undef;
    }

    if ($isWin){
        my $archiveDir = $SAPDB::Install::Config{'RuntimeDir'}.$path_separator.'vcredist';
        my @files;
        unless (opendir (DH, $archiveDir)){
            $self->AddError ("Cannot open archive directory '$archiveDir': $!");
            return undef;
        }
        @files = readdir (DH);
        closedir (DH);  
        $self->{'vc_redistributable'} = ScanFilesForVCRedists(
            $archiveDir,
            \@files,
            $sysinfo,
            $self
        );
    } 
    my $progressHandler = $self->GetProgressHandler ();
    if(defined $self->{'vc_redistributable'}){
        my $success = 0;
        ($success, $self->{'reboot_required'}) = InstallVCRedists(
            $self->{'vc_redistributable'},
            $self,
            $progressHandler
        );
        if(!$success) {
            return undef;
        }
    }    
    my $msg = $self->AddMessage ("Creating root");
    $self->SetFormatInfo ($msg, 'h1','Create root');
    my $submsglst = new SDB::Install::MsgLst ();
    
    if (!defined $self->createRoot ($submsglst)){
        $self->AddSubMsgLst ($msg, $submsglst);
        return undef;
    }

    $self->AddSubMsgLst ($msg, $submsglst);

    if (!defined $self->createuserfromconfig ($password)){
        return undef;
    }

    chown ($self->getUID(),
           $self->getGID(),
           $instance->get_hostNameDir($hostName));

    if (!$self->tryChangeNfsAccess($instconfig->getOwnInstance())) {
        return undef;
    }

    if (!defined $self->chownRoot ()){
        return undef;
    }
    
    $self->dumpProperties ();

    my $fsMan = $self->getFileSystemManager ();
    $msg = $self->getMsgLst()->addMessage ("Creating local file system");
    $fsMan->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!$isWin){
        if (!$fsMan->createUsrSapSidLinks ()){
            $self->AddError ("Cannot create /usr/sap links",$fsMan->getErrMsgLst ());
            return undef;
        }
    
        if (!$self->adaptSystemConfiguration(undef, $instconfig)) {
            return undef;
        }
    }

    $fsMan->set_instanceNr ($instanceNr);

    
    if (!defined $instance){
        $self->AddError ('No $gProductNameEngine instance found');
        return undef;
    }
    
    if ($isWin){
        $instance->set_localUsrSapSidDir ($self->{_usrSapSid});
    }
    else{
        if (!defined $fsMan->createInstanceLink ()){
            $self->AddError ("Cannot create instance link", $fsMan);
            return undef;
        }
    }
    
    $instance->{_user} = $self->{_user};
    $self->{_user}->resetMsgLstContext();
    my $rc = $self->{_user}->configureHome ($self->get_globalTrexInstallSupportDir(),
                                            $instance->get_instanceDir,
                                            $hostName);
    $self->AddMessage (undef, $self->{_user});
    return $rc;
}


sub addHostCreateInstance {
    my ($self,
        $hostName,
        $password,
        $instconfig,
        $sysinfo,
        $instanceNr,
        $instance) = @_;

    my $fsMan = $self->getFileSystemManager ();
    $fsMan->set_instanceNr ($instanceNr);
    my $msg = $self->AddProgressMessage ("Adding host '$hostName' to instance '$instanceNr'...");
    $self->SetFormatInfo ($msg, 'h1', 'Add host to instance');
    $instance->setMsgLstContext([$msg->getSubMsgLst ()]);
    my $rc = $instance->addHost($fsMan, $hostName, $instconfig, $self, $sysinfo);
    if (!defined $rc) {
        $self->AddError ("Add host to instance failed", $instance);
    }
    return $rc;
}


sub addHostFinishPersFile {
    my ($self, $hostName, $instconfig, $instance) = @_;

    my $persInfo = {
            'HostName'          => $hostName,
            'HostRoles'         => join(',', @{$instconfig->getHostRoles()}),
            'HostFailoverGroup' => $instconfig->getValue('HostFailoverGroup'),
            'InstallDate'       => iso8601DateTime(),
    };

    my $partition = $instconfig->getValue ('StoragePartitionNumber');
    $persInfo->{StoragePartitionNumber} = $partition if (defined $partition);

    my $workergroup = $instconfig->getValue ('WorkerGroup');
    $persInfo->{WorkerGroup} = $workergroup if (defined $workergroup);


    if (!$instconfig->pers_store(undef, $persInfo)) {
        $self->AddError ("Cannot write status file '"
                         . $instconfig->pers_filename() . "'", $instconfig);
        return undef;
    }
    return 1;
}


sub addHostStartInstance {
    my ($self, $password, $instconfig, $instance) = @_;

    if ($instconfig->getValue ('NoStart')){
		return 1;
	}

    my $msg = $self->AddProgressMessage ("Starting $gProductNameEngine...");
    $self->SetFormatInfo ($msg, 'h1', 'Start');
    $instance->setMsgLstContext([$msg->getSubMsgLst ()]);
    if (!$isWin){
        my $timeout = $instconfig->getTimeout ('start_service');
        if (!defined $timeout){
            $timeout = 90;
        }
        if (!defined $instance->startLocalSapStartSrv (undef,
                                                       $timeout,
                                                       $instconfig->isUseHttps())){
            $self->AddError ("Cannot start sapstartsrv", $instance);
            return undef;
        }
    }

    if (!defined $instance->startHost (
                $self->getUser()->getSidAdmName(),
                $password, # used for Windows only
                undef,     # localhost
                $instconfig->getTimeout ('start_instance'),
                $instconfig->isUseHttps())){
        $self->AddError ("Start of $gProductNameEngine instance failed", $instance);
        return undef;
    }
    return 1;
}


sub isOutstandingAddHostStepInitHostDir {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_INIT_HOST_DIR);
}

sub isOutstandingAddHostStepInstallSAPProfile {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_INSTALL_SAPPROFILE);
}

sub isOutstandingAddHostStepCreateInstProfile {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_CREATE_INST_PROFILE);
}

sub isOutstandingAddHostStepExecuteHdbnsutil {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_EXECUTE_HDBNSUTIL);
}

sub isOutstandingAddHostStepCreateService {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_CREATE_SERVICE);
}

sub isOutstandingAddHostStepCreateStreaming {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_CREATE_STREAMING);
}

sub isOutstandingAddHostStepCreateES {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_CREATE_ES);
}

sub isOutstandingAddHostStepConfigureLss {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_CONFIGURE_LSS);
}

#sub isOutstandingAddHostStepCreateXS2 {
#    my ($self, $instconfig) = @_;
#    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_CREATE_XS2);
#}

sub isOutstandingAddHostStepStartSystem {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_START_SYSTEM);
}

sub isOutstandingAddHostStepRestartServices {
    my ($self, $instconfig) = @_;
    return $instconfig->isOutstandingAddHostStep($STEP_ADDHOST_RESTART_SERVICES);
}

sub addHostCreateHostDir {
    my ($self,
        $hostName,
        $instconfig,
        $instance) = @_;

    my $hostDir = $instance->get_hostNameDir($hostName);
    if (!-e $hostDir) {
        my $config = {'mode' => 0775};
        if (!$isWin && $self->getUser()->exists()) {
            $config->{uid} = $self->getUID();
            $config->{gid} = $self->getGID();
        }
        $self->AddMessage ("Creating directory '$hostDir'");
        if (!defined makedir ($hostDir, $config)) {
            $self->AddError ("Cannot create directory '$hostDir'", $config);
            return undef;
        }
    }
    return 1;
}

sub setCallback{
    $_[0]->{_f_callback} = $_[1];
    my $instances = $_[0]->getNewDBInstances();
    if (defined $instances){
        foreach my $instance (@$instances){
            $instance->setCallback ($_[1]);
        }
    }
}


sub createuserfromconfig{
    my ($self,$password, $mdcUsers) = @_;
    my $admUserName = $self->getUser()->getSidAdmName();

    my $msg = $self->AddMessage ("Creating user");

    $self->SetFormatInfo ($msg,'h1','Create User');
    
    my $errlst = new SDB::Install::MsgLst();
    
    my $cfg = $self->getUserConfig ($errlst);
    
    if (!defined $cfg){
        $self->AddError (undef, $errlst);
        return undef;
    }

    if ($isWin){
        if ($cfg->{domain} ne $self->{_user}->getSidAdm()->{bsname}->{first}){
            #reset user to incorporate correct domain:
            $self->{_user} = new SDB::Install::NewDBUser ($self->{_sid},$cfg->{domain});
        }
    }
    else{
        if ($self->{_user}->exists){
            $self->AddSubMessage ($msg, 'User already exists');
            $msg = $self->AddSubMessage ($msg, 'Adapting existing user');
            $self->{_user}->setMsgLstContext([$msg->getSubMsgLst ()]);
            if (!defined $self->{_user}->adaptExistingUser($mdcUsers)){
                $self->AddError ("Cannot adapt existing user", $self->{_user});
                return undef;
            }
            return 1;
        }
    }
    my $rc;
    #($templateDir, 'SAP System Administrator', $id, $password, $sapsysid, $home, $shell)
    $self->{_user}->setMsgLstContext([$msg->getSubMsgLst ()]);
    if ($isWin){
        $rc = $self->{_user}->create (undef, "$gProductNameEngine System Administrator", undef, $password);
    }
    else{
        $rc = $self->{_user}->create (undef, $cfg->{sidadm_comment},
            $cfg->{sidadm_id},$password, $cfg->{sapsys_groupid},
            $cfg->{sidadm_home}, $cfg->{sidadm_shell}, $mdcUsers);
    }
    
    if (!defined $rc){
        $self->AddError ("Createuserfromconfig: failed creating user", $self->{_user});# TODO which user?
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Changes 'nobody' UIDs of data/log/inst path into system administrator UID
# if necessary.

sub tryChangeNfsAccess() {

    my ($self,
        $instance  # SDB::Install::SAPInstance::TrexInstance
       ) = @_;

    if ($isWin) {
        return 1;
    }

    my $errlst   = new SDB::Install::MsgLst ();
    my $instpath = $self->{_globalSidDir};
    my $datapath = (defined $instance) ? $instance->getDataPath(1) : undef;
    my $logpath  = (defined $instance) ? $instance->getLogPath(1)  : undef;

    foreach my $path ($instpath, $datapath, $logpath) {
        next if (!defined $path);
        if (!nfsidmap ($path, 300, $self->getMsgLst, $errlst, $self->{_f_callback})){
            $self->setErrorMessage ("nfsidmap for '$path' failed", $errlst);
            return undef;
        }
    }
    return 1;
}


our $logLocationPatternTemplate = '^hdb_%s_([A-Za-z])+_\d\d\d\d-\d\d-\d\d_\d\d\.\d\d\.\d\d$';

sub findLogFileLocations{
    my ($self) = @_;
    my @dirs = ($ENV{TMP},$ENV{TEMP});
    if (!$isWin){
        push @dirs, ('/tmp','/var/tmp');
    }
    my ($fileid);
    my (@visited,$i,$found);
    my @result;
    my @entries;
    my $pattern = sprintf ($logLocationPatternTemplate, $self->{_sid});
    foreach my $dir (@dirs){
        if (!$dir){
            next;
        }
        if (!-d $dir){
            next;
        }
        $fileid = getFileId (\$dir);
        $found = 0;
        foreach $i (0..$#visited){
            if (isSameFile2 ($visited[$i],undef, $fileid)){
                $found = 1;
                last;
            }
        }
        if ($found){
            next;
        }
        push @visited, $dir;
        if (!opendir (DH, $dir)){
            $self->PushError ("Cannot open directory '$dir': $!");
            next;
        }
        push @entries, grep {/$pattern/} readdir (DH);
        closedir (DH);
        foreach my $entry (@entries){
            $entry = $dir . $path_separator . $entry;
            if (-d $entry){
                push @result, $entry;
            }
        }
    }
    return \@result;
}


#-------------------------------------------------------------------------------
# Installs or update SAPHostagent if not '--install_hostagent=off' is specified.
#
# Parameter: $instconfig  SDB::Install::Configuration
#
# Returns int retCode

sub instUpdSaphostagent {

    my ($self, $instconfig) = @_;

    my $installParam = $instconfig->getValue('InstallHostagent');

    if (!defined $installParam || $installParam) {

        if (!installSaphostagent($self->get_globalSidDir(),
                     $instconfig->getValue('HostagentPassword'),
                     $self->getMsgLst(),
                     $instconfig->getValue('HostagentUserName'),
                     1, # isProgressMsg
                     $gLogDir)) { # the cwd to restore to (after a temp. chdir)

            $self->AddError ('Cannot install SAP Host Agent', $self->getErrMsgLst ());
            return undef;
         }
    }
    else {
        $self->AddMessage('Installation/update of SAP Host Agent disabled');
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Removes the status files and writes a message into the log.

sub removePersFiles {
    my ($self, $instconfig) = @_;
    my $msg = $self->getMsgLst()->addMessage ("Removing status file");
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $instconfig->pers_remove()) {
        $self->setErrorMessage('Unable to remove status file ',
                               $instconfig->getErrMsgLst());
    }
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of pending hosts.
#
# Looks for the pers-file hdbaddhost in each subdirectory
# of the instance diretory '<sapmnt>/<SID>/HDB<nr>'.

sub getPendingAddHosts {
    my ($self, $hashWanted) = @_;

    my $instance  = $self->getNewDBInstances()->[0];
    return undef if (!defined $instance);

    my $instanceDir = $instance->get_localInstanceDir();
    return undef if (!defined $instanceDir);

    my @subDirs;
    if (opendir (DH, $instanceDir)) {
        @subDirs = grep {-d $instanceDir . $path_separator . $_} readdir(DH);
        closedir (DH);
    }
    return undef if (!@subDirs);

    require SDB::Install::Configuration::AddHost;
    my $instconfig = new SDB::Install::Configuration::AddHost();
    my @pendingHosts;
    my %hashPendingHosts;

    foreach my $currDir (@subDirs) {

        if (($currDir eq '.') || ($currDir eq '..') ||
            ($currDir eq 'backup') ||
            ($currDir eq 'exe'   ) ||
            ($currDir eq 'work'  )) {
                next;
        }
        my $pendingStep = $instconfig->getAddHostPendingStep($instance, $currDir);
        if (defined $pendingStep) {
            my $info = "Add host '$currDir' is pending at step '"
                     . $instconfig->pers_getstepname($pendingStep) . "'.";
            push @pendingHosts, $info;
            $hashPendingHosts{$currDir} = $info;
        }
    }

    return undef if (!@pendingHosts);

    return ($hashWanted) ? \%hashPendingHosts : \@pendingHosts;
}


#-------------------------------------------------------------------------------
# Reads persistent files of hdbreg and hdbrename.
# Returns a pending information or undef if register/rename is not pending.

sub getRegisterOrRenamePending {

    my ($self) = @_;

    require SDB::Install::Configuration::HdbReg;
    my $instConfig        = new SDB::Install::Configuration::HdbReg();
    $instConfig->{sapSys} = $self;
    my $action            = 'Register';
    my $outstandingHosts  = $instConfig->getOutstandingHosts();

    if (!defined $outstandingHosts) {
        require SDB::Install::Configuration::Rename;
        $instConfig           = new SDB::Install::Configuration::Rename();
        $instConfig->{sapSys} = $self;
        $action               = 'Rename';
        $outstandingHosts     = $instConfig->getOutstandingHosts();
    }

    return (defined $outstandingHosts)
           ? "$action is pending on $outstandingHosts"
           : undef;
}

#-------------------------------------------------------------------------------
# Reads the persistency file and returns information about the pending state.

sub getPendingInfoFromPersFile {

    my ($action,      				# e.g 'Installation'
        $preposition, 				# e.g 'of', as in 'Installation of'
        $instconfig,  				# e.g. SDB::Install::Configuration::NewDB
        $stepNames,   				# e.g. \@STEP_NAMES of SDB::Install::SAPSystem
        $prefix,      				# e.g. for indentation
        $twoLiner     				# if true, linebreak will be inserted before the "(", if applicable 
       ) = @_;

    my $persExists = $instconfig->pers_exists();
    my $pers;
    if($persExists) {
        $pers = $instconfig->pers_load();
    }
    else{
        $pers = $instconfig->validateOldLocalPersistencyFile ();
        if (!defined $pers){
            return undef;
        }
    }
    my $info;
    if(not defined $pers) {
        $info = $prefix."$action pending";
    }
    else {
        my $pendingVersion;
        if (defined $pers->{'_kitversion'}) {
        	$pendingVersion = $pers->{'_kitversion'};
        } else {
        	$pendingVersion = "<unknown version>";
        }
        my $step = $pers->{'step'};
        my $phase = $pers->{$gPhaseOptionPerskey};
        my $prepared = "since ";
        if(defined $phase) {
            if($phase eq $gPhasePrepareOption) {
                $prepared = "prepared at "
            }
            elsif($phase eq $gPhaseOfflineOption) {
                $prepared = "prepared, offline since "
            }
        }
        my $linebreak = $twoLiner ? "\n" : " ";
        $info = $prefix.$action." $preposition version $pendingVersion is pending$linebreak($prepared".$instconfig->pers_date_string().")";
        if ((defined $step) && (not defined $phase)) {
            my $stepString = (defined $stepNames->[$step]) ? $stepNames->[$step] : $step;
            $info .= " at step \'$stepString\'";
        }
    }
    return $info . "\n";
}

sub addHostRestartServices {
    my ($self,$hostName,$instconfig,$instance,$roleMap) = @_;
    if ($instconfig->getValue ('NoStart')){
        return 1;
    }
    my $services = [];
    if ($roleMap->{$gHostRoleStreaming} ) {
        push(@$services,'streamingserver');
    }
    if ($roleMap->{$gHostRoleEsWorker} ||
        $roleMap->{$gHostRoleEsStandby}) {
        push(@$services,'esserver');
    }
    if ($roleMap->{$gHostRoleAcceleratorWorker} ||
        $roleMap->{$gHostRoleAcceleratorStandby}) {
        push(@$services, ('etsserver', 'etbserver'));
    }

    if (!@$services) {
        return 1;
    }
    my $msg = $self->AddProgressMessage ("Starting services...");
    $instconfig->getOwnInstance(1);
    $self->SetFormatInfo ($msg, 'h1', 'Start');
    my $serviceManager = SDB::Install::ServiceManager->new();
    $serviceManager->setMsgLstContext([$msg->getSubMsgLst(), $self->getErrMsgLst()]);
    return $serviceManager->startServices($instconfig, $services,[$hostName]);
}

1;
