package SDB::Install::SAPSystemRename;

use strict;
use SDB::Install::FsManager;
use SDB::Install::Globals qw ($gLogDir
                              $gOperationRenameHost
                              $gProductNameEngine);
use SDB::Install::MsgLst;
use SDB::Install::Installation;
use SDB::Install::NewDBUser;
use SDB::Install::SAPInstance;
use SDB::Install::SAPSystem;
use SDB::Install::SysVars   qw($isWin $path_separator);
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::Tools qw (isMountPoint);
use SDB::Install::Saphostagent qw(unregisterInstanceService);
use SDB::Install::System qw (changeOwn);
use SDB::Common::Utils qw(createXSSpaceSAPUserName createXSSpaceProdUserName);
use SDB::Install::Installer;

our @ISA = qw (SDB::Install::SAPSystemUtilities);


#-------------------------------------------------------------------------------
# Constructor

sub new {
    my $self = shift->SUPER::new (@_);
    return $self;
}


#-------------------------------------------------------------------------------
# Renames a single or master host
# This method is called from SDB::Install::App::Console::Rename
#
# Parameter:  $app    SDB::Install::App::Console::HdbReg
#             $sapSys SDB::Install::SAPSystem
#
# Returns int retCode

sub renameMaster {
    my ($self, $app, $sapSys) = @_;

    my $oldSid              = $sapSys->get_sid();
    my $instconfig          = $app->getInstconfig();
    my $newSid              = $instconfig->getValue ('newSID');
    my $sidChanged          = ($oldSid ne $newSid);
    my $sysinfo             = $app->GetSysInfo();
    my $actionProgressive   = $app->getActionProgressive();
    my $actionDone          = $app->getActionDone();
    my $newNr               = $instconfig->getValue ('InstanceNumber');
    my $hostmap_hash        = $instconfig->getValue ('HostMap');
    my $oldPassword         = $instconfig->getValue ('srcPasswd');
    my $newPassword         = $instconfig->getValue ('Password');
    my $keepUser            = $instconfig->getValue ('KeepUser');
    my $keepUserDir         = $instconfig->getValue ('KeepUserHomeDir');
    my $instance            = $instconfig->getOwnInstance();

    $instance->SetProgressHandler ($sapSys->GetProgressHandler());    

    $instconfig->addMsgHostMap($sapSys);

    my $isSidOrNumberChanged = 0;
    if (!$instconfig->isCheckOnly()) {

        if (defined $oldPassword) {

            my $remoteHosts =  $instconfig->getRemoteHosts();

            if (defined $remoteHosts){
                $self->disableAutoFailOver($instconfig,$sapSys->getMsgLst());
            }
            my $systemIsOffline = $instconfig->getValue('SystemIsOffline');
            if (!$systemIsOffline){
                if (!$self->stopSystem($sapSys,
                               $instance,
                               $oldPassword,
                               $instconfig->getTimeout ('stop_service'),
                               $instconfig->getTimeout ('stop_instance'),
                               $instconfig->isUseHttps(),
                               $instconfig->getValue ('SSOCertificate'))) {
                    return undef;
                }
            }
            if ($sidChanged) {
                $self->tryKillProcessesRestartHandover($sapSys, $instance);

                if (defined $remoteHosts) {

                    my @unchangedHosts;
                    my $hostMap         = $instconfig->getValue('HostMap');
                    my $remoteHostNames = $remoteHosts->getHostNames();
                    # $remoteHostNames does not contain old host names
                    # $hostMap contains mapping: old host name => new host name
                    foreach my $currHost (@$remoteHostNames) {
                        if (defined $hostMap->{$currHost}) {
                            push @unchangedHosts, $currHost;
                        }
                    }

                    if (@unchangedHosts
                        && !$self->stopRemoteHosts($instconfig,
                                                   $sapSys,
                                                   $oldPassword,
                                                   \@unchangedHosts)) {
                        return undef;
                    }
                }
            }
        }

        if ($sidChanged && !$instconfig->getValue('SkipModifySudoers')) {
            $sapSys->tryRemoveSudoersEntries();
        }

        my $shouldSkipHostagentCalls = $instconfig->shouldSkipHostagentCalls();
        $isSidOrNumberChanged = ( $sidChanged || ($newNr != $instance->get_nr()) );
        if (!$shouldSkipHostagentCalls && $isSidOrNumberChanged) {

            unregisterInstanceService($sapSys,
                                      0,   # $isProgressMsg
                                      $gLogDir,
                                      $oldSid,
                                      $instance->get_nr(), # old nr
                                      $instance->get_host(),
                                      -1); # $timeoutSeconds (-1 = synchronous)
            # ignore saphostagent errors
        }
        $self->cleanupSAPServicesEntry($sapSys, $instance, $oldSid, $newSid);
        # ignore cleanup error
    }
    
    my $oldUid;
    my $oldGid;
    my $user_changed = 0;

    if (!$isWin){
        my $errlst = new SDB::Install::MsgLst();
        my $cfg = $sapSys->getUserConfig ($errlst);
        if (!defined $cfg){
            $sapSys->AddError (undef, $errlst);
            return undef;
        }
        $oldUid = $cfg->{sidadm_id};
        $oldGid = $cfg->{sapsys_groupid};
    }

    # new IDs may be used if SID is changed or the system is registered on a new host
    my $newUid = $instconfig->getValue('UID');
    my $newGid = $instconfig->getValue('GID');
    $newUid    = $oldUid if (!$sidChanged && !defined $newUid);
    $newGid    = $oldGid if (!$sidChanged && !defined $newGid);

    if (($oldUid != $newUid) || ($oldGid != $newGid)) {
        $user_changed = 1;
    }

    my $oldUsrSapSid    = $sapSys->getUsrSapSid();
    my $newUsrSapSid    = join ($path_separator, '','usr','sap', $newSid);
    my $oldGlobalSidDir = $sapSys->get_globalSidDir();
    my $newGlobalSidDir = $oldGlobalSidDir;
    $newGlobalSidDir    =~ s/$oldSid$/$newSid/;

    if ($instconfig->isCheckOnly()) {

        if ($sidChanged && -d $newGlobalSidDir) {
            # skip checking remote hosts if '/hana/shared/<newSid>' already exists
            $sapSys->AddMessage ("Directory '$newGlobalSidDir' already exists");
            return 1;
        }

        if (!-d $oldUsrSapSid) {
            # skip checking remote hosts in case of register
            $sapSys->AddMessage ("Directory '$oldUsrSapSid' does not exist");
            return 1;
        }
        $instconfig->setMsgLstContext ([$sapSys->getMsgLst ()]);

        my $isHostChanged = 0;
        my $hostMap       = $instconfig->{params}->{HostMap}->{value};
        if (defined $hostMap) {
            foreach my $currOld (keys %$hostMap) {
                if ($currOld ne $hostMap->{$currOld}) {
                    $isHostChanged = 1;
                    last;
                }
            }
        }

        if (!$isHostChanged &&
            !$self->_tryRenameRemoteHosts($instconfig,
                                          $actionProgressive,
                                          $actionDone,
                                          $newSid,
                                          $newNr)) {
            $sapSys->AddError (undef, $instconfig);
            return undef;
        }
        return 1;
    }

    my $oldHostRoles = $instance->getActiveHostRoles();
    if (!defined $oldHostRoles) {
        $sapSys->setErrorMessage (undef, $instance->getErrMsgLst());
        return undef;
    }

    my $fsManager = new SDB::Install::FsManager ($oldUsrSapSid,
                                                 $oldGlobalSidDir,
                                                 $newUid,
                                                 $newGid);
                                                  
    $fsManager->set_instanceNr ($instance->get_nr ());
    $sapSys->setFileSystemManager($fsManager);

    my $oldProfileName = $instance->get_profileName ();
    
    if ($sidChanged) {
        $sapSys->{_globalSidDir} = $newGlobalSidDir;
        $sapSys->{_usrSapSid}    = $newUsrSapSid;
        $sapSys->setRegistryPath ($sapSys->get_globalTrexInstallTemplateDir ());
        $user_changed = 1;
        $instconfig->setCurrentSID($newSid);
        $instance->changeSid ($newSid);
    }

    #
    # delete the origin user
    #
    if ($sidChanged && !$keepUser){

        $self->deleteSysAdminUser($sapSys->getMsgLst(),
                                  $oldSid,
                                  $keepUserDir,
                                  $instconfig->getIgnore('scan_password_database'),
                                  'origin');
    }

    $sapSys->{_user} = new SDB::Install::NewDBUser ($newSid);

    # Normally '/usr/sap/<oldSID>' is renamed into '/usr/sap/<newSID>'
    #
    # If '/usr/sap/<oldSID>' does not exist
    # or if the old home directory has to be preserved,
    # '/usr/sap/<newSID>' has to be created if it does not exist before

    if (!-d $newUsrSapSid && ($keepUser || $keepUserDir || !-d $oldUsrSapSid)) {
        # create '/usr/sap/<newSID>'
        if (!$sapSys->createRoot($sapSys, $newUid, $newGid)) {
            $sapSys->AddError ("Cannot create root '$newUsrSapSid'", $sapSys);
            return undef;
        }
    }

    $instconfig->tryRemoveUnregisterPersFile();

    #
    # rename reference data installation dir in scriptserver.ini
    #
    
    my $referenceDataPath = $instance->getReferenceDataPath();
    if(defined $referenceDataPath) {
        my $newReferenceDataPath = $instance->getNewVolumePath($referenceDataPath,
                                                               $newSid, $newNr,
                                                               $oldSid, $instance->{_nr},
                                                               $instconfig->getValue('Target'));
        my $saveContext = $instance->setMsgLstContext($instconfig->getMsgLstContext());
        $instance->setReferenceDataPath($newReferenceDataPath);
        $instance->setMsgLstContext($saveContext);
    }

    #
    # rename directories && symbolic links
    #
    
    my $msg = $sapSys->AddMessage ("Updating file system layout...");
    $fsManager->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $fsManager->renameFiles ($oldUsrSapSid,
                                          $newUsrSapSid,
                                          $oldGlobalSidDir,
                                          $newGlobalSidDir,
                                          $newNr,
                                          $oldSid,
                                          $newSid)) {
    
        $sapSys->AddError (undef, $fsManager);
        return undef;
    }

    if ($sidChanged){

        if (-d $oldUsrSapSid) {
            $self->cleanupUsrSapSidDirectory($sapSys->getMsgLst(),
                                             undef,
                                             $oldSid);     # ignore errors
        }
        my $installerMsg = $sapSys->getMsgLst()->addMessage ("Updating internal installer paths...");
        $fsManager->setMsgLstContext ([$msg->getSubMsgLst ()]);

        my $installer = new SDB::Install::Installer ();
        $installer->setMsgLstContext ([$installerMsg->getSubMsgLst ()]);
        $installer->renameHanaSID ($oldSid, $newSid, $instconfig->getValue('Target'));
    }

    my ($rc, $configureUser) =
            $self->tryCreateSysAdminUser($instconfig,
                                         $sapSys,
                                         $instconfig->getValue('Password'));
    if (!$rc) {
        return undef;
    }
    if (($configureUser > 0) || $instconfig->getValue('InitUserHomeDir')) {
        $user_changed = 1;
    }

    if (!$isWin){
        $fsManager->setUserSettings($newUid, $newGid, $sapSys->getShmGID);
        if (!$fsManager->createLocalIpcDirectory ()){
            $sapSys->AddError ("Cannot create ipc directory", $fsManager);
            $sapSys->AddMessage (undef, $fsManager);
            return undef;
        }
        $sapSys->AddMessage (undef, $fsManager);
    }

    #
    # change owner
    #
    
    if (!$isWin && $user_changed){

        $msg = $sapSys->AddProgressMessage ('Changing owner'
            . " (uid: $oldUid -> $newUid, gid: $oldGid -> $newGid) of files...");
        my $errlst = new SDB::Install::MsgLst;

        if (!changeOwn ($oldUid,
                        $newUid,
                        $oldGid,
                        $newGid,
                        $newGlobalSidDir,
                        $errlst,
                        $msg->getSubMsgLst(), # info list
                        SKIP_CHOWN_DIR(),     # skip path map
                        0,                    # change root IDs
                        1,                    # never change root IDs
                       )) {

            $sapSys->AddError ("Cannot change owner",$errlst);
            return undef;
        }

        if (!changeOwn ($oldUid,
                        $newUid,
                        $oldGid,
                        $newGid,
                        $newUsrSapSid,
                        $errlst,
                        $msg->getSubMsgLst(), # info list
                        SKIP_CHOWN_DIR(),     # skip path map
                        1                     # change root IDs
                       )) {

            $sapSys->AddError ("Cannot change owner",$errlst);
            return undef;
        }
        
        if (!defined $sapSys->changePackageOwnerInRegistry ($oldUid,
                                                            $newUid,
                                                            $oldGid,
                                                            $newGid)){
            return undef;
        }
    }
    
    $msg = $sapSys->AddProgressMessage ("Updating system configuration files...");
    
    
    my $sapprofileinitempate = new SDB::Install::IniFile (
        $sapSys->get_globalTrexInstallTemplateDir () .
        $path_separator . 'sapprofile.ini');
    
    my $defaultprofile = new SDB::Install::IniFile (
        $sapSys->get_ProfileDir () .
        $path_separator .'DEFAULT.PFL');
    
    my $inis = [$sapprofileinitempate, $defaultprofile];
    
    my @hostmap;
    
    foreach my $origin (keys %$hostmap_hash){
        push @hostmap, [$origin,$hostmap_hash->{$origin}];
    }

    $rc = 1;
    my ($val, $hostkey,$i);
    
    foreach my $ini (@$inis){
        if ($ini->ErrorState()){
            $sapSys->PushError (undef, $ini);
            $rc = 0;
            next;
        }
        $ini->setValue(undef, 'SAPSYSTEMNAME', $newSid);
        if (defined $ini->getValue (undef, 'SAPSYSTEM')){
            $ini->setValue(undef, 'SAPSYSTEM', sprintf ('%02d', $newNr));
        }
        foreach $hostkey ('SAPLOCALHOST','SAPGLOBALHOST'){
            $val = $ini->getValue (undef, $hostkey);
            if (!$val){
                next;
            }
            foreach $i (0..$#hostmap){
                if ($val eq $hostmap[$i]->[0]){
                    $ini->setValue (undef, $hostkey, $hostmap[$i]->[1]);
                    last;
                }
            }
        }
    }

    $sapprofileinitempate->setValue(undef, 'DIR_PROFILE', $instance->getLocalProfileDir());

    my $oldUserConfig = $sapSys->getUserConfig() || {};
    if ($user_changed){
        my $spaceIsolationUserData = {};

        # Update the information about XS space isolation OS users
        if(exists($oldUserConfig->{xs_space_user_prod})){
            $spaceIsolationUserData->{xs_space_user_prod} = createXSSpaceProdUserName($newSid);
            $spaceIsolationUserData->{xs_space_user_prod_id} = $instconfig->getValue('XSSpaceUserIdProd');
        }
        if(exists($oldUserConfig->{xs_space_user_sap})){
            $spaceIsolationUserData->{xs_space_user_sap} = createXSSpaceSAPUserName($newSid);
            $spaceIsolationUserData->{xs_space_user_sap_id} = $instconfig->getValue('XSSpaceUserIdSAP');
        }
        if (!$sapSys->saveConfig($spaceIsolationUserData)) {
            return undef;
        }
    }

    if (!$rc){
        return undef;
    }
    foreach my $ini (@$inis){
        $ini->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $ini->write ()){
            $sapSys->PushError (undef, $ini);
            $rc = 0;
        }
    }
    if (!$rc){
        return undef;
    }

    my $oldNr = $instance->{_nr};

    if ($oldNr != $newNr){
        $instance->changeInstanceNr ($oldNr, $newNr);
    }

    if (!$self->adaptOperatingSystemConfig($sapSys, $instconfig)) {
        return undef;
    }

    $sapSys->_linkRedHatCompatCPlusPlusLib ();

    $msg = $sapSys->AddProgressMessage ("$actionProgressive instance...");
    
    $instance->{_user} = $sapSys->getUser();
    $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $instance->renameInstance ($oldSid,
                                            $newSid,
                                            $oldNr,
                                            $newNr,
                                            $hostmap_hash,
                                            $instconfig,
                                            !$isWin && $user_changed
                                                ? [$oldUid, $newUid, $oldGid, $newGid]
                                                : undef,
                                            $sysinfo,
                                            $user_changed,
                                            $oldHostRoles)) {

        $sapSys->AddError ("$actionProgressive instance failed", $instance);
        return undef;
    }

    #
    # adjust volume paths
    #

    $msg = $sapSys->AddProgressMessage ("Creating sapservices entry...");
    $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $instance->createService ($newPassword)){
        $sapSys->AddError ("Cannot create sapservices entry", $instance);
        return undef;
    }

    # The actual renaming of the XS2 should be done just once. Therefore it's done in the renameMaster sub
    if($instance->existsXS2Dir() && !$instance->renameXS2($instconfig, $oldUserConfig, $newSid)){
        $sapSys->AddError ("$actionProgressive instance failed", $instance);
        return undef;
    }

    if (!$instconfig->trySetAutoStartLocal($instance, $sapSys)) {
        return undef;
    }

    if (!$self->startSapstartsrv($sapSys, $instance, $instconfig)) {
        return undef;
    }

    $self->setRenameMasterStartedHandler ($instconfig);

    my $startSystemWanted = 0;

    my $lssInstance = $instance->getLssInstance(undef, 1);
    if (defined $lssInstance) {
        $lssInstance->setMsgLstContext($sapSys->getMsgLstContext());
        if (!$lssInstance->storeUserInfo($sapSys)) {
            return undef;
        }
    }

    if (!$instconfig->getValue('NoStart')) {
        if($instconfig->noHostStart()) {
            $startSystemWanted = 1;
        }
        elsif (!$self->startLocalInstance($sapSys,
                                          $instance,
                                          $newSid,
                                          $newPassword, # used for Windows only
                                          $instconfig)) {
            return undef;
        }
    }

    $instconfig->{sapSys} = $sapSys;
    $instconfig->setMsgLstContext ([$sapSys->getMsgLst]);

    if (defined $instconfig->getRemoteHosts()) {
        my $isRemoteReconfigWanted =
                ($instance->isUsingInternalInterface()
                 ||
                 ($instance->isUsingGlobalInterface() &&
                  defined $instance->getInternalNetworkPrefix())) ? 1 : 0;

        if (!$self->_tryRenameRemoteHosts($instconfig,
                                          $actionProgressive,
                                          $actionDone,
                                          undef, # $newSid
                                          undef, # $newNr
                                          $isRemoteReconfigWanted, # is searial
                                         )) {
            $sapSys->AddError (undef, $instconfig);
            return undef;
        }
    }

    if ($startSystemWanted) {

        if (!$self->startSystem($sapSys,
                                $instconfig,
                                $newPassword,
                                $app->getAction())) {
            return undef;
        }
    }

    if($self->isDisabledAutoFailOver()){
        $self->enableAutoFailover($instconfig,$sapSys->getMsgLst());
    }

    if ($instconfig->getValue ('ChangeSystemPasswd')){
        if (!$self->changeSystemUserPassword($sapSys, $instance, $instconfig)) {
            return undef;
        }
    }
    if($isSidOrNumberChanged){
        $self->deleteOrphanedSymlinks($instance, $oldSid, $oldNr);
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Renames a slave host
# This method is called from SDB::Install::App::Console::Rename
#
# Parameter: SDB::Install::SAPSystem            $sapSys
#            SDB::Install::Configuration::NewDB $instconfig
#            SDB::Install::SysInfo              $sysinfo
#
# Returns int retCode

sub renameSlave{
    
    my ($self, $sapSys, $instconfig, $sysinfo) = @_;
        
    my $user_changed = 0;
    my $newSid       = $instconfig->getValue ('newSID');
    my $oldSid       = $instconfig->getValue ('SID');
    my $sidChanged   = ($oldSid ne $newSid);
    my $newPassword  = $instconfig->getValue ('Password');
    my $newNr        = $instconfig->getValue ('InstanceNumber');
    my $oldNr        = $instconfig->getValue ('oldInstanceNumber');
    my $nrChanged    = ($oldNr != $newNr);
    my $keepUser     = $instconfig->getValue ('KeepUser');
    my $keepUserDir  = $instconfig->getValue ('KeepUserHomeDir');
    my $instance     = $instconfig->getOwnInstance();
    my $newHostName  = $instance->get_host();
    
    $instance->SetProgressHandler ($sapSys->GetProgressHandler());

    if ($sidChanged && !$instconfig->getValue('SkipModifySudoers')) {
        $sapSys->tryRemoveSudoersEntries();
    }

    my $shouldSkipHostagentCalls = $instconfig->shouldSkipHostagentCalls();
    my $isSidOrNumberChanged = ($sidChanged || $nrChanged);

    if (!$shouldSkipHostagentCalls || $isSidOrNumberChanged) {

        unregisterInstanceService($sapSys,
                                  0,   # $isProgressMsg
                                  $gLogDir,
                                  $oldSid,
                                  $oldNr,
                                  $instance->get_host(),
                                  -1); # $timeoutSeconds (-1 = synchronous)
        # ignore saphostagent errors

        $self->cleanupSAPServicesEntry($sapSys, $instance, $oldSid, $newSid);
        # ignore cleanup error
    }

    my ($oldUid, $oldGid, $newUid, $newGid);
    
    my $oldUsrSapSid       = join($path_separator, '', 'usr', 'sap', $oldSid);
    my $newUsrSapSid       = $sapSys->getUsrSapSid();
    my $existsOldUsrSapSid = (-d $oldUsrSapSid) ? 1 : 0;

    if (!$isWin){
        if (-l $oldUsrSapSid){
            $oldUid = (lstat (_))[4];
            $oldGid = (lstat (_))[5];
        }
        elsif (-e $oldUsrSapSid){
            $oldUid = (stat (_))[4];
            $oldGid = (stat (_))[5];
        }
        my $errlst = new SDB::Install::MsgLst();
        my $cfg    = $sapSys->getUserConfig ($errlst);
        if (!defined $cfg){
            $sapSys->AddError (undef, $errlst);
            return undef;
        }
        $newUid = $cfg->{sidadm_id};
        $newGid = $cfg->{sapsys_groupid};
    }

    if (!-d $newUsrSapSid && ($keepUser || $keepUserDir || !-d $oldUsrSapSid)) {
        # create '/usr/sap/<newSID>'
        if (!$sapSys->createRoot($sapSys, $newUid, $newGid)) {
            $sapSys->AddError ("Cannot create root '$newUsrSapSid'", $sapSys);
            return undef;
        }
    }
    
    my ($currUsrSapSid, $currNr);

    if ($existsOldUsrSapSid) {
        $currUsrSapSid = $oldUsrSapSid;
        $currNr        = $oldNr;
    }
    else {
        $currUsrSapSid = $newUsrSapSid;
        $currNr        = $newNr;
    }

    my $fsManager = new SDB::Install::FsManager ($currUsrSapSid,
                                                 $sapSys->get_globalSidDir(),
                                                 $newUid,
                                                 $newGid);

    $fsManager->set_instanceNr ($currNr);
    $sapSys->setFileSystemManager($fsManager);
    
    #
    # rename symbolic links
    #
    
    my $msg = $sapSys->AddMessage ("Updating file system layout...");
    $fsManager->setMsgLstContext ([$msg->getSubMsgLst ()]);
    my $rc = defined $fsManager->renameSlaveFiles($newUsrSapSid, $newNr);

    if (!$rc) {
        $sapSys->AddError (undef, $fsManager);
        return undef;
    }

    my $configureUser     = 0;
    ($rc, $configureUser) = $self->tryCreateSysAdminUser($instconfig,
                                                         $sapSys,
                                                         $newPassword);
    if (!$rc) {
        return undef;
    }
    $instance->{_user} = new SDB::Install::NewDBUser($newSid);
    $instconfig->tryRemoveUnregisterPersFile();

    if (!$isWin){
        
        $fsManager->setUserSettings ($newUid, $newGid, $sapSys->getShmGID);
        
        if (!$fsManager->createLocalIpcDirectory ()){
            
            $sapSys->AddError ("Cannot create ipc directory", $fsManager);
            $sapSys->AddMessage (undef, $fsManager);
            return undef;
        }
        
        $sapSys->AddMessage (undef, $fsManager);
    }

    #
    # delete the origin user
    #
    if ($sidChanged && !$keepUser) {

        $self->deleteSysAdminUser($sapSys->getMsgLst(),
                                  $oldSid,
                                  $keepUserDir,
                                  $instconfig->getIgnore('scan_password_database'),
                                  'origin'); # ignore error
    }

    my $chownList = [ $instconfig->{oldUID}, $newUid, $instconfig->{oldGID}, $newGid ];
    if (!$self->adaptSlaveVolumePaths($instconfig, $chownList, $sapSys)) {
        return undef;
    }

    if ($sidChanged && -d $oldUsrSapSid) {
        $self->cleanupUsrSapSidDirectory($sapSys->getMsgLst(),
                                         undef,
                                         $oldSid);     # ignore errors
    }
    
    #
    # change owner
    #
    
    if (!$isWin && $sidChanged && defined $oldUid && defined $oldGid){
        
        $msg = $sapSys->AddProgressMessage ('Changing owner'
                . " (uid: $oldUid -> $newUid, gid: $oldGid -> $newGid)"
                . " of '$newUsrSapSid'...");
        my $errlst     = new SDB::Install::MsgLst;

        if (!changeOwn ($oldUid,
                        $newUid,
                        $oldGid,
                        $newGid,
                        $newUsrSapSid,
                        $errlst,
                        $msg->getSubMsgLst(), # info list
                        SKIP_CHOWN_DIR(),     # skip path map
                        1                     # change root IDs
                       )) {

            $sapSys->AddError ("Cannot change owner",$errlst);
            return undef;
        }
    }
    
    if (!$isWin &&
       (($configureUser > 0) || $instconfig->getValue('InitUserHomeDir'))) {
        $sapSys->getUser()->setMsgLstContext ([$msg->getSubMsgLst ()]);
        $sapSys->getUser()->configureHome 
                                ($sapSys->get_globalTrexInstallSupportDir (),
                                 $instance->get_instanceDir(),
                                 $newHostName,
                                 {$oldSid => $newSid});
                                 
    }
    
    $instance->createHWInfoFile ($sysinfo);

    if ($instance->isUsingInternalInterface() || ($instance->isUsingGlobalInterface() && $instance->getInternalNetworkPrefix())){
        $instance->setMsgLstContext ([$sapSys->getMsgLst ()]);
        if (!defined $instance->reconfigureTrexNet ($instconfig)){
            $sapSys->setErrorMessage (undef, $instance->getErrMsgLst());
            return undef;
        }
    }

    if ( ! $self->adaptOperatingSystemConfig($sapSys, $instconfig)) {
        return undef;
    }

    $msg = $sapSys->AddProgressMessage ("Creating sapservices entry...");
    $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $instance->createService ($newPassword)){
        $sapSys->AddError ("Cannot create sapservices entry", $instance);
        return undef;
    }

    my $oldHostName = $instconfig->{oldHost};
    $oldHostName    = $newHostName if (!defined $oldHostName);

    if ($instconfig->{isRenameAccelerator}) {
        my $chownList   = ($isWin) ? undef
                                   : [$instconfig->{oldUidAccelerator}, $newUid,
                                      $instconfig->{oldGidAccelerator}, $newGid];

        if (!$instance->renameAcceleratorInstance($instconfig,
                                                  $oldSid,      $newSid,
                                                  $oldNr,       $newNr,
                                                  $oldHostName, $newHostName,
                                                  $chownList)) {
            return undef;
        }
    }

    if ($instconfig->{isRenameExtendedStorage}) {
        my $chownList   = ($isWin) ? undef
                                   : [$instconfig->{oldUidEs}, $newUid,
                                      $instconfig->{oldGidEs}, $newGid];

        if (!$instance->renameExtendedStorageInstance($instconfig,
                                                     $oldSid,      $newSid,
                                                     $oldNr,       $newNr,
                                                     $oldHostName, $newHostName,
                                                     $chownList)) {
            return undef;
        }
    }

    if ($instconfig->{isRenameXS2}) {
        return undef if(!$instance->recreateXSSpaceIsolationUsers($instconfig, $oldSid, $newSid));
        return undef if(!$instance->recreateXsEaDataPath($instconfig, $oldHostName));
    }

    my $lssInstance = $instance->getLssInstance();
    if (defined $lssInstance) {
        if (!$instance->renameLssInstance($instconfig) || !$lssInstance->reconfig()) {
            return undef;
        }
        if ($instconfig->getValue('IsRegisterLSS')) {
            return undef if (!$lssInstance->restorePSE($instconfig));
        }
    }

    if (!$instconfig->trySetAutoStartLocal($instance)) {
        return undef;
    }

    $instconfig->pers_remove();

    if (!$self->startSapstartsrv($sapSys, $instance, $instconfig)) {
        return undef;
    }

    if(!$instconfig->noHostStart() && !$instconfig->getValue('NoStart')) {
        $self->setRenameMasterStartedHandler ($instconfig);
        if (!$self->startLocalInstance($sapSys,
                                       $instance,
                                       $newSid,
                                       $newPassword, # used for Windows only
                                       $instconfig)) {
            return undef;
        }
    }

    $instconfig->{sapSys} = $sapSys;
    if($isSidOrNumberChanged){
        $self->deleteOrphanedSymlinks($instance, $oldSid, $oldNr);
    }

    return 1;
}

sub deleteOrphanedSymlinks{
    my ($self, $instance, $oldSid, $oldNr) = @_;

    my $traceDir = $instance->get_hostNameTraceDir();
    my $pysap    = $traceDir.$path_separator.sprintf ('py.sap%s_HDB%02d', $oldSid, $oldNr);
    my $hdbsap   = $traceDir.$path_separator.sprintf ('hdb.sap%s_HDB%02d', $oldSid, $oldNr);

    if (-l $pysap && !unlink ($pysap)){
        $self->AddWarning ("Cannot delete symlink '$pysap': $!");
    }

    if (-l $hdbsap && !unlink ($hdbsap)){
        $self->AddWarning ("Cannot delete symlink '$hdbsap': $!");
    }
}

#-------------------------------------------------------------------------------
# In case of distributed system, hdbrename is called for each remote host
# in parallel.
# Returns undef in case of an error

sub _tryRenameRemoteHosts {

    my ($self,
        $instconfig,
        $actionProgressive, # string  (e.g. 'Renaming')
        $actionDone,        # string  (e.g. 'renamed')
        $newSid,            # used in case of CheckOnly otherwise undef
        $newNr,             # used in case of CheckOnly otherwise undef
        $isSerialExecution,
       ) = @_;

    if (!defined $instconfig->getRemoteHosts()) {
        return 1;
    }

    my $additionalParamIds = ['IsRegister',
                              'InstallHostagent',
                              'KeepUser',
                              'KeepUserHomeDir',
                              'KeepXsUsers',
                              'SkipModifySudoers',
                              'ChangeLSSBackupPassword',
                              'NoStart'];

    my $savedNewSID         = $instconfig->{params}->{newSID}->{value};
    my $savedInstanceNumber = $instconfig->{params}->{InstanceNumber}->{value};

    if (defined $newSid && defined $newNr) {
        # In case of check only, the installation path exists under the old name
        $instconfig->{params}->{newSID}->{value}         = $newSid;
        $instconfig->{params}->{InstanceNumber}->{value} = $newNr;
        push @$additionalParamIds, 'newSID';
        push @$additionalParamIds, 'InstanceNumber';
    }

    my $rc = $instconfig->excuteRemoteSerialOrParallel
                                ($actionProgressive,
                                 $actionDone,
                                 'hdbrename',     # excutable
                                 'RenameHost',    # main program
                                 $gOperationRenameHost, # hostctrl operation
                                 ['HostagentPassword', 'Password', 'srcPasswd', 'LSSPassword', 'LSSBackupPassword', 'TargetLSSBackupPassword'],
                                 $additionalParamIds,
                                 undef, # $cmdLineParams
                                 undef, # $remoteHosts
                                 undef, # $remoteInstconfig
                                 undef, # $hostctrlParamMap
                                 undef, # $templateParamRepl
                                 undef, # $executeOnlyOnHosts
                                 $isSerialExecution);

    $instconfig->{params}->{newSID}->{value}         = $savedNewSID;
    $instconfig->{params}->{InstanceNumber}->{value} = $savedInstanceNumber;
    return $rc;
}




#-------------------------------------------------------------------------------
# Sends 'hdbreg --start/stop_only=instance' to remote hosts.
#
# Returns int retCode

sub stopRemoteHosts {

    my ($self,
        $instconfig,    # SDB::Install::Configuration::AnyConfig
        $sapSys,        # SDB::Install::SAPSystem
        $sidAdmPassword,# used for hostctrl and to stop the remote instances
        $stopThisHosts, # array containing subset of host names to stop
       ) = @_;

    my $stopParmID   = 'StopOnly';
    my $passwdParmID = 'Password';
    my $params       = $instconfig->{params};
    my $cntParams    = scalar keys %$params;
    my $savedPwParam = $params->{$passwdParmID};
    my $savedNostart = $params->{NoStart}->{value};

    delete $params->{$passwdParmID};
    $params->{$passwdParmID} =
            $instconfig->SDB::Install::Configuration::AnyConfig::getParamPassword
                                                                 ($cntParams+1);
    $params->{$stopParmID} = $instconfig->getParamStopOnly($cntParams+2);
    $params->{$passwdParmID}->{value} = $sidAdmPassword;
    $params->{$stopParmID}->{value}   = 'instance';
    $params->{NoStart}->{value}       = 0;

    $instconfig->{sapSys} = $sapSys;
    $instconfig->setMsgLstContext ([$sapSys->getMsgLst()]);
    my $rc = $self->startStopRemoteHosts($instconfig,
                                         $sapSys,
                                         $stopParmID,
                                         'Stopping',
                                         'Stopped',
                                         $stopThisHosts);
    delete $params->{$stopParmID};
    delete $params->{$passwdParmID};
    $params->{$passwdParmID} = $savedPwParam;
    $params->{NoStart}->{value} = $savedNostart;
    return $rc;
}


sub getInstanceRenameArgs{
    my ($self, $instconfig) = @_;
    my $newSid       = $instconfig->getValue ('newSID');
    my $oldSid       = $instconfig->getValue ('SID');
    $oldSid          = $newSid if (!defined $oldSid); # retry after pers file is deleted
    my $newNr        = $instconfig->getValue ('InstanceNumber');
    my $oldNr        = $instconfig->getValue ('oldInstanceNumber');
    my $hostHashMap = $instconfig->getValue ('HostMap');
    return [$oldSid, $newSid, $newNr, $oldNr, $hostHashMap];
}


sub isDisabledAutoFailOver{
    return shift->{_isDisabledFailOver};
}

sub enableAutoFailover{
    my ($self,$config,$msgLst) = @_;
    $config->getOwnInstance()->enableAutoFailover($msgLst);
    $self->{_isDisabledFailOver} = 0;
}

sub disableAutoFailOver{
    my ($self,$config,$msgLst) = @_;
    $config->getOwnInstance()->disableAutoFailover($msgLst);
    $self->{_isDisabledFailOver} = 1;
}
1;
