package SDB::Install::SSH::Factory;

use strict;
use warnings;

use SDB::Install::MsgLst;
use SDB::Install::RPM;
use SDB::Install::Version;
use SDB::Install::SSH::LibsshSession;
use SDB::Install::SSH::Libssh2Session;
use SDB::Install::System qw (loadSSLRequiringPackage loadPackageSafely);
use SDB::Install::Globals qw ($gUseLibssh2);

our $isLegacyLibsshExitSignalSupported;

sub new {
    my $class = shift;
    my $self = {
        libsshFlavour => undef,
    };
    return bless $self, $class;
}

sub getSshSession {
    my ($self) = @_;
    if (!defined $self->{libsshFlavour}) {
        return undef if (!$self->loadLibsshPackage());
    }

    if ($self->{libsshFlavour} eq 'modern') {
        return SDB::Install::SSH::LibsshSession->new();
    }
    elsif ($self->{libsshFlavour} eq 'legacy') {
        return SDB::Install::SSH::Libssh2Session->new($isLegacyLibsshExitSignalSupported);
    }
    return undef;
}

sub loadLibsshPackage {
    my ($self, $errlst) = @_;
    return 1 if (defined $self->{libsshFlavour});
    $errlst //= SDB::Install::MsgLst->new();
# prefer libssh if available
    if (!$gUseLibssh2 && $self->_getModernLibsshPackage($errlst)) {
        return $self->_loadModernLibsshSession($errlst);
    }
    if ($self->_getLegacyLibsshPackage($errlst)) {
        return $self->_loadLegacyLibsshSession($errlst);
    }
    return undef;
}

sub _getModernLibsshPackage {
    my ($self, $errlst) = @_;
    my $rpm = SDB::Install::RPM->new();
    my $libsshPackage = $rpm->getPackage('libssh4') // $rpm->getPackage ('libssh');
    if (!defined $libsshPackage) {
        $errlst->addError("Failed to detect 'libssh4' or 'libssh' packages on system");
        return undef;
    }
    my $versionString = $libsshPackage->getVersion();
    $versionString =~ s/[^\d\.].*$//;
    if (!$versionString) {
        $errlst->addError("Failed to parse libssh package version");
        return undef;
    }
    my $minSupportedVersion = SDB::Install::Version->new(0, 8, 5);
    my $version = SDB::Install::Version->new(split('\.', $versionString));
    if (!$version->isNewerThan($minSupportedVersion) && !$version->isEqual($minSupportedVersion)) {
        my $minSupportedVersionString = $minSupportedVersion->asString();
        $errlst->addError("Minimum supported libssh version is $minSupportedVersionString. Detected version on system is $versionString.");
        $errlst->addError("If a newer version of libssh is not available for your OS, consider installing libssh2.");
        return undef;
    }
    return $libsshPackage;
}

sub _loadModernLibsshSession {
    my ($self, $errlst) = @_;
    if (!loadPackageSafely('Libssh::Session', $errlst)) {
        return 0;
    }
    if (!loadPackageSafely('Libssh::Sftp', $errlst)) {
        return 0;
    }
    $self->{libsshFlavour} = 'modern';
    return 1;
}

sub _getLegacyLibsshPackage {
    my ($self, $errlst) = @_;
    my $rpm = SDB::Install::RPM->new();
    return $rpm->getPackage('libssh2-1') // $rpm->getPackage ('libssh2');
}

sub _loadLegacyLibsshSession {
    my ($self, $errlst) = @_;

    if (!loadSSLRequiringPackage('Net::SSH2', $errlst, $errlst)) {
        return 0;
    }
    if (!defined $isLegacyLibsshExitSignalSupported) {
        $self->_setLegacyLibsshExitSignalSupport();
    }
    $self->{libsshFlavour} = 'legacy';
    return 1;
}

sub _setLegacyLibsshExitSignalSupport{
    my ($self) = @_;

    my $result;
    my $versionString = Net::SSH2->version ();
    $versionString =~ s/[^\d\.].*$//;
    my $version = new SDB::Install::Version (split ('\.', $versionString));
    my $minVersion = new SDB::Install::Version (1,2,8);
    if ($minVersion->isNewerThan ($version)) {
        #
        # used SDK doesn't support it
        #
        $isLegacyLibsshExitSignalSupported  = 0;
    }
    else {
        #
        # try to check libssh2 runtime
        #
        $isLegacyLibsshExitSignalSupported  = 0; #default is false
        require SDB::Install::RPM;
        my $rpm = new SDB::Install::RPM ();
        my $libssh2Package = $self->_getLegacyLibsshPackage();
        if (!defined $libssh2Package) {
            return;
        }
        $versionString = $libssh2Package->getVersion ();
        $versionString =~ s/[^\d\.].*$//;
        if ($versionString){
            $version = new SDB::Install::Version (split ('\.', $versionString));
            if (!$minVersion->isNewerThan ($version)){
                $isLegacyLibsshExitSignalSupported  = 1;
            }
        }
    }
}

1;