package SDB::Install::ServerPluginOnlineOperation;

use strict;
use warnings;

use base 'SDB::Install::Base';

our $CHECK_MSG_STRING             = 'Checking whether online %s is supported.';
our $NOT_SUPPORTED_STRING         = 'Server plugin doesn\'t support online %s.';
our $SUPPORTED_STRING             = 'Server plugin supports online %s.';
our $ACTION_STRING                = 'operation';

sub new{
    my $self = shift->SUPER::new();
    ($self->{pluginKey}, $self->{pluginManifest}) = @_;
    return $self;
}

sub _getCheckPreparationMsgLst{
    my ($self) = @_;
    if (!defined $self->{_checkMsgLst}){
        $self->{_checkMsgLst} = $self->getMsgLst()->addMessage(
            sprintf($CHECK_MSG_STRING, $self->_getAction()))->getSubMsgLst();
    }
    return $self->{_checkMsgLst};
}

sub _getAction{
    return $ACTION_STRING;
}

sub _callDBProcWithParams{
    my ($self, $stmnt, $params, $sizes, $msglst) = @_;
    my $sqlOk = 1;
    my $sql = $self->{sqlConnection};
    $msglst = defined $msglst ? $msglst : $self->getMsgLst();
    $sql->setMsgLstContext([$msglst]);
    if (!defined $sql->prepare ($stmnt)){
        $self->setErrorMessage ("Prepare '$stmnt' failed.", $sql->getErrMsgLst ());
        $sqlOk = 0;
    }
    else{
        for my $i (0..(@$params - 1)){
        my $oneBasedIdx = $i + 1;
        my $size = undef;
            if(defined $sizes && exists $sizes->[$i]) {
                $size = $sizes->[$i];
            }
            if (!defined $sql->bindParam ($oneBasedIdx, $params->[$i], $size)) {
                $msglst->setErrorMessage ("Cannot bind parameter # $oneBasedIdx in statement '$stmnt'.", $sql->getErrMsgLst());
                $sqlOk = 0;
                last;
            }
        }
    }
    if ($sqlOk){
        my $sql_execute = $stmnt;
        foreach my $param (@$params){
            $sql_execute =~ s/\?/'$param'/;
        }
        if (!defined $sql->execute()){
            $self->setErrorMessage ("Execution of '$sql_execute' failed.", $sql->getErrMsgLst ());
            $sqlOk = 0;
        }
    }
    return $sqlOk;
}

sub _getCurrentSqlUser{
    my ($self) = @_;
    my $sql = $self->{sqlConnection};
    if (!defined $sql->execute ("SELECT CURRENT_USER FROM DUMMY")){
        $self->setErrorMessage ("Cannot get current sql user", $sql->getErrMsgLst ());
        return undef;
    }
    my $resultSet = $sql->getResultSet ();
    if (!defined $resultSet){
        $self->setErrorMessage ('Cannot get result set', $sql->getErrMsgLst ());
        return undef;
    }
    return $resultSet->[0]->[0];
}

sub _checkSqlUserPrivilegesByRole{
    my (
        $self,
        $roleName
    ) = @_;
    my $sql = $self->{sqlConnection};
    my $msg = $self->getMsgLst()->addMessage('Checking whether the sql user has sufficient role to perform online registration operations.');
    $sql->setMsgLstContext([$msg->getSubMsgLst]);
    my $sqlUser = $self->_getCurrentSqlUser();
    if ($sqlUser eq 'SYSTEM'){
        return 1;
    }
    if (!defined $sql->execute ("SELECT ROLE_NAME FROM EFFECTIVE_ROLES WHERE USER_NAME = CURRENT_USER AND ROLE_NAME = '$roleName'")){
        $self->setErrorMessage ("Cannot check privileges of sql user '$sqlUser'", $sql->getErrMsgLst ());
        return undef;
    }
    my $resultSet = $sql->getResultSet ();
    if (!defined $resultSet){
        $self->setErrorMessage ('Cannot get result set', $sql->getErrMsgLst ());
        return undef;
    }
    if (!@$resultSet){
        $self->setErrorMessage ("Database user '$sqlUser' requires role '$roleName'");
        return undef;
    }
    return 1;
}

sub _checkSqlUserDBProcExecutePermission{
    my (
        $self,
        $schemaName,
        $DBProcName
    ) = @_;
    my $sql = $self->{sqlConnection};
    my $msg = $self->getMsgLst()->addMessage('Checking whether the sql user has sufficient privileges to perform online registration operations.');
    $sql->setMsgLstContext([$msg->getSubMsgLst]);
    my $sqlUser = $self->_getCurrentSqlUser();
    if ($sqlUser eq 'SYSTEM'){
        return 1;
    }
    my $sqlString = "select top 1 1 from sys.granted_privileges "
            ."where grantee = current_user "
            ."and schema_name = ? "
            ."and object_name = ? "
            ."and privilege = 'EXECUTE' "
            ."and grantee_type = 'USER' "
            ."and is_valid = 'TRUE'";
    if (!defined $sql->prepare($sqlString)){
        $self->setErrorMessage ("Prepare '$sqlString' failed.", $sql->getErrMsgLst ());
        return undef;
    }

    my @params = ($schemaName, $DBProcName);

    for my $i (0..(@params - 1)){
        my $oneBasedIdx = $i + 1;
        if (!defined $sql->bindParam ($oneBasedIdx, $params[$i])) {
            $self->setErrorMessage ("Cannot bind parameter # $oneBasedIdx in statement '$sqlString'.", $sql->getErrMsgLst());
            return undef;
        }
    }
    my $sql_execute = $sqlString;
    foreach my $param (@params){
        $sql_execute =~ s/\?/'$param'/;
    }
    if (!defined $sql->execute()){
        $self->setErrorMessage ("Execution of '$sql_execute' failed.", $sql->getErrMsgLst ());
        return undef;
    }
    my $resultSet = $sql->getResultSetOfPreparedStatement();
    if (!defined $resultSet){
        $self->setErrorMessage ('Cannot get result set.', $sql->getErrMsgLst ());
        return undef;
    }
    if (!@$resultSet){
        $self->setErrorMessage ("Database user '$sqlUser' requires execute permissions on procedure '$schemaName.$DBProcName'.");
        return undef;
    }
    return 1;
}

sub checkPreconditionOffline{
    my ($self) =  @_;
    my $msglst = $self->_getCheckPreparationMsgLst();
    my $rc = $self->{pluginManifest}->isServerPluginOnlineInstallable();
    if ($rc){
        $msglst->addMessage(sprintf($SUPPORTED_STRING, $self->_getAction()));
    }
    else{
        $msglst->addMessage(sprintf($NOT_SUPPORTED_STRING, $self->_getAction()));
    }
    return $rc;
}

sub checkPreconditionOnline{
    my ($self, $sqlConnection) = @_;
    my $msg = $self->getMsgLst()->addMessage("Checking if connection to the Database is established");
    if (!defined($sqlConnection)) {
        $msg->getSubMsgLst()->addMessage("Connection to the Database is not available.");
        return 0;
    }
    $self->{sqlConnection} = $sqlConnection;
    return 1;
}

sub isScriptServerRestartRequired{
    return 0;
}

sub setNoScriptServerRestart{
    return 1;
}

sub prepare{
    my ($self) = @_;
    $self->{onlineOperationOk} = 1;
    return 1;
}

sub do{
    return 1;
}

sub cleanup{
    my ($self) = @_;
    delete $self->{sqlConnection};
    return 1;
}

1;
