package SDB::Install::System::FuserHangChecker;

use strict;
use warnings;
use parent 'SDB::Install::Base';

use SDB::Install::System::Mounts;
use SDB::Install::System qw (which);
use LCM::ProcessExecutor;
use File::Temp;
use File::Basename qw (dirname);
use File::Spec;
use SDB::Install::Installer;


our $statExe;
our $nfsTimeout       = 10;
our $procstateTimeout = 30;

our $procStateScriptContent = << '_PROCSTATE_SCRIPT_';
package SAPDB::Install::ProcState;
bootstrap __PACKAGE__;
sub run{
    $| = 1;
    print "Finding busy files...\n";
    my $ps = __PACKAGE__->new(SDB_PROCSTATE_FLAG_MODULES());
    print "...Done.\n"
}
run();
exit 1;
_PROCSTATE_SCRIPT_

sub _accessMount{
    my ($self, $dir, $msglst) = @_;
    $statExe = defined $statExe ? $statExe : which('stat');
    if (!defined $statExe){
        $self->setErrorMessage("Program 'stat' not found");
        return undef;
    }
    return $self->_runWithTimeout($statExe,[$dir],$nfsTimeout,$msglst);
}

sub _getProcStateScript{
    my $fh = File::Temp->new();
    $fh->print($procStateScriptContent);
    $fh->close();
    return $fh;
}

sub _runProcState{
    my ($self) = @_;
    my $msg = $self->getMsgLst()->addMessage('Checking installer runtime...');
    my $installer = new SDB::Install::Installer();
    my $runtimeDir = $installer->GetRuntimeDir();
    my $installerDir = dirname($runtimeDir);
    my $installerExe = File::Spec->catfile($installerDir,$installer->GetProgramName);
    my $fh = $self->_getProcStateScript();
    return $self->_runWithTimeout($installerExe, ['--script',$fh->filename()],$procstateTimeout,$msg->getSubMsgLst());
}

sub _runWithTimeout{
    my ($self,$prog,$args,$timeout,$msglst) = @_;
    my $exer = new LCM::ProcessExecutor($prog,$args,undef,undef,undef,undef,undef,undef, $timeout);
    $exer->setMsgLstContext([$msglst,$self->getErrMsgLst()]);
    my $rc = $exer->executeProgram();
    if ($exer->wasATimeout()){
        return 0;
    }
    if (defined $rc && $rc == 0){
        return 1;
    }
    return 2;
}

sub _checkMounts{
    my ($self) = @_;
    my $msg = $self->getMsgLst()->addMessage('Checking NFS mounts...');
    my $msglst = $msg->getSubMsgLst();
    my $mnts = new SDB::Install::System::Mounts();
    $mnts->setMsgLstContext([$msglst,$self->getErrMsgLst()]);
    my $nfsMounts = $mnts->getMountsWithType('nfs');
    if (!defined $nfsMounts){
        return undef;
    }
    if (!@$nfsMounts){
        $msglst->addMessage("There is no NFS mount.");
        return 1;
    }
    $nfsMounts = $mnts->getMountsWithOption('hard', $nfsMounts);
    if (!defined $nfsMounts){
        return undef;
    }
    if (!@$nfsMounts){
        $msglst->addMessage("There is no NFS mount with mount option 'hard'.");
        return 1;
    }
    my ($dir, $submsg);
    my @timeoutDetected;
    foreach my $mnt (@$nfsMounts){
        $dir = $mnt->{mnt_dir};
        $submsg = $msglst->addMessage("Accessing NFS mount '$dir' ...");
        if (!$self->_accessMount($dir,$submsg->getSubMsgLst())){
            push @timeoutDetected, $mnt;
        }
    }

    if (!@timeoutDetected){
        $msglst->addMessage('No hanging NFS mount detected');
        return 1;
    }

    $submsg = $msglst->addMessage(int(@timeoutDetected) . ' mounts detected, which don\'t respond in time:');
    my $submsglst = $submsg->getSubMsgLst();
    foreach my $mnt (@timeoutDetected){
        $submsglst->addMessage($mnt->{mnt_dir});
    }
    return 0;
}

sub check{
    my ($self) = @_;
    my $rc = $self->_checkMounts();
    if (defined $rc && $rc == 0){
        $rc = $self->_runProcState();
        if (defined $rc && $rc == 0){
            $self->setErrorMessage("Finding busy executables and libraries is hanging. Installation process will most likely hang as well!");
            $self->appendErrorMessage("Try to find blocking processes and kill them, or reboot your system!");
        }
    }
    return $rc;
}

1;
