#! /usr/bin/perl
#
#


package BuildPackage::Tools;

use Cwd;
use Exporter;
use Digest::SHA qw(sha256_hex);
use Time::Local;
use SDB::Install::DebugUtilities;
use File::Basename qw (dirname basename);
use strict;

our @ISA =('Exporter');
our @EXPORT = qw (getMD5Sum add_file insert_file remove_file getRealPathName
			do_cmd makedir find deltree copy readini
			normalizePath DecodeBoolTag 
			enumFilesRecursively dataDeepCopy createTgz createInstallerPackage);

#-----------------------------------------------------------------------------

sub setErrMsgHandler{
	my ($handler) = @_;
	*err = $handler;
}

#-----------------------------------------------------------------------------

sub setMsgHandler{
	my ($handler) = @_;
	*msg = $handler;
}

#-----------------------------------------------------------------------------

sub getMD5Sum{
	my ($file) = @_;
	-f $file || err("cannot get md5sum of $file: $!\n") && return undef;
	my $md5sum = `md5sum $file 2>&1`;
	chomp($md5sum);
      my $output = $md5sum;
	$md5sum =~ s/\s+\S+//;
      $md5sum =~ /^[a-f|0-9]{32}$/ || err ("cannot get md5sum of $file: $output\n") && return undef;
	return $md5sum;
}

#-----------------------------------------------------------------------------

sub copy{
	my ($src,$dst,$config)=@_;
	$src=~s/\\/\//g;
	$dst=~s/\\/\//g;
	if(not -f $src) {
	     err("ERR: System::copy(): file \"$src\" not found\n");
	     return 0;
	}
	my @statbuf = stat($src);
	if(not open(SRC,$src)) {
	    err("ERR: System::copy(): cannot open file \"$src\": $!\n");
	    return 0;
	}
	$config->{'binmode'} && binmode(SRC);
	my $dst_name;
	if(-d $dst){
		$src=~/([^\/]*)$/ and my $file=$1;
		my $dperm_changed = 0; 
		my @dstatbuf;
		if($^O !~ /mswin/i and $< != 0){
			@dstatbuf = stat($dst);
			if($< == $dstatbuf[4]){
				if(($dstatbuf[2] & 0200) == 0){
					chmod($dstatbuf[2] | 0200,$dst);
					$dperm_changed = 1;
				}
			}
		}
		if(not open(DST,">$dst/$file")) {
		    err("ERR: System::copy(): cannot create $dst/$file: $!\n");
		    return 0;
		}
		chmod($dstatbuf[2],$dst) if $dperm_changed;
		$dst_name = "$dst/$file";
	}
	elsif(-f $dst){
		if($^O !~ /mswin/i and $< != 0){
			my @dstatbuf = stat($dst);
			if($< == $dstatbuf[4]){
				if(($dstatbuf[2] & 0200) == 0){
					chmod($dstatbuf[2] | 0200,$dst);
				}
			}
		}
		if(not open(DST,">$dst")) {
		    err("ERR: System::copy(): cannot create $dst\n");
		    return 0;
		}
		$dst_name = $dst;
	}
	else{
		my $dir = $dst;
		$dir=~s/[^\/]*$//;
		$dst_name = $dst;
		unless(-d $dir){
			if($config->{'createdir'}){
				my $perm = 0775; #default
				$perm = $config->{'dir_perm'} if defined $config->{'dir_perm'}; 
				if(not makedir($dir,$perm)) {
				    err("ERR: Tools::copy(): cannot create directory $dir: $!\n");
				    return 0;
				}
			}
			else{
				err("ERR: Tools::copy(): cannot create $dst - parent dir missing\n");
				return 0;
			}
		}  
		my $dperm_changed = 0; 
		my @dstatbuf;
		if($^O !~ /mswin/i and $< != 0){
			@dstatbuf = stat($dir);
			if($< == $dstatbuf[4]){
				if(($dstatbuf[2] & 0200) == 0){
					chmod($dstatbuf[2] | 0200,$dir);
					$dperm_changed = 1;
				}
			}
		}
		if(not open(DST,">$dst")) {
		    err("ERR: System::copy(): cannot create $dst: $!\n");
		    return 0;
		}
		chmod($dstatbuf[2],$dir) if $dperm_changed;
	}
	$config->{'binmode'} && binmode(DST);
	my $bufSize=(stat SRC)[11] || 32768;
	my ($buf, $written);
	while(my $len = sysread(SRC,$buf,$bufSize)){
		unless(defined $len){
			next if $! =~ /^Interrupted/;
			err("ERR: Tools::copy(): read failure: $!\n");
			return 0;
		} 
		my $offset=0;
		while($len){
			$written=syswrite(DST,$buf,$len,$offset);
			if(not defined $written) {
			    err("ERR: Tools::copy(): write failure: $!\n");
			    return 0;
			}
			$len-=$written;
			$offset+=$written;
		}	
	}
	close(SRC);
	close(DST);
	
	my $atime = $statbuf[8] <= 0 ? undef : $statbuf[8];
	if(!utime($atime, $statbuf[9], $dst_name)) {
		#dumpThings(\@statbuf, 3, 2);
		err("cannot set time of copied file \"$dst_name\": $!\n");
		return 0;
	}

	unless($^O =~ /mswin/i){
		if (defined $config->{mode}){
			if(not chmod ($config->{mode},$dst_name)) {
			    err("cannot set mode of copied file \"$dst_name\": $!\n");
			    return 0;
			}
		}
		else{
			$statbuf[2] &= 0777;
			if (defined $config->{mask}){
				$statbuf[2] &= ~$config->{mask}	
			}
			if(not chmod($statbuf[2],$dst_name)) {
			    err("cannot set mode of copied file \"$dst_name\": $!\n");
			    return 0;
			}
		}
		unless($config->{'nochown'} || $>){
			if(not chown($statbuf[4],$statbuf[5],$dst_name)) {
			    err("cannot set uid/gid of copied file \"$dst_name\": $!\n");
			    return 0;
			}
		}
	}
	
	#msg("copied file \"$src\" to \"$dst\"\n");
	return 1;
}

#-----------------------------------------------------------------------------

sub add_file{
        my ($file,$target_dir,$target,$list) = @_;
        -f $file or return undef;
        -d $target_dir || return undef;
        $target =~/^\// and return undef;
        $file =~ s/\\/\//g;
        $target_dir =~ s/\\/\//g;
        $target_dir =~ s/\/$//g;
        $file =~ s/\\/\//g;
        my ($dir) = ($target =~ /^(.*)\/\w+$/);
        if ($dir =~ /\S/){
                makedir($target_dir.'/'.$dir) || return undef;
        }
        copy($file,$target_dir.'/'.$target) || return undef;
         my $md5sum = `md5sum $file`;
        $md5sum =~ s/\s+\S+//;
        $md5sum =~ /^[a-f|0-9]{32}$/ || return undef;
        my $list_file = $target_dir.'/'.$list;
        my @list_buf;
        open (LST,$list_file) || return undef;
        while(<LST>){
                chomp;
                /\S/ or next;
                /^"$target"/ and next;
                push @list_buf,$_;
        }
        close(LST);
        push @list_buf,"\"$target\" $md5sum";
        open (LST,'>'.$list_file) || return undef;
        foreach (sort @list_buf){
                print LST "$_\n";
        }
        close(LST);
        return 1;
}

#-----------------------------------------------------------------------------

sub insert_file {
        my ($buildroot,$file,$list)=@_;
        -d $buildroot || return undef;
        -f "$buildroot/$file" || return undef;
        -f "$buildroot/$list" || return undef;
        my @list_buf;
        open (LST,"$buildroot/$list") || return undef;
        while(<LST>){
                chomp;
                /\S/ or next;
                /^"$file"/ and next;
                push @list_buf,$_;
        }
        close(LST);
        my $md5sum = `md5sum $buildroot/$file`;
        $md5sum =~ s/\s+\S+//;
        $md5sum =~ /^[a-f|0-9]{32}$/ || return undef;
        push @list_buf,"\"$file\" $md5sum";
        open (LST,">$buildroot/$list") || return undef;
        foreach (sort @list_buf){
                print LST "$_\n";
        }
        close(LST);
        return 1;
}

#-----------------------------------------------------------------------------

sub remove_file{
        my ($buildroot,$file,$list) = @_;
        $buildroot =~ s/\\/\//g;
        $buildroot =~ s/\/$//;
        $file =~ s/\\/\//g;
        my $list_file = $buildroot.'/'.$list;
        -f $list_file || return undef;
        my $path = $buildroot.'/'.$file;
        -f $path || return undef;
        unlink($path) || return undef;
        open (LST,$list_file) || return undef;
        my $found = 0;
        my @list_buf;
        while(<LST>){
                chomp;
                /\S/ or next;
                unless(/"$file"/){
					$found = 1;
					next;
                }
                push @list_buf,$_;
        }
        close(LST);
        return undef unless $found;
        open (LST,'>'.$list_file) || return undef;
        foreach ( @list_buf){
                print LST "$_\n";
        }
        close(LST);
        return 1;
}

#-----------------------------------------------------------------------------

sub do_cmd{
        my ($cmd) = @_;
        msg("processing $cmd\n");
        my $outtxt = `$cmd 2>&1`;
        if($? != 0){
                err("$cmd: $outtxt\n");
                return;
        }
        return 1;
}

#-----------------------------------------------------------------------------

our $max_tries = 30;
sub deltree{
    my (
        $dir, $skip_toplevel
    ) = @_;
    if(not -d $dir) {
    	return 0;
    }
	my @statbuf = stat ($dir);
	if (($^O !~ /mswin/i) && ($> == $statbuf[4]) && (($statbuf[2] & 0700) != 0700)){
		chmod ($statbuf[2] | 0700, $dir);
	}
    opendir(DH,$dir) || return 0;
    my @content = readdir(DH);
    closedir(DH);
    my @files = grep { -f "$dir/$_" || (!-e "$dir/$_" && -l "$dir/$_")} @content;
    my @dirs = grep { -d "$dir/$_" && ! /^\.{1,2}$/} @content;
    #
    #  enable write permission in directory to delete entries:
	if($^O !~ /mswin/i) {
		if($#files > -1 || $#dirs > -1){
			my @statbuf = stat($dir);
			my $mask;
			if($> == $statbuf[4]){
				$mask = 0200;
			}
			else{
				my $found;
				foreach my $gid (split(' ',$) )){
					if($gid == $statbuf[5]){
						$found = 1;
						last;
					} 
				}
				if($found){
					$mask = 020;
				}
				else{
					$mask = 002;
				}
			}
			if(($statbuf[2] & $mask) == 0){
				chmod($statbuf[2] | $mask,$dir) || err("cannot enable write permissions for $dir\n");
			}
		}
	}
    foreach (@dirs){
        -l "$dir/$_" and (unlink("$dir/$_") || err("cannot delete symbolic link $dir/$_: $!\n") && return 0);
        deltree("$dir/$_") || return 0;
    }
	my $success;
    foreach (@files){
        -l "$dir/$_" and (unlink("$dir/$_") || err("cannot delete symbolic link $dir/$_: $!\n") && return 0);
		if (-f "$dir/$_" && !unlink("$dir/$_")){
			if ($^O =~ /mswin/i){
				$success = 0;
				my $f = $_;
				foreach (0..($max_tries - 1)){
					sleep (1);
					print "RETRYING ($_) delete file $dir/$f\n";
					if (unlink("$dir/$f")){
						$success = 1;
						last;
					}
				}
				if (!$success){
					err("cannot delete file $dir/$f (tried $max_tries times): $!\n");
					return 0;
				}
			}
			else{
				err("cannot delete file $dir/$_: $!\n");
				return 0;
			}
		}
			
    }
    if (!$skip_toplevel){
        chmod (0777, $dir);
        rmdir($dir) or err("cannot delete directory $dir: $!\n") and return 0;
    }
    return 1;
}

#-----------------------------------------------------------------------------

sub makedir{
        my ($dir,$mode,$ref_uid,$ref_gid) = @_;
        $dir =~ s/\\/\//g;
        $dir =~ s/\/$//;
        -d $dir && return 1;
		my ($parent) = ($dir =~ /^(.*)\/[^\/]+$/);
	    
		if($dir !~ /\// || -d $parent){
				
				my $pperm_changed = 0;
				my @pstatbuf;

				if($parent =~ /\S/){
					if($^O !~ /mswin/i and $< != 0){
						@pstatbuf = stat ($parent);
						my $dirmode = $pstatbuf[2] & 0777;	
						if($pstatbuf[4] == $<){
							if(($pstatbuf[2] & 0200) == 0){
								$pperm_changed = 1;
								chmod($pstatbuf[2] | 0200,$parent);
							}
						}
					}
				}

				mkdir($dir,$mode) || return 0;
                unless($^O =~ /mswin/i){
					#
					# workaround for perl bug
					#
					my @statbuf = stat ($dir);
					my $filemode = $statbuf[2] & 0777;
					if ($filemode != $mode) {
						chmod ($mode, $dir) || return 0;
					}
					if(defined $$ref_uid || defined $$ref_gid){
						my ($uid,$gid) = ($statbuf[4],$statbuf[5]);
						defined $$ref_uid and $uid = $$ref_uid;
						defined $$ref_gid and $gid = $$ref_gid;
						chown($uid,$gid,$dir) or $! = "cannot change ownership of directory $dir: $!" and return 0; 
					}	
				}
				chmod($pstatbuf[2],$parent) if ($pperm_changed);
				return 1;
        }
        makedir($parent,$mode) || return 0;
 		
		my $pperm_changed = 0;
		my @pstatbuf;

		if($parent =~ /\S/){
			if($^O !~ /mswin/i and $< != 0){
				@pstatbuf = stat ($parent);
				my $dirmode = $pstatbuf[2] & 0777;	
				if($pstatbuf[4] == $<){
					if(($pstatbuf[2] & 0200) == 0){
						$pperm_changed = 1;
						chmod($pstatbuf[2] | 0200,$parent);
					}
				}
			}

		}
		
		mkdir($dir,$mode) || return 0;
		unless($^O =~ /mswin/i){
			#
			# workaround for perl bug
			#
			my @statbuf = stat ($dir);
			my $filemode = $statbuf[2] & 0777;
			if ($filemode != $mode) {
				chmod ($mode, $dir) || return 0;
			}
		}
		chmod($pstatbuf[2],$parent) if ($pperm_changed);
		return 1;
}

#-----------------------------------------------------------------------------

sub readini{
	my ($file) = @_;
	my %returnvalue;
	open(FH,$file) or return undef;
	my $section = 'main';
	while(<FH>){
		chomp;
		if(/^\s*\[.+\]\s*$/){
			($section) = (/^\s*\[(.+)\]\s*$/);	
		}
		elsif(/^.+=.+$/){
			my ($key,$value) = (/^(.+)=(.+)$/);
			$key =~ s/^\s*//;
			$key =~ s/\s*$//;
			$value =~ s/^\s*//;
			$value =~ s/\s*$//;
			if(exists $returnvalue{$section} && ref($returnvalue{$section}) eq  'HASH'){
				$returnvalue{$section}->{$key} = $value;		
			}
			else{
				$returnvalue{$section} = {$key => $value}; 	
			}
		}
	}
	close(FH);
	return %returnvalue;
}

#-----------------------------------------------------------------------------

sub find {
        my ($dir,$type,$root) = @_;
        my @returnvalue;
        my $tdir = defined $root ? $root . '/' . $dir : $dir;
        $tdir =~ s/\/+$// if ($tdir =~ /[^\/]/);

        -d $tdir or return undef;
        opendir(DH,$tdir) || return undef;
        my @content = readdir(DH);
        closedir(DH);

        my @files = grep { -f "$tdir/$_" } @content;
        my @dirs = grep { -d "$tdir/$_" && ! /^\.{1,2}$/} @content;


        foreach(@dirs){
                -l "$dir/$_" and next;
                my @result = find( $dir ? "$dir/$_" : $_,$type,$root);
                $#result > -1 && push @returnvalue,@result;
        }

        if($type !~ /^d$/i){
                foreach (@files){
                        push @returnvalue, $dir ? "$dir/$_" : $_;
                }
        }
        if($type !~ /^f$/i){
                foreach (@dirs){
                        push @returnvalue, $dir ? "$dir/$_" : $_;
                }
        }
        return @returnvalue;
}

#-----------------------------------------------------------------------------

sub modeStr2Num{
	my ($modstr) = @_;
	$modstr =~ /^[rwxs-]{9}$/ or err("mode string \"$modstr\" has wrong format\n"); 
	$modstr =~ s/[wrx]/1/g;
	$modstr =~ s/-/0/g;
	return unpack("N", pack("B32", substr("0" x 32 . $modstr, -32)));
}


#-----------------------------------------------------------------------------

sub normalizePath{
    my ($rPath) = @_;
    $$rPath =~ s/\\/\//g;
    $$rPath =~ s/\/{2,}/\//g;
    $$rPath =~ s/\/$//;
    if ($^O =~ /mswin/i){
        $$rPath =~ tr/[A-Z]/[a-z]/;
    }
    return 1;
}

#-----------------------------------------------------------------------------

our %path_cache;

#-----------------------------------------------------------------------------

sub getRealPathName{
	my ($path,$root) = @_;
	if ($^O =~ /mswin/i){
		return $path;
	}
	my $rv = '';

	my $cache_key = "$root/$path";

	normalizePath (\$cache_key);

	if (exists $path_cache{$cache_key}){
		return $path_cache{$cache_key};
	}

	my ($parentpath,$node)  = ($path =~ /(.*)[\\\/]+([^\\\/]+)$/);
	
	if (defined $parentpath){
		$rv = getRealPathName ($parentpath,$root);
		if (opendir (PP,"$root/$parentpath")){
			my $pattern = '^'.$node.'$';
			my ($result) = grep {/$pattern/i} readdir (PP);
			if (defined $result){
				$rv .= '/'.$result; 
			}
			else{
				$rv .= '/'.$node; 
			}
			closedir (PP);
		}
		else{
			$rv .= '/'.$node; 
		}
	}
	else{
		if (opendir (PP,"$root")){
			my $pattern = '^'.$node.'$';
			my ($result) = grep {/$pattern/i} readdir (PP);
			if (defined $result){
				$rv = $result; 
			}
			else{
				$rv = $node; 
			}
			closedir (PP);
			
		}
		else{
			$rv = $node;
		}	
		
	}
	$path_cache{$cache_key} = $rv;
	return $rv;
}

#--------------------------------------------------------------------------------------------------

sub DecodeBoolTag{
    my (
        $entry
    ) = @_;
    my $retcode = 0;
    if(defined $entry) {
        if($entry eq 'yes' || $entry eq 'true' || $entry eq '1') {
            $retcode = 1;
        }
    }
    return $retcode;
}

#-----------------------------------------------------------------------------

sub enumFilesRecursively {
    my (
        $absolutePrefix, # an absolute path used to prefix $dir. Optional; if undef,
                         # $dir must be absolute.
        $dir,            # return value is relative to the parent directory of the highest directory
                         # given by $dir.
        $fileHashRef,    # return value, must be a hash ref.
        $lowerCaseOutput,
        $includeFileNamePattern,
        $excludeFileNamePattern,
        $includeOnlyWhenExtensionPatternExists,
        $packageInFiles,
        $errObj,         # error handler, typically the calling object, optional
        $depth           # max search depth
    ) = @_;
    $dir =~ s/(^\s*|\s*$)//g;
    my $fullPath = $dir;
    if(defined $absolutePrefix) {
	    $absolutePrefix =~ s/(^\s*|\s*$)//g;
	    if(!$absolutePrefix eq '') {
	    	$absolutePrefix =~ s/(\\$|\/$)//g;
	    	$fullPath = $absolutePrefix.'/'.$dir;
	    }
    }
    if(!-d $fullPath) {
    	if(defined $errObj) {
    	   $errObj->{'last_error'} = "$fullPath does not exist or is not a directory";
    	}
        return undef;
    }
    if(!defined $lowerCaseOutput) {
    	$lowerCaseOutput = 0;
    }
    my $rc = undef;
    $rc = opendir(DH, $fullPath);
    if(!$rc) {
        $errObj->{'last_error'} = "Could not open directory $fullPath";
        return undef;
    }
    my @content = readdir(DH);
    closedir(DH);
    my @files = grep { -f "$fullPath/$_" } @content;
    if (!defined $depth || $depth > 0){
        my @dirs = grep { -d "$fullPath/$_" && ! /^\.{1,2}$/} @content;
        if (defined $depth){
            $depth--;
        }
        foreach(@dirs){
            if(-l "$fullPath/$_") {
                # we don't follow symlinks
                next;
            }
            $rc = enumFilesRecursively($absolutePrefix,
                                       "$dir/$_",
                                       $fileHashRef,
                                       $lowerCaseOutput,
                                       $includeFileNamePattern,
                                       $excludeFileNamePattern,
                                       $includeOnlyWhenExtensionPatternExists,
                                       $packageInFiles,
                                       $errObj,
                                       $depth);
            if(!defined $rc) {
            	return undef;
            }
        }
    }
    foreach (@files){
    	my $file = "$dir/$_";
    	$file =~ s/\\/\//g;
    	my $include = 1;
    	my $exclude = 0;
        if(defined $includeFileNamePattern) {
        	$include = 0;
        	if($file =~ /$includeFileNamePattern/) {
                if(defined $includeOnlyWhenExtensionPatternExists) {
                    my $basename = basename($file);
                    $basename =~ s/(.*)\..*$/$1/;
                    # grepping the "in_files" structure for each file again and again is inefficient
                    # for very large packages, using a hash could improve it:
                    my @exts = grep { $_->{'path'} =~ /$basename$includeOnlyWhenExtensionPatternExists$/ } @{$packageInFiles};
                    if(@exts) {
                        $include = 1;
                    }
                }
                else {
                    $include = 1;
                }
        	}
        }
        if(defined $excludeFileNamePattern && $file =~ /$excludeFileNamePattern/) {
            $exclude = 1;
        }
        if($include && not $exclude) {
	    	if($lowerCaseOutput) {
	    		$file = lc $file;
	    	}
	        $fileHashRef->{$file} = $file;
        }
    }
    return $fileHashRef;
}

#----------------------------------------------------------------------------

=functionComment

performs a deep copy of the data structure referenced by the parameter.
Returns a reference to the copy.
self-referential structures will cause infinite recursions, and
cross-referential structures will cause duplications!

=cut

sub dataDeepCopy {
    my (
        $p_srcRef
    ) = @_;
    my $refType = ref($p_srcRef);
    my $tgtRef = undef;
    if($refType eq "ARRAY") {
    	$tgtRef = [];
        foreach my $value (@$p_srcRef) {
                push(@$tgtRef, dataDeepCopy($value));
        }
    }
    elsif($refType eq "HASH") {
    	$tgtRef = {};
        while (my ($key,$value) = each %$p_srcRef) {
                $tgtRef->{$key} = dataDeepCopy($value);
        }
    }
    elsif($refType eq "SCALAR") {
    	my $value = $$p_srcRef;
    	$tgtRef = \$value;
    }
    elsif($refType =~ /^(CODE|REF|GLOB|LVALUE|FORMAT|IO|VSTRING|Regexp)$/) {
    	$tgtRef = $p_srcRef;
    }
    elsif($refType =~ /\S/) { 
    	$tgtRef = {};
        while (my ($key,$value) = each %$p_srcRef) {
                $tgtRef->{$key} = dataDeepCopy($value);
        }
    }
    else {
        $tgtRef = $p_srcRef;
    }
    return $tgtRef;
}

#--------------------------------------------------------------------------------------------------

sub createTgz{
    my ($archive, $rootDir, $tarFileList) = @_;
    require SAPDB::Install::GnuTar;
    my $gnutar = new SAPDB::Install::GnuTar ();
    $gnutar->OpenArchive ($archive, 'w');
    my $fh;
    if (!open ($fh, $tarFileList)){
        $gnutar->CloseArchive ();
       err ("... Error, cannot open tar file list  '$tarFileList': $!\n");
       next;
    }
    my $error = 0;
    my @expand;
    my ($fileName, $filePath);
    while ($fileName = <$fh>){
        if (!$fileName){
            next;
        }
        chomp ($fileName);
        $filePath = $rootDir . '/'. $fileName;
        if (!-e $filePath){
            err ("... Error, cannot add file '$filePath' to archive '$archive': $!\n");
            return undef;
        }
        if (-d _ ){
            @expand = find ('', 'f', $filePath);
        }
        else{
            @expand = ();
        }
        if (!defined $gnutar->Add($filePath, $fileName)){
            err ("... Error, cannot add file '$filePath' to archive '$archive': $!\n");
            $error = 1;
            last;
        }
        foreach my $foundFile (@expand){
            if (!defined $gnutar->Add($filePath . '/' . $foundFile, $fileName . '/' . $foundFile)){
                err ("... Error, cannot add file '$filePath/$foundFile' to archive '$archive': $!\n");
                $error = 1;
                last;
            }
        }
        if ($error){
            last;
        }
    }
    close ($fh);
    $gnutar->CloseArchive ();
    if ($error){
        return undef;
    }
    return 1;
}

sub createInstallerPackage{
    my (
        $archive,         # 1
        $pData,           # 2
        $list,            # 3
        $script,          # 4
        $content,         # 5
        $pmask,           # 6
        $pigz,            # 7
        $threads,         # 8
        $pigzRsyncable   # 9
    ) = @_;
    require SAPDB::Install::GnuTar;
    my $gnutar = new SAPDB::Install::GnuTar ();
    if (defined $pigz && $threads > 1){
        my @compressorArgs = ('-p',$threads, '-9', '-c');
        if ($pigzRsyncable){
            push @compressorArgs, '--rsyncable';
        }
        my $rc = $gnutar->setCompressor ($pigz, \@compressorArgs);
    }
    if (!defined $gnutar->OpenArchive ($archive, 'w', $pmask)){
      err ("... Error, cannot create archive '$archive': $!");
      return undef;
    }
    my $fh;
    my $packageListFile = "$archive.lst";
    if (!open ($fh, '>', $packageListFile)){
       err ("... Error, cannot open file '$packageListFile': $!");
       return undef;
    }
    print $fh "LSS-HASHES\nHash: SHA256\n\n";
    if (defined $pData){
        my $fileKey = 'PACKAGEDATA';
	    if (!defined $gnutar->AddScalar ($fileKey, $pData)){
	        err ("... Error, add PACKAGEDATA to archive  '$archive': $!");
	        return undef;
	    }
        my $entry = createPackageListEntry($fileKey, undef, $pData);
        print $fh $entry;
    }
    if (defined $list){
        my $fileKey = 'files.lst';
	    if (!defined $gnutar->AddScalar ($fileKey, $list)){
	        err ("... Error, add file list to archive '$archive': $!");
	        return undef;
	    }
        my $entry = createPackageListEntry($fileKey, undef, $list);
        print $fh $entry;
    }
    if (defined $script){
        my $fileKey = 'script.pm';
        if (!defined $gnutar->Add ($script, $fileKey,1,0440)){
            err ("... Error, add script to archive '$archive': $!");
            return undef;
        }
        my $entry = createPackageListEntry($fileKey, $script);
        print $fh $entry;
    }
    my (@localCopy, $type);
    foreach my $i (0 .. (scalar (@$content) - 1)){
        my ($type,@localCopy) = @{$content->[$i]};
        if ($type == 0){
            if (!defined $gnutar->Add (@localCopy)){
                err ("... Error, add file '$localCopy[0]' to archive '$archive': $!");
                return undef;
            }
            my $from = $localCopy[0];
            my $fileKey = $localCopy[1];
            my $entry = createPackageListEntry($fileKey, $from);
            return undef if (! defined $entry);
            print $fh $entry;
        }elsif ($type == 1){
            if (!defined $gnutar->AddLink (@localCopy)){
                err ("... Error, add link '$localCopy[0]' to archive '$archive': $!");
                return undef;
            }
            my $fileKey = $localCopy[0];
            my $linkTarget = $localCopy[1];
            my $entry = createPackageListEntry($fileKey, undef, $linkTarget);
            return undef if (! defined $entry);
            print $fh $entry;
        }
    }
    if (!close ($fh)){
       err ("... Error, cannot close file '$packageListFile': $!");
       return undef;
    }
    if (!defined $gnutar->CloseArchive ()){
        err ("... Error, cannot close archive '$archive': $!");
        return undef;
    }

    return 1;
}

sub _getFileShaHashCode {
    my ($filepath) = @_;

    my $sha = Digest::SHA->new(256);
    $sha->addfile($filepath);
    return $sha->hexdigest();
}

sub createPackageListEntry{
    my ($fileKey, $from, $content) = @_;

    my $checksum;
    my $isExecutable;
    if (defined $from) {
        $checksum = _getFileShaHashCode($from);
        $isExecutable = isExecutable($from);
        return undef if (! defined $isExecutable);
    } else {
        $checksum = sha256_hex($content);
        $isExecutable = 0;
    }
    my $execChar = $isExecutable ? 'X' : ' ';
    return "$checksum $execChar $fileKey\n";
}

our @ELF = ('7F', '45', '4C', '46' );
sub isExecutable{
    my ($from) = @_;

    my ($fh, $buffer);
    if (!open($fh, $from)) {
       err ("... Error, cannot open file '$from': $!");
       return undef;
    }
    binmode($fh);
    if (!eof($fh) && !read($fh, $buffer, 4)) {
       err ("... Error, cannot read from file '$from': $!");
       return undef;
    };
    return 0 if (length $buffer < 4);
    my $rc = 1;
    for (0 .. 3) {
        my $hex = sprintf('%02X', ord(substr($buffer, $_, 1)));
        if ($hex ne $ELF[$_]) {
            $rc = 0;
            last;
        }

    }
    if (!close($fh)) {
       err ("... Error, cannot close file '$from': $!");
       return undef;
    }
    return $rc;
}

#--------------------------------------------------------------------------------------------------



1;






