package LCM::Component;

use strict;
use warnings;

use SDB::Install::Configuration::ExternalProgramConfiguration;
use SDB::Install::SysVars qw($isWin $path_separator $currentPlatformName);
use SDB::Install::Globals qw ($gProductNameAccelerator
                              $gProductNameEs
                              $gProductNameStreaming
                              $gProductNameRDSync
                              $gProductNameXS2
                              $gProductNameEngine
                              $gProductNameClient
                              $gProductNameStudio
                              $gProductNameAFL
                              $gProductNameLCA
                              $gProductNameHLM
                              $gProductNameSDA
                              $gProductNameLSS
                              $gKeynameEngine
                              $gKeynameClient
                              $gKeynameStudio
                              $gKeynameAFL
                              $gKeynameLCA
                              $gKeynameHLM
                              $gKeynameSDA
                              $gKeynameStreaming
                              $gKeynameEs
                              $gKeynameAccelerator
                              $gKeynameRDSync
                              $gKeynameXS
                              $gKeynameXS2
                              $gKeynameOfficeClient
                              $gKeynameInstaller
                              $gKeynameCockpitStack
                              $gKeynameLSS
                              $gSignatureManifestName
                              determineSignedManifestRelativePath);
use Exporter;
use base qw (SDB::Install::Base Exporter LCM::Component::Registrable);
use LCM::Component::Registrable qw($RETURN_CODE_SUCCESS $RETURN_CODE_ERROR);
use LCM::TraceLoggingEnvironment;
use SAPDB::Install::Hostname;
use SDB::Install::RemoteHostctrlHosts;
use SDB::Install::NewDBUser;
use SDB::Common::Utils qw(trim);
use LCM::Component::Installable::InstallationKitChecker;
use SDB::Install::Tools qw (getFilenameTimeStamp repeat);
use File::Spec;
use File::Basename;
use LCM::Manifests::SAPSignature;
use SDB::Install::HdbInstallerOutputParser qw (parseHdbInstallerErrorMessages);
use SDB::Install::System qw (getSAPDrive);
use File::stat;
use SDB::Install::LSS::TrustLevelModifier::TrustLevelModifierFactory;

use constant{
    PREPARE_PHASE   => "prepare",
    OFFLINE_PHASE   => "offline",
    CONFIGURE_PHASE => "configure",
    ONLINE_PHASE    => "online",
    SYSTEM_ONLINE_PHASE  => "system_online",
    ALL             => "all",
};


our @EXPORT = qw (
    COMPONENT_NAME_INSTALLER
    STATUS_INITIAL
    STATUS_FINISHED_SUCCESS
    STATUS_FINISHED_ERROR
    STATUS_FINISHED_WARNING
    $componentKeynameToBatchKey
    parseLogFileLocation
    PREPARE_PHASE
    OFFLINE_PHASE
    CONFIGURE_PHASE
    ONLINE_PHASE
    SYSTEM_ONLINE_PHASE
    ALL
    $componentBatchKeyToKeycaption
    getComponentKeyCaptionByKeyName
);

sub COMPONENT_NAME_INSTALLER    {'Resident hdblcm'}

use constant STATUS_INITIAL     => 0;
use constant STATUS_FINISHED_SUCCESS     => 1;
use constant STATUS_FINISHED_ERROR  => 2;
use constant STATUS_FINISHED_WARNING  => 3;

our $componentKeynameToBatchKey = {
		$gKeynameEngine       => "server",
		$gKeynameClient       => "client",
		$gKeynameStudio       => "studio",
		$gKeynameAFL          => "afl",
		$gKeynameLCA          => "lcapps",
		$gKeynameHLM          => "hlm",
		$gKeynameSDA          => "smartda",
		$gKeynameStreaming    => "streaming",
		$gKeynameEs           => "es",
		$gKeynameAccelerator  => "ets",
		$gKeynameRDSync       => "rdsync",
		$gKeynameXS2          => "xs",
		$gKeynameCockpitStack => "cockpit",
		$gKeynameLSS          => "lss",
};

our $internalComponentKeynameToBatchKey = {
	$gKeynameInstaller   => "hdblcm",
};

our $keynameToComponentKeycaption = {
        $gKeynameEngine       => "SAP HANA Database",
        $gKeynameClient       => "SAP HANA Database Client",
        $gKeynameStudio       => "SAP HANA Database Studio",
        $gKeynameAFL          => "SAP HANA AFL (Misc)",
        $gKeynameLCA          => "SAP HANA LCAPPS",
        $gKeynameHLM          => "SAP HANA Lifecycle Manager",
        $gKeynameSDA          => "SAP HANA Smart Data Access",
        $gKeynameStreaming    => "$gProductNameStreaming",
        $gKeynameEs           => "$gProductNameEs",
        $gKeynameAccelerator  => "$gProductNameAccelerator",
        $gKeynameRDSync       => "$gProductNameRDSync",
        $gKeynameXS2          => "$gProductNameXS2",
        $gKeynameLSS          => "$gProductNameLSS",
};

our $componentBatchKeyToKeycaption = {
        $componentKeynameToBatchKey->{$gKeynameEngine}      => $gProductNameEngine,
        $componentKeynameToBatchKey->{$gKeynameClient}      => $gProductNameClient,
        $componentKeynameToBatchKey->{$gKeynameStudio}      => $gProductNameStudio,
        $componentKeynameToBatchKey->{$gKeynameAFL}         => $gProductNameAFL,
        $componentKeynameToBatchKey->{$gKeynameLCA}         => $gProductNameLCA,
        $componentKeynameToBatchKey->{$gKeynameHLM}         => $gProductNameHLM,
        $componentKeynameToBatchKey->{$gKeynameSDA}         => $gProductNameSDA,
        $componentKeynameToBatchKey->{$gKeynameStreaming}   => $gProductNameStreaming,
        $componentKeynameToBatchKey->{$gKeynameEs}          => $gProductNameEs,
        $componentKeynameToBatchKey->{$gKeynameAccelerator} => $gProductNameAccelerator,
        $componentKeynameToBatchKey->{$gKeynameXS2}         => $gProductNameXS2,
        $componentKeynameToBatchKey->{$gKeynameLSS}         => $gProductNameLSS,
};

sub new {
    my ( $class, $manifestDir, $manifest, $componentManager, $instconfig, $isSidAdmUserExecution ) = @_;
    my $self = $class->SUPER::new ();

	$manifestDir = File::Spec->canonpath($manifestDir);

    $self->{manifestDir} = $manifestDir;
    $self->{manifest} = $manifest;
    $self->{componentManager} = $componentManager;
    $self->{instconfig} = $instconfig;
	$self->setSidAdmUserExecution($isSidAdmUserExecution);
    $self->setStatus(STATUS_INITIAL);
    $self->{warnings} = [];

    return $self;
}

sub getAllComponentsBatchKeys {
	my ($isUninstall) = @_;

	my @componentValues = (
		$componentKeynameToBatchKey->{$gKeynameClient},
		$componentKeynameToBatchKey->{$gKeynameEs},
		$componentKeynameToBatchKey->{$gKeynameAccelerator},
		$componentKeynameToBatchKey->{$gKeynameLCA},
		$componentKeynameToBatchKey->{$gKeynameEngine},
		$componentKeynameToBatchKey->{$gKeynameSDA},
		$componentKeynameToBatchKey->{$gKeynameStreaming},
		$componentKeynameToBatchKey->{$gKeynameRDSync},
		$componentKeynameToBatchKey->{$gKeynameXS2},
		$componentKeynameToBatchKey->{$gKeynameStudio},
	);

	push (@componentValues, $componentKeynameToBatchKey->{$gKeynameHLM}) if ($isUninstall);
	push (@componentValues, $componentKeynameToBatchKey->{$gKeynameAFL});

	my $productSpecificAFLs = ['sca', 'sop', 'eml', 'rme', 'rtl', 'trp'];

	push (@componentValues, @$productSpecificAFLs);

	return @componentValues;
}

sub addComponentInfosToMsgLst{
    my ($self,$msglst) = @_;
    $msglst->addMessage ("Product: ".$self->getComponentName());
    $msglst->addMessage ("Version: ".$self->{manifest}->getVersion());
    my $gitHash = $self->{manifest}->getValue ('git-hash');
    if (defined $gitHash){
       $msglst->addMessage ("GitHash: " . $gitHash );
    }
}

sub getPath{
    return $_[0]->{manifestDir};
}

sub setPath{
    $_[0]->{manifestDir} = $_[1];
}

sub selectComponent{
    $_[0]->{isSelected} = defined $_[1] ? $_[1] : 1;
}

sub isComponentSelected{
    return $_[0]->{isSelected};
}

sub canSelectComponent{
	return defined $_[0]->{canSelect} ? $_[0]->{canSelect} : $_[0]->isInternal() ? 0 : 1;
}

sub setCanSelectComponent{
	$_[0]->{canSelect} = $_[1];
}

sub canUpgradeComponent{
    return $_[0]->{canUpgrade};
}

sub setCanUpgradeComponent {
    $_[0]->{canUpgrade} = $_[1];
}

sub isInternal{
	return 0;
}

sub getManifest{
    return $_[0]->{manifest};
}

sub setManifest{
    $_[0]->{manifest} = $_[1];
}

sub getVersion{
	return $_[0]->{manifest}->getVersion();
}

sub getVersionObject {
    return $_[0]->getManifest()->getVersionObject();
}

sub getComponentName{
    return $_[0]->getManifest ()->getValue ('keycaption');
}

sub getComponentKeyName{
    return $_[0]->getManifest ()->getValue ('keyname');
}

sub getComponentId{
    return $_[0]->getManifest ()->getCompId ();
}

sub getComponentManager{
    return $_[0]->{componentManager};
}

sub getVerificationString {
    my ($self) = @_;
    return $self->getComponentName();
}

sub getComponentBatchKey{
	my $self = shift;
	my $key = $self->getManifest()->getComponentKey();
	
	if (defined $key) {
		return $key;
	}

	my $batchKey = $componentKeynameToBatchKey->{$self->getComponentKeyName()};
	if (defined $batchKey){
		return $batchKey;
	}

	return $internalComponentKeynameToBatchKey->{$self->getComponentKeyName()};
}

sub getComponentKeyCaptionByKeyName{
    my $caption = $keynameToComponentKeycaption->{$_[1]};
    if (not defined $caption) {
       $caption = $_[1];
    }
    return $caption;
}

sub getNumberOfExpectedOutputLines{
    return undef;
}

sub getIgnorePersistency{
    return $_[0]->{ignorePersistency};
}

sub setIgnorePersistency{
    $_[0]->{ignorePersistency} = $_[1];
}

sub setProgressHandler{
    $_[0]->{progressHandler} = $_[1];
}

sub setLogLocation{
	my ($self, $logLocation) = @_;
	my $phase = $self->getPhase();

	$self->{"logLocation"} = $logLocation;

	if (defined $phase) {
		if ($phase eq PREPARE_PHASE) {
			$self->setLogLocationForPreparePhase($logLocation);
		} elsif ($phase eq OFFLINE_PHASE) {
			$self->setLogLocationForOfflinePhase($logLocation);
		} elsif ($phase eq CONFIGURE_PHASE) {
            $self->setLogLocationForConfigurePhase($logLocation);
        } elsif ($phase eq ONLINE_PHASE) {
			$self->setLogLocationForOnlinePhase($logLocation);
		} elsif ($phase eq SYSTEM_ONLINE_PHASE) {
            $self->setLogLocationForSystemOnlinePhase($logLocation);
        }
	}
}

sub getLogLocation{
    my ($self) = @_;
    return $self->{logLocation};
}

sub setLogLocationForPreparePhase{
    my ($self, $logLocation) = @_;
    $self->{logLocationPreparePhase} = $logLocation;
}

sub getLogLocationForPreparePhase{
    my ($self) = @_;
    return $self->{logLocationPreparePhase};
}

sub setLogLocationForOfflinePhase{
    my ($self, $logLocation) = @_;
    $self->{logLocationOfflinePhase} = $logLocation;
}

sub getLogLocationForOfflinePhase{
    my ($self) = @_;
    return $self->{logLocationOfflinePhase};
}

sub setLogLocationForConfigurePhase {
    my ($self, $logLocation) = @_;
    $self->{logLocationConfigurePhase} = $logLocation;
}

sub getLogLocationForConfigurePhase {
    my ($self) = @_;
    return $self->{logLocationConfigurePhase};
}

sub setLogLocationForOnlinePhase{
    my ($self, $logLocation) = @_;
    $self->{logLocationOnlinePhase} = $logLocation;
}

sub getLogLocationForOnlinePhase{
    my ($self) = @_;
    return $self->{logLocationOnlinePhase};
}

sub setLogLocationForSystemOnlinePhase{
    my ($self, $logLocation) = @_;
    $self->{logLocationSystemOnlinePhase} = $logLocation;
}

sub getLogLocationForSystemOnlinePhase{
    my ($self) = @_;
    return $self->{logLocationSystemOnlinePhase};
}

sub setStatus{
    my ($self, $status) = @_;
    $self->{status} = $status;
}

sub getStatus{
    my ($self) = @_;
    return $self->{status};
}

sub initProgressHandler{
    my ($self) = @_;
    my $progressHandler = $self->getProgressHandler();
    return 1 if !$progressHandler;

    my $numberOfLines = $self->getNumberOfExpectedOutputLines ();
    $progressHandler->InitProgress($numberOfLines, 0) if defined($numberOfLines);
    $progressHandler->incrementIndentationDepth() if $progressHandler->can('incrementIndentationDepth');
}

sub getProgressHandler{
    my ($self) = @_;
    return $self->{progressHandler};
}


sub getExternalHdbInstallerConfiguration{
    my ($self) = @_;
    my $instconfig = new SDB::Install::Configuration::ExternalProgramConfiguration (
                            $self->getHdbInstallExecutable(),
                            ["--archive_dir=$self->{manifestDir}"]);
    my $msg = $self->getMsgLst ()->addMessage ("Detecting known command line options");
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
    my $rc = $instconfig->init();
    if (!defined $rc){
        $self->setErrorMessage ("Cannot detect command line options dynamically.",
                                    $instconfig->getErrMsgLst ());
        return undef;
    }
    if (!$rc) {
        return $self->getFallbackHdbInstallerConfiguration();
    }
    return $instconfig;
}

sub getLcmSdkVersion{
    my ($self, $executable) = @_;
    my $externalConfig = SDB::Install::Configuration::ExternalProgramConfiguration->new($executable);
    $externalConfig->setMsgLstContext($self->getMsgLstContext());
    my $rc = $externalConfig->init();
    if (!defined $rc){
        $self->setErrorMessage("Cannot detect LCMsdk version consumed by $gProductNameLSS.", $externalConfig->getErrMsgLst());
        return undef;
    }
    my $version = $externalConfig->getInstallerVersion();
    $self->getMsgLst()->addMessage("Detected LCMsdk version of $gProductNameLSS is $version.");
    return $version;
}

sub getFallbackHdbInstallerConfiguration {
    return SDB::Install::Configuration::Generic->new();
}

#
# get hdbinst output and parse for top level error messages
# returns a SDB::Install::MsgLst containing the error messages
#

sub getHdbInstallerErrorMessages{
    my ($self, $outputLines) = @_;
    return parseHdbInstallerErrorMessages ($outputLines);
}

sub parseLogFileLocation {
	my ($self, $outputLines) = @_;
	if (!defined $outputLines){
		return undef;
	}
	my $log_location;
	my $pattern = "Log file written to \'(.*?)\'";
	foreach my $line (reverse @$outputLines){
		( $log_location ) = $line =~ /$pattern/i;
		
		if (defined $log_location) {
			$log_location =~ s/\.log/\.msg/g;
			if(-f $log_location){
			     return $log_location;
			}
			$log_location =~ s/\.msg/\.log/g;
			return $log_location;
		}
	}
}


#
# Make sure that logs and traces are not overwritten by several HDB installers.
# Append _<component keyName> to global log file copy and installer trace, which
# are defined in the process environment.
#

sub prepareHdbInstallerEnvironment{
	my($self,$componentKeyName) = @_;
    return LCM::TraceLoggingEnvironment::PrepareComponentHdbInstallerEnvironment($self,$componentKeyName);
}


# Returns the path (e.g. Linux: /hana/shared/<SID>', Windows: '/usr/sap/<SID>')
sub getInstSidPath {
    my ( $self, $instconfig ) = @_;

    my $instPath = undef;
    my $sid      = $instconfig->getValue('SID');

    if($isWin) {
        my $drive = $instconfig->getValue('Drive');
        if (!defined $drive) {
            $drive = getSAPDrive ($self->getErrMsgLst ());
            if (!defined $drive){
                return undef;
            }
        }
        $instPath = File::Spec->catfile($drive, 'usr', 'sap', $sid);
    }
    else {
        my $target = $instconfig->getValue('Target');
        $instPath = File::Spec->catfile($target, $sid);
    }
    return $instPath;
}

#----------------------------------------------------------------------------
#
# Configure HDB components.
#
# $_[0] - self
# $_[1] - instconfig
# $_[2] - ref to array containing hdbcomponents that are to be configured
# the name is used to form the path to the component.
# For example 'hdbclinet' is the name of the hdbclient and it's directory
# 
# $_[3] - shared directory
# $_[4] - SID
# $_[5] - LCM::HDBComponentsConfigurator action
# $_[6] - fail message
# $_[7] - remote hosts list or undef if the system is not distributed
#
sub configureHdbComponents {
    my ($self, $loggerClass, $componentNames, $sharedDir, $sid, $action) = @_;
    my $configurator = $self->getHDBComponentsConfigurator ();
    my $saveCntxt = $self->setMsgLstContext([$loggerClass->getMsgLst ()]);
    my $rc;

    $rc = $configurator->configureLocalHdbComponents ( $componentNames, $action, $sharedDir, $sid );
    $self->setMsgLstContext($saveCntxt);
    if($self->hasError ($rc, $self->getComponentsConfigurationFailedMsg($componentNames, $action))){
     	$self->setMsgLstContext($saveCntxt);
       	return undef;	
     }
    return 1;
}

sub getHDBComponentsConfigurator {
    # do not set log location to the hdbclientreg log file
    my $doNotSetLogLocation = 1;
    return LCM::HDBComponentsConfigurator->new ( $_[0], $doNotSetLogLocation );
}

sub getComponentsConfigurationFailedMsg {
    my ($self, $componentNames, $action) = @_;
    my $componentCaptions = join ', ', @{$componentNames};
    my $msgTempalte = "Failed to %s %s on the localhost.";
    ($action) = $action =~ /^action_(.+)$/g;
    $action //= 'configure';

    return sprintf($msgTempalte, $action, $componentCaptions);
}

sub isLSS{
    my ($self) = @_;
    my $manifest = $self->getManifest();
    return 1 if (defined $manifest && $manifest->isLSS());
    return 0;
}

sub isServer{
	my ($self) = @_;
	my $manifest = $self->getManifest();
	return 1 if (defined $manifest && $manifest->isServer());
	return 0;
}

sub isClient{
    my ($self) = @_;
    my $manifest = $self->getManifest();
    return 1 if (defined $manifest && $manifest->isClient());
    return 0;
}

sub isXS2{
    my ($self) = @_;
    my $manifest = $self->getManifest();
    return 1 if (defined $manifest && $manifest->isXS2());
    return 0;
}

sub isCockpitStack {
    my ($self) = @_;
    my $manifest = $self->getManifest();
    return 1 if (defined $manifest && $manifest->isCockpitStack());
    return 0;
}

sub isXS2Application {
    my ($self) = @_;
    my $manifest = $self->getManifest();
    return 1 if (defined $manifest && $manifest->isXS2Application());
    return 0;
}

sub isInstaller {
    my ($self) = @_;
    my $manifest = $self->getManifest();
    return 1 if (defined $manifest && $manifest->isInstaller());
    return 0;
}

sub isServerPlugin{
	my ($self) = @_;
	my $manifest = $self->getManifest();
	if (defined $manifest && $manifest->isServerPlugin()) {
		return 1;
	}
	my $keyname = $self->getComponentKeyName();
	if ($keyname eq $gKeynameAFL || $keyname eq $gKeynameLCA) {
		return 1;
	} else {
		return 0;
	}
}

sub hasError {
    my ( $self, $rc, $message, $componentHandler ) = @_;
    my $logger = $componentHandler ? $componentHandler : $self;
    if ( !defined $rc ) {        
        $logger->getErrMsgLst()->addMessage($message);
        if($logger->can("getProgressHandler"))
        {
            my $progressHandler = $logger->getProgressHandler();
            if(defined $progressHandler && $progressHandler->can("StepFinished"))
            {
                $progressHandler->StepFinished($message);
            }
        }
        return 1;
    }
    return 0;
}

sub addWarning {
	my ( $self, $warning ) = @_;
    
    push (@{$self->{warnings}}, $warning);
}

sub getWarnings {
	my ($self) = @_;
	return $self->{warnings};
}

sub resetWarnings {
	my ($self) = @_;
	$self->{warnings} = [];
}

sub setPersistedVersion {
	my ( $self, $persistedVerison ) = @_;
	$self->{_persistedVersion} = $persistedVerison;
}

sub getPersistedVersion {
	my ($self) = @_;
	return $self->{_persistedVersion};
}

sub _executeThroughSHAOperation {
	my ($self, $instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs, $action, $customPasswordInput, $hosts) = @_;

	my $rc = 1;
	my $hostname = hostname();

	my $cmpName = $self->getComponentName();
	my $progress_message = "$action of $cmpName on host '%s'...";
	my $done_message     = "Successful $action of $cmpName on host '%s'.";
	my $error_message    = "$action of $cmpName on host '%s' failed!";

    $hosts //= [$hostname];
	my $remoteExecution = new SDB::Install::RemoteHostctrlHosts(@{$hosts});

	if ($isShowProgressMsgs) {
		$remoteExecution->setOutputHandler ($hostname, $self->getProgressHandler ());
	}

	my $password = $instconfig->getValue('Password');
    if (defined $password) {
		$remoteExecution->useSidadm();
	}

	my $msg = $self->getMsgLst ()->addMessage ("$action of $cmpName through SAP Host Agent Operation.");
	$remoteExecution->resetMsgLstContext ();
	my $saveCntxt = $remoteExecution->setMsgLstContext ([$msg->getSubMsgLst()]);

	my $remoteExecutionReturnCode = $remoteExecution->executeHostctrlParallel($operation,
                                                        $instconfig, # instconfig
                                                        undef, # param IDs
                                                        $passwordKeys, # password IDs
                                                        undef, # remote ignore
                                                        undef, # remote timeout
                                                        $progress_message, # progress_message,
                                                        $done_message, # done_message,
                                                        $error_message, # error_message,
                                                        $optionMap,
                                                        undef, # host option map
                                                        $hosts, # only on hosts
                                                        undef, # do not fail
														1, # Suppress Console msgs
                                                        undef,
                                                        $customPasswordInput # modified xml password stream
														);
	if ($self->_hasError($remoteExecution->getExitCode(0))) { # retrieve the actual exit code from the operation
            $instconfig->PushError ("$action of $cmpName through SAP Host Agent Operation failed!", $remoteExecution);
            $rc = undef;
	}

	$self->setLogLocation($self->parseLogFileLocation($remoteExecution->getOutputBuffer()));
	$remoteExecution->setMsgLstContext ($saveCntxt);

	return $rc;
}

sub _hasError {
	my ($self, $returnCode) = @_;
	return !defined($returnCode) || ($returnCode != 0);
}

sub buildSHAOptionsMap {
	my ($self, $instconfig, $optionMap, $args, $isChangeHdbLogsOwnership) = @_;

	if (not defined $optionMap) {
		$optionMap = {};
	}

	for my $arg (@$args) {
		$arg =~ s/^(-{1,2})//;
		my @optionValueArr = split(/=/, $arg);
		my $option = $optionValueArr[0];
		my $value = $optionValueArr[1];
		if (($self->_isParamTypeBoolean($instconfig, $option)) and (not defined $value)) {
				$value = 'on';
		}
		if($option eq 'machine_utilization'){
			$optionMap->{'M_U'} = $value;
			next;
		}
		if($option eq 'skip_hostagent_password'){
			$optionMap->{'S_H_P'} = $value;
			next;
		}
		if($option eq 'skip_modify_sudoers'){
			$optionMap->{'SUD'} = $value;
			next;
		}
        if($option eq 'secure_store'){
            $optionMap->{'S_S'} = $value;
            next;
        }
        if($option eq 'remote_execution'){
            $optionMap->{'R_E'} = $value;
            next;
        }
        if($option eq 'install_hostagent'){
            $optionMap->{'I_H'} = $value;
            next;
        }

		$optionMap->{uc($option)} = $value;
	}

	if ($isChangeHdbLogsOwnership) {
		my $user = new SDB::Install::NewDBUser($instconfig->getValue('SID'));
		my $uid = $user->uid();
		my $gid = $user->gid();
		if (defined $uid && defined $gid) {
			$optionMap->{'INSTLOG_UID'} = $uid;
			$optionMap->{'INSTLOG_GID'} = $gid;
		}
	}

	return $optionMap;
}

sub _isParamTypeBoolean {
    my ($self, $instconfig, $option) = @_;

    for my $paramKey (keys %{$instconfig->{params}}){
        my $opt = $instconfig->{params}->{$paramKey}->{opt};
        my $type = $instconfig->{params}->{$paramKey}->{type};
        next if (!defined($opt) || !defined($type));

        if ((index($opt, $option) != -1) && ($type eq 'boolean')) {
            return 1;
        }
    }
    return 0;
}

sub register{
	return $RETURN_CODE_SUCCESS;
}

sub unregister{
	return $RETURN_CODE_SUCCESS;
}

sub isCompatibleWithScopeInstance {
	return 1;
}

sub isSidAdmUserExecution {
	my ($self) = @_;
	return ($self->{_isSidAdmUserExecution}) ? 1 : 0;
}

sub setSidAdmUserExecution {
	my ($self, $value) = @_;
	$self->{_isSidAdmUserExecution} = $value;
}

sub shallExecuteUpdateHost {
	return 0;
}

sub _getUnsupportedDbFeatures {
    my ( $self ) = @_;
    if(defined $self->{_unsupportedDbFeaturesMap}){
    	return $self->{_unsupportedDbFeaturesMap};
    }
    $self->{_unsupportedDbFeaturesMap} = ();
    my $manifest = $self->getManifest();
    my $unsuportedFeatureValue = $manifest->getValue("unsupported-db-features");
    if(defined $unsuportedFeatureValue){
	    my @unsuportedFeatureList = split( ',',$unsuportedFeatureValue );
	    %{$self->{_unsupportedDbFeaturesMap}} = map { trim($_) => 1 } @unsuportedFeatureList;
    }
    return $self->{_unsupportedDbFeaturesMap};
}

sub isFeatureUnsupported{
	my($self,$feature) = @_;
	my $unsupportedFeatureMap = $self->_getUnsupportedDbFeatures();
	my $isUnsupported = exists($unsupportedFeatureMap->{trim($feature)});
	return $isUnsupported;
}

sub getHostRoles {
	return [];
}

sub createTimestamp {
    return getFilenameTimeStamp();
}

sub isInPendingState {
	return 0;
}

sub getProgressMsg {
	...
}

sub getDeprecatedInfo {
    return undef;
}

sub setPhase {
    $_[0]->{phase} = $_[1];
}

sub getPhase {
    return $_[0]->{phase};
}

sub getDefaultPhase {
	return ONLINE_PHASE;
}

sub supportsPhases {
	return scalar @{$_[0]->getSupportedPhases()} > 0;
}

sub getSupportedPhases {
	my ($self) = @_;
	return $self->getManifest()->getSupportedPhases();
}

sub isSystemRestartRequired {
	my ($self) = @_;
	return $self->getManifest()->isSystemRestartRequired();
}

sub isInstanceRestartRequired {
	my ($self) = @_;
	return $self->getManifest()->isInstanceRestartRequired();
}

sub getServicesToRestart {
    my ($self, $configuration) = @_;
    # don't disable any services in case of no start as those will not be started after the manual start
    return [] if($configuration->isUpdate() && $configuration->isServerNoStart());
    return $self->getManifest()->getServicesToRestart();
}

sub isInstanceStopRequired {
    my ($self, $configuration) = @_;
    return 0 if $self->isComponentFinished($configuration) && !$self->isUpdateToNewerVersion();
    return 0 if !$self->isInstanceRestartRequired();
    return $self->_isStopRequired($configuration);
}

sub isSystemStopRequired {
    my ($self, $configuration) = @_;
    return 0 if $self->isComponentFinished($configuration) && !$self->isUpdateToNewerVersion();
    return 0 if !$self->isSystemRestartRequired();
    return $self->_isStopRequired($configuration);
}

sub _isStopRequired {
	my ($self, $configuration) = @_;

	if ($self->_canUsePersistency($configuration)) {
		return !$self->isNextPersistedPhase(ONLINE_PHASE, $configuration);
	}

	return 1;
}

sub _canUsePersistency {
	my ($self, $configuration) = @_;

	if ($self->can('hasPersistenceFile') && 
		$self->hasPersistenceFile($configuration) && 
		!$self->getIgnorePersistency() && 
		!$configuration->getIgnore('check_pending_upgrade')) {
		return 1;
	}

	return 0;
}

sub canSetDependencyPhase {
    my( $self ) = @_;
    my $supportedPhasesCount = scalar @{$self->getSupportedPhases()};
    return 1 if $supportedPhasesCount == 0;
    if($supportedPhasesCount == 1 and $self->supportsPhase(ONLINE_PHASE)){
        return 0;
    }
    return 1;
}

sub supportsPhase {
	my($self, $phase) = @_;
	return 0 if (!defined $phase);
	return ( grep {$_ eq $phase} @{$self->getSupportedPhases()} ) ? 1 : 0 ;
}

sub getLastSupportedPhase {
    my ($self) = @_;
    my $sortedPhases = [PREPARE_PHASE, OFFLINE_PHASE, CONFIGURE_PHASE, ONLINE_PHASE, SYSTEM_ONLINE_PHASE];
    my @supportedSortedPhases = grep { $self->supportsPhase($_) } @{$sortedPhases};
    return @supportedSortedPhases ? $supportedSortedPhases[-1] : undef;
}

sub isPostAddHostsComponent {
    return 0;
}

sub canBeCalledInPhase {
	my($self, $phase, $configuration) = @_;
	my $manifest = $self->getManifest();

	if ($self->_canUsePersistency($configuration)) {
		my $persistency = $self->getPersistenceXMLObject($configuration);
		return 0 if (!defined $persistency);
		my $nextPhase = $persistency->getNextPhaseString();
		return 0 if (!defined $nextPhase);
		return 0 if (!$self->isCurrentEqualOrAfterPersistedPhase($phase, $nextPhase));
	}

	return 0 if $self->isComponentFinished($configuration) && !$self->isUpdateToNewerVersion();
	return 1 if $self->supportsPhase(ALL);
	return 1 if $self->supportsPhase($phase);
	my $componentPhase = defined $self->getPhase() ? $self->getPhase() : $self->getDefaultPhase();
	return $phase eq $componentPhase;
}

sub isCurrentEqualOrAfterPersistedPhase {
	my ($self, $currentPhase, $persistedNextPhase) = @_;
    my $previousPhasesMap = {
        PREPARE_PHASE()       => [ PREPARE_PHASE ],
        OFFLINE_PHASE()       => [ PREPARE_PHASE, OFFLINE_PHASE ],
        CONFIGURE_PHASE()     => [ PREPARE_PHASE, OFFLINE_PHASE, CONFIGURE_PHASE ],
        ONLINE_PHASE()        => [ PREPARE_PHASE, OFFLINE_PHASE, CONFIGURE_PHASE, ONLINE_PHASE],
        SYSTEM_ONLINE_PHASE() => [ PREPARE_PHASE, OFFLINE_PHASE, CONFIGURE_PHASE, ONLINE_PHASE, SYSTEM_ONLINE_PHASE ],
        ALL()                 => [ PREPARE_PHASE, OFFLINE_PHASE, CONFIGURE_PHASE, ONLINE_PHASE, SYSTEM_ONLINE_PHASE ],
    };

    return 1 if ($persistedNextPhase eq '<unknown>');
    return grep { $persistedNextPhase eq $_ } @{$previousPhasesMap->{$currentPhase} // []};
}

sub isNextPersistedPhase {
    my ($self, $phase, $configuration) = @_;
    my $persistency = $self->getPersistenceXMLObject($configuration);
    return 0 if !defined $persistency;
    return $persistency->getNextPhaseString() eq $phase;
}

sub isComponentFinished{
	my ($self, $configuration) = @_;
    my $componentBatchKey = $self->getComponentBatchKey();
    return 0 if !$componentBatchKey;

    my $persistedSteps = $configuration->getPersistedSteps();
    return 0 if !$persistedSteps;

    my @components = split(",", $persistedSteps);
    foreach my $component (@components){
        return 1 if($componentBatchKey eq $component);
    }
    return 0;
}

sub isInstalled{
	...
}

sub _getSupportedDbFeatures {
    my ( $self ) = @_;
    if(defined $self->{_supportedDbFeaturesMap}){
    	return $self->{_supportedDbFeaturesMap};
    }
    $self->{_supportedDbFeaturesMap} = ();
    my $manifest = $self->getManifest();
    my $supportedFeatureValue = $manifest->getValue("supported-db-features");
    if(defined $supportedFeatureValue){
	    my @supportedFeatureList = split( ',',$supportedFeatureValue );
	    %{$self->{_supportedDbFeaturesMap}} = map { trim($_) => 1 } @supportedFeatureList;
    }
    return $self->{_supportedDbFeaturesMap};
}

sub isFeatureSupported{
	my($self,$feature) = @_;
	my $unsupportedFeatureMap = $self->_getUnsupportedDbFeatures();
	my $isUnSupported = exists($unsupportedFeatureMap->{trim($feature)});
	return 0 if($isUnSupported);
	my $supportedFeatureMap = $self->_getSupportedDbFeatures();
	my $isSupported = exists($supportedFeatureMap->{trim($feature)});
	return $isSupported;
}

sub _calculateSignaturePath {
    my ($self) = @_;
    my $manifestDir = $self->getPath();
    if (!defined $manifestDir) {
        return undef;
    }
    for my $offset (0..2) { # Either same level or one/two dirs back
        my $candidate = File::Spec->catfile(repeat(\&dirname, $offset, $manifestDir), $gSignatureManifestName);
        my $statObject = File::stat::stat($candidate);
        return $candidate if ($statObject && -f $statObject && -r $statObject);
    }

    return undef;
}

sub getSignature {
    my ($self, $noCache) = @_;
    if (!exists($self->{signature}) || $noCache) {
        my $signaturePath = $self->_calculateSignaturePath();
        $self->{signature} = $signaturePath ? LCM::Manifests::SAPSignature->new($signaturePath) : undef;
    }
    return $self->{signature};
}

sub isCurrentPlatformSupported {
    my ($self, $msgList) = @_;
    my $manifest = $self->getManifest();
    if (!$manifest->isPlatformSupported($currentPlatformName)){
        my $componentPlatforms = join(", ",@{$manifest->getSupportedPlatforms()});
        $msgList->addMessage ("Component ".$self->getComponentName()." in ".$self->getPath()." is for another platform $componentPlatforms." ) if(defined $msgList);
        return 0;
    }
    return 1;
}

sub isSupportedByLSS {
    my ($self) = @_;
    my $supportedFeatures = $self->getManifest()->getSupportedDbFeatures();
    return grep { $_ eq 'localsecurestore'} @$supportedFeatures;
}

sub getComponentKeyVendor{
    my ($self)  = @_;

    return $self->getManifest ()->getValue ('keyvendor')
}

sub isDependencyForComponent{
    my($self,$component) = @_;
    my $dependentManifest = $component->getManifest();
    my $parser = $dependentManifest->getDependencyParser();
    return 0 if(defined $parser && $parser == 1); #no 'required-components' found

    if(!defined $parser){
        $self->getMsgLst()->addWarning("Parsing dependencies for ".$component->getComponentName()." failed", $dependentManifest->getErrMsgLst());
        return 0;
    }

    my $keyVendor = $self->getComponentKeyVendor();
    my $dependencyVendors = $parser->getDependenciesVendors();
    return 0 if (! grep {/$keyVendor/} @$dependencyVendors);

    my $dependencyKeyNames = $parser->getDependenciesKeyNamesForVendor($keyVendor);
    my $keyname = $self->getComponentKeyName();
    return 0 if !(grep {/$keyname/} @$dependencyKeyNames);
    return 1 if($dependentManifest->checkComponentDependency($self->getManifest()));

    return 0;
}

sub isDeprecated {
    return 0;
}

sub getCustomPasswordInput {
    return undef;
}

sub getManifestDirname {
    my $self = shift;
    my $componentPath = $self->getPath();
    return (grep {$_} File::Spec->splitdir($componentPath))[-1];
}

sub getChecksumDir {
    ...
}

sub modifyTrustLevelInLss {
    my ($self, $config) = @_;
    my $trustLevelModifier = SDB::Install::LSS::TrustLevelModifier::TrustLevelModifierFactory->getInstance($config, $self);
    $trustLevelModifier->setMsgLstContext($self->getMsgLstContext());
    return $trustLevelModifier->execute();
}

1;

