package LCM::Component::Installable;

use strict;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Version;
use SDB::Install::RemoteHosts;
use SDB::Install::Globals qw ($gProductNameEngine $gProductNameClient determineSignedManifestRelativePath $gShortProductNameLSS);
use SAPDB::Install::Hostname;
use File::Basename qw (dirname);
use File::stat;
use LCM::Component;
use LCM::FileUtils;
use LCM::Persistence::XML;
use LCM::Component::Installable::InstallationKitChecker;
use File::Spec;
use SDB::Install::LSS::Registrator::ComponentRegistratorFactory;

use experimental qw (smartmatch);

our @ISA = qw (LCM::Component);

my $PERS_KEY_COMPONENT_PHASE = 'cmpNextPhase';
my $PERS_KEY_COMPONENT_STEP = 'cmpCurrentStep';

our $hdbInstallExecutable = 'hdbinst';
our $hdbUpdateExecutable = 'hdbinst';

if ($isWin){
    $hdbInstallExecutable .= '.exe';
    $hdbUpdateExecutable .= '.exe';
}

sub new {
    my $self = shift->SUPER::new (@_);
    my $manifest = $self->{manifest};
    if (!defined $manifest){
        return $self;
    }
    my $classOfComponent;
    my $isSidAdmExecution = $self->isSidAdmUserExecution();
    if ($manifest->isServer()){
        $classOfComponent = ($isSidAdmExecution) ?
        					'LCM::Component::Installable::HDBServer::HDBServerSidAdm' : 
        					'LCM::Component::Installable::HDBServer::HDBServerRoot';
    } elsif ($manifest->isClient()){
        $classOfComponent = ($isSidAdmExecution) ?
        					'LCM::Component::Installable::HDBClient::HDBClientSidAdm' : 
        					'LCM::Component::Installable::HDBClient::HDBClientRoot';
    } elsif ($manifest->isStudio()){
        $classOfComponent = 'LCM::Component::Installable::HDBStudio';
    } elsif ($manifest->isLCAPPS()){
		$classOfComponent = 'LCM::Component::Installable::HDBServerPlugin';
    } elsif ($manifest->isAFL()){
        $classOfComponent = 'LCM::Component::Installable::HDBServerPlugin';
    } elsif ($manifest->isHLM()){
        $classOfComponent = 'LCM::Component::Installable::HLM';
    } elsif ($manifest->isSDA()){
        $classOfComponent = 'LCM::Component::Installable::SDA';
    } elsif ($manifest->isInstaller()){
        $classOfComponent = ($isSidAdmExecution) ?
        					'LCM::Component::Installable::ResidentInstaller::ResidentInstallerSidAdm' : 
        					'LCM::Component::Installable::ResidentInstaller::ResidentInstallerRoot';
    } elsif ($manifest->isCockpitStack()){
        $classOfComponent = 'LCM::Component::Installable::CockpitStack';
    } elsif ($manifest->isStreaming()){
        $classOfComponent = 'LCM::Component::Installable::Streaming';
    } elsif ($manifest->isES()){
        $classOfComponent = 'LCM::Component::Installable::ES';
    } elsif ($manifest->isAccelerator()){
        $classOfComponent = 'LCM::Component::Installable::Accelerator';
    } elsif ($manifest->isXS2()){
        $classOfComponent = 'LCM::Component::Installable::XS2';
    } elsif ($manifest->isXS2Application()){
        $classOfComponent = 'LCM::Component::Installable::XS2Application';
    } elsif ($manifest->isServerPlugin()){
        my $componentKey = $manifest->getComponentKey();
        if(!$componentKey){
	    	$self->setErrorMessage("Component manifest of '" . $self->getComponentName() . "' is missing property 'component-key'.");
	        return $self;
    	}
        $classOfComponent = 'LCM::Component::Installable::HDBServerPlugin';
    } elsif ( $manifest->isReferenceData() ){
    	$classOfComponent = 'LCM::Component::Installable::ReferenceData';
    } elsif ( $manifest->isLSS() ){
        $classOfComponent = ($isSidAdmExecution) ?
                            'LCM::Component::Installable::LSS::LSSSidAdm' :
                            'LCM::Component::Installable::LSS::LSSRoot';
    }
    if (!defined $classOfComponent){
        $self->addWarning('Unknown component ' . $self->getComponentName());
        return $self;
    }

    eval("require $classOfComponent;");
    if ($@){
        $self->setErrorMessage ("Cannot load component class '$classOfComponent': $@");
        return $self;
    }
    bless ($self, $classOfComponent);
    return $self;
}

sub isResumable {
	my ( $self, $instconfig ) = @_;
	my $isComponentResumable = $self->getComponentName() ~~ @{$self->getResumableComponentNames()} ? 1 : 0;
	return $isComponentResumable;
}

sub getResumableComponentNames {
	return [$gProductNameEngine, ];	
}

sub deletePersistenceFile {
    my ($self, $configuration, $checkCustomFile) = @_;
    my $name = $self->getComponentName();
    if (!defined $self->hasPersistenceFile($configuration, $checkCustomFile)) {
        $self->getErrMsgLst()->addError("Failed to detect if component $name has a persistence file");
        return undef;
    }
    return 1 if (!$self->hasPersistenceFile($configuration, $checkCustomFile));

    my $sapSystem = $configuration->getSAPSystem();
    my $persFilePath = $self->_getPersistenceFilePath($sapSystem);
    $self->getMsgLst()->addMessage("Deleting persistence file '$persFilePath' of component $name");
    return deleteFile($persFilePath, $self->getErrMsgLst());
}

sub hasPersistenceFile {
	my ($self, $configuration) = @_;
	my $sapSystem = $configuration->getSAPSystem();

	return undef if(!defined($sapSystem));

	return File::stat::stat($self->_getPersistenceFilePath($sapSystem)) ? 1 : 0;
}

sub getPersistenceXMLObject {
	my ($self, $configuration) = @_;
	if (defined $self->{_persistenceXML}) {
		return $self->{_persistenceXML};
	}

    return undef if !$self->hasPersistenceFile($configuration);
	my $sapSystem = $configuration->getSAPSystem();
	my $filePath = $self->_getPersistenceFilePath($sapSystem);
	my $persistenceXMLObject = LCM::Persistence::XML->new($filePath);

	if ($persistenceXMLObject->errorState()) {
		$configuration->getMsgLst()->addWarning($persistenceXMLObject->getErrorString());
	}

	$self->{_persistenceXML} = $persistenceXMLObject;

	return $persistenceXMLObject;
}

sub getAction {
	return $_[0]->isUpdate() ? 'Update' : 'Install';
}

sub _setMsg {
	$_[0]->{msg} = $_[1];
}

sub _getMsg {
	return $_[0]->{msg}; 
}

sub _setSaveContext {
	$_[0]->{saveContext} = $_[1];
}

sub _getSaveContext {
	return $_[0]->{saveContext}; 
}

sub _buildArgs {
	my ($self, $instconfig) = @_;
	my $isUsingPhases = $instconfig->usePhases();
	my $currentPhase = $isUsingPhases ? $self->getPhase() : '';
	my $isServerComponent = $self->isServer();

	if($isUsingPhases && $self->supportsPhase($currentPhase)){
		return $isServerComponent ? [ "--phase=$currentPhase" ] : [ '-phase', $currentPhase ];
	}
	return [];
}

sub installComponent {
    $_[0]->setErrorMessage ("installComponent not implemented");
    return undef;
}

sub updateComponent {
    $_[0]->setErrorMessage ("updateComponent not implemented");
    return undef;
}

sub getExecutable {
    my ($self) = @_;
    return $self->getHdbUpdateExecutable() if($self->isUpdate());
    return $self->getHdbInstallExecutable();
}

sub getHdbInstallExecutable{
    my ($self) = @_;
    return File::Spec->catfile($self->getInstallerDir(), $hdbInstallExecutable);
}

sub getHdbUpdateExecutable{
    my ($self) = @_;
    return File::Spec->catfile($self->getInstallerDir(), $hdbUpdateExecutable);
}

sub getInstallAnywareExecutable{
    my ($self) = @_;
    return File::Spec->catfile($self->getPath(), 'setup.bin');
}

sub getInstallActionString{
    return $_[0]->isUpdate () ? 'Updating' : 'Installing';
}

sub isUpdate{
    my ($self) = @_;
    return (defined $self->getInstalledComponent()) ? 1 : 0;
}

sub isUpdateToNewerVersion {
    my($self) = @_;
    my $installedComponent = $self->getInstalledComponent();

    return 0 if(!$self->isUpdate());
    return 0 if(!defined($installedComponent));

    my $newVersion = $self->getManifest()->getVersionObject();
    my $installedVersion = $installedComponent->getManifest()->getVersionObject();
    return $newVersion->isNewerThan($installedVersion);
}

sub isDowngrade {
    my($self) = @_;
    my $installedComponent = $self->getInstalledComponent();

    return 0 if(!$self->isUpdate());
    return 0 if(!defined($installedComponent));

    my $newVersion = $self->getManifest()->getVersionObject();
    my $installedVersion = $installedComponent->getManifest()->getVersionObject();
    return (!$newVersion->isEqual($installedVersion) && !$newVersion->isNewerThan($installedVersion));
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    my $installedComponent = $self->getInstalledComponent();

    return 1 if (!$stackUpdate);
    return 0 if (!defined($installedComponent));

    my $newVersion = $self->getManifest()->getVersionObject();
    my $installedVersion = $installedComponent->getManifest()->getVersionObject();
    return ($installedVersion->isNewerThan($newVersion)) ? 0 : 1;
}

sub getInstalledComponent{
    my ($self) = @_;
    my $systemComponentManager = $self->getSystemComponentManager();
    if (defined $systemComponentManager) {
        return $systemComponentManager->getComponentByKeyName($self->getComponentKeyName());
    }
    return undef;
}

sub requireSystemRestart {
	return 0;
}

sub getNotCompatibleWithScopeInstanceMessage {
	return "";
}

sub getSlppLogFileName {
	return undef;
}

sub installThroughSHAOperation {
    my ($self, $instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs) = @_;
    my $action = $self->isUpdate() ? 'Update' : 'Installation';
    my $customPasswordInput = $self->getCustomPasswordInput($instconfig);

    my $checker = LCM::Component::Installable::InstallationKitChecker->new();
    if($checker->isKitOwnedByRoot($self->getKitCheckerRootDir(), $self->getFilelistFileName())){
        return $self->_executeThroughSHAOperation($instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs, $action, $customPasswordInput);
    }
    my $remoteExecution = new SDB::Install::RemoteHostctrlHosts(hostname());
    my $cmpName = $self->getComponentName();
    my $timestamp = $self->createTimestamp();

    $remoteExecution->useSidadm() if (defined($instconfig->getValue('Password')));

    $self->getMsgLst ()->addMessage ("Installation kit of $cmpName is not owned by root");
    my $msg = $self->getMsgLst ()->addMessage ("Copying installer of $cmpName as root...");
    $remoteExecution->setMsgLstContext ([$msg->getSubMsgLst()]);
    if($self->_executeAdditionalOperation($instconfig, $remoteExecution, $self->getCopyInstallerOperation(), $timestamp)){
        $instconfig->getMsgLst()->addError ("Copying of ${cmpName}'s installer through SAP Host Agent Operation failed!", $remoteExecution);
        return undef;
    }
    
    my $copiedInstallerDirectory = $self->getCopiedInstallerDirPrefix() . $timestamp;
    my $newInstallationKitPath = File::Spec->catdir($instconfig->getValue('Target'), uc($instconfig->getSID()), $copiedInstallerDirectory);
    $self->updateSHAOptionsMap($optionMap, $newInstallationKitPath);
    my $returnCode = $self->_executeThroughSHAOperation($instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs, $action, $customPasswordInput);
    
    $msg = $self->getMsgLst ()->addMessage ("Removing $cmpName installer copy...");
    $remoteExecution->setMsgLstContext ([$msg->getSubMsgLst()]);
    if($self->_executeAdditionalOperation($instconfig, $remoteExecution, $self->getDeleteCopiedInstallerOperation(), $timestamp)){
        $instconfig->getMsgLst()->addWarning("Removing $cmpName installer copy through SAP Host Agent Operation failed!", $remoteExecution);
    }
    
    return $returnCode;
}

sub _executeAdditionalOperation {
    my ($self, $instconfig, $remoteExecution, $operation, $timestamp) = @_;
    my $sourceDirectory = $self->getInstallerDir();
    my $signature_dir = determineSignedManifestRelativePath($sourceDirectory);
    my $optionsMap = {
        SAPMNT => $instconfig->getValue('Target'),
        SID => uc($instconfig->getSID()),
        SOURCE_DIR => $sourceDirectory,
        SUFFIX => $timestamp,
        SIGNATURE_DIR => $signature_dir
    };
    my $passwordKeys = ['Password', 'HostagentPassword'];
    return $remoteExecution->executeHostctrlParallel($operation, $instconfig, undef, $passwordKeys, undef, undef, undef, undef, undef, $optionsMap, undef, undef, undef, 1);
}

sub getProgressMsg {
	my ($self) = @_;
	my $componentName = $self->getComponentName();
	my $phase = $self->getPhase();

	if(!defined $phase || !$self->supportsPhase($phase)){
		my $actionPrefix = $self->isUpdate ? 'Updat' : 'Install';    
		return $actionPrefix."ing " . $componentName;
	}

	if($self->isUpdate()){
		if($phase eq PREPARE_PHASE){
			return "Preparing update of $componentName";
		}
		if($phase eq OFFLINE_PHASE){
			return "Updating $componentName";
        }
        if($phase eq CONFIGURE_PHASE){
            return "Configuring $componentName";
        }
        if($phase eq ONLINE_PHASE){
            return "Finalizing update of $componentName";
        }
        if($phase eq SYSTEM_ONLINE_PHASE){
            return "Updating content of $componentName";
        }
	} else {
		if($phase eq PREPARE_PHASE){
			return "Preparing installation of $componentName";
		}
		if($phase eq OFFLINE_PHASE){
			return "Installing $componentName";
		}
        if($phase eq CONFIGURE_PHASE){
            return "Configuring $componentName";
        }
		if($phase eq ONLINE_PHASE){
			return "Finalizing installation of $componentName";
		}
        if($phase eq SYSTEM_ONLINE_PHASE){
            return "Installing content of $componentName";
        }
	}
}

sub _getPersistenceFilePath {
	my ($self, $sapSystem) = @_;
	my $fileName = $self->getPersistenceXmlName();
	my $globalSidDirectory = $sapSystem->get_globalSidDir();
    return File::Spec->catfile($globalSidDirectory, $fileName);
}

sub getPersistenceXmlName() {
    my ($self) = @_;
    my $componentKey = $self->getComponentBatchKey();
    return sprintf('pending_%s.xml', $componentKey);
}

sub isInstalled{
	my $self = shift;
	return 0;
}

sub requiresSqlSysPasswd{
   return 0;
}

sub isSigned {
    return 0;
}

sub getInstallerDir {
    my ($self) = @_;
    my $manifestDir = $self->getPath();
    return dirname($manifestDir);
}

sub getKitCheckerRootDir {
    my ($self) = @_;
    return $self->isCockpitStack() ? $self->getPath() : $self->getInstallerDir();
}

sub verifyFilesPermissions{
    my ($self, $errorMsgLst) = @_;
    return 1 if (!$self->shouldVerifyFiles());
    my $checker = LCM::Component::Installable::InstallationKitChecker->new();
    return $checker->isKitNotWritableForOthers($self->getKitCheckerRootDir(), $errorMsgLst);
}

sub checkFilesForPipes {
    my ($self, $errorMsgLst) = @_;
    return 1 if (!$self->shouldVerifyFiles());
    my $checker = LCM::Component::Installable::InstallationKitChecker->new();
    return $checker->scanForPipes($self->getKitCheckerRootDir(), $errorMsgLst);
}

sub shouldVerifyFiles {
    my $self = shift;
    return !$self->isInternal();
}

sub requiresRootPrivilege {
    return 0;
}

sub getServicesToRestart{
    my ($self, $configuration) = @_;
    if ($configuration->isUpdate()) {
        my $instance = $configuration->getOwnInstance();
        my $isSecondarySystem = $instance ? $instance->isSecondarySystem($self->getMsgLst()) : 0;
        return [] if $isSecondarySystem;
    }
    return $self->SUPER::getServicesToRestart($configuration);
}

sub getSystemComponentManager{
    my $self = shift;
    return $self->getComponentManager()->getSystemComponentManager();
}

sub getApplicableIgnoreString {
    my ($self, $ignoreOption) = @_;
    my $componentConfig = $self->getExternalHdbInstallerConfiguration();
    return undef if (!defined $componentConfig || !defined $ignoreOption);

    my $cmpIgnoreArgs = $ignoreOption->getSetKeysValidFor ($componentConfig->getOptionIgnore ());
    return $ignoreOption->getOptionStringFromArgList ($cmpIgnoreArgs);
}

sub getLstFileLocation {
    my ($self) = @_;
    return $self->getPath();
}

sub getLSSRegistrationPhase {
    my ($self) = @_;
    return OFFLINE_PHASE;
}

sub registerInLss {
    my ($self, $configuration) = @_;
    my $hashRegistrator = SDB::Install::LSS::Registrator::ComponentRegistratorFactory->getInstance($configuration, $self);
    $hashRegistrator->setMsgLstContext($self->getMsgLstContext());
    $self->persistCustomStep($configuration, sprintf("Register %s binaries in $gShortProductNameLSS", $self->getComponentName()));
    my $rc = $hashRegistrator->execute();
    $self->handleCustomPersistenceFile($configuration) if $rc;
    return $rc;
}

sub createPendingFile {
    my ($self, $configuration, $xmlString) = @_;
    my $errorList = SDB::Install::MsgLst->new();
    my $sid = $configuration->getSID();
    my $sapSystem = $configuration->getSAPSystem();
    if (!defined $sapSystem) {
        $self->getErrMsgLst()->addError(sprintf("Failed to detect SAP System while creating %s persistence file.", $self->getComponentName()));
        return undef;
    }
    my $user = $sapSystem->getUser();
    my $uid  = $user->uid();
    my $gid  = $user->gid();
    my $persistenceFilePath = $self->_getPersistenceFilePath($sapSystem);

    if(!writeToFile($persistenceFilePath, $xmlString, 0644, $uid, $gid, $errorList)){
        $self->getErrMsgLst()->addError("Failed to write file '$persistenceFilePath' to the file system", $errorList);
        return undef;
    }

    return 1;
}

sub persistCustomStep {
    my ($self, $instconfig, $step) = @_;
    my $xmlGenerator = $self->createPersistenceXMLGenerator();
    $xmlGenerator->setCurrentStepString($step);
    if ($self->hasPersistenceFile($instconfig)) {
        my $persistenceXML = $self->getPersistenceXMLObject($instconfig);
        my $nextPhase = $persistenceXML->getNextPhaseString();
        my $currStep = $persistenceXML->getCurrentStepString();
# As this method would reuse the same name as the one, which the installer of the
# corresponding component would have left, here we take care to persist the step and the
# next phase of the component installer before overwriting the file so that we can later
# restore them.
        $xmlGenerator->addExtension($PERS_KEY_COMPONENT_PHASE, $nextPhase) if $nextPhase;
        $xmlGenerator->addExtension($PERS_KEY_COMPONENT_STEP, $currStep) if $currStep;
    }
    return $self->createPendingFile($instconfig, $xmlGenerator->generatePersistenceXMLString());
}

sub hasOverwrittenComponentPersFile {
    my ($self, $instconfig) = @_;
    my $persistenceXML = $self->getPersistenceXMLObject($instconfig);
    my @persistedComponentData = map { $persistenceXML->getExtension($_) } ($PERS_KEY_COMPONENT_PHASE, $PERS_KEY_COMPONENT_STEP);
    return grep { $_ } @persistedComponentData;
}

sub handleCustomPersistenceFile {
    my ($self, $instconfig) = @_;
    if (!$self->hasOverwrittenComponentPersFile($instconfig)) {
        return $self->deletePersistenceFile($instconfig, 1);
    }
    my $persistenceXML = $self->getPersistenceXMLObject($instconfig);
    my $componentPersStep = $persistenceXML->getExtension($PERS_KEY_COMPONENT_STEP);
    my $componentNextPhase = $persistenceXML->getExtension($PERS_KEY_COMPONENT_PHASE);
    my $generator = $self->createPersistenceXMLGenerator();
    $generator->setCurrentStepString($componentPersStep);
    $generator->setNextPhaseString($componentNextPhase);

    $self->getMsgLst()->addMessage(sprintf("Restoring persistence file of '%s'...", $self->getComponentName()));
    return $self->createPendingFile($instconfig, $generator->generatePersistenceXMLString());
}

sub isStepPending {
    my ($self, $instconfig, $step) = @_;
    my $persistenceXML = $self->getPersistenceXMLObject($instconfig);
    if ($persistenceXML) {
        return ($persistenceXML->getCurrentStepString() eq $step) ? 1 : 0;
    }
    return 0;
}

sub createPersistenceXMLGenerator {
    my ($self) = @_;
    my $componentName = $self->getComponentName();
    my $isUpdate = $self->isUpdate();
    my $actionString = $isUpdate ? 'update' : 'install';
    my $currentStepString = sprintf("%s %s", ucfirst($actionString), $componentName);
    my $xmlGenerator = SDB::Install::Persistence::XMLGenerator->new();
    $xmlGenerator->setTargetVersionString($self->getVersion());
    $xmlGenerator->setComponentNameString($componentName);
    $xmlGenerator->setCurrentStepString($currentStepString);
    $xmlGenerator->setActionString($actionString);
    $xmlGenerator->setNextPhaseString($self->getPhase() // $self->getDefaultPhase());
    if ($isUpdate) {
        $xmlGenerator->setSourceVersionString($self->getInstalledComponent()->getVersion());
    }
    return $xmlGenerator;
}

sub getChecksumDir {
    my ($self) = @_;
    return $self->getPath();
}

1;
