package LCM::Component::Installable::InstallationKitExecutePermissionsChecker;

use File::Spec;
use Fcntl ':mode';
use File::stat;
use LCM::FileUtils qw(hasFileExecutePermissions);
use strict;
use parent 'LCM::Component::Installable::InstallationKitChecker';

sub new {
    my($class, $config, $selectedComponents, $user, $uid, $gid) = @_;
    my $self = shift->SUPER::new();
    $self->{config} = $config;
    $self->{selectedComponents} = $selectedComponents;
    $self->{user} = $user;
    $self->{uid} = $uid;
    $self->{gid} = $gid;
    return $self;
}

sub check {
    my ($self) = @_;
    my $config = $self->{config};
    my $user = $self->{user};
    my $subErrList = SDB::Install::MsgLst->new();
    $config->getMsgLst()->addMessage("Check selected components execute permissions...");

    my $rc = 1;
    my $userExists = (defined $user && $user->exists());
    $user->switchto() if($userExists);
    foreach my $component (@{$self->{selectedComponents}}){
        next if($component->isInternal());
        my $executable = $component->getExecutable();
        next if(! defined $executable);

        if(defined $executable && !defined File::stat::stat($executable)){
            my $compName = $component->getComponentName();
            my $errlst = $config->getErrMsgLst();
            if($!{'ENOENT'}){
                $errlst->addError("$compName executable '$executable' does not exist.");
            }
            else{
                my $userName = $user->getSidAdmName();
                $errlst->addError("$compName executable '$executable' cannot be accessed as user $userName: $!");
                $errlst->addError("The $userName user requires read and execute permissions for the directory that contains the installation medium.");
            }
            $rc = 0;
            next;
        }
        $rc = 0 if(!$self->checkExecutePermissions($executable, $userExists, $subErrList));
    }
    $user->switchback() if($userExists);

    if(!$subErrList->isEmpty()){
        my $errMsgList = $config->getErrMsgLst();
        my $filteredErrList = $subErrList->filterDuplicatedMessages();
        $errMsgList->addError("The following software installation files are missing execute permissions:", $filteredErrList);
        $errMsgList->addError("Ensure that no permissions are changed by the umask when extracting or copying the folders.");
        $rc = 0;
    }
    return $rc;
}

sub checkExecutePermissions {
    my ($self, $filePath, $userExists, $errorList) = @_;
    my $rc;
    if($userExists){
        $rc = $self->checkExistingUserExecutePermissions($filePath, $errorList);
    } else{
        $rc = $self->checkNonExistingUserExecutePermissions($filePath, $errorList);
    }
    return $rc;
}

sub checkExistingUserExecutePermissions{
    my ($self, $filePath, $errorList) = @_;
    if(defined $filePath && ! -x $filePath){
        $errorList->addError($filePath) if($errorList);
        return 0;
    }
    return 1;
}

sub checkNonExistingUserExecutePermissions{
    my ($self, $filePath, $errorList) = @_;
    my $uid = $self->{uid};
    my $gid = $self->{gid};
    if(! hasFileExecutePermissions($filePath, $uid, $gid)){
        $errorList->addError($filePath) if($errorList);
        return 0;
    }
    my $parentDir = $self->_getParentDir($filePath);
    my $file = File::stat::stat($parentDir);
    if($file && S_ISDIR($file->mode())){
        return $self->checkNonExistingUserExecutePermissions($parentDir, $errorList) if($parentDir ne File::Spec->rootdir());
    }
    return 1;
}

1;