package LCM::Component::Installable::LSS::LSSBase;

use strict;
use base qw(LCM::Component::LCMsdkInstallable LCM::Component::InternallyPersistedInstallable);
use LCM::ProcessExecutor;
use LCM::Component qw (OFFLINE_PHASE);
use LCM::Utils::CommonUtils qw(buildTimeoutsString getHdbmodifyTimeoutValues);
use SDB::Install::Globals qw(
    $gProductNameLSS
    $gProductName
    $gLogDir
);
use File::Spec;
use File::Basename qw(dirname);

my $CONFIGURE_LSS_STEP = 'Configure LSS across hosts';
my $REG_EXISTING_CMP_STEP = 'Register already installed components';

sub installComponent {
    my ($self, $instconfig) = @_;
    my ($saveCntxt, $msg) = $self->_adaptMessagesContext(sprintf("%s...", $self->getProgressMsg()));
    $self->initProgressHandler();

    my $rc = $self->_shouldCallLSSInstaller($instconfig) ? $self->_installLSSComponent($instconfig) : 1;
    if ($rc && $self->_shouldCallConfigureLSS($instconfig)) {
        $self->persistCustomStep($instconfig, $CONFIGURE_LSS_STEP);
        $rc = $self->_configureLSSonSystem($instconfig);
        $self->handleCustomPersistenceFile($instconfig) if $rc;
    }

    if ($rc && $self->_shouldRegisterExistingComponents($instconfig)) {
        $self->persistCustomStep($instconfig, $REG_EXISTING_CMP_STEP);
        $rc = $self->_registerExistingComponents($instconfig);
        $self->handleCustomPersistenceFile($instconfig) if $rc;
    }

    $msg->endMessage(undef, $self->_getEndMessage($rc));
    $self->setMsgLstContext($saveCntxt);
    return $rc;
}

sub _registerExistingComponents {
    my ($self, $config) = @_;
    my $componentsToReg = $self->getExistingComponentsToRegister($config);
    for my $cmp (@$componentsToReg) {
        my $cmpName = $cmp->getComponentName();
        $cmp->setMsgLstContext($self->getMsgLstContext());
        return 0 if (!$cmp->registerInLss($config));
    }
    return 1;
}

sub getExistingComponentsToRegister {
    my ($self, $config) = @_;
    my $lssCmps = $config->getInstalledLssComponents();
    return [] if (!@$lssCmps);

    my $mcm = $config->getMediumComponentManager();
    [grep {!$mcm->isComponentSelected($_->getComponentKeyName)} @$lssCmps];
}

sub updateComponent {
    my ($self) = shift;
    return $self->installComponent(@_);
}

sub _installLSSComponent {
    # Implement in the LSSRoot.pm and LSSSidAdm.pm
    ...
}

sub _configureLSSonSystem {
    my ($self, $instconfig) = @_;
    my $componentName = $self->getComponentName();
    my $oldCtx = $self->_adaptMessagesContext("Configuring $componentName across the $gProductName landscape...");
    $self->getProgressHandler()->incrementIndentationDepth();

    my $rc = $self->_callHdbmodify($instconfig);

    $self->getProgressHandler()->decrementIndentationDepth();
    $self->setMsgLstContext($oldCtx);
    return $rc;
}

sub _callHdbmodify {
    my ($self, $instconfig) = @_;
    my $sapSystem = $instconfig->getSAPSystem();
    my $hdbmodifyPath = File::Spec->catfile($sapSystem->get_globalTrexInstallDir(), 'bin', 'hdbmodify');
    my $passwordStream = $instconfig->getXmlPasswordStream(['Password', 'LSSPassword', 'LSSBackupPassword', 'RootPassword']);
    my $hdbmodifyArgs = ['--configure_lss', '-b', '--read_password_from_stdin=xml', "--instlog_dir=$gLogDir", '--nostart'];
    my $timeoutString = buildTimeoutsString($instconfig, getHdbmodifyTimeoutValues());
    push(@{$hdbmodifyArgs}, $timeoutString) if $timeoutString;
    push(@{$hdbmodifyArgs}, sprintf("--scope=%s", $instconfig->getValue('Scope'))) if($instconfig->hasValue('Scope'));
    push(@{$hdbmodifyArgs}, sprintf("--remote_execution=%s", $instconfig->getValue('RemoteExecution')));
    my $rootUser = $instconfig->getValue('RootUser');
    push(@{$hdbmodifyArgs}, sprintf("--root_user=%s", $rootUser)) if(defined $rootUser);

    my $executor = LCM::ProcessExecutor->new($hdbmodifyPath, $hdbmodifyArgs, $passwordStream);
    $executor->setOutputHandler($self->getProgressHandler());
    $executor->setMsgLstContext($self->getMsgLstContext());
    my $rc = $executor->executeProgram();
    return (defined($rc) && $rc == 0) ? 1 : 0;
}

sub _adaptMessagesContext {
    my ($self, $messageString) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($messageString);
    my $oldCtx = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    if (wantarray) {
        return ($oldCtx, $msg);
    }
    return $oldCtx;
}

sub _getEndMessage {
    my ($self, $returnCode) = @_;

    my $message  = ($self->isUpdate() ?'Update' : 'Installation') .' of ' . $self->getComponentName();
    $message .= ($returnCode == 1) ? ' finished' : ' failed';
    return $message;
}

sub _shouldCallConfigureLSS {
    my ($self, $config) = @_;
    my $phase = $self->getPhase();
    return 0 if $self->isUpdate();
    return 1 if !$phase;
    return $phase eq OFFLINE_PHASE && !$self->isStepPending($config, $REG_EXISTING_CMP_STEP);
}

sub _shouldCallLSSInstaller {
    my ($self, $config) = @_;
    my $ignoreUpgrade = $config->getIgnore('check_pending_upgrade');
    my $hasPendingCustomSteps = $self->isStepPending($config, $CONFIGURE_LSS_STEP)
                             || $self->isStepPending($config, $REG_EXISTING_CMP_STEP);
    return $ignoreUpgrade || !$hasPendingCustomSteps;
}

sub _shouldRegisterExistingComponents {
    my ($self, $config) = @_;
    my $phase = $self->getPhase();
    return 0 if $self->isUpdate();
    return 1 if !$phase;

    my $existingComponentsToRegister = $self->getExistingComponentsToRegister($config);
    return $phase eq OFFLINE_PHASE && @$existingComponentsToRegister
}

sub getComponentName {
    return $gProductNameLSS;
}

sub _buildArgs {
    my ($self, $instconfig) = @_;
    my @arguments = ();
    my $ignoreOption = $instconfig->getOptionIgnore();
    my $lssIgnoreString = $self->getApplicableIgnoreString($ignoreOption);
    return undef if !defined($lssIgnoreString);

    push( @arguments, '--shared_path='.$self->getLSSSharedDir($instconfig));
    push( @arguments, '--sid='.$instconfig->getSID());
    if (!$self->isUpdate()){
        push( @arguments, '--userid='.$instconfig->getValue('LSSUserID'));
        push( @arguments, '--groupid='.$instconfig->getValue('LSSGroupID'));
        push( @arguments, '--home='.$instconfig->getValue('LSSUserHomeDir'));
        push( @arguments, '--shell='.$instconfig->getValue('LSSUserShell'));
        push( @arguments, '--sapsys_gid='.$self->getGID($instconfig));
        push( @arguments, '--system_usage='.$instconfig->getSystemUsage());
    }
    push( @arguments, $lssIgnoreString) if($lssIgnoreString);
    push( @arguments, '--instlog_dir='.$gLogDir);

    return \@arguments;
}

sub getLSSSharedDir{
    my ($self, $instconfig) = @_;
    my $shared_dir;
    if ($self->isUpdate()){
        my $installedDir = $self->getInstalledComponent()->getInstallationPath();
        $shared_dir = dirname($installedDir);
    } else {
        $shared_dir = $instconfig->getValue('LSSInstPath');
    }
    return $shared_dir;
}

sub getGID {
    my ($self, $instconfig) = @_;
    return $instconfig->getValue('GID') if ! $instconfig->isUpdate();
    my $sapSystem = $instconfig->getSAPSystem();
    return $sapSystem->getGID();
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    return ($stackUpdate) ? $self->SUPER::getDefaultSelection ($stackUpdate) : 0;
}

sub getNumberOfExpectedOutputLines {
    return 5;
}

sub getSlppLogFileName {
    return 'lss.log'
}

sub getDefaultPhase {
    return OFFLINE_PHASE;
}

sub getCustomPasswordInput {
    my ($self, $config) = @_;
    return $config->getCustomPasswordInput(['LSSBackupPassword', 'LSSPassword'], ['backup_password', 'password']);
}

sub requireSystemRestart {
    return 1;
}

sub isSigned {
    return 1;
}

1;