package LCM::Component::Installable::XS2;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw ($gDirNameXS2
                              $gLogDir
                              $gProductNameXS2
                              $gShortProductNameXS2
                              $gHostRoleXS2Worker
                              $gHostRoleXS2Standby
                              $gTmpDir);
use LCM::ProcessExecutor;
use LCM::Installer;
use LCM::FileUtils;
use LCM::Component qw ( OFFLINE_PHASE );
use SDB::Common::Utils qw(createXSSpaceSAPUserName createXSSpaceProdUserName);
use LCM::Utils::CommonUtils qw(getXSInstallerTimeoutValues buildTimeoutsString);
use File::Basename qw (dirname);
use File::stat;
use SDB::Install::SAPSystemXS2;
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::System qw (enableWritePermissions);
use SAPDB::Install::System::Unix qw (lchown);
use SDB::Install::SAPSystemUtilities;
use SDB::Install::Configuration qw($bool_true_pattern);
use SDB::Install::Configuration::Generic;
use strict;

use base 'LCM::Component::LCMsdkInstallable';

#
# Create symlinks to the xs2 scripts in the instance exe directory.
# Prevents one restart of HANA services after initial XS installation
#

sub _createExeSymlinks{
    my ($self, $instconfig) = @_;
    my $msg = $self->getMsgLst ()->addMessage ("Creating xs2 symlinks");
    my $hdbInstance = $instconfig->getOwnInstance();
    if (!defined $hdbInstance){
        return undef;
    }
    my $exeDir = $hdbInstance->get_instanceExeDir ();
    my $stat = File::stat::stat($exeDir);
    if (!defined $stat){
        $msg->getSubMsgLst()->addMessage ("Cannot access path '$exeDir': $!");
        return 1;
    }
    my $restoreMode = enableWritePermissions($exeDir, $stat, $msg->getSubMsgLst());
    my $exeLink;
    foreach my $link ('hdbxscontroller', 'hdbxsexecagent', 'hdbxsuaaserver'){
        $exeLink = "$exeDir/$link";
        if (-l $exeLink){
            $msg->getSubMsgLst ()->addMessage ("Link '$exeLink' already exists");
            next;
        }
        if (symlink ("../../../$gDirNameXS2/bin/$link", $exeLink)){
            $msg->getSubMsgLst ()->addMessage ("Link '$exeLink' created");
            if (!$>){
                lchown($stat->[4], $stat->[5], $exeLink);
            }
        }
        else{
            $msg->getSubMsgLst ()->addMessage ("Cannot create link '$exeLink': $!");
        }
    }
    if (defined $restoreMode){
        $msg->getSubMsgLst ()->addMessage ("Restoring permissions of '$exeDir'");
        chmod ($restoreMode, $exeDir);
    }
    return 1;
}

sub installComponent {
    my ( $self, $instconfig) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext( [ $msg->getSubMsgLst() ] );
    my $sapSystem = $self->getSAPSystem ( $instconfig, $msg, $saveCntxt);
    my $trexInstance = $instconfig->getOwnInstance(1); # no cache

    return undef if(!defined($sapSystem));

    # TODO - Consider future refactoring of all the ifs
    if (!defined($trexInstance)) {
        $self->setErrorMessage ("Own instance not found");
        $msg->endMessage( undef, 'Install ' . $self->getComponentName());
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    if(!$self->_handleXS2InstallDir($instconfig, $sapSystem)) {
        $msg->endMessage(undef, $self->_getEndMessage(undef, "Installation"));
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    if(!$self->_storeXSSpaceIsolationUserData($instconfig, $sapSystem)){
        my $endMessage = $self->_getEndMessage(undef, "Installation");
        $msg->endMessage (undef, $endMessage);
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    $sapSystem->SetProgressHandler   ($instconfig->getMsgLst()->getProgressHandler());
    $trexInstance->SetProgressHandler($instconfig->getMsgLst()->getProgressHandler());
    $self->_createExeSymlinks ($instconfig); # <= just to save one HANA restart after initial installation
    my $util = new SDB::Install::SAPSystemUtilities();
    my $instconfigSaveContext = $instconfig->setMsgLstContext([$msg->getSubMsgLst(), $self->getErrMsgLst()]);
    my $needsSSL = $instconfig->handleGlobalIniISCMode('ssl', 1); # 1 means check only; returns 0 if there is nothing to change
    if(!defined($needsSSL)) {
        $msg->endMessage (undef, $self->_getEndMessage(0, "Installation"));
        $self->setMsgLstContext($saveCntxt);
        $instconfig->setMsgLstContext($instconfigSaveContext);
        return undef;
    }
    my $iscMode = $instconfig->getValue('ISCMode');
    my $shouldChangeIscMode = ($needsSSL && $iscMode eq 'ssl') || (!$needsSSL && $iscMode eq 'standard');
    if($shouldChangeIscMode && !$self->_changeISCMode($instconfig, $util, $sapSystem, $trexInstance, $iscMode)){
        $msg->endMessage (undef, $self->_getEndMessage(0, "Installation"));
        $self->setMsgLstContext($saveCntxt);
        $instconfig->setMsgLstContext($instconfigSaveContext);
        return undef;
    }
    $instconfig->setMsgLstContext($instconfigSaveContext);

    my $returnCode = $self->_installXS2Component($instconfig, $sapSystem, $trexInstance);
    my $endMessage = $self->_getEndMessage($returnCode, "Installation");
    $msg->endMessage (undef, $endMessage);
    $self->setMsgLstContext($saveCntxt);
    return $returnCode;
}

sub _storeXSSpaceIsolationUserData {
    my ($self, $configuration, $sapSystem) = @_;
    my $filePath = $sapSystem->getUserConfigFile();
    my $spaceIsolationUserData = {};

    if($self->_isUseProdSpaceIsolation($configuration)){
        $spaceIsolationUserData->{xs_space_user_prod} = createXSSpaceProdUserName($configuration->getSID());
        $spaceIsolationUserData->{xs_space_user_prod_id} = $configuration->getValue('XSSpaceUserIdProd');
    }
    if($self->_isUseSAPSpaceIsolation($configuration)){
        $spaceIsolationUserData->{xs_space_user_sap} = createXSSpaceSAPUserName($configuration->getSID());
        $spaceIsolationUserData->{xs_space_user_sap_id} = $configuration->getValue('XSSpaceUserIdSAP');
    }

    return 1 if(scalar(keys(%{$spaceIsolationUserData})) == 0);

    $self->getMsgLst()->AddMessage('Writing XS Advanced space OS users to user config...');
    if(!$sapSystem->saveConfig($spaceIsolationUserData)){
        $self->setErrorMessage ("Writing XS Advanced space OS users to user config failed");
        return undef;
    }
    return 1;
}

sub _changeISCMode {
    my ($self, $configuration, $util, $sapSystem, $trexInstance, $iscMode) = @_;

    return undef if(!$configuration->stopSystemForModify($util, $sapSystem, $trexInstance, 0, 1));
    return undef if(!defined($configuration->handleGlobalIniISCMode($iscMode, 0)));
    return 1;
}

sub _isUseSAPSpaceIsolation {
	my ($self, $configuration) = @_;
	my $useSAPSpaceIsolation = $configuration->getValue('XSSAPSpaceIsolation') =~ /$bool_true_pattern/;

	return !$configuration->isSkipped('XSSAPSpaceIsolation') && $useSAPSpaceIsolation;
}

sub _isUseProdSpaceIsolation {
	my ($self, $configuration) = @_;
	my $useProdSpaceIsolation = $configuration->getValue('XSSpaceIsolation') =~ /$bool_true_pattern/;

	return !$configuration->isSkipped('XSSpaceIsolation') && $useProdSpaceIsolation;
}

sub _handleXS2InstallDir {
    my ($self, $instconfig, $sapSystem) = @_;
    if ($self->_shouldCreateXSDir($instconfig, $sapSystem)) {
        return $self->_createXSDir($sapSystem);
    }
    if ($self->_shouldAdjustXSDirPermissions($sapSystem)) {
        return $self->_adjustXSDirMode($sapSystem);
    }
    return 1;
}

sub _shouldCreateXSDir {
    my ($self, $instconfig, $sapSystem) = @_;
    my $xsDir = $self->getXS2InstalledDir($sapSystem);
    return 0 if File::stat::stat($xsDir);
    return 1 if !$instconfig->isOptimizedExecution();
    return 1 if ($self->getPhase() eq OFFLINE_PHASE);
    return 0;
}

sub _shouldAdjustXSDirPermissions {
    my ($self, $sapSystem) =@_;
    my $xsDir = $self->getXS2InstalledDir($sapSystem);
    return 1 if (File::stat::stat($xsDir) && GetPermissions($xsDir) ne '755');
    return 0;
}

sub _createXSDir {
    my ($self, $sapSystem) = @_;
    my $xsDirPath = $self->getXS2InstalledDir($sapSystem);
    my $user = $sapSystem->getUser();
    $self->getMsgLst()->addMessage("Creating XSA installation dir '$xsDirPath'...");

    my $msgLst = SDB::Install::MsgLst->new();
    if (!createDirectory($xsDirPath, 0755, $user->uid(), $user->gid(), 0, $msgLst)) {
        $self->setErrorMessage("Failed to create '$xsDirPath' directory", $msgLst);
        return undef;
    }

    return 1;
}

sub _adjustXSDirMode {
    my ($self, $sapSystem) = @_;
    my $xsDirPath = $self->getXS2InstalledDir($sapSystem);
    my $user = $sapSystem->getUser();
    $self->getMsgLst()->addMessage("Adjusting ownership and permissions of XSA installation dir '$xsDirPath'...");

    my $msgLst = SDB::Install::MsgLst->new();
    if(!changeFileOwnership($xsDirPath, $user->uid(), $user->gid(), $msgLst)) {
        $self->setErrorMessage("Failed to change ownership of '$xsDirPath");
        return undef;
    }

    if(!changeFilePermissions($xsDirPath, 0755, $msgLst)) {
        $self->setErrorMessage("Failed to change permissions of '$xsDirPath'");
        return undef;
    }

    return 1;
}

sub updateComponent {
    my ( $self, $instconfig ) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext( [ $msg->getSubMsgLst() ] );
    my $sapSystem = $self->getSAPSystem ( $instconfig, $msg, $saveCntxt);
    if (not defined $sapSystem) {
        return undef;
    }
    my $trexInstance = $instconfig->getOwnInstance(1); # no cache
    if (not defined $trexInstance) {
        $self->setErrorMessage ("Own instance not found");
        $msg->endMessage( undef, 'Update ' . $self->getComponentName());
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }
    $sapSystem->SetProgressHandler   ($instconfig->getMsgLst()->getProgressHandler());
    $trexInstance->SetProgressHandler($instconfig->getMsgLst()->getProgressHandler());

    if(!$self->_handleXS2InstallDir($instconfig, $sapSystem)) {
        $msg->endMessage(undef, $self->_getEndMessage(undef, "Update"));
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    my $instconfigSaveContext = $instconfig->setMsgLstContext([$msg->getSubMsgLst(), $self->getErrMsgLst()]);
    my $needsSSL = $instconfig->handleGlobalIniISCMode('ssl', 1); # 1 means check only; returns 0 if there is nothing to change
    if(!defined($needsSSL)) {
        $msg->endMessage (undef, $self->_getEndMessage(0, "Update"));
        $self->setMsgLstContext($saveCntxt);
        $instconfig->setMsgLstContext($instconfigSaveContext);
        return undef;
    }
    my $iscMode = $instconfig->getValue('ISCMode');
    my $shouldChangeIscMode = ($needsSSL && $iscMode eq 'ssl') || (!$needsSSL && $iscMode eq 'standard');
    if($shouldChangeIscMode) {
        my $util = new SDB::Install::SAPSystemUtilities();
        if(!$self->_changeISCMode($instconfig, $util, $sapSystem, $trexInstance, $iscMode)){
            $msg->endMessage (undef, $self->_getEndMessage(0, "Update"));
            $self->setMsgLstContext($saveCntxt);
            $instconfig->setMsgLstContext($instconfigSaveContext);
            return undef;
        }
    }
    #- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    my $returnCode = $self->_installXS2Component($instconfig, $sapSystem, $trexInstance);
    if(!$returnCode) {
        $self->setMsgLstContext($saveCntxt);
        $instconfig->setMsgLstContext($instconfigSaveContext);
        return undef;
    }
    #- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    my $endMessage = $self->_getEndMessage($returnCode, "Update");
    $msg->endMessage (undef, $endMessage);
    $self->setMsgLstContext($saveCntxt);
    $instconfig->setMsgLstContext($instconfigSaveContext);
    return $returnCode;
}

sub _getEndMessage {
    my ($self, $returnCode, $type) = @_;
    my $message = "$type of " . $self->getComponentName();
    $message .= (defined $returnCode && ($returnCode == 1)) ? ' finished' : ' failed';
    return $message;
}

sub _installXS2Component {
    my ($self, $instconfig, $sapSystem, $trexInstance) = @_;
    my $exer = $self->createProcessExecutor($instconfig, $sapSystem, $trexInstance);
    return undef if !defined($exer);

    $self->initProgressHandler ();
    $exer->setOutputHandler($self->getProgressHandler ());
    my $exitCode = $exer->executeProgram();

    $self->getMsgLst ()->addMessage(undef, $exer->getMsgLst());
    $self->setLogLocation($self->parseLogFileLocation($exer->getOutputLines()));

    if (!defined $exitCode || $exitCode){
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($exer->getOutputLines());
        my $actionErr = $self->isUpdate() ?'Update ' : 'Installation ';
        my $errorMessage = "$actionErr of " . $self->getComponentName() . ' failed';
        $self->setErrorMessage ($errorMessage, ($errMsgLst->isEmpty() ? $exer->getErrMsgLst() : $errMsgLst));
        return undef;
    }
    return 1;
}

sub getSAPSystem {
    my ( $self, $instconfig, $msg, $saveCntxt) = @_;

    my $systems = $instconfig->CollectSAPSystems (undef, 1);
    my $sid = $instconfig->getValue("SID");
    my $sapSystem = $systems->{$sid};
    if (not defined $sapSystem) {
        $self->setErrorMessage ("No SAP system '$sid' found");
        $msg->endMessage( undef, 'Install ' . $self->getComponentName());
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }
    return $sapSystem;
}

sub _buildArgs {
    my ($self, $instconfig, $inst_dir, $trexInstance) = @_;
    my $systemUser = (defined $instconfig->getValue("SystemUser")) ? $instconfig->getValue("SystemUser") : "SYSTEM";
    my $dbMode = (lc($trexInstance->getDbMode()) eq 'multidb') ? 'multiple_containers' : 'single_container';
    my $args = [
        '--batch',
        '--read_password_from_stdin=xml',
        '--ignore_unknown_option', ################## temporarily
        '--installation_path='    . join($path_separator, $inst_dir, $gDirNameXS2),
        '--instlog_dir='          . $gLogDir,
        '--instance_number='      . $trexInstance->get_nr(),
        '--sid='                  . $trexInstance->get_sid(),
        '--system_db_host='       . $trexInstance->get_host(),
        '--system_db_user='       . $systemUser,
        '--system_db_mode='       . $dbMode,
        '--org_name='             . $instconfig->getValue("OrgName"),
        '--org_manager_user='     . $instconfig->getValue("OrgManagerUser"),
    ];
    if($instconfig->hasValue('TenantUser')){
        push(@{$args}, sprintf('--system_user_default_tenant=%s', $instconfig->getValue('TenantUser')));
    }
    if(! $self->isUpdate()) {
        push(@{$args}, sprintf('--prod_space_name=%s', $instconfig->getValue("ProdSpaceName")));
        push(@{$args}, sprintf('--routing_mode=%s', $instconfig->getValue("RoutingMode")));
        if ($instconfig->hasValue('XSTenantDatabaseName')) {
            my $tenantName = $instconfig->getValue('XSTenantDatabaseName');
            push(@{$args}, sprintf('--runtime_db_tenant=%s', $tenantName));
            push(@{$args}, sprintf('--prod_space_tenant_name=%s', $tenantName));
        } elsif ($instconfig->isSystemInCompatibilityMode()) {
            push(@{$args}, sprintf('--prod_space_tenant_name=%s', $instconfig->getSID()));
        }
        my $domainName = $instconfig->getValue("XsDomainName");
        if ($domainName) {
            push(@{$args}, sprintf('--default_domain=%s', $domainName));
        }
        if($self->_isUseSAPSpaceIsolation($instconfig)){
            push(@{$args}, sprintf('--execution_os_user_sap=%s', createXSSpaceSAPUserName($instconfig->getSID())));
        }
        if($self->_isUseProdSpaceIsolation($instconfig)){
            push(@{$args}, sprintf('--execution_os_user_prod=%s', createXSSpaceProdUserName($instconfig->getSID())));
        }
        if($instconfig->hasValue('XsEaDataPath') && !$instconfig->isSkipped('XsEaDataPath')){
            push(@{$args}, sprintf('--ea_data_path=%s', $instconfig->getValue('XsEaDataPath')));
        }
        if($instconfig->hasValue('XSCertPem') && !$instconfig->isSkipped('XSCertPem')){
            push(@{$args}, sprintf('--xs_cert_pem=%s', MyRealPath($instconfig->getValue('XSCertPem'))));
        }
        if($instconfig->hasValue('XSCertKey') && !$instconfig->isSkipped('XSCertKey')){
            push(@{$args}, sprintf('--xs_cert_key=%s', MyRealPath($instconfig->getValue('XSCertKey'))));
        }
        if($instconfig->hasValue('XSTrustPem') && !$instconfig->isSkipped('XSTrustPem')){
            push(@{$args}, sprintf('--xs_trust_pem=%s', MyRealPath($instconfig->getValue('XSTrustPem'))));
        }
    }
    my $timeoutString = buildTimeoutsString($instconfig, getXSInstallerTimeoutValues());
    push(@{$args}, $timeoutString) if $timeoutString;

    my $ignoreOption = $instconfig->getOptionIgnore();
    my $xsIgnoreString = $self->getApplicableIgnoreString($ignoreOption);
    return undef if !defined($xsIgnoreString);

    my $firstSupportedPhase = ($self->getSupportedPhases())->[0] if scalar ($self->getSupportedPhases()) > 0;
    my $firstCall = ! $instconfig->isOptimizedExecution() || $self->getPhase() eq $firstSupportedPhase;
    my $shouldIgnorePendingInstall = $instconfig->getIgnore ('check_pending_upgrade') ||
                                         $self->_shouldIgnorePersistedVersion($instconfig);
    if ($firstCall && $shouldIgnorePendingInstall){
        if($xsIgnoreString){
            $xsIgnoreString = join(',', $xsIgnoreString, 'check_pending_installation');
        } else {
            $xsIgnoreString = '--ignore=check_pending_installation';
        }
    }

    push(@{$args}, $xsIgnoreString) if $xsIgnoreString;
    push @{$args}, @{$self->SUPER::_buildArgs($instconfig)};
    return $args;
}

sub createProcessExecutor {
    my ($self, $instconfig, $sapSystem, $trexInstance) = @_;
    my $user           = $sapSystem->getUser();
    my $inst_dir       = $sapSystem->get_globalSidDir();
    my $uid            = $user->uid();
    my $gid            = $user->gid();
    my $stdinPasswords = $self->getCustomPasswordInput($instconfig);
    my $command       = $self->getHdbInstallExecutable();
    my $args          = $self->_buildArgs($instconfig, $inst_dir, $trexInstance);

    return undef if !defined($args);

    if ($self->isSidAdmUserExecution()) {
        return new LCM::ProcessExecutor($command, $args, $stdinPasswords);
    }

    return new LCM::ProcessExecutor($command, $args, $stdinPasswords, dirname ($command), undef, $uid, $gid);
}

sub getXS2InstalledDir {
    my ($self, $sapSystem) = @_;
    return join($path_separator, $sapSystem->get_globalSidDir(), $gDirNameXS2);
}

sub getNumberOfExpectedOutputLines{
    return 75;
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    return ($stackUpdate) ? $self->SUPER::getDefaultSelection($stackUpdate) : 0;
}

sub getComponentName {
    return $gProductNameXS2;
}

sub getSlppLogFileName {
    return 'xs.log'
}

sub requireSystemRestart {
    return 0;
}

sub parseLogFileLocation {
    my ($self, $outputLines) = @_;
    if (!defined $outputLines){
        return undef;
    }
    my $log_location;
    my $pattern = "Log file written to [\']*([^\']*)[\']*";
    foreach my $line (reverse @$outputLines){
        if (substr ($line, -1) eq '.') {
           chop $line;
        }
        ( $log_location ) = $line =~ /$pattern/i;

        if (defined $log_location) {
            return $log_location;
        }
    }
}

sub getHostRoles {
    return [$gHostRoleXS2Worker, $gHostRoleXS2Standby];
}

sub _startSystem {
    my (
        $self,
        $util,
        $sapSys,
        $instconfig
    ) = @_;
    $sapSys->resetMsgLstContext();
    my $rc = $util->startSystem($sapSys,
                                $instconfig,
                                $instconfig->getValue('Password'),
                                'Reconfigure');
    $self->getMsgLst->appendMsgLst($sapSys->getMsgLst());
    if (!$rc) {
        $self->setErrorMessage('Cannot start system', $sapSys->getErrMsgLst());
        return undef;
    }
    return 1;
}

sub requiresSqlSysPasswd{
    return 1;
}

sub isSigned {
    return 1;
}

sub getCustomPasswordInput {
    my ($self, $config) = @_;
    my $stdinPasswords = $config->getXmlPasswordStream();
    my $stdinPasswordsContent = $stdinPasswords->[0];
    $stdinPasswordsContent =~ s/tenantdb_user_password/system_pwd_default_tenant/g;
    return [$stdinPasswordsContent];
}

1;
