package LCM::Component::Installable::XS2Application;

use strict;
use File::Basename qw (dirname);
use SDB::Install::NewDBUser;
use SDB::Install::System qw(isAdmin getDirectoryFilenames);
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw($gShortProductNameXS2);
use LCM::App::ApplicationContext;
use SDB::Install::Persistence::XMLGenerator;
use File::Spec;
use Cwd qw (realpath);
use LCM::FileUtils qw(writeToFile);

use parent 'LCM::Component::InternallyPersistedInstallable';

sub _install {
    my ($self, $configuration) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext([ $msg->getSubMsgLst() ]);
    my $xsClient = $configuration->getXSClient();
    $xsClient->setMsgLstContext($self->getMsgLstContext());

    $self->initProgressHandler();
    my $returnCode = $self->_deployXsApplication($configuration, $xsClient);

    $msg->endMessage(undef, $self->_getEndMessage($returnCode));
    $self->setMsgLstContext($saveCntxt);
    return $returnCode;
}

sub _update {
    my ($self, $configuration) = @_;
    my $isSameVersionDeployed = $self->isAlreadyDeployed($configuration);
    if ($isSameVersionDeployed && !$configuration->getIgnore('check_version')) {
        my $cmpName = $self->getComponentName();
        my $version = $self->getVersion();
        $self->getMsgLst()->addMessage("Version '$version' of '$cmpName' is already deployed. Skipping redeployment...");
        return 1;
    }
    return $self->_install($configuration);
}

sub requireSystemRestart {
	return 0;
}

sub getSlppLogFileName {
	my ($self) = @_;
	return $self->getManifest()->getComponentKey() . 'log';
}

sub isAlreadyDeployed {
    my ($self, $configuration) = @_;
    my $scm = $configuration->getSystemComponentManager();
    my $installedComponent = $scm ? $scm->getComponentByKeyName($self->getComponentKeyName()) : undef;
    return $installedComponent && $installedComponent->getVersionObject()->isEqual($self->getVersionObject());
}

sub _deployXsApplication {
    my ($self, $configuration, $xsClient) = @_;
    my $shouldUseNoStart = $self->_shouldStartAfterDeployemnt($configuration) ? 0 : 1;
    my $extensionDescriptorsCSV = join(',', @{$self->_getExtentionDescriptors($configuration) || []});
    my $appPath = $self->getPath();

    if (!$xsClient->deployApplication($appPath, $shouldUseNoStart, $extensionDescriptorsCSV)) {
        my $actionErr    = $self->isUpdate() ? 'Update' : 'Installation';
        my $errorMessage = "$actionErr of " . $self->getComponentName() . ' failed';
        $self->appendErrorMessage($errorMessage);
        return undef;
    } elsif ( ! $self->isUpdate() ) {
        $self->detectCockpitAppsUrls($configuration, $xsClient->getOutputLines());
    }
    return 1;
}

sub detectCockpitAppsUrls {
    my ($self, $instconfig, $outputLines) = @_;
    my $pattern = "Application \"(cockpit-admin-web-app|cockpit-web-app)\" started and available at \"(.*)\"";
    
    return undef if (!defined($outputLines));

    for my $line (reverse(@{$outputLines})){
        $instconfig->{$1} = $2 if($line =~ /$pattern/i);
    }
}

sub _shouldStartAfterDeployemnt {
    my ($self, $configuration) = @_;
    my $ownKeyName = $self->getComponentKeyName();
    my $noStartKeyNamesCSV = $configuration->getValue('XSComponentsNoStart') || '';
    my @selectedKeyNamesNoStart = split(/\s*,\s*/, $noStartKeyNamesCSV);
    my $isSelectedForNoStart = grep { $_ eq $ownKeyName || $_ eq 'all' } @selectedKeyNamesNoStart;
    my $isNoStartParameterActive = !$configuration->isSkipped('XSComponentsNoStart');

    return ($isNoStartParameterActive && $isSelectedForNoStart) ? 0 : 1;
}

sub _getDefaultExtentionDescriptorLocation{
	my ($self, $configuration) = @_;
	my $Xs2ApplicationComponents = $self->getComponentManager()->getXs2ApplicationComponents();
	my @selectedXSAppDirs = ();
	
	foreach(@$Xs2ApplicationComponents){
		if( $_->isComponentSelected()){
			my ($volume,$directories,$file) = File::Spec->splitpath($_->getPath());
			push( @selectedXSAppDirs , $directories) if(! grep(/^$directories$/,@selectedXSAppDirs) );
		}
	}
	return \@selectedXSAppDirs;
}

sub _getExtentionDescriptors{
	my ($self, $configuration) = @_;
	my $extensionDescriptorFilesNames;
	my @extensionDescriptorFiles = ();
	if(!$configuration->isSkipped('XSComponentsCfg') && $configuration->hasValue('XSComponentsCfg')){
	     my $xsComponentsCfgValue = $configuration->getValue('XSComponentsCfg');
	     if ($xsComponentsCfgValue ne "") {
             my $extentionDescriptorDir = realpath($xsComponentsCfgValue);
             $extensionDescriptorFilesNames = $self->_getExntionDescriptorsFromDir($extentionDescriptorDir);
             @extensionDescriptorFiles = map(File::Spec->catfile($extentionDescriptorDir,$_),@$extensionDescriptorFilesNames)	         
	     }
    }
    my $defaultExtentionDescriptorLocation = $self->_getDefaultExtentionDescriptorLocation();
    foreach my $location (@$defaultExtentionDescriptorLocation){
    	my $locationDescriptors = $self->_getExntionDescriptorsFromDir($location);
    	foreach my $descriptor (@$locationDescriptors){
    		push( @extensionDescriptorFiles , File::Spec->catfile($location,$descriptor)) if(! grep(/^$descriptor$/,@$extensionDescriptorFilesNames) );
    	}
    }
    return \@extensionDescriptorFiles
}

sub _getExntionDescriptorsFromDir{
	my ($self, $extentionDescriptorDir) = @_;
	my $filenames = getDirectoryFilenames($extentionDescriptorDir);
	my @extensionDescriptorFiles = grep {$_ =~ /^(.+)\.[Mm][Tt][Aa][Ee][Xx][Tt]$/} @$filenames;
	return \@extensionDescriptorFiles;
}

sub _getEndMessage {
    my ($self, $returnCode) = @_;
    my $action = $self->isUpdate() ? "Update" : "Installation";
    my $suffix = ($returnCode == 1) ? ' finished' : ' failed';

    return sprintf("%s of %s %s", $action, $self->getComponentName(), $suffix);
}

# Custom .XML name because this is not a real component
# but a mock component responsible for seleted XSA Apps
sub getPersistenceXmlName() {
    return 'pending_xs_components.xml';
}

sub isDependencyForComponent{
	my ($self, $component) = @_;
	my $manifest = $self->getManifest();
	my $name = $self->getComponentName();
	my $technicalName = $manifest->getComponentKey();
	my $scvVersion = $manifest->getValue('release');
	my $spLelvel = $manifest->getValue('rev-number');
	my $patchLelvel = $manifest->getValue('rev-patchlevel');
	my $dependencyObject = new LCM::Utils::XsApplicationDependency::InclusiveDependency($name, $technicalName, $scvVersion, $spLelvel, $patchLelvel);
	my $targetSet = { $technicalName => $dependencyObject };
	my $dependencies = $component->getManifest()->getValue('dependencies-array') || [];

	for my $dependencyArray (@{$dependencies}){
		for my $dependencyObject (@{$dependencyArray}){
			next if(!$dependencyObject->isa('LCM::Utils::XsApplicationDependency::InclusiveDependency'));
			return 1 if($dependencyObject->isSatisfied($targetSet));
		}
	}
	return 0;
}

1;