package LCM::ComponentManager::SystemComponentManager;

use strict;
use SDB::Install::System qw(readLink isLink isDirectory isRegularFile);
use SDB::Install::SysVars qw($path_separator $isWin);
use LCM::Component::Installed;
use SDB::Install::Globals qw(
    $gDirNameResidentInstaller
    $gKeynameEngine
    $gKeynameClient
    $gKeynameStudio
    $gKeynameAFL
    $gKeynameLCA
    $gKeynameHLM
    $gKeynameSDA
    $gKeynameStreaming
    $gKeynameEs
    $gKeynameAccelerator
    $gKeynameRDSync
    $gKeynameXS2
    $gKeynameOfficeClient
    $gKeynameCockpitStack
    $gKeynameLSS
    $gXSASpaceSAP
    GetComponentDirsCollection
);
use SDB::Install::SAPSystem qw(CollectSAPSystems);
use LCM::Manifests::ManifestFactory;
use LCM::ComponentManager::InstalledXSApplicationsManager;
use LCM::FileUtils qw(listDirectory);
use File::Spec;
use File::Basename qw (dirname);
use File::stat qw();
use SDB::Install::LSS::LssInstance;

use base qw (LCM::ComponentManager);


sub new {
    my ($class) = shift();
    my $self = $class->SUPER::new (@_);
    $self->setXSAppsManager(LCM::ComponentManager::InstalledXSApplicationsManager->new());
    return $self;
}

sub setXSAppsManager {
    my ($self, $manager) = @_;
    $self->{xsAppsManager} = $manager;
}

sub getXSAppsManager {
    my ($self) = @_;
    return $self->{xsAppsManager};
}

sub detectComponentsBySid {
    my ($self, $sid) = @_;
    my $systems = CollectSAPSystems (undef, 1);
    my $sapSystem = $systems->{$sid};
    if (!defined $sapSystem){
        my $config = $self->{instconfig};
# Workaround: See comment in UpdateHostConfigurationFactory
        my $isBeforeRenameWithContinue = defined ($config) ? $config->{isBeforeRenameWithContinue} : undef;
        my $isRenameConfiguration = defined($config) && ($config->isa('LCM::Configuration::RenameSystemConfiguration') || $isBeforeRenameWithContinue);
        my $isRegisterRename = $isRenameConfiguration && $config->can('isRegisterScenario') && $config->isRegisterScenario();
        my $isUpdateAfterRenameConfiguration = defined($config) && $config->isa('LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationInRename');
        my $isContinueRename = $isRenameConfiguration && ($config->{isContinueOption} || $isBeforeRenameWithContinue);
        
        if (!$isRegisterRename && !$isContinueRename && !$isUpdateAfterRenameConfiguration) {
            $self->setErrorMessage ("No SAP system '$sid' found");
        }
        return 0;
    }
    return $self->detectComponentsBySapSystem ($sapSystem)
}

sub detectComponentsBySapmntSid {
    my ($self, $sapmnt, $sid) = @_;
    my $sapSystem = new SDB::Install::SAPSystem ();
    if (!defined $sapSystem->initWithGlobDir ($sid, $sapmnt)){
        $self->setErrorMessage (undef, $sapSystem->getErrMsgLst());
        return undef;
    }
    return $self->detectComponentsBySapSystem ($sapSystem)
}

sub detectComponentsBySapSystem {
    my ($self, $sapSystem) = @_;
    my $msglst = $self->getMsgLst ();
    my $sid = $self->{sid} = $sapSystem->get_sid();
    my $msg = $msglst->addMessage("Looking for components of system ${sid}...");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);

    if (!$self->_detectServer($sapSystem)) {
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    for my $componentDir (@{$self->_getComponentDirs($sapSystem)}) {
        my $manifestPath = File::Spec->catfile($componentDir, 'manifest');
        my $manifest = LCM::Manifests::ManifestFactory::createComponentManifest($manifestPath);

        $self->detectComponentByManifest($componentDir, $manifest);
    }

    # Check for installed XS2 applications on the system if XSA is detected
    # At this points XSA should be detected if it's present on the system
    if ($self->isComponentAvailable($gKeynameXS2)) {
        $self->_detectInstalledXS2Components($sapSystem);
    }

    $self->setMsgLstContext ($saveCntxt);
    return 1;
}

sub _detectInstalledXS2Components {
    my ($self, $sapSys) = @_;
    my $installPath = File::Spec->catfile($sapSys->get_target(), $self->getSid());
    my $xsAppsManager = $self->getXSAppsManager();

    $xsAppsManager->detectInstalledXsComponents($installPath);
    $self->addComponent($_) for @{$xsAppsManager->getAppsInCurrentRegistry()};
}

sub _getComponentDirs {
    my ($self, $sapSys) = @_;
    my $hanamntSid = $isWin ? $sapSys->getUsrSapSid () : $sapSys->get_globalSidDir();
    my @componentDirs = map { File::Spec->catfile($hanamntSid, $_) } @{GetComponentDirsCollection()}; # All components under <sapmnt>/<SID>
    push @componentDirs, File::Spec->catfile($hanamntSid, $gDirNameResidentInstaller, 'instruntime');

    my $hdbInstance = $sapSys->getNewDBInstances ()->[0];
    if ($hdbInstance) {
        my $lssInstance = $hdbInstance->getLssInstance();
        if ($lssInstance) {
            push @componentDirs, $lssInstance->getLssSidDir();
        }

        push @componentDirs, @{$self->_detectReferenceDataManifestDirs($hdbInstance) // []};
        push @componentDirs, @{$self->_detectPluginsManifestDirs($hdbInstance) // []};
    }

    return [ grep { SDB::Install::System::isRegularFile(File::Spec->catfile($_, 'manifest')) } @componentDirs ];
}

sub switchXSComponentRegistry {
    my ($self, $newRegistry) = @_;
    my $xsAppsManager = $self->getXSAppsManager();
    $xsAppsManager->setXSComponentRegistry($newRegistry);
    $self->_refreshXSComponents();
}

sub _refreshXSComponents {
    my ($self) = @_;
    my $xsAppsManager = $self->getXSAppsManager();
    my @platformComponents = grep { !$_->isXS2Application() } @{$self->getDetectedComponents()};
    $self->clearDetectedComponents();

    $self->addComponent($_) for @platformComponents;
    $self->addComponent($_) for @{$xsAppsManager->getAppsInCurrentRegistry()};
}

sub _detectServer {
    my ($self, $sapSystem) = @_;
    my $sapSystemManifest = $sapSystem->getManifest ();
    if (!defined $sapSystemManifest){
        $self->setErrorMessage ('Cannot get manifest', $sapSystem->getErrMsgLst());
        return undef;
    }
    my $serverManifest = LCM::Manifests::ManifestFactory::createComponentManifest($sapSystemManifest->getFileName());
    if (!$self->detectComponentByManifest($sapSystem->get_globalTrexInstallDir(), $serverManifest)){
        return undef;
    }
    return 1;
}

sub _detectPluginsManifestDirs {
    my ($self, $hdbInstance) = @_;
    my $msg = $self->getMsgLst()->addMessage("Looking for server plugins...");
    my $saveCntxt = $hdbInstance->setMsgLstContext([$msg->getSubMsgLst()]);

    my @serverPlugins = grep { $_ } @{$hdbInstance->getPluginInstallations() // []};
    return [ map { $_->getManifestDir() } @serverPlugins ];
}

sub _detectReferenceDataManifestDirs {
    my ($self, $hdbinstance) = @_;
    my $msg = $self->getMsgLst()->addMessage ('Detecting Reference Data Installations...');
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
    my $result = [];

    my $refInstallDir = $self->_getReferenceDataInstallDir( $hdbinstance );
    if (!$refInstallDir) {
        $self->setMsgLstContext($saveCntxt);
        return $result;
    }

    my @dirContents = grep {!/^\.{1,2}$/ } @{listDirectory($refInstallDir, $self->getErrMsgLst()) // []};
    if (!@dirContents) {
        $self->setMsgLstContext($saveCntxt);
        return $result;
    }

    foreach my $subdir (@dirContents){
        my $link = File::Spec->catfile($refInstallDir, $subdir);
	    next if !SDB::Install::System::isLink($link);

	    my $refdata = SDB::Install::System::readLink($link);
	    my $parentDir = dirname($refdata);
        push @{$result}, $parentDir;
    }

    $self->setMsgLstContext($saveCntxt);
    return $result;
}

sub _getReferenceDataInstallDir {
    my ($self, $hdbinstance) = @_;
    my $globalSidDir = $hdbinstance->get_globalSidDir();
    my $configDir = join ($path_separator, $globalSidDir, 'global', 'hdb', 'custom' ,'config');
    my $scriptServerIniPath = File::Spec->catfile($configDir, 'scriptserver.ini');
    return 0 if(!SDB::Install::System::isRegularFile($scriptServerIniPath));

    my $layeredCfg = $hdbinstance->getLayeredConfig ();
    if (!defined $layeredCfg){
        $self->appendErrorMessage ('Cannot get layered configuration', $layeredCfg);
        return undef;
    }
    
    $layeredCfg->resetMsgLstContext();
    my $scriptServerIni = $layeredCfg->getIniFile ('scriptserver.ini');
    if (!defined $scriptServerIni){
        $self->appendErrorMessage ('Cannot get scriptserver.ini', $layeredCfg);
        return undef;
    }

    $scriptServerIni->resetMsgLstContext();
    if (!defined $scriptServerIni->readValues ()){
        $self->appendErrorMessage ('Cannot read scriptserver.ini', $scriptServerIni);
        return undef;
    }

    my $referenceDataPath = $scriptServerIni->getValue ('adapter_framework','dq_reference_data_path');
    if(! defined $referenceDataPath || !SDB::Install::System::isDirectory($referenceDataPath)){
        return 0;
    }

    return $referenceDataPath;
}

sub detectComponentByManifest{
    my ($self, $installerDir, $manifest) = @_;
    my $component;
    my $msg = $self->getMsgLst ()->addMessage ("Checking component...");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
    $component = new LCM::Component::Installed ($installerDir, $manifest, $self, undef, $self->isSidAdmUserExecution()); # Installable acts as a factory for its subclasses.
    if ($component->errorState){
        $self->appendErrorMessage ("Invalid component", $component->getErrMsgLst());
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
    $component->addComponentInfosToMsgLst ($msg->getSubMsgLst);
    if (!$self->addComponent ($component)){
        $self->setErrorMessage ('Cannot add component', $self->getErrMsgLst());
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
    $self->setMsgLstContext ($saveCntxt);
    return 1;
}

sub getSortedComponentKeynames{
	my ($self) = @_;
	my @sortedComponentKeynames;

	push (@sortedComponentKeynames,
		$gKeynameHLM,
		$gKeynameOfficeClient,
		$gKeynameStudio,
		$gKeynameClient );

	foreach my $component(@{$self->getServerPluginComponents()}) {
		push (@sortedComponentKeynames, $component->getComponentKeyName());
	}

    foreach my $component(@{$self->getReferenceDataComponents()}) {
        push (@sortedComponentKeynames, $component->getComponentKeyName());
    }

	push (@sortedComponentKeynames,
		$gKeynameSDA,
		$gKeynameStreaming,
		$gKeynameEs,
		$gKeynameRDSync,
		$gKeynameAccelerator,
		$gKeynameXS2,
        $gKeynameCockpitStack,
        $gKeynameEngine,
        $gKeynameLSS);

	return \@sortedComponentKeynames;
}

sub getSid {
    my ($self) = @_;
	return $self->{sid};
}

sub isComponentAvailable {
	my($self, $componentKeyname) = @_;
	return (defined $self->getComponentByKeyName($componentKeyname));
}

sub hasIncompatibleToMultiDbComponents{
    my $self = shift;
    my $unsupportComponentNames = $self->getUnsupportedMultiDBComponents();
    return scalar(@$unsupportComponentNames) > 0;
}

sub getUnsupportedMultiDBComponents{
  my $self = shift;
    my $components = $self->getAllComponents();

    my @unsupportedComponents = grep {
        $_->isFeatureUnsupported('multidb') ||
        $_->isFeatureUnsupported('convert_to_multidb')
    } @{$components};

    my @unsupportComponentNames = map { $_->getComponentName() } @unsupportedComponents;
    return \@unsupportComponentNames;
}

sub hasUninstallableComponents {
    my ($self) = @_;
    return grep { $_->canBeSelectedForUninstall() } @{$self->getAllComponents()};
}

1;