package LCM::Configuration::ComponentValidation::XsApplicationDependencyCheck;

use strict;
use warnings;
use LCM::Utils::XsApplicationDependency::InclusiveDependency;
use SDB::Install::Globals qw($gFlavourCockpit);


sub new{
    my ($class, $config) = @_;
    my $self = bless({}, $class);
    $self->_setConfiguration($config);
    return $self;
}

sub _setConfiguration {
    my ($self, $config) = @_;
    $self->{_config} = $config;
}

sub _getConfiguration {
    my ($self) = @_;
    return $self->{_config};
}

sub checkXS2ComponentDependencies {
    my ($self) = @_;
    my $config = $self->_getConfiguration();
    my $mcm = $config->getMediumComponentManager();
    my $xsComponents = $mcm->getXs2ApplicationComponents();
    my %xsComponentMap = map { lc($_->getComponentKeyName()) => $_ } @{$xsComponents};
    my $result = 1;

    for my $xs2Component (@{$xsComponents}) {
        next if(! $xs2Component->isComponentSelected());

        my $manifest = $xs2Component->getManifest();
        my $dependencies = $manifest->getValue('dependencies-array');

        # If dependency component is detected - use its name,
        # because values in the SL_MANIFEST.XML might be inconsistent
        for my $dependencyArray (@{$dependencies}){
            for my $dependencyObject (@{$dependencyArray}){
                my $technicalName = lc($dependencyObject->getTechnicalName());
                my $detectedComponent = $xsComponentMap{$technicalName};

                if(defined($detectedComponent)){
                    my $name = $detectedComponent->getComponentName() || $dependencyObject->getName();
                    $dependencyObject->setName($name);
                }
            }
        }

        my $targetSet = $self->_buildDependencySet($config->gatherTargetSetOfComponents());
        my ($isSatisfied, $errorMessage) = $manifest->checkComponentDependency($targetSet);
        if(!$isSatisfied){
            $config->getErrMsgLst()->addError($errorMessage);
            $result = 0;
        }
    }
    return $result;
}

sub checkDependenciesToXs2Components {
    my ($self) = @_;
    my $config = $self->_getConfiguration();

    for my $component (@{$config->gatherTargetSetOfComponents()}) {
        next if !$self->shouldCheckDependencyToApplication($component);

        if (!$self->_hasValidXs2ComponentDependencies($component)) {
            return 0;
        }
    }

    return 1;
}

sub shouldCheckDependencyToApplication {
    my ($self, $component) = @_;
    my $config = $self->_getConfiguration();

    if ($config->isCockpitOnPlatformAction()) {
# In the Cockpit on Platform case, all the platform components and XS apps, which do not
# belong to the target Cockpit Space will depend on the Apps in the SAP space => it doesn't
# make sense to validate their dependencies to xs components, which will be installed in totally
# different one. Furthermore this check would've otherwise always failed
        return 0 if !$component->isCockpitStack();
    } elsif ($component->isCockpitStack()) {
# No need to validate dependencies of the Cockpit Stack if it has been installed on HANA Platform and the
# current config is not CockpitOnHANAConfiguration because the configuration works with the 'sap' space in
# this case and all the apps, the cockpit stack depends on, are installed in another space => this check will always fail.
        my $scm = $config->getSystemComponentManager();
        my $installedFlavour = $scm ? $scm->getHANAFlavour() : undef;
        if (defined($installedFlavour) && $installedFlavour ne $gFlavourCockpit) {
            return 0;
        }
    }

    return 1;
}

sub _hasValidXs2ComponentDependencies {
    my ($self, $component) = @_;
    my $manifest = $component->getManifest();
    my $parser = $manifest->getDependencyParser();
    return 1 if ($parser == 1); # No dependencies

    if (!defined $parser) {
        $self->getErrMsgLst()->appendMsgLst($manifest->getErrMsgLst ());
        return 0;
    }

    my $config = $self->_getConfiguration();
    my $dependecySet = $self->_buildDependencySet($config->gatherTargetSetOfComponents(), 1); # 1 - xs apps only
    my @targetComponentsKeynames = map { uc($_->getComponentKeyName()) } @{$config->gatherTargetSetOfComponents()};
    foreach my $vendor (@{$parser->getDependenciesVendors()}) {
        foreach my $keyname (@{$parser->getDependenciesKeyNamesForVendor($vendor)}) {
# Check for platform components here as well because we do not have a way of knowing whether a given
# component keyname, which comes from the dependecy parser belongs to a XS application of to a platform
# component.
            if (!grep { $_ eq uc($keyname) } @targetComponentsKeynames) {
                my $caption = $component->getComponentKeyCaptionByKeyName($keyname);
                my $cmpType = $component->isInstalled() ? 'installed' : 'selected';
                my $requiresMessage = $config->_getRequiresComponentMessage($component, $keyname,
                    $parser->getMinVersion($vendor, $keyname),
                    $parser->getMaxVersion($vendor, $keyname),
                    "The $cmpType component ");

                $config->onComponentValidationMessage($requiresMessage, 'SelectedXs2Applications');
                if (!$config->isPrepareUpdate()) {
                    return 0;
                }
            }
            foreach my $xs2ComponentKeyname (keys %$dependecySet) {
                next if (lc($keyname) ne $xs2ComponentKeyname);

                my $xs2Component = $dependecySet->{$xs2ComponentKeyname};
                my $rel = $xs2Component->getScvVersion();
                my $sp = $xs2Component->getSpLevel();
                my $pl = $xs2Component->getPatchLevel();
                my $minDependency = $parser->getMinVersion($vendor, $keyname);
                my $maxDependency = $parser->getMaxVersion($vendor, $keyname);
                if ((defined $minDependency) && (!$minDependency->isCompatible($rel, $sp, $pl)) ||
                    (defined $maxDependency) && (!$maxDependency->isCompatible($rel, $sp, $pl))) {
                    my $msg;
                    my $caption = $component->getComponentKeyCaptionByKeyName($keyname);
                    my $requiresMessage = $config->_getRequiresComponentMessage($component, $keyname, $minDependency, $maxDependency);
                    if ($component->isInstalled()) {
                        $msg = "An already installed component " . $component->getComponentName() . " is not compatible with " .
                            $caption . " version '" . $xs2Component->getVersionString() . "'. " . $requiresMessage;
                            if($component->getComponentBatchKey() eq 'lcapps'){
                                $msg.=" To proceed with the update, follow the instructions in SAP Note 2223318.";
                            } elsif ($manifest->isServerPlugin()) {
                                $msg.=" Updates for plugins are available on the SAP Support Portal. For withdrawn plugins, follow SAP Note 2293092.";
                            }
                    } else {
                        my $selectedStr = $component->isInstaller() ? 'Auto selected' : 'Selected';
                        $msg = "$selectedStr component " . $component->getComponentName() . " is not compatible with " .
                            $caption . " version '" . $xs2Component->getVersionString() . "'. " . $requiresMessage;
                    }
                    $config->onComponentValidationMessage($msg, 'SelectedXs2Applications');
                    if (!$config->isPrepareUpdate()) {
                        return 0;
                    }

                    return 1;
                }
            }
        }
    }
    return 1;
}

sub _buildDependencySet {
    my ($self, $targetComponents, $xsAppsOnly) = @_;
    my $result = {};

    for my $component (@{$targetComponents}) {
        next if $xsAppsOnly && !$component->isXS2Application();
        my $manifest = $component->getManifest();
        my $name         = $component->getComponentName();
        my $componentKey = $component->getComponentKeyName();
        my $scvVersion   = $manifest->getValue('release');
        my $spLelvel     = $manifest->getValue('rev-number');
        my $patchLelvel  = $manifest->getValue('rev-patchlevel');
        my $dependencyObject = new LCM::Utils::XsApplicationDependency::InclusiveDependency($name, lc($componentKey), $scvVersion, $spLelvel, $patchLelvel);

        $result->{lc($componentKey)} = $dependencyObject;
    }

    return $result;
}

1;