package LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration;

use strict;

use parent qw ( SDB::Install::Configuration::AnyModifyConfig
                LCM::Configuration::AddHostsCertifiedHostConfig
                LCM::Configuration::GenericStackAny
                LCM::Configuration::XS2Configuration
                LCM::Configuration::AutoInitializeFamilyServicesConfiguration);

use File::Spec;

use LCM::App::ApplicationContext;
use LCM::Configuration::GenericStackAny;
use LCM::HostsParser qw( ExtractLocalAndRemoteHosts ParseHostStringToMap appendDefaultValuesToAddHostsCmd );
use LCM::HostsUtility qw (IsHostPartOfSystem);
use LCM::SapHostAgentFunctions;
use LCM::Configuration::ParametersCreator;
use LCM::App::AddHostsUtils;
use LCM::SummaryTreeBuilders::AddHostsSummaryTreeBuilder;
use LCM::Task qw($gFlavourProductName);

use LCM::Configuration::ValueChangeListeners::AddHostSIDListener;
use LCM::Configuration::ValueChangeListeners::AddHostAddHostsListener;
use LCM::Configuration::ValueChangeListeners::AddHosts::RemoteCredentialsListener;
use LCM::Configuration::ValueChangeListeners::AddHostHostagentPasswordListener;
use LCM::Configuration::ValueChangeListeners::AddHostRemoteExecutionListener;
use LCM::Configuration::ValueChangeListeners::SHAVersionListener;
use LCM::Configuration::ValueChangeListeners::AddHosts::XSUserIsolationListener;
use LCM::Configuration::ValueChangeListeners::CheckLocalSystemRequirementsListener;
use LCM::Configuration::ValueChangeListeners::PostCollectHostInfoChecksListener;
use LCM::Configuration::ValueChangeListeners::AddHosts::AddHostRootCredentialsListener;
use LCM::Configuration::ValueChangeListeners::ConfigureInternalNetwork::ListenInterfaceRestartListener;
use LCM::Configuration::ValueChangeListeners::ConfigureInternalNetwork::InternalNetworkRestartListener;
use LCM::Configuration::Hosts::AddHosts::Validators::LSS::OSUserCheck;
use LCM::Configuration::Hosts::AddHosts::Validators::LSS::InstallationPathCheck;

use SAPDB::Install::Hostname;

use SDB::Install::Configuration qw($bool_true_pattern);
use SDB::Install::SysVars;
use SDB::Install::System qw(isSidadmin);
use SDB::Install::Configuration::AnyModifyConfig;
use SDB::Install::Globals qw ($gSapsysGroupName
                              $gHostRoleAcceleratorWorker
                              $gProductNameEngine
                              $gProductNameSystem
                              $gHostRoleXS2Standby
                              $gHostRoleXS2Worker
                              $gHostRoleStandby
                              $gHostRoleWorker
                              $gHostRoleEsWorker
                              $gXSParametersConstraint
                              $gXSParametersAddHostOrRoleConstraint
                              $gKeynameEngine
                              $gShortProductNameLSS);
use SDB::Install::Configuration::TenantUserListener;
use SDB::Install::Configuration::LSSUserStringListener;
use SDB::Install::Configuration::LSSPasswordListener;

use experimental qw (smartmatch);

our $TITLE = "Add Hosts to $gFlavourProductName System";
our $HOSTS_ACTION = "add_hosts";

our @EXPORT_OK = qw( $HOSTS_ACTION );

sub new {
    my $self  = shift->SUPER::new (@_);

    my $order = 0;
    my $acceleratorOnly = "Valid with $gHostRoleAcceleratorWorker/standby only";
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();

    $self->{params} = {
    	'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $ini_section_addhosts),
        ( ! $isWin ? ( 'RemoteExecution' => $self->getParamRemoteExecution($order++, $ini_section_addhosts, 0),
                      'UseHttp'         => $self->getParamUseHttp        ($order++, $ini_section_addhosts), )
                   : () ),
        'InstallHostagent'   => $self->getParamInstallHostagent  ($order++, $ini_section_addhosts, undef, 1),
        'Target'             => $self->getParamTarget            ($order++, $ini_section_addhosts),
        'SID'                => $self->getParamSID               ($order++, $ini_section_addhosts),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $ini_section_addhosts,undef,$tenantCredentialsConstraint),
        'AddHosts'           => $self->getParamAddHosts          ($order++, $ini_section_addhosts, 1),
        'InstallSSHKey'      => $self->getParamInstallSSHKey     ($order++, $ini_section_addhosts),
        'RootUser'           => $self->SDB::Install::Configuration::AnyConfig::getParamRootUser ($order++, $ini_section_addhosts),
        'RootPassword'       => $self->getParamRootPassword      ($order++, $ini_section_addhosts),
        'SkipHostagentPw'    => $self->getParamSkipHostagentPw   ($order++, $ini_section_addhosts, undef, 0),
        'HostagentPassword'  => $self->getParamHostagentPassword ($order++, $ini_section_addhosts),
        'HostsProperties' => LCM::Configuration::ParametersCreator::GetParamHostsProperties( $order++, $ini_section_addhosts ),
        'NoStart'            => $self->getParamNoStart           ($order++, $ini_section_addhosts,
                                "Do not start added hosts and do not start $flavourProductName System",
                                "Does not start added hosts and does not start $flavourProductName System",
                                0), # not hidden
        'AutoAddXS2Roles'  => LCM::Configuration::ParametersCreator::GetParamAutoAddXS2Roles($order++, $ini_section_addhosts, $gXSParametersConstraint),
        'ImportContentXS2'   => $self->getParamImportContentXS2  ($order++, $ini_section_addhosts),
        'ListenInterface'    => $self->getParamListenInterface   ($order++, $ini_section_addhosts, 1),
        'InternalNetwork'    => $self->getParamInternalNetwork   ($order++, $ini_section_addhosts, 1, undef, 1, 0, 1),
        'Password'           => $self->getParamPassword          ($order++, $ini_section_addhosts),
        'InitialHostagentPassword'  => $self->getParamHostagentPassword ($order++, $ini_section_addhosts),
        'SystemUser'         => $self->getParamSystemUser        ($order++, $ini_section_addhosts, 1, $acceleratorOnly),
        'SQLSysPasswd'       => $self->getParamSQLSysPasswd      ($order++, $ini_section_addhosts, 'passwd', 1, undef, $acceleratorOnly),
        'AcceleratorUser'    => $self->getParamAcceleratorUser   ($order++, $ini_section_addhosts, $acceleratorOnly),
        'AcceleratorPassword'=> $self->getParamAcceleratorPassword($order++,$ini_section_addhosts, $acceleratorOnly),
        'OrgManagerUser'     => $self->getParamOrgManagerUser    ($order++, $ini_section_addhosts, $gXSParametersAddHostOrRoleConstraint),
        'OrgManagerPassword' => $self->getParamOrgManagerPassword($order++, $ini_section_addhosts, $gXSParametersAddHostOrRoleConstraint),
        'CertificatesHostmap'=> $self->getParamCertificatesHostmap($order++, $ini_section_addhosts),
# The 'UID' dummy param is added to the configuration
# because otherwise the CollectOtherHostInfos
# method won't collect existing sidadms on remote hosts
# The same is valid for the 'GID' param, but for the 'sapsys' groups.
        ($isWin
                ? ()
                : (
                     'UID' => $self->getParamUID($order++),
                     'GID' => $self->getParamGID($order++),
                     'SkipModifySudoers' => $self->getParamSkipModifySudoers($order++, $ini_section_addhosts),
                  ),
         ),
        'TenantUser'     => $self->getParamTenantUser    ($order++, $ini_section_addhosts,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++, $ini_section_addhosts,1,$tenantCredentialsConstraint),
        'LSSPassword'    => $self->getParamLSSPassword($order++, $ini_section_addhosts),
        'Ignore'         => LCM::Configuration::ParametersCreator::GetParamIgnore($order++, $ini_section_addhosts, $self->getIgnoreValues()),
    };

    $self->setSkip('SkipHostagentCalls', isSidadmin());
    for my $paramID ('UID', 'GID') {
        $self->setSkip($paramID, 1);
        $self->setHidden($paramID, 1);
    }
    $self->setType('HostagentPassword', 'passwd');
    $self->setType('OrgManagerPassword', 'passwd');
    $self->{params}->{HostsProperties}->{custom_input} = \&LCM::App::AddHostsUtils::readHostsProperties;
	$self->{params}->{AddHosts}->{custom_input} = \&LCM::App::AddHostsUtils::readInteractiveAddHostsInput;
	$self->{_shaHelper} = new LCM::SapHostAgentFunctions(undef);
	$self->_reconfigureParameters();
	$self->addListeners();
	$self->setCollectOtherHostInfosRequired(1);

	return $self;
}

sub InitDefaults {
    my $self = shift();
    my $sapsysGid = getgrnam($gSapsysGroupName);
    if (defined($sapsysGid)) {
        $self->{localSapsysGID} = [ $sapsysGid, hostname() ];
    }
    return $self->SUPER::InitDefaults(@_);
}

sub checkHostagentPassword {
	my ($self, $value) = @_;

	if ( ! $value ) {
        $self->PushError( "$self->{params}->{HostagentPassword}->{str} must not be empty." );
        return undef;
    }

	require LCM::Configuration::Hosts::AddHosts::Validators::HostagentPasswordValidator;
    my $shaPasswdValidator = new LCM::Configuration::Hosts::AddHosts::Validators::HostagentPasswordValidator( $self );
    if ( ! $shaPasswdValidator->validate( $value, $self ) ) {
    	return undef;
    }

	$self->_prepareCertificatesHostmap( $self->getValue('AddHosts') );

	return 1;
}

sub checkLocalHostagentPassword {
    my ($self, $value ) = @_;
    return $self->SUPER::checkHostagentPassword( $value );
}

sub isOnlyLocalHostForAdding{
	my ($self) = @_;
	my $remoteHostsInputString = $self->getSelectedRemoteHosts();
	return if ( defined $remoteHostsInputString ) ? 0 : 1;
}

sub _reconfigureParameters {
	my ($self) = @_;

	my $params = $self->{params};

	# Target and SID are set by SBD::Install::Installer::getInstallation
	$params->{SID}->{hidden}    = 1;
	$params->{SID}->{set_interactive} = 0;
	$params->{Target}->{hidden} = 1;
	$params->{Target}->{set_interactive} = 0;

	$params->{AddHosts}->{set_interactive} = 0;
	$params->{AddHosts}->{mandatory} = 1;
	$params->{Password}->{skip}              = 1;
	$params->{RootUser}->{init_with_default} = 1;
	$params->{RootUser}->{set_interactive}   = 1;
	$params->{InstallHostagent}->{skip} = 0;
	$self->setDefault('InstallHostagent', 1);
	$params->{HostagentPassword}->{skip} = 1;
	$params->{SkipHostagentPw}->{skip}       = 1;
	$params->{CertificatesHostmap}->{skip} = 1;
	$params->{CertificatesHostmap}->{isValid} = 1;

	$params->{SystemUser}->{skip}          = 1;
	$params->{SQLSysPasswd}->{skip}        = 1;
	$params->{AcceleratorUser}->{skip}     = 1;
	$params->{AcceleratorPassword}->{skip} = 1;
}

sub addListeners {
    my ($self) = @_;

    $self->addParameterListener( 'SID', LCM::Configuration::ValueChangeListeners::AddHostSIDListener->new());
    $self->addParameterListener( 'SID', LCM::Configuration::ValueChangeListeners::CheckLocalSystemRequirementsListener->new());
    $self->addParameterListener( 'AddHosts', LCM::Configuration::ValueChangeListeners::AddHostAddHostsListener->new());
    $self->addParameterListener( 'AddHosts', LCM::Configuration::ValueChangeListeners::AddHosts::RemoteCredentialsListener->new());
    $self->addParameterListener( 'HostagentPassword', LCM::Configuration::ValueChangeListeners::AddHostHostagentPasswordListener->new());
    $self->addParameterListener( 'RemoteExecution', LCM::Configuration::ValueChangeListeners::AddHostRemoteExecutionListener->new());
    $self->addParameterListener( 'RootUser', LCM::Configuration::ValueChangeListeners::AddHosts::AddHostRootCredentialsListener->new());
    $self->addParameterListener( 'RootUser', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    $self->addParameterListener( 'RootUser', LCM::Configuration::Hosts::AddHosts::Validators::LSS::InstallationPathCheck->new('RootUser'));
    $self->addParameterListener( 'RootPassword', LCM::Configuration::ValueChangeListeners::AddHosts::AddHostRootCredentialsListener->new());
    $self->addParameterListener( 'RootPassword', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    $self->addParameterListener( 'RootPassword', LCM::Configuration::Hosts::AddHosts::Validators::LSS::OSUserCheck->new('RootPassword'));
    $self->addParameterListener( 'RootPassword', LCM::Configuration::Hosts::AddHosts::Validators::LSS::InstallationPathCheck->new('RootPassword'));
    $self->addParameterListener( 'Password', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    $self->addParameterListener( 'Password', LCM::Configuration::ValueChangeListeners::PostCollectHostInfoChecksListener->new());
    $self->addParameterListener( 'Password', LCM::Configuration::Hosts::AddHosts::Validators::LSS::OSUserCheck->new('Password'));
    $self->addParameterListener( 'HostagentPassword', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    $self->addParameterListener( 'HostagentPassword', LCM::Configuration::Hosts::AddHosts::Validators::LSS::OSUserCheck->new('HostagentPassword'));
    $self->addParameterListener( 'HostagentPassword', LCM::Configuration::Hosts::AddHosts::Validators::LSS::InstallationPathCheck->new('HostagentPassword'));
    $self->addParameterListener( 'Password', LCM::Configuration::ValueChangeListeners::AddHosts::XSUserIsolationListener->new('Password'));
    $self->addParameterListener( 'AddHosts',$self->getAutoInitFamilyServicesParameterHanlder('AddHosts'));
    $self->addParameterListener( 'TenantUser',SDB::Install::Configuration::TenantUserListener->new());
    $self->addParameterListener( 'AddHosts', SDB::Install::Configuration::LSSUserStringListener->new());
    $self->addParameterListener( 'Password', SDB::Install::Configuration::LSSPasswordListener->new());
    $self->addParameterListener('ListenInterface', LCM::Configuration::ValueChangeListeners::ConfigureInternalNetwork::ListenInterfaceRestartListener->new());
    $self->addParameterListener('InternalNetwork', LCM::Configuration::ValueChangeListeners::ConfigureInternalNetwork::InternalNetworkRestartListener->new());
    $self->addXS2Listeners();
}

sub CollectOtherHostInfos {
	my ($self) = @_;

	my $rc = $self->SUPER::CollectOtherHostInfos();
	$self->handleHostagentPassword(1);

	return $rc;
}

sub handleHostagentPassword {
	my ($self,$hanldeRemoteHosts) = @_;

	return if($self->isUseSAPHostagent());

	my $shouldSkipHostagentPassword = $self->shouldSkipHostagentPassword($hanldeRemoteHosts);
	$self->{params}->{HostagentPassword}->{skip} = 1;
	$self->{params}->{InitialHostagentPassword}->{skip} = $shouldSkipHostagentPassword;
}

sub shouldSkipHostagentPassword {
    my ($self,$hanldeRemoteHosts) = @_;
# _TODO_ why not getValue?
    my $installShaBatchValue = $self->getBatchValue('InstallHostagent');
    my $installHostagentValue = defined($installShaBatchValue) ? $installShaBatchValue : $self->getDefault("InstallHostagent");

    return 1 if(!$installHostagentValue);
    return 1 if(!$hanldeRemoteHosts);
    return $self->_sapadmUserExistsOnAllAdditionalHosts();
}

#_TODO_ This doesn't work at all and must be refactored to use the logic from GSC
sub _sapadmUserExistsOnAllAdditionalHosts {
    my ($self) = @_;
    my $additionalRemoteHosts = $self->getAdditionalRemoteHosts();

    return 1 if(!defined($additionalRemoteHosts));
    return 1 if(!exists($self->{remoteSidadms})); # CollectOtherHostInfos is not executed

    my $remoteSapadmUsers = $self->{remoteSidadms}->{sapadm} || [];
    my $remoteHostNames = $additionalRemoteHosts->getHostNames() || [];
    my %remoteSapadmUsersMap = map { $_->[0] => 1 } @{$remoteSapadmUsers};

    for my $additionalHost (@{$remoteHostNames}){
        return 0 if(!$remoteSapadmUsersMap{$additionalHost});
    }
    return 1;
}

#_TODO_ Unused method
sub getRemoteHostsWithNotInstalledSHA {
	my ($self) = @_;

    my @remoteHostsWithNotInstalledSHA = ();
    my $additionalRemoteHosts = $self->getAdditionalRemoteHosts();
	my $remoteHostnames = $additionalRemoteHosts->getHostNames();

	#CollectOtherHostInfos is not executed
	if (not exists $self->{remoteSaphostagents}) {
		return \@remoteHostsWithNotInstalledSHA;
	}

    my $remoteHostAgents = $self->{remoteSaphostagents};

    for my $remoteHostname (@$remoteHostnames) {
        if (not $remoteHostAgents->{$remoteHostname}->{version}) {
            push (@remoteHostsWithNotInstalledSHA, $remoteHostname);
        }
    }

    return \@remoteHostsWithNotInstalledSHA;
}

sub getRemoteHostsObjectsForCollectHostInfo {
    my ($self) = @_;

    my $additionalRemoteHosts = $self->getAdditionalRemoteHosts();
    if(defined $additionalRemoteHosts){
    	return [$additionalRemoteHosts];
    }
    return [];
}

sub getRemoteHostsObjects {
    my ($self) = @_;

    my $remoteHostsAlreadyPartOfLandscape = $self->{_remoteHosts};
    my $additionalRemoteHosts = $self->getAdditionalRemoteHosts();
    my $resultArray = [];

    if(defined $remoteHostsAlreadyPartOfLandscape){
    	push @{$resultArray}, $remoteHostsAlreadyPartOfLandscape;
    }
    if(defined $additionalRemoteHosts){
    	push @{$resultArray}, $additionalRemoteHosts;
    }

    return $resultArray;
}

sub getProductName {
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    $TITLE =~ s/\Q$gFlavourProductName\E/$flavourProductName/g;
	return $TITLE;
}

sub getAction {
	return $HOSTS_ACTION;
}

sub getSection {
	return '';
}

sub CheckParams{
	my ($self, $batchMode) = @_;
	my $rc = $self->SUPER::CheckParams( $batchMode );

	if (!$rc){
		return $rc;
	}

	my $addHostsParam = $self->{params}->{AddHosts};
	my $addHostsOpt = $addHostsParam->{opt};
	my $addHostsBatchValue = $addHostsParam->{batchValue};
	my $isGui = $self->{options}->{isGUI};

	if ( (!$isGui) && (!$batchMode && ((!defined $addHostsBatchValue) || ($addHostsBatchValue eq ""))) ){
		$self->{params}->{AddHosts}->{set_interactive} = 1;
		$self->{params}->{HostsProperties}->{skip} = 0;
		$self->{params}->{HostsProperties}->{set_interactive} = 1;
	}

	return $rc;
}

sub getSID {
	my ($self) = @_;
    my $sid = $self->getValue('SID');

    if ( not defined $sid ){
    	$sid= $self->getDefault('SID');
    }

    return $sid;
}

sub isOneAdditionalRemoteHostForAdding{
	my ($self) = @_;

	my $remoteHosts = $self->getSelectedRemoteHosts();

	if (defined $remoteHosts) {
		my $remoteHostsMap = ParseHostStringToMap($remoteHosts);
		if (scalar (keys %$remoteHostsMap) eq 1){
			return 1;
		}
	}

	return 0;
}

sub getSelectedRemoteHosts {
	my ($self) = @_;

	my $addHostsValue = $self->getValue("AddHosts");
	my ($localHost, $remoteHosts) = ExtractLocalAndRemoteHosts($addHostsValue, $self);

	return $remoteHosts;
}

sub checkPassword {
	my ($self, $password) = @_;

	if (not defined $password){
		$self->appendErrorMessage ("Password is not defined.");
		return 0;
	} elsif (length($password) == 0) {
		$self->appendErrorMessage ("Password cannot be empty.");
		return 0;
	}

	my $rc = $self->SUPER::checkPassword($password);

	if ($rc) {
		my $value = $self->getValue('AddHosts');
		# Fill the values in the CertificatesHostmap parameter
	    # when all new hostnames are gathered and validated
		$self->_prepareCertificatesHostmap($value);
	}

	return $rc;
}

sub _prepareCertificatesHostmap {
	my ($self, $value) = @_;

   	my $certificatesHostmapParam = $self->{params}->{CertificatesHostmap};

	return if(!defined $certificatesHostmapParam);

	$certificatesHostmapParam->{default_map} = undef;
   	$certificatesHostmapParam->{origin_values} = undef;

	my $parsedNewHosts = ParseHostStringToMap($value);
   	$certificatesHostmapParam->{skip} = 0;

	if( defined $parsedNewHosts ) {
		my @newHosts = keys %$parsedNewHosts;
   		$certificatesHostmapParam->{origin_values} = \@newHosts;
	}
	$self->fillCertificatesHostmapDefaultValues();
}

sub checkRole {
    my ($self, $role, $host) = @_;
	if ( ! $self->SUPER::checkRole( $role ) ) {
		return undef;
	}

    if($role =~ /($gHostRoleWorker|$gHostRoleStandby)/){
        return if(!$self->canAddHanaHost($host));
    }

	return ( $role eq $gHostRoleEsWorker && ! $self->canAddEsWorkerHost() ) ? undef : 1;
}

sub checkRootUser {
	my ($self, $value) = @_;

	if ( ! $self->SUPER::checkRootUser( $value ) ) {
		return undef;
	}

    if (! $self->checkPathsExistenceOnHosts()) {
        return 0;
    }

	return 1;
}

sub checkRootPassword {
    my ($self, $value) = @_;

    if ( ! $self->SUPER::checkRootPassword( $value ) ) {
        return undef;
    }

    if (! $self->checkPathsExistenceOnHosts()) {
        return 0;
    }

    return 1;
}

sub checkAddHosts {
    my ($self, $value) = @_;

    if ( ! $self->SUPER::checkAddHosts( $value ) ) {
        return undef;
    }

    if (! $self->checkPathsExistenceOnHosts($value)) {
        return 0;
    }

    return 1;
}

sub isHLMInstalled {
	my ($self) = @_;
	my $target = $self->getValue( "Target" );
	my $sid = $self->getValue( "SID" );
	my $hlmManifestPath = File::Spec->catfile($target, $sid, 'HLM', 'manifest');
	return (-e $hlmManifestPath);
}

sub getSystemComponentManager {
	my ($self) = @_;
	my $scm;
	if (IsHostPartOfSystem()) {
		my $installer = new LCM::Installer();
		$installer->setMsgLstContext([$self->getMsgLst()]);
		$scm = $installer->getOwnSystemComponentManager();
		if (!defined $scm){
			$self->setErrorMessage("Cannot determine the installed '$gProductNameEngine'. Start the tool from its original location inside the database.");
			return undef;
		}
	} else {
		my $target = $self->getValue("Target");
		my $sid = $self->getValue("SID");
		$scm = new LCM::ComponentManager::SystemComponentManager();
		$scm->detectComponentsBySapmntSid($target,$sid);
		if (!$scm->isComponentAvailable($gKeynameEngine)){
			$self->setErrorMessage("Cannot determine the installed '$gProductNameEngine'. Start the tool from its original location inside the database.");
			return undef;
		}
	}
	return $scm;
}

sub _shouldSkipAutoAddXS2Roles {
	my ($self) = @_;

	return 1 if(! $self->isSpecialComponentInstalled($gHostRoleXS2Worker));

	my @hostsForAddition = split(',', $self->getValue('AddHosts'));
	my $isAddingDbHost = grep { $_ =~ /:role=($gHostRoleWorker|$gHostRoleStandby)/} @hostsForAddition;
    my $isAddingXsHost = grep { $_ =~ /:role=($gHostRoleXS2Worker|$gHostRoleXS2Standby)/} @hostsForAddition;

	return 1 if(! $isAddingDbHost || $isAddingXsHost);

	my $rolesInfoMap = $self->getOwnInstance()->getHostRolesInfo();
	for my $host (keys(%{$rolesInfoMap})){
		my @roles = split('\s+', $rolesInfoMap->{$host});
		my $hasDbRole = grep { $_ =~ /$gHostRoleWorker|$gHostRoleStandby/ } @roles;
		my $hasXsRole = grep { $_ =~ /$gHostRoleXS2Worker|$gHostRoleXS2Standby/ } @roles;

		return 0 if($hasXsRole && $hasDbRole);
	}
	return 1;
}

sub _shouldSkipImportContentXS2{
    my ($self) = @_;
    return 1 unless $self->isSpecialComponentInstalled($gHostRoleXS2Worker);

    my @hostsForAddition = split(',', $self->getValue('AddHosts'));
    my $isAddingXsWorkerHost = grep { $_ =~ /:role=($gHostRoleXS2Worker)/} @hostsForAddition;
    return 1 unless $isAddingXsWorkerHost;

    my $rolesInfoMap = $self->getOwnInstance()->getHostRolesInfo();
    for my $host (keys(%{$rolesInfoMap})){
        my @roles = split('\s+', $rolesInfoMap->{$host});
        my $hasXsWorkerRole = grep { $_ =~ /$gHostRoleXS2Worker/ } @roles;
        return 1 if $hasXsWorkerRole ;
    }
    return 0;
}

sub isHostagentInstalled {
	my ($self) = @_;
	return $self->{_shaHelper}->isHostagentInstalled();
}

sub setAddHosts {
	my ($self, $value) = @_;
	my $rc = $self->SUPER::setAddHosts($value);
	if (!$rc) {
		return 0;
	}

	my $parser = new LCM::HostsParser();
	my ($addHostsCmd, $parserErrMsgLst) = $parser->appendDefaultValuesToAddHostsCmd($value);

	if (!defined $addHostsCmd) {
		$self->PushError("Error parsing string: $value\n", $parserErrMsgLst) if defined $parserErrMsgLst;
		return 0;
	}

	$self->{params}->{AddHosts}->{value} = $addHostsCmd;

	my $shouldSkipAutoAdd = $self->_shouldSkipAutoAddXS2Roles();
	my $shouldSkipImportContent = $self->_shouldSkipImportContentXS2();
	$self->setSkip('AutoAddXS2Roles', $shouldSkipAutoAdd);
    $self->setSkip('ImportContentXS2', $shouldSkipImportContent);
	return $rc;
}

sub setHostagentPassword {
	my $self = shift();
	my ($value) = @_;
	my $rc = $self->SUPER::setHostagentPassword(@_);
    #check data/log paths after the CollectOtherHostInfo
    if ($self->isUseSAPHostagent() && (! $self->checkPathsExistenceOnHosts())) {
        return 0;
    }
	if ($rc) {
		$self->{params}->{InitialHostagentPassword}->{value} = $value;
	}
	return $rc;
}

sub setInitialHostagentPassword {
	my ($self, $value) = @_;
	return $self->setHostagentPassword($value);
}

sub isCollectOtherHostInfosRequired {
	my $self = shift();
    return $self->{_collectOtherHostInfo};
}

sub setCollectOtherHostInfosRequired {
    my ($self,$collectOtherHostInfo) = @_;
    $self->{_collectOtherHostInfo} = $collectOtherHostInfo;
}

sub getAutoAssignXs2RolesMap {
	my ($self) = @_;
	my $additionalHostsString = $self->isSkipped('AddHosts') ? '' : $self->getValue('AddHosts');
	my $allHostString = [ split(',', $additionalHostsString) ];
	my $resultMap = {};

	for my $hostString (@{$allHostString}) {
		if($hostString =~ /^([^:]+)/) {
			$resultMap->{$1} = $self->_determineApplicableXs2Role($hostString);
		}
	}
	return $resultMap;
}

sub createSummaryTree {
    my ($self) = @_;
    return LCM::SummaryTreeBuilders::AddHostsSummaryTreeBuilder::buildSummaryTree ($self);
}

sub _determineApplicableXs2Role {
	my ($self, $hostString) = @_;

	return '' if (!$hostString);
    return '' if($hostString =~ /:role=($gHostRoleXS2Standby|$gHostRoleXS2Worker)/);
    return $gHostRoleXS2Worker if($hostString =~ /:role=$gHostRoleWorker/ || $hostString !~ /:role/);
    return $gHostRoleXS2Standby if($hostString =~ /:role=$gHostRoleStandby/);
    return '';
}

sub willInstallOrUpdateSHA {
    my ($self) = @_;
    my $installSHAValue = $self->getValue('InstallHostagent');
    return (!defined $installSHAValue || ($installSHAValue =~ /$bool_true_pattern/)) ? 1 : 0;
}

sub isAutoInitializeServicesApplicableParameterCallback {
    my $self = shift;
    return $self->isAddHostsValueApplicableForAutoInitialize(@_);
}

sub getAdditionalHanaHosts {
    my ($self) = @_;
    my @dbHosts = ();
    foreach my $role ($gHostRoleWorker, $gHostRoleStandby){
        push @dbHosts, @{$self->getAddHostsWithRole($role)};
    }
    return \@dbHosts;
}

sub setInteractiveAddHostInput {
    my($self, $interactiveInput) = @_;
    $self->{interactiveInput} = $interactiveInput;
}

sub isInteractiveAddHostInput {
    my($self) = @_;
    return $self->{interactiveInput};
}

#-------------------------------------------------------------------------------
#Verifies interactive AddHost 'Role' input for the DB hosts
#Lss UID,GID, Installation path are collected from remote hosts during CollectOtherInfo, which is executed after 'RootUser'
#   - interactive mode the role is set after 'RootUser'
#   - non-interactive mode the role is set before 'RootUser => therefore this check is skipped, the role check will be executed via listener

sub canAddHanaHost {
    my($self, $hostname ) = @_;

    if($self->isInteractiveAddHostInput()){
        my $lssValidations = [LCM::Configuration::Hosts::AddHosts::Validators::LSS::InstallationPathCheck->new(), LCM::Configuration::Hosts::AddHosts::Validators::LSS::OSUserCheck->new()];
        foreach my $lssValidation(@$lssValidations){
            $lssValidation->setMsgLstContext($self->getMsgLstContext());
            return 0 if(!$lssValidation->validateInteractiveDbRole($self,$hostname));
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
#override handleAddOrRemoveHosts so that the default role value is used only in non-interactive mode
#Lss UID,GID,Installation path are collected from remote hosts during CollectOtherInfo, which is executed after 'RootUser'
#   - interactive mode the role is set after 'RootUser' => the lss user and path validation is done in checkRole
#   - non-interactive mode the role is set before 'RootUser' => the lss user and path validation is done as listeners on 'RootUser',etc params

sub handleAddOrRemoveHosts {
    my ($self,
        $specifiedHosts,      # e.g. --addhosts=<specifiedHostList>
        $paramID,             # valid values: 'AddHosts', 'RemoveHosts', 'AddRoles', 'RemoveRoles'
        $doNotFailOnLocalHost, # 1 - the list may contain a local host
    ) = @_;
    my $doNotUseDefaultRoleValue = $self->isInteractiveAddHostInput();
    return $self->SUPER::handleAddOrRemoveHosts($specifiedHosts, $paramID, $doNotFailOnLocalHost, $doNotUseDefaultRoleValue);
}

sub getLssInstance {
    my ($self) = @_;
    if(!defined $self->{lssInstance}){
        my $instance = $self->getOwnInstance();
        return undef if(!$instance);
        $instance->setMsgLstContext($self->getMsgLstContext());
        my $noCheckInstalled = 1;
        $self->{lssInstance} = $instance->getLssInstance($noCheckInstalled);
        $self->{lssInstance}->setMsgLstContext($self->getMsgLstContext());
        my $skipLssLinkTarget = 1;
        if(!$self->{lssInstance}->isInstalled($skipLssLinkTarget)){
            $self->getMsgLst()->addMessage("No $gShortProductNameLSS detected on the system");
            $self->{lssInstance} = undef;
        } else {
            $self->getMsgLst()->addMessage("Detected $gShortProductNameLSS on the system");
        }
    }
    return $self->{lssInstance};
}

sub checkPathExistenceOnHost {
    my ($self, $pathExistenceMap, $path, $host) = @_;
    return 0 if(!$self->SUPER::checkPathExistenceOnHost($pathExistenceMap, $path, $host));
    if( $self->usesNonSharedStorage() ) {
        return 0 if (!$self->checkNonSharedStoragePath($pathExistenceMap, $path,[$host]));
    }
    return 1;
}

sub usesStorageApi {
    my ($self) = @_;
    my $instance = $self->getOwnInstance();
    return 1 if ($instance && $instance->usesStorageApi());
    return 0;
}

1;
