package LCM::Configuration::LSS::LSSConfiguration;

use strict;
use SDB::Install::SysVars qw($isWin);
use SDB::Install::Globals qw(
    $gShortProductNameLSS
    $gHostRoleWorker
    $gHostRoleStandby
    $gKeynameLSS
    $gProductName
    $gOperationCheckLssPath
    $gKeynameEngine
    $gSapsysGroupName
);
use SDB::Common::Utils qw(getSidcryptName checkOsUserID createUsername createGroupname checkOsGroupID);
use SDB::Install::LSS::LssSidcryptUser;
use SAPDB::Install::Hostname;
use File::Basename qw(dirname);
use SDB::Common::BuiltIn;
use SDB::Install::Tools qw(arrayToHash);

use parent qw(SDB::Install::Configuration::LSSConfiguration LCM::Configuration::GenericStackAny);

sub defineLSSParams {
    my ($self) = @_;
    my $order = scalar keys %{ $self->{params} };
    my $section = 'lss';
    my $lssParams = {
        'LSSInstPath'    => $self->getParamLSSInstPath( $order++, $section ),
        'LSSPassword'    => $self->getParamLSSPassword( $order++, $section),
        'LSSUserID'      => $self->getParamLSSUserID( $order++, $section),
        'LSSGroupID'     => $self->getParamLSSGroupID( $order++, $section ),
        'LSSUserHomeDir' => $self->getParamLSSUserHomeDir( $order++, $section, '/usr/sap/$SID/lss/home' ),
        'LSSUserShell'   => $self->getParamLSSUserShell( $order++, $section ),
        'LSSBackupPassword' => $self->getParamLSSBackupPassword( $order++, $section ),
    };
    map { $self->{params}->{$_} = $lssParams->{$_} } keys %$lssParams;
    return 1;
}

sub enableLSSParams {
    my ($self) = @_;
    my $shouldEnable = $self->isInstallingLSS();

    foreach my $paramId (@{$self->getLSSParams()}) {
        $self->setSkip($paramId, !$shouldEnable);
    }
}

sub initLSSUserID {
    my ($self) = @_;
    my $freeUID = $self->getFreeOsUserID(undef, $self->getReservedUserIDs());

    if (!defined $self->getBatchValue('LSSUserID')) {
        $self->setDefault('LSSUserID', $freeUID);
    }
    return 1;
}

sub initLSSGroupID {
    my ($self) = @_;
    my $freeGID = $self->getFreeGroupID(undef, $self->getReservedGroupIDs());

    if (!defined $self->getBatchValue('LSSGroupID')) {
        $self->setDefault('LSSGroupID', $freeGID);
    }
    return 1;
}

sub checkLSSUserID {
    my ($self, $value) = @_;
    my $userName = getSidcryptName($self->getTargetLssSid());
    return 0 if(!checkOsUserID($self, 'LSSUserID' ,$userName, $value, 0, 0, $self->getRemoteDBHosts()));
    return $self->checkReservedOSUserIDs($value, 'LSSUserID');
}

sub checkLSSGroupID {
    my ($self, $value) = @_;
    return 0 if(!$self->checkLSSGroupName($value));

    my $groupName = $self->getSidcryptGroupName();
    if($groupName eq $gSapsysGroupName){
        my $userName = getSidcryptName($self->getTargetLssSid());
        $self->appendErrorMessage("Primary group of user '$userName' cannot be '$gSapsysGroupName'");
        return 0;
    }
    return 0 if (!checkOsGroupID($self, 'LSSGroupID', $groupName, $value, 0, $self->getRemoteDBHosts()));
    return $self->checkReservedOSGroupIDs($value, 'LSSGroupID');
}

sub getExistingLssUserId {
    my ($self) = @_;
    my $userName = getSidcryptName($self->getTargetLssSid());
    return $self->getExistingUserId($userName);
}

sub checkLSSGroupName {
    my ($self, $groupId) = @_;
    my $userId = $self->getExistingLssUserId();
    if(defined $userId){
        return $self->checkExistingSidcryptUserGroupName($userId);
    }

    if($self->isDistributedSystem()){
        return $self->checkLSSGroupIDOnRemoteHosts($groupId);
    }

    my $localGroupName = SDB::Common::BuiltIn->get_instance()->getgrgid($groupId);
    if(defined($localGroupName)){
        $self->setSidcryptGroupName($localGroupName);
    }

    return 1;
}

sub checkLSSGroupIDOnRemoteHosts {
    my ($self, $groupId) = @_;
    return 1 if(!defined $self->{remoteGids});

    my @dbHosts = @{$self->getAllDatabaseHosts()};
    my $remoteGroups = $self->{remoteGids}->{$groupId} // [];
    my @existingGroups = ();
    foreach my $host (@dbHosts) {
        push @existingGroups, map($_->[1], grep{ $_->[0] eq $host} @{$remoteGroups});
    }
    my $localGroupName = SDB::Common::BuiltIn->get_instance()->getgrgid($groupId);
    my @existingGroupsInfo = @{$remoteGroups};

    my $localHost = $self->getLocalHanaHost();
    my $isLocalDbHost = (grep{ $_ eq $localHost} @dbHosts);
    if($isLocalDbHost && defined($localGroupName)){
        push @existingGroups, $localGroupName;
        push @existingGroupsInfo, [$localHost, $localGroupName];
    }

    my $numberOfExistingGroups = scalar(@existingGroups);
    return 1 if($numberOfExistingGroups == 0);

    my $namesOfExistingGroups = arrayToHash(\@existingGroups);
    if(scalar(keys %{$namesOfExistingGroups}) > 1){
        $self->appendErrorMessage("Name of the group with ID '$groupId' is not the same on all hosts with $gHostRoleWorker and $gHostRoleStandby roles");
        return 0;
    }

    if(scalar(@dbHosts) == $numberOfExistingGroups){
        $self->setSidcryptGroupName($existingGroups[0]);
        return 1;
    }

    my $defaultName = getSidcryptName($self->getTargetLssSid());
    if($existingGroups[0] ne $defaultName){
        my @hostsWithMissingGroup = ();
        foreach my $host (@dbHosts) {
            push @hostsWithMissingGroup, $host if(grep { $_->[0] ne $host } @existingGroupsInfo);
        }
        $self->appendErrorMessage("If you want to use group different from '$defaultName', ensure that it exists on all hosts with $gHostRoleWorker and $gHostRoleStandby roles.");
        $self->appendErrorMessage("Group '$existingGroups[0]' with group ID $groupId does not exist on host(s) @hostsWithMissingGroup");
        return 0;
    }
    $self->setSidcryptGroupName($defaultName);
    return 1;

}

sub checkExistingSidcryptUserGroupName {
    my ($self, $userId) = @_;
    my $userName = getSidcryptName($self->getTargetLssSid());
    my $localSidcryptUser = SDB::Install::LSS::LssSidcryptUser->new($userName);

    my ($existingGroupId, $existingGroupName);
    if($localSidcryptUser->exists()){
        $existingGroupId = $localSidcryptUser->gid();
        $existingGroupName = $localSidcryptUser->group();
    } else {
        $existingGroupId = $self->{remoteUids}->{$userId}->[0]->[2];
        $existingGroupName = $self->{remoteGids}->{$existingGroupId}->[0]->[1];
    }

    my $batchValue = $self->getBatchValue('LSSGroupID');
    if(defined $batchValue && $existingGroupId != $batchValue){
        $self->appendErrorMessage("Group ID $batchValue cannot be used. User '$userName' already belongs to group '$existingGroupName' with group ID $existingGroupId");
        return 0;
    }
    $self->setSidcryptGroupName($existingGroupName);
    return 1;
}

sub getRemoteDBHosts {
    my ($self) = @_;
    my $dbHosts = $self->getAllDatabaseHosts();
    my @remoteDbHosts = grep {$self->getLocalHanaHost() !~ /$_/} @{$dbHosts};
    return \@remoteDbHosts;
}

sub getAllDatabaseHosts {
    my ($self) = @_;
    my @dbHosts = ();
    foreach my $role ($gHostRoleWorker, $gHostRoleStandby){
        push @dbHosts, @{$self->getSystemHostsWithRole($role)};
        push @dbHosts, @{$self->getAddHostsWithRole($role)};
    }
    my $server = $self->getComponentManager()->getComponentByKeyName($gKeynameEngine);
    if(defined $server && $server->isComponentSelected() && !$server->isUpdate()){
        push @dbHosts, $self->getLocalHanaHost();
    }
    return \@dbHosts;
}

sub checkLSSUserShell {
    my ($self, $shell) = @_;
    return $self->checkShell($shell);
}

sub isLSSInstalled {
    my ($self) = @_;
    my $lssInstance = SDB::Install::LSS::LssInstance->new(undef, $self->getTargetLssSid());
    return $lssInstance->isConfiguredOnHost();
}

sub handleLssUser {
    my ($self) = @_;
    my $targetSid = $self->getTargetLssSid();
    $self->setDefault('LSSUserHomeDir', "/usr/sap/$targetSid/lss/home");
    my $existingId = $self->getExistingLssUserId();
    my $rc;
    if(defined($existingId)){
        $rc = $self->handleExistingOsID($existingId, 'LSSUserID');
        $rc &&= $self->setExistingLSSGroupID($existingId);
    } else {
        $rc = $self->initLSSUserID();
    }
    return $rc;
}

sub setExistingLSSGroupID {
    my ($self, $sidcrypUID) = @_;
    my $sidcryptName = getSidcryptName($self->getTargetLssSid());
    my $sidcryptUser = SDB::Install::LSS::LssSidcryptUser->new($sidcryptName);
    my $gid = $sidcryptUser->exists() ? $sidcryptUser->gid() : $self->{remoteUids}->{$sidcrypUID}->[0]->[2];

    my $rc = $self->setValue('LSSGroupID', $gid);
    $self->setHidden('LSSGroupID', 1) if($rc);
    return $rc;
}

sub handleLssGroup {
    my ($self) = @_;
    my $batchValue = $self->getBatchValue('LSSGroupID');
    return 0 if(defined $batchValue && !$self->checkLSSGroupName($batchValue));

    my $groupName = $self->getSidcryptGroupName();
    my $existingId = $self->getExistingGroupId($groupName);
    if(defined($existingId)){
        return $self->handleExistingOsID($existingId, 'LSSGroupID');
    } else {
        return $self->initLSSGroupID();
    }
}

sub getSidcryptGroupName {
    my ($self) = @_;
    my $defaultName = getSidcryptName($self->getTargetLssSid());
    return $self->{sidcryptGroupName} // $defaultName;
}

sub setSidcryptGroupName {
    my ($self, $groupName) = @_;
    $self->{sidcryptGroupName} = $groupName;
}

sub getLSSParams {
    return [ 'LSSInstPath', 'LSSPassword', 'LSSUserID', 'LSSGroupID',
        'LSSUserHomeDir', 'LSSUserShell', 'LSSBackupPassword'];
}

sub isInstallingLSS {
    my ($self) = @_;
    my $isLssSelected = $self->isComponentSelected($gKeynameLSS);
    my $isLssUpdate   = $self->isUpdatingComponent($gKeynameLSS);
    return $isLssSelected && !$isLssUpdate;
}

sub initLSSGroupAndUserID {
    my ($self) = @_;
    return 1 if !$self->isInstallingLSS();

    my $rc = $self->handleLssUser();
    if(!defined $self->getExistingLssUserId()){
        $rc &&= $self->handleLssGroup();
    }
    return $rc;
}

sub setLSSPasswordType {
    my ($self) = @_;
    my $dbHostsWithSidcryptUser = $self->getDbHostsWithSidcryptUser();
    my $passwordType = @$dbHostsWithSidcryptUser ? 'passwd' : 'initial_passwd';
    $self->setType('LSSPassword', $passwordType);
}

sub isLSSPasswordRequired {
    my ($self) = @_;
    return $self->isComponentSelected($gKeynameLSS) && !$self->isUpdatingComponent($gKeynameLSS);
}

sub checkLSSInstPath {
    my ($self, $lssInstPath) = @_;
    my $sapmntDir = $self->getValue('Target');
    if(defined($sapmntDir) && ($lssInstPath =~ /^$sapmntDir/)) {
        $self->appendErrorMessage("$gShortProductNameLSS installation path cannot be the same as or under the $gProductName installation directory");
        return 0;
    }
    my @remoteDbHosts = @{$self->getRemoteDBHosts()};
    return 1 if(! scalar(@remoteDbHosts) > 0 );

    if($self->getValue('Scope') eq 'instance'){
        $self->getMsgLst()->addMessage("$gShortProductNameLSS installation path cannot be checked on remote hosts, because scope parameter is set to instance.");
        $self->getMsgLst()->addMessage("Ensure that $lssInstPath is accessible on ".join(', ', @remoteDbHosts));
        return 1;
    }

    if(!File::stat::stat($lssInstPath)){
        $self->setErrorMessage ("Directory '$lssInstPath' doesn't exist");
        return undef;
    }

    my $errorList = SDB::Install::MsgLst->new();
    if(!$self->checkLssInstPathOnRemoteHosts($lssInstPath, $errorList)){
        my $sidMountPoint = File::Spec->catdir($lssInstPath, $self->getTargetLssSid());
        if(!$self->checkLssInstPathOnRemoteHosts($sidMountPoint)){
# We need to add only the messages from the check of $lssInstPath as this is what the user
# has really provided. The errors from the $sidMountPoint shouldn't be visualized.
            $self->setErrorMessage(undef, $errorList);
            return undef;
        };
    }
    return 1;
}

sub checkLssInstPathOnRemoteHosts {
    my($self, $path, $errorList) = @_;
    my $success = 1;
    my $remoteDbHostNames = join(',',@{$self->getRemoteDBHosts()});
    my $output = undef;
    $errorList //= SDB::Install::MsgLst->new();

    if ($self->getComponentManager()->isSidAdmUserExecution() || $self->SDB::Install::Configuration::AnyConfig::isUseSAPHostagent()) {
        my $sidadmExecution = new SDB::Install::RemoteHostctrlHosts(hostname());
        $sidadmExecution->setMsgLstContext($self->getMsgLstContext());

        my $rc = $sidadmExecution->executeHostctrlParallel( $gOperationCheckLssPath,
                                    $self, # instconfig
                                    undef, # param IDs
                                    ['Password'], # password IDs
                                    undef, # remote ignore
                                    undef, # remote timeout
                                    undef, #progress mssage
                                    "$gShortProductNameLSS installation path checked", #done message
                                    undef, #error msg
                                    {   'SAPMNT' => $self->getSAPSystem()->get_target(),
                                        'SID'  => $self->getTargetLssSid(),
                                        'PATH' => $path,
                                        'HOSTNAMES' => $remoteDbHostNames,
                                    },
                                    undef, # host option map
                                    undef, # only on hosts
                                    undef, # do not fail
                                    1, # suppresses console output
                                    0); # isCollectHostInfo

        if( $rc != 0 ){
            $success = 0;
        }

        $output = $sidadmExecution->getOutputBuffer();
    } else {
        my $rootUser = $self->getValue('RootUser');
        my $sid = $self->getTargetLssSid();
        my @args = (
            '-main',
            'SDB::Install::App::Console::LSS::CheckInstallationPath::main',
            "--path=$path",
            "--hostnames=$remoteDbHostNames",
            $rootUser ? "--root_user=$rootUser" : '',
            "--sid=$sid",
        );
        my $app = LCM::App::ApplicationContext::getInstance()->getApplication();
        my $installer = new LCM::Installer();
        my $runtimeDirectory = $installer->getRuntimeDir();
        my $cmd = File::Spec->catfile(dirname($runtimeDirectory), $app->getProgramName());

        my $stdinLines = $self->getXmlPasswordStream(['RootPassword']);
        my $processExecutor = LCM::ProcessExecutor->new($cmd,\@args, $stdinLines);
        my $exitCode = $processExecutor->executeProgram();
        $self->getMsgLst ()->addMessage(undef, $processExecutor->getMsgLst());

        if (!defined $exitCode || $exitCode){
            $success = 0;
        }
        $output = $processExecutor->getOutputLines();
    }
    if(!$success){
        chomp @{$output};
        $errorList->addError($_) for @{$output};
    }
    return $success;
}

1;