package LCM::Configuration::RemoveHostRolesConfiguration;
use parent qw (SDB::Install::Configuration::AnyMultiHostConfig LCM::Configuration::AutoInitializeFamilyServicesConfiguration);

use strict;
use experimental qw (smartmatch);

use LCM::App::ApplicationContext;
use LCM::App::RemoveHostRolesUtils qw(readInteractiveRemoveHostRolesInput getRemovableRolesForExistingHosts);
use LCM::Configuration::ValueChangeListeners::FillingRolesHostmapListener;
use LCM::Configuration::ValueChangeListeners::RemoveHostRoles::AcceleratorCredentialsListener;
use LCM::Configuration::ValueChangeListeners::RemoveHostRoles::RemoteCredentialsListener;
use LCM::Configuration::ValueChangeListeners::RemoveHostRoles::SIDListener;
use SDB::Install::Configuration::TenantUserListener;
use LCM::Configuration::ParametersCreator;
use LCM::HostsParser qw(IsLocalHost);
use LCM::Installer;
use LCM::SummaryTreeBuilders::RemoveHostRolesSummaryTreeBuilder;
use LCM::Utils::CommonUtils qw ( getSidadmName getSystemSidadmUid getLocalHostSidadmUid );
use LCM::Utils::RemoveRoles qw(canRemoveHostRole shouldCheckRoleRemovalStatus);

use SDB::Install::Configuration::AnyMultiHostConfig qw ($validHostRoles);
use SDB::Install::Globals qw ($gHostRoleAcceleratorWorker $mComponentsRoles $gHostRoleXS2Standby $gHostRoleXS2Worker $gShortProductNameXS2 $gXSParametersRemoveHostOrRolesConstraint);
use SDB::Install::System qw(isSidadmin $hostname_regex $ipv4_regex);
use SDB::Install::SysVars qw ($isWin);
use SDB::Install::SAPInstance::TrexInstance;

sub new {
    my $class = shift();
    my $self = $class->SUPER::new(@_);
    my $installer = new LCM::Installer();
    $self->{systemComponentManager} = $installer->getOwnSystemComponentManager();
    if (!defined($self->{systemComponentManager})) {
        return 0;
    }
    $self->_defineAddHostRolesParams();
    $self->_addListeners();
    $self->changePasswordStr(lc($installer->getSid()) . "adm");
    return $self;
}

# Override
sub getLocalHanaHost {
    my $self = shift;
    my $instance = $self->getOwnInstance();
    my $hostname = $instance->get_host() if ( defined $instance);
    $hostname    = lc(hostname())        if (!defined $hostname);
    return $hostname;
}

# Override
sub handleHostsPropertiesSkipValue { return; }

# Override
sub getIgnoreValues { return undef; }

# Override
sub _handleInitializationOfPendingConfiguration { return 1; }

sub _defineAddHostRolesParams {
    my ($self) = @_;
    my $order = 0;
    my $section = "RemoveHostRoles";

    my $acceleratorConstraint = "Valid with $gHostRoleAcceleratorWorker/standby only";
    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();
    $self->{params} = {
        'Target' => $self->getParamTarget($order++, $section),
        'NoStart' => $self->getParamNoStart($order++, $section, "Do not start hosts with modified roles", "Does not start hosts with modified roles", 0),
        (
            !$isWin
            ? (
                 'RemoteExecution'  => $self->getParamRemoteExecution($order++, $section, 0),
                 'UseHttp'          => $self->getParamUseHttp($order++, $section),
                 'SAPControlUseHttp'=> $self->getParamSAPControlUseHttp($order++, $section),
                 'InstallSSHKey'    => $self->getParamInstallSSHKey($order++, $section),
              )
            : ()
        ),
        'SID' => $self->getParamSID($order++, $section),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $section,undef,$tenantCredentialsConstraint),
        (
            !$isWin
            ? (
                 'RemoveRoles'      => LCM::Configuration::ParametersCreator::getParamRemoveHostRoles($self, $order++, $section),
                 'RootUser'         => $self->getParamRootUser($order++, $section),
                 'RootPassword'     => $self->getParamRootPassword ($order++, $section),
              )
            : ()
        ),
        'Password'            => $self->getParamPassword($order++, $section),
        'SystemUser'          => $self->getParamSystemUser($order++, $section),
        'SQLSysPasswd'        => $self->getParamSQLSysPasswd($order++, $section, 'passwd'),
        'AcceleratorUser'     => $self->getParamAcceleratorUser($order++, $section, $acceleratorConstraint),
        'AcceleratorPassword' => $self->getParamAcceleratorPassword($order++, $section, $acceleratorConstraint, 'passwd'),
        'KeepXsUsers'         => $self->getParamKeepXsUsers($order++, $section, $gXSParametersRemoveHostOrRolesConstraint),
        (
            !$isWin
            ? (
                 'SkipModifySudoers'   => $self->getParamSkipModifySudoers($order++, $section)
              )
            : ()
        ),
        'TenantUser'     => $self->getParamTenantUser    ($order++, $section,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++, $section,1,$tenantCredentialsConstraint),
    };
    $self->setMandatory('RemoveRoles');
# SystemUser is needed only if removing ets hosts
    $self->setSkip('SystemUser');
    $self->setSkip('SQLSysPasswd');
    $self->{params}->{Password}->{str} = sprintf($self->{params}->{Password}->{str_templ}, $self->{sidadmUser});
    $self->{params}->{RemoveRoles}->{custom_input} = \&LCM::App::RemoveHostRolesUtils::readInteractiveRemoveHostRolesInput;
}

sub InitDefaults {
    my $self = shift();

    return undef if(!$self->SUPER::InitDefaults(@_));

    my $fillHostRolesListener = new LCM::Configuration::ValueChangeListeners::FillingRolesHostmapListener();
    $fillHostRolesListener->onValueChange(undef, $self);

    my $hostToRemovableRolesMap = getRemovableRolesForExistingHosts($self);
    my $canRemoveRoles = grep { length($hostToRemovableRolesMap->{$_}) > 1 } keys(%{$hostToRemovableRolesMap});
    if(!$canRemoveRoles){
        my $instance = $self->getOwnInstance();
        my $hostRoleInfo = join(', ', @{$instance->getAllHostsWithRoles(1)});
        my $hosts = $instance->get_allhosts();
        my $errorMessage = "There aren't any hosts with roles which can be removed.";
        $errorMessage .= "\nCurrent host setup: $hostRoleInfo";
        $errorMessage .= "\n\nThe Remove Host Roles scenario is only valid for hosts with more than one assigned role.";
        if (scalar(@$hosts) > 1) {
            $errorMessage .= "\nConsider using the Remove Hosts scenario for multiple host systems.";
        }
        $self->getErrMsgLst()->addError($errorMessage);
        return undef;
    }
    return 1;
}

sub CheckParams{
    my ($self, $batchMode) = @_;
    my $rc = $self->SUPER::CheckParams($batchMode);
    return $rc if (!$rc);

    my $removeRolesBatchValue = $self->{params}->{RemoveRoles}->{batchValue};
    my $isGui = $self->{options}->{isGUI};

    if ( (!$isGui) && (!$batchMode && ((!defined $removeRolesBatchValue) || ($removeRolesBatchValue eq ""))) ){
        $self->{params}->{RemoveRoles}->{set_interactive} = 1;
    }
    return $rc;
}

# SID listener is notified after setSID subroutine, SID parameter must be set.
# This is from GenericStackUpdateConfiguration
# Consider future refactoring
sub setSID {
    my ($self, $value) = @_;
    my $sidParamStr = $self->getString('SID');
    my $rc = $self->checkExistingSID($value, $sidParamStr);

    return $rc if (!$rc);

    $self->{params}->{SID}->{value} = $value;
    $self->{current_sid} = $value;

    return 1;
}

sub setRemoveRoles {
    my ($self, $hostname, $roles) = @_;

    if (!$self->checkRemoveRoles($hostname, $roles)) {
        return 0;
    }
    $self->{params}->{RemoveRoles}->{value}->{ lc($hostname) } = $roles;
    return 1;
}

sub checkRemoveRoles {
    my ($self, $host, $roles) = @_;
    if (!defined $roles || $roles eq "") {
        $self->appendErrorMessage ("There is no specified role to remove from host '$host'.");
        return 0;
    }
    if (! $host ~~ @{$self->getParamOriginValues('RemoveRoles')} ) {
        $self->appendErrorMessage ("Host '$host' is not part of the system.");
        return 0;
    }
    my $instance = $self->getOwnInstance();
    my $currentHostRoles = $instance->getHostRolesByIniFile($host);
    if (!defined $currentHostRoles) {
        $self->appendErrorMessage ("Could not get current host roles for host '$host'.");
        return 0;
    }
    if (scalar(@$currentHostRoles) == 1) {
        $self->appendErrorMessage ("Host '$host' only has a single role and it cannot be removed.");
        return 0;
    }
    my $rc = 1;
    # Filter out the xs roles, they will be checked later in checkEntriesRemoveRoles
    my @removeRolesArray = grep { $_ !~ /($gHostRoleXS2Worker|$gHostRoleXS2Standby)/ } split(',', $roles);
    for my $role (@removeRolesArray) {
        if (!($role ~~ @$currentHostRoles)){
            $self->appendErrorMessage("Host role '$role' is not assigned to host '$host'.");
            $rc = 0;
            next;
        }
        my $roleExists = (defined $validHostRoles->{$role}) ? 1 : 0;
        my $isRoleRemovable = ($validHostRoles->{$role}->{removable}) ? 1 : 0;
        if (! $roleExists || ! $isRoleRemovable) {
            $self->appendErrorMessage("Remove host role '$role' not supported");
            $rc = 0;
            next;
        }

        if ( shouldCheckRoleRemovalStatus($self, $role) && !canRemoveHostRole($self, $host, $role)) {
            $rc = 0;
            next;
        }
    }

    if (scalar(@$currentHostRoles) == scalar(@removeRolesArray)) {
        $self->appendErrorMessage("Host '$host' must be left with at least one role");
        $rc = 0;
    }
    return $rc;
}

# Prevents removing the xscontroller's xs_worker role
# while other hosts with XS roles exist on the system
sub checkEntriesRemoveRoles {
    my ($self) = @_;
    my $xsControllerHost = $self->getOwnInstance()->getXsControllerHostname();
    return 1 if (!defined $xsControllerHost);

    my $removeRoles = $self->getValue('RemoveRoles');
    my $xsControllerRemoveRoles = $removeRoles->{$xsControllerHost};
    my $removedRole = $1 if ($xsControllerRemoveRoles =~ /($gHostRoleXS2Worker|$gHostRoleXS2Standby)/);
    if (defined $removedRole && $self->willHaveAnyRemainingXsRoles()) {
        my $errMsg = "Cannot remove '$removedRole' role from host '$xsControllerHost' because the $gShortProductNameXS2 Controller is running on this host. " .
                     "You must remove all other $gShortProductNameXS2 roles from the remaining hosts first.";
        $self->appendErrorMessage($errMsg);
        return 0;
    }
    return 1;
}

sub getSystemComponentManager {
    return $_[0]->{systemComponentManager};
}

# Mimics implementation of the same method from
# AddHostRoles/GenericStackUnisntallation/IntrernalNetworkConfiguration
# Consider future refactoring
sub checkRootUser {
    my ($self, $remoteRootUser) = @_;

    if(defined $self->{_remoteHosts}){
        $self->{_remoteHosts}->destroy();
        $self->{_remoteHosts} = undef;
    }

    $self->setSkip('RootPassword', 1);
    my $returnCode = $self->SUPER::checkRootUser($remoteRootUser);
    return $returnCode;
}

# Basically mimics the InternalNetworkConfiguration::checkPassword method
# Consider future refactoring
sub checkPassword {
    my ($self, $password) = @_;
    return 0 if (!$self->initRemoteHosts());

    if(!defined $password){
        $self->appendErrorMessage ("Password is not defined.");
        return 0;
    }elsif(length($password) == 0){
        $self->appendErrorMessage ("Password value cannot be empty.");
        return 0;
    } else {
        require LCM::Configuration::NewDBUser::User;
        my $sidadmUser = new LCM::Configuration::NewDBUser::User(getSidadmName($self));
        $sidadmUser->setMsgLstContext($self->getMsgLstContext());
        my $rc = $sidadmUser->verifyPassword($password, $self);
        if (!defined $rc){
            $self->getMsgLst()->appendMsgLst ($sidadmUser->getErrMsgLst());
            $self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
            return 1;
        }
        # Fix for bug 72608. Prevent doubled 'Unknown user password combination' message. The other message comes from 'LCM::Configuration::NewDBUser::User;verifyPasswordSHA'.
        if($rc != 1 && !isSidadmin($self->getSID())) {
            $self->appendErrorMessage ('Unknown user password combination');
        }
        return $rc;
    }
}


sub _addListeners {
    my ($self) = @_;
    $self->addParameterListener('SID', LCM::Configuration::ValueChangeListeners::RemoveHostRoles::SIDListener->new());
    $self->addParameterListener('RemoveRoles', LCM::Configuration::ValueChangeListeners::RemoveHostRoles::AcceleratorCredentialsListener->new());
    $self->addParameterListener('RemoveRoles', LCM::Configuration::ValueChangeListeners::RemoveHostRoles::RemoteCredentialsListener->new());
    $self->addParameterListener('RemoveRoles', $self->getAutoInitFamilyServicesParameterHanlder('AddRoles'));
    $self->addParameterListener('TenantUser', SDB::Install::Configuration::TenantUserListener->new());
}

sub getInvalidHostsForSelectionMessage {
    my ($self, $hostToRemoveRolesMap) = @_;
    my @invalidHostsForSelection = grep {! length($hostToRemoveRolesMap->{$_})} keys(%$hostToRemoveRolesMap);
    if (scalar @invalidHostsForSelection) {
        my $invalidhostsString = join(', ', @invalidHostsForSelection);
        return "Note: No removable roles exist for host(s) $invalidhostsString.";
    }
    return undef;
}

sub isRemovingLocalHostRoles {
    my ($self) = @_;
    my $removeRolesValue = $self->getValue('RemoveRoles');
    return undef if (! defined($removeRolesValue) );
    for my $host ( keys(%{$removeRolesValue}) ) {
        return 1 if (IsLocalHost($host));
    }
    return 0;
}

sub isRemovingRemoteHostRoles {
    my ($self) = @_;
    my $removeRolesValue = $self->getValue('RemoveRoles');
    return undef if (! defined($removeRolesValue) );
    my $numberOfHosts = keys(%{$removeRolesValue});
    return 1 if ($numberOfHosts > 1);
    return 1 if ($numberOfHosts == 1 && !$self->isRemovingLocalHostRoles());
    return 0;
}

sub willHaveAnyRemainingXsRoles {
    my ($self) = @_;
    return ($self->willHaveRemainingXsWorkerRoles() || $self->willHaveRemainingXsStandbyRoles());
}

sub willHaveRemainingXsWorkerRoles {
    my ($self) = @_;
    my $ownInstance = $self->getOwnInstance();
    my $numberOfXsWorkerHosts = grep {$_ =~ /$gHostRoleXS2Worker/} values %{$ownInstance->getHostRolesInfo()};
    return 0 if (! $numberOfXsWorkerHosts);

    my $removeRolesValue = $self->getValue('RemoveRoles');
    my $xsWorkerRolesRemoved = grep {$_ =~ /$gHostRoleXS2Worker/} values %$removeRolesValue;
    return ($xsWorkerRolesRemoved == $numberOfXsWorkerHosts) ? 0 : 1;
}

sub willHaveRemainingXsStandbyRoles {
    my ($self) = @_;
    my $ownInstance = $self->getOwnInstance();
    my $numberOfXsStandbyHosts = grep {$_ =~ /$gHostRoleXS2Standby/} values %{$ownInstance->getHostRolesInfo()};
    return 0 if (! $numberOfXsStandbyHosts);

    my $removeRolesValue = $self->getValue('RemoveRoles');
    my $xsStandbyRolesRemoved = grep {$_ =~ /$gHostRoleXS2Standby/} values %$removeRolesValue;
    return ($xsStandbyRolesRemoved == $numberOfXsStandbyHosts) ? 0 : 1;
}

sub shouldRemoveXsControllerHostRolesSeparately {
    my ($self) = @_;
    return 0 if (! $self->existsXsControllerHost());

    my $willHaveRemainingXsRoles     = $self->willHaveAnyRemainingXsRoles();
    my $isLocalHostXsController      = $self->isLocalHostXsController();
    my $numberOfRemoteXsRolesRemoved = $self->getNumberOfRemoteXsRolesRemoved();
    return (!$willHaveRemainingXsRoles && !$isLocalHostXsController && $numberOfRemoteXsRolesRemoved > 1);
}

sub existsXsControllerHost {
    my ($self) = @_;
    my $xsControllerHost = $self->getOwnInstance()->getXsControllerHostname();
    return (defined $xsControllerHost) ? 1 : 0;
}

sub isLocalHostXsController {
    my ($self) = @_;
    return 0 if (! $self->existsXsControllerHost());
    my $xsControllerHost = $self->getOwnInstance()->getXsControllerHostname();
    my $localHost = $self->getOwnInstance()->get_host();
    return ($localHost eq $xsControllerHost);
}

sub getNumberOfRemoteXsRolesRemoved {
    my ($self) = @_;
    my $removeRolesValue = $self->getValue('RemoveRoles');
    my $localHost = $self->getOwnInstance()->get_host();
    my $numberOfRemovedRoles = grep {$_ ne $localHost && $removeRolesValue->{$_} =~ /$gHostRoleXS2Worker|$gHostRoleXS2Standby/} keys %$removeRolesValue;
    return $numberOfRemovedRoles;
}

sub getStaticDialogs {
    return [
        'LCM::Gui::Dialogs::ConfigurationSummaryDialog',
        'LCM::Gui::Dialogs::ConfigurationExecutionDialog',
        'LCM::Gui::Dialogs::FinalizeDialog'
    ];
}

sub getConfigureDialogs {
    my ($self, $wizard) = @_;
    require LCM::Gui::Dialogs::RemoveHostRoles::RemoveHostRolesPropertiesDialog;
    require LCM::Gui::Dialogs::SystemUserPropertiesDialog;
    require LCM::Gui::Dialogs::AddHosts::AcceleratorPropertiesDialog;
    return [ new LCM::Gui::Dialogs::RemoveHostRoles::RemoveHostRolesPropertiesDialog($wizard),
             new LCM::Gui::Dialogs::SystemUserPropertiesDialog($wizard),
             new LCM::Gui::Dialogs::AddHosts::AcceleratorPropertiesDialog($wizard)];
}

sub createSummaryTree {
    my $self = shift;
    return LCM::SummaryTreeBuilders::RemoveHostRolesSummaryTreeBuilder::buildSummaryTree($self);
}
sub getAction {
    my ($self) = @_;
    return "remove_host_roles";
}

sub getProductName {
    my ($self) = @_;
    return "Remove Host Roles";
}

sub getSummaryTitle {
    my ($self) = @_;
    return "Remove host roles parameters";
}

sub getAutoInitializeServicesAction {
    return "removing";
}

sub getTimeoutValues{
    return [ qw(start_service stop_service start_instance stop_instance) ];
}

sub isAutoInitializeServicesApplicableParameterCallback{
    my $self = shift;
    return $self->isAddRolesValueApplicableForAutoInitialize(@_);
}

1;

