package LCM::Configuration::ValueChangeListeners::InstallUpdate::SelectedComponentsListenerBase;

use strict;
use LCM::Component qw($componentKeynameToBatchKey);
use LCM::Utils::CommonUtils;
use LCM::SignatureFileChecker;
use SDB::Install::System qw(isAdmin);
use SDB::Install::Globals qw(
    $gSignatureManifestName
    $gProductNameEngine
    $gKeynameXS2
    $gKeynameEngine
    $gKeynameInstaller
    $gOptionOptimizedExecutionMode
    $gOptionStandardExecutionMode
    $gKeynameLSS
);
use LCM::FeatureManager::GenericFeatureManager qw(LSS_TRUST_UNSIGNED_SERVER);
use LCM::FeatureManager::Feature;
use SDB::Install::Version;

sub new {
	return bless({}, shift());
}

sub onValueChange {
	my ($self, $selectedCmpsValue, $instconfig) = @_;

	return undef if(!$instconfig->fillSelectedComponents($selectedCmpsValue));
	return undef if(!$instconfig->handleSelectionOfResidentInstaller());
	return undef if(!$instconfig->validateDependencies());
    return undef if(!$instconfig->validateLSSDependency());
	return undef if(!$self->_checkSignatureFile($instconfig));

	$instconfig->_recomputeRequiredParameters();
	$instconfig->fillCertificatesHostmapDefaultValuesAndLocalData();
	$instconfig->setSkip('AutoAddXS2Roles', $self->_shallSkipAutoAddXs2Roles($instconfig));
	$instconfig->setSkip('ImportContentXS2', $self->_shallSkipImportContentXS2($instconfig));
	$instconfig->setSkip('ISCMode', $self->_shallSkipISCMode($instconfig));
	$instconfig->setSkip('XSComponentsCfg', $self->_shallSkipXSComponentsCfg($instconfig));
	$self->_initISCModeDefaultValue($instconfig);
    $self->_handleExecutionModeDefaultValue($instconfig);

	my $skipReferenceDataPath = $self->_shallSkipReferenceDataPath($instconfig);
	$instconfig->setSkip('ReferenceDataPath', $skipReferenceDataPath);
    $self->_skipUnsupportedParams($instconfig);
    $self->_handleLssRelatedFeatures($instconfig);
	return 1;
}

sub addResumeActionProgressMsg {
	my ($self, $configuration) = @_;

	if ($configuration->getComponentManager()->existsSelectedComponentInPendingState($configuration)) {
		my $persistenceData = $configuration->pers_load();
		return if (!defined $persistenceData);
		my $pendingStepName = $configuration->pers_getstepname($persistenceData->{step});
		$configuration->AddProgressMessage ("Resuming pending " . $configuration->getAction() . " at step '${pendingStepName}'");
	}
}

sub _shallSkipAutoAddXs2Roles {
	my ($self, $configuration) = @_;
	my $componentManager = $configuration->getComponentManager();
	my $xs2Component = $componentManager->getComponentByKeyName($gKeynameXS2);

	my $hasAutoAddXS2RolesPersistedValue = defined $configuration->getPersistedValue('AutoAddXS2Roles');
	my $isPendingInstall = ! $configuration->isUpdate() && $configuration->isPostServerResumeInstallation();
	return 0 if $isPendingInstall && $hasAutoAddXS2RolesPersistedValue;

	my $isPendingUpdate = $configuration->isUpdate() && $configuration->isPendingUpdate();
	return 0 if $isPendingUpdate && $hasAutoAddXS2RolesPersistedValue;

	return defined($xs2Component) && $xs2Component->isComponentSelected() && ! $xs2Component->isUpdate() ? 0 : 1;
}

sub _shallSkipImportContentXS2 {
    my ($self, $configuration) = @_;
    my $componentManager = $configuration->getComponentManager();
    my $xs2Component = $componentManager->getComponentByKeyName($gKeynameXS2);

    return defined($xs2Component) && $xs2Component->isComponentSelected() ? 0 : 1;
}

sub _shallSkipISCMode {
    my ($self, $configuration) = @_;
    my $componentManager = $configuration->getComponentManager();

    for my $componentKeyname ($gKeynameXS2, $gKeynameEngine){
        my $component = $componentManager->getComponentByKeyName($componentKeyname);
        return 0 if(defined($component) && $component->isComponentSelected())
    }
    return 1;
}

sub _shallSkipXSComponentsCfg {
    my ($self, $configuration) = @_;
    my $componentManager = $configuration->getComponentManager();
    my $xs2Component = $componentManager->getComponentByKeyName($gKeynameXS2);
    return defined($xs2Component) && $xs2Component->isComponentSelected() ? 0 : 1;
}

sub _checkSignatureFile {
	my ($self, $instconfig) = @_;
	my $componentManager = $instconfig->getComponentManager();
	my $installerComponent = $componentManager->getComponentByKeyName($gKeynameInstaller);

	return 1 if(! defined($installerComponent));
	return 1 if(! $installerComponent->isComponentSelected());
	if($instconfig->getIgnore('check_signature_file') && isAdmin()){
		$instconfig->getMsgLst()->addMessage("Ignoring $gSignatureManifestName file check due to --ignore command line switch");
		return 1;
	}

	if($instconfig->getIgnore('check_signature_file')){
		$instconfig->getErrMsgLst()->addError("Ignore option 'check_signature_file' can only be used when the tool is started with root privileges");
		$instconfig->setNoRetry('SelectedComponents', 1);
		return undef;
	}

	my $signaturePathCandidate = $installerComponent->getPath();
	my $sapEnvironment = $instconfig->isSAPEnvironment();
	my $signatureChecker = new LCM::SignatureFileChecker($signaturePathCandidate, $sapEnvironment);
	my $signatureCheckResult = $signatureChecker->checkFile();

    if  (!$signatureCheckResult) {
        $instconfig->getErrMsgLst()->appendMsgLst($signatureChecker->getErrMsgLst());
        return undef;
    } elsif (!$signatureChecker->getMsgLst()->isEmpty()) {
        my $msgLst = $signatureChecker->getMsgLst();
        $instconfig->getMsgLst()->appendMsgLst($msgLst);
        my $message = $msgLst->getMsgLstString();
        $instconfig->addParameterWarning("SelectedComponents", $$message);
        return 1;
    }
    return 1;
}

sub _handleExecutionModeDefaultValue {
    my ($self, $instconfig) = @_;
    my $executionMode = $instconfig->shouldUseOptimizedExecution() ? $gOptionOptimizedExecutionMode : $gOptionStandardExecutionMode;

    $instconfig->setDefault('ExecutionMode', $executionMode);
}

sub _shallSkipReferenceDataPath {
    my ($self, $configuration) = @_;

    my $selectedComponents = $configuration->getComponentManager()->getSelectedComponents();
    
    for my $component ( @$selectedComponents ) {
        if ( $component->getManifest()->isReferenceData() ) {
            return 0;
        }
    }
    return 1;
}

sub _initISCModeDefaultValue {
	...
}

sub _handleDefaultTenantUserCredentials{
    my ($self, $instconfig) = @_;
    return if (!$instconfig->isSystemInCompatibilityMode());

    my $mcm = $instconfig->getComponentManager();
    my $willInstallXS2 = $mcm->willInstallComponent($gKeynameXS2, $instconfig);

    $instconfig->setSkip('TenantUser', !$willInstallXS2);
    $instconfig->setSkip('SQLTenantUserPassword', !$willInstallXS2);

    my $systemUser = $instconfig->getValue('SystemUser');
    $instconfig->setDefault('TenantUser', $systemUser) if $systemUser && $willInstallXS2;
}

# Skip these parameters if the feature is not supported by the HANA, which will be installed
# This will prevent the underlying hdbinst/hdbupd to fail with Unknown option: --secure_store/--volume_encryption
sub _skipUnsupportedParams {
    my ($self, $config) = @_;
    my $featureManager = $config->getFeatureManager();

    $config->setSkip('SecureStore', !$featureManager->isSecureStoreSupported());
    if(!$config->isUpdate()){
        $config->setSkip('VolumeEncryption', !$featureManager->isVolumeEncryptionSupported());
    }
}

sub _handleLssRelatedFeatures {
    my ($self, $config) = @_;
    $self->_addTrustUnsignedServerFeature($config);

    my $isLssSelected = $config->isComponentSelected($gKeynameLSS);
    my $isServerSelected = $config->isComponentSelected($gKeynameEngine);

    my $hdbInstance = $config->getOwnInstance();
    my $isLssInstalled =  (defined $hdbInstance && $hdbInstance->getLssInstance()) ? 1 : 0;

    my $isTrustUnsignedApplicable = $isLssSelected || ($config->isUpdate() && $isLssInstalled && $isServerSelected) ? 1 : 0;
    $config->setSkip('LSSTrustUnsignedServer', !$isTrustUnsignedApplicable);
}

sub _addTrustUnsignedServerFeature {
    my ($self, $config) = @_;
    my $componentManager = $config->getComponentManager();
    my $lssComponent = $componentManager->getComponentByKeyName($gKeynameLSS);
    my $isLssSelected = defined($lssComponent) && $lssComponent->isComponentSelected();
    my $hdbInstance = $config->getOwnInstance();
    my $lssInstance = defined $hdbInstance ? $hdbInstance->getLssInstance() : undef;
    return if(!$isLssSelected && ! defined $lssInstance);

    my $lssExecutable;
    if(!$isLssSelected){
        $lssComponent = $config->getSystemComponentManager()->getComponentByKeyName($gKeynameLSS);
        $lssExecutable = File::Spec->catfile($lssInstance->getLssSidDir(), 'install', 'hdbinst');
    } else{
        $lssExecutable = $lssComponent->getExecutable();
    }

    $lssComponent->setMsgLstContext($config->getMsgLstContext());
    my $featureManager = $config->getFeatureManager();
    my $sourceVersion = SDB::Install::Version->new(split(/\.|-/, $lssComponent->getLcmSdkVersion($lssExecutable)));
    $featureManager->addFeature(LCM::FeatureManager::Feature->new(LSS_TRUST_UNSIGNED_SERVER,  ['2','5','30'], undef, $sourceVersion));
}

1;
