package LCM::Configuration::ValueChangeListeners::Uninstall::ForceListener;

use strict;
use LCM::App::ApplicationContext;
use SDB::Install::Globals qw($gKeynameEngine $gHostRoleXS2Worker);
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::Utils::RemoveHosts qw(IsRemovingLocalHost IsRemovingRemoteHost);
use LCM::Utils::RemoveRoles qw(IsRemovingLocalRole IsRemovingRemoteRole canRemoveHostRole shouldCheckRoleRemovalStatus);

####
# Important - if changed - changes might apply to the Web::ForceListener as well
####

sub new { return bless({}, shift()); }

sub onValueChange {
    my ($self, $value, $configuration) = @_;
    my $isForced = ($value =~ $bool_true_pattern);
    my $applicationContext = LCM::App::ApplicationContext::getInstance();
    my $hdblcmMode = $applicationContext->getMode();

    if( $isForced && IsRemovingLocalHost($configuration)){
        $configuration->getErrMsgLst()->addError('Selected components require removal of the local host, which is not allowed. Start the uninstallation from a different host');
        return undef;
    }

    my $shallSkipKeepUser = $self->_shallSkipUserParameters($configuration, $isForced);
    my $shallSkipRemoveHosts = $self->_shallSkipRemoveHosts($configuration, $isForced);
    my $shallSkipRemoveRoles = $self->_shallSkipRemoveRoles($configuration, $isForced);

    $configuration->setSkip('KeepUser', $shallSkipKeepUser);
    $configuration->setSkip('KeepUserHomeDir', $shallSkipKeepUser);
    $configuration->setSkip('RemoveHosts', $shallSkipRemoveHosts);
    $configuration->setSkip('RemoveHostRoles', $shallSkipRemoveRoles);

    if(!$isForced){
        my $result = $self->_checkNonForceUninstallation($configuration);
        my $shallSetNoRetry = !$result && $hdblcmMode eq 'CLI' ? 1 : 0;
        $configuration->setNoRetry('ForceRemoveHosts', $shallSetNoRetry);
        return $result;
    }

    if(! $self->checkCanRemoveSelectedComponents($configuration)){
        $configuration->setNoRetry('ForceRemoveHosts', 1);
        return undef;
    }
    return 1;
}

sub checkCanRemoveSelectedComponents {
    my ($self, $configuration) = @_;
    my $rc = 1;
    $rc &&= $self->checkCanRemoveHost($configuration) if !$configuration->isSkipped('RemoveHosts');
    $rc &&= $self->checkCanRemoveRoles($configuration) if !$configuration->isSkipped('RemoveRoles');
    return $rc;
}

sub checkCanRemoveHost {
    my ($self, $configuration) = @_;
    my $rc = 1;
    my @hosts = split(',', $configuration->getValue('RemoveHosts'));
    foreach my $host (@hosts) {
        $rc = 0 if !$self->canRemoveHost($configuration, $host);
    }
    return $rc;
}

sub canRemoveHost {
    my ($self, $configuration, $host) = @_;
    my $instance = $configuration->getOwnInstance();
    my $saveCtx = $instance->setMsgLstContext([ SDB::Install::MsgLst->new(), SDB::Install::MsgLst->new() ]);
    my $canSafelyRemoveHost = $instance->getHostRemoveStatus($host, $configuration);
    my $rc = 1;
    $configuration->getMsgLst()->appendMsgLst($instance->getMsgLst());

    if (! $canSafelyRemoveHost) {
        $configuration->appendErrorMessage("Cannot remove host '$host'", $instance->getErrMsgLst());
        $rc = 0;
    }

    $instance->setMsgLstContext($saveCtx);
    return $rc;
}

sub checkCanRemoveRoles {
    my ($self, $configuration) = @_;
    my $rc = 1;
    my $hosts = $configuration->getValue('RemoveRoles');
    my $xsControllerHost = $configuration->getOwnInstance()->getXsControllerHostname();
    my $remainingNonControllerXSWorkerCount = $self->_remainingNonControllerXSWorkerCount($configuration, $xsControllerHost);
    foreach my $host (keys %$hosts) {
        my @roles = split(',', $hosts->{$host});
        foreach my $role (@roles) {
            if (not ($role eq $gHostRoleXS2Worker && $host eq $xsControllerHost && $remainingNonControllerXSWorkerCount > 0)){
                return 1 if ! shouldCheckRoleRemovalStatus($configuration, $role);
                return canRemoveHostRole($configuration, $host, $role);
            }
        }
    }
    return $rc;
}

sub _remainingNonControllerXSWorkerCount {
    my ($self, $configuration, $xsControllerHost) = @_;
    my $sapSystem = $configuration->getSAPSystem();
    my $hostRolesInfo = $sapSystem->getNewDBInstances()->[0]->getHostRolesInfo();
    my $remainingXSWorkerCount = 0;
    foreach my $host (keys %{$hostRolesInfo}) {
        next if $host eq $xsControllerHost;
        if ( $hostRolesInfo->{$host} =~ /$gHostRoleXS2Worker/ ) {
            $remainingXSWorkerCount += 1;
        }
    }
    return $remainingXSWorkerCount;
}

sub _shallSkipRemoveHosts {
    my ($self, $configuration, $isForced) = @_;

    return 1 if(!$isForced);
    return IsRemovingRemoteHost($configuration) ? 0 : 1;
}

sub _shallSkipRemoveRoles {
    my ($self, $configuration, $isForced) = @_;

    return 1 if(!$isForced);
    return IsRemovingLocalRole($configuration) || IsRemovingRemoteRole($configuration) ? 0 : 1;
}

sub _shallSkipUserParameters {
    my ($self, $configuration, $isForced) = @_;
    my $isRemovingRemoteHost = IsRemovingRemoteHost($configuration);

    return $isForced && $isRemovingRemoteHost ? 0 : 1;
}

sub _checkNonForceUninstallation {
    my ($self, $configuration) = @_;
    my $selectedComponentsCsv = $configuration->getValue('SelectedComponents');
    my $result = 1;

    for my $componentBatchKey (split(/\s*,\s*/, $selectedComponentsCsv)) {
        my $component = $configuration->getComponentByBatchKey($componentBatchKey);

        next if(!defined($component));
        next if(!$component->can('preCheckUninstallComponent'));
        next if($component->preCheckUninstallComponent($configuration));

        $result = undef; # We have encountered an error, but we want to check for the other components too.
    }
    return $result;
}

1;