package LCM::Configuration::ValueChangeListeners::Update::ServerDetectionListener;

use strict;
use SDB::Install::Globals qw (
    $gKeynameEngine
    $gKeynameInstaller
    $gProductName
    $gUpgradeRestrictionsMessage
    $gSignatureManifestName
    GetServerNotFoundMessage
    GetNewerServerAlreadyInstalledMessage
);
use SDB::Install::System;
use SDB::Install::User;


sub new {
    return bless({}, shift());
}

sub onValueChange {
    my ($self, $value, $instconfig) = @_;
    my $mcm = $instconfig->getComponentManager();
    my $hanaServer = $mcm->getComponentByKeyName($gKeynameEngine);
    my $installer = $mcm->getComponentByKeyName($gKeynameInstaller);

    if(!defined($hanaServer)) {
        $instconfig->setErrorMessage(GetServerNotFoundMessage("$gProductName System update"));
        return 0;
    }

    if (!defined($hanaServer->getSignature()) && !SDB::Install::System::isAdmin()) {
        my $currentUsername = SDB::Install::User->new()->getname();
        $instconfig->setErrorMessage("Update system cannot be performed as user '$currentUsername' without $gSignatureManifestName file in the installation kit.");
        return 0;
    }

    if (!defined($installer)) {
        $instconfig->setErrorMessage("Failed to detect external installer component.");
        return 0;
    }

    if (!$self->_isServerUpdatable($instconfig, $hanaServer)) {
        return 0;
    }

# Artificially select the installer, so that the SignatureCheck listener will verify its signature
    $installer->setIsSigned(1);
    $installer->selectComponent(1);
# Set default value so that CheckParams will trigger the signature verification listener
    $instconfig->setDefault('ExternalExecutablePath', $instconfig->getExternalHdblcmPath(), 1);
    return 1;
}

sub _isServerUpdatable {
    my ($self, $instconfig, $detectedServer) = @_;
    my $scm = $instconfig->getSystemComponentManager();
    my $installedServer = $scm ? $scm->getComponentByKeyName($gKeynameEngine) : undef;
    if(!defined($installedServer)) {
        $instconfig->setErrorMessage("Failed to detect installed $gProductName Database server");
        return 0;
    }

    my $srcManifest = $installedServer->getManifest();
    my $trgManifest = $detectedServer->getManifest();
    my $srcVersion = $srcManifest->getVersionObject();
    my $trgVersion = $trgManifest->getVersionObject();

    if ($srcVersion->isNewerThan($trgVersion)) {
        if (!$instconfig->getIgnore('check_version')) {
            my $errorMessage = GetNewerServerAlreadyInstalledMessage($trgVersion->asString(), $srcVersion->asString());
            $instconfig->setErrorMessage($errorMessage);
            return 0;
        }
        $instconfig->getMsgLst()->addMessage("Ignoring version check errors due to command line switch '--ignore'");
    }

    if (!$trgManifest->checkSPSUpgradeLimitation($srcManifest)) {
        $instconfig->setErrorMessage($gUpgradeRestrictionsMessage, $trgManifest->getErrMsgLst());
        return 0;
    }

    return 1;
}

1;