package LCM::Fcgi::App::LCMFcgiApp;

use strict;
use File::Basename;
use File::Spec;
use SDB::Web::FcgiDispatcher qw(loop);
use LCM::App::UpdateSystemUtils;
use LCM::HostsUtility qw( GetNumberOfSystemHosts );
use LCM::Fcgi::SLPP;
use LCM::Installer;
use Getopt::Long;
use LCM::DevelopmentTrace;
use LCM::Slpp::ErrorState;
no warnings 'exec';
use SAPDB::Install::Hostname;
use SDB::Install::Globals qw ($gKeynameEngine $gKeynameInstaller $gKeynameSDA $gProductNameSystem $gFailedSignatureVerificationMessage $gProductNameEngine $gFlavourPlatform);
use SDB::Install::Configuration qw($bool_true_pattern $bool_false_pattern);

use LCM::Slpp::SlppAdapter;
use LCM::Slpp::SlppExecutionHandler qw($HDBLCM_ERROR_MESSAGE);
use LCM::App::ApplicationContext;
use LCM::Fcgi::App::EnvironmentUtilities;
use LCM::App::AbstractApplication qw( getURL openURL );
use LCM::Fcgi::App::ProcessActivityListener;
use LCM::FileUtils;
use LCM::Utils::CommonUtils qw(isSingleDB isSystemReplication);
use base qw (LCM::App::AbstractApplication SDB::Install::App::Console);

my $MAX_PROCESS_INACTIVITY_IN_SECONDS = (60 * abs(int($ENV{'HDBLCMWEB_INACTIVE_TIMEOUT'} // 60)));
my ($SID, $SCENARIO, $SERVER_PATH, $TIMEOUT, $OPT1, $OPT2, $OPT3, $OPT4, $OPT5) = ('', '', '', '', '', '', '', '', '');
# $OPT1 can take 'true' or 'false'. 
# True means that a signature verification of the installation kit will be performed
# False - no signature verification of the installation kit

# Override
sub getProgramName {
    return 'hdblcmweb';
}

sub new {
    my $self = shift->SUPER::new();
    $self->InitCmdLineArgs([]);
    $self->{hdblcmweb} = 1;
    LCM::App::ApplicationContext::getInstance()->setApplication($self);

    return $self;
}

sub InitTrace {
	my ($self) = @_;
	eval{
		$self->SUPER::InitTrace();
	};
}

# Override
sub main {
	_parseStartupArguments();

# we dont want hdblcmweb show RT errors when it is run by curious users from the command line w/o the proper arguments.
# (it is well visible in the kit and the resident hdblcm, and has such a promising name.)
	if(!defined($SID) || !defined($SCENARIO) || $SID eq '' || $SCENARIO eq '') {
		my ($progname, $url) = (getProgramName(), getURL());
		if( openURL($url) ) {
			print "$progname should not be called from the command line. You can access the web user interface by opening the following URL in your browser: $url\n";
		}
		return -1;
	}
# Prepare execution environment with some required sidadm specific variables/values
	PrepareExecutionEnvironment($SID);

	my $app = new __PACKAGE__;

	my $scenarioPreconditionError = checkScenarioPreconditions($app);

	$app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
	my $slppAdapter = LCM::Slpp::SlppAdapter::getInstance();
	$slppAdapter->init($SID, $SCENARIO, $OPT1, $app, $scenarioPreconditionError);

	if(!defined($scenarioPreconditionError) && shouldCheckForInitialErrors($SCENARIO)) {
		my $message = $slppAdapter->getConfiguration()->getErrorString();
		$scenarioPreconditionError = $message ? new LCM::Slpp::ErrorState("${SCENARIO}_error", 2, "Information", $message) : undef;
	}

	my $executionHandler = LCM::Slpp::SlppExecutionHandler::getInstance();
	$executionHandler->init($slppAdapter, $scenarioPreconditionError, $TIMEOUT);
	$executionHandler->start();
	my $rc = 1;
	{
		my $processActivityListener = new LCM::Fcgi::App::ProcessActivityListener($executionHandler, $slppAdapter, $MAX_PROCESS_INACTIVITY_IN_SECONDS);
		eval {
			$processActivityListener->notify(); # Cheat - start the auto stop timer before any requests are recieved
			# Start the FCGI loop
			loop($SID, $SCENARIO, [ $processActivityListener ]);
			1;
		} or do {
			my $error = $@;
			$rc = 0;
		};
	}

	return $rc;
}

sub getFlavour {
	my ($self) = @_;
	my $installer = LCM::Installer->new();
	if ($installer->isInstalled()) {
        my $installation = $installer->getInstallation();
        return $installation ? $installation->getHANAFlavour() : $gFlavourPlatform;
	} else {
		my $server = getServerComponentFromPath($self->getServerPath(), SDB::Install::MsgLst->new());
		my $manifest = $server ? $server->getManifest() : undef;
		return $manifest ? $manifest->getHANAFlavour() : $gFlavourPlatform;
	}
}

sub isResident {
	return LCM::Installer->new()->isInstalled();
}

sub shouldCheckForInitialErrors {
	my ($scenario) = @_;
	my $shouldCheck = grep { $_ eq $scenario } ('update_system', 'optional_configuration_v3', 'optional_configuration_v5');
	return $shouldCheck;
}

sub checkScenarioPreconditions {
	my ($application) = @_;
	if(!LCM::DevelopmentTrace::isEnabledDevelopmentTrace()){
		my $traceFilePath = LCM::DevelopmentTrace::GetDevelopmentTraceFile();
		my $sidadmUser = lc($SID) . 'adm';
		my $errorId = $SCENARIO . "_error";
		my $message = "Trace file '$traceFilePath' could not be created. Make sure that user '$sidadmUser' has permissions to create this file";
		return new LCM::Slpp::ErrorState($errorId, 1, $HDBLCM_ERROR_MESSAGE, $message);
	}
	if(LCM::Installer->new()->isInstalled() && $SCENARIO eq 'update_system'){
		return handleUpdateSystemScenario($application);
	}
	
	if($SCENARIO eq 'remove_hosts'){
		return _checkRemoveHostsPreconditions();
	}
	if($SCENARIO eq 'uninstall_system'){
		return _checkUninstallPreconditions();
	}
	return undef;
}

sub handleUpdateSystemScenario {
	my ($application) = @_;
	my $errorList = SDB::Install::MsgLst->new();
	my $errorStateObject = _checkUpdateSystemPreconditions($errorList);

	$application->setErrorMessage("Update system not possible", $errorList);
	return $errorStateObject;
}

sub _checkUpdateSystemPreconditions {
	my ($errorList) = @_;
	my $errorId = "${SCENARIO}_error";
	my $serverPath = getServerPathFromUserInput($SERVER_PATH, $errorList);
	if (!defined($serverPath)) {
		return LCM::Slpp::ErrorState->new($errorId, 1, $HDBLCM_ERROR_MESSAGE, ${$errorList->getMsgLstString()});
	}

	if(isPathToServerBinaries($serverPath)){
		$serverPath = File::Basename::dirname($serverPath);
	}

	my $installerPath = getLCMFromServerPath($serverPath, 'hdblcmweb');
	if(!updateNonResidentWebResources($installerPath, $errorList)) {
		return LCM::Slpp::ErrorState->new($errorId, 1, $HDBLCM_ERROR_MESSAGE, ${$errorList->getMsgLstString()});
	}

	if (!checkVerifySignatureOption()) {
		return LCM::Slpp::ErrorState->new($errorId, 1, $HDBLCM_ERROR_MESSAGE, "Value '$OPT1' is invalid. Please use 'true' or 'false'!");
	}

	my $shouldVerifySignature = shouldVerifySignature($errorList);
	if (!defined ($shouldVerifySignature)) {
		return LCM::Slpp::ErrorState->new($errorId, 1, $HDBLCM_ERROR_MESSAGE, ${$errorList->getMsgLstString()});
	}

	if ($shouldVerifySignature) {
		my $installerComponent = getLCMComponentFromPath($installerPath, $errorList);
		if (!validateExternalInstaller($installerComponent, $errorList)){
			$errorList->addError("\n$gFailedSignatureVerificationMessage");
			$errorList->addError("To ignore this error and continue with the update, use the checkbox to disable the verification.");
			my $errDisplayName = "Authenticity verification of the installation kit failed";
			return LCM::Slpp::ErrorState->new($errorId, 3, $errDisplayName, ${$errorList->getMsgLstString()});
		}
	}

	executeNonResidendHdblcm($installerPath, $serverPath, $shouldVerifySignature, $errorList);
	# Nonresident hdblcm was not started, something obviously went wrong
	return LCM::Slpp::ErrorState->new($errorId, 1, $HDBLCM_ERROR_MESSAGE, ${$errorList->getMsgLstString()});
}

sub executeNonResidendHdblcm {
	my ($installerPath, $serverPath, $shouldVerifySignature, $errorList) = @_;
	my $newHdblcmwebExecutable = File::Spec->catfile($installerPath, 'hdblcmweb');
	my @newArgs = ("-sid=$SID", "-scenario=$SCENARIO", "-server_path=$serverPath", "-timeout=$TIMEOUT");
	push @newArgs, "-opt1=".($shouldVerifySignature ? 'true': 'false');

#	Delegate execution to hdbweb from the provided server
	exec($newHdblcmwebExecutable, @newArgs);

	$errorList->initMsgLst();
	$errorList->addError("Execution of '$newHdblcmwebExecutable' failed : $!");
}

sub _checkRemoveHostsPreconditions {
	if(GetNumberOfSystemHosts() == 1){
		my $errorId = $SCENARIO . "_error";
		my $message = "Cannot remove hosts from a single-host system.";
		return new LCM::Slpp::ErrorState($errorId, 2, "Information", $message);
	}
	return undef;
}

sub _checkUninstallPreconditions {
	require LCM::Component;

	my $systemComponentManager = LCM::Installer->new()->getOwnSystemComponentManager();
	my $errorId = "${SCENARIO}_error";
	my $message = "There aren't any additional components which can be uninstalled";

	if(!defined($systemComponentManager)) {
		return new LCM::Slpp::ErrorState($errorId, 2, "Information", $message);
	}

	return $systemComponentManager->hasUninstallableComponents() ? undef : LCM::Slpp::ErrorState->new($errorId, 2, "Information", $message);
}

sub checkVerifySignatureOption {
	return 1 if !$OPT1;
	return ($OPT1 =~ /$bool_true_pattern/ || $OPT1 =~ /$bool_false_pattern/);
}

sub shouldVerifySignature {
	my ($errorList) = @_;
	return $OPT1 =~ /$bool_true_pattern/ if($OPT1);

# If the resident installer has valid signature => enable verifiation by default
	my $defaultValue = validateResidentInstaller($errorList);
	if (!defined $defaultValue) {
		return undef;
	}
	return $defaultValue;
}

sub _parseStartupArguments {
	GetOptions ('sid=s' => \$SID, 'scenario=s' => \$SCENARIO, 'server_path=s' => \$SERVER_PATH, 'timeout=i' => \$TIMEOUT,
				'opt1=s' => \$OPT1, 'opt2=s' => \$OPT2, 'opt3=s' => \$OPT3, 'opt4=s' => \$OPT4, 'opt5=s' => \$OPT5);
	if($TIMEOUT !~ /^([1-9]|[1-5][0-9]|60)$/){
		# Set default timeout if -timeout isn't in the interval [1 - 60]
		$TIMEOUT = 3; 
	}
}

sub getServerPath {
    return $SERVER_PATH;
}

1;
